/*
 * Photo Image Print System
 * Copyright (C) 2001-2004 EPSON KOWA Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2001-2004.
 *
 * This file is part of the `ekpstm' program.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include "ekpcom.h"


/* key is attached to beginning of packet */
const char Packet_Key[] = "pcp";


enum _PACKET_DATA
{
  Key_Size = 3,			/* "pcp" */
  Header_Size = 5,		/* "pcp"+size */
  Accsess_Wait = 10,
  Max_Data_Size = 1024		/* max packet size */
};

static int server_sock_fd = -1;


/*
 * Parameter   : none
 * Returns     : returns zero on success, or -1 if an error occurred.
 * Description : open a socket and start a communication with ekpd.
 */
int
sock_open (void)
{
  int sockfd, len;
  struct sockaddr_in address;

  if (server_sock_fd >= 0)
    return 0;

  sockfd = socket (AF_INET, SOCK_STREAM, 0);
  address.sin_family = AF_INET;
  address.sin_addr.s_addr = htonl (INADDR_ANY);
  address.sin_port = htons (35586); /* did hard coding in port number (35586) */
  len = sizeof (address);
  if (connect (sockfd, (struct sockaddr *)&address, len))
    {
      return -1;
    }

  server_sock_fd = sockfd;
  return 0;
}


/*
 * Parameter   : none
 * Returns     : none
 * Description : close a socket and end a communication with ekpd.
 */
void
sock_close (void)
{
  if (server_sock_fd >= 0)
    {
      shutdown (server_sock_fd, 2);
      server_sock_fd = -1;
    }
  return;
}


/*
 * Parameter   : buf      -pointer to write data buffer.
 *               lp_wsize -on success, the number of bytes write is returned.
 * Returns     : returns zero on success, or -1 if an error occurred.
 * Description : send a packet of data to ekpd.
 */
int
sock_write (char* buf, int* lp_wsize)
{
  char packet [Max_Data_Size + Header_Size];
  int size;

  size = *lp_wsize;
  *lp_wsize = 0;

  if (server_sock_fd < 0)
    return -1;

  if (size <= 0)
    return 0;

  memcpy (packet, Packet_Key, Key_Size); /* "pcp" */
  packet[3] = size / 0xFF;	/* date size (upper) */
  packet[4] = size % 0xFF;	/* date size (lower) */
  memcpy (packet + Header_Size, buf, size); /* date */

  size = send (server_sock_fd, packet, size + Header_Size, MSG_NOSIGNAL);

  if (size < 0)
    return -1;

  *lp_wsize = size;
  return 0;
}

/*
 * Parameter   : buf      -pointer to read data buffer
 *               lp_wsize -on success, the number of bytes read is returned.
 * Returns     : returns zero on success, or -1 if an error occurred.
 * Description : receive a packet of data from ekpd.
 */
int
sock_read (char* buf, int* lp_rsize)
{
  char packet [Max_Data_Size + Header_Size];
  int size, dsize;

  size = *lp_rsize;
  *lp_rsize = 0;

  if (server_sock_fd < 0)
    return -1;

  if (size <= 0)
    return 0;

  dsize = recv (server_sock_fd, packet, size, MSG_NOSIGNAL);

  if (dsize < 0)
    return -1;
  else if (dsize < Header_Size)
    return 0;

  if (memcmp (packet, Packet_Key, Key_Size) != 0) /* check of "pcp" */
    return 0;

  /* date size */
  dsize = packet [3];
  dsize <<= 8;
  dsize += packet [4];

  if (dsize == 0)		/* Received a message of error */
    return packet[5];

  if (dsize > size)
    return 0;

  /* copy */
  memcpy (buf, packet + Header_Size, dsize);

  *lp_rsize = dsize;
  return 0;
}


/*
 * Parameter   : none
 * Returns     : returns zero on success, or -1 if an error occurred.
 * Description : reopen a socket and restart a communication with ekpd.
 */
int
sock_reopen (void)
{
  sock_close ();
  return sock_open ();
}


/*
 * Parameter   : buf      -pointer to write data buffer.
 *               lp_wsize -on success, the number of bytes read is returned.
 * Returns     : returns zero on success, or -1 if an error occurred.
 * Description : version to ignore reply of sock_write.
 */
int
sock_write_abandon_reply (char* buf, int* lp_wsize)
{
  char dummy[128];
  int len = 128;
  int tmp_size = *lp_wsize;

  if (sock_write (buf, lp_wsize) < 0)
    {
      sock_reopen ();
      *lp_wsize = tmp_size;
      sock_write (buf, lp_wsize);
    }

  if (*lp_wsize <= 0)
    return -1;

  sock_read (dummy, &len);
  return 0;
}
