/*
 * Photo Image Print System
 * Copyright (C) 2001-2004 EPSON KOWA Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2001-2004.
 *
 * This file is part of the `ekpstm' program.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "interface.h"
#include "support.h"

#include "stm.h"
#include "ekpcom.h"
#include "error.xpm"
#include "error_i3.xpm"
#include "error_i4.xpm"
#include "error_i6.xpm"
#include "error_i8.xpm"
#include "error_ia.xpm"
#include "printing1.xpm"
#include "printing2.xpm"
#include "printing3.xpm"
#include "ready.xpm"
#include "ready_i1.xpm"
#include "warning.xpm"

typedef enum
	{
		STS_ERR_FATAL,	/* fatal error */
		STS_ERR_JAM,	/* paper jam error */
		STS_ERR_NO_INK,	/* ink end error */
		STS_ERR_NO_PAPER, /* no paper error */
		STS_ERR_PRINTER, /* <no support> */
		STS_NOT_READY,	/* printer busy */
		STS_PRINTING,	/* printing */
		STS_IDLE,	/* idle */
		STS_NO_REPLY,	/* not response */
		STS_MOTOR_OFF,	/* motor is off */
		STS_EKPD_DOWN,	/* ekpd stops */
		STS_ERR_PART_CHANGE, /* maintenance call */
		STS_ERR_ROLL,	/* fed incorrectly */
		STS_ERR_REPETITION, /* multi-page feed error */
		STS_PRINTER_RESET, /* cancel it */
		STS_DEFAULT	/* initial condition */
	}PRINTER_STS;

typedef struct{
	GtkWidget *window;	/* top window */
	GtkWidget *prt_pix;	/* printer pixmap */

	PRINTER_STS cur_status;	/* status */
	gint num_pict;		/* number of pixmaps */
	gint cur_pict;		/* number of pixmap in representation */
	gint timer;		/* gtk+ timer */
	GdkPixmap *pixmap[4];	/* printer pixmap list */
	GdkBitmap *mask[4];	/* mask list */

	int stm_st;	   /* status flag of status monitor oneself */
}PRINTSTSSTRUCT;


static PRINTSTSSTRUCT _status_str;


static PRINTER_STS ReadStatuslogfile (gint*, gint*);
static void SetAllUserButtonHidden (void);
static void DisplayInformation (GtkText*, char*);
static int GetPrinterStatus (gpointer);
static void auto_down_count_start (void);
static void auto_down_count (int);

static PRINTER_STS
ReadStatuslogfile (gint *black_rest, gint *color_rest)
{
	char *lpInk, *lpSts, *lpErr;
	char lpReply[1024], StsCode[3], ErrCode[3]; /* 1024 : max of status strings */
	char K[3], C[3], M[3], Y[3], LC[3], LM[3], DY[3]; /* 3 : amount of code strings */
	char get_status_command[] = { 's', 't', 0x01, 0x00, 0x01 };
	int com_len, rep_len;
  
	/* initialization */
	*black_rest = 0;
	*color_rest = 0;
	StsCode[2] = 0;
	ErrCode[2] = 0;
	memcpy (K, "-1", 3);
	memcpy (C, "-1", 3);
	memcpy (M, "-1", 3);
	memcpy (Y, "-1", 3);
	memcpy (LC, "-1", 3);
	memcpy (LM, "-1", 3);
	memcpy (DY, "-1", 3);

	com_len = sizeof (get_status_command);
	rep_len = 1024;

	/* get printer status */
	if (sock_write (get_status_command, &com_len) < 0)
	{
		if (sock_reopen () < 0)
		{
			return STS_EKPD_DOWN;
		}
		return STS_NO_REPLY;
	}
  
	sock_read (lpReply, &rep_len);
	if (rep_len == 0)
	{
		return STS_NO_REPLY;
	}

	/* ink residual quantity */
	lpInk = strstr(lpReply, "IQ:");
	if(lpInk){
		long rest;
		char *stopstring;

    
		strncpy(K, lpInk + 3, 2);
		strncpy(C, lpInk + 5, 2);
		strncpy(M, lpInk + 7, 2);
		strncpy(Y, lpInk + 9, 2);
		if (lpInk[11] != ';')	/* 6 ink */
		{
			strncpy(LC, lpInk + 11, 2);
			strncpy(LM, lpInk + 13, 2);

			if (lpInk[15] != ';')	/* 7 ink */
				strncpy(DY, lpInk + 15, 2);
		}

		*black_rest = (gint)strtol(K, &stopstring, 16);
		*color_rest = (gint)strtol(C, &stopstring, 16);
		rest = strtol (M, &stopstring, 16);
		if ((gint)rest < *color_rest) *color_rest = (gint)rest; 
		rest = strtol (Y, &stopstring, 16);
		if ((gint)rest < *color_rest) *color_rest = (gint)rest;

		rest = strtol (LC, &stopstring, 16);
		if ((rest >= 0) && ((gint)rest < *color_rest)) *color_rest = (gint)rest; 
    
		rest = strtol (LM, &stopstring, 16);
		if ((rest >= 0) && ((gint)rest < *color_rest)) *color_rest = (gint)rest; 

		rest = strtol (DY, &stopstring, 16);
		if ((rest >= 0) && ((gint)rest < *color_rest)) *color_rest = (gint)rest; 
	}
  
	/* status analysis */
	lpSts = strstr(lpReply, "ST:");
	if(lpSts){
		strncpy(StsCode, lpSts + 3, 2);
		if(!strcmp(StsCode, "00")){

			/* error status analysis */
			lpErr = strstr(lpReply, "ER:");
			if(lpErr){
				strncpy(ErrCode, lpErr + 3, 2);
				if(!strcmp(ErrCode, "04")) return STS_ERR_JAM;
				if(!strcmp(ErrCode, "05")) return STS_ERR_NO_INK;
				if(!strcmp(ErrCode, "06")) return STS_ERR_NO_PAPER;
				if(!strcmp(ErrCode, "10")) return STS_ERR_PART_CHANGE;
				if(!strcmp(ErrCode, "11")) return STS_ERR_ROLL;
				if(!strcmp(ErrCode, "12")) return STS_ERR_REPETITION;
				if(!strcmp(ErrCode, "0B")) return STS_ERR_PRINTER;
			}
			return STS_ERR_FATAL;
		}
		if(!strcmp(StsCode, "01") || !strcmp(StsCode, "07")) return STS_NOT_READY;
		if(!strcmp(StsCode, "03") || !strcmp(StsCode, "02")) return STS_PRINTING;
		if(!strcmp(StsCode, "04")) return STS_IDLE;
		if(!strcmp(StsCode, "09")) return STS_MOTOR_OFF;
		if(!strcmp(StsCode, "CN")) return STS_PRINTER_RESET;
	}
	return STS_NO_REPLY;
}


/* turn every button into non-representation */
static void
SetAllUserButtonHidden ()
{
	GtkWidget *widget;
  
	widget = lookup_widget (_status_str.window, "eject_button");
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "continue_button");
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "reset_button");
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "ink_button");
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "cancel_button");
	gtk_widget_set_sensitive (widget, TRUE);
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "close_button");
	gtk_widget_hide (widget);

	return;
}


/* indicate detailed information in information box */
static void
DisplayInformation (GtkText *text, char *info)
{
	gtk_text_set_point (text, 0);
	gtk_text_forward_delete (text, gtk_text_get_length (text));
	gtk_text_insert (text, NULL, NULL, NULL, info, -1);

	/* return automatic scrolling to the top */
	gtk_text_set_point (text, 0);
	gtk_text_insert (text, NULL, NULL, NULL, " ", -1);
	gtk_text_backward_delete (text, 1);
	return;
}

static int
GetPrinterStatus (gpointer data)
{
	PRINTER_STS status;
	gint i, black_rest, color_rest;
	GtkWidget *widget, *txt, *label;

	/* Post of ink residual quantity */
	status = ReadStatuslogfile(&black_rest, &color_rest);

	widget = lookup_widget (_status_str.window, "black_prog");
	gtk_progress_bar_update(GTK_PROGRESS_BAR(widget), (gfloat)black_rest / 100);
	widget = lookup_widget (_status_str.window, "color_prog");
	gtk_progress_bar_update(GTK_PROGRESS_BAR(widget), (gfloat)color_rest / 100);

	/* Auto down */
	if (status == STS_IDLE)
		auto_down_count (0);	/* count */
	else
		auto_down_count (1);	/* reset */
  
  
	if (status == STS_EKPD_DOWN)
	{
		widget = create_msg_no_ekpd ();
		gtk_widget_show (widget);
		return 0;
	}
  
	if (status == STS_MOTOR_OFF)
		status = _status_str.cur_status;
  
	if (status == _status_str.cur_status)
	{
		if ((status == STS_ERR_FATAL || status == STS_ERR_JAM)
		    &&  _status_str.stm_st == 2)
		{
			SetAllUserButtonHidden();

			widget = lookup_widget (_status_str.window, "reset_button");
			gtk_widget_show(widget);
			_status_str.stm_st = 0;
		}
      
		/* Post of animated image */
		if(_status_str.num_pict == 1)
			return 1;

		_status_str.cur_pict++;

		if(_status_str.cur_pict > _status_str.num_pict - 1)
			_status_str.cur_pict = 0;

		gtk_pixmap_set (GTK_PIXMAP(_status_str.prt_pix),
				_status_str.pixmap[_status_str.cur_pict],
				_status_str.mask[_status_str.cur_pict]);
		return 1;
	}
  
	if (_status_str.cur_status != STS_DEFAULT)
	{
		/* According to a state, all the buttons from which a display
		 * state changes are made un-displaying. */
		for (i = 0; i < _status_str.num_pict; i++)
			gdk_pixmap_unref (_status_str.pixmap[i]);
	}
	/* A re-setup of PIXMAP, an information display, and the state of
	 * a button are changed by state changes. */
	_status_str.cur_pict = 0;
	_status_str.stm_st = 0;

	label = lookup_widget (_status_str.window, "st_label");
	txt = lookup_widget (_status_str.window, "st_txt");

	switch(status)
	{
	case STS_ERR_FATAL:
		_status_str.stm_st = 1;
      
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_ia);
		gtk_label_set_text(GTK_LABEL(label), _("General error"));
		DisplayInformation(GTK_TEXT(txt), _("Delete all print jobs and turn the printer off. "
						    "Remove any foreign objects from inside the printer. "
						    "After a few minutes, turn the printer back on. "
						    "If the problem persists, contact your dealer."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_JAM:
		_status_str.stm_st = 1;
      
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i8);
		gtk_label_set_text(GTK_LABEL(label), _("Paper jam"));
		DisplayInformation(GTK_TEXT(txt), _("Press the Maintenance button on the printer or click the Eject button when it appears on the screen. Remove any remaining jammed paper by hand."));
		SetAllUserButtonHidden();

#ifndef NO_ACTION
		widget = lookup_widget (_status_str.window, "eject_button");
		gtk_widget_show(widget);
#endif
		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);

		break;

	case STS_ERR_NO_INK:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i4);

		gtk_label_set_text(GTK_LABEL(label), _("Ink out"));
		DisplayInformation(GTK_TEXT(txt), _("Ink is out.\n"
						    "Replace the ink cartridge."));

		SetAllUserButtonHidden();

#ifdef INK_CHENGE_OK
		widget = lookup_widget (_status_str.window, "ink_button");
		gtk_widget_show(widget);
#endif

		if (_status_str.cur_status == STS_IDLE || _status_str.cur_status == STS_DEFAULT)
			widget = lookup_widget (_status_str.window, "close_button");
		else
			widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);

		break;

	case STS_ERR_NO_PAPER:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i6);
		gtk_label_set_text(GTK_LABEL(label), _("Paper not loaded correctly"));
		DisplayInformation(GTK_TEXT(txt), _("Reload the paper and press the Maintenance button on the printer or click the Continue button when it appears on the screen. "
						    "To cancel print jobs, click the Cancel button."));
		SetAllUserButtonHidden();

#ifndef NO_ACTION
		widget = lookup_widget (_status_str.window, "continue_button");
		gtk_widget_show(widget);
#endif
		widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);

		break;
		/*
		  case STS_ERR_PRINTER:
		  _status_str.num_pict = 2;
		  _status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
		  &_status_str.mask[0], NULL,
		  (gchar**)error);
		  _status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
		  &_status_str.mask[1], NULL,
		  (gchar**)error_ia);
		  gtk_label_set_text(GTK_LABEL(label), _(""));
		  DisplayInformation(GTK_TEXT(txt), _(""));
		  SetAllUserButtonHidden();

		  widget = lookup_widget (_status_str.window, "close_button");
		  gtk_widget_show(widget);
		  break;
		*/

	case STS_ERR_PART_CHANGE:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_ia);
		gtk_label_set_text(GTK_LABEL(label), _("Maintenance call"));
		DisplayInformation(GTK_TEXT(txt), _("Parts inside your printer require maintenance or replacement. "
						    "Contact your dealer."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;
      
	case STS_ERR_ROLL:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_ia);
		gtk_label_set_text(GTK_LABEL(label), _("The paper has fed incorrectly"));
		DisplayInformation(GTK_TEXT(txt), _("Press the roll paper button to return the paper to the correct position."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;
      
	case STS_ERR_REPETITION:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i6);
		gtk_label_set_text(GTK_LABEL(label), _("Multi-page feed error"));
		DisplayInformation(GTK_TEXT(txt), _("Several pages have been fed into the printer at once. "
						    "Remove and reload the paper, then press the maintenance button."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;
      
	case STS_NOT_READY:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)ready);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)ready_i1);
		gtk_label_set_text(GTK_LABEL(label), _("Printer busy"));
		DisplayInformation(GTK_TEXT(txt), _("The printer is busy cleaning the print head or charging ink. "
						    "Please wait."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_PRINTING:
		auto_down_count_start ();

		_status_str.num_pict = 3;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)printing1);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)printing2);
		_status_str.pixmap[2] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[2], NULL,
								     (gchar**)printing3);
		gtk_label_set_text(GTK_LABEL(label), _("Printing"));
		DisplayInformation(GTK_TEXT(txt), _("Please wait."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);

		break;

	case STS_IDLE:
		_status_str.num_pict = 1;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)ready);
		gtk_label_set_text(GTK_LABEL(label), _("Ready"));
		DisplayInformation(GTK_TEXT(txt), _("Ready to print."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_PRINTER_RESET:
		_status_str.stm_st = 1;
      
		_status_str.num_pict = 1;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)warning);
		gtk_label_set_text(GTK_LABEL(label), _("Cancel"));
		DisplayInformation(GTK_TEXT(txt), _("Turn the printer off, and remove any jammed paper. "
						    "After a few minutes, turn the printer back on."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	default:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i3);
		gtk_label_set_text(GTK_LABEL(label), _("Communication error"));
		DisplayInformation(GTK_TEXT(txt), _("Check all connections and make sure all devices are on. "
						    "If the power was turned off during printing, cancel the print job. "
						    "If the error does not clear, see your manual."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;
	}

	gtk_pixmap_set(GTK_PIXMAP(_status_str.prt_pix),
		       _status_str.pixmap[0],
		       _status_str.mask[0]);
	_status_str.cur_status = status;
	return 1;
}



static int _down_count_start;
static int _down_limit;
static int _down_counter;

/* attachment of autodown */
void
auto_down_set (int sec)
{
	_down_count_start = 0;
	_down_limit = sec;
	_down_counter = 0;
	return;
}

/* start monitor for autodown */
static void
auto_down_count_start (void)
{
	_down_count_start = 1;
	return;
}

/* decide execution of autodown */
static void
auto_down_count (int flag)
{
	if (_down_limit <= 0 || _down_count_start == 0)
		return;

	if (flag)			/* Refresh */
		_down_counter = 0;
	else
		_down_counter ++;

	if (_down_counter >= _down_limit)
		gtk_main_quit ();

	return;
}

/* initialization of status monitor */
void
init_stm (GtkWidget* window)
{
	/* Widget making for icon representation */
	_status_str.prt_pix = lookup_widget (window, "prt_pix");
	
	_status_str.num_pict = 1;
	_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(window->window,
							     &_status_str.mask[0], NULL,
							     (gchar**)warning);
	gtk_pixmap_set(GTK_PIXMAP(_status_str.prt_pix),
		       _status_str.pixmap[0],
		       _status_str.mask[0]);
	gtk_widget_show (_status_str.prt_pix);
	
	_status_str.window = window;
	_status_str.cur_status = STS_DEFAULT;
	_status_str.cur_pict = 0;
	_status_str.num_pict = 1;
	_status_str.stm_st = 0;
	_status_str.timer = gtk_timeout_add(1000, GetPrinterStatus, NULL);
	return;
}

/* end of status monitor */
void
end_stm (void)
{
	int i;
  
	/* PIXMAPλȤ򳰤 */
	for (i = 0; i < _status_str.num_pict; i++)
		gdk_pixmap_unref (_status_str.pixmap[i]);
  
	/* ޡκ */
	gtk_timeout_remove(_status_str.timer);
  
	/* close socket */
	sock_close ();
  
	gtk_main_quit();
}

/* cancel process */
void
set_cancel_status (void)
{
	GtkWidget* widget;

	/* "error return" being anonymous */
	/* 
	   if (_status_str.stm_st == 1)
	   _status_str.stm_st = 2;
	*/
	widget = lookup_widget (_status_str.window, "cancel_button");
	gtk_widget_set_sensitive (widget, FALSE);
	return;
}
