/*
 * Photo Image Print System
 * Copyright (C) 2001-2004 EPSON KOWA Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2001-2004.
 *
 * This file is part of the `ekpstm' program.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "inkbox.h"

#define INKBOX_XPM_WIDTH 38
#define INKBOX_XPM_HEIGHT 75

#define INK_COLOR_PALETTE 6
#define INK_MAX 19
#define INK_MIN 72

#define SIDE_OFS 30
#define SIDE_WIDTH 5
#define SIDE_START 2
#define SIDE_END 6

/*   +-----1
 *  /     /|
 * +-----2 |
 * |     | |
 * |     | |
 * |     | |
 * |     | |
 * |     | |
 * |     | |
 * |     | 3
 * |     |/
 * 4-----+
 *
 * 1 : ( 34 , 13 )
 * 2 : ( 28 , 19 )
 * 3 : ( 34 , 71 )
 * 4 : (  6 , 65 )
 */

static char ink_table[][12] = {
	/*  K */ "X c #000000",
	/*  C */ "X c #00FFFF",
	/*  M */ "X c #FF00FF",
	/*  Y */ "X c #FFFF00",
	/* LC */ "X c #7FFFFF",
	/* LM */ "X c #FF7FFF",
	/* DY */ "X c #CFCF00",
	/* LK */ "X c #7F7F7F",
	/* MK */ "X c #000000"
};

static char * inkbox_xpm[] = {
	"37 68 6 1",
	"  c None",
	". c #000000",
	"+ c #D0CFC0",
	"@ c #FFFFFF",
	"# c #7F7F7F",
	"X c #FFFFFF",
	"          .......................... ",
	"         .++++++++++++++++++++++++@.#",
	"        .++++++++++++++++++++++++@#.#",
	"       .++++++++++++++++++++++++@##.#",
	"      .++++++++++++++++++++++++@##..#",
	"     .++++++++++++++++++++++++@##.#.#",
	"    .@@@@@@@@@@@@@@@@@@@@@@@@@##.#+.#",
	"    .++++++++++++++++++++++++@#.#++.#",
	"    .++++++++++++++++++++++++@.#+++.#",
	"... ..........................#++++.#",
	"    .#########################+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"... .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"... .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"... .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++++.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@++++#.#",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+++#.##",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@++#.## ",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@+#.##  ",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@#.##   ",
	"    .#@@@@@@@@@@@@@@@@@@@@@@@@.##    ",
	"     .#######################.##     ",
	"...   .......................##      ",
	"       #######################       "
};

char **
inkbox_xpm_new (void)
{
	char ** xpm_buf;
	int i;

	xpm_buf = calloc (INKBOX_XPM_HEIGHT, sizeof (char *));
	if (xpm_buf == NULL) return NULL;

	for (i = 0; i < INKBOX_XPM_HEIGHT; i ++)
	{
		xpm_buf[i] = calloc (INKBOX_XPM_WIDTH, sizeof (char));
		if (xpm_buf[i] == NULL) return NULL;

		strcpy (xpm_buf[i], inkbox_xpm[i]);
	}

	return xpm_buf;
}


void
inkbox_xpm_ref (char ** xpm_buf)
{
	int i;

	if (xpm_buf == NULL) return;

	for (i = 0; i < INKBOX_XPM_HEIGHT; i ++)
	{
		if (xpm_buf[i] != NULL)
			free (xpm_buf[i]);
	}
	free (xpm_buf);

	return;
}


void
inkbox_chenge_volume (char * xpm_buf[], unsigned long id, int value)
{

	int start;
	int i;
	enum Inkbox_Id color;

	assert (xpm_buf != NULL);
	assert (value >= 0 && value <= 100);

	switch (id) {
	case 0x1101:
	case 0x1140:
		color = INKBOX_K; break;
	case 0x3202:
		color = INKBOX_C; break;
	case 0x4304:
		color = INKBOX_M; break;
	case 0x5408:
		color = INKBOX_Y; break;
	case 0x6210:
		color = INKBOX_LC; break;
	case 0x7320:
		color = INKBOX_LM; break;
	case 0x9440:
		color = INKBOX_DY; break;
	case 0xA140:
		color = INKBOX_LK; break;
	case 0xB140:
	case 0xB101:
		color = INKBOX_MK; break;
	default:
		return;
	}

	/* set color */
	strcpy (xpm_buf[INK_COLOR_PALETTE], ink_table[color]);

	start = INK_MIN - (value / 2) - 2;

	for (i = INK_MAX - 6; i < INK_MIN; i++)
	{
		int side_ofs, side_num;
		int ink_ofs;
		char *line;

		line = xpm_buf[i];

		if (i >= INK_MAX)
		{
			int front;
			
			if (i < start)
				front = '@';
			else if (i == start)
				front = '+';
			else
				front = 'X';
			
			memset (line + 6, front, 23);
		}

		if (i < INK_MAX - SIDE_START)
		{
			side_ofs = INK_MAX - i - SIDE_START;
			side_num = SIDE_WIDTH - side_ofs;
		}

		else if (i >= INK_MIN - SIDE_END)
		{
			side_ofs = 0;
			side_num = INK_MIN - i - SIDE_START + 1;
		}

		else
		{
			side_ofs = 0;
			side_num = 5;
		}

		if (i < start - SIDE_END)
		{
			memset (line + SIDE_OFS + side_ofs, '+', side_num);
		}

		else if (i >= start - SIDE_START)
		{
			memset (line + SIDE_OFS + side_ofs, '#', side_num);
		}

		else
		{
			ink_ofs = start - i - SIDE_START;
			memset (line + SIDE_OFS + side_ofs, '+', ink_ofs - side_ofs);
			memset (line + SIDE_OFS + ink_ofs, '#', side_num - (ink_ofs - side_ofs));
		}
	}
	return;
}

void
inkbox_chenge_table (int inkset)
{
	if (inkset == INKSET_CMYKcmDY)
		strncpy (ink_table[INKBOX_DY], "X c #CFCF00", 12);

	else if (inkset == INKSET_CMYKcmLk)
		strncpy (ink_table[INKBOX_DY], "X c #000000", 12);

	return;
}

/*
int
main (int argc, char *argv[])
{
	int i;
	char ** box;

	assert (argc > 1);

	box = inkbox_xpm_new ();
	assert (box != NULL);

	inkbox_chenge_volume (box, atoi (argv[1]), INKBOX_K);
	
	for (i = 0; i < INKBOX_XPM_HEIGHT; i++)
	{
		printf ("%s\n", box[i]);
	}

	inkbox_xpm_ref (box);
	return 0;
}

*/
