#include <ncurses.h>
#include <stdlib.h>
#include <signal.h>
#include <time.h>
#include <ctype.h>
#include <unistd.h>
#include <string.h>

#define GAME_VERSION "v0.03"
#define GAME_NAME "plonx"

#define DEF_MAPX 20
#define DEF_MAPY 20
#define DEF_LVL  10
#define DEF_DELAY 50000

#define MAP_PLAYER	 0
#define MAP_BOMB	-1
#define MAP_VISITED	-2

#define END_QUIT	1
#define END_BOMB	2
#define END_TRAIL	3
#define END_EDGE	4
#define END_PLAYER	5

#define CLR_PLAYER	COLOR_PAIR(COLOR_WHITE)	| A_BOLD
#define CLR_VISITED	COLOR_PAIR(COLOR_BLUE)	| A_NORMAL
#define CLR_BOMB	COLOR_PAIR(COLOR_RED)	| A_BOLD
#define CLR_NUMBER	COLOR_PAIR(COLOR_GREEN)	| A_NORMAL
#define CLR_TEXT	COLOR_PAIR(COLOR_WHITE)	| A_DIM
#define CLR_ATTENTION	COLOR_PAIR(COLOR_WHITE)	| A_BOLD

#define CHR_PLAYER	'@'
#define CHR_VISITED	'#'
#define CHR_BOMB	'*'

#define PKEY_QUIT	'q'
#define PKEY_HELP	'?'
#define PKEY_N		'k'
#define PKEY_NE		'u'
#define PKEY_E		'l'
#define PKEY_SE		'n'
#define PKEY_S		'j'
#define PKEY_SW		'b'
#define PKEY_W		'h'
#define PKEY_NW		'y'

struct mapdata {
  int mapx, mapy, map_level;
  int *map;
};

struct playerdata {
  int player_x, player_y;
  int player_score;
};

unsigned int screen_delay = DEF_DELAY;


void
curses_start(void)
{
  initscr();
  cbreak();
  noecho();
  keypad(stdscr, TRUE);
  curs_set(0);
  start_color();

  init_pair(COLOR_BLACK, COLOR_BLACK, COLOR_BLACK);
  init_pair(COLOR_RED, COLOR_RED, COLOR_BLACK);
  init_pair(COLOR_GREEN, COLOR_GREEN, COLOR_BLACK);
  init_pair(COLOR_YELLOW, COLOR_YELLOW, COLOR_BLACK);
  init_pair(COLOR_BLUE, COLOR_BLUE, COLOR_BLACK);
  init_pair(COLOR_MAGENTA, COLOR_MAGENTA, COLOR_BLACK);
  init_pair(COLOR_CYAN, COLOR_CYAN, COLOR_BLACK);
  init_pair(COLOR_WHITE, COLOR_WHITE, COLOR_BLACK);

  refresh();

  flushinp();
}


void
curses_stop(void)
{
  erase();
  refresh();
  echo();
  nocbreak();
  endwin();
}


void
bail(char *message)
{
  curses_stop();
  fprintf(stderr, "%s\n", message);
  exit(EXIT_FAILURE);
}


void
sigint_handler()
{
  curses_stop();
  bail("Whee!");
}


void
sigwinch_handler()
{
  endwin();
  curses_start();
  return;
}

#define map_put(x,y,ch,m) m->map[y*m->mapx + x] = ch
#define map_get(x,y,m) m->map[y*m->mapx + x]

void
map_init(int level, int width, int height, struct mapdata *m, struct playerdata *plr)
{
  int x, y, d;

  if (width < 10) width = 10;
  if (width > (COLS - 40)) width = (COLS - 40);

  if (height < 10) height = 10;
  if (height >= LINES) height = LINES - 1;

  m->mapx = width;
  m->mapy = height;

  m->map = (int *)malloc(sizeof(int) * width * height);

  if (!m->map) bail("Error: could not get mem for map");

  if (level < 0) level = 0;
  else if (level > (m->mapx*m->mapy)/10) level = (m->mapx*m->mapy)/10;

  m->map_level = level;

  for (x = 0; x < m->mapx; x++)
    for (y = 0; y < m->mapy; y++)
      map_put(x,y, (rand() % 6) + 1, m);

  for (d = 0; d < level; d++) {
    do {
      x = rand() % m->mapx;
      y = rand() % m->mapy;
    } while (map_get(x,y,m) == MAP_BOMB);
    map_put(x,y, MAP_BOMB, m);
  }

  do {
    x = rand() % m->mapx;
    y = rand() % m->mapy;
  } while (map_get(x,y,m) == MAP_BOMB);
  map_put(x,y, MAP_PLAYER,m);

  plr->player_x = x;
  plr->player_y = y;
}


void
map_done(struct mapdata *m)
{
  free(m->map);
  m->mapx = m->mapy = 0;
}


void
map_print(struct mapdata *m)
{
  int x,y;
  chtype ch;

  for (x = 0; x < m->mapx; x++)
    for (y = 0; y < m->mapy; y++) {
      switch (map_get(x,y,m)) {
      case MAP_BOMB:	ch = CHR_BOMB | CLR_BOMB; break;
      case MAP_VISITED:	ch = CHR_VISITED | CLR_VISITED; break;
      case MAP_PLAYER:	ch = CHR_PLAYER | CLR_PLAYER; break;
      default: ch = ('0' + (map_get(x,y,m) % 10)) | CLR_NUMBER;
      }
      mvaddch(y + 1, x, ch);
    }
}

void
print_clear(struct mapdata *map)
{
  int x;
  int y;

  attrset(CLR_TEXT);
  for (y = 0; y < LINES; y++)
    for (x = map->mapx + 3; x < COLS; x++)
      mvprintw(y, x, " ");
}

void
print_help(struct mapdata *map)
{
  int x = map->mapx + 3;

  print_clear(map);

  attrset(CLR_TEXT);
  mvprintw(1, x, "This is you:");
  attrset(CLR_PLAYER);
  mvprintw(1, x + 17, "%c", CHR_PLAYER);
  attrset(CLR_TEXT);
  mvprintw(3, x, "This is a bomb:");
  attrset(CLR_BOMB);
  mvprintw(3, x + 17, "%c", CHR_BOMB);

  attrset(CLR_TEXT);
  mvprintw(5, x, "Do not hit bombs, or the upper");
  mvprintw(6, x, "or lower edges of the map.");
  mvprintw(7, x, "Left and right edges wrap around.");

  mvprintw(9, x, "You move by using");

  mvprintw(11, x, "7 8 9        %c %c %c", PKEY_NW, PKEY_N, PKEY_NE);
  mvprintw(13, x, "4   6   or   %c   %c", PKEY_W, PKEY_E);
  mvprintw(15, x, "1 2 3        %c %c %c", PKEY_SW, PKEY_S, PKEY_SE);

  mvprintw(17, x, "You move ");
  attrset(CLR_NUMBER);
  printw("1");
  attrset(CLR_TEXT);
  printw("-");
  attrset(CLR_NUMBER);
  printw("6");
  attrset(CLR_TEXT);
  printw(" spaces forward,");

  mvprintw(18, x, "depending on what number you");
  mvprintw(19, x, "first pick up on that move.");

  mvprintw(21, x, "To quit, press '%c'", PKEY_QUIT);

  getch();

  print_clear(map);
}

void
print_screen(struct mapdata *map, struct playerdata *plr)
{
  attrset(CLR_TEXT);
  mvprintw(0, map->mapx + 3, "%s %s", GAME_NAME, GAME_VERSION);

  mvprintw(2, map->mapx + 3, "Map: (%i,%i), Level: %i", map->mapx, map->mapy, map->map_level);

  mvprintw(7, map->mapx + 3, "Score: %i", plr->player_score);


  map_print(map);
}


int
move_player(int dir, struct mapdata *map, struct playerdata *plr)
{
  int dirx[9] = { 0, 0, 1, 1, 1, 0,-1,-1,-1};
  int diry[9] = { 0,-1,-1, 0, 1, 1, 1, 0,-1};
  int x, y;
  int dist;

  if (!dir) return 0;

  dir = dir % 9;

  map_put(plr->player_x,plr->player_y, MAP_VISITED, map);

  x = plr->player_x + dirx[dir];
  y = plr->player_y + diry[dir];

  if (y < 0 || y >= map->mapy) return END_EDGE;

  if (x < 0) x += map->mapx;
  else if (x >= map->mapx) x = x % (map->mapx);

  dist = map_get(x,y,map);

  if (dist > MAP_PLAYER) {
    dist = (dist % 10);
    plr->player_score += dist;
    dist--;
    while (dist > 0) {
      map_put(x,y, MAP_VISITED,map);

      x += dirx[dir];
      y += diry[dir];

      if (y < 0 || y >= map->mapy) return END_EDGE;

      if (x < 0) x += map->mapx;
      else if (x >= map->mapx) x = x % (map->mapx);

      if (map_get(x,y,map) == MAP_BOMB) return END_BOMB;
      else if (map_get(x,y,map) == MAP_VISITED) return END_TRAIL;
      else if (map_get(x,y,map) == MAP_PLAYER) return END_PLAYER;

      dist--;
      map_put(x,y, MAP_PLAYER,map);

      if (screen_delay > 0) {
	print_screen(map, plr);
	refresh();
	usleep(screen_delay);
      }
    }
    plr->player_x = x;
    plr->player_y = y;
  } else
    if (map_get(x,y,map) == MAP_BOMB) return END_BOMB;
    else if (map_get(x,y,map) == MAP_VISITED) return END_TRAIL;
    else if (map_get(x,y,map) == MAP_PLAYER) return END_PLAYER;

  map_put(plr->player_x,plr->player_y, MAP_PLAYER,map);

  return 0;
}


void
show_end(int end, struct mapdata *map)
{
  char *s;

  switch (end) {
  case END_QUIT: s = "quit"; break;
  case END_BOMB: s = "hit a bomb"; break;
  case END_TRAIL: s = "hit your trail"; break;
  case END_EDGE: s = "hit the edge of the board"; break;
  case END_PLAYER: s = "hit another player"; break;
  default: s = "What!?";
  }

  attrset(CLR_ATTENTION);
  mvprintw(10, map->mapx + 3, "You %s!", s);

  attrset(CLR_TEXT);
  mvprintw(12, map->mapx + 3, "Press any key to exit");

  getch();
}


void
main_loop(struct mapdata *map, struct playerdata *plr)
{
  int input;
  int quit = 0;
  int dir;

  plr->player_score = 0;

  while (quit == 0) {

    dir = 0;

    print_screen(map, plr);
    input = getch();
    flushinp();

    input = tolower(input);

    switch(input) {
    case PKEY_HELP: print_help(map); break;
    case PKEY_QUIT: quit = END_QUIT; break;
    case PKEY_N:  case '8': dir = 1; break;
    case PKEY_NE: case '9': dir = 2; break;
    case PKEY_E:  case '6': dir = 3; break;
    case PKEY_SE: case '3': dir = 4; break;
    case PKEY_S:  case '2': dir = 5; break;
    case PKEY_SW: case '1': dir = 6; break;
    case PKEY_W:  case '4': dir = 7; break;
    case PKEY_NW: case '7': dir = 8; break;
    default: dir = 0; break;
    }

    if (dir)
      quit = move_player(dir, map, plr);
  }

  print_screen(map, plr);
  show_end(quit, map);
}


int
main(int argc, char **argv)
{
  int wid = DEF_MAPX;
  int hei = DEF_MAPY;
  int lvl = DEF_LVL;
  int seed = 0;
  int use_seed = 0;
  struct playerdata plrd;
  struct mapdata map;

  printf("%s %s\n", GAME_NAME, GAME_VERSION);

  if (argc > 1) {
    int n = 1;

    while (n < argc) {
      char *s = argv[n++];

      while (*s == '-') s++;

      if (!strcmp(s, "wid") && (n < argc)) {
	wid = atoi(argv[n++]);
      } else
      if (!strcmp(s, "hei") && (n < argc)) {
	hei = atoi(argv[n++]);
      } else
      if (!strcmp(s, "level") && (n < argc)) {
	lvl = atoi(argv[n++]);
      } else
      if (!strcmp(s, "seed") && (n < argc)) {
	seed = atoi(argv[n++]);
	use_seed = 1;
      } else
      if (!strcmp(s, "delay") && (n < argc)) {
	screen_delay = atoi(argv[n++]);
      } else
	if (!strcmp(s, "h") || !strcmp(s, "help")) {
	  printf("%s [--wid X] [--hei Y] [--level N] [--seed N] [--delay N]\n", argv[0]);
	  return 0;
      }
    }
  }

  curses_start();

  if (COLS < 80 || LINES < 25)
    bail("Error: your terminal size must be at least 80x25");

  signal(SIGINT, sigint_handler);
  signal(SIGWINCH, sigwinch_handler);

  if (use_seed) {
    srand(seed);
  } else {
    srand(time(0));
  }

  map_init(lvl, wid, hei, &map, &plrd);
  main_loop(&map, &plrd);
  map_done(&map);

  curses_stop();

  printf("Your score: %i\n", plrd.player_score);

  return 0;
}
