/***************************************************************************
                          fam.cpp  -  fam-based monitoring
                             -------------------
    begin                : do jan 9 2003
    copyright            : (C) 2003 by Meilof
    email                : meilof@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <zones.h>
#include <poslib/poslib.h>
#include <primary.h>
#include <masterfile.h>
#include <configuration.h>
#include <lib.h>
#include <posadisrc.h>
#include <zonefile.h>

#include "fam_w32.h"
#include <stdio.h>

extern "C" bool /*fam_LTX_*/lib_feed_setting(const char *opt, const char *values) {
  if (strcmpi(opt, "fam_directory") == 0) {
    pos_log(context_conf, log_error, "The fam_directory command is no longer supported and will be ignored. Use configdir instead.");
    return true;
  }
  return false;
}

void *fam_thread(void *arg);

extern "C" void /*fam_LTX_*/lib_end_setting() {
  pthread_t tr;
  if (!confdir[0]) throw PException("monitor: no configuration directory given");
  posthread_create(&tr, fam_thread, confdir);
}


void *fam_thread(void *dir) {
  pos_log(context_server, log_info, "monitor: starting monitoring on %s", (char *)dir);

  FAMConnection fam;
  FAMRequest req;
  FAMEvent ev;
  struct timeval tv;

  int ret;
#ifndef _WIN32
  fd_set famset;
  int fd;
#endif

  char fname[PATH_MAX];
  domainname dom, retdom;
  Zone *z;

  ret = FAMOpen2(&fam, "Posadis monitor module");
  if (ret != 0) {
    pos_log(context_server, log_error, "monitor: could not connect to FAM server: %d:%d", ret, FAMErrno);
    return NULL;
  }

  ret = FAMMonitorDirectory(&fam, (char *)dir, &req, NULL);
  if (ret != 0) {
    pos_log(context_server, log_error, "monitor: could not monitor %s", (char *)dir);
    FAMClose(&fam);
    return NULL;
  }
  
#ifndef _WIN32
  fd = FAMCONNECTION_GETFD(&fam);
#endif

  while(1) {
#ifndef _WIN32
    FD_ZERO(&famset);
    FD_SET(fd, &famset);
#endif
    tv.tv_sec = 1;
    tv.tv_usec = 0;
    if (pos_quitting() || (ret=FAMselect(&fam, &req, fd + 1, &famset, &tv)) < 0) {
      pos_log(context_server, log_info, "monitor: closing down");
      FAMCancelMonitor(&fam, &req);
      FAMClose(&fam);
      return NULL;
    }
    if (FAM_FD_ISSET(&fam, fd, &famset)) {
      do {
        if (FAMNextEvent(&fam, &ev) < 0) {
          pos_log(context_server, log_error, "monitor: call to FAMNextEvent() failed");
          FAMCancelMonitor(&fam, &req);
          FAMClose(&fam);
          return NULL;
        }

        if (strlen(ev.filename) + strlen((char *)dir) + 1 > PATH_MAX) {
          pos_log(context_server, log_error, "monitor: path too long for file %s", ev.filename);
          continue;
        }

        strcpy(fname, (char *)dir);
        ret = strlen(fname) - 1;
        if (fname[ret] != SEP) { fname[ret + 1] = SEP; fname[ret + 2] = '\0'; }
        strcat(fname, ev.filename);

        if (is_masterfile(ev.filename, dom)) {
          pthread_mutex_lock(&m_auth_zones);
          try {
            if (ev.code == FAMDeleted) {
              /* file deleted! */
              try {
                remove_base_zone(dom, (BaseZone **)&auth_root_zone);
              } catch (PException p) { }
              pos_log(context_server, log_info, "monitor: master file %s removed", ev.filename);
            } else if (ev.code == FAMCreated || ev.code == FAMExists) {
              z = new PrimaryZone();
              add_authoritative_zone(dom, z);
              z->feed_setting("file", fname);
              z->end_setting();
            } else if (ev.code == FAMChanged) {
              z = lookup_authoritative_zone(dom, &retdom, false);
              if (dom != retdom || z->type != Z_PRIMARY) {
                throw PException(true, "master file %s does not correspond to primary zone", ev.filename);
              }
              delete ((PrimaryZone *)z)->rootdomain;
              ((PrimaryZone *)z)->rootdomain = new ZoneDomain(NULL, "");
              z->feed_setting("reset", "");
              z->feed_setting("file", fname);
              z->end_setting();
            }
          } catch (PException p) {
            pos_log(context_server, log_error, "monitor: error updating file info for %s: %s", ev.filename, p.message);
          }
          pthread_mutex_unlock(&m_auth_zones);
        } else if (is_zonefile(ev.filename, dom)) {
          pthread_mutex_lock(&m_auth_zones);
          try {
            if (ev.code == FAMDeleted) {
              /* file deleted! */
              try {
                remove_base_zone(dom, (BaseZone **)&auth_root_zone);
              } catch (PException p) { }
              pos_log(context_server, log_info, "monitor: zone file %s removed", ev.filename);
            } else if (ev.code == FAMCreated || ev.code == FAMExists) {
              load_zone_file(dom, fname, true);
              pos_log(context_server, log_info, "monitor: zone file %s loaded", ev.filename);
            } else if (ev.code == FAMChanged) {
              try { remove_base_zone(dom, (BaseZone **)&auth_root_zone); } catch (PException p) { }
              load_zone_file(dom, fname, true);
              pos_log(context_server, log_info, "monitor: zone file %s reloaded", ev.filename);
            }
          } catch (PException p) {
            pos_log(context_server, log_error, "monitor: error updating file info for %s: %s", ev.filename, p.message);
          }
          pthread_mutex_unlock(&m_auth_zones);

        }
      } while (FAMPending(&fam) == 1);
    }
  }

  FAMCancelMonitor(&fam, &req);
  FAMClose(&fam);
  return NULL;
}
