/*
 * fam_w32.cpp - FAM implementation for Win32
 * Copyright (C) 2003 Meilof Veeningen <meilof@myrealbox.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#include <windows.h>
#include <io.h>
#include <poslib/poslib.h>
#include "fam_w32.h"

int FAMErrno = 0;

bool file_info::operator<(file_info& inf) {
  return (strcmpi(fname.c_str(), inf.fname.c_str())==-1);
}

FAMEvent::FAMEvent() {
  filename = NULL;
}

FAMEvent::~FAMEvent() {
  if (filename) delete filename;
}

FAMEvent::FAMEvent(const FAMEvent& ev) {
  filename = strdup(ev.filename);
  code = ev.code;
}

FAMEvent& FAMEvent::operator=(FAMEvent& ev) {
  filename = strdup(ev.filename);
  code = ev.code;
  return *this;
}

int FAMOpen2(FAMConnection *conn, const char *appname) {
  return 0;
}

bool build_filelist(const char *dir, stl_list(file_info)& files) {
  struct _finddata_t data;
  file_info inf;
  int handle;
  if ((handle = _findfirst(dir, &data)) == -1) return false;

  do {
    inf.fname = data.name;
    inf.writetime = data.time_write;
    inf.used = true;
    files.push_back(inf);
  } while (_findnext(handle, &data) == 0);
  files.sort();

  _findclose(handle);
  return true;
}

int FAMMonitorDirectory(FAMConnection *conn, const char *dir, FAMRequest *req,
                        void *nulldata) {
  req->dirname = dir;
  if (req->dirname[req->dirname.size() - 1] == '\\')
    req->dirname.resize(req->dirname.size() - 1);
    
  /* build initial file list */
  if (!build_filelist((req->dirname + "\\*.*").c_str(), req->filelist)) {
    FAMErrno = 1;
    return -1;
  }
  
  stl_list(file_info)::iterator it = req->filelist.begin();
  FAMEvent ev;
  while (it != req->filelist.end()) {
    if (ev.filename) free((char*)ev.filename);
    ev.filename = strdup(it->fname.c_str());
    ev.code = FAMExists;
    conn->pending_events.push_back(ev);
    it++;
  }

  /* create handle */
  req->changehandle = FindFirstChangeNotification(req->dirname.c_str(), FALSE,
    FILE_NOTIFY_CHANGE_FILE_NAME | FILE_NOTIFY_CHANGE_LAST_WRITE);
  if (req->changehandle == INVALID_HANDLE_VALUE) {
    FAMErrno = 2;
    return -1;
  }                     
    
  return 0;
}

void FAMClose(FAMConnection *conn) {
}

void FAMCancelMonitor(FAMConnection *conn, FAMRequest *req) {
  FindCloseChangeNotification(req->changehandle);
}

bool find_changes(stl_string dirname, FAMConnection *conn,
                  stl_list(file_info)& filelist) {
  FAMEvent ev;
  stl_list(file_info) newfiles;
  stl_list(file_info)::iterator oldit, newit;
  if (!build_filelist((dirname + "\\*.*").c_str(), newfiles)) return false;
  oldit = filelist.begin();
  newit = newfiles.begin();
  bool haveold, havenew;
  while (oldit != filelist.end() || newit != newfiles.end()) {
    haveold = (oldit != filelist.end()); havenew = (newit != newfiles.end());
    if (haveold && havenew && strcmpi(oldit->fname.c_str(), newit->fname.c_str()) == 0) {
      if (oldit->writetime != newit->writetime) {
        if (ev.filename) free((char*)ev.filename);
        ev.filename = strdup(newit->fname.c_str()); 
        ev.code = FAMChanged;
        conn->pending_events.push_back(ev);
      }
      oldit++;
      newit++;
    } else if (haveold && (!havenew || *oldit < *newit)) {
      if (ev.filename) free((char*)ev.filename);
      ev.filename = strdup(oldit->fname.c_str());
      ev.code = FAMDeleted;
      conn->pending_events.push_back(ev);
      oldit++;
    } else if (havenew && (!haveold || *newit < *oldit)) {
      if (ev.filename) free((char*)ev.filename);
      ev.filename = strdup(newit->fname.c_str());
      ev.code = FAMCreated;
      conn->pending_events.push_back(ev);
      newit++;
    } else {
      break;
    }
  }
  filelist = newfiles;
  return true;
}


int FAMWait(FAMConnection *conn, FAMRequest *req, struct timeval *tv) {
  switch (WaitForSingleObject(req->changehandle,
                              tv->tv_usec / 1000 + tv->tv_sec * 1000)) {
    case WAIT_TIMEOUT:
      return 0;
    case WAIT_OBJECT_0:
      /* find out what's changed */
      if (!find_changes(req->dirname, conn, req->filelist)) {
        FAMErrno = 3;
        return -1;
      }
      FindNextChangeNotification(req->changehandle);
      return 1;
    }
}

bool FAMIsData(FAMConnection *conn) {
  if (conn->pending_events.size()) return true; else return false;
}

int FAMNextEvent(FAMConnection *conn, FAMEvent *event) {
  *event = *(conn->pending_events.begin());
  conn->pending_events.pop_front();
}

int FAMPending(FAMConnection *conn) {
  if (conn->pending_events.size()) return 1; else return 0;
}


