"""
This module contains an C{L{OpenIDStore}} implementation with no
persistent backing, for use only by limited consumers.
"""

from openid import cryptutil
from openid.store.interface import OpenIDStore

class DumbStore(OpenIDStore):
    """
    This is a store for use in the worst case, when you have no way of
    saving state on the consumer site.  Using this store makes the
    consumer vulnerable to replay attacks (though only within the
    lifespan of the tokens), as it's unable to use nonces.  Avoid
    using this store if it is at all possible.

    Most of the methods of this class are implementation details.
    Users of this class need to worry only about the C{L{__init__}}
    method.

    @sort: __init__
    """
    def __init__(self, secret_phrase):
        """
        Creates a new DumbStore instance.  For the security of the
        tokens generated by the library, this class attempts to at
        least have a secure implementation of C{L{getAuthKey}}.

        When you create an instance of this class, pass in a secret
        phrase.  The phrase is hashed with sha1 to make it the correct
        length and form for an auth key.  That allows you to use a
        long string as the secret phrase, which means you can make it
        very difficult to guess.

        Each C{L{DumbStore}} instance that is created for use by your
        consumer site needs to use the same C{secret_phrase}.

        @param secret_phrase: The phrase used to create the auth key
            returned by C{L{getAuthKey}}

        @type secret_phrase: C{str}
        """
        self.auth_key = cryptutil.sha1(secret_phrase)

    def storeAssociation(self, server_url, association):
        """
        This implementation does nothing.
        """
        pass

    def getAssociation(self, server_url, handle=None):
        """
        This implementation always returns C{None}.


        @return: C{None}

        @rtype: C{None}
        """
        return None

    def removeAssociation(self, server_url, handle):
        """
        This implementation always returns C{False}.


        @return: C{False}

        @rtype: C{bool}
        """
        return False

    def storeNonce(self, nonce):
        """
        This implementation does nothing.
        """
        pass

    def useNonce(self, nonce):
        """
        In a system truly limited to dumb mode, nonces must all be
        accepted.  This therefore always returns C{True}, which makes
        replay attacks feasible during the lifespan of the token.


        @return: C{True}

        @rtype: C{bool}
        """
        return True

    def getAuthKey(self):
        """
        This method returns the auth key generated by the constructor.


        @return: The auth key generated by the constructor.

        @rtype: C{str}
        """
        return self.auth_key

    def isDumb(self):
        """
        This store is a dumb mode store, so this method is overridden
        to return C{True}.


        @return: C{True}

        @rtype: C{bool}
        """
        return True
