
/* except.q: exception handling for the builtin and standard library functions
   07-26-01, 03-04-02, 02-18-03, 03-18-03 AG */

/* This file is part of the Q programming system.

   The Q programming system is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   The Q programming system is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

/* This script defines sample default rules for all the builtin and standard
   library operations, which throw an exception whenever such an operation
   "fails". The exception handling implemented here is fairly simplistic:
   just throw back the offending expression protected with a special
   constructor. There are different constructors, belonging to different
   Exception subtypes, for the builtins and each part of the standard library,
   such as builtin_err of type BuiltinException for the builtins, stdlib_err
   of type StdlibException for the operations in stdlib.q, etc.

   There also is a simple handler function which can be supplied as the first
   argument to `catch'. When invoked with an exception, i.e., a value of type
   Exception, it prints an appropriate error message and halts. Otherwise it
   just returns the argument value unchanged.

   NOTE: These definitions are only examples, which should be adapted for the
   particular application. Note that Q is essentially an exception-free
   language because of its rewriting semantics. Thus if you want to report
   exceptions for certain operations you *always* have to add corresponding
   rules yourself. Judicious use of exception rules, in particular for
   operations involving side-effects can help to locate certain bugs in a
   script which otherwise are hard to find. But adding exceptions to each and
   every operation in a program generally is a bad idea because it interferes
   with symbolic evaluation. Therefore exception rules are *not* included in
   the standard library by default.

   EXAMPLES: Default error message printed by the interpreter in response to
   an unhandled exception:

   ==> 17+4*3/0
   ! Exception
   builtin_err (12/0)
   >>> 17+4*3/0
               ^

   Use our own exception handler:

   ==> catch exception (17+4*3/0)
   ! Exception: builtin_err (12/0)

   You can also handle exceptions in the Q machine. E.g., let's force a stack
   overflow:

   ==> catch exception (iter 100000000 (+) 1)
   ! Exception: syserr 5

   Note that 5 is the error code for an expression stack overflow; see the
   manual for the list of all built-in exception codes.  */

public exception X;

exception X:Exception	= fwrites ERROR ("! Exception: "++str X++"\n") ||
			  halt;
exception X		= X otherwise;

/* The following rules are at the lowest possible priority level so as not to
   interfere with the standard library and user equations. */

@-0x80000000

/* Note that some operations are missing in this list, because they never fail
   unless a hard error condition occurs. */

/* builtins ******************************************************************/

public type BuiltinException : Exception = special const builtin_err X;

/* operators */

X^Y			= throw (builtin_err (X^Y));
X!Y			= throw (builtin_err (X!Y));

-X			= throw (builtin_err (-X));
#X			= throw (builtin_err (#X));
not X			= throw (builtin_err (not X));

X*Y			= throw (builtin_err (X*Y));
X/Y			= throw (builtin_err (X/Y));
X div Y			= throw (builtin_err (X div Y));
X mod Y			= throw (builtin_err (X mod Y));
X and Y			= throw (builtin_err (X and Y));
X and then Y		= throw (builtin_err (X and then Y));

X++Y			= throw (builtin_err (X++Y));
X+Y			= throw (builtin_err (X+Y));
X-Y			= throw (builtin_err (X-Y));
X or Y			= throw (builtin_err (X or Y));
X or else Y		= throw (builtin_err (X or else Y));

X<Y			= throw (builtin_err (X<Y));
X>Y			= throw (builtin_err (X>Y));
(X=Y)			= throw (builtin_err (X=Y));
X<=Y			= throw (builtin_err (X<=Y));
X>=Y			= throw (builtin_err (X>=Y));
X<>Y			= throw (builtin_err (X<>Y));

/* functions */

succ X			= throw (builtin_err (succ X));
pred X			= throw (builtin_err (pred X));

shl N COUNT		= throw (builtin_err (shl N COUNT));
shr N COUNT		= throw (builtin_err (shr N COUNT));

exp X			= throw (builtin_err (exp X));
ln X			= throw (builtin_err (ln X));
sqrt X			= throw (builtin_err (sqrt X));
sin X			= throw (builtin_err (sin X));
cos X			= throw (builtin_err (cos X));
atan X			= throw (builtin_err (atan X));
atan2 X Y		= throw (builtin_err (atan2 X Y));
seed N			= throw (builtin_err (seed N));


sub Xs I J		= throw (builtin_err (sub Xs I J));
substr S K L		= throw (builtin_err (substr S K L));
pos S1 S		= throw (builtin_err (pos S1 S));

trunc X			= throw (builtin_err (trunc X));
round X			= throw (builtin_err (round X));
float X			= throw (builtin_err (float X));
int X			= throw (builtin_err (int X));
frac X			= throw (builtin_err (frac X));
ord X			= throw (builtin_err (ord X));
chr N			= throw (builtin_err (chr N));
list Xs			= throw (builtin_err (list Xs));
tuple Xs		= throw (builtin_err (tuple Xs));
str X			= throw (builtin_err (str X));
strq X			= throw (builtin_err (strq X));
val S			= throw (builtin_err (val S));
valq S			= throw (builtin_err (valq S));

fopen NAME MODE		= throw (builtin_err (fopen NAME MODE));
popen CMD MODE		= throw (builtin_err (popen CMD MODE));
fclose F		= throw (builtin_err (fclose F));
read			= throw (builtin_err read);
readq			= throw (builtin_err readq);
readc			= throw (builtin_err readc);
reads			= throw (builtin_err reads);
fread F			= throw (builtin_err (fread F));
freadq F		= throw (builtin_err (freadq F));
freadc F		= throw (builtin_err (freadc F));
freads F		= throw (builtin_err (freads F));
write X			= throw (builtin_err (write X));
writeq X		= throw (builtin_err (writeq X));
writec X		= throw (builtin_err (writec X));
writes S		= throw (builtin_err (writes S));
fwrite F X		= throw (builtin_err (fwrite F X));
fwriteq F X		= throw (builtin_err (fwriteq F X));
fwritec F X		= throw (builtin_err (fwritec F X));
fwrites F S		= throw (builtin_err (fwrites F S));
eof			= throw (builtin_err eof);
feof F			= throw (builtin_err (feof F));
flush			= throw (builtin_err flush);
fflush F		= throw (builtin_err (fflush F));

trap ACT SIG		= throw (builtin_err (trap ACT SIG));
time			= throw (builtin_err time);
sleep X			= throw (builtin_err (sleep X));

/* clib **********************************************************************/

import clib;

public type ClibException : Exception = special const clib_err X;

islower C		= throw (clib_err (islower C));
isupper C		= throw (clib_err (isupper C));
isalpha C		= throw (clib_err (isalpha C));
isdigit C		= throw (clib_err (isdigit C));
isxdigit C		= throw (clib_err (isxdigit C));
isalnum C		= throw (clib_err (isalnum C));
ispunct C		= throw (clib_err (ispunct C));
isspace C		= throw (clib_err (isspace C));
isgraph C		= throw (clib_err (isgraph C));
isprint C		= throw (clib_err (isprint C));
iscntrl C		= throw (clib_err (iscntrl C));
isascii C		= throw (clib_err (isascii C));
tolower S		= throw (clib_err (tolower S));
toupper S		= throw (clib_err (toupper S));
bytestr X		= throw (clib_err (bytestr X));
bcat Bs			= throw (clib_err (bcat Bs));
bsize B			= throw (clib_err (bsize B));
byte I B		= throw (clib_err (byte I B));
bsub B I J		= throw (clib_err (bsub B I J));
bcmp B1 B2		= throw (clib_err (bcmp B1 B2));
bint B			= throw (clib_err (bint B));
bfloat B		= throw (clib_err (bfloat B));
bstr B			= throw (clib_err (bstr B));
fdopen FD MODE		= throw (clib_err (fdopen FD MODE));
freopen NAME MODE F	= throw (clib_err (freopen NAME MODE F));
fileno F		= throw (clib_err (fileno F));
setvbuf F MODE		= throw (clib_err (setvbuf F MODE));
tmpnam			= throw (clib_err tmpnam);
tmpfile			= throw (clib_err tmpfile);
ftell F			= throw (clib_err (ftell F));
fseek F N WHENCE	= throw (clib_err (fseek F N WHENCE));
rewind F		= throw (clib_err (rewind F));
gets			= throw (clib_err gets);
fgets F			= throw (clib_err (fgets F));
fget F			= throw (clib_err (fget F));
ungetc C		= throw (clib_err (ungetc C));
fungetc C		= throw (clib_err (fungetc C));
printf FORMAT ARGS	= throw (clib_err (printf FORMAT ARGS));
fprintf F FORMAT ARGS	= throw (clib_err (fprintf F FORMAT ARGS));
sprintf FORMAT ARGS	= throw (clib_err (sprintf FORMAT ARGS));
scanf FORMAT		= throw (clib_err (scanf FORMAT));
fscanf F FORMAT		= throw (clib_err (fscanf F FORMAT));
sscanf S FORMAT		= throw (clib_err (sscanf S FORMAT));
rename OLD NEW		= throw (clib_err (rename OLD NEW));
unlink NAME		= throw (clib_err (unlink NAME));
truncate NAME LEN	= throw (clib_err (truncate NAME LEN));
getcwd			= throw (clib_err getcwd);
chdir NAME		= throw (clib_err (chdir NAME));
mkdir NAME MODE		= throw (clib_err (mkdir NAME MODE));
rmdir NAME		= throw (clib_err (rmdir NAME));
readdir NAME		= throw (clib_err (readdir NAME));
link OLD NEW		= throw (clib_err (link OLD NEW));
symlink OLD NEW		= throw (clib_err (symlink OLD NEW));
readlink NAME		= throw (clib_err (readlink NAME));
mkfifo NAME MODE	= throw (clib_err (mkfifo NAME MODE));
access NAME MODE	= throw (clib_err (access NAME MODE));
chmod NAME MODE		= throw (clib_err (chmod NAME MODE));
chown NAME UID GID	= throw (clib_err (chown NAME UID GID));
lchown NAME UID GID	= throw (clib_err (lchown NAME UID GID));
utime NAME TIMES	= throw (clib_err (utime NAME TIMES));
umask N			= throw (clib_err (umask N));
stat NAME		= throw (clib_err (stat NAME));
lstat NAME		= throw (clib_err (lstat NAME));
system CMD		= throw (clib_err (system CMD));
fork			= throw (clib_err fork);
exec PROG ARGS		= throw (clib_err (exec PROG ARGS));
spawn PROG ARGS		= throw (clib_err (spawn PROG ARGS));
_spawn MODE PROG ARGS	= throw (clib_err (_spawn MODE PROG ARGS));
nice INC		= throw (clib_err (nice INC));
exit N			= throw (clib_err (exit N));
pause			= throw (clib_err pause);
raise SIG		= throw (clib_err (raise SIG));
kill SIG PID		= throw (clib_err (kill SIG PID));
getpid			= throw (clib_err getpid);
getppid			= throw (clib_err getppid);
wait			= throw (clib_err wait);
waitpid PID OPTIONS	= throw (clib_err (waitpid PID OPTIONS));
isactive STATUS		= throw (clib_err (isactive STATUS));
isexited STATUS		= throw (clib_err (isexited STATUS));
exitstatus STATUS	= throw (clib_err (exitstatus STATUS));
issignaled STATUS	= throw (clib_err (issignaled STATUS));
termsig STATUS		= throw (clib_err (termsig STATUS));
isstopped STATUS	= throw (clib_err (isstopped STATUS));
stopsig STATUS		= throw (clib_err (stopsig STATUS));
getenv NAME		= throw (clib_err (getenv NAME));
setenv NAME VAL		= throw (clib_err (setenv NAME VAL));
setuid UID		= throw (clib_err (setuid UID));
setgid GID		= throw (clib_err (setgid GID));
seteuid UID		= throw (clib_err (seteuid UID));
setegid GID		= throw (clib_err (setegid GID));
setreuid RUID EUID	= throw (clib_err (setreuid RUID EUID));
setregid RGID EGID	= throw (clib_err (setregid RGID EGID));
getuid			= throw (clib_err getuid);
geteuid			= throw (clib_err geteuid);
getgid			= throw (clib_err getgid);
getegid			= throw (clib_err getegid);
getlogin		= throw (clib_err getlogin);
getgroups		= throw (clib_err getgroups);
setgroups GIDS		= throw (clib_err (setgroups GIDS));
getpgid PID		= throw (clib_err (getpgid PID));
setpgid PID PGID	= throw (clib_err (setpgid PID PGID));
getpgrp			= throw (clib_err getpgrp);
setpgrp			= throw (clib_err setpgrp);
getsid PID		= throw (clib_err (getsid PID));
setsid			= throw (clib_err setsid);
open NAME FLAGS MODE	= throw (clib_err (open NAME FLAGS MODE));
close FD		= throw (clib_err (close FD));
dup FD			= throw (clib_err (dup FD));
dup2 OLDFD NEWFD	= throw (clib_err (dup2 OLDFD NEWFD));
pipe			= throw (clib_err pipe);
fstat FD		= throw (clib_err (fstat FD));
fchdir FD		= throw (clib_err (fchdir FD));
fchmod FD MODE		= throw (clib_err (fchmod FD MODE));
fchown FD UID GID	= throw (clib_err (fchown FD UID GID));
ftruncate FD LEN	= throw (clib_err (ftruncate FD LEN));
fsync FD		= throw (clib_err (fsync FD));
fdatasync FD		= throw (clib_err (fdatasync FD));
bread FD COUNT		= throw (clib_err (bread FD COUNT));
bwrite FD MEM		= throw (clib_err (bwrite FD MEM));
lseek FD POS WHENCE	= throw (clib_err (lseek FD POS WHENCE));
fcntl FD CMD ARG	= throw (clib_err (fcntl FD CMD ARG));
select FILES		= throw (clib_err (select FILES));
isatty FD		= throw (clib_err (isatty FD));
ttyname FD		= throw (clib_err (ttyname FD));
ctermid			= throw (clib_err ctermid);
openpty			= throw (clib_err openpty);
forkpty			= throw (clib_err forkpty);
tcgetattr FD		= throw (clib_err (tcgetattr FD));
tcsetattr FD WHEN ATTR	= throw (clib_err (tcsetattr FD WHEN ATTR));
tcsendbreak FD DURATION	= throw (clib_err (tcsendbreak FD DURATION));
tcdrain FD		= throw (clib_err (tcdrain FD));
tcflush FD QUEUE	= throw (clib_err (tcflush FD QUEUE));
tcflow FD ACTION	= throw (clib_err (tcflow FD ACTION));
tcgetpgrp FD		= throw (clib_err (tcgetpgrp FD));
tcsetpgrp FD PGID	= throw (clib_err (tcsetpgrp FD PGID));
errno			= throw (clib_err errno);
seterrno N		= throw (clib_err (seterrno N));
perror S		= throw (clib_err (perror S));
strerror N		= throw (clib_err (strerror N));
uname			= throw (clib_err uname);
getpwuid UID		= throw (clib_err (getpwuid UID));
getpwnam NAME		= throw (clib_err (getpwnam NAME));
getpwent		= throw (clib_err getpwent);
getgrgid GID		= throw (clib_err (getgrgid GID));
getgrnam NAME		= throw (clib_err (getgrnam NAME));
getgrent		= throw (clib_err getgrent);
crypt KEY SALT		= throw (clib_err (crypt KEY SALT));
gethostbyname HOST	= throw (clib_err (gethostbyname HOST));
gethostent		= throw (clib_err gethostent);
getprotobyname NAME	= throw (clib_err (getprotobyname NAME));
getprotobynumber PROTO	= throw (clib_err (getprotobynumber PROTO));
getprotoent		= throw (clib_err getprotoent);
getservbyname NAME	= throw (clib_err (getservbyname NAME));
getservbyport PORT	= throw (clib_err (getservbyport PORT));
getservent		= throw (clib_err getservent);
socket FAMILY TYPE PROTO= throw (clib_err (socket FAMILY TYPE PROTO));
socketpair FAMILY TYPE PROTO
			= throw (clib_err (socketpair FAMILY TYPE PROTO));
shutdown SOCKET HOW	= throw (clib_err (shutdown SOCKET HOW));
closesocket SOCKET	= throw (clib_err (closesocket SOCKET));
bind SOCKET ADDR	= throw (clib_err (bind SOCKET ADDR));
listen SOCKET N		= throw (clib_err (listen SOCKET N));
accept SOCKET		= throw (clib_err (accept SOCKET));
connect SOCKET ADDR	= throw (clib_err (connect SOCKET ADDR));
getsockname SOCKET	= throw (clib_err (getsockname SOCKET));
getpeername SOCKET	= throw (clib_err (getpeername SOCKET));
getsockopt SOCKET LEVEL OPT
			= throw (clib_err (getsockopt SOCKET LEVEL OPT));
setsockopt SOCKET LEVEL OPT DATA
			= throw (clib_err (setsockopt SOCKET LEVEL OPT DATA));
recv SOCKET FLAGS	= throw (clib_err (recv SOCKET FLAGS));
send SOCKET FLAGS DATA	= throw (clib_err (send SOCKET FLAGS DATA));
recvfrom SOCKET FLAGS	= throw (clib_err (recvfrom SOCKET FLAGS));
sendto SOCKET FLAGS DATA= throw (clib_err (sendto SOCKET FLAGS DATA));
thread_no THREAD	= throw (clib_err (thread_no THREAD));
this_thread		= throw (clib_err this_thread);
thread X		= throw (clib_err (thread X));
return X		= throw (clib_err (return X));
cancel THREAD		= throw (clib_err (cancel THREAD));
result THREAD		= throw (clib_err (result THREAD));
yield			= throw (clib_err yield);
active THREAD		= throw (clib_err (active THREAD));
canceled THREAD		= throw (clib_err (canceled THREAD));
setsched THREAD POL PRIO= throw (clib_err (setsched THREAD POL PRIO));
getsched THREAD		= throw (clib_err (getsched THREAD));
mutex			= throw (clib_err mutex);
errorchecking_mutex	= throw (clib_err errorchecking_mutex);
recursive_mutex		= throw (clib_err recursive_mutex);
lock MUTEX		= throw (clib_err (lock MUTEX));
unlock MUTEX		= throw (clib_err (unlock MUTEX));
try MUTEX		= throw (clib_err (try MUTEX));
cond			= throw (clib_err cond);
signal COND		= throw (clib_err (signal COND));
broadcast COND		= throw (clib_err (broadcast COND));
await COND		= throw (clib_err (await COND));
sem			= throw (clib_err sem);
send SEM X		= throw (clib_err (send SEM X));
receive SEM		= throw (clib_err (receive SEM));
get_size SEM		= throw (clib_err (get_size SEM));
get_bound SEM		= throw (clib_err (get_bound SEM));
ref X			= throw (clib_err (ref X));
put REF X		= throw (clib_err (put REF X));
get REF			= throw (clib_err (get REF));
tzname			= throw (clib_err tzname);
timezone		= throw (clib_err timezone);
daylight		= throw (clib_err daylight);
ctime T			= throw (clib_err (ctime T));
gmtime T		= throw (clib_err (gmtime T));
localtime T		= throw (clib_err (localtime T));
mktime TM		= throw (clib_err (mktime TM));
asctime TM		= throw (clib_err (asctime TM));
strftime FORMAT TM	= throw (clib_err (strftime FORMAT TM));
clock			= throw (clib_err clock);
times			= throw (clib_err times);
fnmatch PATTERN S	= throw (clib_err (fnmatch PATTERN S));
glob PATTERN		= throw (clib_err (glob PATTERN));
regmatch OPTS REGEX S	= throw (clib_err (regmatch OPTS REGEX S));
regnext			= throw (clib_err regnext);
regdone			= throw (clib_err regdone);
regex OPTS REGEX S EXPR	= throw (clib_err (regex OPTS REGEX S EXPR));
regstart		= throw (clib_err regstart);
regskip			= throw (clib_err regskip);
reg N			= throw (clib_err (reg N));
regpos N		= throw (clib_err (regpos N));
regend N		= throw (clib_err (regend N));
regs			= throw (clib_err regs);
pow M N			= throw (clib_err (pow M N));
root M N		= throw (clib_err (root M N));
intsqrt M		= throw (clib_err (intsqrt M));
powmod K M N		= throw (clib_err (powmod K M N));
invmod K M		= throw (clib_err (invmod K M));
isprime N		= throw (clib_err (isprime N));
gcd M N			= throw (clib_err (gcd M N));
lcm M N			= throw (clib_err (lcm M N));
fact M N		= throw (clib_err (fact M N));
rem M N			= throw (clib_err (rem M N));
jacobi M N		= throw (clib_err (jacobi M N));
sort P Xs		= throw (clib_err (sort P Xs));

/* complex *******************************************************************/

import complex;

public type ComplexException : Exception = special const complex_err X;

arg Z			= throw (complex_err (arg Z));
re Z			= throw (complex_err (re Z));
im Z			= throw (complex_err (im Z));
conj Z			= throw (complex_err (conj Z));

/* cond **********************************************************************/

import cond;

public type CondException : Exception = special const cond_err X;

ifelse P X Y		= throw (cond_err (ifelse P X Y));
switch CASES		= throw (cond_err (switch CASES));
match X CASES		= throw (cond_err (match X CASES));

/* error *********************************************************************/

import error;

public type ErrorException : Exception = special const error_err X;

error S			= throw (error_err (error S));

/* graphics ******************************************************************/

/* Note that the graphics script is not by default included by the prelude,
   but will be imported into your program with this module. You might wish to
   remove this section if you don't need the graphics script in your
   program. */

import graphics;

public type GraphicsException : Exception = special const graphics_err X;

moveto X Y		= throw (graphics_err (moveto X Y));
rmoveto DX DY		= throw (graphics_err (rmoveto DX DY));
lineto X Y		= throw (graphics_err (lineto X Y));
rlineto DX DY		= throw (graphics_err (rlineto DX DY));
curveto X1 Y1 X2 Y2 X3 Y3
			= throw (graphics_err (curveto X1 Y1 X2 Y2 X3 Y3));
rcurveto DX1 DY1 DX2 DY2 DX3 DY3
			= throw (graphics_err
				 (rcurveto DX1 DY1 DX2 DY2 DX3 DY3));
arc X Y R A1 A2		= throw (graphics_err (arc X Y R A1 A2));
narc X Y R A1 A2	= throw (graphics_err (narc X Y R A1 A2));
arct X1 Y1 X2 Y2 R	= throw (graphics_err (arct X1 Y1 X2 Y2 R));
charpath S T		= throw (graphics_err (charpath S T));
show S			= throw (graphics_err (show S));
translate TX TY		= throw (graphics_err (translate TX TY));
scale SX SY		= throw (graphics_err (scale SX SY));
rotate A		= throw (graphics_err (rotate A));
setlinewidth X		= throw (graphics_err (setlinewidth X));
setlinecap N		= throw (graphics_err (setlinecap N));
setlinejoin N		= throw (graphics_err (setlinejoin N));
setdash Xs DX		= throw (graphics_err (setdash Xs DX));
setgray N		= throw (graphics_err (setgray N));
setrgbcolor R G B	= throw (graphics_err (setrgbcolor R G B));
sethsbcolor H S B	= throw (graphics_err (sethsbcolor H S B));
setcmykcolor C M Y K	= throw (graphics_err (setcmykcolor C M Y K));
setfont S X		= throw (graphics_err (setfont S X));
setcolor C		= throw (graphics_err (setcolor C));
copies N		= throw (graphics_err (copies N));
psfile S		= throw (graphics_err (psfile S));
psstr S			= throw (graphics_err (psstr S));
ps S			= throw (graphics_err (ps S));

/* lambda ********************************************************************/

import lambda;

public type LambdaException : Exception = special const lambda_err X;

lambda X Y		= throw (lambda_err (lambda X Y));

_H X Y			= throw (lambda_err (_H X Y));
_HL X Y			= throw (lambda_err (_HL X Y));
_HT X Y			= throw (lambda_err (_HT X Y));
_HK X Y Z		= throw (lambda_err (_HK X Y Z));
__H X Y			= throw (lambda_err (__H X Y));
__HL X Y		= throw (lambda_err (__HL X Y));
__HT X Y		= throw (lambda_err (__HT X Y));
__HK X Y Z		= throw (lambda_err (__HK X Y Z));

/* list **********************************************************************/

/* none */

/* math **********************************************************************/

import math;

public type MathException : Exception = special const math_err X;

asin X			= throw (math_err (asin X));
acos X			= throw (math_err (acos X));
tan X			= throw (math_err (tan X));
lg X			= throw (math_err (lg X));
log X			= throw (math_err (log X));
sinh X			= throw (math_err (sinh X));
cosh X			= throw (math_err (cosh X));
tanh X			= throw (math_err (tanh X));
asinh X			= throw (math_err (asinh X));
acosh X			= throw (math_err (acosh X));
atanh X			= throw (math_err (atanh X));

/* sort **********************************************************************/

import sort;

public type SortException : Exception = special const sort_err X;

msort P Xs		= throw (sort_err (msort P Xs));
qsort P Xs		= throw (sort_err (qsort P Xs));

/* stdtypes ******************************************************************/

import stdtypes;

public type StdtypesException : Exception = special const stdtypes_err X;

array Xs		= throw (stdtypes_err (array Xs));
array2 Xs		= throw (stdtypes_err (array2 Xs));
bag Xs			= throw (stdtypes_err (bag Xs));
delete A X		= throw (stdtypes_err (delete A X));
dict XYs		= throw (stdtypes_err (dict XYs));
first A			= throw (stdtypes_err (first A));
hdict XYs		= throw (stdtypes_err (hdict XYs));
heap Xs			= throw (stdtypes_err (heap Xs));
insert A X		= throw (stdtypes_err (insert A X));
keys D			= throw (stdtypes_err (keys D));
last A:Array		= throw (stdtypes_err (last A));
last M:Bag		= throw (stdtypes_err (last M));
last D:Dict		= throw (stdtypes_err (last D));
last M:Set		= throw (stdtypes_err (last M));
member A X		= throw (stdtypes_err (member A X));
members A		= throw (stdtypes_err (members A));
members2 A		= throw (stdtypes_err (members2 A));
mkarray X N		= throw (stdtypes_err (mkarray X N));
mkarray2 X NM		= throw (stdtypes_err (mkarray2 X NM));
mkdict Y Xs		= throw (stdtypes_err (mkdict Y Xs));
mkhdict Y Xs		= throw (stdtypes_err (mkhdict Y Xs));
rmfirst A		= throw (stdtypes_err (rmfirst A));
rmlast A		= throw (stdtypes_err (rmlast A));
set Xs			= throw (stdtypes_err (set Xs));
update A K X		= throw (stdtypes_err (update A K X));
update2 A IJ X		= throw (stdtypes_err (update2 A IJ X));
vals D			= throw (stdtypes_err (vals D));

/* stream ********************************************************************/

import stream;

public type StreamException : Exception = special const stream_err X;

stream Xs		= throw (stream_err (stream Xs));
foldl1 F Xs:Stream	= throw (stream_err (foldl1 F Xs));
foldr1 F Xs:Stream	= throw (stream_err (foldr1 F Xs));
hd Xs:Stream		= throw (stream_err (hd Xs));
init Xs:Stream		= throw (stream_err (init Xs));
last Xs:Stream		= throw (stream_err (last Xs));
pop Xs:Stream		= throw (stream_err (pop Xs));
tl Xs:Stream		= throw (stream_err (tl Xs));
top Xs:Stream		= throw (stream_err (top Xs));
numstream N		= throw (stream_err (numstream N));
numstreamby K N		= throw (stream_err (numstreamby K N));

/* string ********************************************************************/

import string;

public type StringException : Exception = special const string_err X;

chars S			= throw (string_err (chars S));
join DELIM Xs		= throw (string_err (join DELIM Xs));
split DELIM S		= throw (string_err (split DELIM S));
strcat Xs		= throw (string_err (strcat Xs));

/* stdlib ********************************************************************/

/* Make sure that these definitions come last because some of the default
   rules here are overridden above (overloaded operations!). */

import stdlib;

public type StdlibException : Exception = special const stdlib_err X;

abs X			= throw (stdlib_err (abs X));
all P Xs		= throw (stdlib_err (all P Xs));
any P Xs		= throw (stdlib_err (any P Xs));
append Xs Y		= throw (stdlib_err (append Xs Y));
cat Xs			= throw (stdlib_err (cat Xs));
cons X Xs		= throw (stdlib_err (cons X Xs));
do F Xs			= throw (stdlib_err (do F Xs));
drop N Xs		= throw (stdlib_err (drop N Xs));
dropwhile P Xs		= throw (stdlib_err (dropwhile P Xs));
filter P Xs		= throw (stdlib_err (filter P Xs));
foldl F A Xs		= throw (stdlib_err (foldl F A Xs));
foldl1 F Xs		= throw (stdlib_err (foldl1 F Xs));
foldr F A Xs		= throw (stdlib_err (foldr F A Xs));
foldr1 F Xs		= throw (stdlib_err (foldr1 F Xs));
fst Xs			= throw (stdlib_err (fst Xs));
hd Xs			= throw (stdlib_err (hd Xs));
hds Xss			= throw (stdlib_err (hds Xss));
init Xs			= throw (stdlib_err (init Xs));
iter N F A		= throw (stdlib_err (iter N F A));
last Xs			= throw (stdlib_err (last Xs));
map F Xs		= throw (stdlib_err (map F Xs));
mklist X N		= throw (stdlib_err (mklist X N));
null Xs			= throw (stdlib_err (null Xs));
nums N M		= throw (stdlib_err (nums N M));
numsby K N M		= throw (stdlib_err (numsby K N M));
pop Xs			= throw (stdlib_err (pop Xs));
prd Xs			= throw (stdlib_err (prd Xs));
push Xs X		= throw (stdlib_err (push Xs X));
reverse Xs		= throw (stdlib_err (reverse Xs));
scan F A Xs		= throw (stdlib_err (scan F A Xs));
scan1 F Xs		= throw (stdlib_err (scan1 F Xs));
sgn X			= throw (stdlib_err (sgn X));
snd Xs			= throw (stdlib_err (snd Xs));
sum Xs			= throw (stdlib_err (sum Xs));
take N Xs		= throw (stdlib_err (take N Xs));
takewhile P Xs		= throw (stdlib_err (takewhile P Xs));
tl Xs			= throw (stdlib_err (tl Xs));
tls Xss			= throw (stdlib_err (tls Xss));
top Xs			= throw (stdlib_err (top Xs));
transpose Xss		= throw (stdlib_err (transpose Xss));
trd Xs			= throw (stdlib_err (trd Xs));
tuplecat Xs		= throw (stdlib_err (tuplecat Xs));
unzip Xs:List		= throw (stdlib_err (unzip Xs:List));
unzip3 Xs:List		= throw (stdlib_err (unzip3 Xs:List));
zip Xs Ys		= throw (stdlib_err (zip Xs Ys));
zip3 Xs Ys Zs		= throw (stdlib_err (zip3 Xs Ys Zs));
zipwith F Xs Ys		= throw (stdlib_err (zipwith F Xs Ys));
zipwith3 F Xs Ys Zs	= throw (stdlib_err (zipwith3 F Xs Ys Zs));
