
/* q.c: the Q interpreter */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <setjmp.h>
#include "qdefs.h"

static char		signon[] = Q_SIGNON;
static char   		usage[] = Q_USAGE;
static char		opts[MAXSTRLEN];
static char		terms[] = TERMS;
static char		copying[] = COPYING;
static char		helpmsg[] = HELPMSG;

static char           *self = "q";
static char            qcprog[MAXSTRLEN] = "qc";

static bool		batch = 0, docompile = 1, donecompile = 0,
  doexitrc = 0, oset = 0;

bool		eflag = 0, qflag = 0, iflag = 0, hflag = 0, Vflag = 0,
  gflag = 0, norl = 0;

char *source = NULL;
char *which = NULL;

static unload_dlls(void);
#ifdef USE_READLINE
static void fini_readline(void);
#endif

void exitproc(void)
{
  FILE *fp;
  char fname[MAXSTRLEN];
  int i;

  if (donecompile) remove(code);
  if (doexitrc && exitrc && chkfile(expand(fname, exitrc)) &&
      (fp = fopen(fname, "r"))) {
    fclose(fp);
    quitflag = 0;
    parsesrc(fname, 1);
  }
  unload_dlls();
  lt_dlexit();
#if defined (USE_READLINE)
  fini_readline();
  rl_deprep_terminal();
#endif
}

void fatal(char *s)
{
  fprintf(stderr, "%s: %s\n", self, s);
  doexitrc = 0;
  exit(1);
}

void echo(char *s)
{
  char *t = s;
  while (*s && isspace(*s)) ++s;
  if (*s != '@')
    printf("%s\n", t);
}

#ifdef HAVE_BACKTRACE
/* Obtain a backtrace and print it to `stdout'. */
void print_trace(void)
{
  void *array[50];
  size_t size;

  size = backtrace (array, 50);

  fprintf(stderr, "%d stack frames\n", (int)size);
  backtrace_symbols_fd(array, size, fileno(stderr));
}
#endif

#if defined(USE_READLINE) && defined(HAVE_POSIX_SIGNALS)
/* used to implement nonlocal exit while reading input using readline */
jmp_buf longjmp_target;
volatile int longjump_set = 0;
#endif

RETSIGTYPE
break_handler(int sig)
/* handle SIGINT */
{
  SIGHANDLER_RESTORE(sig, break_handler);
  brkflag = 1;
#if defined(USE_READLINE) && defined(HAVE_POSIX_SIGNALS)
  if (longjump_set) longjmp(longjmp_target, 1);
#endif
  SIGHANDLER_RETURN(0);
}

RETSIGTYPE
term_handler(int sig)
/* handle SIGTERM */
{
  SIGHANDLER_RESTORE(sig, term_handler);
  quitflag = 1;
#if defined(USE_READLINE) && defined(HAVE_POSIX_SIGNALS)
  if (longjump_set) longjmp(longjmp_target, 1);
#endif
  SIGHANDLER_RETURN(0);
}

RETSIGTYPE
tty_handler(int sig)
/* handle SIGTTIN/SIGTTOU */
{
  SIGHANDLER_RESTORE(sig, tty_handler);
#if defined (USE_READLINE)
  rl_deprep_terminal();
#endif
#ifdef SIGTTIN
  fprintf(stderr, "%s[pid %d]: stopped (tty %s)\n", self, getpid(),
	  (sig==SIGTTIN)?"input":"output");
#endif
#ifndef _WIN32
  raise(SIGSTOP);
#endif
  SIGHANDLER_RETURN(0);
}

RETSIGTYPE
segv_handler(int sig)
/* handle fatal program errors */
{
  static volatile bool recursive = 0;
  /* when we come here, many things can already be broken; we proceed (with
     fingers crossed) anyway */
  if (recursive)
    SIGHANDLER_RETURN(0);
  else
    recursive = 1;
#if defined (USE_READLINE)
  rl_deprep_terminal();
#endif
#ifdef HAVE_BACKTRACE
#ifdef HAVE_STRSIGNAL
  fprintf(stderr, "%s[pid %d]: caught signal %d (%s), printing backtrace... ",
	  self, getpid(), sig, strsignal(sig));
#else
  fprintf(stderr, "%s[pid %d]: caught signal %d, printing backtrace... ",
	  self, getpid(), sig);
#endif
  fflush(stderr);
  print_trace();
#else
#ifdef HAVE_STRSIGNAL
  fprintf(stderr, "%s[pid %d]: caught signal %d (%s), exiting\n",
	  self, getpid(), sig, strsignal(sig));
#else
  fprintf(stderr, "%s[pid %d]: caught signal %d, exiting\n",
	  self, getpid(), sig);
#endif
#endif
  fflush(NULL);
  syssignal(sig, SIG_DFL);
  raise(sig);
  SIGHANDLER_RETURN(0);
}

RETSIGTYPE (*old_handler)() = NULL;

void push_sigint(RETSIGTYPE (*new_handler)())
{
  if (!old_handler)
    old_handler = sigint(new_handler);
}

void pop_sigint(void)
{
  if (old_handler) {
    sigint(old_handler);
    old_handler = NULL;
  }
}

/* gmp memory handlers */

static void *
gmp_allocate (size)
     size_t size;
{
  void *ret;

  ret = malloc (size);
  return ret;
}

static void *
gmp_reallocate (oldptr, old_size, new_size)
     void *oldptr;
     size_t old_size;
     size_t new_size;
{
  void *ret;

  ret = realloc (oldptr, new_size);
  return ret;
}

static void
gmp_free (blk_ptr, blk_size)
     void *blk_ptr;
     size_t blk_size;
{
  free (blk_ptr);
}

/* safe (non-leaking) version of _mpz_realloc */

void *my_mpz_realloc(mpz_ptr m, mp_size_t new_size)
{
  mpz_t m1;
  memcpy(m1, m, sizeof(mpz_t));
  if (_mpz_realloc(m, new_size))
    return m->_mp_d;
  else {
    if (m1->_mp_d) mpz_clear(m1);
    return NULL;
  }
}

/* readline support */

#ifdef USE_READLINE

static char **fsyms = NULL, **vsyms = NULL;

static char *csyms[] = {
  "break", "cd", "chdir", "clear", "completion_matches", "copying", "debug",
  "dec", "def", "echo", "edit", "fix", "format", "help", "hex", "histfile",
  "histsize", "import", "imports", "load", "ls", "memsize", "modules", "oct",
  "off", "on", "path", "prompt", "pwd","run", "save", "sci", "source",
  "stacksize", "stats", "std", "undef", "which", "who", "whos",
  NULL
};

static comp(x, y)
     char **x, **y;
{
  return strcmp(*x, *y);
}

static build_fsym_table()
{
  int i, fno;
  fsyms = (char**)calloc(2*(symtbsz-BINARY)+1, sizeof(char*));
  if (!fsyms) return;
  for (i = 0, fno = BINARY; fno < symtbsz; fno++) {
    char *base = strsp+symtb[fno].pname, name[MAXSTRLEN];
    if (('a' <= *base && *base <= 'z' ||
	 'A' <= *base && *base <= 'Z' ||
	 *base == '_') &&
	!(symtb[fno].flags & TSYM)
	/*&& (!(symtb[fno].flags & VSYM)||symtb[fno].x)*/) {
      pname(name, fno);
      fsyms[i++] = strdup(name);
      if (visible(fno) && unique(fno)) {
	if (symtb[fno].modno == NONE)
	  sprintf(name, "::%s", strsp + symtb[fno].pname);
	else
	  sprintf(name, "%s::%s", strsp + modtb[symtb[fno].modno],
		  strsp + symtb[fno].pname);
	fsyms[i++] = strdup(name);
      }
    }
  }
  qsort(fsyms, i, sizeof(char*), comp);
}

static build_vsym_table()
{
  int i, fno;
  vsyms = (char**)calloc(2*tmptbsz+1, sizeof(char*));
  if (!vsyms) return;
  for (i = 0, fno = symtbsz; fno < symtbsz+tmptbsz; fno++) {
    char *base = strsp+symtb[fno].pname, name[MAXSTRLEN];
    if (('a' <= *base && *base <= 'z' ||
	 'A' <= *base && *base <= 'Z' ||
	 *base == '_') &&
	!(symtb[fno].flags & TSYM) &&
	(!(symtb[fno].flags & VSYM)||symtb[fno].x)) {
      pname(name, fno);
      vsyms[i++] = strdup(name);
      if (visible(fno) && unique(fno)) {
	if (symtb[fno].modno == NONE)
	  sprintf(name, "::%s", strsp + symtb[fno].pname);
	else
	  sprintf(name, "%s::%s", strsp + modtb[symtb[fno].modno],
		  strsp + symtb[fno].pname);
	vsyms[i++] = strdup(name);
      }
    }
  }
  qsort(vsyms, i, sizeof(char*), comp);
}

static search(table, text, len)
     char **table, *text;
     int len;
{
  /* hmm, this should be binary search, maybe next time ... ;-) */
  int i = 0;
  while (table[i] && strncmp(text, table[i], len) > 0) i++;
  return i;
}

static clear_fsym_table()
{
  if (!fsyms) return;
  free(fsyms);
  fsyms = NULL;
}

static clear_vsym_table()
{
  if (!vsyms) return;
  free(vsyms);
  vsyms = NULL;
}

static char *
sym_generator (text, state)
     char *text;
     int state;
{
  static int i_csym, i_fsym, i_vsym, len;
  int cmp;
  char *name;
     
  if (!state)
    {
      len = strlen (text);
      if (!fsyms)
	build_fsym_table();
      if (!vsyms)
	build_vsym_table();
      if (!fsyms || !vsyms)
	fatal("memory overflow");
      i_csym = search(csyms, text, len);
      i_fsym = search(fsyms, text, len);
      i_vsym = search(vsyms, text, len);
    }

  /* command keywords */
  while (name = csyms[i_csym])
    {
      i_csym++;

      if ((cmp = strncmp (name, text, len)) == 0)
	return (strdup(name));
      else if (cmp > 0)
	break;
    }

  /* function or predefined var symbol */
  while (name = fsyms[i_fsym])
    {
      i_fsym++;

      if ((cmp = strncmp (name, text, len)) == 0)
	return (strdup(name));
      else if (cmp > 0)
	break;
    }

  /* temporary variable symbols */
  while (name = vsyms[i_vsym])
    {
      i_vsym++;
      
      if ((cmp = strncmp (name, text, len)) == 0)
	return (strdup(name));
      else if (cmp > 0)
	break;
    }
  
  return ((char *)NULL);
}

#ifndef HAVE_RL_COMPLETION_MATCHES
#define rl_completion_matches completion_matches
#endif

char **sym_completion (char *text, int start, int end)
{
  return rl_completion_matches (text, sym_generator);
}

int use_readline = 0;
static bool history_read = 0;

static void init_readline(void)
{
  rl_readline_name = "Q";
  rl_basic_word_break_characters = " \t\n\"\\'`@$><=,;|%&~{[(";
  rl_attempted_completion_function = (CPPFunction *)sym_completion;
  if (!history_read && !gflag && use_readline) {
    char h[MAXSTRLEN];
    if (histmax > 0)
      read_history(expand(h, histfile));
    stifle_history(histmax);
    history_read = 1;
  }
}

static void fini_readline(void)
{
  if (history_read && histmax > 0) {
    char h[MAXSTRLEN];
    write_history(expand(h, histfile));
  }
}
#endif

static char *getline1(fp, prompt)
     FILE *fp;
     char *prompt;
{
  char *buf, *bufp;
#ifdef USE_READLINE
  if (fp == stdin && use_readline && isatty(fileno(stdin))) {
    static char *last = NULL;
#ifdef HAVE_POSIX_SIGNALS
    save_sigmask();
    if (setjmp(longjmp_target) == 0) {
      longjump_set = 1;
      buf = readline(prompt);
      longjump_set = 0;
    } else {
      /* clean up */
      longjump_set = 0;
      rl_free_line_state();
      rl_cleanup_after_signal();
/*       printf("\n"); */
      buf = malloc(sizeof(char));
      if (!buf) fatal("memory overflow");
      *buf = 0;
      restore_sigmask();
    }
#else
    buf = readline(prompt);
#endif
    if (buf && *buf &&
	(history_length <= 0 ||
	 strcmp(buf, history_get(history_length-1+history_base)->line) != 0))
      add_history(buf);
    return buf;
  } else {
#endif
    int chunksz = 10000;
    int actsz, l;
    buf = malloc(chunksz*sizeof(char));
    actsz = chunksz;
    bufp = buf;
    if (bufp) {
      *bufp = 0;
      if (iflag && fp == stdin) {
	printf("%s", prompt);
	fflush(stdout);
      }
    }
    while (buf && !ferror(fp) && !feof(fp)) {
      int k;
      if (!fgets(bufp, chunksz, fp) || ferror(fp) || feof(fp) ||
	  (l = strlen(bufp)) > 0 && bufp[l-1] == '\n')
	break;
      /* enlarge the buffer */
      k = bufp-buf+l;
      actsz += chunksz;
      buf = realloc(buf, actsz*sizeof(char));
      bufp = buf+k;
    }
    if (buf)
      if (*buf) {
	l = strlen(buf);
	if (buf[l-1] == '\n')
	  buf[--l] = 0;
	buf = realloc(buf, (strlen(buf)+1)*sizeof(char));
      } else {
	free(buf);
	return NULL;
      }
    if (buf)
      if (ferror(fp) || feof(fp) && !*buf) {
	free(buf);
	return NULL;
      } else
	return buf;
    else
      fatal("memory overflow");
#ifdef USE_READLINE
  }
#endif
}

static contd();

int actlineno = 0;

static char *ps = NULL, *psx = NULL, *psdef = PROMPT;

void new_xprompt(void)
{
  if (psx && psx != psdef) free(psx);
  ps = psx = NULL;
}

static char *xprompt(char *prompt)
{
  if (ps != prompt) {
    /* expand placeholders in prompt string */
    int l, sl = strlen(sysinfo), vl = strlen(version), wl, WL, ml, ML;
    char *s, *buf, wd[MAXSTRLEN], WD[MAXSTRLEN];
    char *m = (mainno==-1)?"":strsp+fnametb[mainno];
    char *M = (mainno==-1)?"":strsp+modtb[mainno];
    if (!getcwd(wd, MAXSTRLEN)) strcpy(wd, "");
    basename(WD, wd, 0);
    wl = strlen(wd); WL = strlen(WD);
    ml = strlen(m); ML = strlen(M);
    s = ps = prompt; l = strlen(ps);
    if (psx && psx != psdef) free(psx);
    while ((s = strstr(s, "\\v"))) {
      l += vl-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\s"))) {
      l += sl-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\w"))) {
      l += wl-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\W"))) {
      l += WL-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\m"))) {
      l += ml-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\M"))) {
      l += ML-2; s++;
    }
    if ((psx = (char*)malloc((l+1)*sizeof(char))) &&
	(buf = (char*)malloc((l+1)*sizeof(char)))) {
      strcpy(psx, ps);
      while ((s = strstr(psx, "\\v"))) {
	strcpy(buf, s+2); strcpy(s, version);
	strcpy(s+vl, buf);
      }
      while ((s = strstr(psx, "\\s"))) {
	strcpy(buf, s+2); strcpy(s, sysinfo);
	strcpy(s+sl, buf);
      }
      while ((s = strstr(psx, "\\w"))) {
	strcpy(buf, s+2); strcpy(s, wd);
	strcpy(s+wl, buf);
      }
      while ((s = strstr(psx, "\\W"))) {
	strcpy(buf, s+2); strcpy(s, WD);
	strcpy(s+WL, buf);
      }
      while ((s = strstr(psx, "\\m"))) {
	strcpy(buf, s+2); strcpy(s, m);
	strcpy(s+ml, buf);
      }
      while ((s = strstr(psx, "\\M"))) {
	strcpy(buf, s+2); strcpy(s, M);
	strcpy(s+ML, buf);
      }
      free(buf);
    } else {
      if (psx) free(psx);
      psx = psdef;
    }
  }
  return psx;
}

char *getline(FILE *fp, char *prompt)
{
  char *buf;
  int l;
#ifdef USE_READLINE
  if (fp == stdin)
    clear_vsym_table();
#endif
  if (!(buf = getline1(fp, xprompt(prompt)))) return NULL;
  l = strlen(buf);
  actlineno++;
  while (contd(buf)) {
    char *buf2;
    if (!(buf2 = getline1(fp, prompt2))) break;
    actlineno++;
    buf[l] = '\n'; 
    if ((buf = realloc(buf, (l+strlen(buf2)+2)*sizeof(char)))) {
      strcpy(buf+l+1, buf2);
      l += strlen(buf2)+1;
      free(buf2);
    } else
      fatal("memory overflow");
  }
  return buf;
}

static int contd(char *p)
{
  int strmode = 0, l;
  if (!p) return 0;
  l = strlen(p);
  for (; *p; p++) {
    if (strmode) {
      if (*p == '\\') {
	if (!*++p)
	  return 1;
      } else if (*p == '"')
	strmode = 0;
    } else if (*p == '%' || strncmp(p, "//", 2) == 0)
      return 0;
    else if (*p == '"')
      strmode = 1;
  }
  return l >= 1 && p[-1] == '\\';
}

/* command line options */

/* this should be plenty */
#define MAXARGC 1200

static char	       *qcargv[MAXARGC+1];
static int		qcargc;

static wsarg(char *arg)
{
  while (*arg)
    if (isspace(*arg))
      return 1;
    else
      arg++;
  return 0;
}

static char *quotearg(char *arg)
{
  if (!arg) return NULL;
#ifdef _WIN32
  /* MS C lib kludge: empty command args or args containing whitespace need
     quoting */
  if (!*arg || wsarg(arg)) {
    char *newarg = malloc(2*strlen(arg)+3), *s;
    if (!newarg)
      return 0;
    s = newarg;
    *s++ = '"';
    while (*arg) {
      if (*arg == '"') *s++ = '\\';
      *s++ = *arg++;
    }
    *s++ = '"';
    *s = 0;
    return newarg;
  } else
#endif
    return strdup(arg);
}

static qcarg(char *arg)
{
  if (qcargc >= MAXARGC)
    fatal("too many command line parameters");
  else if (arg && !(arg = quotearg(arg)))
    fatal("memory overflow");
  else {
    if (qcargv[qcargc]) free(qcargv[qcargc]);
    qcargv[qcargc++] = arg;
  }
}

static set_qcarg(int i, char *arg)
{
  if (i >= MAXARGC)
    fatal("too many command line parameters");
  else if (arg && !(arg = quotearg(arg)))
    fatal("memory overflow");
  else {
    if (qcargv[i]) free(qcargv[i]);
    qcargv[i] = arg;
  }
}

static char *list = "", *hsz = "";
static struct option longopts[] = Q_OPTS;

static int
getintarg(char *s, int *i)
{
  char *t = s;
  while (isspace(*t)) t++;
  s = t;
  while (isdigit(*t)) t++;
  if (t == s) return 0;
  while (isspace(*t)) t++;
  if (*t) return 0;
  *i = atoi(s);
  return 1;
}

static void
parse_opts(int argc, char **argv, int pass)
     /* pass = 0 denotes source, 1 command line pass */
{
  int c, longind;
  optind = 0;
  while ((c = getopt_long(argc, argv, Q_OPTS1, longopts,
			  &longind)) != EOF)
    switch (c) {
    case Q_GNUCLIENT:
      gflag = 1;
      break;
    case Q_DEBUG_OPTIONS: {
      char opts[MAXSTRLEN];
      strcpy(opts, optarg?optarg:"");
      if (!debug_parse_opts(opts)) {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid option string `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    } case Q_BREAK:
      brkdbg = 1;
      break;
    case Q_PROMPT:
      prompt = optarg;
      break;
    case Q_DEC:
      imode = 0;
      break;
    case Q_HEX:
      imode = 1;
      break;
    case Q_OCT:
      imode = 2;
      break;
    case Q_STD: {
      int prec;
      if (!optarg)
	prec = 15;
      else if (!getintarg(optarg, &prec))
	prec = -1;
      if (prec >= 0) {
	fmode = 0; fprec = prec; sprintf(fformat, STDFORMAT, prec);
      } else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid precision `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_SCI: {
      int prec;
      if (!optarg)
	prec = 15;
      else if (!getintarg(optarg, &prec))
	prec = -1;
      if (prec >= 0) {
	fmode = 1; fprec = prec; sprintf(fformat, SCIFORMAT, prec);
      } else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid precision `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_FIX: {
      int prec;
      if (!optarg)
	prec = 2;
      else if (!getintarg(optarg, &prec))
	prec = -1;
      if (prec >= 0) {
	fmode = 2; fprec = prec; sprintf(fformat, FIXFORMAT, prec);
      } else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid precision `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_HISTFILE:
      histfile = optarg;
      break;
    case Q_HISTSIZE: {
      int sz = optarg?atoi(optarg):0;
      if (optarg && getintarg(optarg, &sz) && sz >= 0)
	histmax = sz;
      else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid size `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_INITRC:
      initrc = optarg;
      break;
    case Q_NO_INITRC:
      initrc = NULL;
      break;
    case Q_EXITRC:
      exitrc = optarg;
      break;
    case Q_NO_EXITRC:
      exitrc = NULL;
      break;
    case Q_NO_EDITING:
      norl = 1;
      break;
    case Q_GC_MSGS:
      gc_v = 1;
      break;
    case Q_GC: {
      int t;
      gc_flag = 1;
      if (optarg)
	if (getintarg(optarg, &t)) {
	  if (t < 0 || t > 100) {
	    fprintf(stderr, "%s: bad tolerance value `%s', using default\n",
		    self, t);
	    t = 0;
	  }
	} else {
	  char msg[MAXSTRLEN];
	  sprintf(msg, "invalid tolerance value `%s'", optarg);
	  fatal(msg);
	}
      else
	t = 0;
      gc_tol = ((double)t)/100.0;
      break;
    }
    case Q_STACKSIZE: {
      int sz;
      if (optarg && getintarg(optarg, &sz) && sz >= 0)
	if (sz == 0 || sz >= STACKMIN)
	  stackmax = sz;
	else {
	  fprintf(stderr, "%s: bad stack size `%s', using default\n", self,
		  sz);
	  stackmax = STACKMAX;
	}
      else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid size `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_MEMSIZE: {
      int sz;
      if (optarg && getintarg(optarg, &sz) && sz >= 0) {
	if (sz == 0 || sz >= MEMMIN)
	  memmax = sz;
	else {
	  fprintf(stderr, "%s: bad memory size `%s', using default\n", self,
		  sz);
	  memmax = MEMMAX;
	}
	lastblksz = memmax % XBLKSZ;
	maxnblks = memmax/XBLKSZ+((memmax <= 0||lastblksz==0)?0:1);
	if (lastblksz == 0) lastblksz = XBLKSZ;
      } else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid size `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case 'd':
      debug = 1;
      break;
    case 'e':
      eflag = 1;
      break;
    case 'h':
      hflag = 1;
      break;
    case 'i':
      iflag = 1;
      break;
    case 'q':
      qflag = 1;
      break;
    case 'c':
    case 's':
      /* these will be taken care of later ... */
      batch = 1;
      break;
    case 'V':
      Vflag = 1;
      break;
    /* qc options: */
    case QC_NO_PRELUDE:
      prelude = NULL;
      if (pass) qcarg("--no-prelude");
      break;
    case QC_PRELUDE:
      prelude = optarg?optarg:prelude;
      if (pass) {
	qcarg("--prelude");
	qcarg(prelude);
      }
      break;
    case 'n':
      /* ignored; we need the code file ;-) */
      break;
    case 'l':
      if (pass) {
	list = optarg?optarg:list;
	qcarg("-l");
	qcarg(list);
      }
      break;
    case 'o':
      code = optarg?optarg:code;
      oset = 1;
      if (pass) {
	qcarg("-o");
	qcarg(code);
      }
      break;
    case 'p':
      if (optarg && !pass) {
	change_qpath(optarg);
	if (!qpath) fatal("memory overflow");
      }
      break;
    case 't':
      if (pass) {
	hsz = optarg?optarg:hsz;
	qcarg("-t");
	qcarg(hsz);
      }
      break;
    case 'v':
      if (pass) qcarg("-v");
      break;
    case 'w':
      if (pass) qcarg("-w");
      break;
    default:
      exit(1);
    }
}

static int sargc = 0;
static char **sargv = NULL;

static void
get_source_opts(FILE *fp)
{
  char s[MAXSTRLEN];
  int i;

  sargc = 1;
  sargv = aalloc(1, sizeof(char*));
  *sargv = strdup(self);
  while (!feof(fp) && !ferror(fp) &&
	 fgets(s, MAXSTRLEN, fp)) {
    int l = strlen(s);
    if (l > 0 && s[l-1] == '\n') s[l-1] = '\0', l--;
    if (l == 0)
      continue;
    else if (strncmp(s, "#!", 2) == 0)
      if (isspace(s[2])) {
	char *p = s+3;
	while (isspace(*p)) p++;
	sargv = arealloc(sargv, sargc, 1, sizeof(char*));
	sargv[sargc++] = strdup(p);
      }else
	continue;
    else
      break;
  }
  sargv = arealloc(sargv, sargc, 1, sizeof(char*));
  sargv[sargc] = NULL;
}

/* spawn child processes */

#ifndef _WIN32
static volatile int pid = 0;
static RETSIGTYPE
spawn_term_handler(int sig)
/* pass termination signals to child process */
{
  if (pid) kill(pid, sig);
  SIGHANDLER_RETURN(0);
}
#endif

static
spawn(char *prog, char *argv[])
{
  RETSIGTYPE (*oldint)(), (*oldterm)(), (*oldhup)();
  int status;
#ifdef _WIN32
  oldint = sigint(SIG_IGN);
  status = spawnvp(P_WAIT, prog, argv);
  if (status < 0)
    fatal("exec failed -- check installation");
  else {
    sigint(oldint);
    return status;
  }
#else
  oldterm = sigterm(SIG_IGN);
  switch ((pid = fork())) {
  case 0:
    execvp(prog, argv);
  case -1:
    fatal("exec failed -- check installation");
  }
  oldint = sigint(SIG_IGN);
  sigterm(spawn_term_handler);
  oldhup = sighup(spawn_term_handler);
  waitpid(pid, &status, 0);
  sigint(oldint);
  sigterm(oldterm);
  sighup(oldhup);
  return status;
#endif
}

/* rerun the interpreter */

static resolve(), init_dlls();
extern int ximpsz;
extern char **ximp;

int rerun(int argc, char **argv)
{
  char *_source = (argv&&argc>0)?argv[0]:source;
  char fname[MAXSTRLEN], fname2[MAXSTRLEN], msg[MAXSTRLEN];
  FILE *fp = NULL;
  int i, nargs;
  EXPR *args, *in, *out, *err;

  /* check the new source script */
  if (!_source || !*_source ||
      chkfile(searchlib(fname, _source)) &&
      (fp = fopen(fname, "rb")) != NULL ||
      chkfile(searchlib(fname, strcat(strcpy(fname2, _source), ".q"))) &&
      (fp = fopen(fname, "rb")) != NULL) {
    int res;
    if (!fp || !(res = iscode(fp))) {
      /* check whether the file compiles ok */
      if (fp) fclose(fp);
      set_qcarg(qcargc-2, qpath);
      set_qcarg(qcargc-1, (_source||!ximpsz)?_source:"");
      if (!oset) {
	code = tmpnam(NULL);
	set_qcarg(qcargc-4, code);
      }
      for (i = 0; i < ximpsz; i++)
	qcarg(ximp[i]);
      remove(code);
      if (spawn(qcprog, qcargv)) {
	sprintf(msg, qmmsg[COMPILE_ERR],
		(_source && *_source)?_source:"script");
	error(msg);
	for (i = 0; i < ximpsz; i++)
	  free(qcargv[--qcargc]), qcargv[qcargc] = NULL;
	set_qcarg(qcargc-1, source);
	remove(code);
	return 0;
      } else
	for (i = 0; i < ximpsz; i++)
	  free(qcargv[--qcargc]), qcargv[qcargc] = NULL;
      source = _source;
      donecompile = 1;
    } else {
      fclose(fp);
      if (res == -1) {
	error(qmmsg[FILE_FORMAT_ERR]);
	return 0;
      }
      source = _source;
      code = source;
      if (ximpsz) {
	int i;
	for (i = 0; i < ximpsz; i++)
	  free(ximp[i]);
	ximpsz = 0;
      }
    }
  } else {
    sprintf(msg, qmmsg[FILE_NOT_FOUND], _source);
    errno = 0;
    error(msg);
    return 0;
  }
  /* source exitrc file if necessary */
  if (doexitrc && exitrc && chkfile(expand(fname, exitrc)) &&
      (fp = fopen(fname, "r"))) {
    fclose(fp);
    parsesrc(fname, 1);
    if (quitflag || thr0->qmstat == QUIT) exit(0);
  }
  /* kill all threads and wait for them to finish */
  kill_threads(); wait_threads();
  /* purge all variables (except builtins) */
  qmfree(thr0, symtb[DEFVAROP].x);
  symtb[DEFVAROP].x = NULL;
  symtb[DEFVAROP].flags &= ~MODIF;
  for (i = BUILTIN; i<symtbsz+tmptbsz; i++)
    if (symtb[i].x) {
      qmfree(thr0, symtb[i].x);
      symtb[i].x = NULL;
      symtb[i].flags &= ~MODIF;
    }
  /* we only keep the ARGS variable if we have been invoked without args */
  if (argc > 0 && symtb[ARGSOP].x) {
    qmfree(thr0, symtb[ARGSOP].x);
    symtb[ARGSOP].x = NULL;
  } else {
    /* record the number of arguments (needed below) */
    EXPR *x = symtb[ARGSOP].x;
    nargs = 0;
    while (x->fno == CONSOP) {
      nargs++;
      x = x->data.args.x2;
    }
  }
  /* clean up the main stack and the heap */
  clear(1);
  /* save the builtin variables s.t. we can recover them after the symbol
     table has been reinitialized */
  in = symtb[INPUTOP].x;
  out = symtb[OUTPUTOP].x;
  err = symtb[ERROROP].x;
  if (argc <= 0) args = symtb[ARGSOP].x;
  /* unload dlls */
  unload_dlls();
  /* purge the code table */
  if (codesp) free(codesp); codesp = NULL; codespsz = 0;
  if (strsp) free(strsp); strsp = NULL; strspsz = tmpspsz = 0;
  atmpspsz = TMPSPSZ;
  if (limbsp) free(limbsp); limbsp = NULL; limbspsz = 0;
  if (hashtb) free(hashtb); hashtb = NULL; hashtbsz = 0;
  if (symtb) free(symtb); symtb = NULL; symtbsz = tmptbsz = 0;
  atmptbsz = TMPTBSZ;
  if (statetb) free(statetb); statetb = NULL; statetbsz = 0;
  if (transtb) free(transtb); transtb = NULL; transtbsz = 0;
  if (roffstb) free(roffstb); roffstb = NULL; roffstbsz = 0;
  if (matchtb) free(matchtb); matchtb = NULL; matchtbsz = 0;
  if (inittb) free(inittb); inittb = NULL; inittbsz = 0;
  if (modtb) free(modtb); modtb = NULL;
  if (fnametb) free(fnametb); fnametb = NULL; modtbsz = 0;
  /* reinitialize */
  readtables();
  resolve();
  reinit();
  if (donecompile) remove(code);
  donecompile = 0;
  /* standard I/O streams */
  symtb[INPUTOP].x = in;
  symtb[OUTPUTOP].x = out;
  symtb[ERROROP].x = err;
  /* ARGS variable */
  if (argc > 0) {
    int count = 0; char **argv0 = argv;
    while (argc-- > 0) {
      char *s;
      if (*argv)
	s = *argv++;
      else
	s = strdup("");
      if (!s)
	fatal("memory overflow");
      else if (!pushstr(thr0, s))
	fatal("stack overflow");
      else
	count++;
    }
    free(argv0);
    if (!pushfun(thr0, NILOP))
      fatal("stack overflow");
    while (count-- > 0)
      if (!pushfun(thr0, CONSOP))
	fatal("memory overflow");
    symtb[ARGSOP].x = (void*) thr0->xsp[-1];
    thr0->xsp = thr0->xst;
  } else {
    /* KLUDGE ALERT: we have to fix the nil cell in the last component, since
       this is a preallocated function symbol and the symbol array may have
       moved after reinit() */
    if (nargs == 0)
      symtb[ARGSOP].x = qmnew(funexpr(thr0, NILOP));
    else {
      EXPR *x = args;
      while (nargs-- > 1)
	x = x->data.args.x2;
      x->data.args.x2 = qmnew(funexpr(thr0, NILOP));
      symtb[ARGSOP].x = args;
    }
  }
  /* `which' */
  if (which) free(which);
  if (mainno == -1)
    which = "";
  else
    which = strsp+fnametb[mainno];
  which = strdup(which);
  if (!which) fatal("memory overflow");
  /* module initializations */
  init_dlls();
  /* script initialization code */
  errno = 0;
  if (inittbsz) {
    int i = 0, done = 1;
    start_init();
    while (done && i < inittbsz) {
      if (!evaldef(inittb[i]))
	done = checkbrk || thr0->qmstat == HALT;
      i++;
    }
    end_init();
    if (quitflag || thr0->qmstat == QUIT)
      exit(0);
    else if (thr0->qmstat != OK && thr0->qmstat != HALT) {
      error(qmmsg[thr0->qmstat]);
      if (thr0->qmstat == XCEPT && thr0->xsp > thr0->xst) {
	printx(thr0->xsp[-1]); printf("\n");
      }
    }
    clear(0);
    clearerr(stdin);
  }
  /* source initrc file */
  doexitrc = 1;
  if (iflag && initrc && chkfile(expand(fname, initrc)) &&
      (fp = fopen(fname, "r"))) {
    fclose(fp);
    parsesrc(fname, 1);
    if (quitflag || thr0->qmstat == QUIT) exit(0);
  }
  /* notify gnu server */
  if (gflag)
    if (source)
      gcmd_s("q-run-cmd", source);
    else
      gcmd_b("q-run-cmd", 0);
#if defined (USE_READLINE)
  clear_fsym_table();
  clear_vsym_table();
#endif
  new_xprompt();
  errno = 0;
  return 1;
}

/* additional interface functions required by libq */

EXPR *_qinter_intexpr(long i)
{
  return intexpr(get_thr(), i);
}

EXPR *_qinter_mpzexpr(mpz_t z)
{
  return mpzexpr(get_thr(), z);
}

EXPR *_qinter_floatexpr(double f)
{
  return floatexpr(get_thr(), f);
}

EXPR *_qinter_strexpr(char *s)
{
  return strexpr(get_thr(), s);
}

EXPR *_qinter_fileexpr(FILE *fp)
{
  return fileexpr(get_thr(), fp);
}

EXPR *_qinter_pipeexpr(FILE *fp)
{
  return pipeexpr(get_thr(), fp);
}

EXPR *_qinter_funexpr(int fno)
{
  return funexpr(get_thr(), fno);
}

EXPR *_qinter_usrexpr(int type, void *vp)
{
  return usrexpr(get_thr(), type, vp);
}

EXPR *_qinter_vectexpr(int n, EXPR **xv)
{
  return vectexpr(get_thr(), n, xv);
}

EXPR *_qinter_mpz_floatexpr(double f)
{
  mpz_t z;
  double ip, fp;
  int sz;
  fp = modf(f, &ip);
  if (ip < 0) ip = -ip;
  sz = log(ip)/log(2)+1;
  if (sz < 0) return 0;
  sz = sz/(CHAR_BIT*sizeof(mp_limb_t)) + 2;
  mpz_init(z);
  if (z->_mp_d && my_mpz_realloc(z, sz)) {
    EXPR *x;
    int __sz;
    mpz_set_d(z, f);
    __sz = mpz_size(z);
    if (__sz < sz && !my_mpz_realloc(z, __sz)) {
      get_thr()->qmstat = MEM_OVF;
      return NULL;
    }
    x = mpzexpr(get_thr(), z);
    return x;
  } else {
    get_thr()->qmstat = MEM_OVF;
    return NULL;
  }
}

EXPR *_qinter_consexpr(int fno, EXPR *x1, EXPR *x2)
{
  if (fno == APPOP) {
    THREAD *thr = get_thr();
    bool actmode = thr->mode;
    EXPR *ret;
    /* force quote mode on */
    thr->mode = 1;
    ret = consexpr(thr, fno, x1, x2);
    thr->mode = actmode;
    return ret;
  } else
    return consexpr(get_thr(), fno, x1, x2);
}

int getintexpr(EXPR *x, long *i)
{
  if (x->fno == INTVALOP && my_mpz_fits_slong_p(x->data.z)) {
    *i = mpz_get_si(((EXPR*)x)->data.z);
    return 1;
  } else
    return 0;
}

int getmpzexpr(EXPR *x, mpz_t z)
{
  if (x->fno == INTVALOP) {
    memcpy(z, ((EXPR*)x)->data.z, sizeof(mpz_t));
    return 1;
  } else
    return 0;
}

int getmpz_floatexpr(EXPR *x, double *f)
{
  mpz_t z;
  if (x->fno == INTVALOP) {
    *f = mpz_get_d(((EXPR*)x)->data.z);
    return 1;
  } else
    return 0;
}

EXPR *_qinter_eval(EXPR *x)
{
  THREAD *thr = get_thr();
  if (x && eval(thr, x)) {
    thr->xsp[-1]->refc--;
    return *--thr->xsp;
  } else
    return NULL;
}

void _qinter_free(EXPR *x)
{
  qmfree(get_thr(), x);
}

int issym(int sym)
{
  return sym >= BINARY && sym < symtbsz+tmptbsz &&
    !(symtb[sym].flags & TSYM);
}

int istype(int type)
{
  return type >= BINARY && type < symtbsz &&
    (symtb[type].flags & TSYM);
}

int isusrtype(int type)
{
  return istype(type) && (symtb[type].flags & EXT);
}

void _qinter_error(void)
{
  THREAD *thr = get_thr();
  if (thr->qmstat == OK) thr->qmstat = EXT_ERR;
}

/* dl interface */

char **dll_name = NULL;
lt_dlhandle *dll_handle = NULL;
void (**dll_init)() = NULL, (**dll_fini)() = NULL;
thread_atfork_t *dll_atfork = NULL;

void thread_atfork(void (*prepare)(void), void (*parent)(void),
		   void (*child)(void), int modno)
{
  dll_atfork[modno].prepare = prepare;
  dll_atfork[modno].parent = parent;
  dll_atfork[modno].child = child;
}

extern __libq_init();

static resolve()
{
  int fno, count;

  /* initialize */

  if (modtbsz <= 0) return 1;

  dll_name = (char**)calloc(modtbsz, sizeof(char*));
  dll_handle = (lt_dlhandle*)calloc(modtbsz, sizeof(void*));
  dll_init = (void(**)())calloc(modtbsz, sizeof(void(**)()));
  dll_fini = (void(**)())calloc(modtbsz, sizeof(void(**)()));
  dll_atfork = (thread_atfork_t*)calloc(modtbsz, sizeof(thread_atfork_t));

  if (!dll_name || !dll_handle || !dll_init || !dll_fini || !dll_atfork)
    fatal("memory overflow");

  /* set up libq interface (back links into the interpreter) */

  __libq_init(_qinter_intexpr,
	      _qinter_mpzexpr,
	      _qinter_mpz_floatexpr,
	      _qinter_floatexpr,
	      _qinter_strexpr,
	      _qinter_fileexpr,
	      _qinter_pipeexpr,
	      _qinter_funexpr,
	      _qinter_usrexpr,
	      _qinter_consexpr,
	      _qinter_vectexpr,

	      getintexpr,
	      getmpzexpr,
	      getmpz_floatexpr,
	    
	      _qinter_eval,
	    
	      _qinter_free,

	      issym,
	      istype,
	      isusrtype,
	      getsym,
	      gettype,

	      init_thread,
	      exit_thread,
	      fini_thread,
	      this_thread,
	      release_lock,
	      acquire_lock,
	      acquire_tty,
	      release_tty,
	      thread_atfork,

	      _qinter_error);

  /* resolve external symbols */

  for (fno = BINARY; fno < symtbsz; fno++)
    if (symtb[fno].flags & EXT) {
      int xfno = symtb[fno].xfno, modno = symtb[xfno].modno;
      char *fname = strsp+fnametb[modno];
      char sym[MAXSTRLEN];
      if (!dll_name[modno]) {
	char modname[MAXSTRLEN];
	basename(modname, fname, '.');
	dll_name[modno] = strdup(modname);
	if (!dll_name[modno])
	  fatal("memory overflow");
	else if (!(dll_handle[modno] = lt_dlopenext(dll_name[modno])))
	  fprintf(stderr, "%s: error loading module\n", fname);
	else {
	  char initmod[MAXSTRLEN], init[MAXSTRLEN], fini[MAXSTRLEN];
	  void (*__initmod)();
	  sprintf(initmod, "__%s__initmod", dll_name[modno]);
	  sprintf(init, "__%s__init", dll_name[modno]);
	  sprintf(fini, "__%s__fini", dll_name[modno]);
	  if (!(__initmod = lt_dlsym(dll_handle[modno], initmod))) {
	    fprintf(stderr, "%s: invalid module header\n", fname);
	    lt_dlclose(dll_handle[modno]);
	    dll_handle[modno] = NULL;
	  } else {
	    void (*__init)() = lt_dlsym(dll_handle[modno], init);
	    void (*__fini)() = lt_dlsym(dll_handle[modno], fini);
	    (*__initmod)(modno, gmp_allocate, gmp_reallocate, gmp_free);
	    dll_init[modno] = __init;
	    dll_fini[modno] = __fini;
	  }
	}
      }
      if (symtb[fno].flags & TSYM)
	strcpy(sym, "__D__");
      else
	strcpy(sym, "__F__");
      strcat(strcat(sym, dll_name[modno]), "_");
      strcat(sym, strsp+symtb[xfno].pname);
      if (dll_handle[modno] &&
	  !(symtb[fno].f = lt_dlsym(dll_handle[modno], sym)) &&
	  !(symtb[fno].flags & TSYM)) {
	char pn[MAXSTRLEN];
	fprintf(stderr, "%s: unresolved symbol `%s'\n", fname,
		pname(pn, fno));
      } else if (symtb[fno].f && symtb[fno].argc > maxargs)
	maxargs = symtb[fno].argc;
    }

  /* check for errors */

  for (count = 0, fno = BINARY; fno < symtbsz; fno++)
    if ((symtb[fno].flags & EXT) && !(symtb[fno].flags & TSYM) &&
	!symtb[fno].f)
      count++;
  if (count) {
    fprintf(stderr, "Warning: %d unresolved external symbol%s\n",
	    count, count>1?"s":"");
    fflush(stderr);
    return 0;
  } else
    return 1;
}

static init_dlls()
{
  if (dll_name) {
    int i;
    for (i = 0; i < modtbsz; i++)
      if (dll_name[i] && dll_init[i])
	(*dll_init[i])();
  }
}

static unload_dlls()
{
  if (dll_name) {
    int i;
    for (i = 0; i < modtbsz; i++)
      if (dll_name[i] && dll_fini[i])
	(*dll_fini[i])();
    for (i = 0; i < modtbsz; i++)
      if (dll_name[i] && dll_handle[i])
	lt_dlclose(dll_handle[i]);
    for (i = 0; i < modtbsz; i++)
      if (dll_name[i])
	free(dll_name[i]);
    free(dll_name); dll_name = NULL;
    free(dll_handle); dll_handle = NULL;
    free(dll_init); dll_init = NULL;
    free(dll_fini); dll_fini = NULL;
    free(dll_atfork); dll_atfork = NULL;
  }
}

/* interface to gnuserv */

static gnuclient(char *sexp)
{
  static char *gnuclient_prog = NULL;
  static char *argv[] = { NULL, "-q", "-eval", NULL, NULL };
  if (!gnuclient_prog && !(gnuclient_prog = getenv("GNUCLIENT_PROGRAM")))
    argv[0] = gnuclient_prog = "gnuclient";
  argv[3] = sexp;
  spawn(gnuclient_prog, argv);
}

void gcmd(char *name)
{
  char *sexp = malloc(strlen(name)+3);
  if (sexp) {
    sprintf(sexp, "(%s)", name);
    gnuclient(sexp);
    free(sexp);
  }
}

void gcmd_i(char *name, int i)
{
  char *sexp = malloc(strlen(name)+100);
  if (sexp) {
    sprintf(sexp, "(%s %d)", name, i);
    gnuclient(sexp);
    free(sexp);
  }
}

void gcmd_b(char *name, int b)
{
  char *sexp = malloc(strlen(name)+100);
  if (sexp) {
    sprintf(sexp, "(%s %s)", name, b?"t":"nil");
    gnuclient(sexp);
    free(sexp);
  }
}

void gcmd_s(char *name, char *s)
{
  char *sexp = malloc(strlen(name)+4*strlen(s)+100);
  if (sexp) {
    char *buf = malloc(4*strlen(s)+1);
    if (buf) {
      sprintf(sexp, "(%s \"%s\")", name, pstr(buf, s));
      gnuclient(sexp);
      free(buf);
    }
    free(sexp);
  }
}

/* main program */

main(int argc, char **argv)
{
  int c, bargc = argc, longind;
  char **bargv = argv, *s;
  char fname[MAXSTRLEN], fname2[MAXSTRLEN], msg[MAXSTRLEN];
  char prefix[MAXSTRLEN];
  FILE *fp;

#ifdef _WIN32
  InstallSignalHandler();
#endif

  atexit(exitproc);

  sprintf(fformat, STDFORMAT, fprec);

  LTDL_SET_PRELOADED_SYMBOLS();

  /* get program name: */
  self = argv[0];

  /* check to see whether we were invoked with a pathname; in this case we
     also invoke the compiler from there */
  if (*(dirname(prefix, self)))
    sprintf(qcprog, "%s%s", prefix, "qc");

  /* get environment strings: */
  if ((s = getenv("QPATH")) != NULL)
    init_qpath(s);
  else
    init_qpath(QPATH);
  if (!qpath) fatal("memory overflow");
  
  /* set code file id: */
  sprintf(outid, OUTID, version, sysinfo);
  
  /* initialize the qc command line */
  qcarg(qcprog);

  /* scan the command line to obtain the source/code file name: */
  opterr = 0;
  while ((c = getopt_long(argc, argv, Q_OPTS1,
			  longopts, &longind)) != EOF)
    if (c == 'p' && optarg) {
      /* set the path so that we can find the source file even before the
	 remaining command line options have been parsed */
      change_qpath(optarg);
      if (!qpath) fatal("memory overflow");
    }
  opterr = 1;

  if (argc-optind >= 1) {
    source = argv[optind];
    if (!source) fatal("memory overflow");
    if (!*source) goto opts;
  } else
    goto opts;

  /* check to see whether the source actually is a precompiled bytecode file,
     then we can skip the compilation step */
    
  if (chkfile(searchlib(fname, source)) &&
      (fp = fopen(fname, "rb")) != NULL ||
      chkfile(searchlib(fname, strcat(strcpy(fname2, source), ".q"))) &&
      (fp = fopen(fname, "rb")) != NULL) {
    if (iscode(fp)) {
      fclose(fp);
      code = source;
      docompile = 0;
    } else {
      /* get options from the source file */
      if (!(fp = freopen(fname, "r", fp)))
	/* this shouldn't happen, but Murphy knows ... */
	goto err;
      get_source_opts(fp);
      fclose(fp);
      parse_opts(sargc, sargv, 0);
    }
  } else {
  err:
    sprintf(msg, qmmsg[FILE_NOT_FOUND], source);
    fatal(msg);
  }

 opts:

  /* process command line options: */
  parse_opts(argc, argv, 1);
  argc -= optind, argv += optind;

  if (Vflag) {
    printf(signon, version, sysinfo, year);
    printf(copying);
    printf(helpmsg, self);
    exit(0);
  }
  if (hflag) {
    char p[100];
    printf(usage, self);
    sprintf(opts, Q_OPTMSG, "dec", EXITRC, "std", HISTFILE, HISTMAX,
	    INITRC, MEMMAX, QPATH, pstr(p, PROMPT), STACKMAX, HASHTBSZ);
    fputs(opts, stdout);
    exit(0);
  }

  /* if no code file name is set, make a temporary file name */
  if (!oset) {
    char *_code = tmpnam(NULL);
    qcarg("-o");
    qcarg(_code);
    if (docompile) code = _code;
  }

  /* make sure that -p is the last option on the qc command line before
     the source file, rerun() depends on it */
  qcarg("-p");
  qcarg(qpath);
  qcarg(source);

  if (!docompile) goto run;

  /* if we come here, we have a source file which we compile first */

  remove(code);
  if (spawn(qcprog, qcargv))
    exit(1);
  else
    donecompile = 1;

 run:
  /* now we should have a bytecode file which we can finally run */

  /* make dynamic copies of the setup variables */
  prompt = strdup(prompt);
  histfile = strdup(histfile);
  if (!(prompt && histfile))
    fatal("memory overflow");

  /* initialize: */
  if (lt_dlinit() || lt_dlsetsearchpath(qpath)) 
    fatal("error initializing module interface");
  readtables();
  resolve();
  init();
  /* install gmp memory handlers */
  mp_set_memory_functions(gmp_allocate, gmp_reallocate, gmp_free);

  /* if all went well, we can now safely remove the code file if we created
     it */
  if (donecompile) remove(code);
  donecompile = 0;

  /* get the file name of the main module */
  if (mainno == -1)
    which = "";
  else
    which = strsp+fnametb[mainno];
  which = strdup(which);
  if (!which) fatal("memory overflow");

  /* set up standard devices: */
  symtb[INPUTOP].x = (void*) fileexpr(thr0, stdin);
  ((EXPR*)symtb[INPUTOP].x)->refc = 1;
  symtb[OUTPUTOP].x = (void*) fileexpr(thr0, stdout);
  ((EXPR*)symtb[OUTPUTOP].x)->refc = 1;
  symtb[ERROROP].x = (void*) fileexpr(thr0, stderr);
  ((EXPR*)symtb[ERROROP].x)->refc = 1;

  /* process remaining command line arguments: */
  {	int count = 0;
  while (argc-- > 0) {
    char *s;
    if (*argv)
      s = strdup(*argv++);
    else
      s = strdup("");
    if (!s)
      fatal("memory overflow");
    else if (!pushstr(thr0, s))
      fatal("stack overflow");
    else
      count++;
  }
  if (!pushfun(thr0, NILOP))
    fatal("stack overflow");
  while (count-- > 0)
    if (!pushfun(thr0, CONSOP))
      fatal("memory overflow");
  symtb[ARGSOP].x = (void*) thr0->xsp[-1];
  thr0->xsp = thr0->xst;
  }

  if (!(batch && !iflag) && !qflag && (iflag || isatty(fileno(stdin)) &&
				       isatty(fileno(stdout)))) {
    /* sign-on: */
    printf(signon, version, sysinfo, year);
    printf(terms);
  }

  /* install signal handlers: */
  sigint(break_handler); sigterm(term_handler); sighup(term_handler);
#ifdef SIGTTIN
  syssignal(SIGTTIN, tty_handler);
#endif
#ifdef SIGTTOU
  syssignal(SIGTTOU, tty_handler);
#endif

  /* handle fatal program errors */
#ifdef SIGFPE
  syssignal(SIGFPE, segv_handler);
#endif
#ifdef SIGILL
  syssignal(SIGILL, segv_handler);
#endif
#ifdef SIGSEGV
  syssignal(SIGSEGV, segv_handler);
#endif
#ifdef SIGBUS
  syssignal(SIGBUS, segv_handler);
#endif
#ifdef SIGTRAP
  syssignal(SIGTRAP, segv_handler);
#endif
    
  /* initialize random seed: */
  seedMT(((unsigned long)time(NULL)) << 1 | 1);

  /* initialize external modules: */
  init_dlls();

  /* execute the script's initialization code (def and undef): */
  errno = 0;
  if (inittbsz) {
    int i = 0, done = 1;
    start_init();
    while (done && i < inittbsz) {
      if (!evaldef(inittb[i]))
	done = checkbrk || thr0->qmstat == HALT;
      i++;
    }
    end_init();
    if (quitflag || thr0->qmstat == QUIT)
      exit(0);
    else if (thr0->qmstat != OK && thr0->qmstat != HALT) {
      error(qmmsg[thr0->qmstat]);
      if (thr0->qmstat == XCEPT && thr0->xsp > thr0->xst) {
	printx(thr0->xsp[-1]); printf("\n");
      }
    }
    clear(0);
    clearerr(stdin);
  }

  /* execute -c and -s options, if any */
  if (!iflag && batch) {
    if (sargc && sargv) {
      optind = 1;
      while ((c = getopt_long(sargc, sargv, Q_OPTS1, longopts,
			      &longind)) != EOF) {
	switch (c) {
	case 'c':
	  if (eflag) echo(optarg);
	  parsex(optarg);
	  break;
	case 's':
	  parsesrc(optarg, 1);
	  break;
	default:
	  break;
	}
	if (quitflag || thr0->qmstat == QUIT) exit(0);
      }
    }
    optind = 1;
    while ((c = getopt_long(bargc, bargv, Q_OPTS1, longopts,
			    &longind)) != EOF) {
      switch (c) {
      case 'c':
	if (eflag) echo(optarg);
	parsex(optarg);
	break;
      case 's':
	parsesrc(optarg, 1);
	break;
      default:
	break;
      }
      if (quitflag || thr0->qmstat == QUIT) exit(0);
    }
    exit(0);
  }

  /* if we come here, we are running interactively; source the initrc file
     and enter the evaluation loop */

  quitflag = 0;
  thr0->qmstat = OK;
  doexitrc = 1;
  if (initrc && chkfile(expand(fname, initrc)) && (fp = fopen(fname, "r"))) {
    fclose(fp);
    parsesrc(fname, 1);
    if (quitflag || thr0->qmstat == QUIT) exit(0);
  }

  iflag = iflag || (isatty(fileno(stdin)) &&
		    isatty(fileno(stdout)));
  quitflag = 0;
  thr0->qmstat = OK;
#ifdef USE_READLINE
  use_readline = (iflag && !norl && isatty(fileno(stdin)));
  init_readline();
#endif

  /* the read/eval/print loop */
  errno = 0;
  while (1) {
    char *buf;
    brkflag = 0;
    /* release the lock on the interpreter while we're waiting for input */
    release_lock();
    /* get hold of the input line */
    acquire_input();
    clearerr(stdin);
    if (feof(stdin) || !(buf = getline(stdin, prompt))) {
      if (iflag) putchar('\n');
      release_input();
      acquire_lock();
      break;
    }
    release_input();
    acquire_lock();
    if (!iflag && eflag) echo(buf);
    parsex(buf);
    if (quitflag || thr0->qmstat == QUIT) break;
    free(buf);
    fflush(stderr);
    fflush(stdout);
  }

  fflush(stderr);
  fflush(stdout);

  thr0->qmstat = OK;
  exit(0);
}
