
/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qcdefs.h"

/*

void left_hand_side(x)
EXPR *x;
- compile the left-hand side

void right_hand_side(x)
EXPR *x;
- compile the right-hand side

void qualifier(x)
EXPR *x;
- compile the qualifier

void where_clause(x,y)
EXPR *x, *y;
- compile a where clause

void definition(x,y)
EXPR *x, *y;
- compile a definition

void write_TA()
- write the constructed TA to the code file

*/

/*
 * We employ the incremental technique described in Graef: Left-To-Right Tree
 * Pattern Matching, Proc. RTA 1991, Springer 1991 (LNCS 488) to construct a
 * tree automaton (TA) for the left-hand sides of equations. The technique is
 * slightly modified in order to cope with type guards. The basic outline
 * of the technique is as follows. Initially, the automaton is empty. For
 * each rule, we produce a trie from the lhs pattern of the rule (considered
 * as a string of variable and function symbols). This trie is then merged
 * with the TA obtained so far. The latter process is similar to merging two
 * deterministic finite automata, but it also takes into account the
 * variables (see the mergestatep() routine below).
 *
 * To simplify matters, the automaton only has transitions on variable and
 * function symbols. Other values (number and string constants), for the
 * purpose of pattern matching, are treated as (typed) variables; during code
 * generation, we will generate the appropriate Q code queries for verifying
 * at runtime that these subexpressions actually have the specified constant
 * values.
 *
 * The following tree data structure is convenient to use during the
 * construction of the automaton (note that the output TA data structure
 * does not allow to manipulate multiple TAs, which is required during the
 * construction). The constructed TA is stored in the code table when
 * all rules have been compiled (see write_TA() below).
 *
 */

typedef struct staten *Statep;
typedef struct transn *Transp;
typedef struct rulen *Rulep;

typedef struct staten {
	int             sno;	/* state number */
	int		ntrans;	/* number of transitions */
	int		trans;	/* offset in transition table */
	int		nrules; /* number of matched rules */
	int		roffs;	/* offset in rule offset table */
	short		minc;	/* min arg count for NONE type app */
	short		maxc;	/* max arg count for NONE type app */
	Statep		s;	/* next state for NONE type app */
	Transp          tlist;	/* transitions, NULL if none */
	Rulep           rlist;	/* accepted rules, NULL in nonfinal state */
}               Staten;

typedef struct transn {
	short		type;	/* type tag */
	short           fno;	/* function number (0 if variable) */
	short		argc;	/* max arg count for application node */
	Statep          nextst;	/* pointer to next state */
	Transp          next;	/* other transitions in this state, NULL if
				 * none */
}               Transn;

typedef struct rulen {
	int		prio;	/* priority of the rule */
	int		addr;	/* entry point of the rule */
	Rulep           next;	/* other rules (NULL if none) */
}               Rulen;

static Statep   start = NULL;	/* pointer to start state of the TA */

static int
arity(int fno)
{
  switch (fno) {
  case CONSOP:
  case PAIROP:
  case APPOP:
    return (2);
  default:
    return (0);
  }
}

static Statep
newstatep(void)
{
  Statep          s = (Statep) calloc(1, sizeof(Staten));

  if (s == NULL)
    fatal(qcmsg[MEM_OVF]);
  else
    return (s);
}

static Transp
newtransp(void)
{
  Transp          t = (Transp) calloc(1, sizeof(Transn));

  if (t == NULL)
    fatal(qcmsg[MEM_OVF]);
  else
    return (t);
}

static Rulep
newrulep(void)
{
  Rulep           r = (Rulep) calloc(1, sizeof(Rulen));

  if (r == NULL)
    fatal(qcmsg[MEM_OVF]);
  else
    return (r);
}

static void freetransp(Transp), freerulep(Rulep);

static void
freestatep(Statep s)
{
  if (s != NULL) {
    freestatep(s->s);
    freetransp(s->tlist);
    freerulep(s->rlist);
    free(s);
  }
}

static void
freetransp(Transp t)
{
  if (t != NULL) {
    freestatep(t->nextst);
    freetransp(t->next);
    free(t);
  }
}

static void
freerulep(Rulep r)
{
  if (r != NULL) {
    freerulep(r->next);
    free(r);
  }
}

static Statep 		s0, s;

static void
mkstatep1(EXPR *x)
{
  Transp		t;
  s->tlist = t = newtransp();
  t->type = x->type==NONE||x->argc==x->expc?x->type:0;
  t->argc = x->argc;
  t->fno = x->fno < RESERVED?0:x->fno;
  s = t->nextst = newstatep();
  if (arity(x->fno) > 0) {
    mkstatep1(x->x1);
    mkstatep1(x->x2);
  }
}

static Transp cptransp0(Transp);
static Rulep cprulep(Rulep);

static Statep
cpstatep0(Statep s)
{
  Statep s1;

  if (!s)
    return (NULL);
  else {
    s1 = newstatep();
    s1->tlist = cptransp0(s->tlist);
    s1->rlist = cprulep(s->rlist);
    return (s1);
  }
}

static Transp
cptransp0(Transp t)
{
  Transp          t1;

  if (!t)
    return (NULL);
  else {
    t1 = newtransp();
    t1->type = t->type==NONE?0:t->type;
    t1->fno = t->fno;
    t1->argc = t->argc;
    t1->nextst = cpstatep0(t->nextst);
    t1->next = cptransp0(t->next);
    return (t1);
  }
}

static Statep
scanstatep(Statep s)
{
  if (!s || !s->tlist)
    return s;
  if (s->tlist->type == NONE) {
    s->tlist->type = 0;
    s->s = cpstatep0(s->tlist->nextst);
    s->minc = s->maxc = s->tlist->argc;
  }
  s->tlist->nextst = scanstatep(s->tlist->nextst);
  return s;
}

static Statep
mkstatep(EXPR *x)
{
  s0 = s = newstatep();
  mkstatep1(x);
  /* add rule list to final state: */
  s->rlist = newrulep();
  s->rlist->prio = prio;
  s->rlist->addr = codespsz;
  /* replace NONE apps */
  return scanstatep(s0);
}

static Transp cptransp(Transp);
static Rulep cprulep(Rulep);

static Statep
cpstatep(Statep s)
{
  Statep s1;

  if (!s)
    return (NULL);
  else {
    s1 = newstatep();
    s1->s = cpstatep(s->s);
    s1->minc = s->minc;
    s1->maxc = s->maxc;
    s1->tlist = cptransp(s->tlist);
    s1->rlist = cprulep(s->rlist);
    return (s1);
  }
}

static Statep
vcpstatep(Statep s, int n)
{
  Statep          s0, s1;
  Transp		t;

  if (n <= 0)
    return (cpstatep(s));
  else {
    s0 = s1 = newstatep();
    t = s1->tlist = newtransp();
    for (n--; n; n--, t = s1->tlist = newtransp())
      s1 = t->nextst = newstatep();
    t->nextst = cpstatep(s);
    return (s0);
  }
}

static Transp
cptransp(Transp t)
{
  Transp          t1;

  if (!t)
    return (NULL);
  else {
    t1 = newtransp();
    t1->type = t->type;
    t1->fno = t->fno;
    t1->argc = t->argc;
    t1->nextst = cpstatep(t->nextst);
    t1->next = cptransp(t->next);
    return (t1);
  }
}

static Rulep
cprulep(Rulep r)
{
  Rulep           r1;

  if (!r)
    return (NULL);
  else {
    r1 = newrulep();
    r1->prio = r->prio;
    r1->addr = r->addr;
    r1->next = cprulep(r->next);
    return (r1);
  }
}

static Transp mergetranspv(Transp, Transp*),
  scantransp(Transp, int, int, Statep);
static Rulep mergerulep(Rulep, Rulep);

static Statep
mergestatep(Statep s1, Statep s2)
{
  if (!s1)
    return (cpstatep(s2));
  else if (!s2)
    return (s1);
  else {
    Transp t1 = NULL, t2 = NULL;
    /* merge the transitions of s1 and s2: */
    if (s1->s)
      t1 = scantransp(s2->tlist, s1->minc, s1->maxc,
		      s1->s);
    if (s2->s)
      t2 = scantransp(s1->tlist, s2->minc, s2->maxc,
		      s2->s);
    s1->s = mergestatep(s1->s, s2->s);
    s1->minc = s1->minc<=s2->minc?s1->minc:s2->minc;
    s1->maxc = s1->maxc>=s2->maxc?s1->maxc:s2->maxc;
    if (t1)
      s1->tlist = mergetranspv(s1->tlist, &t1);
    if (t2)
      s2->tlist = mergetranspv(s2->tlist, &t2);
    freetransp(t1); freetransp(t2);
    s1->tlist = mergetranspv(s1->tlist, &s2->tlist);
    /* merge the rule lists: */
    s1->rlist = mergerulep(s1->rlist, s2->rlist);
    return (s1);
  }
}

#define transcmp(t1,t2) (t1->fno==t2->fno?t2->type-t1->type:t1->fno-t2->fno)

static
subtype(int type1, int type2)
{
   for (; type1 && type1 != type2; type1 = symtb[type1].type)
   	;
   return type1 == type2;
}

static Transp
mergetranspv1(Transp t, int type, Statep s)
{
  if (!t)
    return NULL;
  else if (subtype(t->type, type) && (t->fno || t->type != type)) {
    Statep s1 = vcpstatep(s, arity(t->fno));
    t->nextst = mergestatep(t->nextst, s1);
    freestatep(s1);
  }
  t->next = mergetranspv1(t->next, type, s);
  return t;
}

static Transp
mergetransp(Transp t1, Transp t2)
{
  int cmp;
  if (!t1)
    return cptransp(t2);
  else if (!t2)
    return t1;
  else if ((cmp = transcmp(t1, t2)) < 0) {
    t1->next = mergetransp(t1->next, t2);
    return t1;
  } else if (cmp > 0) {
    Transp t = newtransp();
    t->type = t2->type;
    t->fno = t2->fno;
    t->argc = t2->argc;
    t->nextst = cpstatep(t2->nextst);
    t->next = mergetransp(t1, t2->next);
    return t;
  } else {
    t1->argc = t1->argc>=t2->argc?t1->argc:t2->argc;
    t1->nextst = mergestatep(t1->nextst, t2->nextst);
    t1->next = mergetransp(t1->next, t2->next);
    return t1;
  }
}

static Transp
mergetranspv(Transp t1, Transp *t2)
{
  Transp t;
  for (t = *t2; t && !t->fno; t = t->next)
    t1 = mergetranspv1(t1, t->type, t->nextst);
  for (t = t1; t && !t->fno; t = t->next)
    *t2 = mergetranspv1(*t2, t->type, t->nextst);
  t1 = mergetransp(t1, *t2);		
  return t1;
}

static Transp
addapp(Transp t, int type, int argc, Statep s)
{
  static Transp t1;
  if (!t)
    t = t1 = newtransp();
  else
    t1 = t1->next = newtransp();
  t1->type = type;
  t1->fno = APPOP;
  t1->argc = argc;
  t1->nextst = cpstatep(s);
  return t;
}

static Transp
mergetransp1(Transp t1, Transp t2)
{
  int cmp;
  if (!t1)
    return cptransp(t2);
  else if (!t2)
    return t1;
  else if ((cmp = transcmp(t1, t2)) < 0) {
    t1->next = mergetransp1(t1->next, t2);
    return t1;
  } else if (cmp > 0) {
    Transp t = newtransp();
    t->type = t2->type;
    t->fno = t2->fno;
    t->argc = t2->argc;
    t->nextst = cpstatep(t2->nextst);
    t->next = mergetransp1(t1, t2->next);
    return t;
  } else {
    t1->argc = t1->argc>=t2->argc?t1->argc:t2->argc;
    t1->next = mergetransp1(t1->next, t2->next);
    return t1;
  }
}

static Transp
scantransp(Transp t, int minc, int maxc, Statep s)
{
  Transp t1, tt1 = NULL, tt2 = NULL;
  for (t1 = t; t1 && !t1->fno; t1 = t1->next)
    if (t1->type && symtb[t1->type].argc >= minc)
      tt1 = addapp(tt1, t1->type, maxc, s);
  for (; t1 && t1->fno <= APPOP; t1 = t1->next)
    if (t1->fno == APPOP && t1->type && t1->argc >= minc)
      tt2 = addapp(tt2, t1->type, maxc, s);
  if (tt1) {
    tt1 = mergetransp1(tt1, tt2);
    freetransp(tt2);
    return tt1;
  } else
    return tt2;
}

static Rulep
mergerulep(Rulep r1, Rulep r2)
{
  Rulep           r;

  /* mergesort: */
  if (r1 == NULL)
    return (cprulep(r2));
  else if (r2 == NULL)
    return (r1);
  else if (r1->prio > r2->prio) {
    r1->next = mergerulep(r1->next, r2);
    return (r1);
  } else if (r1->prio < r2->prio) {
    r = newrulep();
    r->prio = r2->prio;
    r->addr = r2->addr;
    r->next = mergerulep(r1, r2->next);
    return (r);
  } else if (r1->addr < r2->addr) {
    r1->next = mergerulep(r1->next, r2);
    return (r1);
  } else if (r1->addr > r2->addr) {
    r = newrulep();
    r->prio = r2->prio;
    r->addr = r2->addr;
    r->next = mergerulep(r1, r2->next);
    return (r);
  } else {
    r1->next = mergerulep(r1->next, r2->next);
    return (r1);
  }
}

/* merge( x ) merges expression x into the TA rooted at start: */

static void
merge(EXPR *x)
{
  Statep          s;

  s = mkstatep(x);
  start = mergestatep(start, s);
  freestatep(s);
}

/*
 * assign( s ) assigns state, transition and rule numbers to the states in
 * the subtree rooted at s (in preorder):
 */

static void
assign(Statep s)
{
  Transp          l;
  Rulep		r;

  if (s != NULL) {
    if (statetbsz == INT_MAX)
      fatal(qcmsg[CODETB_OVF]);
    else {
      s->sno = statetbsz++;
      s->ntrans = 0;
      s->trans = transtbsz;
      for (l = s->tlist; l; l = l->next)
	if (transtbsz == INT_MAX)
	  fatal(qcmsg[CODETB_OVF]);
	else
	  s->ntrans++, transtbsz++;
      s->nrules = 0;
      s->roffs = roffstbsz;
      for (r = s->rlist; r; r = r->next)
	if (roffstbsz == INT_MAX)
	  fatal(qcmsg[CODETB_OVF]);
	else
	  s->nrules++, roffstbsz++;
      for (l = s->tlist; l; l = l->next)
	assign(l->nextst);
    }
  }
}

/* write_states( s ) writes the states in the TA rooted at s to the code
   file: */

static void
write_states(Statep s)
{
  Transp          t;

  if (s != NULL) {
    write_state(s->ntrans, s->trans, s->nrules, s->roffs);
    for (t = s->tlist; t; t = t->next)
      write_states(t->nextst);
  }
}

/* write_transs( s ) writes the transitions in the TA rooted at s to the code
   file: */

static void
write_transs(Statep s)
{
  Transp          t;

  if (s != NULL) {
    for (t = s->tlist; t; t = t->next)
      write_trans(t->type, t->fno, t->nextst->sno);
    for (t = s->tlist; t; t = t->next)
      write_transs(t->nextst);
  }
}

/* write_roffss( s ) writes the rule offsets in the TA rooted at s to the
   code file: */

static void
write_roffss(Statep s)
{
  Transp          t;
  Rulep		r;

  if (s != NULL) {
    for (r = s->rlist; r != NULL; r = r->next)
      write_roffs(r->addr);
    for (t = s->tlist; t != NULL; t = t->next)
      write_roffss(t->nextst);
  }
}

#ifdef DEBUG

/* debugging operations: */

extern int _modtb[], _fnametb[];
extern int _modtbsz;

static
printsym(int fno)
{
  if (fno)
    if (symtb[fno].modno >= 0)
      printf("%s::%s", strsp+_modtb[symtb[fno].modno],
	     strsp+symtb[fno].pname);
    else
      printf("%s", strsp+symtb[fno].pname);
  else
    printf("_");
}

static void
printop(OPREC *op)
{
  int             i;
  char            s[MAXSTRLEN];

  printf("%05d: ", op-tcodesp+codespsz);
  switch (op->opcode) {
  case RETOP:
    printf("!RET");
    break;
  case MATCHOP:
    printf("?MATCH #%d", op->opargs.m);
    break;
  case LVALOP:
    printf("[%d]", op->opargs.lval.offs);
    for (i = 0; i < op->opargs.lval.plen; i++)
      printf("$%d", getpath(op->opargs.lval.p, i));
    if (op->opargs.lval.vsym >= 0)
      printf(" (%s)", strsp+op->opargs.lval.vsym);
    break;
  case QUERYOP:
    printf("?");
    break;
  case INFOP:
    printf("!INFO %s, line %d", strsp+_fnametb[op->opargs.info.modno],
	   op->opargs.info.lineno);
    break;
  case POPOP:
    printf("!POP");
    break;
  case INTVALOP: {
    mpz_t z;
    if (getint(z, op->opargs.iv.len, op->opargs.iv.l)) {
      printf("%s", pmpz(s, z));
      mpz_clear(z);
    }
    break;
  }
  case FLOATVALOP:
    printf("%s", pfloat(s, op->opargs.fv));
    break;
  case STRVALOP:
    printf("\"%s\"", pstr(s, strsp + op->opargs.sv));
    break;
  default:
    printsym(op->opcode);
    break;
  }
  if (op->mode)
    printf(" *");
  printf("\n");
}

static void
printstate(Statep s)
{
  Transp          t;
  Rulep           r;

  if (s != NULL) {
    printf("\n%d:", s->sno);
    for (r = s->rlist; r; r = r->next)
      printf(" %05d", r->addr);
    printf("\n");
    for (t = s->tlist; t; t = t->next) {
      printf("\t");
      if (t->type) {
	printf("<");
	printsym(t->type);
	printf(">");
      }
      printsym(t->fno);
      printf(" : %d\n", t->nextst->sno);
    }
    for (t = s->tlist; t; t = t->next)
      printstate(t->nextst);
  }
}

#endif

static int saveaddr, saveexpr;
static EXPR *l;
static PATH p;
static byte plen;
byte offs;

static
isconstx(EXPR *x)
{
  /* walk down the spine */
  while (x->fno == APPOP)
    x = x->x1;
  return x->fno < RESERVED || symtb[x->fno].flags & CST;
}

void
debug_info(void)
{
  extern int yylineno;
  /* generate debugging information */
  geninfop(modno, yylineno);
}

void
fix_debug_info(void)
{
  extern int yylineno;
  /* fix debugging information */
  fixinfop(modno, yylineno);
}

void
left_hand_side(EXPR *x)
{
  if (isconstx(x))
    yyerror(qcmsg[INVALID_DEF]);
  if (nerrs)
    return;
  /* store lhs pattern: */
  l = x;
  /* generate queries and determine variable positions: */
  offs = 0; plen = 0; p = 0;
  queries(x);
  /* save current instruction address: */
  saveaddr = tcodespsz;
}

void
queries(EXPR *x)
{
  int vno;
  switch (x->fno) {
  case VAROP:
    vno = x->tag.vno;
    if (strcmp(tmpsp+vartb[vno].pname, "_"))
      if (vartb[vno].plen && vartb[vno].offs == offs) {
	genop(IDOP); set_mode();
	genlval(vartb[vno].offs, vartb[vno].plen, vartb[vno].p, mkvarsym(vno));
	genop(APPOP); set_mode();
	genlval(offs, plen, p, mkvarsym(vno));
	genop(APPOP);
	genop(QUERYOP);
      } else {
	vartb[vno].offs = offs;
	vartb[vno].plen = plen;
	vartb[vno].p = p;
      }
    break;
  case INTVALOP:
  case FLOATVALOP:
  case STRVALOP:
    genop(IDOP); set_mode();
    genlval(offs, plen, p, -1);
    genop(APPOP); set_mode();
    switch (x->fno) {
    case INTVALOP:
      genintval(x->tag.iv); break;
    case FLOATVALOP:
      genfloatval(x->tag.fv); break;
    case STRVALOP:
      genstrval(x->tag.sv); break;
    }
    genop(APPOP);
    genop(QUERYOP);
    break;
  default:
    if (symtb[x->fno].flags & VSYM)
      yyerror(qcmsg[UNBOUND_VAR]);
    if (arity(x->fno) > 0)
      if (plen >= MAXDEPTH)
	fatal(qcmsg[EXPRTB_OVF]);
      else {
	plen++;
	setpath(&p, plen-1, 0);
	queries(x->x1);
	setpath(&p, plen-1, 1);
	queries(x->x2);
	setpath(&p, plen-1, 0);
	plen--;
      }
    break;
  }
}

static void
queries0(EXPR *x)
{
  int vno;
  switch (x->fno) {
  case VAROP:
    vno = x->tag.vno;
    if (strcmp(tmpsp+vartb[vno].pname, "_") && vartb[vno].offs) {
      vartb[vno].offs = offs;
      vartb[vno].plen = plen;
      vartb[vno].p = p;
    }
    break;
  case INTVALOP:
  case FLOATVALOP:
  case STRVALOP:
    break;
  default:
    if (symtb[x->fno].flags & VSYM)
      break;
    if (arity(x->fno) > 0)
      if (plen >= MAXDEPTH)
	fatal(qcmsg[EXPRTB_OVF]);
      else {
	plen++;
	setpath(&p, plen-1, 0);
	queries0(x->x1);
	setpath(&p, plen-1, 1);
	queries0(x->x2);
	setpath(&p, plen-1, 0);
	plen--;
      }
    break;
  }
}

void
mark(void)
{
  saveexpr = exprtbsz;
}

static void
clearexpr(void)
{
  int i;
  /* clear all but the left-hand side expression */
  for (i = saveexpr; i < exprtbsz; i++)
    if (exprtb[i].fno == INTVALOP)
      mpz_clear(exprtb[i].tag.iv);
  exprtbsz = saveexpr;
  /* regenerate the paths of the left-hand side variables (might have been
     overridden by where clauses) */
  offs = 0; plen = 0; p = 0;
  queries0(l);
  /* scratch all leftovers from where clauses */
  for (i = 0; i < VARTBSZ; i++)
    if (vartb[i].offs)
      vartb[i].dflag = 0;
}

void
right_hand_side(EXPR *x)
{
  if (nerrs) return;
  /* merge lhs pattern into TA: */
  merge(l);
  /* generate code for the right-hand side: */
  expression(x);
  /* generate return instruction: */
  genop(RETOP);
#ifdef DEBUG
  {
    int addr;
    for (addr = 0; addr < tcodespsz; addr++)
      printop(tcodesp+addr);
  }
#endif
  /* flush the code table and reset for next rhs: */
  write_code();
  tcodespsz = saveaddr;
  clearexpr();
}

void
qualifier(EXPR *x)
{
  if (!x || nerrs) return;
  /* generate code for the qualifying expression: */
  expression(x);
  /* generate query instruction: */
  genop(QUERYOP);
}

static Statep *_matchtb = NULL;
static int amatchtbsz = 0;

static
add_match(Statep s)
{
  if (!_matchtb)
    if (!(_matchtb = (Statep*)aalloc(MATCHTBSZ, sizeof(Statep))))
      fatal(qcmsg[CODETB_OVF]);
    else
      amatchtbsz = MATCHTBSZ;
  else if (matchtbsz >= amatchtbsz)
    if (!(_matchtb = (Statep*)arealloc(_matchtb, amatchtbsz,
				       MATCHTBSZ/4, sizeof(Statep))))
      fatal(qcmsg[CODETB_OVF]);
    else
      amatchtbsz += MATCHTBSZ/4;
  _matchtb[matchtbsz] = s;
  return matchtbsz++;
}

void
where_clause(EXPR *x, EXPR *y)
{
  if (nerrs) return;
  /* generate code for the matchee */
  expression(y);
  /* generate pattern and matching operation */
  genmatch(add_match(mkstatep(x)));
  /* generate queries */
  if (offs++ >= 255)
    fatal(qcmsg[CODETB_OVF]);
  plen = 0; p = 0;
  queries(x);
}

void
init_def(void)
{
  offs = 0; plen = 0; p = 0;
}

static bool lvars = 0;

void
definition(EXPR *x, EXPR *y)
{
  if (nerrs) return;
  /* generate def/undef op */
  genop(y?DEFOP:UNDEFOP); set_mode();
  lvars = 1, expression(x), lvars = 0;
  genop(APPOP);
  if (y) {
    set_mode();
    expression(y);
    genop(APPOP);
  }
  /* generate return instruction: */
  genop(RETOP);
#ifdef DEBUG
  {
    int addr;
    for (addr = 0; addr < tcodespsz; addr++)
      printop(tcodesp+addr);
  }
#endif
  /* add entry point to init table: */
  add_init(codespsz);
  /* flush the code table and reset for next def: */
  write_code();
  clear();
}

static void
nexpression(EXPR *x)
{
  switch (x->fno) {
  case VAROP:
    if (!lvars && !strcmp(tmpsp+vartb[x->tag.vno].pname, "_"))
      yyerror(qcmsg[INVALID_VAR]);
    else if (vartb[x->tag.vno].offs || vartb[x->tag.vno].plen)
      /* lhs value */
      genlval(vartb[x->tag.vno].offs, vartb[x->tag.vno].plen,
	      vartb[x->tag.vno].p, mkvarsym(x->tag.vno));
    else {
      /* free variable symbol treated as function symbol */
      int vno = mkfvar(tmpsp + vartb[x->tag.vno].pname);
      symtb[vno].flags |= DCL;
      genop(vno);
    }
    break;
  case INTVALOP:
    genintval(x->tag.iv);
    break;
  case FLOATVALOP:
    genfloatval(x->tag.fv);
    break;
  case STRVALOP:
    genstrval(x->tag.sv);
    break;
  case CONSOP:
  case PAIROP:
    nexpression(x->x1);
    nexpression(x->x2);
    genop(x->fno);
    break;
  case APPOP:
    nexpression(x->x1);
    set_mode();
    nexpression(x->x2);
    genop(APPOP);
    break;
  default:
    genop(x->fno);
    break;
  }
}

void
expression(EXPR *x)
{
  /* optimize the special case of top-level saturated sequences (X||Y), which
     are compiled to X !POP Y instead of an ordinary application construct */
  if (x->fno == APPOP && x->x1->fno == APPOP && x->x1->x1->fno == SEQOP) {
    expression(x->x1->x2);
    genop(POPOP);
    nexpression(x->x2);
  } else
    nexpression(x);
}

void
write_TA(void)
{
  int i;
  assign(start);
  matchtb = NULL;
  if (matchtbsz > 0 && !(matchtb = (int*)aalloc(matchtbsz, sizeof(int))))
    fatal(qcmsg[MEM_OVF]);
  for (i = 0; i < matchtbsz; i++) {
    assign(_matchtb[i]);
    matchtb[i] = _matchtb[i]->sno;
  }
  write_states(start);
  for (i = 0; i < matchtbsz; i++)
    write_states(_matchtb[i]);
  write_transs(start);
  for (i = 0; i < matchtbsz; i++)
    write_transs(_matchtb[i]);
  write_roffss(start);
  for (i = 0; i < matchtbsz; i++)
    write_roffss(_matchtb[i]);
#ifdef DEBUG
  printstate(start);
  for (i = 0; i < matchtbsz; i++) {
    printf("\n*** ?MATCH #%d ***\n", i);
    printstate(_matchtb[i]);
  }
#endif
}
