
/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qdefs.h"

/*

Interface to the built-in functions:

The built-in functions are implemented as parameterless functions;
argument expressions are placed in the global args vector (module qm).

It is supposed that each function returns zero if the function
application failed (due to mismatch of arguments, or other error
conditions in which case qmstat should be set accordingly), and
nonzero in case of a successful application. In the latter case, the
result is pushed on top of the stack.

List of built-in functions:

qmdef()
- process variable def
qmundef()
- process variable undef

qmconcat()
- string/list/tuple concatenation
qmadd()
- integer/float addition
qmmin()
- integer/float subtraction
qmmul()
- integer/float multiplication
qmfdiv()
- integer/float division
qmdiv()
- integer division
qmmod()
- remainder of integer division
qmpow()
- integer/float exponentiation
qmidx()
- indexing
qmumin()
- unary minus
qmhash()
- size operator
qmunquote()
- unquote operator
qmforce()
- force operator
qmor()
- logical or (special)
qmand()
- logical and (special)
qmorelse()
- logical or (short-circuit)
qmandthen()
- logical and (short-circuit)
qmnot()
- logical not
qmle()
- check for "less than"
qmgr()
- check for "greater than"
qmeq()
- check for "equal"
qmleq()
- check for "less than or equal"
qmgeq()
- check for "greater than or equal"
qmneq()
- check for "not equal"
qmid()
- check for "identical" (special)

qmseq()
- sequence operator

qmshl()
- integer bit shift left
qmshr()
- integer bit shift right
qmpred()
- predecessor function
qmsucc()
- successor function

qmexp()
- exponential function
qmln()
- natural logarithm
qmsqrt()
- square root
qmsin()
- sine
qmcos()
- cosine
qmatan()
- arc tan
qmatan2()
- arc tan of 2 args
qmrandom()
- random number
qmseed()
- initialize random number generator

qmsub()
- return subsequence of string, list or tuple
qmsubstr()
- return substring of a string
qmpos()
- determine position of a substring in a string

qmint()
- return integral part of a floating point number
qmfrac()
- return fractional part of a floating point number
qmtrunc()
- truncate a float to an integer value
qmround()
- round a float to an integer value
qmfloat()
- convert an integer to a floating point number
qmhashnum()
- return a 32 bit hash code for any expression
qmord()
- convert a character or enumeration constant to an integer
qmchr()
- convert an integer to a character
qmlist()
- convert a tuple to a list
qmtuple()
- convert a list to a tuple
qmstr()
- convert expression to string
qmval()
- convert string to expression
qmstrq()
- convert quoted expression to string
qmvalq()
- convert string to quoted expression

qmisspecial()
- verify that argument is a special form
qmisconst()
- verify that argument is a constant
qmisfun()
- verify that argument is other function symbol
qmisdef()
- check whether variable has been assigned a value
qmflip()
- flip args of binary function

qmread()
- read an expression from the terminal
qmreadq()
- read a quoted expression from the terminal
qmreadc()
- read a character from the terminal
qmreads()
- read a string from the terminal
qmwrite()
- write an expression to the terminal
qmwriteq()
- write a quoted expression to the terminal
qmwritec()
- write a character to the terminal
qmwrites()
- write a string to the terminal

qmfread()
- read an expression from a file
qmfreadq()
- read a quoted expression from a file
qmfreadc()
- read a character from a file
qmfreads()
- read a string from a file
qmfwrite()
- write an expression to a file
qmfwriteq()
- write a quoted expression to a file
qmfwritec()
- write a character to a file
qmfwrites()
- write a string to a file

qmfopen()
- open a file
qmpopen()
- open a pipe
qmfclose()
- close a file
qmeof()
- check for end-of-file on terminal
qmfeof()
- check for end-of-file on file
qmflush()
- flush output buffer of terminal
qmfflush()
- flush output buffer of a file

qmversion()
- return interpreter version number
qmsysinfo
- return string describing the host system (hardware-vendor-os)
qmwhich
- return absolute path of given script
qmhalt()
- halt execution
qmquit()
- quit the interpreter
qmbreak()
- interrupt evaluation and invoke the Q code debugger
qmcatch()
- handle an exception
qmthrow()
- throw an exception
qmtrap()
- catch signals
qmfail()
- let current rule fail
qmtime()
- return the current time
qmsleep()
- sleep for given number of seconds

*/

#if 0

/* This routine is now implemented non-recursively below. */

static
xeq(EXPR *x, EXPR *y)
/* check two terms for syntactic equality */
{
  if (x == y)
    return (1);
  else if (x->fno != y->fno)
    return (0);
  else
    switch (x->fno) {
    case INTVALOP:
      return mpz_cmp(x->data.z, y->data.z) == 0;
    case FLOATVALOP:
      return (x->data.f == y->data.f);
    case STRVALOP:
      return (strcmp(x->data.s, y->data.s) == 0);
    case BADFILEVALOP:
    case FILEVALOP:
      return (x->data.fp == y->data.fp);
    case USRVALOP:
      return (x->data.vp == y->data.vp);
    case CONSOP:
    case PAIROP:
    case APPOP:
      return (xeq(x->data.args.x1, y->data.args.x1) &&
	      xeq(x->data.args.x2, y->data.args.x2));
    case VECTOP: {
      int i, n = x->data.vect.n, m = y->data.vect.n;
      if (n != m)
	return (0);
      else {
	for (i = 0; i < n; i++)
	  if (!xeq(x->data.vect.xv[i], y->data.vect.xv[i]))
	    return (0);
	return (1);
      }
    }
    default:
      return (1);
    }
}

#else

/* This operation should be "failsafe", so we actually implement it
   non-recursively. */

static struct xstk_entry {
  EXPR *x, *y;
} *xstk = NULL;

static int xstkp = 0, xstka = 0;

static
xeq(EXPR *x, EXPR *y)
/* check two terms for syntactic equality */
{
  int mark = xstkp;
 loop:
  if (x == y)
    goto pop;
  else if (x->fno != y->fno)
    goto exit;
  else
    switch (x->fno) {
    case INTVALOP:
      if (mpz_cmp(x->data.z, y->data.z) == 0)
	goto pop;
      else
	goto exit;
    case FLOATVALOP:
      if (x->data.f == y->data.f)
	goto pop;
      else
	goto exit;
    case STRVALOP:
      if (strcmp(x->data.s, y->data.s) == 0)
	goto pop;
      else
	goto exit;
    case BADFILEVALOP:
    case FILEVALOP:
      if (x->data.fp == y->data.fp)
	goto pop;
      else
	goto exit;
    case USRVALOP:
      if (x->data.vp == y->data.vp)
	goto pop;
      else
	goto exit;
    case CONSOP:
    case PAIROP:
    case APPOP:
      if (xstkp >= xstka) {
	if (xstka >= INT_MAX ||
	    !(xstk = xstka?
	      realloc(xstk, (xstka+10240)*sizeof(struct xstk_entry)):
	      malloc(10240*sizeof(struct xstk_entry))))
	  fatal("memory overflow");
	else
	  xstka += 10240;
      }
      xstk[xstkp].x = x;
      xstk[xstkp++].y = y;
      x = x->data.args.x1;
      y = y->data.args.x1;
      goto loop;
    case VECTOP: {
      int i, n = x->data.vect.n, m = y->data.vect.n;
      if (n != m)
	goto exit;
      else {
	for (i = 0; i < n; i++)
	  if (!xeq(x->data.vect.xv[i], y->data.vect.xv[i]))
	    goto exit;
	goto pop;
      }
    }
    default:
    pop:
      while (xstkp > mark && x == xstk[xstkp-1].x->data.args.x2) {
	x = xstk[--xstkp].x;
	y = xstk[xstkp].y;
      }
      if (xstkp > mark) {
	x = xstk[xstkp-1].x->data.args.x2;
	y = xstk[xstkp-1].y->data.args.x2;
	goto loop;
      }
    }
  return 1;
 exit:
  xstkp = mark;
  return 0;
}

#endif

static int avtbsz = 0, vtbsz = 0, *vtb = NULL;

static
add_vtb(int fno, EXPR *x)
{
  if (fno == DEFVAROP)
    return 1;
  else if (symtb[fno].xp)
    return xeq(x, symtb[fno].xp);
  else {
    if (vtbsz >= avtbsz) {
      int *vtb1;
      if ((vtb1 = (int*)arealloc(vtb, avtbsz, 64, sizeof(int)))) {
	vtb = vtb1;
	avtbsz += 64;
      } else {
	thr0->qmstat = MEM_OVF;
	return 0;
      }
    }
    vtb[vtbsz++] = fno;
    symtb[fno].xp = qmnew(x);
    return 1;
  }
}

static int clear_vtb(THREAD *thr)
{
  int i;
  for (i = 0; i < vtbsz; i++) {
    qmfree(thr, symtb[vtb[i]].xp);
    symtb[vtb[i]].xp = NULL;
  }
  vtbsz = 0;
}

static
xmatch(THREAD *thr, EXPR *x, EXPR *y)
{
  if ((symtb[x->fno].flags & VSYM))
    return add_vtb(x->fno, y);
  else if (x->fno == PAIROP && y->fno == VECTOP) {
    int i = 0, n = y->data.vect.n;
    while (x->fno == PAIROP && i < n)
      if (!xmatch(thr, x->data.args.x1, y->data.vect.xv[i]))
	return 0;
      else {
	x = x->data.args.x2;
	i++;
      }
    if (x->fno == PAIROP)
      return 0;
    if ((symtb[x->fno].flags & VSYM)) {
      if (i >= n)
	/* end of the vector has been reached */
	y = funexpr(thr, VOIDOP);
      else {
	/* copy vector of remaining elements (assert: n>i>0) */
	EXPR **yv = malloc((n-i)*sizeof(EXPR*));
	int j;
	if (!yv) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
	for (j = i; j < n; j++)
	  yv[j-i] = qmnew(y->data.vect.xv[j]);
	y = vectexpr(thr, n-i, yv);
      }
      if (y)
	return add_vtb(x->fno, y);
      else
	return 0;
    } else
      return 0;
  } else if (x->fno != y->fno)
    return 0;
  else
    switch (x->fno) {
    case INTVALOP:
      return mpz_cmp(x->data.z, y->data.z) == 0;
    case FLOATVALOP:
      return x->data.f == y->data.f;
    case STRVALOP:
      return strcmp(x->data.s, y->data.s) == 0;
    case BADFILEVALOP:
    case FILEVALOP:
      return x->data.fp == y->data.fp;
    case USRVALOP:
      return x->data.vp == y->data.vp;
    case CONSOP:
    case PAIROP:
    case APPOP:
      return xmatch(thr, x->data.args.x1, y->data.args.x1) &&
	xmatch(thr, x->data.args.x2, y->data.args.x2);
    case VECTOP: {
      int i, n = x->data.vect.n, m = y->data.vect.n;
      if (n!=m)
	return 0;
      else {
	for (i = 0; i < n; i++)
	  if (!xmatch(thr, x->data.vect.xv[i], y->data.vect.xv[i]))
	    return 0;
	return 1;
      }
    }
    default:
      return 1;
    }
}

static
qmdef(THREAD* thr)
{
  EXPR **args = thr->args;
  if (xmatch(thr, args[0], args[1])) {
    int i, res = 1;
    for (i = 0; i < vtbsz; i++)
      if (!setvar(vtb[i], symtb[vtb[i]].xp)) {
	res = 0;
	break;
      } else if (init_mode)
	symtb[vtb[i]].flags &= ~MODIF;
    clear_vtb(thr);
    if (res && pushfun(thr, VOIDOP)) {
      thr->nredns--;
      return 1;
    } else
      return 0;
  } else {
    clear_vtb(thr);
    thr->qmstat = MATCH_ERR;
    return 0;
  }
}

static
qmundef(THREAD* thr)
{
  EXPR **args = thr->args;
  if (!setvar(args[0]->fno, NULL))
    return 0;
  else {
    if (init_mode)
      symtb[args[0]->fno].flags &= ~MODIF;
    if (pushfun(thr, VOIDOP)) {
      thr->nredns--;
      return 1;
    } else
      return 0;
  }
}

static
strconcat(THREAD *thr, char *s1, char *s2)
 /* concatenate two strings and push the result on the stack. */
{
  char           *s;
  int		l1 = strlen(s1), l2 = strlen(s2);
  
  if (l1 >= INT_MAX - l2 || (s = malloc(l1+l2+1)) == NULL) {
    thr->qmstat = MEM_OVF;
    return (0);
  } else
    return (pushstr(thr, strcat(strcpy(s, s1), s2)));
}

static
listconcat(THREAD *thr, EXPR *x1, EXPR *x2)
 /* concatenate two lists and push the result on the stack. */
{
  int             n;

  for (n = 0; x1->fno == CONSOP; x1 = x1->data.args.x2) {
    n++;
    if (!push(thr, x1->data.args.x1))
      return (0);
  }
  if (x1->fno == NILOP) {
    if (!push(thr, x2))
      return (0);
    for (; n > 0; n--)
      if (!pushfun(thr, CONSOP))
	return (0);
    return (1);
  } else {
    for (; n > 0; n--)
      qmfree(thr, *--thr->xsp);
    return (0);
  }
}

static
vectconcat(THREAD *thr, int n1, EXPR **xv1, int n2, EXPR **xv2)
 /* concatenate two vectors and push the result on the stack. */
{
  EXPR          **xv = NULL;

  if (n1 >= INT_MAX - n2 || n1+n2>0 && (xv = malloc((n1+n2)*sizeof(EXPR*)))
      == NULL) {
    thr->qmstat = MEM_OVF;
    return 0;
  } else {
    int i;
    for (i = 0; i < n1; i++)
      xv[i] = qmnew(xv1[i]);
    for (i = 0; i < n2; i++)
      xv[n1+i] = qmnew(xv2[i]);
    return pushvect(thr, n1+n2, xv);
  }
}

static
tupleconcat(THREAD *thr, EXPR *x1, EXPR *x2)
 /* concatenate two tuples and push the result on the stack. */
{
  int             n;

  if (x1->fno == VECTOP)
    if (x2->fno == VECTOP)
      return vectconcat(thr, x1->data.vect.n, x1->data.vect.xv,
			x2->data.vect.n, x2->data.vect.xv);
    else {
      int i;
      for (i = 0; i < x1->data.vect.n; i++)
	if (!push(thr, x1->data.vect.xv[i]))
	  return (0);
      n = x1->data.vect.n;
    }
  else
    for (n = 0; x1->fno == PAIROP; x1 = x1->data.args.x2) {
      n++;
      if (!push(thr, x1->data.args.x1))
	return (0);
    }
  if (x1->fno == VECTOP || x1->fno == VOIDOP) {
    if (!push(thr, x2))
      return (0);
    for (; n > 0; n--)
      if (!pushfun(thr, PAIROP))
	return (0);
    return (1);
  } else {
    for (; n > 0; n--)
      qmfree(thr, *--thr->xsp);
    return (0);
  }
}

static qmconcat(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case STRVALOP:
    if (args[1]->fno == STRVALOP)
      return (strconcat(thr, args[0]->data.s,
			args[1]->data.s));
    else
      return (0);
  case NILOP:
  case VOIDOP:
    return (push(thr, args[1]));
  case CONSOP:
    return (listconcat(thr, args[0], args[1]));
  case PAIROP:
  case VECTOP:
    return (tupleconcat(thr, args[0], args[1]));
  default:
    return (0);
  }
}

/* NOTE: for maximum performance, we should make qmadd/min/mul/div/mod/umin
   reuse an argument where possible (i.e., where refc=1) */

/* inline code for the time-critical stuff in the bigint ops */

#define __max(x,y) (((x)>=(y))?(x):(y))
#define __abs(x) (((x)>=0)?(x):-(x))
#define __sz(z) __abs((z)->_mp_size)

#define __mpzop2(f,sz,x,y) \
{ mpz_t __z; \
mpz_init(__z); \
if (__z->_mp_d && my_mpz_realloc(__z, sz)) { \
  int __sz; \
  f(__z, x, y); \
  if (!__z->_mp_d) { \
    thr->qmstat = MEM_OVF; \
    return 0; \
  } \
  __sz = mpz_size(__z); \
  if (__sz < sz && !my_mpz_realloc(__z, __sz)) { \
    thr->qmstat = MEM_OVF; \
    return 0; \
  } \
  return pushmpz(thr, __z); \
} else { \
  thr->qmstat = MEM_OVF; \
  return 0; \
}}

#define __mpzop1(f,sz,x) \
{ mpz_t __z; \
mpz_init(__z); \
if (__z->_mp_d && my_mpz_realloc(__z, sz)) { \
  int __sz; \
  f(__z, x); \
  if (!__z->_mp_d) { \
    thr->qmstat = MEM_OVF; \
    return 0; \
  } \
  __sz = mpz_size(__z); \
  if (__sz < sz && !my_mpz_realloc(__z, __sz)) { \
    thr->qmstat = MEM_OVF; \
    return 0; \
  } \
  return pushmpz(thr, __z); \
} else { \
  thr->qmstat = MEM_OVF; \
  return 0; \
}}

static qmadd(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP: {
      /* needed: max size of arg 0 and arg 1, plus possible carry */
      int sz = __max(__sz(args[0]->data.z), __sz(args[1]->data.z))+1;
      /* aargh, this is terrible, but we have to check for possible
	 overflows here */
      if (sz < 0) return 0;
      __mpzop2(mpz_add, sz, args[0]->data.z, args[1]->data.z);
    }
    case FLOATVALOP:
      return (pushfloat(thr, mpz_get_d(args[0]->data.z) +
			args[1]->data.f));
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, args[0]->data.f +
			mpz_get_d(args[1]->data.z)));
    case FLOATVALOP:
      return (pushfloat(thr, args[0]->data.f +
			args[1]->data.f));
    default:
      return (0);
    }
  default:
    return (0);
  }
}

static qmmin(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP: {
      /* needed: max size of arg 0 and arg 1, plus possible carry */
      int sz = __max(__sz(args[0]->data.z), __sz(args[1]->data.z))+1;
      if (sz < 0) return 0;
      __mpzop2(mpz_sub, sz, args[0]->data.z, args[1]->data.z);
    }
    case FLOATVALOP:
      return (pushfloat(thr, mpz_get_d(args[0]->data.z) -
			args[1]->data.f));
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, args[0]->data.f -
			mpz_get_d(args[1]->data.z)));
    case FLOATVALOP:
      return (pushfloat(thr, args[0]->data.f -
			args[1]->data.f));
    default:
      return (0);
    }
  default:
    return (0);
  }
}

static qmmul(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP: {
      /* needed: total size of arg 0 and arg 1 */
      int sz = __sz(args[0]->data.z)+__sz(args[1]->data.z);
      if (sz < 0) return 0;
      __mpzop2(mpz_mul, sz, args[0]->data.z, args[1]->data.z);
    }
    case FLOATVALOP:
      return (pushfloat(thr, mpz_get_d(args[0]->data.z) *
			args[1]->data.f));
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, args[0]->data.f *
			mpz_get_d(args[1]->data.z)));
    case FLOATVALOP:
      return (pushfloat(thr, args[0]->data.f *
			args[1]->data.f));
    default:
      return (0);
    }
  default:
    return (0);
  }
}

static qmfdiv(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      if (mpz_sgn(args[1]->data.z) != 0)
	return (pushfloat(thr, mpz_get_d(args[0]->data.z) /
			  mpz_get_d(args[1]->data.z)));
      else
	return (0);
    case FLOATVALOP:
      if (args[1]->data.f != 0.)
	return (pushfloat(thr, mpz_get_d(args[0]->data.z) /
			  args[1]->data.f));
      else
	return (0);
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      if (mpz_sgn(args[1]->data.z) != 0)
	return (pushfloat(thr, args[0]->data.f /
			  mpz_get_d(args[1]->data.z)));
      else
	return (0);
    case FLOATVALOP:
      if (args[1]->data.f != 0.)
	return (pushfloat(thr, args[0]->data.f /
			  args[1]->data.f));
      else
	return (0);
    default:
      return (0);
    }
  default:
    return (0);
  }
}

static qmdiv(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      mpz_sgn(args[1]->data.z) != 0) {
    /* needed: size of arg 0 minus size of arg 1 plus 1 */
    int sz = __sz(args[0]->data.z)-__sz(args[1]->data.z)+1;
    if (sz < 0) sz = 0;
    __mpzop2(mpz_tdiv_q, sz, args[0]->data.z, args[1]->data.z);
  } else
    return (0);
}

static qmmod(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      mpz_sgn(args[1]->data.z) != 0) {
    /* needed: size of arg 1 */
    int sz = __sz(args[1]->data.z);
    __mpzop2(mpz_tdiv_r, sz, args[0]->data.z, args[1]->data.z);
  } else
    return (0);
}

static qmpow(THREAD* thr)
{
  EXPR **args = thr->args;
  double ip;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      if (mpz_sgn(args[0]->data.z) != 0 ||
	  mpz_sgn(args[1]->data.z) != 0)
	return (pushfloat(thr, pow(mpz_get_d(args[0]->data.z),
			      mpz_get_d(args[1]->data.z))));
      else
	return (0);
    case FLOATVALOP:
      if ((mpz_sgn(args[0]->data.z) >= 0 ||
	   modf(args[1]->data.f, &ip) == 0.0) &&
	  (mpz_sgn(args[0]->data.z) != 0 ||
	   args[1]->data.f != 0.0))
	return (pushfloat(thr, pow(mpz_get_d(args[0]->data.z),
			      args[1]->data.f)));
      else
	return (0);
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      if (args[0]->data.f != 0.0 ||
	  mpz_sgn(args[1]->data.z) != 0)
	return (pushfloat(thr, pow(args[0]->data.f,
			      mpz_get_d(args[1]->data.z))));
      else
	return (0);
    case FLOATVALOP:
      if ((args[0]->data.f >= 0.0 ||
	   modf(args[1]->data.f, &ip) == 0.0) &&
	  (args[0]->data.f != 0.0 ||
	   args[1]->data.f != 0.0))
	return (pushfloat(thr, pow(args[0]->data.f,
			      args[1]->data.f)));
      else
	return (0);
    default:
      return (0);
    }
  default:
    return (0);
  }
}

/* NOTE: qmidx, qmhash, qmsub, qmsubstr and qmpos should probably use mpz's for
   indexing. At least as soon as handling of 2GB strings and lists becomes
   everyday business. But in that future long values might have 128 bits
   anyway. ;-) */

static qmidx(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[1]->fno == INTVALOP && my_mpz_fits_slong_p(args[1]->data.z)) {
    long		i = mpz_get_si(args[1]->data.z);
    switch (args[0]->fno) {
    case STRVALOP: {
      char           *s, *s1 = args[0]->data.s,	s2[2];
      if (i >= 0 && i < strlen(s1))
	if ((s = strdup(charstr(s2, s1[i]))) == NULL) {
	  thr->qmstat = MEM_OVF;
	  return (0);
	} else
	  return (pushstr(thr, s));
      else
	return (0);
    }
    case CONSOP: {
      EXPR           *x = args[0];
      for (; i > 0 && x->fno == CONSOP; i--)
	x = x->data.args.x2;
      if (x->fno == CONSOP && i >= 0)
	return (push(thr, x->data.args.x1));
      else
	return (0);
    }
    case PAIROP: {
      EXPR           *x = args[0];
      for (; i > 0 && x->fno == PAIROP; i--)
	x = x->data.args.x2;
      if (x->fno == PAIROP && i >= 0)
	return (push(thr, x->data.args.x1));
      else
	return (0);
    }
    case VECTOP:
      if (i >= 0 && i < args[0]->data.vect.n)
	return (push(thr, args[0]->data.vect.xv[i]));
      else
	return (0);
    default:
      return (0);
    }
  } else
    return (0);
}

static qmumin(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP) {
    /* needed: size of arg 0 */
    int sz = __sz(args[0]->data.z);
    __mpzop1(mpz_neg, sz, args[0]->data.z);
  } else if (args[0]->fno == FLOATVALOP)
    return (pushfloat(thr, -args[0]->data.f));
  else
    return (0);
}

static qmhash(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR           *x = args[0];
  long		l;

  switch (x->fno) {
  case STRVALOP:
    l = (long) strlen(x->data.s);
    if (l >= 0)
      return (pushint(thr, l));
    else
      return (0);
  case NILOP:
  case CONSOP:
    for (l = 0; x->fno == CONSOP; l++)
      x = x->data.args.x2;
    if (x->fno == NILOP && l >= 0)
      return (pushint(thr, l));
    else
      return (0);
  case VOIDOP:
  case PAIROP:
    for (l = 0; x->fno == PAIROP; l++)
      x = x->data.args.x2;
    if (x->fno == VOIDOP && l >= 0)
      return (pushint(thr, l));
    else
      return (0);
  case VECTOP:
    return (pushint(thr, x->data.vect.n));
  default:
    return (0);
  }
}

static qmunquote(THREAD* thr)
{
  EXPR **args = thr->args;
  if (eval(thr, args[0])) {
    EXPR *x = thr->xsp[-1];
    if (x->fno == APPOP && x->data.args.x1->fno == QUOTEOP) {
      x = qmnew(x->data.args.x2);
      qmfree(thr, thr->xsp[-1]);
      thr->xsp[-1] = x;
      if (!thr->mode) {
	int res = eval(thr, (x = *--thr->xsp));
	qmfree(thr, x);
	return res;
      }
      return 1;
    } else
      return 1;
  } else
    return 0;
}

static qmforce(THREAD* thr)
{
  EXPR **args = thr->args;
  return eval(thr, args[0]);
}

static qmor(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0], *y = args[1];
  if (x->fno == INTVALOP && y->fno == INTVALOP) {
    int sz = __max(__sz(x->data.z), __sz(y->data.z)) + 1;
    if (sz < 0) return 0;
    __mpzop2(mpz_ior, sz, x->data.z, y->data.z);
  } else if (y->type != BOOLTYPE)
    return (0);
  else if (x->fno == FALSEOP)
    return (push(thr, y));
  else if (x->fno == TRUEOP)
    return (push(thr, x));
  else
    return (0);
}

static qmorelse(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR	       *x = args[0], *y = args[1];
  if (x->fno == FALSEOP)
    return (eval(thr, y));
  else if (x->fno == TRUEOP)
    return (push(thr, x));
  else
    return (0);
}

static qmand(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0], *y = args[1];
  if (x->fno == INTVALOP && y->fno == INTVALOP) {
    int sz = __max(__sz(x->data.z), __sz(y->data.z)) + 1;
    if (sz < 0) return 0;
    __mpzop2(mpz_and, sz, x->data.z, y->data.z);
  } else if (y->type != BOOLTYPE)
    return (0);
  else if (x->fno == TRUEOP)
    return (push(thr, y));
  else if (x->fno == FALSEOP)
    return (push(thr, x));
  else
    return (0);
}

static qmandthen(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR	       *x = args[0], *y = args[1];
  if (x->fno == TRUEOP)
    return (eval(thr, y));
  else if (x->fno == FALSEOP)
    return (push(thr, x));
  else
    return (0);
}

static qmnot(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP) {
    int sz = __sz(args[0]->data.z) + 1;
    if (sz < 0) return 0;
    __mpzop1(mpz_com, sz, args[0]->data.z);
  } else if (args[0]->fno == TRUEOP)
    return (pushfun(thr, FALSEOP));
  else if (args[0]->fno == FALSEOP)
    return (pushfun(thr, TRUEOP));
  else
    return (0);
}

static
xcmp(EXPR *x, EXPR *y, int *result)
 /* compare two terms */
{
  if (x->fno >= BINARY && y->fno >= BINARY &&
      symtb[x->fno].type && symtb[x->fno].type == symtb[y->fno].type &&
      symtb[symtb[x->fno].type].fno_min) {
    *result = x->fno-y->fno;
    return 1;
  } else if (x->fno != y->fno)
    switch (x->fno) {
    case INTVALOP:
      if (y->fno == FLOATVALOP) {
	double xf = mpz_get_d(x->data.z);
	if (xf < y->data.f)
	  *result = -1;
	else if (xf > y->data.f)
	  *result = 1;
	else
	  *result = 0;
	return (1);
      } else
	return (0);
    case FLOATVALOP:
      if (y->fno == INTVALOP) {
	double yf = mpz_get_d(y->data.z);
	if (x->data.f < yf)
	  *result = -1;
	else if (x->data.f > yf)
	  *result = 1;
	else
	  *result = 0;
	return (1);
      } else
	return (0);
    default:
      return (0);
    }
  else
    switch (x->fno) {
    case INTVALOP:
      *result = mpz_cmp(x->data.z, y->data.z);
      return (1);
    case FLOATVALOP:
      if (x->data.f < y->data.f)
	*result = -1;
      else if (x->data.f > y->data.f)
	*result = 1;
      else
	*result = 0;
      return (1);
    case STRVALOP:
      *result = strcmp(x->data.s, y->data.s);
      return (1);
    default:
      return (0);
    }
}

static qmle(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (xcmp(args[0], args[1], &result))
    if (result < 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmgr(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (xcmp(args[0], args[1], &result))
    if (result > 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmeq(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (xcmp(args[0], args[1], &result))
    if (result == 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmleq(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (xcmp(args[0], args[1], &result))
    if (result <= 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmgeq(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (xcmp(args[0], args[1], &result))
    if (result >= 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmneq(THREAD* thr)
{
  EXPR **args = thr->args;
  int             result;

  if (xcmp(args[0], args[1], &result))
    if (result != 0)
      return (pushfun(thr, TRUEOP));
    else
      return (pushfun(thr, FALSEOP));
  else
    return (0);
}

static qmid(THREAD* thr)
{
  EXPR **args = thr->args;
  if (xeq(args[0], args[1]))
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmseq(THREAD* thr)
{
  EXPR **args = thr->args;
  return (push(thr, args[1]));
}

static shl(THREAD *thr, mpz_t z, int i)
{
  int n = sizeof(mp_limb_t)*8, m = __sz(z), r = i/n, k = i%n, s;
  mpz_t u;
  if (m > INT_MAX-r-1) return 0;
  mpz_init(u);
  if (m == 0) return pushmpz(thr, u);
  if (!u->_mp_d || !my_mpz_realloc(u, m+r+1)) {
    thr->qmstat = MEM_OVF;
    return 0;
  }
  mpz_set(u, z);
  if (k) {
    mp_limb_t x = mpn_lshift(u->_mp_d, u->_mp_d, m, k);
    u->_mp_d[m] = x;
  } else
    u->_mp_d[m] = 0;
  for (s = m; s >= 0; s--) u->_mp_d[s+r] = u->_mp_d[s];
  for (s = 0; s < r; s++) u->_mp_d[s] = 0;
  m += r;
  if (u->_mp_d[m]) m++;
  if (z->_mp_size < 0)
    u->_mp_size = -m;
  else
    u->_mp_size = m;
  if (__sz(z)+r+1 != m && !my_mpz_realloc(u, m)) {
    thr->qmstat = MEM_OVF;
    return 0;
  }
  return pushmpz(thr, u);
}

static shr(THREAD *thr, mpz_t z, int i)
{
  int n = sizeof(mp_limb_t)*8, m = __sz(z), r = i/n, k = i%n, s;
  mpz_t u;
  mpz_init(u);
  if (r >= m) return pushmpz(thr, u);
  if (!u->_mp_d || !my_mpz_realloc(u, m)) {
    thr->qmstat = MEM_OVF;
    return 0;
  }
  for (s = r; s < m; s++) u->_mp_d[s-r] = z->_mp_d[s];
  m -= r;
  if (k) mpn_rshift(u->_mp_d, u->_mp_d, m, k);
  if (!u->_mp_d[m-1]) m--;
  if (z->_mp_size < 0)
    u->_mp_size = -m;
  else
    u->_mp_size = m;
  if (!my_mpz_realloc(u, m)) {
    thr->qmstat = MEM_OVF;
    return 0;
  }
  return pushmpz(thr, u);
}

static qmshl(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      my_mpz_fits_slong_p(args[1]->data.z)) {
    long i = mpz_get_si(args[1]->data.z);
    if (i > 0)
      return shl(thr, args[0]->data.z, i);
    else if (i == INT_MIN)
      /* calculating -i will overflow */
      return 0;
    else if (i < 0)
      return shr(thr, args[0]->data.z, -i);
    else {
      mpz_t u;
      mpz_init(u);
      if (!u->_mp_d || !my_mpz_realloc(u, __sz(args[0]->data.z))) {
	thr->qmstat = MEM_OVF;
	return 0;
      }
      mpz_set(u, args[0]->data.z);
      return pushmpz(thr, u);
    }
  } else
    return 0;
}

static qmshr(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      my_mpz_fits_slong_p(args[1]->data.z)) {
    long i = mpz_get_si(args[1]->data.z);
    if (i > 0)
      return shr(thr, args[0]->data.z, i);
    else if (i == INT_MIN)
      /* calculating -i will overflow */
      return 0;
    else if (i < 0)
      return shl(thr, args[0]->data.z, -i);
    else {
      mpz_t u;
      mpz_init(u);
      if (!u->_mp_d || !my_mpz_realloc(u, __sz(args[0]->data.z))) {
	thr->qmstat = MEM_OVF;
	return 0;
      }
      mpz_set(u, args[0]->data.z);
      return pushmpz(thr, u);
    }
  } else
    return 0;
}

static qmpred(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->type == CHARTYPE && *args[0]->data.s > CHAR_MIN) {
    char *s = malloc(2*sizeof(char));
    if (s) {
      s[0] = (*args[0]->data.s)-1;
      s[1] = 0;
      return pushstr(thr, s);
    } else {
      thr->qmstat = MEM_OVF;
      return 0;
    }
  } else if (args[0]->type && symtb[args[0]->type].fno_min &&
	     args[0]->fno > symtb[args[0]->type].fno_min)
    return pushfun(thr, args[0]->fno-1);
  else
    return 0;
}

static qmsucc(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->type == CHARTYPE &&
      ((unsigned char)*args[0]->data.s) < UCHAR_MAX) {
    char *s = malloc(2*sizeof(char));
    if (s) {
      s[0] = (*args[0]->data.s)+1;
      s[1] = 0;
      return pushstr(thr, s);
    } else {
      thr->qmstat = MEM_OVF;
      return 0;
    }
  } else if (args[0]->type && symtb[args[0]->type].fno_min &&
	     args[0]->fno < symtb[args[0]->type].fno_max)
    return pushfun(thr, args[0]->fno+1);
  else
    return 0;
}

static qmexp(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    return (pushfloat(thr, exp(mpz_get_d(args[0]->data.z))));
  case FLOATVALOP:
    return (pushfloat(thr, exp(args[0]->data.f)));
  default:
    return (0);
  }
}

static qmln(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    if (mpz_sgn(args[0]->data.z) > 0)
      return (pushfloat(thr, log(mpz_get_d(args[0]->data.z))));
    else
      return (0);
  case FLOATVALOP:
    if (args[0]->data.f > 0)
      return (pushfloat(thr, log(args[0]->data.f)));
    else
      return (0);
  default:
    return (0);
  }
}

static qmsqrt(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    if (mpz_sgn(args[0]->data.z) >= 0)
      return (pushfloat(thr, sqrt(mpz_get_d(args[0]->data.z))));
    else
      return (0);
  case FLOATVALOP:
    if (args[0]->data.f >= 0)
      return (pushfloat(thr, sqrt(args[0]->data.f)));
    else
      return (0);
  default:
    return (0);
  }
}

static qmsin(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    return (pushfloat(thr, sin(mpz_get_d(args[0]->data.z))));
  case FLOATVALOP:
    return (pushfloat(thr, sin(args[0]->data.f)));
  default:
    return (0);
  }
}

static qmcos(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    return (pushfloat(thr, cos(mpz_get_d(args[0]->data.z))));
  case FLOATVALOP:
    return (pushfloat(thr, cos(args[0]->data.f)));
  default:
    return (0);
  }
}

static qmatan(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    return (pushfloat(thr, atan(mpz_get_d(args[0]->data.z))));
  case FLOATVALOP:
    return (pushfloat(thr, atan(args[0]->data.f)));
  default:
    return (0);
  }
}

static qmatan2(THREAD* thr)
{
  EXPR **args = thr->args;
  switch (args[0]->fno) {
  case INTVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, atan2(mpz_get_d(args[0]->data.z),
			      mpz_get_d(args[1]->data.z))));
    case FLOATVALOP:
      return (pushfloat(thr, atan2(mpz_get_d(args[0]->data.z),
			      args[1]->data.f)));
    default:
      return (0);
    }
  case FLOATVALOP:
    switch (args[1]->fno) {
    case INTVALOP:
      return (pushfloat(thr, atan2(args[0]->data.f,
			      mpz_get_d(args[1]->data.z))));
    case FLOATVALOP:
      return (pushfloat(thr, atan2(args[0]->data.f,
			      args[1]->data.f)));
    default:
      return (0);
    }
  default:
    return (0);
  }
}

static qmrandom(THREAD* thr)
{
  EXPR **args = thr->args;
  mpz_t z;
  mpz_init(z);
  if (z->_mp_d) {
    mpz_set_ui(z, randomMT());
    return pushmpz(thr, z);
  } else {
    thr->qmstat = MEM_OVF;
    return 0;
  }
}

static qmseed(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP && mpz_sgn(args[0]->data.z) >= 0) {
    seedMT(mpz_get_ui(args[0]->data.z) << 1 | 1);
    return (pushfun(thr, VOIDOP));
  } else
    return (0);
}

static qmsub(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[1]->fno == INTVALOP && my_mpz_fits_slong_p(args[1]->data.z) &&
      args[2]->fno == INTVALOP && my_mpz_fits_slong_p(args[2]->data.z)) {
    long		i = mpz_get_si(args[1]->data.z);
    long		j = mpz_get_si(args[2]->data.z);
    long		c, l;
    if (i < 0) i = 0; c = j-i+1;
    switch (args[0]->fno) {
    case STRVALOP: {
      char           *s1 = args[0]->data.s, *s2;
      l = strlen(s1);
      if (i >= l || j < i)
	l = 0;
      else if ((l -= i) > c)
	l = c;
      if (l < 0)
	l = 0;
      if ((s2 = malloc(l+1)) == NULL) {
	thr->qmstat = MEM_OVF;
	return (0);
      }
      if (l > 0)
	substr(s2, s1+i, l);
      else
	*s2 = 0;
      return (pushstr(thr, s2));
    }
    case NILOP:
    case CONSOP: {
      EXPR *x = args[0], *x2;
      if (j < i)
	return pushfun(thr, NILOP);
      while (x->fno == CONSOP && i > 0)
	x = x->data.args.x2, i--;
      x2 = x; l = c;
      while (x2->fno == CONSOP && c > 0)
	x2 = x2->data.args.x2, c--;
      if (x2->fno == NILOP)
	return push(thr, x);
      else if (c > 0)
	return 0;
      for (x2 = x, c = 0; c < l; x2 = x2->data.args.x2, c++)
	if (!push(thr, x2->data.args.x1)) return 0;
      if (!pushfun(thr, NILOP)) return 0;
      for (c = 0; c < l; c++)
	if (!pushfun(thr, CONSOP)) return 0;
      return 1;
    }
    case VOIDOP:
    case PAIROP: {
      EXPR *x = args[0], *x2, **xv;
      if (j < i)
	return pushfun(thr, VOIDOP);
      while (x->fno == PAIROP && i > 0)
	x = x->data.args.x2, i--;
      x2 = x; l = c;
      while (x2->fno == PAIROP && c > 0)
	x2 = x2->data.args.x2, c--;
      if (x2->fno == VOIDOP)
	/* actually this case shouldn't arise (tuple is a legal vector and
	   hence should be handled in the default case) */
	return push(thr, x);
      else if (c > 0)
	return 0;
      /* build a vector from the collected tuple members */
      if ((xv = (EXPR**)malloc(l*sizeof(EXPR*))) == NULL) {
	thr->qmstat = MEM_OVF;
	return (0);
      }
      for (x2 = x, c = 0; c < l; x2 = x2->data.args.x2, c++)
	xv[c] = qmnew(x2->data.args.x1);
      return pushvect(thr, l, xv);
    }
    case VECTOP: {
      EXPR **xv, **xv1 = args[0]->data.vect.xv;
      l = args[0]->data.vect.n;
      if (i >= l || j < i)
	l = 0;
      else if ((l -= i) > c)
	l = c;
      if (l <= 0)
	return pushfun(thr, VOIDOP);
      else if ((xv = (EXPR**)malloc(l*sizeof(EXPR*))) == NULL) {
	thr->qmstat = MEM_OVF;
	return (0);
      }
      for (c = 0; c < l; c++)
	xv[c] = qmnew(xv1[i+c]);
      return pushvect(thr, l, xv);
    }
    default:
      return (0);
    }
  } else
    return 0;
}

static qmsubstr(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP &&
      args[1]->fno == INTVALOP && my_mpz_fits_slong_p(args[1]->data.z) &&
      args[2]->fno == INTVALOP && my_mpz_fits_slong_p(args[2]->data.z)) {
    char           *s1 = args[0]->data.s, *s2;
    long		i = mpz_get_si(args[1]->data.z);
    long		c = mpz_get_si(args[2]->data.z);
    long		l;

    if (i < 0) i = 0;
    l = strlen(s1);
    if (i >= l || c <= 0)
      l = 0;
    else if ((l -= i) > c)
      l = c;
    if (l < 0)
      l = 0;
    if ((s2 = malloc(l+1)) == NULL) {
      thr->qmstat = MEM_OVF;
      return (0);
    }
    if (l > 0)
      substr(s2, s1+i, l);
    else
      *s2 = 0;
    return (pushstr(thr, s2));
  } else
    return (0);
}

static qmpos(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP && args[1]->fno == STRVALOP) {
    char           *s1 = args[0]->data.s;
    char           *s2 = args[1]->data.s;
    char           *s;
    
    if ((s = strstr(s2, s1)) != NULL)
      return (pushint(thr, (long) (s - s2)));
    else
      return (pushint(thr, (long) -1));
  } else
    return (0);
}

static qmint(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FLOATVALOP) {
    double ip, fp;
    fp = modf(args[0]->data.f, &ip);
    return (pushfloat(thr, ip));
  } else
    return (0);
}

static qmfrac(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FLOATVALOP) {
    double ip, fp;
    fp = modf(args[0]->data.f, &ip);
    return (pushfloat(thr, fp));
  } else
    return (0);
}

static qmtrunc(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FLOATVALOP) {
    double ip, fp, dsz;
    int sz;
    fp = modf(args[0]->data.f, &ip);
    /* estimate the number of limbs required */
    dsz = log(__abs(ip))/log(2)/((double)CHAR_BIT*sizeof(mp_limb_t))+1.0;
    if (dsz < 1.0) dsz = 1.0; /* this can't happen?? */
    /* add an extra limb to be safe */
    sz = ((int)dsz)+1;
    /* this shouldn't happen but ... ;-) */
    if (((double)INT_MAX) <= dsz || sz < 0) return 0;
    __mpzop1(mpz_set_d, sz, ip);
  } else
    return (0);
}

static qmround(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FLOATVALOP) {
    double ip, fp, dsz;
    int sz;
    fp = modf(args[0]->data.f, &ip);
    ip += (fp>=0.5)?1:(fp<=-0.5)?-1:0;
    dsz = log(__abs(ip))/log(2)/((double)CHAR_BIT*sizeof(mp_limb_t))+1.0;
    if (dsz < 1.0) dsz = 1.0;
    sz = ((int)dsz)+1;
    if (((double)INT_MAX) <= dsz || sz < 0) return 0;
    __mpzop1(mpz_set_d, sz, ip);
  } else
    return (0);
}

static qmfloat(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP)
    return (pushfloat(thr, mpz_get_d(args[0]->data.z)));
  else
    return (0);
}

static unsigned mpz_hash(mpz_t z)
{
  unsigned h = 0;
  int i, len = z->_mp_size;
  if (len < 0) len = -len;
  for (i=0; i<len; i++)
    h ^= z->_mp_d[i];
  if (z->_mp_size < 0)
    h = -h;
  return h;
}

static unsigned float_hash(double d)
{
  unsigned h;
  char *c;
  int i;
  c = (char*)&d;
  for (h=0, i=0; i<sizeof(double); i++) {
    h += c[i] * 971;
  }
  return h;
}

static unsigned str_hash(char *s)
{
  unsigned h = 0, g;
  while (*s) {
    h = (h<<4)+*(s++);
    if ((g = (h & 0xf0000000)))	{
      h = h^(g>>24);
      h = h^g;
    }
  }
  return h;
}

static unsigned ptr_hash(void *p)
{
  return (unsigned)p;
}

static unsigned expr_hash(EXPR *x)
{
  switch (x->fno) {
  case INTVALOP:
    return mpz_hash(x->data.z);
  case FLOATVALOP:
    return float_hash(x->data.f);
  case STRVALOP:
    return str_hash(x->data.s);
  case FILEVALOP: case BADFILEVALOP:
    return ptr_hash(x->data.fp);
  case VECTOP: {
    int i, h;
    h = x->data.vect.n;
    for (i = 0; i < x->data.vect.n; i++) {
	h = (h<<1) | (h<0 ? 1 : 0);
	h ^= expr_hash(x->data.vect.xv[i]);
    }
    return (unsigned)h;
  }
  case USRVALOP:
    return ptr_hash(x->data.vp);
  case CONSOP: case PAIROP: case APPOP: {
    int h;
    h = expr_hash(x->data.args.x1);
    h = (h<<1) | (h<0 ? 1 : 0);
    h ^= expr_hash(x->data.args.x2);
    return (unsigned)h;
  }
  default:
    return (unsigned)x->fno;
  }
}

static qmhashnum(THREAD* thr)
{
  EXPR **args = thr->args;
  return pushuint(thr, expr_hash(args[0]));
}

static qmord(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->type == CHARTYPE)
    return (pushint(thr, (long) (unsigned char) args[0]->data.s[0]));
  else if (args[0]->type && symtb[args[0]->type].fno_min)
    return pushint(thr, args[0]->fno - symtb[args[0]->type].fno_min);
  else
    return (0);
}

static qmchr(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP &&
      my_mpz_fits_slong_p(args[0]->data.z)) {
    long i = mpz_get_si(args[0]->data.z);
    if (i >= 0 && i < 256) {
      char            s[2], *t;
      if ((t = strdup(charstr(s, (char) i)))
	  == NULL) {
	thr->qmstat = MEM_OVF;
	return (0);
      } else
	return (pushstr(thr, t));
    } else
      return (0);
  } else
    return (0);
}

static qmlist(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == VECTOP) {
    int i, n = args[0]->data.vect.n;
    EXPR **xv = args[0]->data.vect.xv;
    EXPR *x = funexpr(thr, NILOP);
    for (i = n-1; x && i >= 0; i--) {
      EXPR *y = consexpr(thr, CONSOP, xv[i], x);
      if (!y) qmfree(thr, x);
      x = y;
    }
    return push(thr, x);
  } else if (args[0]->fno == VOIDOP)
    return pushfun(thr, NILOP);
  else
    return 0;
}

static qmtuple(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0];
  int n = 0;
  while (x->fno == CONSOP) {
    n++;
    x = x->data.args.x2;
  }
  if (x->fno == NILOP) {
    EXPR **xv = (n>0)?malloc(n*sizeof(EXPR*)):NULL;
    int i = 0;
    x = args[0];
    while (x->fno == CONSOP) {
      xv[i++] = qmnew(x->data.args.x1);
      x = x->data.args.x2;
    }
    return pushvect(thr, n, xv);
  } else
    return 0;
}

static qmstr(THREAD* thr)
{
  EXPR **args = thr->args;
  char            *s, *sprintx();
  if ((s = sprintx(args[0])) == NULL) {
    thr->qmstat = MEM_OVF;
    return (0);
  } else
    return (pushstr(thr, s));
}

static qmval(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP)
    if (sparsex(args[0]->data.s)) {
      EXPR *x = *--thr->xsp;
      int val = eval(thr, x);
      qmfree(thr, x);
      return (val);
    } else {
      if (thr->qmstat == SYNTAX_ERR || thr->qmstat == STR_ERR ||
	  thr->qmstat == BAD_SYM || thr->qmstat == BAD_REF)
	thr->qmstat = OK;
      return (0);
    }
  else
    return (0);
}

#define isquote(x,y) (((x)->fno==APPOP&&(x)->data.args.x1->fno==QUOTEOP)?\
		      (((y)=(x)->data.args.x2),1):0)

static qmstrq(THREAD* thr)
{
  EXPR **args = thr->args;
  char            *s, *sprintx();
  EXPR		*x;
  if (isquote(args[0], x))
    if ((s = sprintx(x)) == NULL) {
      thr->qmstat = MEM_OVF;
      return (0);
    } else
      return (pushstr(thr, s));
  else
    return (0);
}

static qmvalq(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP)
    if (pushfun(thr, QUOTEOP))
      if (sparsex(args[0]->data.s))
	return (pushfun(thr, APPOP));
      else {
	qmfree(thr, *--thr->xsp);
	if (thr->qmstat == SYNTAX_ERR || thr->qmstat == STR_ERR ||
	    thr->qmstat == BAD_SYM || thr->qmstat == BAD_REF)
	  thr->qmstat = OK;
	return (0);
      }
    else
      return (0);
  else
    return (0);
}

static qmisspecial(THREAD* thr)
{
  EXPR **args = thr->args;
  unsigned long argv = (args[0]->fno==APPOP)?args[0]->data.args.argv:
    symtb[args[0]->fno].argv;
  if (argv & 1)
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmisconst(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0];
  while (x->fno == APPOP) x = x->data.args.x1;
  if (x->fno < BINARY || (symtb[x->fno].flags & CST))
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmisfun(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno >= BINARY && !(symtb[args[0]->fno].flags & VSYM))
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmisvar(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno >= BINARY && (symtb[args[0]->fno].flags & VSYM))
    return (pushfun(thr, TRUEOP));
  else
    return (pushfun(thr, FALSEOP));
}

static qmisdef(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno >= BINARY && (symtb[args[0]->fno].flags & VSYM))
    return (pushfun(thr, symtb[args[0]->fno].x?TRUEOP:FALSEOP));
  else
    return (0);
}

static qmflip(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x;
  int _mode = thr->mode;
  thr->mode = 1;
  if (push(thr, args[0]) && push(thr, args[2]) &&
      pushfun(thr, APPOP) && push(thr, args[1]) &&
      pushfun(thr, APPOP)) {
    int res = eval(thr, (x = *--thr->xsp));
    qmfree(thr, x);
    thr->mode = _mode;
    return res;
  } else {
    thr->mode = _mode;
    return 0;
  }
}

static qmfread(THREAD* thr);

static qmread(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfread(thr));
  } else
    return (0);
}

static qmfreadq(THREAD* thr);

static qmreadq(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfreadq(thr));
  } else
    return (0);
}

static qmfreadc(THREAD* thr);

static qmreadc(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfreadc(thr));
  } else
    return (0);
}

static qmfreads(THREAD* thr);

static qmreads(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfreads(thr));
  } else
    return (0);
}

static qmfwrite(THREAD* thr);

static qmwrite(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[1] = args[0];
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfwrite(thr));
  } else
    return (0);
}

static qmfwriteq(THREAD* thr);

static qmwriteq(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[1] = args[0];
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfwriteq(thr));
  } else
    return (0);
}

static qmfwritec(THREAD* thr);

static qmwritec(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[1] = args[0];
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfwritec(thr));
  } else
    return (0);
}

static qmfwrites(THREAD* thr);

static qmwrites(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[1] = args[0];
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfwrites(thr));
  } else
    return (0);
}

static qmfread(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    int             ret;
    fp = args[0]->data.fp;
#ifdef USE_THREADS
    release_lock();
#endif
    ret = fparsex(fp);
#ifdef USE_THREADS
    acquire_lock();
#endif
    if (ret) {
      EXPR *x;
      int val;
      x = *--thr->xsp;
      val = eval(thr, x);
      qmfree(thr, x);
      return (val);
    } else {
      if (ferror(fp)) clearerr(fp);
      if (thr->qmstat == SYNTAX_ERR || thr->qmstat == STR_ERR ||
	  thr->qmstat == BAD_SYM || thr->qmstat == BAD_REF)
	thr->qmstat = OK;
      return (0);
    }
  } else
    return (0);
}

static qmfreadq(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    fp = args[0]->data.fp;
    if (pushfun(thr, QUOTEOP)) {
      int ret;
#ifdef USE_THREADS
      release_lock();
#endif
      ret = fparsex(fp);
#ifdef USE_THREADS
      acquire_lock();
#endif
      if (ret)
	return (pushfun(thr, APPOP));
      else {
	if (ferror(fp)) clearerr(fp);
	qmfree(thr, *--thr->xsp);
	if (thr->qmstat == SYNTAX_ERR || thr->qmstat == STR_ERR ||
	    thr->qmstat == BAD_SYM || thr->qmstat == BAD_REF)
	  thr->qmstat = OK;
	return (0);
      }
    } else
      return (0);
  } else
    return (0);
}

static qmfreadc(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    int		c;
    char            s[2], *t;
    fp = args[0]->data.fp;
#ifdef USE_THREADS
    release_lock();
    if (fp == stdin) acquire_tty();
#endif
    c = getc(fp);
    if (c == EOF) {
      if (ferror(fp)) clearerr(fp);
#ifdef USE_THREADS
      if (fp == stdin) release_tty();
      acquire_lock();
#endif
      return (0);
    } else if (thr == thr0 && checkbrk) {
      while (c != '\n' && c != EOF)
	c = getc(fp);
#ifdef USE_THREADS
      if (fp == stdin) release_tty();
      acquire_lock();
#endif
      return (0);
    }
#ifdef USE_THREADS
    if (fp == stdin) release_tty();
    acquire_lock();
#endif
    if ((t = strdup(charstr(s, c))) == NULL) {
      thr->qmstat = MEM_OVF;
      return (0);
    } else
      return (pushstr(thr, t));
  } else
    return (0);
}

static qmfreads(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    char            *s = malloc(MAXSTRLEN*sizeof(char)), *t = s, *r;
    int             a = MAXSTRLEN, l;
    if (!s) {
      thr->qmstat = MEM_OVF;
      return (0);
    }
    *s = 0;
    fp = args[0]->data.fp;
#ifdef USE_THREADS
    release_lock();
    if (fp == stdin) acquire_tty();
    pthread_mutex_lock(&reads_mutex);
#endif
    while((r = fgets(t, MAXSTRLEN, fp)) && *t &&
	  t[(l = strlen(t))-1] != '\n') {
      /* try to enlarge the buffer: */
      int k = t-s+l;
      char *s1;
      if (s1 = (char*) arealloc(s, a, MAXSTRLEN,
				sizeof(char))) {
	s = s1;
	t = s+k;
	a += MAXSTRLEN;
      } else {
	free(s);
#ifdef USE_THREADS
	pthread_mutex_unlock(&reads_mutex);
	if (fp == stdin) release_tty();
	acquire_lock();
#endif
	thr->qmstat = MEM_OVF;
	return (0);
      }
    }
    if (*t && t[(l = strlen(t))-1] == '\n')
      t[l-1] = 0;
    if (!(s = realloc(s, strlen(s)+1)))
      fatal(qmmsg[THIS_CANT_HAPPEN]);
#ifdef USE_THREADS
    pthread_mutex_unlock(&reads_mutex);
    if (fp == stdin) release_tty();
    acquire_lock();
#endif
    if (ferror(fp)) {
      clearerr(fp);
      free(s);
      return (0);
    }
    if (!r && !*s || thr == thr0 && checkbrk) {
      free(s);
      return (0);
    } else
      return (pushstr(thr, s));
  } else
    return (0);
}

/* Define this to have the standard output streams flushed on each write
   operation. This has a high performance penalty, but might be required on
   some systems to get correct output in interactive prompt/input
   situations. */

/* #define FLUSH_STDIO */

#ifdef FLUSH_STDIO
static chkflush(FILE *fp)
{
  if (iflag && (fp == stdout || fp == stderr)) fflush(fp);
}
#endif

static qmfwrite(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    fp = args[0]->data.fp;
    if (!fprintx(fp, args[1])) {
      if (ferror(fp)) clearerr(fp);
      return (0);
    } else {
#ifdef FLUSH_STDIO
      chkflush(fp);
#endif
      return (pushfun(thr, VOIDOP));
    }
  } else
    return (0);
}

static qmfwriteq(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR		*x;
  if (args[0]->fno == FILEVALOP && isquote(args[1], x)) {
    FILE           *fp;
    fp = args[0]->data.fp;
    if (!fprintx(fp, x)) {
      if (ferror(fp)) clearerr(fp);
      return (0);
    } else {
#ifdef FLUSH_STDIO
      chkflush(fp);
#endif
      return (pushfun(thr, VOIDOP));
    }
  } else
    return (0);
}

static qmfwritec(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    if (args[1]->type == CHARTYPE) {
      FILE           *fp;
      fp = args[0]->data.fp;
      if (putc(args[1]->data.s[0], fp) == EOF) {
	clearerr(fp);
	return (0);
      } else {
#ifdef FLUSH_STDIO
	chkflush(fp);
#endif
	return (pushfun(thr, VOIDOP));
      }
    } else
      return (0);
  } else
    return (0);
}

static qmfwrites(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    if (args[1]->fno == STRVALOP) {
      FILE           *fp;
      char           *s = args[1]->data.s;
      fp = args[0]->data.fp;
      while (*s)
	if (putc(*s++, fp) == EOF) {
	  clearerr(fp);
	  return (0);
	}
#ifdef FLUSH_STDIO
      chkflush(fp);
#endif
      return (pushfun(thr, VOIDOP));
    } else
      return (0);
  } else
    return (0);
}

static char modestr[3];

static qmfopen(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP && args[1]->fno == STRVALOP) {
    FILE           *fp;
    char           *name = args[0]->data.s;
    char           *mode = args[1]->data.s;

    if ((mode[0] == 'r' || mode[0] == 'w' || mode[0] == 'a') &&
	(mode[1] == '\0' || mode[1] == 'b' && mode[2] == '\0')) {
      strcpy(modestr, mode);
#ifndef MSDOS
      modestr[1] = '\0';
#endif
      if ((fp = fopen(name, modestr)) == NULL)
	return (0);
      else
	return (pushfile(thr, fp));
    } else
      return (0);
  } else
    return (0);
}

static qmpopen(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP && args[1]->fno == STRVALOP) {
    FILE           *fp;
    char           *cmd = args[0]->data.s;
    char           *mode = args[1]->data.s;

    if ((mode[0] == 'r' || mode[0] == 'w') &&
	(mode[1] == '\0' || mode[1] == 'b' && mode[2] == '\0')) {
      strcpy(modestr, mode);
#ifndef MSDOS
      modestr[1] = '\0';
#endif
      if ((fp = popen(cmd, modestr)) == NULL)
	return (0);
      else {
	setlinebuf(fp);
	return (pushpipe(thr, fp));
      }
    } else
      return (0);
  } else
    return (0);
}

static qmfclose(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    int res;
    res = fclose(args[0]->data.fp);
    args[0]->data.fp = NULL;
    args[0]->fno = BADFILEVALOP;
    if (res)
      return 0;
    else
      return pushfun(thr, VOIDOP);
  } else
    return 0;
}

static qmfeof(THREAD* thr);

static qmeof(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[INPUTOP].x) {
    args[0] = (EXPR*) symtb[INPUTOP].x;
    return (qmfeof(thr));
  } else
    return (0);
}

static qmfeof(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    char            c;
    fp = args[0]->data.fp;
#ifdef USE_THREADS
    release_lock();
    if (fp == stdin) acquire_tty();
#endif
    c = getc(fp);
    if (c == EOF)
      if (ferror(fp)) {
	clearerr(fp);
#ifdef USE_THREADS
	if (fp == stdin) release_tty();
	acquire_lock();
#endif
	return 0;
      } else {
#ifdef USE_THREADS
	if (fp == stdin) release_tty();
	acquire_lock();
#endif
	return (pushfun(thr, TRUEOP));
      }
    else {
      ungetc(c, fp);
#ifdef USE_THREADS
      if (fp == stdin) release_tty();
      acquire_lock();
#endif
      return (pushfun(thr, FALSEOP));
    }
  } else
    return (0);
}

static qmfflush(THREAD* thr);

static qmflush(THREAD* thr)
{
  EXPR **args = thr->args;
  if (symtb[OUTPUTOP].x) {
    args[0] = (EXPR*) symtb[OUTPUTOP].x;
    return (qmfflush(thr));
  } else
    return (0);
}

static qmfflush(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == FILEVALOP) {
    FILE           *fp;
    fp = args[0]->data.fp;
    if (fflush(fp) == EOF) {
      clearerr(fp);
      return 0;
    } else
      return (pushfun(thr, VOIDOP));
  } else
    return (0);
}

static qmversion(THREAD* thr)
{
  return pushstr(thr, strdup(version));
}

static qmsysinfo(THREAD* thr)
{
  return pushstr(thr, strdup(sysinfo));
}

static qmwhich(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == STRVALOP) {
    char *s = args[0]->data.s;
    int l0 = strlen(s), l = strlen(qpath)+l0;
    char *buf = malloc((l+3)*sizeof(char));
    char *name = malloc((l+MAXSTRLEN+3)*sizeof(char));
    if (buf && name) {
      absname(name, searchlib(buf, s));
      if (!chkfile(name)) {
	char *t = malloc((l0+3)*sizeof(char));
	if (!t) goto errexit;
	strcat(strcpy(t, s), ".q");
	absname(name, searchlib(buf, t));
	free(t);
      }
      if (chkfile(name)) {
	name = realloc(name, (strlen(name)+1)*sizeof(char));
	return pushstr(thr, name);
      } else {
	free(buf); free(name);
	return 0;
      }
    } else {
    errexit:
      thr->qmstat = MEM_OVF;
      if (buf) free(buf);
      if (name) free(name);
      return 0;
    }
  } else
    return 0;
}

static qmhalt(THREAD* thr)
{
  thr->qmstat = HALT;
  return (0);
}

static qmquit(THREAD* thr)
{
  thr->qmstat = QUIT;
  return (0);
}

static qmbreak(THREAD* thr)
{
  if (thr->brkdbg) thr->brkflag = 1;
  return (pushfun(thr, VOIDOP));
}

static qmcatch(THREAD* thr)
{
  EXPR **args = thr->args;
  EXPR *x = args[0], *y = args[1];
  long ap = thr->asp-thr->ast;
  if (push_mark(thr, x) && eval(thr, y) && (thr > thr0 || !checkbrk)) {
    pop_mark(thr);
    return 1;
  } else if (thr == thr0 && checkbrk ||
	     thr->qmstat >= BREAK && thr->qmstat <= XCEPT) {
    if (thr->qmstat == OK)
      if (thr == thr0 && quitflag)
	thr->qmstat_save = QUIT;
      else
	thr->qmstat_save = BREAK;
    else
      thr->qmstat_save = thr->qmstat;
    thr->qmstat = XCEPT_CATCH;
    /* rewind the activation stack */
    while (thr->asp-thr->ast > ap) free(*--thr->asp);
    return 0;
  } else {
    /* other error */
    pop_mark(thr);
    return 0;
  }
}

static qmthrow(THREAD* thr)
{
  EXPR *x = *--thr->xsp;
  qmfree(thr, *--thr->xsp);
  *(thr->xsp++) = x;
  thr->qmstat = XCEPT;
  return (0);
}

static bool initsig[NSIGNALS];
static sighandler_t old_handler[NSIGNALS];
#ifdef HAVE_POSIX_SIGNALS
static struct sigaction new_action, old_action;
#endif

static qmtrap(THREAD* thr)
{
  EXPR **args = thr->args;
  long prev_action = 0;
#ifdef HAVE_POSIX_SIGNALS
  static bool init = 0;
  int i;
  if (!init) {
    sigemptyset(&new_action.sa_mask);
    new_action.sa_flags = 0;
    init = 1;
  }
#else
  sighandler_t res;
#endif
  if (args[0]->fno == INTVALOP && args[1]->fno == INTVALOP &&
      my_mpz_fits_slong_p(args[1]->data.z)) {
    int flag = mpz_sgn(args[0]->data.z);
    long sig = mpz_get_si(args[1]->data.z);
    if (sig <= 0 || sig > NSIGNALS) return 0;
#ifdef HAVE_POSIX_SIGNALS
    if (flag <= 0) {
      if (flag == 0)
	/* revert to old action */
	if (initsig[sig])
	  new_action.sa_handler = old_handler[sig];
	else
	  goto skip;
      else
	/* new action: ignore */
	new_action.sa_handler = SIG_IGN;
      if (sigaction(sig, &new_action, &old_action))
	return 0;
      sigdelset(&new_action.sa_mask, sig);
    } else {
      /* new action: sig_handler */
      new_action.sa_handler = sig_handler;
      if (sigaction(sig, &new_action, &old_action))
	return 0;
      sigaddset(&new_action.sa_mask, sig);
    }
    /* update the signal mask for the other signals */
    for (i = 1; i <= NSIGNALS; i++)
      if (i != sig && sigismember(&new_action.sa_mask, i)) {
	new_action.sa_handler = sig_handler;
	sigaction(i, &new_action, NULL);
      }
    /* check previous action */
    if (!initsig[sig])
      prev_action = 0;
    else if (old_action.sa_handler == sig_handler)
      prev_action = 1;
    else
      prev_action = -1;
    /* remember the previous handler */
    if (flag == 0)
      initsig[sig] = 0;
    else if (!initsig[sig]) {
      old_handler[sig] = old_action.sa_handler;
      initsig[sig] = 1;
    }
#else
    /* we don't have POSIX signals, so we'll just have to put up with bad ol'
       signal() */
    if (flag == 0)
      if (initsig[sig])
	res = signal(sig, old_handler[sig]);
      else
	goto skip;
    else if (flag < 0)
      res = signal(sig, SIG_IGN);
    else
      res = signal(sig, sig_handler);
    if (res == SIG_ERR) return 0;
    if (!initsig[sig])
      prev_action = 0;
    else if (res == sig_handler)
      prev_action = 1;
    else
      prev_action = -1;
    if (flag == 0)
      initsig[sig] = 0;
    else if (!initsig[sig]) {
      old_handler[sig] = res;
      initsig[sig] = 1;
    }
#endif
  skip:
    return pushint(thr, prev_action);
  } else
    return 0;
}

static qmfail(THREAD* thr)
{
  if (thr->asp > thr->ast)
    thr->qmstat = XCEPT_FAIL;
  return (0);
}

static qmtime(THREAD* thr)
{
  return pushfloat(thr, systime()/1e3);
}

static qmsleep(THREAD* thr)
{
  EXPR **args = thr->args;
  if (args[0]->fno == INTVALOP &&
      mpz_sgn(args[0]->data.z) >= 0) {
    double d = mpz_get_d(args[0]->data.z);
#ifdef USE_THREADS
    release_lock();
#endif
    syssleep(d*1e3);
#ifdef USE_THREADS
    acquire_lock();
#endif
    return (pushfun(thr, VOIDOP));
  } else if (args[0]->fno == FLOATVALOP &&
	     args[0]->data.f >= 0.0) {
#ifdef USE_THREADS
    release_lock();
#endif
    syssleep(args[0]->data.f*1e3);
#ifdef USE_THREADS
    acquire_lock();
#endif
    return (pushfun(thr, VOIDOP));
  } else
    return (0);
}

/* function table ("specials" are NULLed out): */

int             (*funtb[BUILTIN]) () = {

  NULL, NULL, NULL, NULL, qmdef, qmundef, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL,
  NULL, NULL, NULL, NULL,
  qmconcat, qmadd, qmmin, qmmul, qmfdiv, qmdiv, qmmod, qmpow, qmidx, qmumin,
  qmhash, NULL, qmunquote, qmforce, qmor, qmand, qmorelse, qmandthen, qmnot,
  qmle, qmgr, qmeq, qmleq, qmgeq, qmneq, NULL, qmid,
  qmseq,
  qmshl, qmshr, qmpred, qmsucc,
  qmexp, qmln, qmsqrt, qmsin, qmcos, qmatan, qmatan2, qmrandom, qmseed,
  qmsub, qmsubstr, qmpos,
  qmint, qmfrac, qmtrunc, qmround, qmfloat, qmhashnum, qmord, qmchr,
  qmlist, qmtuple, qmstr, qmval, qmstrq, qmvalq,
  qmisspecial, qmisconst, qmisfun, qmisvar, qmisdef, qmflip,
  qmread, qmreadq, qmreadc, qmreads,
  qmwrite, qmwriteq, qmwritec, qmwrites,
  qmfread, qmfreadq, qmfreadc, qmfreads,
  qmfwrite, qmfwriteq, qmfwritec, qmfwrites,
  qmfopen, qmpopen, qmfclose, qmeof, qmfeof, qmflush, qmfflush,
  qmversion, qmsysinfo, qmwhich, qmhalt, qmquit, qmbreak,
  qmcatch, qmthrow, qmtrap, qmfail, NULL,
  qmtime, qmsleep,
  NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL


};

int             nargs[BUILTIN] = {

  0, 0, 0, 0, 2, 1, 0, 0,
  0, 0, 0, 0, 0, 0, 0,
  0, 0, 0,
  0, 0, 0, 0,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 1,
  1, 1, 1, 1, 2, 2, 2, 2, 1,
  2, 2, 2, 2, 2, 2, 0, 2,
  2,
  2, 2, 1, 1,
  1, 1, 1, 1, 1, 1, 2, 0, 1,
  3, 3, 2,
  1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 3,
  0, 0, 0, 0,
  1, 1, 1, 1,
  1, 1, 1, 1,
  2, 2, 2, 2,
  2, 2, 1, 0, 1, 0, 1,
  0, 0, 1, 0, 0, 0,
  2, 1, 2, 0, 0,
  0, 1,
  0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0

};
