
/* sombrero.q: demo for DXL and Tcl/Tk working in concert in a Q script
   July 10 2001 AG */

include dxl, tk;

/* Plot a "sombrero" f(X,Y) = sin (sqrt (X^2 + Y^2)) / (sqrt (X^2 + Y^2)) in
   three dimensions using DX inside a graphical Tcl/Tk application. This
   script shows how to pass a temporary data file and DXLInput values into a
   visual DX program, how to poll for DXLOutput values (in this case, points
   picked in the image), and how to embed DX in a Tcl/Tk application.

   To start the program, simply execute `sombrero N' where N specifies the
   size of the NxN grid on which the function values are sampled. The sombrero
   routine displays the image inside the Tk main window. You can use the mouse
   inside the image to rotate/pan/zoom the view using the standard DX Display
   interactors, which can be selected in the button strip at the bottom of the
   window:

   - Rotate: dragging with the left mouse button rotates in 3, right mouse
     button in 2 dimensions

   - Pan: dragging with the left mouse button moves the graph around in the
     image

   - Zoom: dragging with the left mouse button downwards zooms in on the
     image, dragging upwards zooms out

   Furthermore, in any mode clicking with the middle mouse button allows you
   to pick a point on the graph, which is displayed in the "Results" field at
   the top of the main window.

   Three additional check buttons at the bottom of the window can be used to
   switch between hardware and software rendering, normal and wireframe
   display, and antialiasing on/off (wireframe display and antialiasing
   automatically switch on hardware rendering).

   The main window also contains an "Expression" field into which you can
   enter arbitrary Q expressions and evaluate them by hitting the carriage
   return key or by pushing the Eval button. This also allows you to evaluate
   some special "commands" to control the DX server and manipulate various
   aspects of the image:

   - reset: reinitialize the DX server and reset the camera to the initial
     position

   - caption S: change the caption on the image (S a string value)

   - scale X: change the elevation of the sombrero (default is 2.5)

   - software, hardware: change the rendering mode

   - antialias N: enable (N=1) or disable (N=0) antialiasing

   - none, box, dots, wireframe: change the rendering approximation

   - save NAME, format FORMAT: save the current image in the given file, and
   set the image format for subsequent saves to the given value (use format ()
   to reset the format to the default)

   Note that `save' simply captures the window contents when in hardware mode,
   while it rerenders the image using the software renderer (Render module)
   in software mode. See the description of the WriteImage module in the DX
   reference guide for information on the supported formats. If no format is
   set, DX picks the format indicated by the filename extension. Otherwise, an
   appropriate default filename extension is supplied.

   Finally, the Quit button terminates the application. Alternatively, you can
   also close the main Tk window. In either case, you are returned to the
   command prompt of the Q interpreter.

   NOTE: Recent OpenDX versions have a bug which causes the DX executive to
   hang when the application is exited and hardware rendering is on. (At least
   this happens on my Linux system, YMMV.) If this happens to you, you will
   have to exit the Q interpreter and kill the dxexec and dxworker processes.

   The visual DX program which actually realizes the plot can be found in
   sombero.net. It should be instructive to take a look at this program (with
   `dx -edit'), in particular to see how the SuperviseWindow and
   SuperviseState modules are used to render the image inside a Tk window, and
   how the pick feature is implemented. */

/* The data. ****************************************************************/

// compute the sombrero function on a regular 2D grid in the range
// (X1,Y1)-(X2,Y2) with a given number N of grid lines

def X1 = -8.0, Y1 = -8.0, X2 = 8.0, Y2 = 8.0;

mydata N    = map f (grid N);

// origin and deltas of a grid with N lines

orig	    = (X1,Y1);
deltas N    = ((X2-X1)/(N-1),(Y2-Y1)/(N-1));

// the sombrero function

f (X,Y)     = sin R / R if R>0 where R = sqrt (X^2 + Y^2);
	    = 1.0 otherwise;

// regular grid (last index varies fastest, since this is what DX expects)

grid N	    = listof (X0+I*DX,Y0+J*DY)
	      (I in nums 0 (N-1), J in nums 0 (N-1))
	      where (X0,Y0) = orig, (DX,DY) = deltas N;

/* DX data file creation. ***************************************************/

/* While smaller amounts of input data can be fed into DX more easily using
   dxl_input, an entire input field is best passed using a temporary data file
   in DX format. Here we create a DX file for 2D regular grid data, using the
   dxl_file convenience function provided by the dxl module. */

data_file NAME N (X0,Y0) (DX,DY) DATA
	    = dxl_file NAME
	      (regular_positions [N,N] [X0,Y0] [[DX,0],[0,DY]])
	      (regular_connections [N,N] "quads")
	      (data "data" "positions" "float" [] DATA);

/* The main program. ********************************************************/

// set up a connection to the DX server

def H = dxl_start "dx -exonly", RET = dxl_load H "sombrero.net";

sombrero N = init_app || run_app tmpnam N if N>1;

// init_app: initialize the interface (create the application's widgets);
// this has been adapted from the tk_examp script

init_app
= // start a new interpreter
  tk_quit ||
  // unmap the main window (will be shown when all widgets have
  // been created, to reduce flickering)
  tk "wm withdraw ." ||
  // set the window title and min dimensions
  tk "wm geometry . 540x520" ||
  tk "wm title . \"Sombrero\"" ||
  // title label
  tk "label .title -font 8x13bold -text \
\"Sombrero: Q/Tk/DX Demo Application\"" ||
  tk "label .descr -text \
\"Type in a Q expression and click Eval to evaluate.\n\
Drag the image in Rotate/Pan/Zoom mode; click Quit to quit.\"" ||
  // main area: input expression and output result with labels
  tk "frame .f" ||
  tk "label .f.expr_l -text \"Expression:\"" ||
  tk "entry .f.expr_e -textvariable expr_val -width 40" ||
  tk "label .f.result_l -text \"Result:\"" ||
  tk "entry .f.result_e -textvariable result_val -width 40" ||
  // image area used as the parent of the DX diplay window
  tk "frame .image -container 1" ||
  // buttons
  tk "frame .buttons" ||
  tk "checkbutton .buttons.hardware -text HW3D \
-command { q hardware_cb $hardware }" ||
  tk "checkbutton .buttons.wireframe -text Wireframe \
-command { q wireframe_cb $wireframe }" ||
  tk "checkbutton .buttons.antialias -text AA \
-command { q antialias_cb $antialias }" ||
  tk "radiobutton .buttons.rot -text Rotate -value 1 \
-command { q rot_cb }" ||
  tk "radiobutton .buttons.pan -text Pan -value 2 \
-command { q pan_cb }" ||
  tk "radiobutton .buttons.zoom -text Zoom -value 3 \
-command { q zoom_cb }" ||
  tk "button .buttons.reset -text Reset \
-command { q reset_cb }" ||
  tk "button .buttons.eval -text Eval \
-command { q eval_cb }" ||
  tk "button .buttons.quit -text Quit \
-command { q quit_cb }" ||
  tk "set selectedButton 1" ||
  // pack widgets into the main window
  // title and main frame
  tk "pack .title .descr" ||
  tk "pack .f -fill x" ||
  // grid layout of the main frame
  tk "grid config .f.expr_l -column 0 -row 0 -sticky w" ||
  tk "grid config .f.expr_e -column 1 -row 0 -sticky we" ||
  tk "grid config .f.result_l -column 0 -row 1 -sticky w" ||
  tk "grid config .f.result_e -column 1 -row 1 -sticky we" ||
  // make entry widgets expand horizontally to fill main window
  tk "grid columnconfigure .f 0 -weight 0" ||
  tk "grid columnconfigure .f 1 -weight 1" ||
  // image area
  tk "pack .image -fill both -expand 1" ||
  // buttons
  tk "pack .buttons" ||
  tk "pack .buttons.hardware .buttons.wireframe .buttons.antialias \
.buttons.rot .buttons.pan .buttons.zoom \
.buttons.reset .buttons.eval .buttons.quit -side left" ||
  // bind Return key in expression entry to evaluation command
  tk "bind .f.expr_e <Return> { q eval_cb }" ||
  // show the main window
  tk "wm deiconify .";

// run_app: create input data, set up DXLInput/Output channels and enter the
// main loop

run_app NAME N
           = data_file NAME N orig (deltas N) (mydata N) || plot NAME N;

plot NAME N
	   = // pass filename and window parameters
	     dxl_input H "datafile" (str NAME) ||
	     dxl_input H "window" (str (val (tk "winfo id .image"))) ||
	     dxl_input H "size"
	     (sprintf "[%s,%s]" (tk "winfo width .image",
				 tk "winfo height .image")) ||
	     dxl_input H "caption"
	     (str (sprintf "The sombrero, on a %dx%d grid" (N,N))) ||
	     // initialize some other settings
	     dxl_input H "mode" "0" ||
	     software || antialias 0 || none || format () ||
	     // DXLOutput for pick positions
	     dxl_output H "pick" ||
	     dxl_exec_on_change H ||
	     install_idle_cb ||
	     main_loop NAME;

// main_loop: the main loop of the application

main_loop NAME
	   = tk_read NAME || main_loop NAME if tk_ready;
	   = dxl_end_exec H || unlink NAME otherwise;

// special commands

reset      = dxl_reset H ||
	     dxl_input H "reset" "1" ||
	     dxl_exec_once H ||
	     dxl_wait H ||
	     dxl_input H "reset" "0" ||
	     dxl_exec_on_change H;

caption S:String
	   = dxl_input H "caption" (str S);

scale X:Num
	   = dxl_input H "scale" (str X);

software   = dxl_input H "rendering_mode" (str "software");
hardware   = dxl_input H "rendering_mode" (str "hardware");

antialias N:Int
	   = dxl_input H "antialias" (str N) if (N=0) or (N=1);

none	   = dxl_input H "rendering_approx" (str "none");
box	   = dxl_input H "rendering_approx" (str "box");
dots	   = dxl_input H "rendering_approx" (str "dots");
wireframe  = dxl_input H "rendering_approx" (str "wireframe");

save NAME:String
	   = dxl_input H "imagefile" (str NAME) ||
	     dxl_wait H ||
	     dxl_input H "imagefile" "NULL";

format ()  = dxl_input H "format" "NULL";
format FORMAT:String
	   = dxl_input H "format" (str FORMAT);

// retrieve the current expression and set expression/result values

private expr, set_result S;

expr	   = tk "return $expr_val";

set_expr S = tk ("set expr_val {" ++ S ++ "}");

set_result S
	   = tk ("set result_val {" ++ S ++ "}");

// check syntax of expression and evaluate

private res, eval;

res 'X     = set_expr "" || set_result (str X);
res X	   = set_result "*** syntax error ***" otherwise;

eval S     = res (valq S);

// callbacks

hardware_cb K
	   = software if K=0;
	   = hardware otherwise;

wireframe_cb K
	   = none if K=0;
	   = tk "set hardware 1" || hardware || wireframe otherwise;


antialias_cb K
	   = antialias 0 if K=0;
	   = tk "set hardware 1" || hardware || antialias 1 otherwise;

rot_cb     = dxl_input H "mode" "0";

pan_cb     = dxl_input H "mode" "1";

zoom_cb    = dxl_input H "mode" "2";

eval_cb    = eval expr;

reset_cb   = reset;

quit_cb    = tk_quit;

idle_cb    = check || install_idle_cb;

// install an `after idle' callback which regularly checks for DX messages

install_idle_cb
	   = tk "after 100 { after idle { q idle_cb } }";

// poll the DX server for messages

check      = dxl_msg (dxl_read H) || check
		 if dxl_ready H and then dxl_check H;
	   = () otherwise;

dxl_msg (NAME,VAL)
	   = set_result (sprintf "%s = %s" (NAME,VAL));
dxl_msg (dxl_error MSG)
	   = set_result (sprintf "*** DXL error: %s ***" MSG);
dxl_msg X  = set_result (sprintf "*** unknown message %s from DXL ***" (str X))
		 otherwise;
