
/* magick.q: the Q programmer's ImageMagick interface
   $Id: magick.q,v 1.33 2004/01/06 01:03:07 agraef Exp $ */

/* This file is part of the Q programming system.

   The Q programming system is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   The Q programming system is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

/* This module provides access to the ImageMagick C API (libMagick), a
   comprehensive library for image manipulation, which is available from
   http://www.imagemagick.org/. All essential ImageMagick operations are
   supported, thus you can create, read and write images in all ImageMagick-
   supported formats, access the pixel data of an image, apply advanced image
   manipulation and effect functions, and draw a variety of graphic primitives
   on an image. Since the external pixel representation is compatible with
   that of the Q-GGI module, you can easily exchange pixel data between images
   and GGI visuals as well. */

import clib;

/***************************************************************************/

/* Manifest constants. */

/* This is used to indicate the default value for various options. */
public var const DEFAULT;
def DEFAULT = 0;

public var const
  /* Compression types for the write_image and image_to_blob functions. */
  NONE, BZIP, FAX, GROUP4, JPEG, LOSSLESS, LZW, RLE, ZIP,

  /* Gravity types used by the DRAW_INFO structure. */
  FORGET, NORTH_WEST, NORTH, NORTH_EAST, WEST, CENTER, EAST, SOUTH_WEST,
  SOUTH, SOUTH_EAST, STATIC,

  /* Colorspace values, used by the quantize and segment operations. */
  RGB, GRAY, TRANSPARENT, OHTA, XYZ, YCbCr, YCC, YIQ, YPbPr, YUV, CMYK,
  SRGB,

  /* Filter types for the resize function. */
  POINT_FILTER, BOX_FILTER, TRIANGLE_FILTER, HERMITE_FILTER,
  HANNING_FILTER, HAMMING_FILTER, BLACKMAN_FILTER, GAUSSIAN_FILTER,
  QUADRATIC_FILTER, CUBIC_FILTER, CATROM_FILTER,  MITCHELL_FILTER,
  LANCZOS_FILTER, BESSEL_FILTER, SINC_FILTER,

  /* Compose operators for the composite function. */
  OVER_OP, IN_OP, OUT_OP, ATOP_OP, XOR_OP, PLUS_OP, MINUS_OP, ADD_OP,
  SUBTRACT_OP, DIFFERENCE_OP, MULTIPLY_OP, BUMPMAP_OP, COPY_OP, COPYRED_OP,
  COPYGREEN_OP, COPYBLUE_OP, COPYOPACITY_OP, CLEAR_OP, DISSOLVE_OP,
  DISPLACE_OP, MODULATE_OP, THRESHOLD_OP, NO_OP, DARKEN_OP, LIGHTEN_OP,
  HUE_OP, SATURATE_OP, COLORIZE_OP, LUMINIZE_OP, SCREEN_OP, OVERLAY_OP,

  /* Noise types for the add_noise function. */
  UNIFORM_NOISE, GAUSSIAN_NOISE, MULTIPLICATIVE_GAUSSIAN_NOISE,
  IMPULSE_NOISE, LAPLACIAN_NOISE, POISSON_NOISE;

private extern magick_vars;
def (NONE, BZIP, FAX, GROUP4, JPEG, LOSSLESS, LZW, RLE, ZIP,

     FORGET, NORTH_WEST, NORTH, NORTH_EAST, WEST, CENTER, EAST, SOUTH_WEST,
     SOUTH, SOUTH_EAST, STATIC,

     RGB, GRAY, TRANSPARENT, OHTA, XYZ, YCbCr, YCC, YIQ, YPbPr, YUV, CMYK,
     SRGB,

     POINT_FILTER, BOX_FILTER, TRIANGLE_FILTER, HERMITE_FILTER,
     HANNING_FILTER, HAMMING_FILTER, BLACKMAN_FILTER, GAUSSIAN_FILTER,
     QUADRATIC_FILTER, CUBIC_FILTER, CATROM_FILTER,  MITCHELL_FILTER,
     LANCZOS_FILTER, BESSEL_FILTER, SINC_FILTER,

     OVER_OP, IN_OP, OUT_OP, ATOP_OP, XOR_OP, PLUS_OP, MINUS_OP, ADD_OP,
     SUBTRACT_OP, DIFFERENCE_OP, MULTIPLY_OP, BUMPMAP_OP, COPY_OP, COPYRED_OP,
     COPYGREEN_OP, COPYBLUE_OP, COPYOPACITY_OP, CLEAR_OP, DISSOLVE_OP,
     DISPLACE_OP, MODULATE_OP, THRESHOLD_OP, NO_OP, DARKEN_OP, LIGHTEN_OP,
     HUE_OP, SATURATE_OP, COLORIZE_OP, LUMINIZE_OP, SCREEN_OP, OVERLAY_OP,

     UNIFORM_NOISE, GAUSSIAN_NOISE, MULTIPLICATIVE_GAUSSIAN_NOISE,
     IMPULSE_NOISE, LAPLACIAN_NOISE, POISSON_NOISE)
	 = magick_vars;

/***************************************************************************/

/* General ImageMagick properties and functions. */

/* Error handling. In case of failure, some operations use the following
   magick_error symbol to return a diagnostic message. You can define this
   symbol to perform error handling as appropriate for the application. */

public magick_error MSG;

/* ImageMagick file formats. The following operation retrieves general
   information about the supported image file formats. It returns a list of
   string tuples of the form (NAME, MODE, DESCRIPTION, VERSION) with the
   following fields:

   - NAME: The name of the format under which it is known to ImageMagick,
     which is also used as a format prefix in filenames and the value of the
     MAGICK field in the image_info structure (see below).

   - MODE: A four-character string of the form "*rw+" specifying whether the
     format supports native blobs, reading, writing, and multi-image files,
     respectively. Unsupported capabilities will be represented by a '-'
     character in the corresponding position.

   - DESCRIPTION: A more verbose description of the file format.

   - VERSION: Any additional version information about the format.

   E.g., to quickly list all supported formats, try something like
   `do (printf "%9s  %4s  %s %s\n") magick_info'. We remark that some of the
   listed formats (such as MPEG) require third-party software to make them
   work, see ImageMagick(1) for details. */

public extern magick_info;

/* ImageMagick resource management (IM > 5.4 only). You can set a limit on the
   resource RES (which can be any of the resource types for ImageMagick's
   -limit option, see ImageMagick(1) for details; currently "memory", "map"
   and "disk" are supported) with the magick_limit function, use the
   magick_limits function to print the current limits on standard output, and
   retrieve the current resource usage figures with magick_resources. */

public extern magick_limit RES VAL, magick_limits, magick_resources;

/* Encoding of color values. Regardless of the internal representation of an
   image, pixel data is always encoded as a byte string of 16 bit RGBA
   values. A value of 0 in the RGB components denotes minimum, 0xffff maximum
   intensity. Similarly, an alpha value of 0 denotes a fully transparent,
   0xffff a fully opaque pixel. Note that this format matches the layout of
   GGI pixel values, which makes it easy to transfer pixel data between
   ImageMagick images and GGI visuals. */

/* The magick_pixel function returns the normalized RGBA color value, as a
   byte string, for a given color specification, which may be either a color
   name in the ImageMagick color database or an (R,G,B,A) tuple with values in
   the range 0..0xffff; the alpha value is optional and is assumed to be
   0xffff if it is missing. The function also packs a list of such values into
   a single byte string. The returned value can be passed as the PIXELS
   parameter of create_image and set_image_pixels.

   The magick_color function converts an RGBA byte string back to an (R,G,B,A)
   tuple or a list of such tuples. */

public extern magick_pixel COLOR, magick_color PIXEL;

/* The magick_colors function returns all color names known to ImageMagick
   which match the given glob pattern. */

public extern magick_colors PATTERN;

/***************************************************************************/

/* The Image type. Objects of this type are created with the create_image,
   read_image, ping_image and blob_to_image operations, and processed by the
   other operations of this module. */

public extern type Image;

public is_image X;

is_image IMG:Image	= true;
is_image _		= false otherwise;

/* The most important attributes of an image are summarized in its INFO
   structure, a tuple of the from (WIDTH, HEIGHT, OFFSET, DEPTH, MATTE,
   MAGICK) with the following components:

   - WIDTH, HEIGHT: The dimensions of the image, in pixel units.

   - OFFSET: A number of initial bytes to be skipped when reading an
     image. Usually, this is set to zero, but there are some special image
     formats (in particular: STEGANO) which may use a non-zero value for this
     field.

   - DEPTH: The bit depth of the image (usually 8 or 16), denotes the number
     of bits used to represent each color component or "plane" (red, green,
     blue, opacity or "alpha").

   - MATTE: An integer value which is 1 if the image has an alpha channel and
     0 otherwise.

   - MAGICK: A string naming one of the image formats understood by
     ImageMagick (such as EPS, GIF, RGB, etc.).

   INFO tuples are used to specify the properties of an image when it is
   created, and the INFO tuple of an existing image can be retrieved with the
   image_info function. */

public extern image_info IMG;

/* Convenience functions to retrieve the different components of an INFO
   structure. */

public image_width IMG, image_height IMG, image_offset IMG, image_depth IMG,
  image_matte IMG, image_magick IMG;

image_width IMG		= INFO!0 where INFO:Tuple = image_info IMG;
image_height IMG	= INFO!1 where INFO:Tuple = image_info IMG;
image_offset IMG	= INFO!2 where INFO:Tuple = image_info IMG;
image_depth IMG		= INFO!3 where INFO:Tuple = image_info IMG;
image_matte IMG		= INFO!4 where INFO:Tuple = image_info IMG;
image_magick IMG	= INFO!5 where INFO:Tuple = image_info IMG;

/* Image attributes related to drawing are kept in a separate DRAW_INFO
   structure which has the following fields:

   - FONT: A string, the name of the font to be used for drawing text.

   - POINTSIZE: Integer or floating point number, the font size.

   - GRAVITY: The default gravity of rendered text. Can be any of the gravity
     values listed in the manifest constants section.

   - FILL, STROKE, UNDERCOLOR: Pixel values specifying the default colors for
     the graphics primitives. The FILL color is used for the interior, STROKE
     for the outline of graphic objects; use FILL = magick_pixel "none" if no
     filling is desired. UNDERCOLOR determines the background color for text
     rendering.

   - BORDER_COLOR: Pixel value, determines the border color for flood fill
     operations.

   - STROKE_WIDTH: Integer or floating point value, the linewidth of outlines
     drawn with the graphic primitives.

   - STROKE_ANTIALIAS, TEXT_ANTIALIAS: Two flags (either true or false),
     indicating whether to perform antialiasing when rendering text and other
     graphic objects.

   - TILE: An image to be tiled when filling a graphic object, () if none.

   The DRAW_INFO structure and its members can be accessed with the operations
   listed below. */

/* FIXME: What about the other font properties (family, style, stretch,
   weight, alignment) and rendering attributes (gradients, fill rule,
   linecaps/-joins, miterlimit, dash offset and pattern) which are
   undocumented but can be found in libMagick's DrawInfo struct? */

public extern draw_info IMG;

public draw_font IMG, draw_pointsize IMG, draw_gravity IMG, draw_fill IMG,
  draw_stroke IMG, draw_undercolor IMG, draw_border_color IMG,
  draw_stroke_width IMG, draw_stroke_antialias IMG, draw_text_antialias IMG,
  draw_tile IMG;

draw_font IMG		= INFO!0 where INFO:Tuple = draw_info IMG;
draw_pointsize IMG	= INFO!1 where INFO:Tuple = draw_info IMG;
draw_gravity IMG	= INFO!2 where INFO:Tuple = draw_info IMG;
draw_fill IMG		= INFO!3 where INFO:Tuple = draw_info IMG;
draw_stroke IMG		= INFO!4 where INFO:Tuple = draw_info IMG;
draw_undercolor IMG	= INFO!5 where INFO:Tuple = draw_info IMG;
draw_border_color IMG	= INFO!6 where INFO:Tuple = draw_info IMG;
draw_stroke_width IMG	= INFO!7 where INFO:Tuple = draw_info IMG;
draw_stroke_antialias IMG
			= INFO!8 where INFO:Tuple = draw_info IMG;
draw_text_antialias IMG	= INFO!9 where INFO:Tuple = draw_info IMG;
draw_tile IMG		= INFO!10 where INFO:Tuple = draw_info IMG;

public extern set_draw_font IMG FONT,
  set_draw_pointsize IMG POINTSIZE,
  set_draw_gravity IMG GRAVITY,
  set_draw_fill IMG FILL,
  set_draw_stroke IMG STROKE,
  set_draw_undercolor IMG UNDERCOLOR,
  set_draw_border_color IMG BORDER_COLOR,
  set_draw_stroke_width IMG STROKE_WIDTH,
  set_draw_stroke_antialias IMG STROKE_ANTIALIAS,
  set_draw_text_antialias IMG TEXT_ANTIALIAS,
  set_draw_tile IMG TILE;

/* To provide support for laying out text in an image, the type_metrics
   function gives access to the type metrics of the current font set on an
   image. It returns a tuple with the following information:

   - PPEM_X, PPEM_Y: pixels per em, character width and height in pixels

   - ASCENT, DESCENT: ascent and descent of the font

   - WIDTH, HEIGHT: text width and height

   - ADVANCE: maximum horizontal advance

   - UNDERLINE_POSITION, UNDERLINE_THICKNESS: underline position and
     thickness

   Note that all figures are in pixel units. Since fractional font sizes are
   supported, the values are reported as floating point numbers. */

public extern type_metrics IMG;

/* A few other image properties can be retrieved and changed with the
   operations listed below. The following properties are currently supported:

   - background_color, border_color, matte_color: The image background, border
     and matte colors, specified as RGBA pixel values.

   - compression: The image's compression method, used as the default method
     when writing the image. This can be any of the compression type values
     defined in the manifest constants section at the beginning of this
     module.

   - fuzz: This integer value is used to match pixels in various operations,
     such as the flood_fill functions. The default value is zero, meaning that
     colors only match if they are identical. You can set this value to a
     positive number to increase the range of colors which match a given pixel
     value.

   - page: The equivalent PostScript page size of the image, specified as a
     quadrupel (X,Y,W,H), where (X,Y) denotes the page offset and (W,H) its
     dimension. This information is used when saving the image in PostScript
     format (in which case the coordinates are interpreted in PostScript
     coordinate space). The (X,Y) coordinates are also used by various image
     manipulation routines which combine images, such as flatten and mosaic
     (in this case the coordinates are in pixels). When setting this value,
     you can either specify the entire quadrupel, or only the (X,Y) values; in
     the latter case only the offset values are changed. */

public extern image_background_color IMG, set_image_background_color IMG PIXEL;
public extern image_border_color IMG, set_image_border_color IMG PIXEL;
public extern image_matte_color IMG, set_image_matte_color IMG PIXEL;

public extern image_compression IMG, set_image_compression IMG COMPRESSION;
public extern image_fuzz IMG, set_image_fuzz IMG FUZZ;
public extern image_page IMG, set_image_page IMG PAGE;

/* The following operations let you retrieve and change named string
   attributes associated with an image. To delete a key from the attributes
   list of an image, specify () as the VAL parameter of set_image_attr.
   Otherwise the given value is added to the attribute (an existing value is
   *not* overridden automatically, you first have to set the attribute to ()
   to accomplish this). Attributes used by ImageMagick include "comment",
   "label" and "signature". In particular, the "label" attribute is useful
   with the montage operation (see below). */

public extern image_attr IMG KEY, set_image_attr IMG KEY VAL;

/* Convenience functions to access the "comment" and "label" attributes. */

public image_comment IMG, set_image_comment IMG COMMENT;

image_comment IMG:Image	= COMMENT
			    where COMMENT:String = image_attr IMG "comment";
set_image_comment IMG:Image COMMENT:String
			= set_image_attr IMG "comment" () ||
			  set_image_attr IMG "comment" COMMENT;

public image_label IMG, set_image_label IMG LABEL;

image_label IMG:Image	= LABEL
			    where LABEL:String = image_attr IMG "label";
set_image_label IMG:Image LABEL:String
			= set_image_attr IMG "label" () ||
			  set_image_attr IMG "label" LABEL;

/***************************************************************************/

/* Retrieve various other useful information about an image. The count_image_
   colors function determines the number of unique colors in an image. The
   remaining functions check whether the image belongs to one of the following
   image classes: gray (all RGB intensities equal), monochrome (all RGB
   intensities equal *and* each RGB value is either zero or max), opaque (all
   alpha values max), and palette (pseudo class image with at most 256 unique
   colors). */

public extern count_image_colors IMG;
public extern is_gray_image IMG, is_monochrome_image IMG, is_opaque_image IMG,
  is_palette_image IMG;

/***************************************************************************/

/* Operations to create and copy images. The create_image function creates a
   new image, given its INFO tuple. Only the first two fields specifying the
   dimensions of the image are mandatory, the remaining fields will be filled
   with suitable defaults if they are omitted or zero. The image will be
   created with the pixels initialized from the given PIXELS argument, a byte
   string of RGBA values representing either a single pixel or a pixel
   sequence whose length matches the dimensions of the image.

   The clone_image function creates an exact copy of the given image. This is
   useful if an image is to be modified but you want to retain the original
   image. */

public extern create_image INFO PIXELS, clone_image IMG;

/* Operations to read and write images. The read_image function reads an image
   from the given source, which may also include a format prefix (such as
   "rgb:file") or indicate a special builtin image object (such as "logo:" or
   "xc:white"); see ImageMagick(1) for details. Additional image properties
   may be specified in the given INFO tuple. For read_image, all fields in the
   INFO structure are optional, but note that for raw image data (e.g., RGB or
   RGBA) you will have to specify at least the dimensions of the image. For
   STEGANO images ("watermark" images hidden in the main image, specified as
   "stegano:file") you also have to specify the offset of the stegano image in
   the file using the OFFSET field of the INFO structure.

   Note that some image files (such as GIF animations or MPEG videos) actually
   contain an entire sequence of images (which are also called "frames")
   rather than a single image; in this case read_image returns a list with all
   the images that were read from the file. You can also specify a subrange of
   frames to be extracted in the input filename (e.g., "video.mpg[50-75]"),
   see ImageMagick(1) for details. Be warned that reading an entire video file
   will eat *huge* amounts of memory. You can work around some of this by
   carefully tuning ImageMagick's resource limits with magick_limit, but, at
   least at the time of this writing, ImageMagick is not really up to handling
   huge amounts of video data yet.

   The ping_image function is like read_image, but does *not* read the actual
   pixel data of an image. This is typically used in conjuction with
   image_info, in order to inspect the properties of an image or image
   sequence without wasting resources for reading the pixels into memory.

   The write_image function writes the given image to the given target,
   designated by a name in the same syntax as for read_image. If the input
   image is actually a list of images, it is written to a single multi-image
   file, if the image format supports that, or to a sequence of files with
   attached frame numbers otherwise. You can also use a %d printf
   specification in the output filename (e.g., "image%02d.miff") which expands
   to the zero-based frame number when the individual frames are written, see
   again ImageMagick(1) for details.

   The compression type and quality of write_image can optionally be indicated
   with the INFO parameter, which can be a singleton COMPRESSION value, or a
   tuple with up to 3 fields which takes the form (COMPRESSION, QUALITY,
   SAMPLING_FACTOR). COMPRESSION can be any of the compression type values
   defined in the "manifest constants" section, QUALITY is an integer (ranging
   from 0 to 100 a.k.a. "worst/fastest" to "best/slowest", with 75 being the
   default) which corresponds to ImageMagick's -quality option, and
   SAMPLING_FACTOR is a string in the same format as ImageMagick's
   -sampling-factor option, see ImageMagick(1) for details.  All fields are
   optional. If no compression type is specified (COMPRESSION = 0 or INFO =
   ()) then the default compression type for the image format is used. */

public extern read_image NAME INFO, ping_image NAME INFO;
public extern write_image NAME IMG INFO;

/* Convert an image or image sequence to a "blob" ("binary large object",
   represented as a byte string), and vice versa. This is useful, e.g., if an
   image is to be stored in a database, or if it is to be manipulated as raw
   data. The MAGICK parameter of image_to_blob can be used to indicate the
   desired target format; if it is () then the format of the image is used as
   the target format. The INFO parameter of image_to_blob can be used to
   denote the desired compression method, and has the same format and meaning
   as with the write_image function. The blob_to_image function takes an image
   INFO structure to specify image properties which cannot be deduced from the
   blob itself; this is necessary, e.g., if the blob is just raw RGB data. */

public extern image_to_blob MAGICK IMG INFO, blob_to_image BLOB INFO;

/***************************************************************************/

/* Get and set the pixel data of an image. The affected area of the image is
   specified by its origin P = (X,Y) and dimension DIM = (W,H). All pixel data
   is encoded as a byte string of 16 bit RGBA values, as discussed above. */

public extern get_image_pixels IMG P DIM;
public extern set_image_pixels IMG P DIM PIXELS;

/***************************************************************************/

/* Image manipulation operations. Please refer to the ImageMagick manual for
   further information about these. Note that, for efficiency reasons, some
   operations are of the "BANG! type", i.e., they modify the given target
   image(s) in-place. In the following, these are marked with `BANG!'. BANG! 
   type functions return () rather than a new image as the result, and if you
   want to retain the original image you will have to clone it beforehand. All
   other operations return a new image, and the original image is left
   unchanged.

   Most operations have the same name as in the C API (without the `Image'
   suffix), with the following exceptions: (1) `FlipImage' and `FlopImage' are
   named `flipy' and `flipx'. (Or was it the other way round? Go figure. ;-)
   (2) `RaiseImage' is named `button' (that's more suggestive, too, and btw a
   `raise' function is already in clib, so we don't want to reuse that name).

   A note on parameters: IMG always denotes a singleton input image, IMGS an
   image list. Some operations also take a secondary image IMG2 as an
   additional (r/o) parameter. P = (X,Y) is used to indicate a position,
   offset or displacement vector, DIM = (W,H) the dimensions of a rectangle,
   RADIUS a radius, SIGMA a standard deviation value, and ANGLE an angle in
   degrees. PIXEL is an RGBA color value in byte string format. Some
   operations have a generic string parameter called VAL, which must be in the
   same format as the parameter for the corresponding ImageMagick command line
   option. Other parameters are mostly different numeric values used by the
   various algorithms. These are explained in more detail in the ImageMagick
   manual. */

/* FIXME: Some of the stuff below really needs further documentation. For the
   time being, please take a look at what the corresponding C or Perl
   functions or ImageMagick command line options do. */

/* Quantization. This operation is used to remap the colors of an image to a
   fixed number of "similar" colors in a given colorspace.

   - The COLORSPACE argument designates the target colorspace, and can be
     either (1) one of the colorspace values listed in the manifest constants
     section (RGB, GRAY etc.), or (2) a reference image IMG2 specifying the
     colormap to be applied to the target image, or (3) a tuple with up to 3
     fields, either (COLORSPACE[,MAXCOLORS][,DITHER]) or (IMG2[,DITHER]). The
     MAXCOLORS value denotes the maximum number of colors in the quantized
     image and defaults to 256. The DITHER flag determines whether to apply
     Floyd/Steinberg dithering to the image; it defaults to true.

   - The target image IMG can also be a list of images which are all to be
     mapped to the same colormap.

   - Please note that this is a BANG! type operation, thus if you you want to
     retain the original images you will have to clone them first. */

public extern quantize IMG COLORSPACE;				// BANG!

/* Resizing, scaling and transformations. */

public extern chop IMG P DIM, crop IMG P DIM, shave IMG DIM;

public extern magnify IMG, minify IMG;
public extern resize IMG DIM FILTER BLUR, sample IMG DIM, scale IMG DIM;
public extern thumbnail IMG DIM; // IM > 5.4 only

public extern flipx IMG, flipy IMG, roll IMG P;
public extern rotate IMG ANGLE, shear IMG SHEAR_ANGLES;
public extern affine_transform IMG MATRIX P;

/* Combining and deconstructing images. */

public extern coalesce IMGS, flatten IMGS, mosaic IMGS;
public extern deconstruct IMGS;
public extern composite IMG OP IMG2 P;				// BANG!
public extern segment IMG COLORSPACE VERBOSE CLUSTER SMOOTH;	// BANG!

/* The montage function creates a montage of thumbnail images. The following
   parameters are set via the INFO tuple (see also montage(1) for a closer
   description of the various options):

   - TITLE: the title of the image, as a string, () for none

   - TILE: the layout of the tiles, e.g.: "4x3", () for default

   - GEOMETRY: the geometry of the tiles, e.g.: "128x128+10+10", () for
     default

   - FRAME: the geometry of the 3D frame to place around the tiles, () if no
     frame is desired; e.g.: "15x15+3+3"

   - BORDERWIDTH: the size of a border to draw around each tile (an integer);
     this is only active if no frame is specified

   - SHADOW: true or false, determines whether an additional shadow is drawn
     behind each tile

   - TEXTURE: a background texture such as "granite:", () if none

   - FONT: the font to be used for the title and tile labels, () if none

   - POINTSIZE: the font size (integer or floating point number)

   - BACK, FORE, MATTE: pixel values used for the image's background,
     foreground and frames, respectively

   All fields are optional, but must be specified in the given order.
   ImageMagick defaults are used for omitted values. You can also specify just
   the title string, or leave all fields empty by specifying () for the INFO
   tuple.

   Note that this operation may also return a list of images, if the tiles do
   not all fit on one page. Also note that to get labels on the images, you
   have to set them first with the set_image_attr function. */

public extern montage IMGS INFO;

/* Image enhancement. */

public extern contrast IMG SHARPEN, equalize IMG;		// BANG!
public extern gamma IMG VAL, level IMG VAL, modulate IMG VAL;	// BANG!
public extern negate IMG, normalize IMG;			// BANG!

/* Effects. */

public extern adaptive_threshold IMG DIM OFFS; // IM > 5.4 only
public extern add_noise IMG NOISE;
public extern blur IMG RADIUS SIGMA;
public extern despeckle IMG;
public extern edge IMG RADIUS;
public extern emboss IMG RADIUS SIGMA;
public extern enhance IMG;
public extern gaussian_blur IMG RADIUS SIGMA;
public extern median_filter IMG RADIUS;
public extern motion_blur IMG RADIUS SIGMA ANGLE;
public extern reduce_noise IMG RADIUS;
public extern shade IMG AZIMUTH ELEVATION;
public extern sharpen IMG RADIUS SIGMA;
public extern spread IMG RADIUS;
public extern threshold IMG VAL;				// BANG!
public extern unsharp_mask IMG RADIUS SIGMA AMOUNT THRESHOLD;

/* Special effects. */

public extern charcoal IMG RADIUS SIGMA;
public extern colorize IMG VAL PIXEL;
public extern convolve IMG KERNEL;
public extern implode IMG AMOUNT;
public extern morph IMGS FRAMES;
public extern oil_paint IMG RADIUS;
public extern solarize IMG THRESHOLD;				// BANG!
public extern stegano IMG IMG2 OFFSET;
public extern stereo IMG IMG2;
public extern swirl IMG ANGLE;
public extern wave IMG AMP WAVELEN;

/* Decorations. */

public extern border IMG DIM;
public extern frame IMG P DIM BEVEL;
public extern button IMG DIM RAISE;				// BANG!

/* Drawing. */

public extern annotate IMG P TEXT;				// BANG!
public extern draw IMG COMMAND;					// BANG!

/* Painting. */

public extern color_flood_fill IMG P FILL BORDER;		// BANG!
public extern matte_flood_fill IMG P OPACITY BORDER;		// BANG!

public extern opaque IMG PIXEL FILL;				// BANG!
public extern transparent IMG PIXEL OPACITY;			// BANG!
