
%{

/* qc.y: yacc source of Q parser and Q compiler main program */

/* Special case constructs (unary minus, 1-tuples) cause a number of parsing
   conflicts which are resolved correctly. */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qcdefs.h"

int      	nerrs, nwarns;
bool		hflag, nflag, vflag, Vflag, wflag;
volatile bool   int_sig;
char           *self = "qc", *list = "";

char		signon[] = QC_SIGNON;
char   		usage[] = QC_USAGE;
char		opts[MAXSTRLEN];
char		copying[] = COPYING;
char		helpmsg[] = HELPMSG;

DECLARE_YYTEXT

static int type, fno_min, fno_max;
static byte flags, sflags;
static unsigned long argv;

static void start_qualifiers(void), qualifiers(void);
static void add_qualifier(EXPR *x);
static void start_where_clauses(void), end_where_clauses(void),
  add_where_clause(EXPR *l, EXPR *r);

%}

%union {
  int ival;
  mpz_t zval;
  double fval;
  char *sval;
  EXPR *xval;
}

/* keywords and multi-character literals: */

%token AND AS CONST DEF DIV ELSE EXTERN IF IMPORT IN INCLUDE MOD NOT OR OTHERWISE
%token PRIVATE PUBLIC SPECIAL THEN TYPE UNDEF VAR WHERE
%token CAT DBAR LEQ GEQ NEQ

/* identifiers and constants: */

%token <ival> STR
%token <sval> UID LID QUID QLID STR1
%token <zval> INT
%token <fval> FLOAT

/* special tokens */

%token ERRTOK EOFTOK

%type <ival> id nid qid fid nfid qfid vid tid ntid qtid fvid nfvid qfvid vid_list
%type <ival> opt_type type_alias id_alias
%type <xval> condition
%type <xval> lexpression lrelation
%type <xval> expression relation addition multiplication unary script
%type <xval> application primary tuple_elems list_elems
%type <ival> op lrelop relop addop mulop unop scriptop quoteop

%start source

%%

/* error recovery is fairly simplistic (panic mode with ';' as stop symbol),
   I should really work out something more sophisticated in the future -AG */

source		: { srcstate(); }
		  program
		;

program		: /* empty */
		| program imports ';'
				{ import(); newdecl(); }
		| program includes ';'
				{ include(); newdecl(); }
		| program priority
				{ newdecl(); }
		| program declaration
				{ newdecl(); }
		| program definition
				{ newrule(); }
		| program rule
				{ newrule(); }
		| program EOFTOK
/* I disabled this one, most often it confuses the parser -AG
		| program error ';' '='
				{ yyerrok; srcstate(); newrule(); newdecl(); }
		  body
				{ newrule(); }
*/
		| program error ';'
				{ yyerrok; srcstate(); newrule(); newdecl(); }
		;

imports		: IMPORT import
		| imports ',' import
		;

import		: STR1
				{ add_import($1, NULL); }
		| STR1 AS STR1
				{ add_import($1, strdup($3)); }
		;

includes	: INCLUDE include
		| includes ',' include
		;

include		: STR1
				{ add_import($1, NULL); }
		| STR1 AS STR1
				{ add_import($1, strdup($3)); }
		;

priority	: '@' INT
				{ priority($2); mpz_clear($2); }
		| '@' '+' INT
				{ priority($3); mpz_clear($3); }
		| '@' '-' INT
				{ mpz_neg($3,$3);
				  priority($3); mpz_clear($3); }
		;

declaration	: prefix
		   		{ type = 0; }
		  headers ';'
		| TYPE qtid type_alias ';'
				{ if (!(symtb[$2].flags&DCL)) {
				    yyerror(qcmsg[DCL_ERROR]);
				    YYERROR;
				  } else
				    astype($2, $3, flags); }
		| scope TYPE qtid type_alias ';'
				{ if (!(symtb[$3].flags&DCL)) {
				    yyerror(qcmsg[DCL_ERROR]);
				    YYERROR;
				  } else
				    astype($3, $4, flags); }
		| TYPE ntid opt_type
		   		{ type = dcltype($2, $3, flags);
				  sflags = flags; fno_min = symtbsz; }
		  opt_header_sects ';'
				{ /* check for enumeration type */
				  int enumtype = 1, i;
				  fno_max = symtbsz-1;
				  for (i = fno_min; i <= fno_max; i++)
				    if (!(symtb[i].flags & CST) ||
					symtb[i].argc > 0) {
				      enumtype = 0; break;
				    }
				  if (enumtype) {
				    symtb[type].fno_min = fno_min;
				    symtb[type].fno_max = fno_max;
				  }
				}
		| scope TYPE ntid opt_type
		   		{ type = dcltype($3, $4, flags);
				  sflags = flags; fno_min = symtbsz; }
		  opt_header_sects ';'
				{ int enumtype = 1, i;
				  fno_max = symtbsz-1;
				  for (i = fno_min; i <= fno_max; i++)
				    if (!(symtb[i].flags & CST) ||
					symtb[i].argc > 0) {
				      enumtype = 0; break;
				    }
				  if (enumtype) {
				    symtb[type].fno_min = fno_min;
				    symtb[type].fno_max = fno_max;
				  }
				}
		| EXTERN TYPE ntid opt_type
		   		{ type = dcltype($3, $4, flags|EXT);
				  sflags = flags; }
		  ';'
		| scope EXTERN TYPE ntid opt_type
		   		{ type = dcltype($4, $5, flags|EXT);
				  sflags = flags; }
		  ';'
		;

type_alias	: /* empty */
				{ $$ = 0; }
		| AS ntid
				{ $$ = $2; }
		;

opt_type	: /* empty */
				{ $$ = 0; }
		| ':' tid
				{ $$ = checktype($2); }
		;

opt_header_sects: /* empty */
		| '=' header_sects
		;
		
header_sects	: header_sect
		| header_sects '|' header_sect
		;

header_sect	:		{ flags = sflags; }
		  opt_prefix headers
		;
		
prefix		: scope
		| modifiers
		| scope modifiers
		;
		
opt_prefix	: /* empty */
		| prefix
		;
		
scope		: PRIVATE
				{ flags = PRIV; }
		| PUBLIC
				{ flags = 0; }
		;
		
modifiers	: modifier
		| modifiers modifier
		;

modifier	: CONST
				{ flags |= CST; }
		| SPECIAL
				{ flags |= SPEC; }
		| EXTERN
				{ flags |= EXT; }
		| VAR
				{ flags |= VSYM; }
		;
		
headers		: header
		| headers ',' { argv = 0; } header
		;

header		: nid vid_list
				{ if ((flags & VSYM) && ($2 || type) ||
				      (flags & VSYM) && (flags & (EXT|SPEC)) ||
				      (flags & CST) && (flags & EXT) ||
				      !(flags & VSYM) &&
				      isupper(strsp[symtb[$1].pname]) ||
				      (symtb[$1].flags & DCL) &&
				      (symtb[$1].flags & VSYM) !=
				      (flags & VSYM)) {
				    yyerror(qcmsg[DCL_ERROR]);
				    YYERROR;
				  } else if (flags & VSYM)
				    dclfvar($1, flags);
				  else
				    dclfun($1, type, $2, argv, flags); }
		| qid vid_list id_alias
				{ if (!(symtb[$1].flags&DCL) ||
				      (flags & VSYM) && ($2 || type) ||
				      (flags & VSYM) && (flags & (EXT|SPEC)) ||
				      (flags & CST) && (flags & EXT) ||
				      !(flags & VSYM) &&
				      isupper(strsp[symtb[$1].pname]) ||
				      (symtb[$1].flags & DCL) &&
				      (symtb[$1].flags & VSYM) !=
				      (flags & VSYM) ||
				      type ||
				      $3 && symtb[$3].modno == modno &&
				      (symtb[$3].flags&DCL)) {
				    yyerror(qcmsg[DCL_ERROR]);
				    YYERROR;
				  } else if (flags & VSYM)
				    asfvar($1, $3, flags);
				  else
				    asfun($1, $3, $2, argv, flags); }
		;

vid_list	: /* empty */
				{ $$ = 0; }
		| vid_list UID
				{ if (flags & SPEC)
				    if ($1 < sizeof(unsigned long)*8)
				      argv |= 1<<$1;
				    else {
				      yyerror(qcmsg[DCL_ERROR]);
				      YYERROR;
				    }
				  $$ = $1+1; }
		| vid_list '~' UID
				{ if (!(flags & SPEC)) {
				    yyerror(qcmsg[DCL_ERROR]);
				    YYERROR;
				  }
				  $$ = $1+1; }
		;

id_alias	: /* empty */
			{ $$ = 0; }
		| AS nid
			{ $$ = $2; }
		;

definition	: DEF defs ';'
		| UNDEF undefs ';'
		;

defs		: def
		| defs ',' def
		;

def		: 		{ debug_info(); init_def(); }
		  lexpression '=' lexpression
				{ definition($2, $4); }
		;

undefs		: undef
		| undefs ',' undef
		;

undef		: id
				{ init_def();
				  if (!(symtb[$1].flags & VSYM))
				    yyerror(qcmsg[INVALID_DEF]);
				  else {
				    symtb[$1].flags |= DCL;
				    debug_info();
				    definition(funexpr($1), NULL);
				  }
				}
		;

rule		: lexpression
				{ debug_info(); left_hand_side($1); mark(); }
		  '=' body
		;

body		:		{ fix_debug_info(); start_qualifiers(); }
		  lexpression qualifiers ';'
				{ qualifiers(); right_hand_side($2); }
		| body '='
				{ fix_debug_info(); start_qualifiers(); }
		  lexpression qualifiers ';'
				{ qualifiers(); right_hand_side($4); }
		;

qualifiers	: /* empty */
		| qualifiers condition
				{ add_qualifier($2); }
		| qualifiers where
		;

condition	: IF expression
				{ $$ = $2; }
		| OTHERWISE
				{ $$ = NULL; }
		;

where		: WHERE
				{ start_where_clauses(); }
		  where_clauses
				{ end_where_clauses(); }
		;

where_clauses	: where_clause
		| where_clauses ',' where_clause
		;

where_clause	: lexpression '=' lexpression
				{ add_where_clause($1, $3); }
		;

lexpression	: lrelation
		| lexpression DBAR lrelation
				{ $$ = binexpr(SEQOP, $1, $3); }
		;

lrelation	: addition
		| addition lrelop addition
				{ $$ = binexpr($2, $1, $3); }
		;

lrelop		: '<'		{ $$ = LEOP; }
		| '>'		{ $$ = GROP; }
		| LEQ		{ $$ = LEQOP; }
		| GEQ		{ $$ = GEQOP; }
		| NEQ		{ $$ = NEQOP; }
		| IN		{ $$ = INOP; }
		;

expression	: relation
		| expression DBAR relation
				{ $$ = binexpr(SEQOP, $1, $3); }
		;

relation	: addition
		| addition relop addition
				{ $$ = binexpr($2, $1, $3); }
		;

relop		: '<'		{ $$ = LEOP; }
		| '>'		{ $$ = GROP; }
		| '='		{ $$ = EQOP; }
		| LEQ		{ $$ = LEQOP; }
		| GEQ		{ $$ = GEQOP; }
		| NEQ		{ $$ = NEQOP; }
		| IN		{ $$ = INOP; }
		;

addition	: multiplication
		| addition addop multiplication
				{ $$ = binexpr($2, $1, $3); }
		| addition '-' multiplication
				{ $$ = binexpr(MINOP, $1, $3); }
		;

addop		: '+'		{ $$ = ADDOP; }
		| CAT		{ $$ = CATOP; }
		| OR		{ $$ = OROP; }
		| OR ELSE	{ $$ = ORELSEOP; }
		;

multiplication	: unary
		| multiplication mulop unary
				{ $$ = binexpr($2, $1, $3); }
		;

mulop		: '*'		{ $$ = MULOP; }
		| '/'		{ $$ = FDIVOP; }
		| DIV		{ $$ = DIVOP; }
		| MOD		{ $$ = MODOP; }
		| AND		{ $$ = ANDOP; }
		| AND THEN	{ $$ = ANDTHENOP; }
		;

/* ! ambiguous rule */

unary		: script
		| '-' INT	{ mpz_neg($2, $2); $$ = intexpr($2); }
		| '-' FLOAT	{ $$ = floatexpr(-$2); }
		| '-' unary	{ $$ = unexpr(UMINOP, $2); }
		| unop unary    { $$ = unexpr($1, $2); }
		;

unop		: '#'		{ $$ = HASHOP; }
		| NOT		{ $$ = NOTOP; }
		;

script		: application
		| application scriptop script
				{ $$ = binexpr($2, $1, $3); }
		;

scriptop	: '^'		{ $$ = POWOP; }
		| '!'		{ $$ = IDXOP; }
		;

application	: primary
		| application primary
				{ $$ = appexpr($1, $2); }
		;

quoteop		: '\''		{ $$ = QUOTEOP; }
		| '`'		{ $$ = UNQUOTEOP; }
		| '~'		{ $$ = FORCEOP; }
		;

primary
		/* constants: */

		: INT		{ $$ = intexpr($1); }
		| FLOAT		{ $$ = floatexpr($1); }
		| STR		{ $$ = strexpr($1); }

		/* variable and function symbols: */

		| '(' op ')'	{ $$ = funexpr($2); }
		| fid		{ symtb[$1].flags |= DCL; $$ = funexpr($1); }
		| qfvid		{ symtb[$1].flags |= DCL; $$ = funexpr($1); }
		| vid		{ vartb[$1].type = 0;
				  $$ = varexpr($1); }
		| vid ':' tid	{ checktype($3);
		                  vartb[$1].type = $3;
				  $$ = varexpr($1); }

		/* quoted expressions: */

		| quoteop primary
				{ $$ = unexpr($1, $2); }

                /* sections: */

		| '(' expression DBAR ')'
				{ $$ = appexpr(funexpr(SEQOP), $2); }

		| '(' DBAR relation ')'
				{ $$ = appexpr(appexpr(funexpr(FLIPOP),
						       funexpr(SEQOP)),
					       $3); }

		| '(' addition relop ')'
				{ $$ = appexpr(funexpr($3), $2); }

		| '(' relop addition ')'
				{ $$ = appexpr(appexpr(funexpr(FLIPOP),
						       funexpr($2)),
					       $3); }

		| '(' addition addop ')'
				{ $$ = appexpr(funexpr($3), $2); }

		| '(' addition '-' ')'
				{ $$ = appexpr(funexpr(MINOP), $2); }

		| '(' addop multiplication ')'
				{ $$ = appexpr(appexpr(funexpr(FLIPOP),
						       funexpr($2)),
					       $3); }

		| '(' multiplication mulop ')'
				{ $$ = appexpr(funexpr($3), $2); }

		| '(' mulop unary ')'
				{ $$ = appexpr(appexpr(funexpr(FLIPOP),
						       funexpr($2)),
					       $3); }

		| '(' application scriptop ')'
				{ $$ = appexpr(funexpr($3), $2); }

		| '(' scriptop script ')'
				{ $$ = appexpr(appexpr(funexpr(FLIPOP),
						       funexpr($2)),
					       $3); }

		/* parenthesized expressions and tuples: */

		| '(' ')'
				{ $$ = funexpr(VOIDOP); }
		| '(' primary ')'
				{ $$ = pairexpr($2, funexpr(VOIDOP)); }
		| '(' expression ')'
				{ $$ = $2; }
		| '(' expression '|' expression ')'
				{ $$ = pairexpr($2, $4); }
		| '(' expression ',' tuple_elems ')'
				{ $$ = pairexpr($2, $4); }


		/* lists: */

		| '[' ']'
			{ $$ = funexpr(NILOP); }
		| '[' list_elems ']'
				{ $$ = $2; }

		;

tuple_elems	: expression
			{ $$ = pairexpr($1, funexpr(VOIDOP)); }
		| expression '|' expression
			{ $$ = pairexpr($1, $3); }
		| expression ',' tuple_elems
			{ $$ = pairexpr($1, $3); }
		;

list_elems	: expression
			{ $$ = consexpr($1, funexpr(NILOP)); }
		| expression '|' expression
			{ $$ = consexpr($1, $3); }
		| expression ',' list_elems
			{ $$ = consexpr($1, $3); }
		;

op		: DBAR		{ $$ = SEQOP; }
		| '<'		{ $$ = LEOP; }
		| '>'		{ $$ = GROP; }
		| '='		{ $$ = EQOP; }
		| LEQ		{ $$ = LEQOP; }
		| GEQ		{ $$ = GEQOP; }
		| NEQ		{ $$ = NEQOP; }
		| IN		{ $$ = INOP; }
		| CAT		{ $$ = CATOP; }
		| '+'		{ $$ = ADDOP; }
		| '-'		{ $$ = MINOP; }
		| OR		{ $$ = OROP; }
		| OR ELSE	{ $$ = ORELSEOP; }
		| '*'		{ $$ = MULOP; }
		| '/'		{ $$ = FDIVOP; }
		| DIV		{ $$ = DIVOP; }
		| MOD		{ $$ = MODOP; }
		| AND		{ $$ = ANDOP; }
		| AND THEN	{ $$ = ANDTHENOP; }
		| '#'		{ $$ = HASHOP; }
		| NOT		{ $$ = NOTOP; }
		| '\''		{ $$ = QUOTEOP; }
		| '`'		{ $$ = UNQUOTEOP; }
		| '~'		{ $$ = FORCEOP; }
		| '^'		{ $$ = POWOP; }
		| '!'		{ $$ = IDXOP; }
		;

id		: fid
		| fvid
		;

nid		: nfid
		| nfvid
		;

qid		: qfid
		| qfvid
		;

fid		: LID
				{ $$ = mkfun($1); }
		| QLID
				{ $$ = mkfun($1); }
		;

nfid		: LID
				{ $$ = mkfun($1); }
		;

qfid		: QLID
				{ $$ = mkfun($1); }
		;

vid		: UID
				{ $$ = mkvar($1); }
		;

fvid		: UID
				{ $$ = mkfvar($1); }
		| QUID
				{ $$ = mkfvar($1); }
		;

nfvid		: UID
				{ $$ = mkfvar($1); }
		;

qfvid		: QUID
				{ $$ = mkfvar($1); }
		;

tid		: UID
				{ $$ = mktype($1); }
		| LID
				{ $$ = mktype($1); }
		| QUID
				{ $$ = mktype($1); }
		| QLID
				{ $$ = mktype($1); }
		;

ntid		: UID
				{ $$ = mktype($1); }
		| LID
				{ $$ = mktype($1); }
		;

qtid		: QUID
				{ $$ = mktype($1); }
		| QLID
				{ $$ = mktype($1); }
		;
		
%%

extern int      yyleng, yylineno;
extern char     *source;

yyerror(s)
	char           *s;
{
	fprintf(stderr, "Error %s, line %d: %s", source, yylineno, s);
	if (
#ifdef YYBISON
	    strcmp(s, "parse error") == 0
#else
	    strcmp(s, "syntax error") == 0
#endif
	   )
		if (*yytext)
			fprintf(stderr, " at or near symbol `%s'", yytext);
		else
			fprintf(stderr, " at end of file");
	fprintf(stderr, "\n");
	nerrs++;
}

yywarn(s)
	char           *s;
{
	if (wflag) {
		fprintf(stderr, "Warning %s, line %d: %s\n", source,
			yylineno, s);
		nwarns++;
	}
}

fatal(s)
	char           *s;
{
	if (source && *source)
		fprintf(stderr, "%s: %s: %s -- compilation aborted\n",
			self, source, s);
	else
		fprintf(stderr, "%s: %s -- compilation aborted\n",
			self, s);
	if (codefp) {
		fclose(codefp);
		remove(code);
	}
	exit(1);
}

#define no(n) n, n==1?"":"s"

static
statistics()
{
	int fno, k, n, b, bmax, btotal, n_data;
	for (n = bmax = btotal = k = 0; k < hashtbsz; k++)
		if (hashtb[k] != NONE) {
			n++;
			for (b = -1, fno = hashtb[k]; fno != NONE;
			     b++, fno = symtb[fno].next)
				;
			btotal += b;
			if (b+1>bmax)
				bmax = b+1;
		}
	n_data = strspsz+limbspsz*sizeof(mp_limb_t);
	printf("%d ops in %d module%s, ", codespsz, no(modtbsz));
	printf("%d byte%s data, ", no(n_data));
	printf("%d symbol%s,\n", no(symtbsz));
	printf("%d hash key%s out of %d, %d collision%s, max bucket size = %d\n",
	       no(n), hashtbsz, no(btotal), bmax);
	printf("%d state%s, %d transition%s, %d offset%s\n",
	       no(statetbsz), no(transtbsz), no(roffstbsz));
}

static void *
gmp_allocate (size)
     size_t size;
{
  void *ret;

  ret = malloc (size);
  if (ret == 0) fatal(qcmsg[MEM_OVF]);
  return ret;
}

static void *
gmp_reallocate (oldptr, old_size, new_size)
     void *oldptr;
     size_t old_size;
     size_t new_size;
{
  void *ret;

  ret = realloc (oldptr, new_size);
  if (ret == 0) fatal(qcmsg[MEM_OVF]);
  return ret;
}

static void
gmp_free (blk_ptr, blk_size)
     void *blk_ptr;
     size_t blk_size;
{
  free (blk_ptr);
}

RETSIGTYPE
break_handler()
/* handle SIGINT and SIGTERM */
{
  /* Since many system functions are unsave to call in a signal
     handler, we simply set a flag here; the corresponding actions
     in response to SIGINT (remove code file, close list file,
     terminate program) will be carried out later in a save
     context. */
  int_sig = 1;
  SIGHANDLER_RETURN(0);
}

checkint()
/* check for pending int_sig */
{
  if (int_sig) fatal("interrupt");
}

newrule()
/* reinitialize for the next rule */
{
  clear(); checkint();
}

newdecl()
/* reinitialize for the next declaration */
{
  flags = PRIV; argv = 0; checkint();
}

priority(z)
     mpz_t z;
/* set a new priority level */
{
  if (my_mpz_fits_slong_p(z))
    prio = mpz_get_si(z);
  else
    yyerror(qcmsg[INVALID_PRIO]);
}

/* qualifier table */

int qual_size, qual_alloc, clause_size, clause_alloc;
static QUAL *qual;
static CLAUSE *clause;

static void start_qualifiers(void)
{
  qual_size = clause_size = 0;
}

static void qualifiers(void)
{
  int i;
  for (i = qual_size-1; i >= 0; i--)
    if (qual[i].x)
      qualifier(qual[i].x);
    else {
      int j;
      for (j = qual[i].start; j < qual[i].end; j++)
	where_clause(clause[j].l, clause[j].r);
    }
}

static void add_qualifier(EXPR *x)
{
  if (!x) return;
  if (qual_size >= qual_alloc)
    if ((qual = arealloc(qual, qual_alloc, 10, sizeof(QUAL))))
      qual_alloc += 10;
    else
      fatal("memory overflow");
  qual[qual_size++].x = x;
}

static void start_where_clauses(void)
{
  if (qual_size >= qual_alloc)
    if ((qual = arealloc(qual, qual_alloc, 10, sizeof(QUAL))))
      qual_alloc += 10;
    else
      fatal("memory overflow");
  qual[qual_size].x = NULL;
  qual[qual_size].start = clause_size;
}

static void end_where_clauses(void)
{
  qual[qual_size++].end = clause_size;
}

static void add_where_clause(EXPR *l, EXPR *r)
{
  if (clause_size >= clause_alloc)
    if ((clause = arealloc(clause, clause_alloc, 10, sizeof(CLAUSE))))
      clause_alloc += 10;
    else
      fatal("memory overflow");
  clause[clause_size].l = l;
  clause[clause_size].r = r;
  clause_size++;
}

static struct option longopts[] = QC_OPTS;
static struct option all_longopts[] = Q_OPTS;

static int
getintarg(char *s, int *i)
{
  char *t = s;
  while (isspace(*t)) t++;
  s = t;
  while (isdigit(*t)) t++;
  if (t == s) return 0;
  while (isspace(*t)) t++;
  if (*t) return 0;
  *i = atoi(s);
  return 1;
}

static void
parse_opts(argc, argv, pass)
     int argc;
     char **argv;
     int pass; /* 0 denotes source, 1 command line pass */
{
  int c, longind;
  optind = 0;
  while ((c = getopt_long(argc, argv,
			  pass?QC_OPTS1:Q_OPTS1,
			  pass?longopts:all_longopts,
			  &longind)) != EOF)
    switch (c) {
    case QC_NO_PRELUDE:
      prelude = NULL;
      break;
    case QC_PRELUDE:
      prelude = optarg?optarg:prelude;
      break;
    case 'h':
      hflag = 1;
      break;
    case 'l':
      list = optarg?optarg:list;
      break;
    case 'n':
      nflag = 1;
      break;
    case 'o':
      code = optarg?optarg:code;
      break;
    case 'p':
      if (optarg) {
	change_qpath(optarg);
	if (!qpath) fatal("memory overflow");
      }
      break;
    case 't': {
      int sz;
      if (optarg && getintarg(optarg, &sz) && sz > 0)
	hashtbsz = sz;
      else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid size `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case 'v':
      vflag = 1;
      break;
#if 0
    case 'w':
      wflag = 1;
      break;
#endif
    case 'V':
      Vflag = 1;
      break;
    /* interpreter options (ignored): */
    case Q_GNUCLIENT:
    case Q_DEBUG_OPTIONS:
    case Q_BREAK:
    case Q_PROMPT:
    case Q_DEC:
    case Q_HEX:
    case Q_OCT:
    case Q_STD:
    case Q_SCI:
    case Q_FIX:
    case Q_HISTFILE:
    case Q_HISTSIZE:
    case Q_INITRC:
    case Q_NO_INITRC:
    case Q_EXITRC:
    case Q_NO_EXITRC:
    case Q_NO_EDITING:
    case Q_GC_MSGS:
    case Q_GC:
    case Q_STACKSIZE:
    case Q_MEMSIZE:
    case 'd':
    case 'e':
    case 'i':
    case 'q':
    case 'c':
    case 's':
      break;
    default:
      exit(1);
    }
}

static int sargc;
static char **sargv;

static void
get_source_opts(fp)
     FILE *fp;
{
  char s[MAXSTRLEN];
  int i;
  bool first = 1;

  sargc = 1;
  if (!(sargv = aalloc(1, sizeof(char*)))) fatal(qcmsg[MEM_OVF]);
  *sargv = strdup(self);
  while (!feof(fp) && !ferror(fp) &&
	 fgets(s, MAXSTRLEN, fp)) {
    int l = strlen(s);
    if (l > 0 && s[l-1] == '\n') s[l-1] = '\0', l--;
    if (l == 0) {
      first = 0;
      continue;
    } else if (strncmp(s, "#!", 2) == 0)
      if (!first && isspace(s[2])) {
	char *p = s+3;
	while (isspace(*p)) p++;
	sargv = arealloc(sargv, sargc, 1, sizeof(char*));
	sargv[sargc++] = strdup(p);
      } else {
	first = 0;
	continue;
      }
    else
      break;
  }
  sargv = arealloc(sargv, sargc, 1, sizeof(char*));
  sargv[sargc] = NULL;
}

main(argc, argv)
	int             argc;
	char          **argv;
{
	int c, longind;
	char *s;

#ifdef _WIN32
	InstallSignalHandler();
#endif

	/* get program name: */
	self = argv[0];

	/* get environment settings: */
	if ((s = getenv("QPATH")) != NULL)
	  init_qpath(s);
	else
	  init_qpath(QPATH);
	if (!qpath) fatal("memory overflow");

	/* scan command line to obtain the first source file name: */
	opterr = 0;
	while ((c = getopt_long(argc, argv, Q_OPTS1,
				longopts, &longind)) != EOF)
	  ;
	opterr = 1;

	/* get options from the main script: */
	if (argc-optind >= 1) {
	  char fname[MAXSTRLEN];
	  FILE *fp;
	  if (chkfile(searchlib(fname, argv[optind])) &&
	      (fp = fopen(fname, "r")) != NULL) {
	    get_source_opts(fp);
	    fclose(fp);
	    parse_opts(sargc, sargv, 0);
	  }
	}

	/* get command line options: */
	parse_opts(argc, argv, 1);
	argc -= optind, argv += optind;

	if (Vflag) {
		printf(signon, version, sysinfo, year);
		printf(copying);
		printf(helpmsg, self);
		exit(0);
	}
	if (hflag) {
		printf(usage, self);
		sprintf(opts, QC_OPTMSG, QPATH, HASHTBSZ);
		fputs(opts, stdout);
		exit(0);
	}

	/* install break and term handlers: */
	sigint(break_handler);
	sigterm(break_handler);
	sighup(break_handler);

	/* install gmp memory handlers */
	mp_set_memory_functions(gmp_allocate, gmp_reallocate, gmp_free);

	/* set code file id: */
	sprintf(outid, OUTID, version, sysinfo);

	/* compile: */
	if (*list) {
	  FILE *fp;
	  if (!(fp = fopen(list, "w"))) {
	    fprintf(stderr, "%s: error creating %s\n",
		    self, list);
	    exit(1);
	  } else {
	    fclose(fp);
	    freopen(list, "w", stderr);
	  }
	}
	if (!(codefp = fopen(code, "wb"))) {
	  fprintf(stderr, "%s: error creating %s\n",
		  self, code);
	  exit(1);
	}
	mainno = -1;
	write_header();
	inittables();
	if (!initlex(argc, argv))
	  /* empty source; skip parse */
	  goto generate;
	if (yyparse() == 0 && nerrs == 0) {
	generate:
	  if (!nflag) {
	    /* generate code: */
	    write_strsp();
	    write_limbsp();
	    write_hashtb();
	    write_symtb();
	    write_TA();
	    write_matchtb();
	    write_inittb();
	    write_modtb();
	    fix_header();
	  }
	  checkint();
	  if (nflag) {
	    fclose(codefp);
	    remove(code);
	  } else
	    fclose(codefp);
	  if (list && !nwarns) {
	    fclose(stderr);
	    remove(list);
	  }
	  if (vflag)
	    if (nflag)
	      printf("(no code generated)\n");
	    else
	      statistics();
	  exit(0);
	} else {
	  checkint();
	  fclose(codefp);
	  remove(code);
	  if (vflag)
	    putchar('\n');
	  exit(1);
	}
}
