
/* qcc.c: the Q external modules compiler */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* Note that this is ANSI code, so if you have a plain K&R compiler it won't
   work. But then you won't be able to build libq.c either. */

#include "qbase.h"

/* We now support libtool, hence the "shared library extension" is always .la
   except for the native Windows port where it is .dll, in order to allow
   building native, i.e., non-libtool modules with MINGW or MSVC. */

#ifndef SHLEXT
#if defined _WIN32
#define SHLEXT ".dll"
#else
#define SHLEXT ".la"
#endif
#endif

#ifndef LIBTOOL
#define LIBTOOL "libtool"
#endif

#ifndef CC
#define CC "gcc"
#endif

#ifndef SH_CC
#define SH_CC LIBTOOL
#endif

#ifndef SH_CFLAGS
#define SH_CFLAGS "--quiet --mode=compile %s"
#endif

#ifndef SH_LD
#define SH_LD LIBTOOL
#endif

#ifndef SH_LDFLAGS
#define SH_LDFLAGS "--quiet --mode=link %s"
#endif

#define OPTS1 "-hkno:vV"

#define OPTS { \
{ "help", 0, NULL, 'h' }, \
{ "keep", 0, NULL, 'k' }, \
{ "link", 0, NULL, 11 }, \
{ "dry-run", 0, NULL, 'n' }, \
{ "output", 1, NULL, 'o' }, \
{ "verbose", 0, NULL, 'v' }, \
{ "version", 0, NULL, 'V' }, \
{ "mingw", 0, NULL, 12 }, \
{ "msc", 0, NULL, 13 }, \
{ "ext", 0, NULL, 14 }, \
{ NULL, 0, NULL, 0 } \
}

#define MAXARGC 1000

char *self = "qcc", sh_cflags[MAXSTRLEN], sh_ldflags[MAXSTRLEN];
int no_compile, verbose, driver, linker, keep;
char *output = NULL, *cext = ".c", *oext = ".lo", *shlext = SHLEXT;
int ldoptc, loptc, coptc, srcc, objc, eargc;
char *ldoptv[MAXARGC+1], *loptv[MAXARGC+1], *coptv[MAXARGC+1],
  *srcv[MAXARGC+1], *objv[MAXARGC+1], *eargv[MAXARGC+1];

void fatal(char *s)
{
  fprintf(stderr, "%s: %s\n", self, s);
  exit(1);
}
  
void addopt(char *s)
{
  if (linker) {
    if (loptc >= MAXARGC)
      fatal("command line too long");
    loptv[loptc++] = s;
  } else {
    if (coptc >= MAXARGC)
      fatal("command line too long");
    coptv[coptc++] = s;
  }
}
  
void addldopt(char *s)
{
  if (ldoptc >= MAXARGC)
    fatal("command line too long");
  ldoptv[ldoptc++] = s;
}

char *newext(char *s, char *ext, char *next)
{
  int l = strlen(s), e = strlen(ext);
  char *t = malloc(l-e+strlen(next)+1);
  if (!t) fatal("memory overflow");
  strncpy(t, s, l-e);
  strcpy(t+l-e, next);
  return t;
}

char *newexts(char *s, char *ext, char *next)
{
  /* strip path from filename */
  char *t, *p = NULL;
  for (t = s; *t; t++)
    if (strchr(DIRSTR, *t))
      p = t;
  if (p)
    p++;
  else
    p = s;
  return newext(p, ext, next);
}

char *getext(char *s)
{
  char *t = strrchr(s, '.'), *p;
  if (t) {
    for (p = t; *p; p++)
      if (strchr(DIRSTR, *p))
	return "";
    return t;
  } else
    return "";
}

void default_output(char *s, char *ext)
{
  if (!output) {
    int l = strlen(s), e = strlen(ext);
    output = malloc(l-e+strlen(shlext)+1);
    if (!output) fatal("memory overflow");
    strncpy(output, s, l-e);
    strcpy(output+l-e, shlext);
  }
}

void addsrc(char *s)
{
  if (srcc >= MAXARGC)
    fatal("command line too long");
  srcv[srcc++] = s;
  if (!output) output = newexts(s, cext, shlext);
}

void addobj(char *s)
{
  if (objc >= MAXARGC)
    fatal("command line too long");
  objv[objc++] = s;
  if (!output) output = newexts(s, oext, shlext);
}

int isfile(char *s, char *ext)
{
  int l = strlen(s);
  if (l >= strlen(ext))
    return !strcmp(s+l-strlen(ext), ext);
  else
    return 0;
}

void addfilearg(char *s)
{
  if (isfile(s, cext)) {
    addsrc(s);
    addobj(newexts(s, cext, oext));
  } else if (isfile(s, oext))
    addobj(s);
  else {
    char msg[1000];
    sprintf(msg, "unrecognized file type: `%s'", s);
    fatal(msg);
  }
}

void addarg(char *s)
{
  if (isfile(s, cext)) {
    addsrc(s);
    addobj(newexts(s, cext, oext));
  } else if (isfile(s, oext))
    addobj(s);
  else
    addopt(s);
}

/* spawn child processes */

#ifndef _WIN32
static volatile int pid = 0;
static RETSIGTYPE
spawn_term_handler(int sig)
/* pass termination signals to child process */
{
  if (pid) kill(pid, sig);
  SIGHANDLER_RETURN(0);
}
#endif

int spawn(char *prog, char *argv[])
{
  int status;
#ifdef _WIN32
  status = spawnvp(P_WAIT, prog, argv);
  if (status < 0)
    fatal("exec failed -- check installation");
  else
    return status;
#else
  sigterm(SIG_IGN);
  switch ((pid = fork())) {
  case 0:
    execvp(prog, argv);
  case -1:
    fatal("exec failed -- check installation");
  }
  sigint(SIG_IGN);
  sigterm(spawn_term_handler);
  sighup(spawn_term_handler);
  waitpid(pid, &status, 0);
  sigterm(SIG_DFL);
  sighup(SIG_DFL);
  return status;
#endif
}

void doexec(int argc, char **argv)
{
  int i;
  if (verbose) {
    printf("%s", argv[0]);
    for (i = 1; i < argc; i++)
      printf(" %s", argv[i]);
    printf("\n");
    fflush(stdout);
  }
  argv[argc] = NULL;
  if (!no_compile) {
    char msg[100];
    int status = spawn(argv[0], argv);
#ifdef _WIN32
    if (status != 0) {
      sprintf(msg, "%s error (status %d)", linker?"linkage":"compilation",
	      status);
      fatal(msg);
    }
#else
    if (!WIFEXITED(status)) {
      sprintf(msg, "child process exited abnormally (status %d)", status);
      fatal(msg);
    } else if ((status = WEXITSTATUS(status)) != 0) {
      sprintf(msg, "%s error (status %d)", linker?"linkage":"compilation",
	      status);
      fatal(msg);
    }
#endif
  }
}

int main(int argc, char **argv)
{
  static struct option longopts[] = OPTS;
  int i, c, longind;
  char pwd[MAXSTRLEN];
  self = argv[0];
  if (!getcwd(pwd, MAXSTRLEN)) fatal("cannot get the current directory");
  sprintf(sh_cflags, SH_CFLAGS, CC);
  sprintf(sh_ldflags, SH_LDFLAGS, CC);
#if defined _WIN32
  driver = 1; oext = ".o"; shlext = ".dll";
#endif
  /* parse compiler and linker flags */
  if (driver == 0) {
    const char *delim = " \t";
    int lc = strlen(sh_cflags), lld = strlen(sh_ldflags);
    char *flags = malloc(((lld>lc)?lld:lc)+1), *tok;
    if (!flags) fatal("memory overflow");
    linker = 0;
    tok = strtok(strcpy(flags, sh_cflags), delim);
    while (tok) {
      addopt(strdup(tok)); tok = strtok(NULL, delim);
    }
    tok = strtok(strcpy(flags, sh_ldflags), delim);
    while (tok) {
      addldopt(strdup(tok)); tok = strtok(NULL, delim);
    }
  }
  /* parse the command line */
  while ((c = getopt_long(argc, argv, OPTS1, longopts,
			  &longind)) != EOF) {
    switch (c) {
    case 1:
      addfilearg(optarg);
      break;
    case 'k':
      keep = 1;
      break;
    case 'n':
      no_compile = verbose = 1;
      break;
    case 'o':
      if (optarg)
	output = optarg;
      else
	fatal("--output option needs filename argument");
      break;
    case 'v':
      verbose = 1;
      break;
    case 'V':
      printf("Q module compiler version %s (%s)\n\
Copyright (c) 2000-%s by Albert Graef\n", VERSION, SYSINFO, YEAR);
      printf(COPYING);
      printf(HELPMSG, self);
      exit(0);
    case 'h':
      printf(
"Usage: %s [options] file ... [-- cc-options ... [--link ld-options ...]]\n",
	     self);
      printf("Options:\n\
--dry-run, -n		Do not actually execute compile commands (also\n\
			switches on verbose mode).\n\
--ext			Print the default output file extension and exit.\n\
--help, -h		Print this list.\n\
--keep, -k		Keep intermediate files produced by compilation.\n\
--mingw			Select the mingw compiler driver (default\n\
			for native Windows compilation).\n\
--msc			Select the MS Visual C compiler driver.\n\
--output=FILE, -o FILE	Specify output file name (default: first source or\n\
			object file name with new extension `%s').\n\
--verbose, -v		Echo compile commands as they are processed.\n\
--version, -V		Display version information and copyright notices.\n\
cc-options		Options to pass to the C compiler.\n\
ld-options		Options to pass to the linker.\n",
	     shlext);
      exit(0);
    case 11:
      fatal("misplaced --link option");
      break;
    case 12:
      driver = 1; oext = ".o"; shlext = ".dll";
      break;
    case 13:
      driver = 2; oext = ".obj"; shlext = ".dll";
      break;
    case 14:
      printf("%s\n", shlext);
      exit(0);
    default:
      exit(1);
    }
  }
  while (optind < argc)
    if (!strcmp(argv[optind], "--link")) {
      linker = 1;
      optind++;
    } else
      addarg(argv[optind++]);
  if (objc <= 0)
    fatal("no source or object files specified");
  /* compile the sources: */
  linker = 0;
  if ((eargc = coptc + 3) > MAXARGC)
    fatal("command line too long");
  eargv[0] = (driver==1)?"gcc":(driver==2)?"cl":SH_CC;
  for (i = 0; i < coptc; i++) eargv[i+1] = coptv[i];
  eargv[coptc+1] = "-c";
  eargv[coptc+3] = NULL;
  for (i = 0; i < srcc; i++) {
    eargv[coptc+2] = srcv[i];
    doexec(eargc, eargv);
  }
  /* link: */
  linker = 1;
  if (driver == 1) {
    /* mingw: use dlltool/dllwrap */
    if ((eargc = objc + loptc + 6) > MAXARGC)
      fatal("command line too long");
    eargv[0] = "dlltool";
    eargv[1] = "--output-def";
    eargv[2] = newext(output, getext(output), ".def");
    for (i = 0; i < objc; i++) eargv[i+3] = objv[i];
    doexec(objc+3, eargv);
    eargv[0] = "dllwrap";
    eargv[1] = "--dllname";
    eargv[2] = output;
    eargv[3] = "--def";
    eargv[4] = newext(output, getext(output), ".def");
    for (i = 0; i < objc; i++) eargv[i+5] = objv[i];
    for (i = 0; i < loptc; i++) eargv[i+objc+5] = loptv[i];
    eargv[loptc+objc+5] = "-lq";
    doexec(eargc, eargv);
  } else if (driver == 2) {
    /* msc: use -LD option and add libq.lib */
    if ((eargc = objc + loptc + 5) > MAXARGC)
      fatal("command line too long");
    eargv[0] = "cl";
    eargv[1] = "-LD";
    eargv[2] = "-o";
    eargv[3] = output;
    for (i = 0; i < objc; i++) eargv[i+4] = objv[i];
    for (i = 0; i < loptc; i++) eargv[i+objc+4] = loptv[i];
    eargv[objc+loptc+4] = "libq.lib";
    doexec(eargc, eargv);
  } else {
    /* standard libtool driver */
    if ((eargc = objc + ldoptc + loptc + 9) > MAXARGC)
      fatal("command line too long");
    eargv[0] = SH_LD;
    for (i = 0; i < ldoptc; i++) eargv[i+1] = ldoptv[i];
    eargv[ldoptc+1] = "-o";
    eargv[ldoptc+2] = output;
    eargv[ldoptc+3] = "-rpath";
    eargv[ldoptc+4] = pwd;
    eargv[ldoptc+5] = "-no-undefined";
    eargv[ldoptc+6] = "-module";
    eargv[ldoptc+7] = "-avoid-version";
    for (i = 0; i < objc; i++) eargv[i+ldoptc+8] = objv[i];
    for (i = 0; i < loptc; i++) eargv[i+ldoptc+objc+8] = loptv[i];
    eargv[loptc+ldoptc+objc+8] = "-lq";
    doexec(eargc, eargv);
    /* make all created files visible in the current directory */
    eargc = 5;
    eargv[0] = LIBTOOL;
    eargv[1] = "--quiet";
    eargv[2] = "cp";
    eargv[3] = output;
    eargv[4] = pwd;
    doexec(eargc, eargv);
  }
  if (!no_compile && !keep) {
    /* clean up: */
    for (i = 0; i < srcc; i++) unlink(newexts(srcv[i], cext, oext));
    if (driver == 1) {
      char *ext = getext(output);
      unlink(newext(output, ext, ".def"));
    } else if (driver == 2) {
      char *ext = getext(output);
      unlink(newext(output, ext, ".exp"));
      unlink(newext(output, ext, ".lib"));
    } else {
      for (i = 0; i < srcc; i++) unlink(newexts(srcv[i], cext, ".o"));
      system("rm .libs/* && rmdir .libs");
    }
  }
  exit(0);
}

