
%{ /* -*-C-*- */

/* qclex.l: lexical analyzer for the Q language */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qcdefs.h"
#include "qc.h"

#ifdef FLEX_SCANNER
#undef yywrap
int yylineno;           /* the current line */
#endif

char *source = NULL;  	/* the source file name */

static char s1[MAXSTRLEN], s2[MAXSTRLEN];
int incmode;

static int abufsz = 0, bufp = 0;
static char *buf = NULL;

static comment(), string();
static void bigint();
%}

%s src isrc

LL			[a-z_]
UL			[A-Z]
L			({LL}|{UL})
O			[0-7]
D			[0-9]
X			[0-9A-Fa-f]
SF			([Ee][+-]?{D}+)
WS			[ \t\f\r]
QUAL			({L}({L}|{D})*)

%%

and			return(AND);
as			return(AS);
const			return(CONST);
def			return(DEF);
div			return(DIV);
else			return(ELSE);
extern			return(EXTERN);
if			return(IF);
import			{ BEGIN isrc; return(IMPORT); }
in			return(IN);
include			{ BEGIN isrc; return(INCLUDE); }
mod			return(MOD);
not			return(NOT);
or			return(OR);
otherwise		return(OTHERWISE);
private			return(PRIVATE);
public			return(PUBLIC);
special			return(SPECIAL);
then			return(THEN);
type			return(TYPE);
undef			return(UNDEF);
var			return(VAR);
where			return(WHERE);
<isrc>\"		{
			char *buf1;
			string();
			buf1 = (char*)malloc((strlen(buf)+1)*sizeof(char));
			if (!buf1)
			  fatal(qcmsg[MEM_OVF]);
			yylval.sval = scanstr(buf1, buf);
			return(STR1);
}
<isrc>{QUAL}		{ yylval.sval =	strdup(yytext); return(STR1); }
<isrc>;			{ BEGIN src; return ';'; }
"++"			return(CAT);
"<="			return(LEQ);
">="			return(GEQ);
"<>"			return(NEQ);
"||"			return(DBAR);
"_"			|
{UL}({L}|{D})*		{ yylval.sval =	strcpy(s1, yytext); return(UID); }
{LL}({L}|{D})*		{ yylval.sval =	strcpy(s1, yytext); return(LID); }
{QUAL}?::_		|
{QUAL}?::{UL}({L}|{D})*	{ yylval.sval =	strcpy(s1, yytext); return(QUID); }
{QUAL}?::{LL}({L}|{D})*	{ yylval.sval =	strcpy(s1, yytext); return(QLID); }
0{O}+			{ bigint(yylval.zval); return(INT); }
0{D}+			{ return(ERRTOK); }
{D}+			|
0[xX]{X}+		{ bigint(yylval.zval); return(INT); }
{D}+{SF}		|
{D}+\.{D}*{SF}?		|
{D}*\.{D}+{SF}?		{ sscanf(yytext, "%lf", &yylval.fval); return(FLOAT); }
\"			{
			char *buf1;
			string();
			buf1 = (char*)malloc((strlen(buf)+1)*sizeof(char));
			if (!buf1)
			  fatal(qcmsg[MEM_OVF]);
			yylval.ival = putstr(scanstr(buf1, buf));
			free(buf1);
			return(STR);
}
{WS}			|
\%.*			|
"//".*			|
^#!.*			;
\n			{
#ifdef FLEX_SCANNER
			yylineno++;
#endif
}
"/*"			comment();
.			return(yytext[0]);

%%

static int      in_comment = 0;
static int      argc;
static char   **argv, **argv0 = NULL, **asv, **asv0 = NULL, *mainfile;

static comment()
{
  register int    c;

  in_comment = 1;
  while ((c = input()) && c != EOF) {
    if (c == '*') {
      if ((c = input()) == '/')
	break;
      else
	unput(c);
    }
#ifdef FLEX_SCANNER
    else if (c == '\n')
      yylineno++;
#endif
  }
  if (!c || c == EOF)
    fatal(qcmsg[OPEN_COMMENT]);
  in_comment = 0;
}

static initbuf()
{
  bufp = 0;
}

static addbuf(char c)
{
  if (bufp >= abufsz)
    if (!(buf = (char*)arealloc(buf, abufsz, 100, sizeof(char))))
      fatal(qcmsg[MEM_OVF]);
    else
      abufsz += 100;
  buf[bufp++] = c;
}

static string()
{
  register int    c;

  in_comment = 1;
  initbuf();
  while ((c = input()) && c != EOF) {
    if (c == '"')
      break;
    else if (c == '\\') {
      if ((c = input()) == '\n') {
#ifdef FLEX_SCANNER
	yylineno++;
#endif
      } else {
	addbuf('\\');
	addbuf(c);
      }
    } else if (c == '\n') {
#ifdef FLEX_SCANNER
      yylineno++;
#endif
      break;
    } else
      addbuf(c);
  }
  addbuf('\0');
  if (c != '"') {
    yyerror(qcmsg[OPEN_STRING]);
    for (; c && c != EOF && c != '"'; c = input()) ;
    if (c != '"') unput('\0');
  }
  in_comment = 0;
}

static char *
skipz(char *s)
{
  while (*s == '0') s++;
  return s;
}

static void*
my_mpz_realloc(m, new_size)
     mpz_ptr m;
     mp_size_t new_size;
{
  mpz_t m1;
  memcpy(m1, m, sizeof(mpz_t));
  if (_mpz_realloc(m, new_size))
    return m->_mp_d;
  else {
    if (m1->_mp_d) mpz_clear(m1);
    return NULL;
  }
}

static void bigint(z)
     mpz_t z;
{
  int sz;
  if (strncmp(yytext, "0x", 2) == 0 ||
      strncmp(yytext, "0X", 2) == 0)
    sz = 4*strlen(skipz(yytext+2));
  else if (*yytext == '0')
    sz = 3*strlen(skipz(yytext+1));
  else
    sz = log(10)/log(2)*strlen(skipz(yytext))+1;
  sz = sz/(CHAR_BIT*sizeof(mp_limb_t)) + 2;
  mpz_init(z); 
  if (z->_mp_d && my_mpz_realloc(z, sz)) {
    int sz1;
    mpz_set_str(z, yytext, 0);
    sz1 = mpz_size(z);
    if (sz1 < sz && !my_mpz_realloc(z, sz1))
      fatal(qcmsg[MEM_OVF]);
  } else
    fatal(qcmsg[MEM_OVF]);
}

int _modtb[MAXFILES], _fnametb[MAXFILES];
int _modtbsz;

static struct FileStack {
#ifdef FLEX_SCANNER
  YY_BUFFER_STATE state;
#endif
  FILE *yyin;
  int prio, modno, yylineno, incl;
  char *source;
  int argc;
  char **argv, **argv0;
  char **asv, **asv0;
} fst[MAXFILES], *fsp;

static
getmodno(s)
	char	       *s;
/* look up module name s in module table, return corresponding module
   number (NONE if not in table) */
{
	int		i;
	for (i = 0; i < _modtbsz; i++)
		if (strcmp(s, strsp+_modtb[i]) == 0)
			return i;
	return NONE;
}

static
getmodno_by_fname(s)
	char	       *s;
/* look up module by file name */
{
	int		i;
	for (i = 0; i < _modtbsz; i++)
		if (strcmp(s, strsp+_fnametb[i]) == 0)
			return i;
	return NONE;
}

static
stacked(modno)
	int 		modno;
/* check whether module is on include stack */
{
	struct FileStack *fsp1;
	for (fsp1 = fst; fsp1 < fsp; fsp1++)
		if (fsp1->modno == modno)
			return 1;
	return 0;
}

static
addmod(modname, fname, s)
     char		*modname, *fname, *s;
/* add module to module table */
{
  if (_modtbsz >= MAXFILES)
    fatal(qcmsg[TOO_MANY_FILES]);
  else {
    _modtb[_modtbsz] = putstr(modname);
    _fnametb[_modtbsz] = putstr(fname);
    modno = _modtbsz++;
    if (s == mainfile) mainno = modno;
    if (mainno > 0) putimp(0, 1);
  }
}

#define HAVE_FILE (-2)

static
opensrc(s, as, save)
     char           *s, *as;
     int save;
{
  char 	       	aname[MAXSTRLEN], fname[MAXSTRLEN], fname2[MAXSTRLEN],
    modname[MAXSTRLEN];
  int		mno;

  if (!s || !*s) return NONE;

  /* parse file name and determine module id: */

  if (!as) {
    basename(modname, s, '.');
    as = modname;
  }
  absname(aname, searchlib(fname, s));
  if (!chkfile(aname)) {
    strcat(strcpy(fname2, s), ".q");
    absname(aname, searchlib(fname, fname2));
  }

  /* check whether module has already been loaded: */

  if ((mno = getmodno(as)) != NONE) {
    if (s == mainfile) mainno = mno;
    /* file already loaded, check for name conflicts and cyclic inclusions */
    if (strcmp(aname, strsp+_fnametb[mno]) != 0) {
      char msg[MAXSTRLEN];
      sprintf(msg, qcmsg[AMBIG_REF], as);
      yyerror(msg);
    } else if (stacked(mno))
      yyerror(qcmsg[CYCLIC_REF]);
    return mno;
  } else if ((mno = getmodno_by_fname(aname)) != NONE) {
    /* module has already been loaded under a different alias; we handle this
       case by manufacturing aliases for all symbols of the module */
    int _modno = modno, sz = symtbsz, i;
    addmod(as, aname, s);
    for (i = BUILTIN; i < sz; i++)
      if (symtb[i].modno == mno) {
	short flags = symtb[i].flags & ~EXT;
	if (flags & TSYM)
	  astype(i, 0, flags);
	else if (flags & VSYM)
	  asfvar(i, 0, flags);
	else
	  asfun(i, 0, symtb[i].argc, symtb[i].argv, flags);
      }
    mno = modno;
    modno = _modno;
    return mno;
  }

  /* open new file: */

  if (save) saveimps();

  if ((yyin = fopen(aname, "r")) == NULL) {
    source = s;
    fatal(qcmsg[FILE_NOT_FOUND]);
  }
#ifdef FLEX_SCANNER
  if (fsp > fst)
    yy_switch_to_buffer(yy_create_buffer(yyin, YY_BUF_SIZE));
#endif

  /* enter into module table and initialize settings: */

  addmod(as, aname, s);
  prio = 0;
  source = strdup(fname);
  yylineno = 1;
  BEGIN src;
  if (vflag)
    printf("%s:\n", s);
  return HAVE_FILE;

}

static
pushfile(incl)
     int incl;
{
  if (fsp - fst >= MAXFILES)
    fatal(qcmsg[TOO_MANY_FILES]);
  else {
#ifdef FLEX_SCANNER
    fsp->state = YY_CURRENT_BUFFER;
#endif
    fsp->yyin = yyin;
    fsp->yylineno = yylineno;
    fsp->prio = prio;
    fsp->modno = modno;
    fsp->source = source;
    fsp->incl = incl;
    fsp->argc = argc;
    fsp->argv = argv;
    fsp->argv0 = argv0;
    fsp->asv = asv;
    fsp->asv0 = asv0;
    fsp++;
  }
}

static popfile()
{
  fsp--;
#ifdef FLEX_SCANNER
  yy_delete_buffer(YY_CURRENT_BUFFER);
  yy_switch_to_buffer(fsp->state);
#endif
  yyin = fsp->yyin;
  yylineno = fsp->yylineno;
  prio = fsp->prio;
  modno = fsp->modno;
  source = fsp->source;
  argc = fsp->argc;
  argv = fsp->argv;
  argv0 = fsp->argv0;
  asv = fsp->asv;
  asv0 = fsp->asv0;
}

static popfile0()
{
  fsp--;
  yyin = fsp->yyin;
  yylineno = fsp->yylineno;
  prio = fsp->prio;
  modno = fsp->modno;
  source = fsp->source;
  argc = fsp->argc;
  argv = fsp->argv;
  argv0 = fsp->argv0;
  asv = fsp->asv;
  asv0 = fsp->asv0;
}

struct {
  char *s, *as;
} impq[MAXFILES];

static int impqsz = 0;

add_import(s, as)
     char *s, *as;
{
  if (impqsz >= MAXFILES)
    fatal(qcmsg[TOO_MANY_FILES]);
  impq[impqsz].s = s;
  impq[impqsz].as = as;
  impqsz++;
}

static
doimport(incl)
     int incl;
{
  int i, mno;
  if (impqsz == 0) return;
  pushfile(incl);
  argc = impqsz;
  argv0 = argv = aalloc(argc, sizeof(char*));
  asv0 = asv = aalloc(argc, sizeof(char*));
  if (!argv || !asv) fatal(qcmsg[MEM_OVF]);
  for (i = 0; i < argc; i++) {
    argv[i] = impq[i].s;
    asv[i] = impq[i].as;
  }
  impqsz = 0;
  while (argc--)
    if ((mno = opensrc(*argv++, *asv++, 1)) == HAVE_FILE) {
      if (argv[-1]) free(argv[-1]); if (asv[-1]) free(asv[-1]);
      return;
    } else {
      if (argv[-1]) free(argv[-1]); if (asv[-1]) free(asv[-1]);
      fsp--;
      modno = fsp->modno;
      if (mno != NONE)
	if (incl)
	  putinc(mno, 0);
	else
	  putimp(mno, 0);
      fsp++;
    }
  if (argv0) free(argv0);
  if (asv0) free(asv0);
  popfile0();
}

import()
{
  doimport(0);
}

include()
{
  doimport(1);
}

static int have_main = 0;

yywrap()
{
  int have_file = 0;
  saveimps();
  fclose(yyin);
  if (in_comment)
    return 1;
  else if (fsp > fst) {
    int mno = modno;
    fsp--;
    modno = fsp->modno;
    if (source) free(source);
    source = fsp->source;
    restoreimps();
    if (fsp->incl)
      putinc(mno, 0);
    else
      putimp(mno, 0);
    fsp++;
    while(argc--)
      if ((have_file = (mno = opensrc(*argv++, *asv++, 1)) == HAVE_FILE)) {
	if (argv[-1]) free(argv[-1]); if (asv[-1]) free(asv[-1]);
	break;
      } else {
	if (argv[-1]) free(argv[-1]); if (asv[-1]) free(asv[-1]);
	fsp--;
	if (mno != NONE)
	  if (fsp->incl)
	    putinc(mno, 0);
	  else
	    putimp(mno, 0);
	fsp++;
      }
    if (!have_file) {
      if (argv0) free(argv0); if (asv0) free(asv0);
      popfile();
    }
    return 0;
  } else {
    static int mno, save = 0;
    if (have_main) {
      mno = modno;
      modno = mainno;
      restoreimps();
      putimp(mno, 0);
      save = 1;
    } else if (mainno != NONE) {
      have_main = 1;
      modno = mainno;
      restoreimps();
    } else if (mainfile && !*mainfile) {
      /* create a dummy main module to hold the global imports */
      mno = modno;
      addmod("", "", mainfile);
      have_main = 1;
      restoreimps();
      if (mno > 0) putimp(mno, 0);
      save = 1;
    }
    while(argc-- && !(have_file = (mno=opensrc(*argv++, NULL, save))
		      == HAVE_FILE)) {
      putimp(mno, 0);
      save = 1;
    }
    if (!have_file && save) saveimps();
    return !have_file;
  }
}

initlex(_argc, _argv)
     int             _argc;
     char          **_argv;
{
  argc = _argc;
  argv = _argv;
  mainfile = (argc < 1)?NULL:*argv;
  while (argc && !**argv) argc--, argv++;
  if (prelude || argc >= 1) {
    fsp = fst;
    BEGIN src;
#ifndef FLEX_SCANNER
    fclose(yyin);
#endif
    if (prelude)
      opensrc(prelude, NULL, 0);
    else {
      if (mainfile && !*mainfile) {
	/* create a dummy main module to hold the global imports */
	addmod("", "", mainfile);
	saveimps();
	have_main = 1;
      }
      argc--;
      opensrc(*argv++, NULL, 0);
    }
    return 1;
  } else
    return 0;
}

srcstate()
{
  BEGIN src;
}
