
/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qdefs.h"

bool            debug_long = 0, brkdbg = 0;
volatile bool   brkflag = 0, quitflag = 0;
bool		gc_flag = 0, gc_v = 0;
double		gc_tol = 0.5;
bool		debug = 0;
int		debug_lock = 0;
bool		init_mode = 0;

static THREAD  *init_thr = NULL;

int lastblksz, maxnblks;

char           *qmmsg[] = {

	"Ok",				/* not used */

	"Break",
	"Halt",				/* not used */
	"Quit",				/* not used */
	"Out of memory",
	"Stack overflow",
	"Stack overflow",
	"Symbol table overflow",
	"Error in conditional",
	"Error in external function",
	"Value mismatch in definition",

	"Exception",
	"Fail",				/* not used */
	"Catch",			/* not used */

	"File %s not found",
	"Error reading code file",
	"Bad code file format",

	"Syntax error",
	"Unterminated string constant",
	"External object",
	"Unknown symbol",
	"Ambiguous symbol",
	"Not a variable",
	"Cannot redefine const variable",
	"Cannot undefine const variable",
	"Bad format specification",
	"No such directory",
	"Too many nested command files",
	"Error writing %s",
	"Error compiling %s",
	"Too many arguments",

	"Unknown variable %s",
	"Invalid command (type ? for help)",

	"This can't happen"

};

static EXPR *x_copy(EXPR *x);
static void xcopy_cleanup(void);

static int evalf(THREAD *thr, int fno);
static int eval_with_frame(THREAD *thr, 
			   EXPR *x, int fno, int *rp, int rc, OPREC *ip,
			   long xbp, int modno, int lineno);

static void lock_debug(THREAD *thr);
static void unlock_debug(THREAD *thr);
static void set_debug(THREAD *thr, bool debug);

static void debug_thread(THREAD *thr, char *msg);
static int rule(THREAD *thr, 
		int fno, long xbp, int addr, OPREC *ip, unsigned level,
		int modno, int lineno);
static void binding(THREAD *thr, int failed, int m, byte offs);
static void reduction(THREAD *thr, int fno, long xbp);
static void default_reduction(THREAD *thr, int fno, long xbp);
static void tail_reduction(THREAD *thr, int fno, long xbp, int fno1);

void error(char *s)
{
  fprintf(stderr, "! %s\n", s);
  fflush(stderr);
}

int             tmpspsz, tmptbsz;
int             atmpspsz = TMPSPSZ, atmptbsz = TMPTBSZ;

int	        maxargs = MAXARGS;

int		xnblks = 0;
XBLK           *xblk;
EXPR           *xheap;
EXPR           *xfreep;

THREAD	        threads[MAXTHREAD], *thr0 = threads;
short		nthreads, nused;

static THREAD  *nthr;

int push_mark(THREAD *thr, EXPR *h)
{
  if (!thr->mark) {
    if ((thr->mark = (Mark*)aalloc(100, sizeof(Mark)))) {
      thr->markp = thr->mark;
      thr->marksz = 100;
    } else {
      thr->qmstat = MEM_OVF;
      return 0;
    }
  } else if (thr->markp-thr->mark == thr->marksz) {
    Mark *mark1;
    if ((mark1 = (Mark*)arealloc(thr->mark, thr->marksz, 100, sizeof(Mark)))) {
      thr->mark = mark1;
      thr->markp = thr->mark+thr->marksz;
      thr->marksz += 100;
    } else {
      thr->qmstat = MEM_OVF;
      return 0;
    }
  }
  thr->markp->xp = thr->xsp-thr->xst;
  thr->markp->h = h;
  thr->markp++;
  return 1;
}

void pop_mark(THREAD *thr)
{
  if (thr->markp > thr->mark) --thr->markp;
}

int get_mark(THREAD *thr, long *xp, EXPR **h)
{
  if (thr->markp > thr->mark) {
    *xp = thr->markp[-1].xp;
    *h = thr->markp[-1].h;
    return 1;
  } else
    return 0;
}

int have_mark(THREAD *thr)
{
  return thr->markp>thr->mark;
}

/* The fx array stores expression nodes preallocated for the function symbols
   in the symbol table. This offers a significant improvement in performance
   when a program is run, since many operations are simply pushes of function
   symbols. Note that there are two versions of the array; the fx0 array is
   used for irreducible terms (created outside of special forms). */

static EXPR    *fx, *fx0;

/* MULTITHREADING SUPPORT */

#ifdef USE_THREADS

static pthread_key_t thr_key;

pthread_mutex_t global_mutex, tty_mutex, parse_mutex, reads_mutex;

static bool input_suspended;
static pthread_mutex_t input_mutex, init_mutex;
static pthread_cond_t input_cond, init_cond;

#ifndef _WIN32

/* do necessary cleanup at fork time */

void atfork_prepare(void)
{
  THREAD *thr;
  int i;
  for (i = 0; i < modtbsz; i++)
    if (dll_atfork[i].prepare)
      (*dll_atfork[i].prepare)();
  pthread_mutex_lock(&init_mutex);
  pthread_mutex_lock(&input_mutex);
#if 0
  /* NOTE: We don't lock the tty and parse mutexes, since that would suspend a
     fork in a background thread when the interpreter sits waiting in the main
     loop. */
  pthread_mutex_lock(&tty_mutex);
  pthread_mutex_lock(&parse_mutex);
#endif
  pthread_mutex_lock(&reads_mutex);
  if (nused > 1)
    for (thr = thr0+1; thr < thr0+nthreads; thr++)
      if (thr->used)
	pthread_mutex_lock(&thr->exit_mutex);
}

void atfork_parent(void)
{
  THREAD *thr;
  int i;
  for (i = 0; i < modtbsz; i++)
    if (dll_atfork[i].parent)
      (*dll_atfork[i].parent)();
  pthread_mutex_unlock(&init_mutex);
  pthread_mutex_unlock(&input_mutex);
#if 0
  /* See notes above. */
  pthread_mutex_unlock(&tty_mutex);
  pthread_mutex_unlock(&parse_mutex);
#endif
  pthread_mutex_unlock(&reads_mutex);
  if (nused > 1)
    for (thr = thr0+1; thr < thr0+nthreads; thr++)
      if (thr->used)
	pthread_mutex_unlock(&thr->exit_mutex);
}

void atfork_child(void)
{
  THREAD *thr, *this = get_thr();
  sigset_t sigset;
  int i;
  pthread_mutexattr_t attr;
  pthread_mutexattr_init(&attr);
  pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);

  for (i = 0; i < modtbsz; i++)
    if (dll_atfork[i].child)
      (*dll_atfork[i].child)();
  pthread_mutex_init(&global_mutex, NULL);
  pthread_mutex_init(&init_mutex, NULL);
  pthread_cond_init(&init_cond, NULL);
  pthread_mutex_init(&input_mutex, NULL);
  pthread_cond_init(&input_cond, NULL);
  pthread_mutex_init(&tty_mutex, NULL);
  pthread_mutex_init(&parse_mutex, &attr);
  pthread_mutex_init(&reads_mutex, NULL);

  pthread_mutex_lock(&global_mutex);
  if (this->debug) {
    input_suspended = 1;
    debug_lock = 1;
    this->debug_lock = 1;
  } else {
    input_suspended = 0;
    debug_lock = 0;
    this->debug_lock = 0;
  }

  /* this thread is the new main thread, hence we unblock all signals */
  sigemptyset(&sigset);
  pthread_sigmask(SIG_SETMASK, &sigset, NULL);

  /* the forking thread becomes the one and only thread in the child process,
     so update the thread table accordingly */
  if (nused > 1)
    for (thr = thr0; thr < thr0+nthreads; thr++)
      if (thr->used) {
	pthread_mutex_init(&thr->exit_mutex, NULL);
	pthread_cond_init(&thr->exit_cond, NULL);
	if (thr != this) {
	  thr->tty_lock = 0;
	  thr->debug_lock = 0;
	  thr->stats_fini = 1;
	  thr->endtime = clock();
	  thr->active = 0;
	  /* since there is no chance that the other threads will be activated
	     again we can as well collect their resources now */
	  thr->sticky = 1;
	  while (thr->xsp > thr->xst)
	    qmfree(thr, *--thr->xsp);
	  while (thr->asp > thr->ast)
	    free(*--thr->asp);
	  if (thr->xst) free(thr->xst); thr->xst = thr->xsp = NULL;
	  if (thr->ast) free(thr->ast); thr->ast = thr->asp = NULL;
	  if (thr->args) free(thr->args); thr->args = NULL;
	  if (thr->mark) free(thr->mark); thr->mark = NULL;
	  if (thr->vartb) free(thr->vartb); thr->vartb = NULL;
	} else {
	  /* update the info of the new main thread */
	  thr->id = pthread_self();
	  /* reinitialize the signal queue */
	  thr->nsig = 0;
	  thr->sigpend = thr->sigblk = 0;
	}
      }

  /*  if (this->debug) debug_thread(this, "thread #%d forked"); */
}

#endif

static void init_threads(void)
{
  pthread_mutexattr_t attr;
  pthread_mutexattr_init(&attr);
  pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);

  pthread_key_create(&thr_key, NULL);

  pthread_mutex_init(&global_mutex, NULL);
  pthread_mutex_init(&init_mutex, NULL);
  pthread_cond_init(&init_cond, NULL);
  pthread_mutex_init(&input_mutex, NULL);
  pthread_cond_init(&input_cond, NULL);
  pthread_mutex_init(&tty_mutex, NULL);
  pthread_mutex_init(&parse_mutex, &attr);
  pthread_mutex_init(&reads_mutex, NULL);

#ifdef HAVE_PTHREAD_ATFORK
  pthread_atfork(atfork_prepare, atfork_parent, atfork_child);
#endif
}

#endif

int init_thread(void)
{
  THREAD *thr;
#ifdef USE_THREADS
  pthread_mutex_lock(&global_mutex);
#else
  if (nthreads) return -1;
#endif
  thr = nthr;
  if (thr)
    nthr = thr->next;
  else if (nthreads >= MAXTHREAD)
    goto err;
  else
    thr = thr0+(nthreads++);

  thr->xstsz = XSTSZ;
  thr->astsz = ASTSZ;
  thr->maxxstsz = thr->maxastsz = stackmax;
  if (thr->maxxstsz > 0 && thr->maxxstsz < thr->xstsz)
    thr->xstsz = thr->maxxstsz;
  if (thr->maxastsz > 0 && thr->maxastsz < thr->astsz)
    thr->astsz = thr->maxastsz;
  if (!(thr->xst = (EXPR **)aalloc(thr->xstsz, sizeof(EXPR*))) ||
      !(thr->ast = (AREC **)aalloc(thr->astsz, sizeof(AREC*))) ||
      !(thr->args = (EXPR**)malloc((maxargs+1)*sizeof(EXPR*))))
    goto alloc_err;

  thr->next = NULL;
  thr->used = 1;
  thr->active = 1;
  thr->released = 0;
  thr->sticky = 0;
  thr->debug_lock = 0;
  thr->level = 0;

  thr->xsp = thr->xst;
  thr->asp = thr->ast;
  thr->qmstat = thr->qmstat_save = OK;
  thr->mode = 0;
  thr->debug = debug;
  thr->brkdbg = brkdbg;
  thr->brkflag = 0;
  thr->nsig = 0;
  thr->sigpend = thr->sigblk = 0;

  thr->mark = thr->markp = NULL;
  thr->marksz = 0;

  thr->vartb = 0;
  thr->nvarsyms = thr->avarsyms = 0;
  thr->lastaddr = NONE;

  thr->maxexprs = thr->nexprs = thr->nredns = 0;
  thr->stats_init = (thr>thr0);
  thr->stats_fini = 0;
  thr->starttime = clock();

#ifdef USE_THREADS
  pthread_setspecific(thr_key, thr);
  thr->id = pthread_self();
  if (thr == thr0)
    pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);
  else {
#ifndef _WIN32
    /* signals are to be handled in the main thread, hence we block all
       signals here */
    sigset_t sigset;
    sigfillset(&sigset);
    pthread_sigmask(SIG_SETMASK, &sigset, NULL);
#endif
    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
  }
  pthread_mutex_init(&thr->exit_mutex, NULL);
  pthread_cond_init(&thr->exit_cond, NULL);
#endif
  nused++;
  if (thr->debug && thr > thr0) debug_thread(thr, "thread #%d started");
  return thr-thr0;

 alloc_err:
  if (thr->xst) free(thr->xst);
  if (thr->ast) free(thr->ast);
  if (thr->args) free(thr->args);
  if (thr+1 < thr0+nthreads) {
    thr->next = nthr;
    nthr = thr;
  } else
    nthreads--;
 err:
#ifdef USE_THREADS
  pthread_mutex_unlock(&global_mutex);
#endif
  return -1;
}

void exit_thread(int id)
{
  THREAD *thr = thr0+id;
#ifndef USE_THREADS
  return;
#else
  if (!thr->used || !thr->active) return;
  thr->stats_fini = 1;
  thr->endtime = clock();
  pthread_mutex_lock(&thr->exit_mutex);
  thr->active = 0;
  pthread_cond_broadcast(&thr->exit_cond);
  pthread_mutex_unlock(&thr->exit_mutex);
  if (thr->released) pthread_mutex_lock(&global_mutex);
  unlock_debug(thr);
  if (thr->tty_lock) pthread_mutex_unlock(&tty_mutex);
  pthread_mutex_unlock(&global_mutex);
  if (thr->debug) debug_thread(thr, "thread #%d exited");
#endif
}

void fini_thread(int id)
{
  THREAD *thr;
#ifndef USE_THREADS
  return;
#else
  thr = thr0+id;
  thr->sticky = 1;
  while (thr->xsp > thr->xst)
    qmfree(thr, *--thr->xsp);
  while (thr->asp > thr->ast)
    free(*--thr->asp);
  if (thr->xst) free(thr->xst);
  if (thr->ast) free(thr->ast);
  if (thr->args) free(thr->args);
  if (thr->mark) free(thr->mark);
  if (thr->vartb) free(thr->vartb);
  pthread_mutex_destroy(&thr->exit_mutex);
  pthread_cond_destroy(&thr->exit_cond);
  thr->used = 0;
  if (thr+1 < thr0+nthreads) {
    thr->next = nthr;
    nthr = thr;
  } else
    nthreads--;
  nused--;
#endif
}

THREAD *get_thr(void)
{
#ifdef USE_THREADS
  return (THREAD*)pthread_getspecific(thr_key);
#else
  return thr0;
#endif
}

void kill_threads(void)
{
#ifdef USE_THREADS
  THREAD *thr;
  if (nused > 1)
    for (thr = thr0+1; thr < thr0+nthreads; thr++)
      if (thr->used)
	pthread_cancel(thr->id);
  pthread_mutex_unlock(&global_mutex);
#endif
}

void wait_threads(void)
{
#ifdef USE_THREADS
  THREAD *thr;
  if (nused > 1)
    for (thr = thr0+1; thr < thr0+nthreads; thr++)
      if (thr->used) {
	pthread_mutex_lock(&thr->exit_mutex);
	while (thr->active)
	  pthread_cond_wait(&thr->exit_cond, &thr->exit_mutex);
	pthread_mutex_unlock(&thr->exit_mutex);
      }
  pthread_mutex_lock(&global_mutex);
#endif
}

int this_thread(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  return thr-thr0;
#else
  return 0;
#endif
}

void release_lock(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  if (thr > thr0) pthread_testcancel();
  unlock_debug(thr);
  pthread_mutex_unlock(&global_mutex);
  thr->released = 1;
#endif
}

void acquire_lock(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  if (init_thr) {
    /* suspend secondary threads until initializations are over */
    pthread_mutex_lock(&init_mutex);
    while (init_thr && thr != init_thr)
      pthread_cond_wait(&init_cond, &init_mutex);
    pthread_mutex_unlock(&init_mutex);
  }
  pthread_mutex_lock(&global_mutex);
  thr->released = 0;
  if (thr > thr0) pthread_testcancel();
  lock_debug(thr);
#endif
}

void start_init(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&init_mutex);
  init_thr = get_thr();
#endif
  init_mode = 1;
#ifdef USE_THREADS
  pthread_mutex_unlock(&init_mutex);
#endif
}

void end_init(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&init_mutex);
#endif
  init_mode = 0;
#ifdef USE_THREADS
  init_thr = NULL;
  pthread_cond_broadcast(&init_cond);
  pthread_mutex_unlock(&init_mutex);
#endif
}

void acquire_input(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&input_mutex);
  while (input_suspended)
    pthread_cond_wait(&input_cond, &input_mutex);
  acquire_tty();
  pthread_mutex_unlock(&input_mutex);
#endif
}

void release_input(void)
{
#ifdef USE_THREADS
  release_tty();
#endif
}

void suspend_input(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&input_mutex);
  input_suspended = 1;
  pthread_mutex_unlock(&input_mutex);
#endif
}

void resume_input(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&input_mutex);
  input_suspended = 0;
  pthread_cond_signal(&input_cond);
  pthread_mutex_unlock(&input_mutex);
#endif
}

void acquire_tty(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  pthread_mutex_lock(&tty_mutex);
  thr->tty_lock = 1;
#endif
}

void release_tty(void)
{
#ifdef USE_THREADS
  THREAD *thr = get_thr();
  pthread_mutex_unlock(&tty_mutex);
  thr->tty_lock = 0;
#endif
}

/* INITIALIZATION: */

/* init(), reinit(): (re)initialize the Q machine. */

/* Note: These routines are only invoked from the main thread. */

static initfx(int fno)
{
  int	       *rp;
  int             rc;
  fx[fno].refc = fx0[fno].refc = 1;
  fx[fno].fno = fx0[fno].fno = fno;
  fx[fno].type = fx0[fno].type = symtb[fno].type;
  fx[fno].argc = fx0[fno].argc = symtb[fno].argc;
  fx[fno].red =
    fno < BUILTIN && funtb[fno] && nargs[fno] == 0 ||
    symtb[fno].f && symtb[fno].argc == 0 ||
    (symtb[fno].flags & VSYM) ||
    match(thr0, fno, &rp, &rc);
  fx0[fno].red = (symtb[fno].flags & VSYM)?1:0;
  fx[fno].pipe = fx0[fno].pipe = 0;
}

void init(void)
{
  int 		fno;
  /* initialize the main thread */
#ifdef USE_THREADS
  init_threads();
#endif
  nthreads = nused = 0; nthr = NULL;
  if (init_thread() == -1)
    fatal("memory overflow");
  /* initialize global data */
  if (!(xblk = (XBLK *)malloc(sizeof(XBLK))) ||
      !(fx = (EXPR *)aalloc(symtbsz, sizeof(EXPR))) ||
      !(fx0 = (EXPR *)aalloc(symtbsz, sizeof(EXPR))))
    fatal("memory overflow");
  initfx(DEFOP);
  initfx(UNDEFOP);
  for (fno = BINARY; fno < symtbsz; fno++)
    initfx(fno);
  xblk->next = NULL;
  xfreep = NULL;
  xheap = xblk->x;
  xnblks = 1;
  lastblksz = memmax % XBLKSZ;
  maxnblks = memmax/XBLKSZ+((memmax <= 0||lastblksz==0)?0:1);
  if (lastblksz == 0) lastblksz = XBLKSZ;
  brkflag = 0;
}

void reinit(void)
{
  int 		fno;
  /* reinitialze the main thread */
  if (thr0->args) free(thr0->args);
  if (!(thr0->args = (EXPR**)malloc((maxargs+1)*sizeof(EXPR*))))
    fatal("memory overflow");
  thr0->qmstat = thr0->qmstat_save = OK;
  thr0->mode = 0;
  thr0->debug = debug;
  thr0->brkdbg = brkdbg;
  thr0->brkflag = 0;
  thr0->nsig = 0;
  thr0->sigpend = thr0->sigblk = 0;
  thr0->maxexprs = thr0->nexprs = thr0->nredns = 0;
  thr0->stats_init = thr0->stats_fini = 0;
  /* reinitialize global data */
  if (fx) free(fx);
  if (fx0) free(fx0);
  if (!(fx = (EXPR *)aalloc(symtbsz, sizeof(EXPR))) ||
      !(fx0 = (EXPR *)aalloc(symtbsz, sizeof(EXPR))))
    fatal("memory overflow");
  initfx(DEFOP);
  initfx(UNDEFOP);
  for (fno = BINARY; fno < symtbsz; fno++)
    initfx(fno);
  brkflag = 0;
}

/* EXPRESSION HEAP MANAGEMENT: */

unsigned long fexprs = 0;

/* x_alloc() allocates an expression */

static EXPR *x_alloc(THREAD *thr)
{
  EXPR		*x;

  if (x = xfreep) {
    if (++thr->nexprs > thr->maxexprs) thr->maxexprs = thr->nexprs;
    xfreep = xfreep->data.xp;
    --fexprs;
#ifdef USE_THREADS
    x->sticky = 0;
    x->thrid = thr-thr0;
#endif
    return x;
  }
  if (xheap >= xblk->x+XBLKSZ) {
    if (maxnblks > 0 && xnblks < maxnblks) {
      /* try to allocate a new block */
      XBLK	       *xblk1 = (XBLK*) malloc(sizeof(XBLK));
      if (xblk1) {
	xblk1->next = xblk;
	xblk = xblk1;
	xheap = xblk->x;
	xnblks++;
      } else
	return NULL;
    } else
      return NULL;
  }
  if (maxnblks > 0 && xnblks == maxnblks && xheap-xblk->x >= lastblksz)
    return NULL;
  else {
    x = xheap++;
    if (++thr->nexprs > thr->maxexprs) thr->maxexprs = thr->nexprs;
#ifdef USE_THREADS
    x->sticky = 0;
    x->thrid = thr-thr0;
#endif
    return x;
  }
}

/* x_free() frees an expression by inserting it into the free list */

inline
static void x_free(THREAD *thr, EXPR *x)
{
#ifdef USE_THREADS
  if (!x->sticky)
#endif
    if (thr->nexprs > 0) --thr->nexprs;
  ++fexprs;
  x->data.xp = xfreep;
  xfreep = x;
}

/* x_collect() implements the usual stop-and-copy garbage collector which is
   used to defrag the expression heap and return unused memory to the system
   pool. It copies the expression objects reachable from def'ed symbols and
   the stack into fresh memory. If there's not enough memory to allocate the
   new memory arena, then the process is aborted and the current memory arena
   is left unchanged. */

/* Note: This routine is only invoked from the main thread, and only if no
   other threads are currently active. */

static XBLK	       *fblk;	/* the free block list */
static XBLK	       *xblk1;	/* the new heap */
static EXPR	       *xheap1;	/* new heap ptr */

static void x_collect(void)
{
  XBLK *fblk1;
  int fno;
  EXPR **xp;
  int lastblksz, xnblks1;
  unsigned long nheap = ((unsigned long)(xnblks-1))*XBLKSZ+(xheap-xblk->x);
  unsigned long mexprs = nheap-fexprs;

  if (gc_v) { printf("garbage collecting ... "); fflush(stdout); }

  /* compute the size of the new memory arena */
  lastblksz = mexprs % XBLKSZ;
  xnblks1 = mexprs/XBLKSZ+((lastblksz==0)?0:1);
  if (xnblks1 <= 0) xnblks1 = 1;

  /* create the new memory arena */
  if ((fblk = fblk1 = (XBLK *)malloc(sizeof(XBLK)))) {
    int i;
    for (i = 1; i < xnblks1; i++)
      if ((fblk1->next = (XBLK *)malloc(sizeof(XBLK))))
	fblk1 = fblk1->next;
      else {
	for (; fblk; fblk = fblk1) {
	  fblk1 = fblk->next;
	  free(fblk);
	}
	break;
      }
  }
  if (fblk)
    fblk1->next = NULL;
  else {
    if (gc_v) printf("failed (not enough memory)\n");
    return;
  }

  /* reset reference counts of preallocated function nodes: */
  for (fno = BINARY; fno < symtbsz; fno++)
    fx[fno].refc = fx0[fno].refc = 1;

  /* initialize block pointers */
  xblk1 = fblk; xheap1 = xblk1->x;
  fblk = fblk->next; xblk1->next = NULL;

  /* copy all referenced objects into the free memory arena */
  if (mexprs > 0) {
    for (fno = INPUTOP; fno < symtbsz+tmptbsz; fno++)
      if (symtb[fno].x)
	symtb[fno].x = (void*) x_copy(symtb[fno].x);
    for (xp = thr0->xst; xp < thr0->xsp; xp++)
      *xp = x_copy(*xp);
    xcopy_cleanup();
  }

  /* discard any remaining free blocks (shouldn't happen but Murphy knows) */
  for (; fblk; fblk = fblk1) {
    fblk1 = fblk->next;
    free(fblk);
  }

  /* discard the old memory arena */
  for (fblk = xblk; fblk; fblk = fblk1) {
    fblk1 = fblk->next;
    free(fblk);
  }

  /* reinitialize the heap: */
  xblk = xblk1;
  xfreep = NULL;
  xheap = xheap1;
  xnblks = xnblks1;
  fexprs = 0;

  if (gc_v) printf("done (moved %ld cells)\n", mexprs);
}

#define MARK -99	/* refc value used to mark "broken hearts" */

#if 0

/* This routine is now implemented non-recursively below. */

static EXPR *x_copy(EXPR *x)
{
  if (x->refc == MARK) {
    /* this is a "broken heart" which has been moved already
       to the location indicated by xp */
    x->data.xp->refc++;
    return x->data.xp;
  } else if (x->fno >= BINARY && x->fno < symtbsz) {
    /* preallocated function node, doesn't have to be copied */
    x->refc++;
    return x;
  } else {
    if (xheap1 >= xblk1->x+XBLKSZ) {
      XBLK *xblk2;
      /* get a new block */
      if (!fblk) fatal(qmmsg[THIS_CANT_HAPPEN]);
      xblk2 = xblk1; xblk1 = fblk;
      fblk = fblk->next; xblk1->next = xblk2;
      xheap1 = xblk1->x;
    }
    /* move x to new location */
    *xheap1 = *x;
    xheap1->refc = 1;
    /* mark x as a broken heart */
    x->refc = MARK;
    x->data.xp = xheap1++;
    /* copy args recursively */
    switch (x->fno) {
    case CONSOP: case PAIROP: case APPOP:
      x->data.xp->data.args.x1 = x_copy(x->data.xp->data.args.x1);
      x->data.xp->data.args.x2 = x_copy(x->data.xp->data.args.x2);
      break;
    case VECTOP: {
      int i, n = x->data.xp->data.vect.n;
      for (i = 0; i < n; i++)
	x->data.xp->data.vect.xv[i] = x_copy(x->data.xp->data.vect.xv[i]);
      break;
    }
    }
    /* return the moved object: */
    return x->data.xp;
  }
}

static void xcopy_cleanup(void)
{
}

#else

/* This operation should be "failsafe", so we actually implement it
   non-recursively. */

static EXPR **xstk = NULL;

static int xstkp = 0, xstka = 0;

static EXPR *x_copy(EXPR *x)
{
  EXPR *y;
  int mark = xstkp;
 loop:
  if (x->refc == MARK) {
    /* this is a "broken heart" which has been moved already
       to the location indicated by xp */
    x->data.xp->refc++;
    y = x->data.xp;
    goto pop;
  } else if (x->fno >= BINARY && x->fno < symtbsz) {
    /* preallocated function node, doesn't have to be copied */
    x->refc++;
    y = x;
    goto pop;
  } else {
    if (xheap1 >= xblk1->x+XBLKSZ) {
      XBLK *xblk2;
      /* get a new block */
      if (!fblk) fatal(qmmsg[THIS_CANT_HAPPEN]);
      xblk2 = xblk1; xblk1 = fblk;
      fblk = fblk->next; xblk1->next = xblk2;
      xheap1 = xblk1->x;
    }
    /* move x to new location */
    *xheap1 = *x;
    xheap1->refc = 1;
    /* mark x as a broken heart */
    x->refc = MARK;
    y = x->data.xp = xheap1++;
    /* copy args recursively */
    switch (y->fno) {
    case CONSOP: case PAIROP: case APPOP:
      if (xstkp >= xstka) {
	if (xstka >= INT_MAX ||
	    !(xstk = xstka?
	      realloc(xstk, (xstka+10240)*sizeof(EXPR*)):
	      malloc(10240*sizeof(EXPR*))))
	  fatal("memory overflow");
	else
	  xstka += 10240;
      }
      xstk[xstkp++] = x;
      x = x->data.args.x1;
      goto loop;
    case VECTOP: {
      int i, n = y->data.vect.n;
      for (i = 0; i < n; i++)
	y->data.vect.xv[i] = x_copy(y->data.vect.xv[i]);
      goto pop;
    }
    default:
    pop:
      while (xstkp > mark && x == xstk[xstkp-1]->data.args.x2) {
	xstk[xstkp-1]->data.xp->data.args.x2 = y;
	x = xstk[--xstkp];
	y = xstk[xstkp]->data.xp;
      }
      if (xstkp > mark) {
	xstk[xstkp-1]->data.xp->data.args.x1 = y;
	x = xstk[xstkp-1]->data.args.x2;
	goto loop;
      }
    }
  }
  /* return the moved object: */
  return y;
}

static void xcopy_cleanup(void)
{
  /* collect temp stack */
  free(xstk);
  xstk = NULL;
  xstkp = xstka = 0;
}

#endif

/* qmfree( x ) decrements the reference count for the expression pointed to by
   x and if the count drops to zero frees the heap space occupied by x. In
   order to achieve robustness and better performance this is implemented
   (mostly) non-recursively using an explicit stack built with the xp field in
   the EXPR data structure. */

void qmfree(THREAD *thr, EXPR *x)
{
  EXPR	       *xp = NULL, *x1;
  if (!x) return;
  do {
    if (!--x->refc) {
      switch (x->fno) {
      case CONSOP: case PAIROP: case APPOP:
      push:
	x1 = x->data.args.x1;
	x->data.xp = xp;
	xp = x;
	x = x1;
	break;
      case STRVALOP:
	free(x->data.s);
	goto pop;
      case INTVALOP:
	mpz_clear(x->data.z);
	goto pop;
      case FILEVALOP:
	if (x->pipe)
	  pclose(x->data.fp);
	else
	  fclose(x->data.fp);
	goto pop;
      case USRVALOP:
	/* invoke external destructor */
	if (x->type)
	  if (symtb[x->type].f) {
	    void (*f)() = symtb[x->type].f;
	    (*f) (x->data.vp);
	  } else if (x->data.vp)
	    free(x->data.vp);
	goto pop;
      case VECTOP: {
	int i, n = x->data.vect.n;
	for (i = 0; i < n; i++)
	  qmfree(thr, x->data.vect.xv[i]);
	if (x->data.vect.xv)
	  free(x->data.vect.xv);
      }
      default:
      pop:
	while (xp && x == xp->data.args.x2) {
	  if (!x->refc) x_free(thr, x);
	  x = xp;
	  xp = x->data.xp;
	}
	if (!x->refc) x_free(thr, x);
	if (xp)
	  x = xp->data.args.x2;
      }
    } else {
#ifdef USE_THREADS
      if (thr->sticky && x->thrid == thr-thr0)
	x->sticky = 1;
#endif
      goto pop;
    }
  } while (xp);
}

/* qmnew( x ) counts a new reference to the expression pointed to by x;
   returns: x. */

inline
EXPR *qmnew(EXPR *x)
{
  if (x) {
    x->refc++;
    return x;
  } else
    return NULL;
}

/* clear(): clear main stack and reinitialize pointers and status. */

/* Note: This routine is only invoked from the main thread. */

void clear(int force_gc)
{
  int defxstsz = XSTSZ, defastsz = ASTSZ; 

  /* free the stack first, to reclaim open file handles and dynamic data */
  while (thr0->xsp > thr0->xst)
    qmfree(thr0, *--thr0->xsp);

  while (thr0->asp > thr0->ast)
    free(*--thr0->asp);

  /* perform garbage collection on the expression heap (only do this if no
     other threads are currently active) */
#ifdef USE_THREADS
  if (nused <= 1)
#endif
    {
      unsigned long d, dmod;

      d = fexprs/XBLKSZ; dmod = fexprs%XBLKSZ;
      if (dmod > 0 && dmod + (XBLKSZ-(xheap-xblk->x)) > XBLKSZ) d++;
      if (force_gc || gc_flag && d > gc_tol*xnblks) x_collect();
    }

  /* Reallocate evaluation and activation stacks to their original sizes such
     that the evaluation loop doesn't get hooked with the Q machine having
     claimed all available memory for the stacks. */

  thr0->maxxstsz = thr0->maxastsz = stackmax;

  if (thr0->maxxstsz > 0 && thr0->maxxstsz < defxstsz)
    defxstsz = thr0->maxxstsz;
  if (thr0->maxastsz > 0 && thr0->maxastsz < defastsz)
    defastsz = thr0->maxastsz;

  if (thr0->xstsz > defxstsz) {
    thr0->xst = (EXPR**)realloc(thr0->xst, defxstsz*sizeof(EXPR*));
    thr0->xstsz = defxstsz;
  }
  if (thr0->astsz > defastsz) {
    thr0->ast = (AREC**)realloc(thr0->ast, defastsz*sizeof(AREC*));
    thr0->astsz = defastsz;
  }
  if (!thr0->xst || !thr0->ast)
    /* This shouldn't happen, but Murphy knows ... */
    fatal(qmmsg[THIS_CANT_HAPPEN]);

  /* reinitialize status variables */
  
  if (thr0->mark) free(thr0->mark);
  thr0->mark = thr0->markp = NULL;
  thr0->marksz = 0;
  
  thr0->xsp = thr0->xst;
  thr0->asp = thr0->ast;
  thr0->qmstat = thr0->qmstat_save = OK;
  thr0->mode = 0;
  thr0->debug = debug;
  thr0->brkdbg = brkdbg;
  thr0->brkflag = 0;
  thr0->nsig = 0;
  thr0->sigpend = thr0->sigblk = 0;
}

/* SIGNAL HANDLING */

static volatile int defer_sig = 0;
static volatile int saved_sig = 0;

RETSIGTYPE sig_handler(int sig)
{
  THREAD *thr;
  SIGHANDLER_RESTORE(sig, sig_handler);
  thr = get_thr();
  if (!thr || sig <= 0 || sig > NSIGNALS)
    SIGHANDLER_RETURN(0);
  /* prevent race conditions */
  if (defer_sig)
    saved_sig = sig;
  else if (thr->nsig < NSIGNALS) {
    int i;
    for (i = 0; i < thr->nsig && thr->sig[i] != sig; i++) ;
    if (i >= thr->nsig)
      /* enqueue the signal */
      thr->sig[thr->nsig++] = sig;
  }
  SIGHANDLER_RETURN(0);
}

inline
static int get_sig(THREAD *thr)
{
  int sig;
  defer_sig++;
  if (thr->sigblk || !thr->nsig) {
    defer_sig--;
    return 0;
  }
  sig = *thr->sig;
  memmove(thr->sig+1, thr->sig, --thr->nsig);
  defer_sig--;
  if (!defer_sig && saved_sig) {
    raise(saved_sig);
    saved_sig = 0;
  }
  return sig;
}

/* EXPRESSION CONSTRUCTORS: */

/* In case one of these operations fails, NULL is returned and qmstat is
   set to MEM_OVF. */

EXPR *intexpr(THREAD *thr, long i)
{
  mpz_t z;
  mpz_init(z);
  if (z->_mp_d) {
    mpz_set_si(z, i); /* this shouldn't fail */
    return mpzexpr(thr, z);
  } else {
    thr->qmstat = MEM_OVF;
    return NULL;
  }
}

EXPR *uintexpr(THREAD *thr, unsigned long i)
{
  mpz_t z;
  mpz_init(z);
  if (z->_mp_d) {
    mpz_set_ui(z, i); /* this shouldn't fail */
    return mpzexpr(thr, z);
  } else {
    thr->qmstat = MEM_OVF;
    return NULL;
  }
}

EXPR *mpzexpr(THREAD *thr, mpz_t z)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = INTVALOP;
    x->type = INTTYPE;
    x->argc = 0;
    x->red = 0;
    x->pipe = 0;
    memcpy(x->data.z, z, sizeof(mpz_t));
  } else {
    mpz_clear(z);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *floatexpr(THREAD *thr, double f)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = FLOATVALOP;
    x->type = FLOATTYPE;
    x->argc = 0;
    x->red = 0;
    x->pipe = 0;
    x->data.f = f;
  } else
    thr->qmstat = MEM_OVF;
  return x;
}

EXPR *strexpr(THREAD *thr, char *s)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = STRVALOP;
    x->type = (!s[0]||s[1])?STRTYPE:CHARTYPE;
    x->argc = 0;
    x->red = 0;
    x->pipe = 0;
    x->data.s = s;
  } else {
    free(s);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *fileexpr(THREAD *thr, FILE *fp)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = FILEVALOP;
    x->type = FILETYPE;
    x->argc = 0;
    x->red = 0;
    x->pipe = 0;
    x->data.fp = fp;
  } else {
    fclose(fp);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *pipeexpr(THREAD *thr, FILE *fp)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = FILEVALOP;
    x->type = FILETYPE;
    x->argc = 0;
    x->red = 0;
    x->pipe = 1;
    x->data.fp = fp;
  } else {
    pclose(fp);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *vectexpr(THREAD *thr, int n, EXPR **xv)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    int i;
    x->refc = 0;
    x->fno = VECTOP;
    x->type = TUPLETYPE;
    x->argc = 0;
    x->red = 0;
    for (i = 0; i < n; i++)
      if (xv[i]->red) {
	x->red = 1;
	break;
      }
    x->pipe = 0;
    x->data.vect.n = n;
    x->data.vect.xv = xv;
  } else {
    if (xv) {
      int i;
      for (i = 0; i < n; i++)
	qmfree(thr, xv[i]);
      free(xv);
    }
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *usrexpr(THREAD *thr, int type, void *vp)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = USRVALOP;
    x->type = type;
    x->argc = 0;
    x->red = 0;
    x->pipe = 0;
    x->data.vp = vp;
  } else {
    if (type)
      if (symtb[type].f) {
	void (*f)() = symtb[type].f;
	(*f) (vp);
      } else if (vp)
	free(vp);
    thr->qmstat = MEM_OVF;
  }
  return x;
}

EXPR *funexpr(THREAD *thr, int fno)
{
  if (fno < symtbsz)
    return (thr->mode?fx:fx0)+fno;
  else {
    EXPR	       *x = x_alloc(thr);
    if (x) {
      x->refc = 0;
      x->fno = fno;
      x->type = symtb[fno].type;
      x->argc = symtb[fno].argc;
      x->red = /*symtb[fno].flags&VSYM?1:*/thr->mode;
      x->pipe = 0;
    } else
      thr->qmstat = MEM_OVF;
    return x;
  }
}

EXPR *consexpr(THREAD *thr, int fno, EXPR *x1, EXPR *x2)
{
  EXPR	       *x = x_alloc(thr);
  if (x) {
    x->refc = 0;
    x->fno = fno;
    if (fno == APPOP) {
      if (x1->fno == FLIPOP) {
	unsigned long argv = (x2->fno==APPOP)?x2->data.args.argv:
	  symtb[x2->fno].argv;
	x->data.args.argv = ((argv & 1) << 1) | ((argv & 2) >> 1);
      } else {
	unsigned long argv = (x1->fno==APPOP)?x1->data.args.argv:
	  symtb[x1->fno].argv;
	x->data.args.argv = argv >> 1;
      }
      if (x1->argc) {
	x->type = x1->type;
	x->argc = x1->argc-1;
      } else {
	x->type = 0;
	x->argc = 0;
      }
      x->red = thr->mode || x1->red || x2->red;
    } else {
      x->data.args.argv = 0;
      x->type = fno==CONSOP?LISTTYPE:fno==PAIROP?TUPLETYPE:0;
      x->argc = 0;
      x->red = x1->red || x2->red;
    }
    x->pipe = 0;
    x->data.args.x1 = qmnew(x1);
    x->data.args.x2 = qmnew(x2);
  } else
    thr->qmstat = MEM_OVF;
  return x;
}

/* STACK OPERATIONS: */

/* stack_avail() checks whether there is room on the stack; the stack
   is enlarged if required (in case this fails qmstat is set to XST_OVF
   or MEM_OVF) */

static int stack_avail(THREAD *thr)
{
  if (thr->maxxstsz > 0 && thr->xsp - thr->xst >= thr->maxxstsz) {
    thr->qmstat = XST_OVF;
    return (0);
  } else if (thr->xsp - thr->xst == thr->xstsz) {
    EXPR **xst1;
    int n = XSTSZ/4;
    if (thr->maxxstsz > 0 && thr->xstsz+n > thr->maxxstsz)
      n = thr->maxxstsz-thr->xstsz;
    if (n <= 0 ||
	!(xst1 = (EXPR**)arealloc(thr->xst, thr->xstsz, n,
				  sizeof(EXPR*)))) {
      thr->qmstat = MEM_OVF;
      return (0);
    } else {
      thr->xst = xst1;
      thr->xsp = thr->xst+thr->xstsz;
      thr->xstsz += n;
    }
  }
  return (1);
}

/* push( x ): push a copy of expression x onto the stack; returns:
   zero iff error (cf. qmstat). */

int push(THREAD *thr, EXPR *x)
{
  if (stack_avail(thr)) {
    *thr->xsp++ = qmnew(x);
    return (1);
  } else
    return (0);
}

/* pushint(): push an integer value */

int pushint(THREAD *thr, long i)
{
  if (stack_avail(thr)) {
    EXPR	       *x = intexpr(thr, i);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushuint(): push an unsigned integer value */

int pushuint(THREAD *thr, unsigned long i)
{
  if (stack_avail(thr)) {
    EXPR	       *x = uintexpr(thr, i);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushmpz(): push a big integer value */

int pushmpz(THREAD *thr, mpz_t z)
{
  if (stack_avail(thr)) {
    EXPR	       *x = mpzexpr(thr, z);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushfloat(): push a float value */

int pushfloat(THREAD *thr, double f)
{
  if (stack_avail(thr)) {
    EXPR	       *x = floatexpr(thr, f);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushstr(): push a string value (STRVALOP) */

int pushstr(THREAD *thr, char *s)
{
  if (stack_avail(thr)) {
    EXPR	       *x = strexpr(thr, s);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushfile(): push a file value */

int pushfile(THREAD *thr, FILE *fp)
{
  if (stack_avail(thr)) {
    EXPR	       *x = fileexpr(thr, fp);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushpipe(): push a file value associated with a pipe */

int pushpipe(THREAD *thr, FILE *fp)
{
  if (stack_avail(thr)) {
    EXPR	       *x = pipeexpr(thr, fp);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushvect(): push a vector (fno = -n) */

int pushvect(THREAD *thr, int n, EXPR **xv)
{
  if (stack_avail(thr)) {
    EXPR	       *x = vectexpr(thr, n, xv);
    if (x) {
      *thr->xsp++ = qmnew(x);
      return 1;
    } else
      return 0;
  } else
    return 0;
}

/* pushfun(fno): push a term constructed from fno. */

int pushfun(THREAD *thr, int fno)
{
  if (stack_avail(thr)) {
    EXPR	       *x;
    switch (fno) {
    case PAIROP:
      if (thr->xsp[-1]->fno == VECTOP || thr->xsp[-1]->fno == VOIDOP) {
	int i, n = (thr->xsp[-1]->fno == VOIDOP)?0:thr->xsp[-1]->data.vect.n;
	if (n && thr->xsp[-1]->refc == 1) {
	  /* prepend a new element to an existing vector */
	  EXPR **xv = realloc(thr->xsp[-1]->data.vect.xv, (n+1)*sizeof(EXPR*));
	  if (!xv) {
	    thr->qmstat = MEM_OVF;
	    return 0;
	  } else {
	    for (i = n; i > 0; i--) xv[i] = xv[i-1];
	    xv[0] = thr->xsp[-2];
	    thr->xsp[-1]->data.vect.xv = xv;
	    thr->xsp[-1]->data.vect.n++;
	    thr->xsp[-1]->red = thr->xsp[-1]->red || thr->xsp[-2]->red;
	    thr->xsp[-2] = thr->xsp[-1];
	    thr->xsp--;
	    return 1;
	  }
	} else {
	  /* construct a new vector */
	  EXPR **xv = malloc((n+1)*sizeof(EXPR*));
	  if (!xv) {
	    thr->qmstat = MEM_OVF;
	    return 0;
	  } else {
	    xv[0] = qmnew(thr->xsp[-2]);
	    for (i = 0; i < n; i++)
	      xv[i+1] = qmnew(thr->xsp[-1]->data.vect.xv[i]);
	    if (x = vectexpr(thr, n+1, xv)) {
	      qmfree(thr, *--thr->xsp); qmfree(thr, *--thr->xsp);
	      *thr->xsp++ = qmnew(x);
	      return 1;
	    } else
	      return 0;
	  }
	}
      }
      /* else part falls through to the following case */
    case CONSOP: case APPOP:
      if (x = consexpr(thr, fno, thr->xsp[-2], thr->xsp[-1])) {
	thr->xsp[-2]->refc--, thr->xsp[-1]->refc--;
	thr->xsp--;
	thr->xsp[-1] = qmnew(x);
	return 1;
      } else
	return 0;
    default:
      if (x = funexpr(thr, fno)) {
	*thr->xsp++ = qmnew(x);
	return 1;
      } else
	return 0;
    }
  } else
    return 0;
}

/* ARGUMENTS: */

/* pushlval(): determine a left-hand side or local variable value and push it
   on the stack. xbp denotes the (relative) base pointer pointing to the
   arguments and local variables on the stack. */

static int pushlval(THREAD *thr,
		    int fno, long xbp, byte offs, byte plen, PATH p)
{
  EXPR           *x;
  int             i = offs?0:1;
  
  if (offs)
    if (fno == APPOP)
      x = thr->xst[xbp+offs+1];
    else
      x = thr->xst[xbp+offs-1];
  else
    x = thr->xst[xbp+getpath(p, 0)];
  for (; i < plen; i++) {
    if (x->fno == VECTOP) {
      int n = x->data.vect.n, k = 0;
      while (i < plen && getpath(p, i) == 1)
	i++, k++;
      if (i < plen)
	/* assert: getpath(p,i)==0, k is the current element */
	x = x->data.vect.xv[k];
      else if (k == n)
	/* end of the vector has been reached */
	return pushfun(thr, VOIDOP);
      else {
	/* copy vector of remaining elements (assert: n>k>0) */
	EXPR **xv = malloc((n-k)*sizeof(EXPR*));
	int j;
	if (!xv) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	}
	for (j = k; j < n; j++)
	  xv[j-k] = qmnew(x->data.vect.xv[j]);
	return pushvect(thr, n-k, xv);
      }
    } else if (getpath(p, i) == 0)
      x = x->data.args.x1;
    else
      x = x->data.args.x2;
  }
  return push(thr, x);
}

/* GLOBAL ENVIRONMENT: */

/* setvar(): associate a (variable) symbol with an expression */

int setvar(int vno, EXPR *x)
{
  if (!(symtb[vno].flags & VSYM)) {
    thr0->qmstat = BAD_DEF;
    return 0;
  } else if (symtb[vno].x && (symtb[vno].flags & CST)) {
    thr0->qmstat = x?BAD_REDEF:BAD_UNDEF;
    return 0;
  } else {
    if (symtb[vno].x || x)
      symtb[vno].flags |= MODIF;
    else
      symtb[vno].flags &= ~MODIF;
    qmfree(thr0, symtb[vno].x);
    symtb[vno].x = (void*) qmnew(x);
    return 1;
  }
}

/* FUNCTION CALLS AND EXPRESSION EVALUATION: */

/* retval(): collect arguments and local variables and set the return value
   for a function application. */

static int retval(THREAD *thr, EXPR **xbp)
{
  EXPR **xp;
  for (xp = xbp; xp < thr->xsp-1; xp++)
    qmfree(thr, *xp);
  *xbp = thr->xsp[-1];
  thr->xsp = xbp+1;
  return 1;
}

/* pushact(): push an activation record on the stack. */

static int pushact(THREAD *thr,
		   int fno, int *rp, int rc, OPREC *ip, long xbp,
		   int modno, int lineno)
{
  if (thr->asp-thr->ast == thr->astsz) {
    AREC **ast1;
    int n = ASTSZ/4;
    if (thr->astsz+n > thr->maxastsz) n = thr->maxastsz-thr->astsz;
    if (n <= 0 ||
	!(ast1 = (AREC**)arealloc(thr->ast, thr->astsz, n,
				  sizeof(AREC*)))) {
      thr->qmstat = AST_OVF;
      return (0);
    } else {
      thr->ast = ast1;
      thr->asp = thr->ast+thr->astsz;
      thr->astsz += n;
    }
  }
  if (!(*thr->asp = (AREC*)malloc(sizeof(AREC)))) {
    thr->qmstat = MEM_OVF;
    return (0);
  }
  (*thr->asp)->fno = fno;
  (*thr->asp)->rp = rp;
  (*thr->asp)->rc = rc;
  (*thr->asp)->ip = ip;
  (*thr->asp)->xbp = xbp;
  (*thr->asp)->modno = modno;
  (*thr->asp)->lineno = lineno;
  thr->asp++;
  return (1);
}

/* popact(): pop an activation record from the stack. */

static void popact(THREAD *thr,
		   int *fno, int **rp, int *rc, OPREC **ip, long *xbp,
		   int *modno, int *lineno)
{
  thr->asp--;
  *fno = (*thr->asp)->fno;
  *rp = (*thr->asp)->rp;
  *rc = (*thr->asp)->rc;
  *ip = (*thr->asp)->ip;
  *xbp = (*thr->asp)->xbp;
  *modno = (*thr->asp)->modno;
  *lineno = (*thr->asp)->lineno;
  free(*thr->asp);
}

/* interface to external functions */

static int dl_funcall(THREAD *thr, EXPR* (*f)(), int argc)
{
  EXPR **argv = malloc((argc+1)*sizeof(EXPR*));
  if (argv) {
    bool actmode = thr->mode;
    EXPR* x;
    memcpy(argv, thr->args, (argc+1)*sizeof(EXPR*));
    thr->mode = 1;
    x = (*f) (argc, argv);
    thr->mode = actmode;
    free(argv);
    if (x) {
      int ret;
      x->refc++;
      ret = eval(thr, x);
      qmfree(thr, x);
      return ret;
    } else
      return 0;
  } else {
    thr->qmstat = MEM_OVF;
    return 0;
  }
}

/* evalb(): evaluate a built-in (or external) function and perform environment
   variable replacement. */

static int evalb(THREAD *thr, int fno)
{
  int             n, *rp, rc;
  EXPR           *x;

  switch (fno) {
  case CONSOP:
  case PAIROP:
    /* constructor: */
    return (pushfun(thr, fno));
  case APPOP:
    /* application: */
    /* walk down the spine, count arguments: */
    for (n = 1, x = thr->xsp[-2]; n < maxargs && x->fno == APPOP; n++)
      x = x->data.args.x1;
    if (x->fno < BUILTIN && funtb[x->fno] != NULL && nargs[x->fno] == n ||
	symtb[x->fno].f && symtb[x->fno].argc == n) {
      int argc = n;
      /* set arguments: */
      thr->args[n] = NULL;
      thr->args[n - 1] = thr->xsp[-1];
      for (x = thr->xsp[-2]; n > 1; x = x->data.args.x1) {
	n--;
	thr->args[n - 1] = x->data.args.x2;
      }
      /* execute function: */
      if (x->fno < BUILTIN && (*funtb[x->fno]) (thr) ||
	  symtb[x->fno].f && dl_funcall(thr, symtb[x->fno].f, argc)) {
	thr->nredns++;
	if (thr->debug) reduction(thr, APPOP, thr->xsp-thr->xst-3);
	retval(thr, thr->xsp-3);
	return (1);
      } else
	return (0);
    } else
      return (0);
  case FALSEOP:
  case TRUEOP:
  case NILOP:
  case VOIDOP:
    /* built-in constant: */
    return (pushfun(thr, fno));
  default:
    if (symtb[fno].x)
      /* defined symbol: */
      return (push(thr, symtb[fno].x));
    else if (symtb[fno].f  && symtb[fno].argc == 0 ||
	     fno < BUILTIN && funtb[fno] != NULL &&
	     nargs[fno] == 0) {
      /* parameterless builtin or external function: */
      thr->args[0] = NULL;
      if (fno < BUILTIN && (*funtb[fno]) (thr) ||
	  symtb[fno].f && dl_funcall(thr, symtb[fno].f, 0)) {
	thr->nredns++;
	if (thr->debug) reduction(thr, fno, thr->xsp-thr->xst-1);
	return (1);
      } else
	return (0);
    } else
      /* other symbol: */
      return (0);
  }
}

/* evalb_with_frame constructs an extra stack frame for a user rule calling a
   builtin or external function, to facilitate debugging in nested calls to
   evalu */

static evalb_with_frame(THREAD *thr, 
			int fno, int _fno, int *rp, int rc, OPREC *ip,
			long xbp, int modno, int lineno)
{
  int           n;
  EXPR         *x;

  switch (fno) {
  case CONSOP:
  case PAIROP:
    /* constructor: */
    return (pushfun(thr, fno));
  case APPOP:
    /* application: */
    /* walk down the spine, count arguments: */
    for (n = 1, x = thr->xsp[-2]; n < maxargs && x->fno == APPOP; n++)
      x = x->data.args.x1;
    if (x->fno < BUILTIN && funtb[x->fno] != NULL && nargs[x->fno] == n ||
	symtb[x->fno].f && symtb[x->fno].argc == n) {
      int argc = n;
      /* set arguments: */
      thr->args[n] = NULL;
      thr->args[n - 1] = thr->xsp[-1];
      for (x = thr->xsp[-2]; n > 1; x = x->data.args.x1) {
	n--;
	thr->args[n - 1] = x->data.args.x2;
      }
      /* execute function: */
      if (!pushact(thr, _fno, rp, rc, ip, xbp, modno, lineno))
	return 0;
      if (x->fno < BUILTIN && (*funtb[x->fno]) (thr) ||
	  symtb[x->fno].f && dl_funcall(thr, symtb[x->fno].f, argc)) {
	free(*--thr->asp);
	thr->nredns++;
	if (thr->debug) reduction(thr, APPOP, thr->xsp-thr->xst-3);
	retval(thr, thr->xsp-3);
	return (1);
      } else {
	free(*--thr->asp);
	return (0);
      }
    } else
      return (0);
  case FALSEOP:
  case TRUEOP:
  case NILOP:
  case VOIDOP:
    /* built-in constant: */
    return (pushfun(thr, fno));
  default:
    if (symtb[fno].x)
      /* defined symbol: */
      return (push(thr, symtb[fno].x));
    else if (symtb[fno].f  && symtb[fno].argc == 0 ||
	     fno < BUILTIN && funtb[fno] != NULL &&
	     nargs[fno] == 0) {
      /* parameterless builtin or external function: */
      thr->args[0] = NULL;
      if (!pushact(thr, _fno, rp, rc, ip, xbp, modno, lineno))
	return 0;
      if (fno < BUILTIN && (*funtb[fno]) (thr) ||
	  symtb[fno].f && dl_funcall(thr, symtb[fno].f, 0)) {
	free(*--thr->asp);
	thr->nredns++;
	if (thr->debug) reduction(thr, fno, thr->xsp-thr->xst-1);
	return (1);
      } else {
	free(*--thr->asp);
	return (0);
      }	
    } else
      /* other symbol: */
      return (0);
  }
}

/*

evalu(): evaluate a user-defined function.

Here you are looking at the very core of the interpreter. evalu() tries to
execute the code for a user-defined rule. If no rule matches, a constructor
term is returned as the default value. If during the execution an error
condition arises (qmstat != OK) the execution is aborted immediately, and
evalu() returns 0; otherwise it returns 1. Exceptions are handled internally
if a handler (catch) is available in the current context.

During execution of a function, the configuration of the Q machine
is described by the following items:

- rp, the rule pointer, pointing to the offset of the currently executing
  rule;

- rc, the count of rules remaining to be processed (including the current
  rule);

- ip, the instruction pointer;

- xbp, the base pointer which points to the first argument of the
  executing rule on the stack (this one actually is an index into the stack
  since the stack may be reallocated during evaluation);

- mode and start, which are used to keep track of special forms.

Besides this, source line information from !INFO instructions is maintained in
the modno and lineno variables.

All state variables except mode are local to evalu in order to make evalu
reentrant. (The mode variable must be made global since it is needed by other
operations like consexpr() and qmunquote(). However, the only way to reenter
evalu() while mode is set is through eval() which takes care of restoring the
original mode value when it finishes. In fact, this only happens through
invokations of the unquote (`) and force (~) builtins, since these are the
only operations which are executed while mode is set.)

Tail calls (calls immediately preceding a !RET instruction) are eliminated
automatically by setting the configuration for the new function and returning
the return value of the tail call as the return value of the original call. In
such a case, no additional stack space is required.

*/

/* macros for invoking the debugger */

#define dbg_rule		\
if (thr->debug)\
     stoplevel = rule(thr, fno, xbp, *rp, ip, level, modno, lineno)

#define dbg_binding(failed,m,offs) \
if (thr->debug)\
     binding(thr, failed, m, offs)

#define dbg_reduction 	\
if (thr->debug || level<stoplevel) \
{ set_debug(thr, 1); stoplevel = 0; reduction(thr, fno, xbp); }

#define dbg_default_reduction \
if (thr->debug || level<stoplevel)\
{ set_debug(thr, 1); stoplevel = 0; default_reduction(thr, fno, xbp); }

#define dbg_tail_reduction	\
if (thr->debug)\
{ set_debug(thr, 1); stoplevel = 0; tail_reduction(thr, fno, xbp, ip->opcode); }

static int evalu(THREAD *thr, int fno)
{
  int	       *rp, *rp1;
  int           rc, rc1;
  OPREC	       *ip;
  long          xbp, start;
  unsigned	level = 0, stoplevel = 0, siglevel = 0;
  int		modno = NONE, lineno = NONE;
  long		sig;

  thr->qmstat = OK;

  if (!match(thr, fno, &rp, &rc))

    /* no matching rule; push default value: */
    return pushfun(thr, fno);

  else {

    /*
     * at this point, we are dealing with a user-defined rule and have rp and
     * rc set to the corresponding rule pointer and counter
     */

__start:				/* start a new function call */

    ip = codesp + *rp;
    if (fno == APPOP)
      xbp = thr->xsp - thr->xst - 2;
    else
      xbp = thr->xsp - thr->xst;
    thr->mode = 0;

__debug:				/* catch debug flag */

#ifdef USE_THREADS
    if (nused > 1) {
      /* allow for cancellation */
      if (thr > thr0) pthread_testcancel();
      /* let another thread hold the candle */
      pthread_mutex_unlock(&global_mutex);
      thr->released = 1;
      if (init_thr) {
	pthread_mutex_lock(&init_mutex);
	while (init_thr && thr != init_thr)
	  pthread_cond_wait(&init_cond, &init_mutex);
	pthread_mutex_unlock(&init_mutex);
      }
      pthread_mutex_lock(&global_mutex);
      thr->released = 0;
      if (thr > thr0) pthread_testcancel();
    }
#endif

    /* make sure that debugging information is available at the beginning of a
       rule: */
    if (modno == NONE && ip->opcode == INFOP) {
      modno = ip->opargs.info.modno;
      lineno = ip->opargs.info.lineno;
      ip++;
    }

    dbg_rule;

__loop:				/* execution cycle */

    /* catch signals, BREAK, QUIT and errors: */

    if (thr->qmstat != OK) return 0;
    if (thr->sigpend) {
      /* catch has caught a signal and now we're processing the handler; make
	 sure we don't get interrupted again until it's done */
      siglevel = (level==0)?0:(level+1);
      thr->sigblk = 1;
      thr->sigpend = 0;
    } else if (level < siglevel)
      /* we're done with the signal handler, reenable signal processing */
      thr->sigblk = 0;
    if ((sig = get_sig(thr))) {
      /* we've received a signal, throw an exception */
      if (pushfun(thr, SYSERROP) && pushint(thr, -sig) &&
	  pushfun(thr, APPOP))
	thr->qmstat = XCEPT;
      if (thr->brkdbg) {
	set_debug(thr, 1);
	goto __debug;
      } else {
	thr->sigpend = 1;
	return 0;
      }
    } else if (thr->brkflag) {
      /* thread-local break (break function) */
      thr->brkflag = 0;
      set_debug(thr, 1);
      thr->qmstat = BREAK;
      goto __debug;
    } else if (thr > thr0)
      ;
    else if (quitflag) {
      /* quit (quit function or termination signal) */
      thr->qmstat = QUIT;
      return 0;
    } else if (brkflag) {
      /* global break (Ctl-C) */
      brkflag = 0;
      thr->qmstat = BREAK;
      if (thr->brkdbg) {
	set_debug(thr, 1);
	goto __debug;
      } else
	return 0;
    }

    switch (ip->opcode) {

    case INFOP:
      /* process debugging information */
      modno = ip->opargs.info.modno;
      lineno = ip->opargs.info.lineno;
      ip++;
      break;

    case RETOP:
      /* return instruction: set the return value and return */
      thr->nredns++; dbg_reduction;
      retval(thr, thr->xst+xbp);
      if (level == 0)
	return 1;
      else {
	popact(thr, &fno, &rp, &rc, &ip, &xbp, &modno, &lineno);
	level--;
	dbg_rule;
	goto check;
      }
      break;

    case POPOP:
      /* pop instruction: pop value from stack */
      qmfree(thr, *--thr->xsp);
      ip++;
      break;

    case LVALOP:
      /* lval instruction: push an lval, evaluate if necessary: */
      if (!pushlval(thr, fno, xbp, ip->opargs.lval.offs, ip->opargs.lval.plen,
		    ip->opargs.lval.p))
	return 0;
      else if (!thr->mode && thr->xsp[-1]->red) {
	/* make a new stack frame */
	if (eval_with_frame(thr, thr->xsp[-1],
			    fno, rp, rc, ip, xbp, modno, lineno)) {
	  qmfree(thr, thr->xsp[-2]);
	  thr->xsp[-2] = thr->xsp[-1];
	  thr->xsp--;
	} else if (thr->qmstat == XCEPT_FAIL) {
	  thr->qmstat = OK;
	  goto __next_rule;
	} else
	  return 0;
      }

check:                          /* check mode flag and advance to
				   next instruction */
	    
      /* this code is reached from cases RETOP, LVALOP, QUERYOP, MATCHOP and
	 the default case */
	    
      if (!thr->mode && ip->mode) {
	EXPR *x = thr->xsp[-1];
	unsigned long argv = (x->fno==APPOP)?x->data.args.argv:
	  symtb[x->fno].argv;
	if (argv & 1) {
	  thr->mode = 1;
	  start = thr->xsp-thr->xst;
	}
      }
      ip++;
      break;

    case QUERYOP:
      /* query instruction: */
      if (thr->xsp[-1]->fno == FALSEOP) {
	/* abort the current rule */
      __next_rule:
	/* clean up the stack: */
	while (thr->xsp-thr->xst > (fno == APPOP?xbp+2:xbp))
	  qmfree(thr, *--thr->xsp);
	if (--rc > 0) {
	  /* try the next rule: */
	  ip = codesp + *++rp;
	  modno = lineno = NONE;
	  goto __debug;
	} else {
	  /* push the default value and return: */
	  dbg_default_reduction;
	  if (!pushfun(thr, fno))
	    return 0;
	  else if (level == 0)
	    return 1;
	  else {
	    popact(thr, &fno, &rp, &rc, &ip, &xbp, &modno, &lineno);
	    level--;
	    dbg_rule;
	    goto check;
	  }
	}
      } else if (thr->xsp[-1]->fno == TRUEOP) {
	/* proceed */
	qmfree(thr, *--thr->xsp);
	ip++;
	dbg_rule;
      } else if (!have_mark(thr)) {
	/* error in conditional */
	int _debug = thr->debug;
	thr->qmstat = COND_ERR;
	if (thr->brkdbg) {
	  set_debug(thr, 1);
	  dbg_rule;
	  if (thr->qmstat == OK) thr->qmstat = HALT;
	}
	set_debug(thr, _debug);
	return 0;
      } else {
	/* error in conditional, will be handled with catch */
	thr->qmstat = COND_ERR;
	return 0;
      }
      break;

    case MATCHOP:
      /* pattern binding: */
      if (!matchp(thr, matchtb[ip->opargs.m])) {
	dbg_binding(1, ip->opargs.m, ((thr->xsp-thr->xst)-xbp-((fno==APPOP)?2:0)));
	goto __next_rule;
      } else
	dbg_binding(0, ip->opargs.m, ((thr->xsp-thr->xst)-xbp-((fno==APPOP)?2:0)));
      ip++;
      dbg_rule;
      break;

    case INTVALOP:
      /* push an int value: */
      {
	mpz_t z;
	if (!getint(z, ip->opargs.iv.len, ip->opargs.iv.l)) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	} else if (!pushmpz(thr, z))
	  return 0;
      }
      ip++;
      break;

    case FLOATVALOP:
      /* push a float value: */
      if (!pushfloat(thr, ip->opargs.fv))
	return 0;
      ip++;
      break;
	    
    case STRVALOP:
      /* push a string value: */
      {
	char           *s;
	if ((s = strdup(strsp + ip->opargs.sv)) ==
	    NULL) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	} else if (!pushstr(thr, s))
	  return 0;
      }
      ip++;
      break;

    case APPOP:
      if (thr->mode && thr->xsp-thr->xst-1 == start)
	thr->mode = 0;
      /* falls through to default case */

    default:
      /* function call: */
      if (thr->mode && !(ip->opcode == APPOP &&
		    (thr->xsp[-2]->fno == UNQUOTEOP ||
		     thr->xsp[-2]->fno == FORCEOP))) {
	/* processing special form */
	if (!pushfun(thr, ip->opcode))
	  return 0;
	/* no need to check, mode is enabled already */
	ip++;
      } else if (evalb_with_frame(thr, ip->opcode,
				  fno, rp, rc, ip, xbp, modno, lineno))
	/* built-in function */
	goto check;
      else if (thr->qmstat == XCEPT_CATCH) {
	long xp;
	EXPR *h, *x = NULL;
	/* handle an exception */
	get_mark(thr, &xp, &h);
	pop_mark(thr);
	/* save throw argument */
	if (thr->qmstat_save == XCEPT) x = *--thr->xsp;
	/* rewind the stack */
	while (thr->xsp-thr->xst > xp) qmfree(thr, *--thr->xsp);
	/* evaluate the handler (special argument) */
	thr->qmstat = OK;
	/* we need an extra stack frame here */
	if (!eval_with_frame(thr, h, fno, rp, rc, ip, xbp, modno, lineno)) {
	  thr->sigpend = 0;
	  if (x) qmfree(thr, x);
	  if (thr->qmstat == XCEPT_FAIL) {
	    /* fail in handler; let the current rule fail (see also case
	       qmstat == XCEPT_FAIL below) */
	    thr->qmstat = OK;
	    goto __next_rule;
	  } else
	    /* exception in handler will be handled by caller */
	    return 0;
	}
	/* pop catch arguments, replace with handler and exception */
	retval(thr, thr->xsp-3);
	if (thr->qmstat_save != XCEPT) {
	  /* exception comes from interpreter, push error code */
	  if (!pushfun(thr, SYSERROP) || !pushint(thr, thr->qmstat_save) ||
	      !pushfun(thr, APPOP)) return 0;
	} else
	  /* exception comes from throw, push argument saved above */
	  *(thr->xsp++) = x;
	/* now handler and exception are on the stack and in the next loop
	   cycle we will reexecute the current (apply) instruction from the
	   original catch call */
	break;
      } else if (thr->qmstat == XCEPT_FAIL) {
	/* pretend that the current rule has failed */
	thr->qmstat = OK;
	/* proceed as with failed qualifier */
	goto __next_rule;
      } else if (!have_mark(thr) &&
		 (thr->qmstat == EXT_ERR || thr->qmstat == XCEPT)) {
	int _debug = thr->debug;
	if (thr->brkdbg) {
	  set_debug(thr, 1);
	  dbg_rule;
	  if (thr->qmstat == OK) thr->qmstat = HALT;
	}
	set_debug(thr, _debug);
	return 0;
      } else if (thr->qmstat != OK)
	/* error condition in builtin */
	return 0;
      else if ((ip+1)->opcode == RETOP) {
	/* tail call */
	if (match(thr, ip->opcode, &rp1, &rc1)) {
	  EXPR **xp, **top = thr->xsp;
	  thr->nredns++; dbg_tail_reduction;
	  /* set up for the new call: */
	  rp = rp1;
	  rc = rc1;
	  fno = ip->opcode;
	  ip = codesp + *rp;
	  if (fno == APPOP) top -= 2;
	  for (xp = thr->xst+xbp; xp < top; xp++)
	    qmfree(thr, *xp);
	  if (fno == APPOP) {
	    thr->xst[xbp] = thr->xsp[-2];
	    thr->xst[xbp+1] = thr->xsp[-1];
	    thr->xsp = thr->xst + xbp + 2;
	  } else
	    thr->xsp = thr->xst + xbp;
	  modno = lineno = NONE;
	  goto __debug;
	} else {
	  /* push the default value: */
	  if (!pushfun(thr, ip->opcode)) return 0;
	  /* no need to check, since RETOP is the next instruction */
	  ip++;
	}
      } else if (!match(thr, ip->opcode, &rp1, &rc1)) {
	/* no matching rule; push default value: */
	if (!pushfun(thr, ip->opcode)) return 0;
	goto check;
      } else {
	/* user-defined rule */
	if (!pushact(thr, fno, rp, rc, ip, xbp, modno, lineno))
	  return 0;
	else
	  level++;
	fno = ip->opcode;
	rp = rp1;
	rc = rc1;
	modno = lineno = NONE;
	goto __start;
      }
      break;
    }		/* switch(ip->opcode) */
	  
    goto __loop;

  }
}

/* evaldef() is a specialized version of evalu() used to execute the
   initialization code */

int evaldef1(int offs)
{
  OPREC	       *ip;
  long	        start;
  long		xbp = thr0->xsp - thr0->xst;

  thr0->mode = 0;
  for (ip = codesp + offs; ; ip++) {
    switch (ip->opcode) {
		
    case INFOP:
      /* ignore */
      break;
      
    case RETOP:
      /* pop value from stack and return: */
      if (thr0->xsp-thr0->xst > xbp) qmfree(thr0, *--thr0->xsp);
      return 1;

    case POPOP:
      /* pop value from stack */
      qmfree(thr0, *--thr0->xsp);
      break;
			
    case INTVALOP:
      /* push an int value: */
      {
	mpz_t z;
	if (!getint(z, ip->opargs.iv.len, ip->opargs.iv.l)) {
	  thr0->qmstat = MEM_OVF;
	  return 0;
	} else if (!pushmpz(thr0, z))
	  return (0);
      }
      break;

    case FLOATVALOP:
      /* push a float value: */
      if (!pushfloat(thr0, ip->opargs.fv))
	return (0);
      break;
      
    case STRVALOP:
      /* push a string value: */
      {
	char           *s;
	if ((s = strdup(strsp + ip->opargs.sv)) ==
	    NULL) {
	  thr0->qmstat = MEM_OVF;
	  return (0);
	} else if (!pushstr(thr0, s))
	  return (0);
      }
      break;

    case APPOP:
      if (thr0->mode && thr0->xsp-thr0->xst-1 == start)
	thr0->mode = 0;
      /* falls through to default case */

    default:
      /* function call: */
      if (thr0->mode && !(ip->opcode == APPOP &&
			  (thr0->xsp[-2]->fno == UNQUOTEOP ||
			   thr0->xsp[-2]->fno == FORCEOP))) {
	/* processing special form */
	if (!pushfun(thr0, ip->opcode))
	  return (0);
      } else if (evalf(thr0, ip->opcode)) {
	if (!thr0->mode && ip->mode) {
	  EXPR *x = thr0->xsp[-1];
	  unsigned long argv = (x->fno==APPOP)?x->data.args.argv:
	    symtb[x->fno].argv;
	  if (argv & 1) {
	    thr0->mode = 1;
	    start = thr0->xsp-thr0->xst;
	  }
	}
	break;
      } else
	return 0;
      
    }
  }
}

int evaldef(int offs)
{
  bool mode = thr0->mode;
  int res;
  lock_debug(thr0);
  res = evaldef1(offs);
  unlock_debug(thr0);
  thr0->mode = mode;
  return res;
}

/* evalf(): evaluate a function call */

static int evalf(THREAD *thr, int fno)
{
  bool _mode = thr->mode;
  int res;
  long sig;
  if (evalb(thr, fno)) {
    /* builtin: */
    if ((sig = get_sig(thr))) {
      if (pushfun(thr, SYSERROP) && pushint(thr, -sig) &&
	  pushfun(thr, APPOP))
	thr->qmstat = XCEPT;
      res = 0;
    } else
      res = 1;
  } else if (thr->qmstat == XCEPT_CATCH) {
    long xp;
    EXPR *h, *x = NULL;
    /* handle an exception */
    get_mark(thr, &xp, &h);
    pop_mark(thr);
    if (thr->qmstat_save == XCEPT) x = *--thr->xsp;
    while (thr->xsp-thr->xst > xp) qmfree(thr, *--thr->xsp);
    thr->qmstat = OK;
    if (!eval(thr, h)) {
      thr->sigpend = 0;
      if (x) qmfree(thr, x);
      return 0;
    }
    retval(thr, thr->xsp-3);
    if (thr->qmstat_save != XCEPT) {
      if (!pushfun(thr, SYSERROP) || !pushint(thr, thr->qmstat_save) ||
	  !pushfun(thr, APPOP)) return 0;
    } else
      *(thr->xsp++) = x;
    res = evalu(thr, APPOP);
  } else if (thr->qmstat != OK)
    /* error condition in builtin: */
    res = 0;
  else
    /* check for user-defined function: */
    res = evalu(thr, fno);
  thr->mode = _mode;
  return res;
}

/* eval(): evaluate an expression and push the result onto the stack. */

static int eval_internal(THREAD *thr, EXPR *x)
{
  if (!x->red)
    return (push(thr, x));
  else switch (x->fno) {
  case CONSOP:
  case PAIROP: {
    /* we do this non-recursively, to prevent stack overflows */
    EXPR *h = x->data.args.x1, *t = x->data.args.x2;
    int n = 0, res;
    while ((res = eval_internal(thr, h)) && ++n && t->fno == x->fno)
      h = t->data.args.x1, t = t->data.args.x2;
    if (!res || !eval_internal(thr, t)) return 0;
    while (n > 0 && (res = pushfun(thr, x->fno))) n--;
    return res;
  }
  case APPOP:
    /* function application: */
    if (!eval_internal(thr, x->data.args.x1))
      return 0;
    if (!thr->mode) {
      EXPR *x1 = thr->xsp[-1];
      unsigned long argv = (x1->fno==APPOP)?x1->data.args.argv:
	symtb[x1->fno].argv;
      if (argv & 1) {
	int res;
	thr->mode = 1;
	res = eval_internal(thr, x->data.args.x2);
	thr->mode = 0;
	if (!res) return 0;
      } else if (!eval_internal(thr, x->data.args.x2))
	return 0;
    } else if (!eval_internal(thr, x->data.args.x2))
      return 0;
    if (thr->mode && thr->xsp[-2]->fno != UNQUOTEOP &&
	thr->xsp[-2]->fno != FORCEOP)
      return pushfun(thr, APPOP);
    else
      return evalf(thr, APPOP);
  default:
    if (x->fno == VECTOP) {
      int i, n = x->data.vect.n;
      EXPR **xv = n?malloc(n*sizeof(EXPR*)):NULL;
      if (n && !xv) {
	thr->qmstat = MEM_OVF;
	return 0;
      }
      for (i = 0; i < n; i++)
	if (!eval_internal(thr, x->data.vect.xv[i])) {
	  int j;
	  for (j = 0; j < i; j++)
	    qmfree(thr, xv[i]);
	  free(xv);
	  return 0;
	} else
	  xv[i] = *--thr->xsp;
      return (pushvect(thr, n, xv));
    } else if (thr->mode)
      return pushfun(thr, x->fno);
    else
      return evalf(thr, x->fno);
  }
}

int eval(THREAD *thr, EXPR *x)
{
  bool _mode = thr->mode;
  int res;
  thr->mode = 0;
  if (thr->level == 0) lock_debug(thr);
  thr->level++;
  res = eval_internal(thr, x);
  thr->level--;
  if (thr->level == 0) unlock_debug(thr);
  thr->mode = _mode;
  return res;
}

/* eval with extra stack frame */

static int eval_with_frame(THREAD *thr, 
			   EXPR *x, int fno, int *rp, int rc, OPREC *ip,
			   long xbp, int modno, int lineno)
{
  int res;
  if (!pushact(thr, fno, rp, rc, ip, xbp, modno, lineno))
    return 0;
  res = eval(thr, x);
  free(*--thr->asp);
  return res;
}

/* The following routines implement a simple debugging facility which
   allows to trace the reductions performed by the Q machine. */

static int printp(THREAD *thr, int fno, long xbp);
static int printc(THREAD *thr, int fno, long xbp, int addr, OPREC *ip0);
static int buildvartb(THREAD *thr, int addr);
static int lastoffs(THREAD *thr, int addr, OPREC *ip0);
static int listvars(THREAD *thr, int offs);
static int savevars(THREAD *thr, int fno, long xbp, int offs);
static void restorevars(THREAD *thr);
static int getvar(THREAD *thr, char *name, int offs);
static void set_print_params(void);
static void reset_print_params(void);

#define MAXITEMS 6	/* maximum number of stacked rules to print */
#define MAXLEVEL 2	/* maximum display depth of expressions */
#define MAXLIST 3	/* maximum number of list/tuple items to print */
#define MAXCHARS 33	/* maximum string length to print */

static int maxlevel0 = MAXLEVEL, maxlist0 = MAXLIST, maxchars0 = MAXCHARS;
static int maxitems0 = MAXITEMS;

static void check_thread(THREAD *thr)
{
  static int act_thrno = -1;
  int thrno = thr-thr0;
  if (thrno != act_thrno && (act_thrno >= 0 || thrno > 0))
    printf("[switching to thread #%d]\n", thrno);
  act_thrno = thrno;
}

static void help(void)
{
	printf(
"\nCommands:\n\n\
? or help	print this list\n\
options		print list of available options\n\
\n\
? EXPR		evaluate expression\n\
. [OPTION ...]	print rule or change options settings\n\
l [K] [N]	list source lines (N lines, K = offset)\n\
p [N]		print rule stack (N rules)\n\
m		print memory usage\n\
v		list local variables\n\
\n\
u [N], d [N]	move up or down N stack levels\n\
t, b		move to the top or bottom of the rule stack\n\
\n\
<CR>		step into the current rule\n\
n		step over the current rule\n\
c		continue\n\
h		halt\n\
q or <EOF>	quit\n\
\n");
}

static void help_options(void)
{
	printf(
"\nOption Commands:\n\n\
. options	print current settings\n\
\n\
. pathnames=y|n	print long script pathnames yes/no\n\
. detail=N	set detail to N expression levels\n\
. maxitems=N	set number of list or tuple items to print\n\
. maxchars=N	set number of string characters to print\n\
. maxstack=N	set number of stack levels to print\n\
\n\
The numeric argument N can also be zero or `all' to specify that all\n\
corresponding items are to be printed.\n\
\n\
Multiple options, separated with whitespace, can be specified in a single\n\
`.' command. Single options must not contain any whitespace.\n\
\n");
}

static void lock_debug(THREAD *thr)
{
#ifdef USE_THREADS
  if (thr->debug && !thr->debug_lock) {
    suspend_input();
    thr->debug_lock = 1;
    debug_lock++;
  }
#endif
}

static void unlock_debug(THREAD *thr)
{
#ifdef USE_THREADS
  if (thr->debug_lock) {
    thr->debug_lock = 0;
    if (!--debug_lock) resume_input();
  }
#endif
}

static void set_debug(THREAD *thr, bool debug)
{
  if (thr->debug != debug) {
    thr->debug = debug;
#ifdef USE_THREADS
    if (debug)
      lock_debug(thr);
    else
      unlock_debug(thr);
#endif
  }
}

static void debug_thread(THREAD *thr, char *msg)
{
  char s[MAXSTRLEN];
  push_sigint(SIG_IGN);
  sprintf(s, msg, thr-thr0);
  printf("[%s]\n", s);
  pop_sigint();
}

/* temporarily increase the maximum stack size for debugging */

static void dbg_stack(THREAD *thr, int n)
{
  if (thr->maxxstsz > 0) thr->maxxstsz += n;
}

static void end_stack(THREAD *thr, int n)
{
  if (thr->maxxstsz > 0) thr->maxxstsz -= n;
}

static void print_source(int modno, int lineno, int lines)
{
  FILE *fp;
  if (modno != NONE && lineno != NONE &&
      (fp = fopen(strsp+fnametb[modno], "r"))) {
    int c, actline = 1;
    while (actline < lineno && (c = getc(fp)) != EOF)
      if (c == '\n')
	actline++;
    while (actline < lineno+lines && (c = getc(fp)) != EOF) {
      putchar(c);
      if (c == '\n')
	actline++;
    }
    fclose(fp);
  } else
    error("Source file not found\n");
}

static void print_rule(THREAD *thr, int fno, long xbp, int addr, OPREC *ip,
		       int modno, int lineno)
{
  char fname[MAXSTRLEN];
  if (modno != NONE && lineno != NONE) {
    if (debug_long)
      strcpy(fname, strsp+fnametb[modno]);
    else
      basename(fname, strsp+fnametb[modno], 0);
    printf("%s, line %d: ", fname, lineno);
  }
  dbg_stack(thr, MAXDEPTH+10);
  if (printp(thr, fno, xbp)) {
    printf("  ==>  ");
    printc(thr, fno, xbp, addr, ip);
  }
  end_stack(thr, MAXDEPTH+10);
  putchar('\n');
}

static void print_stacked_rule(THREAD *thr, int base)
{
  printf("%3d>  ", thr->asp-thr->ast-base);
  print_rule(thr, 
	     thr->asp[-base]->fno, thr->asp[-base]->xbp, *thr->asp[-base]->rp,
	     thr->asp[-base]->ip,
	     thr->asp[-base]->modno, thr->asp[-base]->lineno);
}

static void print_stack(THREAD *thr, 
			int base, int fno, long xbp, int addr, OPREC *ip,
			int modno, int lineno, int maxitems)
{
  AREC	      **abp, **asp1 = thr->asp - base;
  if (maxitems > 0 && asp1-thr->ast > maxitems-1)
    abp = asp1 - maxitems + 1;
  else
    abp = thr->ast;
  if (base) {
    fno = asp1[0]->fno; xbp = asp1[0]->xbp; addr = *asp1[0]->rp;
    modno = asp1[0]->modno; lineno = asp1[0]->lineno;
  }
  printf("stack size: %d\n", thr->asp-thr->ast+1);
  for (; abp < asp1; abp++) {
    printf("%3d>  ", abp-thr->ast);
    print_rule(thr, (*abp)->fno, (*abp)->xbp, *(*abp)->rp, (*abp)->ip,
	       (*abp)->modno, (*abp)->lineno);
    if (thr->qmstat != OK) return;
  }
  printf("%3d>  ", thr->asp-thr->ast-base);
  print_rule(thr, fno, xbp, addr, ip, modno, lineno);
}

static char *debug_parse_opt(char *s, char *t)
{
  int l = strlen(t);
  if (strncmp(s, t, l) == 0)
    return s+l;
  else
    return NULL;
}

char *debug_get_opts(char *s)
{
  char opt[MAXSTRLEN];
  sprintf(opt, "pathnames=%c", debug_long?'y':'n');
  strcpy(s, opt);
  if (maxlevel0)
    sprintf(opt, " detail=%d", maxlevel0);
  else
    sprintf(opt, " detail=all");
  strcat(s, opt);
  if (maxlist0)
    sprintf(opt, " maxitems=%d", maxlist0);
  else
    sprintf(opt, " maxitems=all");
  strcat(s, opt);
  if (maxchars0)
    sprintf(opt, " maxchars=%d", maxchars0);
  else
    sprintf(opt, " maxchars=all");
  strcat(s, opt);
  if (maxitems0)
    sprintf(opt, " maxstack=%d", maxitems0);
  else
    sprintf(opt, " maxstack=all");
  strcat(s, opt);
  return s;
}

int debug_parse_opts(char *s)
{
  int n;
  char *t, *arg;
  for (; *s && isspace(*s); s++) ;
  for (t = strtok(s, " \t\n\r\f"); t; t = strtok(NULL, " \t\n\r\f"))
    if (*t) {
      if (strcmp(t,"options") == 0) {
	char opts[MAXSTRLEN];
	printf("%s\n", debug_get_opts(opts));
      } else if ((arg = debug_parse_opt(t, "pathnames=")))
	if (strlen(arg)==1 && (*arg=='y'||*arg=='n'))
	  debug_long = *arg=='y';
	else
	  return 0;
      else if ((arg = debug_parse_opt(t, "detail=")))
	if (strcmp(arg,"all") == 0)
	  maxlevel0 = 0;
	else if (isdigit(*arg) && (n = atoi(arg)) >= 0)
	  maxlevel0 = n;
	else
	  return 0;
      else if ((arg = debug_parse_opt(t, "maxitems=")))
	if (strcmp(arg,"all") == 0)
	  maxlist0 = 0;
	else if (isdigit(*arg) && (n = atoi(arg)) >= 0)
	  maxlist0 = n;
	else
	  return 0;
      else if ((arg = debug_parse_opt(t, "maxchars=")))
	if (strcmp(arg,"all") == 0)
	  maxchars0 = 0;
	else if (isdigit(*arg) && (n = atoi(arg)) >= 0)
	  maxchars0 = n;
	else
	  return 0;
      else if ((arg = debug_parse_opt(t, "maxstack=")))
	if (strcmp(arg,"all") == 0)
	  maxitems0 = 0;
	else if (isdigit(*arg) && (n = atoi(arg)) >= 0)
	  maxitems0 = n;
	else
	  return 0;
      else
	return 0;
    }
  return 1;
}

static int rule(THREAD *thr, 
		int fno, long xbp, int addr, OPREC *ip, unsigned level,
		int modno, int lineno)
{
  char          s[MAXSTRLEN];
  int		stoplevel = 0;
  static int	debug_init = 1;
  static int	nlines = 1, noffs = 0;
  int		base = 0;
  
  /* further breaks are masked while in the debugger: */
#ifdef USE_THREADS
  acquire_tty();
#endif
  push_sigint(SIG_IGN); if (thr == thr0) brkflag = 0;
  check_thread(thr);
  if (thr->qmstat != OK)
    error(qmmsg[thr->qmstat]);
  thr->qmstat = OK;
  printf("%3d>  ", thr->asp-thr->ast-base);
  print_rule(thr, fno, xbp, addr, ip, modno, lineno);
  /* check whether running interactively: */
  if (!iflag && (!isatty(fileno(stdin)) || !isatty(fileno(stdout))))
    goto exit;
  /* clear EOF on stdin if set: */
  clearerr(stdin);

 command:			/* command loop: */

  if (thr->qmstat != OK)
    goto exit;
  if (debug_init) {
    printf("(type ? for help)\n");
    debug_init = 0;
  }
  printf(prompt3); fflush(stdout);
  if (fgets(s, MAXSTRLEN, stdin) == NULL) {
    thr->qmstat = QUIT;
    if (iflag) putchar('\n');
    goto exit;
  } else {
    char t[MAXSTRLEN];
    int i, l = strlen(s);
    if (l > 0 && s[l-1] == '\n')
      s[l-1] = '\0', l--;
    while (l>0 && isspace(s[l-1]))
      s[l-1] = '\0', l--;
    for (i=0; i<l && isspace(s[i]); i++) ;
    strcpy(t, s+i);
    strcpy(s, t);
  }
  
  if (strcmp(s, "help") == 0)
    help();
  else if (strcmp(s, "options") == 0)
    help_options();
  else if (strlen(s) == 0)
    goto exit;
  else if (!strchr("?lp.ud", s[0]) && strlen(s) > 1)
    error(qmmsg[INVALID_COMMAND]);
  else
    switch (s[0]) {
    case '?': {
      char *t;
      for (t = s+1; *t && isspace(*t); t++) ;
      if (*t) {
	long xp = thr->xsp - thr->xst;
	long ap = thr->asp-thr->ast;
	bool _brkdbg = thr->brkdbg, _debug = thr->debug;
	pop_sigint();
#ifdef USE_THREADS
	release_tty();
#endif
	thr->brkdbg = thr->debug = 0;
	if (sparsex(t)) {
	  EXPR *x = *--thr->xsp;
	  int ret;
	  long _fno = base?thr->asp[-base]->fno:fno;
	  long _xbp = base?thr->asp[-base]->xbp:xbp;
	  int _addr = base?(*thr->asp[-base]->rp):addr;
	  OPREC *_ip = base?thr->asp[-base]->ip:ip;
	  int offs = lastoffs(thr, _addr, _ip);

	  buildvartb(thr, _addr);
	  savevars(thr, _fno, _xbp, offs);
	  ret = eval(thr, x);
	  qmfree(thr, x);
	  check_thread(thr);
	  if (ret) {
	    printx(thr->xsp[-1]);
	    printf("\n");
	    qmfree(thr, *--thr->xsp);
	  }
	  restorevars(thr);
	}
	if (thr->qmstat != OK) {
	  error(qmmsg[thr->qmstat]);
	  if (thr->qmstat == XCEPT && thr->xsp-thr->xst > xp) {
	    printx(thr->xsp[-1]); printf("\n");
	  }
	}
	thr->qmstat = OK;
	while (thr->xsp-thr->xst > xp) qmfree(thr, *--thr->xsp);
	while (thr->asp-thr->ast > ap) free(*--thr->asp);
	thr->brkdbg = _brkdbg; thr->debug = _debug;
#ifdef USE_THREADS
	acquire_tty();
#endif
	push_sigint(SIG_IGN);
	fflush(stderr); fflush(stdout);
	clearerr(stdin);
      } else
	help();
      break;
    }
    case '.':
      if (s[1] != 0) {
	if (!debug_parse_opts(s+1))
	  error(qmmsg[INVALID_COMMAND]);
	else if (gflag) {
	  char opts[MAXSTRLEN];
	  /* prepare to fork */
	  pop_sigint();
#ifdef USE_THREADS
	  release_tty();
#endif
	  gcmd_s("q-debug-options-cmd", debug_get_opts(opts));
	  /* after fork */
#ifdef USE_THREADS
	  acquire_tty();
#endif
	  push_sigint(SIG_IGN);
	}
	break;
      }
    reprint:
      if (base)
	print_stacked_rule(thr, base);
      else {
	printf("%3d>  ", thr->asp-thr->ast-base);
	print_rule(thr, fno, xbp, addr, ip, modno, lineno);
      }
      break;
    case 'v': {
#if 0
      char *t;
      long _xbp = base?thr->asp[-base]->xbp:xbp;
#endif
      int _addr = base?(*thr->asp[-base]->rp):addr;
      OPREC *_ip = base?thr->asp[-base]->ip:ip;
      int offs = lastoffs(thr, _addr, _ip);

      buildvartb(thr, _addr);
#if 0
      for (t = s+1; *t && isspace(*t); t++) ;
      if (*t) {
	for (t = strtok(t, " \t\n\r\f"); t; t = strtok(NULL, " \t\n\r\f"))
	  if (*t) {
	    int v = getvar(thr, t, offs);
	    if (v != NONE) {
	      if (pushlval(thr, fno, _xbp, thr->vartb[v].offs,
			   thr->vartb[v].plen,
			   thr->vartb[v].p)) {
		set_print_params(), printx(thr->xsp[-1]), reset_print_params();
		printf("\n");
		qmfree(thr, *--thr->xsp);
	      }
	    } else {
	      char msg[MAXSTRLEN];
	      sprintf(msg, qmmsg[BAD_VAR], t);
	      error(msg);
	    }
	  }
      } else
#endif
	listvars(thr, offs);
      break;
    }
    case 'm': {
      int nstack = thr->xsp - thr->xst;
      unsigned long nheap;
      unsigned long mexprs;
      
      nheap = ((unsigned long)(xnblks-1))*XBLKSZ+(xheap-xblk->x);
      mexprs = nheap-fexprs;
      if (stackmax > 0)
	printf("stack: %d cells (%d used) out of %d\n",
	       thr->xstsz, nstack, stackmax);
      else
	printf("stack: %d cells (%d used)\n",
	       thr->xstsz, nstack);
      if (memmax > 0)
	printf("heap: %d cells (%d used, %d free) out of %d\n",
	       xnblks*XBLKSZ, mexprs, fexprs, memmax);
      else
	printf("heap: %d cells (%d used, %d free)\n",
	       xnblks*XBLKSZ, mexprs, fexprs);
      break;
    }
    case 'p': {
      char *t;
      for (t = s+1; *t && isspace(*t); t++) ;
      if (*t) {
	int n1;
	if (strcmp(t, "all") == 0)
	  maxitems0 = 0;
	else if (isdigit(*t) && (n1 = atoi(t)) >= 0)
	  maxitems0 = n1;
	else {
	  error(qmmsg[INVALID_COMMAND]);
	  break;
	}
      }
      print_stack(thr, base, fno, xbp, addr, ip, modno, lineno, maxitems0);
      break;
    }
    case 'l': {
      long _modno = base?thr->asp[-base]->modno:modno;
      long _lineno = base?thr->asp[-base]->lineno:lineno;
      char *t;
      for (t = s+1; *t && isspace(*t); t++) ;
      if (*t) {
	char c;
	int n1, k, l = 0;
	if (sscanf(t, "%c%d%n", &c, &k, &l) > 0 && l > 0 &&
	    (c=='+' || c=='-')) {
	  if (c=='-') k = -k;
	  noffs = k;
	  for (t += l; *t && isspace(*t); t++) ;
	}
	if (*t)
	  if (isdigit(*t) && (n1 = atoi(t)) > 0)
	    nlines = n1;
	  else {
	    error(qmmsg[INVALID_COMMAND]);
	    break;
	  }
      }
      if (_lineno+noffs < 1) noffs = -(_lineno-1);
      print_source(_modno, _lineno+noffs, nlines);
      break;
    }
    case 'n':
      thr->debug = 0;
      stoplevel = level+1-base;
      goto exit;
    case 'c':
      set_debug(thr, 0);
      goto exit;
    case 'u': {
      int n = 1;
      char *t;
      for (t = s+1; *t && isspace(*t); t++) ;
      if (*t) {
	int n1;
	if (isdigit(*t) && (n1 = atoi(t)) >= 0)
	  n = n1;
	else {
	  error(qmmsg[INVALID_COMMAND]);
	  break;
	}
      }
      if (base+n > thr->asp-thr->ast) n = thr->asp-thr->ast-base;
      if (n > 0)
	base += n;
      else
	error("Already at top of stack");
      goto reprint;
    }
    case 'd': {
      int n = 1;
      char *t;
      for (t = s+1; *t && isspace(*t); t++) ;
      if (*t) {
	int n1 = atoi(s+1);
	if (isdigit(*t) && (n1 = atoi(t)) >= 0)
	  n = n1;
	else {
	  error(qmmsg[INVALID_COMMAND]);
	  break;
	}
      }
      if (base < n) n = base;
      if (n > 0)
	base -= n;
      else
	error("Already at bottom of stack");
      goto reprint;
    }
    case 't':
      if (base < thr->asp-thr->ast)
	base = thr->asp-thr->ast;
      goto reprint;
    case 'b':
      if (base > 0)
	base = 0;
      goto reprint;
    case 'h':
      thr->qmstat = HALT;
      goto exit;
    case 'q':
      thr->qmstat = QUIT;
      goto exit;
    default:
      error(qmmsg[INVALID_COMMAND]);
      break;
    }
  
  goto command;
  
 exit:
  /* reinstall break handler: */
  pop_sigint();
#ifdef USE_THREADS
  release_tty();
#endif
  return stoplevel;
}

static bool _brkflag;
static int _maxlevel, _maxlist, _maxchars;

static void set_print_params(void)
{
  _brkflag = brkflag;
  _maxlevel = maxlevel;
  _maxlist = maxlist;
  _maxchars = maxchars;
  brkflag = 0;
  maxlevel = maxlevel0;
  maxlist = maxlist0;
  maxchars = maxchars0;
}

static void reset_print_params(void)
{
  brkflag = _brkflag;
  maxlevel = _maxlevel;
  maxlist = _maxlist;
  maxchars = _maxchars;
}

/* rhs varsym table */

static void clearvartb(THREAD *thr)
{
  int i;
  for (i = 0; i < thr->nvarsyms; i++)
    if (thr->vartb[i].pname)
      free(thr->vartb[i].pname);
  if (thr->vartb) {
    free(thr->vartb);
    thr->vartb = NULL;
  }
  thr->nvarsyms = thr->avarsyms = 0;
}

static int vareq(VARREC *v1, VARREC *v2)
{
  return v1->offs == v2->offs && v1->plen == v2->plen &&
    v1->p == v2->p;
}

static int vartbadd(THREAD *thr, VARREC *v, int vsym)
{
  char s[MAXSTRLEN];

  if (thr->nvarsyms >= thr->avarsyms) {
    VARREC *vartb1 = (VARREC*) arealloc(thr->vartb, thr->avarsyms, 10,
					sizeof(VARREC));
    if (!vartb1) return NONE;
    thr->vartb = vartb1; thr->avarsyms += 10;
  }
  if (vsym >= 0)
    strcpy(s, strsp+vsym);
  else
    /*sprintf(s, "$%d", thr->nvarsyms+1);*/
    strcpy(s, "_");
  thr->vartb[thr->nvarsyms].pname = strdup(s);
  thr->vartb[thr->nvarsyms].offs = v->offs;
  thr->vartb[thr->nvarsyms].plen = v->plen;
  thr->vartb[thr->nvarsyms].p = v->p;
  return thr->nvarsyms++;
}

static VARREC *vtb;
static
varcmp(v1, v2)
     int	       *v1, *v2;
{
  return strcmp(vtb[*v1].pname, vtb[*v2].pname);
}

static int listvars(THREAD *thr, int offs)
{
  int             i, n, *v;

  vtb = thr->vartb;
  for (i = n = 0; i < thr->nvarsyms; i++)
    if (vtb[i].offs <= offs && strcmp(vtb[i].pname, "_")) n++;
  if (n == 0)
    return 1;
  else if (!(v = (int *)calloc(n, sizeof(int))))
    return 0;
  for (i = n = 0; i < thr->nvarsyms; i++)
    if (vtb[i].offs <= offs && strcmp(vtb[i].pname, "_")) v[n++] = i;
  /* sort variables */
  qsort(v, n, sizeof(int), (int(*)())varcmp);
  /* remove duplicates */
  for (i = 0; i < n-1; )
    if (strcmp(vtb[v[i]].pname, vtb[v[i+1]].pname) == 0) {
      int j;
      for (j = i+2; j < n; j++) v[j-1] = v[j];
      n--;
    } else
      i++;
  /* print variables */
  for (i = 0; i < n; i++) {
    if (i > 0)
      if (i % 4 == 0)
	printf("\n");
      else
	printf("\t");
    printf("%-15s", vtb[v[i]].pname);
  }
  printf("\n");
  free(v);
  return 1;
}

typedef struct {
  int sym;
  EXPR *x;
} display;

static display *dpy = NULL;
static int ndpy = 0;

static int savevars(THREAD *thr, int fno, long xbp, int offs)
{
  int             i, n, *v;

  vtb = thr->vartb;
  for (i = n = 0; i < thr->nvarsyms; i++)
    if (vtb[i].offs <= offs && strcmp(vtb[i].pname, "_")) n++;
  if (n == 0)
    return 1;
  else if (!(v = (int *)calloc(n, sizeof(int))))
    return 0;
  for (i = n = 0; i < thr->nvarsyms; i++)
    if (vtb[i].offs <= offs && strcmp(vtb[i].pname, "_")) v[n++] = i;
  /* sort variables */
  qsort(v, n, sizeof(int), (int(*)())varcmp);
  /* remove duplicates */
  for (i = 0; i < n-1; )
    if (strcmp(vtb[v[i]].pname, vtb[v[i+1]].pname) == 0) {
      int j;
      for (j = i+2; j < n; j++) v[j-1] = v[j];
      n--;
    } else
      i++;
  /* construct the display */
  if (!(dpy = (display*)calloc(n, sizeof(display)))) {
    free(v);
    return 0;
  }
  for (i = 0; i < n; i++) {
    int sym = mksym(vtb[v[i]].pname);
    if (sym == NONE ||
	!pushlval(thr, fno, xbp, vtb[v[i]].offs, vtb[v[i]].plen,
		  vtb[v[i]].p)) {
      free(v); ndpy = i;
      restorevars(thr);
      return 0;
    }
    dpy[i].sym = sym;
    dpy[i].x = symtb[sym].x;
    symtb[sym].x = *--thr->xsp;
  }
  free(v);
  ndpy = n;
  return 1;
}

static void restorevars(THREAD *thr)
{
  if (dpy) {
    int i;
    for (i = 0; i < ndpy; i++) {
      qmfree(thr, symtb[dpy[i].sym].x);
      symtb[dpy[i].sym].x = dpy[i].x;
    }
    free(dpy);
    dpy = NULL; ndpy = 0;
  }
}

static int getvar(THREAD *thr, char *name, int offs)
{
  int             i, last = NONE;

  if (strcmp(name, "_") == 0) return NONE;
  for (i = 0; i < thr->nvarsyms; i++)
    if (strcmp(thr->vartb[i].pname, name) == 0 &&
	thr->vartb[i].offs <= offs &&
	(last == NONE || thr->vartb[i].offs > thr->vartb[last].offs))
	last = i;
  return last;
}

static int mkvar(THREAD *thr, byte offs, byte plen, PATH p, int vsym)
{
  VARREC          v;
  int             i;

  v.offs = offs; v.plen = plen; v.p = p;
  for (i = 0; i < thr->nvarsyms; i++)
    if (vareq(thr->vartb+i, &v))
      break;
  if (i >= thr->nvarsyms)
    i = vartbadd(thr, &v, vsym);
  if (i != NONE && thr->vartb[i].pname)
    return mksym(thr->vartb[i].pname);
  else
    return NONE;
}

static int pushtmp(THREAD *thr, byte offs, byte plen, PATH p)
{
  int sym = mkvar(thr, offs, plen, p, -1);
  if (sym != NONE)
    return pushfun(thr, sym);
  else
    return 0;
}

static int arity(int fno)
{
  switch (fno) {
  case CONSOP:
  case PAIROP:
  case APPOP:
    return (2);
  default:
    return (0);
  }
}

static int s;
static byte _offs, _plen;
static PATH _p;

static int pushm(THREAD *thr)
{
  int k = statetb[s].trans;

  s = transtb[k].next;
  if (transtb[k].fno) {
    if (arity(transtb[k].fno)) {
      _plen++;
      setpath(&_p, _plen-1, 0);
      if (!pushm(thr)) return 0;
      setpath(&_p, _plen-1, 1);
      if (!pushm(thr)) return 0;
      setpath(&_p, _plen-1, 0);
      _plen--;
    }
    return pushfun(thr, transtb[k].fno);
  } else
    return pushtmp(thr, _offs, _plen, _p);
}

static int printm(THREAD *thr, int m, byte offs)
{
  s = matchtb[m]; _offs = offs; _plen = 0; _p = 0;
  if (!pushm(thr)) return 0;
  if (thr->xsp[-1]->fno == APPOP && thr->xsp[-1]->data.args.x1->fno == APPOP &&
      thr->xsp[-1]->data.args.x1->data.args.x1->fno == EQOP) {
    printf("(");
    set_print_params(), printx(thr->xsp[-1]), reset_print_params();
    printf(")");
  } else
    set_print_params(), printx(thr->xsp[-1]), reset_print_params();
  qmfree(thr, *--thr->xsp);
  return 1;
}

static void binding(THREAD *thr, int failed, int m, byte offs)
{
  push_sigint(SIG_IGN);
  check_thread(thr);
  dbg_stack(thr, 10);
  printf("--  def ");
  if (printm(thr, m, offs)) {
    printf(" = ");
    set_print_params(), printx(thr->xsp[-1]), reset_print_params();
    if (failed) printf("  :( FAILED )");
  }
  end_stack(thr, 10);
  putchar('\n');
  pop_sigint();
}

static void reduction(THREAD *thr, int fno, long xbp)
{
  push_sigint(SIG_IGN);
  check_thread(thr);
  dbg_stack(thr, 10);
  if (fno == APPOP && thr->xst[xbp]->fno == APPOP &&
      thr->xst[xbp]->data.args.x1->fno == DEFOP) {
    EXPR *p = thr->xst[xbp]->data.args.x2, *v = thr->xst[xbp+1];
    printf("--  def ");
    if (p->fno == APPOP && p->data.args.x1->fno == APPOP &&
	p->data.args.x1->data.args.x1->fno == EQOP) {
      printf("(");
      set_print_params(), printx(p), reset_print_params();
      printf(")");
    } else
      set_print_params(), printx(p), reset_print_params();
    printf(" = ");
    set_print_params(), printx(v), reset_print_params();
  } else if (fno == APPOP &&
	     thr->xst[xbp]->fno == UNDEFOP) {
    char s[MAXSTRLEN];
    printf("--  undef %s", pname(s, thr->xst[xbp+1]->fno));
  } else {
    printf("**  ");
    if (printp(thr, fno, xbp)) {
      printf("  ==>  ");
      set_print_params(), printx(thr->xsp[-1]), reset_print_params();
    }
  }
  end_stack(thr, 10);
  putchar('\n');
  pop_sigint();
}

static void default_reduction(THREAD *thr, int fno, long xbp)
{
  push_sigint(SIG_IGN);
  check_thread(thr);
  dbg_stack(thr, 10);
  if (fno == APPOP && thr->xst[xbp]->fno == APPOP &&
      thr->xst[xbp]->data.args.x1->fno == DEFOP) {
    EXPR *p = thr->xst[xbp]->data.args.x2, *v = thr->xst[xbp+1];
    printf("--  def ");
    if (p->fno == APPOP && p->data.args.x1->fno == APPOP &&
	p->data.args.x1->data.args.x1->fno == EQOP) {
      printf("(");
      set_print_params(), printx(p), reset_print_params();
      printf(")");
    } else
      set_print_params(), printx(p), reset_print_params();
    printf(" = ");
    set_print_params(), printx(v), reset_print_params();
    printf("  :( FAILED )");
  } else if (fno == APPOP &&
	     thr->xst[xbp]->fno == UNDEFOP) {
    char s[MAXSTRLEN];
    printf("-- undef %s  :( FAILED )", pname(s, thr->xst[xbp+1]->fno));
  } else {
#if 1
    return;
#else
    printf("**  ");
    printp(thr, fno, xbp);
#endif
  }
  end_stack(thr, 10);
  putchar('\n');
  pop_sigint();
}

static void tail_reduction(THREAD *thr, int fno, long xbp, int fno1)
{
  push_sigint(SIG_IGN);
  check_thread(thr);
  dbg_stack(thr, 10);
  printf("++  "); 
  if (printp(thr, fno, xbp)) {
    printf("  ==>  ");
    printp(thr, fno1, (fno1<BINARY)?thr->xsp-thr->xst-2:0);
  }
  end_stack(thr, 10);
  putchar('\n');
  pop_sigint();
}

static int printp(THREAD *thr, int fno, long xbp)
{
  char		s[MAXSTRLEN];
  if (fno < BINARY) {
    if (push(thr, thr->xst[xbp]))
      if (push(thr, thr->xst[xbp+1]))
	if (pushfun(thr, fno)) {
	  set_print_params(), printx(thr->xsp[-1]), reset_print_params();
	  qmfree(thr, *--thr->xsp);
	} else {
	  qmfree(thr, *--thr->xsp);
	  qmfree(thr, *--thr->xsp);
	  return 0;
	}
      else {
	qmfree(thr, *--thr->xsp);
	return 0;
      }
    else
      return 0;
  } else
    printf("%s", pname(s, fno));
  return 1;
}

static int xlat_pops(THREAD *thr, EXPR **mark)
{
  if (thr->xsp-mark > 1) {
    /* translate "dangling POPs" (toplevel ||) to ordinary applications */
    EXPR *x, *y, **act;
    for (x = *mark, act = mark+1; act < thr->xsp; act++) {
      if ((y = consexpr(thr, APPOP, funexpr(thr, SEQOP), x)) != NULL)
	x = y;
      else {
	if (x != *mark) qmfree(thr, x);
	return 0;
      }
      if ((y = consexpr(thr, APPOP, x, *act)) != NULL)
	x = y;
      else {
	qmfree(thr, x);
	return 0;
      }
    }
    while (thr->xsp > mark) qmfree(thr, *--thr->xsp);
    push(thr, x);
    return 1;
  } else
    return 1;
}

static int buildvartb(THREAD *thr, int addr)
{
  OPREC	       *ip;

  if (thr->lastaddr == addr) return 1;

  thr->lastaddr = addr;
  clearvartb(thr);

  /* enter variables into the var table: */
  for (ip = codesp + addr; ip->opcode != RETOP; ip++)
    if (ip->opcode == LVALOP && ip->opargs.lval.vsym >= 0)
	if (!mkvar(thr, ip->opargs.lval.offs, ip->opargs.lval.plen,
		   ip->opargs.lval.p, ip->opargs.lval.vsym))
	  return 0;
  return 1;
}

static int lastoffs(THREAD *thr, int addr, OPREC *ip0)
{
  OPREC *ip;
  int p = 0;
  for (ip = codesp + addr; ip->opcode != RETOP; ip++)
    if (ip->opcode == MATCHOP)
      if (ip <= ip0)
	p++;
      else
	return p;
  return p;
}

static int printc(THREAD *thr, int fno, long xbp, int addr, OPREC *ip0)
{
  long          top = thr->xsp-thr->xst;
  int           maxoffs = (fno==APPOP)?(top-xbp-2):(top-xbp);
  OPREC	       *ip;
  int		p = 0, q = 0, m = -1;

  if (!buildvartb(thr, addr)) return 0;

  /* build the rhs on the stack: */
  for (ip = codesp + addr; ip->opcode != RETOP; ip++)
    switch(ip->opcode) {
    case INFOP:
    case RETOP:
      break;
    case POPOP:
      /* these are treated later */
      break;
    case LVALOP:
      if (ip->opargs.lval.offs > p) {
	if (!pushtmp(thr, ip->opargs.lval.offs, ip->opargs.lval.plen,
		     ip->opargs.lval.p))
	  return 0;
      } else {
	if (ip->opargs.lval.offs)
	  mkvar(thr, ip->opargs.lval.offs, ip->opargs.lval.plen,
		ip->opargs.lval.p, -1);
	if (!pushlval(thr, fno, xbp, ip->opargs.lval.offs,
		      ip->opargs.lval.plen,
		      ip->opargs.lval.p))
	  return 0;
      }
      break;
    case QUERYOP:
      if (!xlat_pops(thr, thr->xst+top+q)) return 0;
      if (ip < ip0 || q)
	qmfree(thr, *--thr->xsp);
      else
	q++;
      break;
    case MATCHOP:
      if (!xlat_pops(thr, thr->xst+top+q)) return 0;
      if (ip < ip0) p++;
      if (ip < ip0 || q)
	qmfree(thr, *--thr->xsp);
      else {
	m = ip->opargs.m;
	q++;
      }
      break;
    case INTVALOP:
      { mpz_t z;
      if (!getint(z, ip->opargs.iv.len, ip->opargs.iv.l)) {
	thr->qmstat = MEM_OVF;
	return 0;
      } else if (!pushmpz(thr, z))
	return (0);
      }
      break;
    case FLOATVALOP:
      if (!pushfloat(thr, ip->opargs.fv))
	return 0;
      break;
    case STRVALOP:
      {
	char           *s;
	if ((s = strdup(strsp + ip->opargs.sv)) ==
	    NULL) {
	  thr->qmstat = MEM_OVF;
	  return 0;
	} else if (!pushstr(thr, s))
	  return 0;
      }
      break;
    default:
      if (!pushfun(thr, ip->opcode))
	return 0;
      break;
    }
  if (!xlat_pops(thr, thr->xst+top+q)) return 0;
  thr->qmstat = OK;
  set_print_params(), printx(thr->xsp[-1]), reset_print_params();
  qmfree(thr, *--thr->xsp);
  if (q) {
    if (m >= 0) {
      printf(" where ");
      if (printm(thr, m, p+1)) {
	printf(" = ");
	set_print_params(), printx(thr->xsp[-1]), reset_print_params();
      }
    } else {
      /* suppress generated IDOP queries: */
      EXPR *x;
      for (x = thr->xsp[-1]; x->fno == APPOP; x = x->data.args.x1) ;
      if (x->fno != IDOP) {
	printf(" if ");
	set_print_params(), printx(thr->xsp[-1]), reset_print_params();
      }
    }
    qmfree(thr, *--thr->xsp);
  }
  return 1;
}

/* STRING SPACE: */

static int putstr(char *s)
{
  int             k = strspsz + tmpspsz, l = strlen(s);

  while (l >= strspsz + atmpspsz - k) {
    char *strsp1;
    if (!(strsp1 = (char*) arealloc(strsp, strspsz+atmpspsz,
				    TMPSPSZ, sizeof(char))))
      return(NONE);
    else {
      strsp = strsp1;
      atmpspsz += TMPSPSZ;
    }
  }
  strcpy(strsp+k, s);
  tmpspsz += l + 1;
  return (k);
}

/* SYMBOL TABLE: */

static int strhash(char *s, int sz)
{
  unsigned h = 0, g;
  while (*s) {
    h = (h<<4)+*(s++);
    if ((g = (h & 0xf0000000)))	{
      h = h^(g>>24);
      h = h^g;
    }
  }
  return h % sz;
}

#define streq(s1,s2) (!strcmp(s1,s2))

#define NIL (-2)
#define ANY (-3)

static int splitid(char *s, char *mnm)
{
  char *p;
  int mno = ANY;
  *mnm = 0;
  if ((p = strchr(s, ':'))) {
    char t[MAXSTRLEN];
    strcpy(t, p+2);
    *p = 0;
    strcpy(mnm, s);
    if (!*s)
      mno = NONE;
    else {
      mno = getmodno(s);
      if (mno == NONE) mno = NIL;
    }
    strcpy(s, t);
  }
  return mno;
}

static int symprio(int sym)
{
  if (symtb[sym].modno == NONE)
    return -1;
  else if (globs[symtb[sym].modno] & 2)
    return 0;
  else
    return 1;
}

/* mksym(): look up a (possibly qualified) function or variable symbol in the
   global scope, and create a new variable symbol if the symbol does not yet
   exist; sets qmstat to either BAD_REF, BAD_SYM or SYMTB_OVF if symbol is
   not defined or could not be created */

#define matchsym(fno,s) (!(symtb[fno].flags & TSYM)&&\
			 streq((s), strsp+symtb[fno].pname))

int mksym(char *s)
{
  char mnm[MAXSTRLEN];
  int modno = (mainno>=0)?mainno:0, mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int fno, fno1 = NONE, pname;

  if (mno == NIL)
    return NONE;
  if (mno == ANY) {
    /* look for symbol in all global imports */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchsym(fno, s) &&
	  (symtb[fno].modno == NONE || symtb[fno].modno == modno ||
	   globs[symtb[fno].modno]))
	if (symtb[fno].modno == modno) {
	  /* found symbol in main module, done */
	  fno1 = fno;
	  break;
	} else if (symtb[fno].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (fno1 != NONE) {
	  int r1 = fno1, r = fno;
	  while (symtb[r1].ref) r1 = symtb[r1].ref;
	  while (symtb[r].ref) r = symtb[r].ref;
	  if (r1 == r)
	    /* aliases for same symbol, skip */
	    ;
	  else if (symprio(fno) == symprio(fno1)) {
	    /* multiple imports, error */
	    get_thr()->qmstat = BAD_REF;
	    return NONE;
	  } else
	    break;
	} else
	  fno1 = fno;
  } else {
    /* look for qualified symbol in given module */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchsym(fno, s) && symtb[fno].modno == mno) {
	fno1 = fno;
	break;
      }
  }
  fno = fno1;
  if (fno == NONE) {
    if (mno != ANY) {
      /* undefined symbol, not in global namespace */
      get_thr()->qmstat = BAD_SYM;
      return NONE;
    }
    if (symtbsz+tmptbsz > SHRT_MAX) {
      get_thr()->qmstat = SYMTB_OVF;
      return NONE;
    }
    if (tmptbsz >= atmptbsz) {
      SYMREC *symtb1;
      if (!(symtb1 =
	    (SYMREC*)arealloc(symtb,
			      symtbsz+atmptbsz, TMPTBSZ/10,
			      sizeof(SYMREC)))) {
	get_thr()->qmstat = SYMTB_OVF;
	return NONE;
      } else {
	symtb = symtb1;
	atmptbsz += TMPTBSZ/10;
      }
    }
    if ((pname = putstr(s)) == NONE) {
      get_thr()->qmstat = MEM_OVF;
      return NONE;
    }
    fno = symtbsz+tmptbsz++;
    symtb[fno].ref = 0;
    symtb[fno].flags = VSYM;
    symtb[fno].type = 0;
    symtb[fno].fno_min = symtb[fno].fno_max = 0;
    symtb[fno].argc = 0;
    symtb[fno].argv = 0;
    symtb[fno].modno = mainno>=0?mainno:modtbsz>0?0:NONE;
    symtb[fno].xfno = fno;
    symtb[fno].pname = pname;
    symtb[fno].x = symtb[fno].xp = NULL;
    symtb[fno].f = NULL;
    symtb[fno].next = hashtb[k];
    hashtb[k] = fno;
    symtb[fno].flags |= UNIQ|VIS;
  } else {
    while (symtb[fno].ref)
      fno = symtb[fno].ref;
  }
  return fno;
}

/* getsym(): look up a (possibly qualified) function or variable symbol in the
   context of a given module, return the symbol or NONE if it does not
   exist */

static int symprio2(int p, int sym)
{
  if (symtb[sym].modno == NONE)
    return -1;
  else if (impib[p] & 2)
    return 0;
  else
    return 1;
}

static int searchimp(int modno, int mno)
{
  int i, i1 = imports[modno],
    i2 = (modno+1<modtbsz)?imports[modno+1]:imptbsz;
  for (i = i1; i < i2; i++)
    if (imptb[i] == mno)
      return i;
  return NONE;
}

int getsym(char *s, int modno)
{
  char mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int fno, fno1 = NONE, p = NONE, p1, pname;

  if (mno >= 0) {
    /* check that module is in local scope */
    if (searchimp(modno, mno) == NONE)
      mno = NIL;
  }
  if (mno == NIL)
    return NONE;
  if (mno == ANY) {
    /* look for symbol in all local imports */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchsym(fno, s) &&
	  (symtb[fno].modno == NONE || symtb[fno].modno == modno ||
	   (p = searchimp(modno, symtb[fno].modno)) != NONE))
	if (symtb[fno].modno == modno) {
	  /* found symbol in this module, done */
	  fno1 = fno;
	  break;
	} else if (symtb[fno].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (fno1 != NONE) {
	  if (symprio2(p,fno) == symprio2(p1,fno1)) {
	    /* multiple imports, error */
	    return NONE;
	  } else
	    break;
	} else
	  p1 = p, fno1 = fno;
  } else {
    /* look for qualified symbol in given module */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchsym(fno, s) && symtb[fno].modno == mno) {
	if (!(symtb[fno].flags & PRIV) || mno == modno)
	  fno1 = fno;
	break;
      }
  }
  return fno1;
}

/* gettype(): look up a (possibly qualified) type symbol in the context of a
   given module, return the symbol or NONE if it does not exist */

#define matchtype(type,s) ((symtb[type].flags & TSYM)&&\
			   streq((s), strsp+symtb[type].pname))

int gettype(char *s, int modno)
{
  char mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int type, type1 = NONE, p = NONE, p1, pname;

  if (mno >= 0) {
    /* check that module is in local scope */
    if (searchimp(modno, mno) == NONE)
      mno = NIL;
  }
  if (mno == NIL)
    return NONE;
  if (mno == ANY) {
    /* look for symbol in all local imports */
    for (type = hashtb[k]; type != NONE; type = symtb[type].next)
      if (matchtype(type, s) &&
	  (symtb[type].modno == NONE || symtb[type].modno == modno ||
	   (p = searchimp(modno, symtb[type].modno)) != NONE))
	if (symtb[type].modno == modno) {
	  /* found symbol in this module, done */
	  type1 = type;
	  break;
	} else if (symtb[type].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (type1 != NONE) {
	  if (symprio2(p,type) == symprio2(p1,type1)) {
	    /* multiple imports, error */
	    return NONE;
	  } else
	    break;
	} else
	  p1 = p, type1 = type;
  } else {
    /* look for qualified symbol in given module */
    for (type = hashtb[k]; type != NONE; type = symtb[type].next)
      if (matchtype(type, s) && symtb[type].modno == mno) {
	if (!(symtb[type].flags & PRIV) || mno == modno)
	  type1 = type;
	break;
      }
  }
  return type1;
}

/* getmodno() returns the module index of a module name, NONE if
   nonexistent. */

int getmodno(char *name)
{
  int i;
  for (i = 0; i < modtbsz; i++)
    if (streq(name, strsp+modtb[i]))
      return i;
  return NONE;
}

