
%{ /* -*-C-*- */

/* qmlex.l: lexical analyzer for Q interpreter commands */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* make sure we have flex */

#ifndef FLEX_SCANNER
#error "Sorry, this program requires flex."
#endif

#include "qdefs.h"
#include "qmparse.h"

int		pmode = -1;	/* parse mode indicator */
static int	lexinit = 0;	/* initial state */
static char	_sflag;		/* set when input comes from string */
static char    *_s;		/* string buffer */
static char    *_sp;		/* pointer to end of last token */
static FILE    *_fp;		/* input file */

static char     s1[MAXSTRLEN], s2[MAXSTRLEN];

static int abufsz = 0, bufp = 0;
static char *buf = NULL;

static THREAD *thr;

static bigint();
static string();

/* redefined input macro: */

#define getch()     (_sflag?(*_s?*_s++:EOF):getc(_fp))

#undef YY_INPUT
#define YY_INPUT(buf,result,max_size)\
	{\
	int c = getch();\
	result = (c == EOF)?YY_NULL:(buf[0]=c,1);\
	}

#define tok(x) if(_sflag)_sp=_s;BEGIN(0);return x;
#define tok2(x) if(_sflag)_sp=_s;BEGIN(arg);return x;

%}

%s cmd arg

L			[A-Za-z_]
O			[0-7]
D			[0-9]
X			[0-9A-Fa-f]
ID			({L}({L}|{D})*)
SF			([Ee][+-]?{D}+)
XSF			([Pp][+-]?{D}+)
C			([^\n\\\"]|\\.)
WS			[ \t\f\r]
NWS			([^ \t\f\r;]|\\.)
NWSNQ			([^ \t\f\r;"']|\\.)

%%

	if (lexinit) {
	  lexinit = 0;
	  if (pmode == INTERACT || pmode == SOURCE)
	    BEGIN(cmd);
	  yy_set_bol(1);
	  if(_sflag)_sp=_s;
	  return pmode;
	}

<cmd>^"#!"		{ skip(); BEGIN(0); }
<cmd>^{WS}*"@"		;
<cmd>{WS}+		;
<cmd>\?			BEGIN(0);
<cmd>break		tok2(_BREAK_);
<cmd>cd			tok2(_CHDIR_);
<cmd>chdir		tok2(_CHDIR_);
<cmd>clear		tok(_CLEAR_);
<cmd>completion_matches	tok2(_COMPLETION_MATCHES_);
<cmd>copying		tok2(_COPYING_);
<cmd>debug		tok2(_DEBUG_);
<cmd>dec		tok2(_DEC_);
<cmd>def		tok(_DEF_);
<cmd>echo		tok2(_ECHO_);
<cmd>edit		tok2(_EDIT_);
<cmd>fix		tok2(_FIX_);
<cmd>format		tok2(_FORMAT_);
<cmd>help		tok2(_HELP_);
<cmd>hex		tok2(_HEX_);
<cmd>histfile		tok2(_HISTFILE_);
<cmd>histsize		tok2(_HISTSIZE_);
<cmd>import		tok2(_IMPORT_);
<cmd>imports		tok2(_IMPORTS_);
<cmd>load		tok2(_LOAD_);
<cmd>ls			tok2(_LS_);
<cmd>memsize		tok2(_MEMSIZE_);
<cmd>modules		tok2(_MODULES_);
<cmd>oct		tok2(_OCT_);
<cmd>path		tok2(_PATH_);
<cmd>prompt		tok2(_PROMPT_);
<cmd>pwd		tok2(_PWD_);
<cmd>run		tok2(_RUN_);
<cmd>save		tok2(_SAVE_);
<cmd>sci		tok2(_SCI_);
<cmd>\./{WS}		tok2(_SOURCE_);
<cmd>source		tok2(_SOURCE_);
<cmd>stacksize		tok2(_STACKSIZE_);
<cmd>stats		tok2(_STATS_);
<cmd>std		tok2(_STD_);
<cmd>undef		tok(_UNDEF_);
<cmd>which		tok2(_WHICH_);
<cmd>who		tok2(_WHO_);
<cmd>whos		tok(_WHOS_);
<arg>{WS}+		;
<arg>{NWSNQ}{NWS}*	{ static char *buf1 = NULL;
			  if (buf1) free(buf1);
			  if(thr->qmstat != MEM_OVF &&
			     (buf1 = (char*)malloc((yyleng+1)*sizeof(char)))
			     != NULL) {
			    yylval.sval = scanstr(buf1, yytext);
			    tok2(ARG);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok2(ERRTOK);
			  }
			}
<arg>["']		{ static char *buf1 = NULL;
			  if (buf1) free(buf1);
			  string(yytext[0]);
			  if(thr->qmstat == STR_ERR) {
			    tok2(ARGERR);
			  } else if(thr->qmstat != MEM_OVF &&
			     (buf1 = (char*)malloc((strlen(buf)+1)*
						   sizeof(char)))
			     != NULL) {
			    yylval.sval = scanstr(buf1, buf);
			    tok2(ARG);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok2(ERRTOK);
			  }
			}
and			tok(AND);
as			tok(AS);
const			tok(CONST);
def			tok(DEF);
div			tok(DIV);
else			tok(ELSE);
extern			tok(EXTERN);
if			tok(IF);
import			tok(IMPORT);
in			tok(IN);
include			tok(INCLUDE);
mod			tok(MOD);
not			tok(NOT);
or			tok(OR);
otherwise		tok(OTHERWISE);
private			tok(PRIVATE);
public			tok(PUBLIC);
special			tok(SPECIAL);
then			tok(THEN);
type			tok(TYPE);
undef			tok(UNDEF);
var			tok(VAR);
where			tok(WHERE);
"++"			tok(CAT);
"<="			tok(LEQ);
">="			tok(GEQ);
"<>"			tok(NEQ);
"||"			tok(DBAR);
({ID}?::)?{ID}		{ if (yyleng > MAXSTRLEN) {
			    thr->qmstat = BAD_SYM;
			    tok(ERRTOK);
			  }
			  yylval.ival = mksym(yytext);
			  if (yylval.ival == NONE) {
			    tok(ERRTOK);
			  } else {
			    tok(ID);
			  }
			}
"<<"{ID}">>"		|
"<<"{ID}::{ID}">>"	tok(XID);

  /* There is a potential mem leak here -- when scanning an integer or string,
     memory is allocated dynamically which might not be claimed back when the
     parser runs into a syntax error before having processed the token.
     However, with the current grammar this shouldn't happen since *any* legal
     token sequence can be followed by a primary. At least I hope so. ;-) */

0{O}+			{ if (bigint(yylval.zval)) {
			    tok(INT);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
                        }
0{D}+			{ tok(ERRTOK); }
{D}+			|
0[xX]{X}+		{ if (bigint(yylval.zval)) {
			    tok(INT);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
                        }
{D}+{SF}		|
{D}+\.{D}*{SF}?		|
{D}*\.{D}+{SF}?		{ sscanf(yytext, "%lf", &yylval.fval); tok(FLOAT); }
0[xX]{X}+{XSF}		|
0[xX]{X}+\.{X}*{XSF}?	|
0[xX]{X}*\.{X}+{XSF}?	{
#ifdef HAVE_ISO99_PRINTF
                          sscanf(yytext, "%la", &yylval.fval); tok(FLOAT);
#else
			  tok(ERRTOK);
#endif
}
\"			{ char *buf1 = NULL;
			  string('"');
			  if(thr->qmstat == STR_ERR) {
			    tok2(STRERR);
			  } else if(thr->qmstat != MEM_OVF &&
			     (buf1 = (char*)malloc((strlen(buf)+1)
						   *sizeof(char)))
			     != NULL) {
			    yylval.sval = scanstr(buf1, buf);
			    tok(STR);
			  } else {
			    thr->qmstat = MEM_OVF;
			    tok(ERRTOK);
			  }
			}
{WS}			|
"//".*			|
\%.*			/* line-oriented comments allowed */;
\\\n			;
";"			{ if(_sflag)_sp=_s; BEGIN(cmd); return yytext[0]; }
.			|
\n			tok(yytext[0]);

%%

static initbuf()
{
  bufp = 0;
}

static addbuf(char c)
{
  if (bufp >= abufsz)
    if (!(buf = (char*)arealloc(buf, abufsz, 100, sizeof(char)))) {
      thr->qmstat = MEM_OVF;
      return;
    } else
      abufsz += 100;
  buf[bufp++] = c;
}

static string(int d)
{
  register int    c;

  initbuf();
  while ((c = input()) && c != EOF && thr->qmstat != MEM_OVF) {
    if (c == d)
      break;
    else if (c == '\\') {
      if ((c = input()) != '\n') {
	addbuf('\\');
	addbuf(c);
      }
    } else if (c == '\n') {
      break;
    } else
      addbuf(c);
  }
  addbuf('\0');
  if (c != d) {
    thr->qmstat = STR_ERR;
    skip();
  }
}

static char *
skipz(char *s)
{
  while (*s == '0') s++;
  return s;
}

static bigint(z)
     mpz_t z;
{
  int sz;
  if (strncmp(yytext, "0x", 2) == 0 ||
      strncmp(yytext, "0X", 2) == 0)
    sz = 4*strlen(skipz(yytext+2));
  else if (*yytext == '0')
    sz = 3*strlen(skipz(yytext+1));
  else
    sz = log(10)/log(2)*strlen(skipz(yytext))+1;
  sz = sz/(CHAR_BIT*sizeof(mp_limb_t)) + 2;
  mpz_init(z); 
  if (z->_mp_d && my_mpz_realloc(z, sz)) {
    int sz1;
    mpz_set_str(z, yytext, 0);
    sz1 = mpz_size(z);
    if (sz1 < sz && !my_mpz_realloc(z, sz1)) {
      thr->qmstat = MEM_OVF;
      tok(ERRTOK);
    }
    tok(INT);
  } else {
    thr->qmstat = MEM_OVF;
    tok(ERRTOK);
  }
}

#undef yywrap

yywrap()
{
  return 1;
}

peek()
{
  int c = input();
  unput(c);
  return c;
}

skip()
/* skip remainder of input line */
{
  if (_sflag)
    _s += strlen(_s);
  else {
    register int c;
    while ((c = input()) && c != EOF && c != '\n')
      if (c == '\\' && (!(c = input()) || c == EOF))
	break;
  }
}

getln(s)
     char	       *s;
/* get remainder of input line */
{
  if (_sflag) {
    strcpy(s, _s);
    _s += strlen(_s);
  } else {
    register int c;
    while ((c = input()) && c != EOF && c != '\n')
      if (c == '\\') {
	*s++ = c;
	if (!(c = input()) || c == EOF)
	  break;
	else
	  *s++ = c;
      } else
	*s++ = c;
    *s = 0;
  }
}

static struct {
  int	   pmode, lexinit, start;
  char	   _sflag;
  char    *_s, *_sp;
  FILE    *_fp;
  YY_BUFFER_STATE state;
  THREAD *thr;
} lexstack[MAXSTACK], *lexp = NULL;

static lexpush()
{
  if (!lexp)
    lexp = lexstack;
  else {
    /* scanner already running, push state */
    if (lexp - lexstack >= MAXSTACK)
      return 0;
    lexp->pmode = pmode;
    lexp->lexinit = lexinit;
    lexp->start = YYSTATE;
    lexp->_sflag = _sflag;
    lexp->_s = _s;
    lexp->_sp = _sp;
    lexp->_fp = _fp;
    lexp->state = YY_CURRENT_BUFFER;
    lexp->thr = thr;
    yy_switch_to_buffer(yy_create_buffer(NULL, YY_BUF_SIZE));
    lexp++;
  }
  yyrestart(NULL);
  return 1;
}

static lexpop()
{
  if (lexp > lexstack) {
    --lexp;
    pmode = lexp->pmode;
    lexinit = lexp->lexinit;
    _sflag = lexp->_sflag;
    _s = lexp->_s;
    _sp = lexp->_sp;
    _fp = lexp->_fp;
    thr = lexp->thr;
    yy_delete_buffer(YY_CURRENT_BUFFER);
    yy_switch_to_buffer(lexp->state);
    BEGIN(lexp->start);
  } else {
    lexp = NULL;
    yyrestart(NULL);
  }
}

int initlex(void *source, int mode)
{
  if (!lexpush())
    return 0;
  thr = get_thr();
  BEGIN(0);
  pmode = mode;
  lexinit = 1;
  switch (mode) {
  case STRING:
  case INTERACT:
  case SOURCE:
    _sflag = 1;
    _s = _sp = (char*) source;
    break;
  case LINE:
    _sflag = 0;
    _fp = (FILE*) source;
    break;
  }
  return 1;
}

void finilex(void)
{
  lexinit = 0;
  lexpop();
  if (abufsz > 10000) {
    free(buf); buf = NULL; abufsz = 0;
  }
}

char *actchar(void)
{
  if (_sflag)
    return _sp;
  else
    return NULL;
}
