
/* qmmatch.c: TA matching routines for the Q machine */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qdefs.h"

/* The nextstate() function looks up a transition on a given symbol with a
   given type in the transition table. It returns the transition with
   the type which is the "smallest" supertype of the given type. This is
   implemented by a kind of mergesort algorithm which compares the list
   of types for the symbol in the transition table to the list of supertypes
   of the given type. */

/* Note: If your compiler supports it, make sure that nextstate() and the
   following defaultstate() function are inlined. This considerably improves
   performance of the interpreter. */

inline
static
nextstate(int s, int type, int fno)
{
  int k = statetb[s].trans, left = k, right = k+statetb[s].ntrans;

  if (!statetb[s].ntrans) return 0;
  
  if (statetb[s].ntrans <= 5) {
    /* linear search */
    for (; fno > transtb[k].fno && k < right; k++)
      ;
  } else {
    /* binary search */
    int l = left, r = right-1;
    while (l <= r) {
      k = (l+r)/2;
      if (fno < transtb[k].fno)
	r = k-1;
      else if (fno > transtb[k].fno)
	l = k+1;
      else
	break;
    }
    while (k > left && transtb[k-1].fno == fno)
      k--;
  }

  if (k >= right || transtb[k].fno != fno)
    return 0;
		
  for (;;)
    if (type > transtb[k].type)
      type = symtb[type].type;
    else if (type < transtb[k].type) {
      if (++k >= right || transtb[k].fno != fno)
	return 0;
    } else
      return transtb[k].next;
}

/* The defaultstate() function looks up the default transition on a given
   type in the transition table. */

inline
static
defaultstate(int s, int type)
{
  int k = statetb[s].trans, left = k, right = k+statetb[s].ntrans;

  if (k >= right || transtb[k].fno != 0) return 0;
	
  for (;;)
    if (type > transtb[k].type)
      type = symtb[type].type;
    else if (type < transtb[k].type) {
      if (++k >= right || transtb[k].fno != 0)
	return 0;
    } else
      return transtb[k].next;
}

/* The matchx() function matches an expression starting at the given state and
   returns the resulting end state (0 if the match fails). */

/* We have two versions here: the default recursive version which is more
   straightforward and is actually a little faster on my system, and an
   iterative version which only needs constant C stack space. You might wish
   to experiment with both versions to see which one performs best with your
   system and C compiler. */

#if 1

/* recursive version */

inline
static
matchx(int s, EXPR *x)
{
  int s1, fno = x->fno, type = (x->argc)?0:x->type;
#ifdef DEBUG
  char fsym[MAXSTRLEN], tsym[MAXSTRLEN];
  printf("state %d, symbol %s, type %s\n", s,
	 fno>=RESERVED?pname(fsym,fno):"_",
	 type?pname(tsym,type):"<none>");
#endif
  switch (fno) {
  case CONSOP: case PAIROP: case APPOP:
    if ((s1 = nextstate(s, type, fno))) {
      (s = matchx(s1, x->data.args.x1)) && (s = matchx(s, x->data.args.x2));
      return s;
    } else
      return defaultstate(s, type);
  case VECTOP: {
    int i, n = x->data.vect.n;
    EXPR **xv = x->data.vect.xv;
    for (i = 0; i < n; i++)
      if ((s1 = nextstate(s, TUPLETYPE, PAIROP))) {
	if (!(s = matchx(s1, xv[i])))
	  return 0;
      } else
	return defaultstate(s, TUPLETYPE);
    if ((s1 = nextstate(s, TUPLETYPE, VOIDOP)))
      return s1;
    else
      return defaultstate(s, TUPLETYPE);
  }
  default:
    if ((s1 = nextstate(s, type, fno)))
      return s1;
    else
      return defaultstate(s, type);
  }
}

#else

/* non-recursive version */

static EXPR *xstk[1000];
static int xstkp;

inline
static
matchx(int s, EXPR *x)
{
  int s1, mark, fno, type;
#ifdef DEBUG
  char fsym[MAXSTRLEN], tsym[MAXSTRLEN];
#endif
  mark = xstkp;
 loop:
  fno = x->fno;
  type = (x->argc)?0:x->type;
#ifdef DEBUG
  printf("state %d, symbol %s, type %s\n", s,
	 fno>=RESERVED?pname(fsym,fno):"_",
	 type?pname(tsym,type):"<none>");
#endif
  switch (fno) {
  case CONSOP: case PAIROP: case APPOP:
    if ((s1 = nextstate(s, type, fno))) {
      s = s1;
      xstk[xstkp++] = x;
      x = x->data.args.x1;
      goto loop;
    } else {
      s = defaultstate(s, type);
      goto pop;
    }
  case VECTOP: {
    int i, n = x->data.vect.n;
    EXPR **xv = x->data.vect.xv;
    for (i = 0; i < n; i++)
      if ((s1 = nextstate(s, TUPLETYPE, PAIROP))) {
	if (!(s = matchx(s1, xv[i]))) goto pop;
      } else {
	s = defaultstate(s, TUPLETYPE);
	goto pop;
      }
    if ((s1 = nextstate(s, TUPLETYPE, VOIDOP)))
      s = s1;
    else
      s = defaultstate(s, TUPLETYPE);
    goto pop;
  }
  default:
    if ((s1 = nextstate(s, type, fno)))
      s = s1;
    else
      s = defaultstate(s, type);
  pop:
    if (!s) {
      xstkp = mark; return 0;
    }
    while (xstkp > mark && x == xstk[xstkp-1]->data.args.x2)
      x = xstk[--xstkp];
    if (xstkp > mark) {
      x = xstk[xstkp-1]->data.args.x2;
      goto loop;
    }
    break;
  }
  return s;
}

#endif

/* interface functions */

/* match( fno, rp, rc ) matches a term with top symbol fno and
   corresponding number of arguments from the top of the stack against
   the left-hand sides of rules and sets *rp and *rc to the corresponding
   rule pointer and counter if a matching rule is found; returns: zero if
   there is no matching rule, nonzero otherwise. */

int match(THREAD *thr, int fno, int **rp, int *rc)
{
  int s, type;
#ifdef DEBUG
  char fsym[MAXSTRLEN], tsym[MAXSTRLEN];
#endif
  if (statetbsz == 0 || fno < RESERVED)
    return 0;
  switch (fno) {
  case CONSOP:
    type = LISTTYPE;
  match2:
#ifdef DEBUG
    printf("state %d, symbol %s, type %s\n", 0,
	   fno>=RESERVED?pname(fsym,fno):"_",
	   type?pname(tsym,type):"<none>");
#endif
    if ((s = nextstate(0, type, fno)))
      (s = matchx(s, thr->xsp[-2])) && (s = matchx(s, thr->xsp[-1]));
    else
      s = defaultstate(0, type);
    break;
  case PAIROP:
    type = TUPLETYPE;
    goto match2;
  case APPOP:
    type = (thr->xsp[-2]->argc == 1)?thr->xsp[-2]->type:0;
    goto match2;
  default:
    type = !symtb[fno].argc?symtb[fno].type:0;
#ifdef DEBUG
    printf("state %d, symbol %s, type %s\n", 0,
	   fno>=RESERVED?pname(fsym,fno):"_",
	   type?pname(tsym,type):"<none>");
#endif
    if (!(s = nextstate(0, type, fno)))
      s = defaultstate(0, type);
    break;
  }
#ifdef DEBUG
  printf(s?"match succeeded, final state %d\n":"match failed\n", s);
#endif
  if (s != 0 && statetb[s].nrules > 0) {
    *rc = statetb[s].nrules;
    *rp = roffstb + statetb[s].roffs;
    return 1;
  } else
    return 0;
}

/* matchp( s ) matches the expression on the top of the stack, starting at the
   given state */

int matchp(THREAD *thr, int s)
{
  s = matchx(s, thr->xsp[-1]);
#ifdef DEBUG
  printf(s?"match succeeded, final state %d\n":"match failed\n", s);
#endif
  return s;
}
