
/* qmprint.c: unparsing of Q expressions */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qdefs.h"

/* expression outline parameters */

int maxlevel;	/* maximum expression depth */
int maxlist;	/* maximum list/tuple members */
int maxchars;	/* maximum string size */

static char    *bufp;
static int      bufleng, abufleng;
 /* buffer for unparsed expression */
static FILE    *fp;
 /* output file for unparsed expression */
static int      (*putstr) ();
 /* string put routine */

/* sputstr(): write string to buffer */

static sputstr(s)
     char           *s;
{
  int l = strlen(s);

  while (bufleng >= abufleng-l) {
    /* try to enlarge the buffer */
    char *bufp1;
    if (bufp1 = (char*) arealloc(bufp, abufleng, MAXSTRLEN,
				 sizeof(char))) {
      bufp = bufp1;
      abufleng += MAXSTRLEN;
    } else {
      free(bufp);
      return (0);
    }
  }
  strcpy(bufp+bufleng, s);
  bufleng += l;
  return (1);
}

/* fputstr(): write string to file */

static fputstr(s)
     char           *s;
{
  int l = strlen(s);

  if (checkbrk && fp == stdout)
    return (0);
  if (fputs(s, fp) == EOF)
    return (0);
  else
    return (1);
}

/* opstr(): return print representation of an operator symbol */

static char *pad(char *s, int l, int r)
{
  static char buffer[10];
  char *template;
  if (l && r)
    template = " %s ";
  else if (l)
    template = " %s";
  else if (r)
    template = "%s ";
  else
    return s;
  sprintf(buffer, template, s);
  return buffer;
}

static char *
opstr(int fno, int l, int r)
{
  switch (fno) {
  case SEQOP:
    return "||";
  case LEOP:
    return "<";
  case GROP:
    return ">";
  case EQOP:
    return "=";
  case LEQOP:
    return "<=";
  case GEQOP:
    return ">=";
  case NEQOP:
    return "<>";
  case INOP:
    return pad("in",l,r);
  case IDOP:
    return "==";
  case CATOP:
    return "++";
  case ADDOP:
    return "+";
  case MINOP:
    return "-";
  case QUOTEOP:
    return "'";
  case UNQUOTEOP:
    return "`";
  case FORCEOP:
    return "~";
  case OROP:
    return pad("or",l,r);
  case ORELSEOP:
    return pad("or else",l,r);
  case MULOP:
    return "*";
  case FDIVOP:
    return "/";
  case DIVOP:
    return pad("div",l,r);
  case MODOP:
    return pad("mod",l,r);
  case ANDOP:
    return pad("and",l,r);
  case ANDTHENOP:
    return pad("and then",l,r);
  case UMINOP:
    return "-";
  case NOTOP:
    return "not ";
  case HASHOP:
    return "#";
  case POWOP:
    return "^";
  case IDXOP:
    return "!";
  case APPOP:
    return " ";
  }
}

/* opprec(): return precedence of operator symbol */

static opprec(int fno)
{
  switch (fno) {
  case SEQOP:
    return (0);	/* || */
  case LEOP:
  case GROP:
  case EQOP:
  case LEQOP:
  case GEQOP:
  case NEQOP:
  case INOP:
  case IDOP:
    return (1);	/* relational operators */
  case CATOP:
  case ADDOP:
  case MINOP:
  case OROP:
  case ORELSEOP:
    return (2);	/* addition operators */
  case MULOP:
  case FDIVOP:
  case DIVOP:
  case MODOP:
  case ANDOP:
  case ANDTHENOP:
    return (3);	/* multiplication operators */
  case UMINOP:
  case NOTOP:
  case HASHOP:
    return (4);	/* prefix operators */
  case POWOP:
  case IDXOP:
    return (5);	/* super/subscript */
  case APPOP:
    return (6);	/* @ */
  case QUOTEOP:
  case UNQUOTEOP:
  case FORCEOP:
    return (7);	/* quotation */
  default:
    return (-1);	/* no operator */
  }
}

/* islsect(), isrsect(): check for left and right operator sections */

static islsect(EXPR *x)
{
  int p;
  return (x->fno == APPOP && (p = opprec(x->data.args.x1->fno)) >= 0 &&
	  p != 4 && p != 6 && p != 7);
}

static isrsect(EXPR *x)
{
  int p;
  return (x->fno == APPOP && x->data.args.x1->fno == APPOP &&
	  x->data.args.x1->data.args.x1->fno == FLIPOP &&
	  x->data.args.x1->data.args.x2->fno != MINOP &&
	  (p = opprec(x->data.args.x1->data.args.x2->fno)) >= 0 &&
	  p != 4 && p != 6 && p != 7);
}

/* exprprec(): return precedence of an expression */

static exprprec(EXPR *x)
{
  int		p;

  switch (x->fno) {
    /* applications */
  case APPOP:
    if (islsect(x) || isrsect(x))
      return 7; /* section */
    else if (x->data.args.x1->fno == APPOP)
      if ((p = opprec(x->data.args.x1->data.args.x1->fno))
	  != -1)
	return (p);	/* binary op */
      else
	return (6);	/* @ */
    else if ((p = opprec(x->data.args.x1->fno)) == 4 || p == 7)
      return (p);		/* unary op */
    else
      return (6);		/* @ */
    /* atoms: */
  case INTVALOP:
    return (mpz_sgn(x->data.z) < 0 ? 4 : 7);
    /* 4 = prec. of unary minus */
  case FLOATVALOP:
    return (x->data.f < 0.0 ? 4 : 7);
  default:
    return (7);
  }
}

/* pretty-printing operations: */

static print(int level, EXPR *x);

static printlprec(int level, int fno, EXPR *x)
{
  if (x)
    if (opprec(fno) > exprprec(x))
      return ((*putstr) ("(") && print(level+1, x) && (*putstr) (")"));
    else
      return (print(level, x));
  else
    return 1;
}

static printrprec(int level, int fno, EXPR *x)
{
  if (x)
    if (opprec(fno) >= exprprec(x))
      return ((*putstr) ("(") && print(level+1, x) && (*putstr) (")"));
    else
      return (print(level, x));
  else
    return 1;
}

static print1(int level, int fno, EXPR *x)
{
  return ((*putstr) (opstr(fno,0,0)) && printlprec(level, fno, x));
}

static print2l(int level, int fno, EXPR *x, EXPR *y)
{
  return (printlprec(level, fno, x) &&
	  (*putstr) (opstr(fno,x!=NULL,y!=NULL)) &&
	  printrprec(level, fno, y));
}

static print2r(int level, int fno, EXPR *x, EXPR *y)
{
  return (printrprec(level, fno, x) &&
	  (*putstr) (opstr(fno,x!=NULL,y!=NULL)) &&
	  printlprec(level, fno, y));
}

static print2n(int level, int fno, EXPR *x, EXPR *y)
{
  return (printrprec(level, fno, x) &&
	  (*putstr) (opstr(fno,x!=NULL,y!=NULL)) &&
	  printrprec(level, fno, y));
}

static print2(int level, int fno, EXPR *x, EXPR *y)
{
  switch (opprec(fno)) {
    /* relational ops (non-associative): */
  case 1:
    return (print2n(level, fno, x, y));
    /* sub/superscript (right-associative): */
  case 5:
    return (print2r(level, fno, x, y));
    /* others (left-associative): */
  default:
    return (print2l(level, fno, x, y));
  }
}

/* print an expression (outline only if maxlevel > 0), also truncate strings
   and lists/tuples to length maxchars and maxlist if nonzero,
   respectively) */

static char	s1[MAXSTRLEN], s2[MAXSTRLEN];

static char *strunc(char *s)
{
  /* truncate a string to maximum length */
  if (maxchars > 0 && strlen(s) > maxchars)
    strcpy(s+(maxchars-3), "...");
  return s;
}

static print(int level, EXPR *x)
{
  THREAD *thr = get_thr();
  if (maxlevel > 0 && level >= maxlevel)
    return (*putstr) ("...");
  else if (islsect(x))
    if (maxlevel > 0 && level+1 >= maxlevel)
      return (*putstr) ("(...)");
    else
      return (*putstr) ("(") &&
	print2(level+1, x->data.args.x1->fno, x->data.args.x2, NULL) &&
	(*putstr) (")");
  else if (isrsect(x))
    if (maxlevel > 0 && level+1 >= maxlevel)
      return (*putstr) ("(...)");
    else
      return (*putstr) ("(") &&
	print2(level+1, x->data.args.x1->data.args.x2->fno, NULL,
	       x->data.args.x2) &&
	(*putstr) (")");
  switch (x->fno) {
  /* atoms: */
  case INTVALOP: {
    int l = mpz_sizeinbase(x->data.z, (imode==1)?16:(imode==2)?8:10)+1;
    if (mpz_sgn(x->data.z) < 0) l++;
    if (l >= MAXSTRLEN) {
      /* LARGE buffer needed */
      char *s1 = malloc(l*sizeof(char));
      if (s1) {
	int ret = (*putstr) (pmpz(s1, x->data.z));
	free(s1);
	return ret;
      } else {
	thr->qmstat = MEM_OVF;
	return 0;
      }
    } else
      return ((*putstr) (pmpz(s1, x->data.z)));
  }
  case FLOATVALOP:
    return ((*putstr) (pfloat(s1, x->data.f)));
  case STRVALOP:
    if (strlen(x->data.s)*4+2 >= MAXSTRLEN) {
      /* LARGE buffer needed */
      /* Note: Since s may contain non-printable characters,
	 the representation of s may be as large as
	 strlen(s)*4+2 + 1 for \0 */
      char *s1 = malloc((strlen(x->data.s)*4+3)*
			sizeof(char));
      char *s2 = malloc((strlen(x->data.s)*4+3)*
			sizeof(char));
      if (!s1 || !s2) {
	if (s1) free(s1);
	thr->qmstat = MEM_OVF;
	return (0);
      } else if ((*putstr) ((sprintf(s1, "\"%s\"",
				     strunc(pstr(s2, x->data.s))), s1))) {
	free(s1); free(s2);
	return (1);
      } else {
	free(s1); free(s2);
	return (0);
      }
    } else
      /* default buffer is large enough */
      return ((*putstr) ((sprintf(s1, "\"%s\"",
				  strunc(pstr(s2, x->data.s))), s1)));
  case FILEVALOP:
  case BADFILEVALOP:
  case USRVALOP:
    return ((*putstr) ((sprintf(s1, "<<%s>>",
				x->type?pname(s2, x->type):
				"???"), s1)));
  /* [|]: */
  case CONSOP: {
    int count = 1;
    if (maxlevel > 0 && level+1 >= maxlevel)
      return (*putstr) ("[...]");
    if (!(*putstr) ("[") || !print(level+1, x->data.args.x1))
      return (0);
    x = x->data.args.x2;
    if (maxlist > 0)
      while (x->fno == CONSOP && count < maxlist) {
	if (!(*putstr) (",") || !print(level+1, x->data.args.x1))
	  return (0);
	x = x->data.args.x2;
	count++;
      }
    else
      while (x->fno == CONSOP) {
	if (!(*putstr) (",") || !print(level+1, x->data.args.x1))
	  return (0);
	x = x->data.args.x2;
      }
    if (x->fno == NILOP)
      return ((*putstr) ("]"));
    else if (maxlist > 0 && count == maxlist)
      if (x->fno == CONSOP)
	return ((*putstr) (",...]"));
      else
	return ((*putstr) ("|...]"));
    else
      return ((*putstr) ("|") && print(level+1, x) && (*putstr) ("]"));
  }
  /* (|): */
  case PAIROP: {
    int count = 1;
    char *lb = "(", *rb = ")";
    if (x->data.args.x2->fno == VOIDOP &&
	exprprec(x->data.args.x1) < 7) {
      lb = "(("; rb = "))";
    }
    if (maxlevel > 0 && level+1 >= maxlevel)
      return (*putstr) ("(...)");
    if (!(*putstr) (lb) || !print(level+1, x->data.args.x1))
      return (0);
    x = x->data.args.x2;
    if (maxlist > 0)
      while (x->fno == PAIROP && count < maxlist) {
	if (!(*putstr) (",") || !print(level+1, x->data.args.x1))
	  return (0);
	x = x->data.args.x2;
	count++;
      }
    else
      while (x->fno == PAIROP) {
	if (!(*putstr) (",") || !print(level+1, x->data.args.x1))
	  return (0);
	x = x->data.args.x2;
      }
    if (x->fno == VOIDOP)
      return ((*putstr) (rb));
    else if (maxlist > 0 && count == maxlist)
      if (x->fno == PAIROP)
	return ((*putstr) (",...") && (*putstr) (rb));
      else
	return ((*putstr) ("|...") && (*putstr) (rb));
    else
      return ((*putstr) ("|") && print(level+1, x) && (*putstr) (rb));
  }
  /* @: */
  case APPOP:
    if (x->data.args.x1->fno == APPOP)
      switch (opprec(x->data.args.x1->data.args.x1->fno)) {
	/* @: */
      case -1: case 4: case 6: case 7:
	return print2l(level, APPOP, x->data.args.x1,
		       x->data.args.x2);
      default:
	return print2(level, x->data.args.x1->data.args.x1->fno,
		      x->data.args.x1->data.args.x2,
		      x->data.args.x2);
      }
    else if (opprec(x->data.args.x1->fno) == 4 ||
	     opprec(x->data.args.x1->fno) == 7)
      /* unary operators: */
      return (print1(level, x->data.args.x1->fno,
		     x->data.args.x2));
    else
      /* @: */
      return (print2l(level, APPOP, x->data.args.x1,
		      x->data.args.x2));
  /* vectors: */
  case VECTOP: {
    int n = x->data.vect.n;
    if (n == 0)
      return ((*putstr) (pname(s1, VOIDOP)));
    else if (maxlevel > 0 && level+1 >= maxlevel)
      return (*putstr) ("(...)");
    else if (n == 1)
      if (exprprec(x->data.vect.xv[0]) < 7)
	return ((*putstr) ("((") && print(level+1, x->data.vect.xv[0]) &&
		(*putstr) ("))"));
      else
	return ((*putstr) ("(") && print(level+1, x->data.vect.xv[0]) &&
		(*putstr) (")"));
    else {
      int i;
      if (!(*putstr) ("(") || !print(level+1, x->data.vect.xv[0]))
	return 0;
      if (maxlist > 0) {
	for (i = 1; i < maxlist && i < n; i++)
	  if (!(*putstr) (",") || !print(level+1, x->data.vect.xv[i]))
	    return 0;
	if (i < n)
	  return ((*putstr) (",...)"));
	else
	  return ((*putstr) (")"));
      } else {
	for (i = 1; i < n; i++)
	  if (!(*putstr) (",") || !print(level+1, x->data.vect.xv[i]))
	    return 0;
	return ((*putstr) (")"));
      }
    }
  }
  /* other symbols: */
  default:
    return ((*putstr) (pname(s1, x->fno)));
  }
}

/* sprintx() unparses an expression to a null-terminated string. */

char *sprintx(EXPR *x)
{
  int 		retval;
  if (!(bufp = malloc(sizeof(char)*MAXSTRLEN)))
    return (NULL);
  bufleng = 0;
  abufleng = MAXSTRLEN;
  putstr = sputstr;
  retval = print(0, x);
  if (retval) {
    if (!(bufp = realloc(bufp, sizeof(char)*(strlen(bufp)+1))))
      fatal(qmmsg[THIS_CANT_HAPPEN]);
    return (bufp);
  } else {
    return (NULL);
  }
}

/* fprintx() unparses an expression to a file. It returns zero unless
   an error is encountered writing to the file. */

int fprintx(FILE *_fp, EXPR *x)
{
  int 		retval;
  fp = _fp;
  putstr = fputstr;
  retval = print(0, x);
  return (retval || checkbrk);
}

/* printx() unparses an expression to stdout. */

int printx(EXPR *x)
{
  return (fprintx(stdout, x));
}
