/*

qcread.c Connectix QuickCam hardware interface

Copyright (C) 1996, 1997 Alex Belits

This software contains ideas and modified parts from qcam,
written by Scott Laird.

Copyright (C) 1996 by Scott Laird

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL SCOTT LAIRD OR ALEX BELITS BE LIABLE FOR ANY CLAIM,
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "cqc.h"

#define VERSION "0.3"
#define RELEASEDATE "Dec  8 1997"

char *versionline=
"qcread version " VERSION "\n"
"released " RELEASEDATE ", compiled "__DATE__" " __TIME__"\n";

char *helpmessage=
"Connectix QuickCam image reader, available options:\n"
"-V, --version,         print version number\n"
"-f, --config FILE      config file name FILE\n"
"                       (default "DEFAULTCONFIGFILENAME")\n"
"--camera NAME          camera name NAME\n"
"-p, --port N           port number N (0xNNN - hex, NNN - decimal)\n"
"--porttype TYPE        port type TYPE (unknown, unidirectional\n"
"                       or bidirectional)\n"
"-b, --brightness N     brightness N (0-255)\n"
"-c, --contrast N       contrast N (0-255)\n"
"--white N              white level N (0-255)\n"
"-w, --blackoffset N    black offset N (0-255)\n"
"--saturation N         saturation N (0-255)\n"
"--hue N                hue N (0-255)\n"
"--red N                red correction (0.004-1-255)\n"
"--green N              green correction (0.004-1-255)\n"
"--blue N               blue correction (0.004-1-255)\n"
"--black N              black level N (0-255)\n"
"-l, --xoffset N        left image edge offset (0-326)\n"
"-t, --yoffset N        top image edge offset (0-326)\n"
"-x, --width N          image width (2-328)\n"
"-y, --height N         image height (2-328)\n"
"-B, --bpp N            bits per pixel (4, 6 or 24)\n"
"-s, --transfer N       transfer scale (1, 2 or 4)\n"
"--direction DIRECTION  camera direction (up, down, right or left)\n"
"--priority N           scanning priority (\"niceness\" value)\n"
"-v, --verbose          verbose mode\n"
"-h, --help             this message\n"
;

int main(int argc,char **argv,char **env){
int option_index,curropt,verboseflag=0;
char *configfilename=NULL,*cameraname=NULL;

int bidiroverride=0, blackoverride=0, bppoverride=0, brightnessoverride=0,
 contrastoverride=0, directionoverride=0, ysizeoverride=0, hueoverride=0,
 portoverride=0, saturationoverride=0, decimoverride=0, whiteoverride=0,
 blackoffsetoverride=0, xsizeoverride=0, xoverride=0, yoverride=0,
 redoverride=0, greenoverride=0, blueoverride=0, priorityoverride=0;

int bidirvalue=0, blackvalue=0, bppvalue=0, brightnessvalue=0,
 contrastvalue=0, directionvalue=0, ysizevalue=0, huevalue=0,
 portvalue=0, saturationvalue=0, decimvalue=0, whitevalue=0,
 blackoffsetvalue=0, xsizevalue=0, xvalue=0, yvalue=0, priorityvalue=0;

double redvalue=1,greenvalue=1,bluevalue=1;

struct option longoptions[]={
 {"version",0,0,0},
 {"config",1,0,0},
 {"camera",1,0,0},
 {"port",1,0,0},
 {"porttype",1,0,0},
 {"brightness",1,0,0},
 {"contrast",1,0,0},
 {"white",1,0,0},
 {"blackoffset",1,0,0},
 {"saturation",1,0,0},
 {"hue",1,0,0},
 {"black",1,0,0},
 {"yoffset",1,0,0},
 {"xoffset",1,0,0},
 {"height",1,0,0},
 {"width",1,0,0},
 {"bpp",1,0,0},
 {"transfer",1,0,0},
 {"direction",1,0,0},
 {"verbose",0,0,0},
 {"red",1,0,0},
 {"green",1,0,0},
 {"blue",1,0,0},
 {"priority",1,0,0},
 {"help",0,0,0},
 {0,0,0,0}
 };

struct qcam *qc;

qc=qccreate();
if(!qc){
 fputs("qcread: Insufficient memory\n",stderr);
 exit(1);
 }

do{
 option_index=0;
 curropt=getopt_long(argc,argv,"f:ut:l:x:y:p:B:c:w:b:Vs:vh",
  longoptions,&option_index);
 switch(curropt){
  case -1: break;
  case 0: 
   switch(option_index){
    case 0: /* version */
     fputs(versionline,stderr);
     return 0;
    case 1: /* config */
     configfilename=optarg;
     break;
    case 2: /* camera */
     cameraname=optarg;
     break;
    case 3: /* port */
     portoverride=1;
     if(!strncasecmp(optarg,"0x",2)){
      if(!sscanf(optarg+2,"%x",&portvalue)){
       portvalue=0;
       }
      }else{
      portvalue=atoi(optarg);
      }
     break;
    case 4: /* porttype */
     bidiroverride=1;
     if(!strcasecmp(optarg,"bidirectional")) bidirvalue=2;
      else
      if(!strcasecmp(optarg,"unidirectional")) bidirvalue=1;
       else
       if(!strcasecmp(optarg,"unknown")) bidirvalue=0;
        else{
        fprintf(stderr,"wrong value for --porttype: \"%s\"\n",optarg);
        return 1;
        }
     break;
    case 5: /* brightness */
     brightnessoverride=1;
     brightnessvalue=atoi(optarg);
     break;
    case 6: /* contrast */
     contrastoverride=1;
     contrastvalue=atoi(optarg);
     break;
    case 7: /* white */
     whiteoverride=1;
     whitevalue=atoi(optarg);
     break;
    case 8: /* blackoffset */
     blackoffsetoverride=1;
     blackoffsetvalue=atoi(optarg);
     break;
    case 9: /* saturation */
     saturationoverride=1;
     saturationvalue=atoi(optarg);
     break;
    case 10: /* hue */
     hueoverride=1;
     huevalue=atoi(optarg);
     break;
    case 11: /* black */
     blackoverride=1;
     blackvalue=atoi(optarg);
     break;
    case 12: /* yoffset */
     yoverride=1;
     yvalue=atoi(optarg);
     break;
    case 13: /* xoffset */
     xoverride=1;
     xvalue=atoi(optarg);
     break;
    case 14: /* height */
     ysizeoverride=1;
     ysizevalue=atoi(optarg);
     break;
    case 15: /* width */
     xsizeoverride=1;
     xsizevalue=atoi(optarg);
     break;
    case 16: /* bpp */
     bppoverride=1;
     bppvalue=atoi(optarg);
     break;
    case 17: /* transfer */
     decimoverride=1;
     decimvalue=atoi(optarg);
     break;
    case 18: /* direction */
     directionoverride=1;
     if(!strcasecmp(optarg,"up")) directionvalue=0;
      else
      if(!strcasecmp(optarg,"down")) directionvalue=2;
       else
       if(!strcasecmp(optarg,"left")) directionvalue=3;
        else
        if(!strcasecmp(optarg,"right")) directionvalue=1;
	else{
        fprintf(stderr,"wrong value for --direction: \"%s\"\n",optarg);
        return 1;
	}
     break;
    case 19: /* verbose */
     verboseflag=1;
     break;
    case 20: /* red */
     if(sscanf(optarg,"%lf",&redvalue)){
      redoverride=1;
      }else{
      fprintf(stderr,"wrong value for --red: \"%s\"\n",optarg);
      }
     break;
    case 21: /* green */
     if(sscanf(optarg,"%lf",&greenvalue)){
      greenoverride=1;
      }else{
      fprintf(stderr,"wrong value for --green: \"%s\"\n",optarg);
      }
     break;
    case 22: /* blue */
     if(sscanf(optarg,"%lf",&bluevalue)){
      blueoverride=1;
      }else{
      fprintf(stderr,"wrong value for --blue: \"%s\"\n",optarg);
      }
     break;
    case 23: /* priority */
     if(sscanf(optarg,"%d",&priorityvalue)){
      priorityoverride=1;
      }else{
      fprintf(stderr,"wrong value for --priority: \"%s\"\n",optarg);
      }
     break;
    case 24: /* help */
     fputs(helpmessage,stderr);
     return 0;
    }
   break;
  case 'f':
   configfilename=optarg;
   break;
  case 'u':
   bidiroverride=1;
   bidirvalue=1;
   break;
  case 't':
   yoverride=1;
   yvalue=atoi(optarg);
   break;
  case 'l':
   xoverride=1;
   xvalue=atoi(optarg);
   break;
  case 'x':
   xsizeoverride=1;
   xsizevalue=atoi(optarg);
   break;
  case 'y':
   ysizeoverride=1;
   ysizevalue=atoi(optarg);
   break;
  case 'p':
   portoverride=1;
   if(!strncasecmp(optarg,"0x",2)){
    if(!sscanf(optarg+2,"%x",&portvalue)){
     portvalue=0;
     }
    }else{
    portvalue=atoi(optarg);
    }
   break;
  case 'B':
   bppoverride=1;
   bppvalue=atoi(optarg);
   break;
  case 'c':
   contrastoverride=1;
   contrastvalue=atoi(optarg);
   break;
  case 'w':
   blackoffsetoverride=1;
   blackoffsetvalue=atoi(optarg);
   break;
  case 'b':
   brightnessoverride=1;
   brightnessvalue=atoi(optarg);
   break;
  case 'V':
   fputs(versionline,stderr);
  return 0;
  case 's':
   decimoverride=1;
   decimvalue=atoi(optarg);
   break;
  case 'v':
   verboseflag=1;
   break;
  case 'h':
   fputs(helpmessage,stderr);
  return 0;
  default:
   return 1;
  }

 }while(curropt>=0);

if(optind<argc){
 fprintf(stderr,"wrong argument \"%s\"\n",argv[optind]);
 return 1;
 }
qcreadconfigfile(qc,configfilename,cameraname,0);

if(portoverride){
 qc->p0=portvalue;
 qc->p1=portvalue+1;
 qc->p2=portvalue+2;
 }
if(bidiroverride) qc->bidir=bidirvalue;
if(blackoverride) qc->black=blackvalue;
if(bppoverride) qc->bpp=bppvalue;
if(brightnessoverride) qc->brightness=brightnessvalue;
if(contrastoverride) qc->contrast=contrastvalue;
if(directionoverride) qc->direction=directionvalue;
if(ysizeoverride) qc->ysize=ysizevalue;
if(hueoverride) qc->hue=huevalue;
if(saturationoverride) qc->saturation=saturationvalue;
if(decimoverride) qc->decim=decimvalue;
if(whiteoverride) qc->white=whitevalue;
if(blackoffsetoverride) qc->blackoffset=blackoffsetvalue;
if(xsizeoverride) qc->xsize=xsizevalue;
if(xoverride) qc->x=xvalue;
if(yoverride) qc->y=yvalue;
if(redoverride) qc->red=redvalue;
if(greenoverride) qc->green=greenvalue;
if(blueoverride) qc->blue=bluevalue;
if(priorityoverride) qc->scan_priority=priorityvalue;

switch(qcinitset(qc)){
 case -1:
  fputs("qcread: Initialization failed: Can't run as non-root\n",stderr);
  exit(1);
 case -2:
  perror("qcread: Initialization failed");
  exit(1);
 case -3:
  fputs("qcread: Initialization failed: QuickCam not detected\n",stderr);
  exit(1);
 case -4:
  fprintf(stderr,"qcread: Initialization failed: QuickCam not detected or unsupported type\n");
  exit(1);
 case -5:
  fputs("qcread: Initialization failed: QuickCam doesn't work\n",stderr);
  exit(1);
 case -6:
  fputs("qcread: Image setup failed\n",stderr);
  exit(1);
 case -7:
  fputs("qcread: Invalid frame parameters\n",stderr);
  exit(1);
 case -8:
  fputs("qcread: Frame setup failed\n",stderr);
  exit(1);
 case -9:
  fputs("qcread: Frame memory allocation failed\n",stderr);
  exit(1);
 case 0: 
  if(verboseflag){
   fprintf(stderr,"%sdirectional parallel port at 0x%03x with %s Quickcam camera\n",
    qc->bidir==2?"Bi":"Uni",qc->p0,(qc->camv==QCAM_VER_COLOR)?"color":"b&w");
   }
  break;
 default: 
  fprintf(stderr,"qcread: Unknown error\n");
  exit(1);
  }
if(verboseflag) fputs("qcread: Reading frame\n",stderr);
if(qcreadframe(qc)){
 fputs("qcread: Frame read failed\n",stderr);
 exit(1);
 }
if(qc->color){
 printf("P6\n%d %d 255\n",qc->xsize/qc->decim,qc->ysize/qc->decim);
 }else{
 printf("P5\n%d %d %d\n",qc->xsize/qc->decim,qc->ysize/qc->decim,(qc->bpp==4)?15:63);
 }
fflush(stdout);
write(1,qc->frame,qc->framesize);
return 0;
}
