/*
 * Copyright (c) 1998-2001 Caucho Technology -- all rights reserved
 *
 * Caucho Technology permits redistribution, modification and use
 * of this file in source and binary form ("the Software") under the
 * Caucho Developer Source License ("the License").  The following
 * conditions must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Redistributions of the Software in source or binary form must include 
 *    an unmodified copy of the License, normally in a plain ASCII text
 *
 * 3. The names "Resin" or "Caucho" are trademarks of Caucho Technology and
 *    may not be used to endorse products derived from this software.
 *    "Resin" or "Caucho" may not appear in the names of products derived
 *    from this software.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 *
 * CAUCHO TECHNOLOGY AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL CAUCHO OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.      
 */

package example.servlet.basic;

import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;

/**
 * Shows how to get initialization parameters from the configuration file.
 *
 * <p>The following is a sample configuration for the examples.  It uses
 * Resin's short syntax which is more scannable and maintainable.
 *
 * <pre>
 * &lt;web-app>
 *
 * &lt;context-param context-a='context-value-a'/>
 *
 * &lt;servlet servlet-name='servlet-a'
 *          servlet-class='example.servlet.basic.InitServlet'>
 *   &lt;init-param init-a='value-a'/>
 * &lt;/servlet>
 *
 * &lt;servlet servlet-name='servlet-b'
 *          servlet-class='example.servlet.basic.InitServlet'>
 *   &lt;init-param init-a='value-b'/>
 * &lt;/servlet>
 *
 * &lt;servlet-mapping url-pattern='/servlet/*'
 *                  servlet-name='invoker'/>
 *
 * &lt;/web-app>
 * </pre>
 *
 * <p>The specific servlet instance depends on the URL.  In the above example,
 * there are three URLs that will use the InitServlet, but all three will
 * use different servlet instances.
 *
 * <ul>
 * <li>/servlet/servlet-a will use servlet-a defined above.  init-a
 * will be 'value-a'.
 * <li>/servlet/servlet-b will use servlet-b defined above.  init-a
 * will be 'value-b'.
 * <li>/servlet/example.servlet.basic.InitServlet will use an unnamed
 * servlet.  init-a will be null.
 * </ul>
 *
 * <p>The example includes a count variable so you can see how the servlet
 * instances are reused.  Since Resin only creates a single servlet instance,
 * servlet programmers must be aware of threading issues.
 */
public class InitServlet extends HttpServlet {
  // The web-app configuration variable
  private String contextA;
  
  // The servlet configuration variable
  private String initA;

  // A counting variable for the instance
  private static int count;

  // The instance id
  private int id;
  
  /**
   * The <code>init()</code> method is called when the servlet is
   * initialized.  For most servlets, it will be called on the first
   * request to the servlet.  For load-on-startup servlets, it will be
   * called when the web-app initializes.
   *
   * <p>Servlets normally use init() to cache initial lookups.  A typical
   * example is caching a lookup of a database DataSource or an EJB home.
   */
  public void init()
    throws ServletException
  {
    // Gets the application object for the web-app.
    // There is exactly one servletContext for each web-app.
    ServletContext application = getServletContext();

    // The application contains the context-param variables.
    contextA = application.getInitParameter("context-a");

    // Gets a servlet initialization parameter
    initA = getInitParameter("init-a");

    this.id = count++;
  }

  /**
   * Handles GET requests.  Resin will call the doGet method when
   * the browser sends a GET request.
   *
   * @param request the request object contains the data from
   * the browser's request.
   * @param response the response object contains methods to send
   * data back to the browser.
   */
  public void doGet(HttpServletRequest request, HttpServletResponse response)
    throws IOException, ServletException
  {
    // Tells the browser that the data is HTML
    response.setContentType("text/html");
    
    PrintWriter pw = response.getWriter();

    pw.println("<title>Init Parameters</title>");

    pw.println("<table>");

    // Returns the servlet's init parameter
    pw.println("<tr><td>init-a<td>" + initA);
    
    // Returns the application's init parameter
    pw.println("<tr><td>context-a<td>" + contextA);
    
    // Returns the instance id.  This just identifies the particular
    // servlet instance.  The same servlet instance is used
    // for each request.  This means, in particular, that servlets
    // need to be aware of threading.
    pw.println("<tr><td>id<td>" + id);
    
    pw.println("</table>");
  }
}

