/*
 * Copyright (c) 1999 Caucho Technology.  All rights reserved.
 *
 * Caucho Technology permits redistribution, modification and use
 * of this file in source and binary form ("the Software") under the
 * Caucho Developer Source License ("the License").  In particular, the following
 * conditions must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Redistributions of the Software in source or binary form must include 
 *    an unmodified copy of the License, normally in a plain ASCII text
 *
 * 3. The names "Resin" or "Caucho" are trademarks of Caucho Technology and
 *    may not be used to endorse products derived from this software.
 *    "Resin" or "Caucho" may not appear in the names of products derived
 *    from this software.
 *
 * 4. Caucho Technology requests that attribution be given to Resin
 *    in any manner possible.  We suggest using the "Resin Powered"
 *    button or creating a "powered by Resin(tm)" link to
 *    http://www.caucho.com for each page served by Resin.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.

 * CAUCHO TECHNOLOGY AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL CAUCHO OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.
 *
 * @author Scott Ferguson
 *
 * $Id: stream.c,v 1.2 2000/10/13 01:50:38 ferg Exp $
 */

#include <stdlib.h>
#include <errno.h>
#include "rws.h"

void
rws_close(stream_t *s)
{
  if (s->fd >= 0) {
    rws_flush(s);
    close(s->fd);
  }
  s->fd = -1;
}

int
rws_open(stream_t *s, int fd)
{
  memset(s, 0, sizeof(*s));
  s->fd = fd;

  s->write_length = 0;
  s->read_length = 0;
  s->read_offset = 0;
}

int
rws_flush(stream_t *s)
{
  int len;

  if (s->write_length > 0) {
    len = write(s->fd, s->write_buf, s->write_length);
    if (len != s->write_length)
      return -1;

    s->write_length = 0;
  }

  return 0;
}

static int
rws_fill_buffer(stream_t *s)
{
  int i = 0;
  
  if (rws_flush(s) < 0)
    return -1;

  do {
    s->read_length = read(s->fd, s->read_buf, BUF_LENGTH);
  } while (s->read_length < 0 && errno == EINTR && i++ < 10);

  if (s->read_length <= 0) {
    rws_close(s);
    
    return -1;
  }
  
  s->read_offset = 0;
  
  return 0;
}

int
rws_read_byte(stream_t *s)
{
  if (s->read_offset >= s->read_length) {
    if (rws_fill_buffer(s) < 0)
      return -1;
  }

  return s->read_buf[s->read_offset++];
}

void
rws_write(stream_t *s, const char *buf, int length)
{
  /* XXX: writev??? */

  if (s->write_length + length > BUF_LENGTH) {
    if (s->write_length > 0) {
      if (write(s->fd, s->write_buf, s->write_length) < 0)
	rws_close(s);

      s->write_length = 0;
    }

    if (length > BUF_LENGTH) {
      if (write(s->fd, buf, length, 0) < 0)
	rws_close(s);

      return;
    }
  }

  memcpy(s->write_buf + s->write_length, buf, length);
  s->write_length += length;
}

int
rws_read_all(stream_t *s, char *buf, int len)
{
  while (len > 0) {
    int sublen;

    if (s->read_offset >= s->read_length) {
      if (rws_fill_buffer(s) < 0)
	return -1;
    }

    sublen = s->read_length - s->read_offset;
    if (len < sublen)
      sublen = len;

    memcpy(buf, s->read_buf + s->read_offset, sublen);

    buf += sublen;
    len -= sublen;
    s->read_offset += sublen;
  }

  return 1;
}
