/*
 * Copyright (c) 1999-2000 Caucho Technology.  All rights reserved.
 *
 * Caucho Technology permits redistribution, modification and use
 * of this file in source and binary form ("the Software") under the
 * Caucho Public License ("the License").  In particular, the following
 * conditions must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Redistributions of the Software in source or binary form must include 
 *    an unmodified copy of the License, normally in a plain ASCII text
 *
 * 3. The names "Resin" or "Caucho" are trademarks of Caucho Technology and
 *    may not be used to endorse products derived from this software.
 *    "Resin" or "Caucho" may not appear in the names of products derived
 *    from this software.
 *
 * 4. Caucho Technology requests that attribution be given to Resin
 *    in any manner possible.  We suggest using the "Resin Powered"
 *    button or creating a "powered by Resin(tm)" link to
 *    http://www.caucho.com for each page served by Resin.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.

 * CAUCHO TECHNOLOGY AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL Caucho OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.      
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>

#include <pthread.h>

#include "cse.h"
#include "proxy.h"

#define READ(req) ((req->offset < req->length) ? \
                   req->buffer[req->offset++] :  \
                   request_read_byte(req))

int_map_t *g_header_map;

static void
init_header_map()
{
  g_header_map = int_map_create();

  int_map_set(g_header_map, "content-length", CONTENT_LENGTH);
  int_map_set(g_header_map, "host", HOST);
  int_map_set(g_header_map, "connection", CONNECTION);
}

request_t *
request_create()
{
  request_t *req;

  if (! g_header_map) {
    init_header_map();
  }

  req = (request_t *) malloc(sizeof(request_t));

  memset(req, 0, sizeof(request_t));
  
  req->capacity = 8192;
  req->buffer = malloc(req->capacity);
  
  return req;
}

int
request_fill(request_t *req)
{
  if (req->sock < 0)
    return -1;

  req->offset = 0;
  req->length = recv(req->sock, req->buffer, req->capacity, 0);
  
  return req->length;
}

static int
request_read_byte(request_t *req)
{
  if (req->offset < req->length)
    return req->buffer[req->offset++];

  if (request_fill(req) <= 0)
    return -1;
  else
    return req->buffer[req->offset++];
}

static void
request_parse_uri(request_t *req)
{
  int ch;
  int i;

  for (ch = READ(req); isspace(ch); ch = READ(req)) {
  }

  for (i = 0; ch > 0 && ! isspace(ch); ch = READ(req)) {
    if (i < sizeof(req->method) - 1)
      req->method[i++] = ch;
  }
  req->method[i] = 0;

  for (; isspace(ch); ch = READ(req)) {
  }

  for (i = 0; ch > 0 && ! isspace(ch); ch = READ(req)) {
    if (i < sizeof(req->uri) - 1)
      req->uri[i++] = ch;
  }
  req->uri[i] = 0;
  
  for (; ch == ' '; ch = READ(req)) {
  }

  for (i = 0; ch > 0 && ch != '\n'; ch = READ(req)) {
    if (i < sizeof(req->protocol) - 1 && ch != '\n')
      req->protocol[i++] = ch;
  }
  req->protocol[i] = 0;
  if (req->protocol[i - 1] == '\r') {
    req->protocol[i - 1] = 0;
    i--;
  }
  
  if (i == 0) {
    req->version = HTTP_0_9;
    req->keepalive = 0;
  }
  else if (req->protocol[i - 1] >= '1') {
    req->version = HTTP_1_1;
    req->keepalive = 1;
  }
  else {
    req->version = HTTP_1_0;
    req->keepalive = 0;
  }
}

static void
request_parse_headers(request_t *req)
{
  int i = 0;
  int ch = READ(req);

  while (ch != '\n' && ch != '\r') {
    char *key;
    char *value;
    int code;

    key = req->headers + i;
    
    req->header_keys[req->header_size] = i;

    for (; ch > 0 && ch != ' ' && ch != ':'; ch = READ(req)) {
      if (i < sizeof(req->headers) - 1)
        req->headers[i++] = ch;
    }
    req->headers[i++] = 0;

    value = req->headers + i;
    
    req->header_values[req->header_size] = i;
    for (; ch == ' ' || ch == ':'; ch = READ(req)) {
    }

    for (; ch > 0 && ch != '\r' && ch != '\n'; ch = READ(req)) {
      if (i < sizeof(req->headers) - 1)
        req->headers[i++] = ch;
    }
    req->headers[i++] = 0;

    if (req->header_size < 250)
      req->header_size++;

    code = int_map_get(g_header_map, key);
    if (code >= 0) {
      switch (code) {
      case CONTENT_LENGTH:
        req->content_length = atoi(value);
        break;
        
      case CONNECTION:
        if (! strcasecmp(value, "Close"))
          req->keepalive = 0;
        break;
      }
    }
    
    if (ch == '\r')
      ch = READ(req);
    
    if (ch == '\n')
      ch = READ(req);
  }

  req->header_keys[req->header_size] = i;
  req->header_values[req->header_size] = i;
  req->header_length = i;

  if (ch == '\r')
    ch = READ(req);
}

void
request_parse(request_t *req, int sock)
{
  req->sock = sock;
  
  request_parse_uri(req);
  
  req->header_size = 0;
  req->keepalive = 1;
  req->content_length = 0;
  req->content_offset = 0;

  if (req->version >= HTTP_1_0)
    request_parse_headers(req);
}
