/*
 * Copyright (c) 1998-2002 Caucho Technology -- all rights reserved
 *
 * Caucho Technology permits modification and use of this file in
 * source and binary form ("the Software") subject to the Caucho
 * Developer Source License 1.1 ("the License") which accompanies
 * this file.  The License is also available at
 *   http://www.caucho.com/download/cdsl1-1.xtp
 *
 * In addition to the terms of the License, the following conditions
 * must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Each copy of the Software in source or binary form must include 
 *    an unmodified copy of the License in a plain ASCII text file named
 *    LICENSE.
 *
 * 3. Caucho reserves all rights to its names, trademarks and logos.
 *    In particular, the names "Resin" and "Caucho" are trademarks of
 *    Caucho and may not be used to endorse products derived from
 *    this software.  "Resin" and "Caucho" may not appear in the names
 *    of products derived from this software.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 *
 * CAUCHO TECHNOLOGY AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL CAUCHO OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.      
 *
 * @author Scott Ferguson
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netdb.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <dirent.h>
/* probably system-dependent */
#include <dlfcn.h>
#include <jni.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include <signal.h>

#include "cse.h"
#include "version.h"
#include "resin.h"
#include "resin_jni.h"

static config_t *g_config;

static int g_port_count;
static server_socket_t *g_ports[256];

#define MAX_OPTIONS 256

typedef struct options_t {
  char *resin_home;
  char *server_root;
  char *java_home;
  char *jvm_type;
  char *conf;
  char *classpath;
  char *loadpath;
  char *stdout_path;
  char *stderr_path;
  char *pid;

  char *server;

  int verbose;
  
  int start;
  int stop;
  int restart;
  int is_child;

  JavaVMOption options[MAX_OPTIONS];
  int n_options;
} options_t;

static options_t g_options;
java_t g_java;

void
cse_log(char *fmt, ...)
{
#ifdef DEBUG  
  va_list list;

  va_start(list, fmt);
  vfprintf(stderr, fmt, list);
  va_end(list);
#endif
}

void
cse_error(config_t *config, char *fmt, ...)
{
  va_list list;

  va_start(list, fmt);
  vfprintf(stderr, fmt, list);
  va_end(list);

  exit(1);
}

void
cse_set_socket_cleanup(int socket, void *pool)
{
}

void
cse_kill_socket_cleanup(int socket, void *pool)
{
}

int
cse_lock(void *lock)
{
  return 1;
}

void
cse_unlock(void *lock)
{
}

void *
cse_create_lock(config_t *config)
{
  return 0;
}

/**
 * Returns true if the path is an absolute path
 */
static int
is_path_absolute(char *path)
{
  return (path[0] == '/' || path[0] == '\\' ||
          (path[1] == ':' &&
           ((path[0] >= 'a' && path[0] <= 'z') ||
            (path[0] >= 'A' && path[0] <= 'Z'))));
}

/**
 * Add a jvm option
 */
static void
add_jvm_option(char *fmt, ...)
{
  char buf[16 * 1024];
  va_list args;

  va_start(args, fmt);
  vsprintf(buf, fmt, args);
  va_end(args);

  if (strlen(buf) >= sizeof(buf)) {
    fprintf(stderr, "argument was too big.");
    exit(1);
  }

  if (g_options.n_options >= MAX_OPTIONS) {
    fprintf(stderr, "too many options\n");
    exit(1);
  }
  
  g_options.options[g_options.n_options++].optionString = strdup(buf);
}

/**
 * Bind to a port.
 *
 * @param hostname name of the host interface
 * @param port port to bind
 *
 * @return server socket on success, -1 on failure.
 */
static int
cse_bind(char *hostname, int port)
{
  struct sockaddr_in sin;
  int val = 0;
  struct hostent *hostent;
  int sock;

  memset(&sin, 0, sizeof(sin));
  
  if (hostname && *hostname && *hostname != '*') {
    hostent = gethostbyname(hostname);
    if (! hostent || ! hostent->h_addr) {
      fprintf(stderr, "can't lookup host %s\n", hostname);
      return -1;
    }

    memcpy(&sin.sin_addr, hostent->h_addr, sizeof(struct in_addr));
  }
  sin.sin_port = htons(port);

  sock = socket(AF_INET, SOCK_STREAM, 0);
  
  val = 1;
  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &val, sizeof(int)) < 0) {
    fprintf(stderr, "internal error: bad set sock opt\n");
    return -1;
  }
  
  if (bind(sock, (struct sockaddr *) &sin, sizeof (sin)) < 0) {
    return -1;
  }

  listen(sock, 5);

  /* set non-blocking */
  /*
  flags = fcntl(sock, F_GETFL);
  fcntl(sock, F_SETFL, O_NONBLOCK|flags);
  */

  return sock;
}

static void
set_classpath()
{
  char classpath[16 * 1024];
  char path[16 * 1024];
  DIR *dir;
  struct dirent *dir_entry;

  classpath[0] = 0;
  
  if (g_options.classpath)
    strcat(classpath, g_options.classpath);
  
  if (getenv("CLASSPATH")) {
    strcat(classpath, ":");
    strcat(classpath, getenv("CLASSPATH"));
  }

  /* classes from resin/classes */
  sprintf(path, ":%s/classes", g_options.resin_home);
  strcat(classpath, path);

  /* grab jars from resin/lib */
  sprintf(path, "%s/lib", g_options.resin_home);
  dir = opendir(path);
  while (dir && (dir_entry = readdir(dir))) {
    int len = strlen(dir_entry->d_name);
    
    if (! strcmp(dir_entry->d_name + len - 4, ".jar") ||
        ! strcmp(dir_entry->d_name + len - 4, ".zip")) {
      strcat(classpath, ":");
      strcat(classpath, g_options.resin_home);
      strcat(classpath, "/lib/");
      strcat(classpath, dir_entry->d_name);
    }
  }
  if (dir)
    closedir(dir);

  sprintf(path, ":%s/jre/lib/rt.jar", g_options.java_home);
  strcat(classpath, path);

  sprintf(path, ":%s/lib/tools.jar", g_options.java_home);
  strcat(classpath, path);

  /*
   * Apparently, this doesn't get added to the list
   *
  sprintf(path, "%s/jre/lib/ext", g_options.java_home);
  dir = opendir(path);
  while ((dir_entry = readdir(dir))) {
    int len = strlen(dir_entry->d_name);
    
    if (! strcmp(dir_entry->d_name + len - 4, ".jar") ||
        ! strcmp(dir_entry->d_name + len - 4, ".zip")) {
      strcat(classpath, ":");
      strcat(classpath, g_options.java_home);
      strcat(classpath, "/jre/lib/ext/");
      strcat(classpath, dir_entry->d_name);
    }
  }
  closedir(dir);
  */

  g_options.classpath = strdup(classpath);
  
  add_jvm_option("-Djava.class.path=%s", classpath);
}

/**
 * Returns true if prefix + tail is a file.
 */
static int
is_file(char *prefix, char *tail)
{
  struct stat st;
  char buf[8192];

  sprintf(buf, "%s/%s", prefix, tail);

  return ! stat(buf, &st) && S_ISREG(st.st_mode);
}

/**
 * Sets the dynamic loading library path LD_LIBRARY_PATH so the JVM classes
 * will be automatically picked up.
 *
 * @param java_home the JAVA_HOME directory
 * @param resin_home the RESIN_HOME directory
 */
void
set_library_path(char *java_home, char *resin_home, char *type)
{
  char envpath[8192];
  char newpath[8192];

  newpath[0] = 0;

  if (getenv("LD_LIBRARY_PATH"))
    strcat(newpath, getenv("LD_LIBRARY_PATH"));

  strcat(newpath, ":");
  strcat(newpath, resin_home);
  strcat(newpath, "/libexec:");

  /* IBM JDK */
  if (is_file(java_home, "jre/bin/libjava.so")) {
    char jvmlib[8192];

    if (! type)
      type = "classic";

    sprintf(jvmlib, "jre/bin/%s/libjvm.so", type);

    if (! is_file(java_home, jvmlib)) {
      fprintf(stderr, "Can't load -%s JVM at %s/%s.\n",
              type, java_home, jvmlib);
      exit(1);
    }
    
    strcat(newpath, java_home);
    strcat(newpath, "/jre/bin/");
    strcat(newpath, type);
    strcat(newpath, ":");
    
    strcat(newpath, java_home);
    strcat(newpath, "/jre/bin");
  }
  /* Sun JDK */
  else if (is_file(java_home, "jre/lib/" CPU "/libjava.so")) {
    char jvmlib[8192];

    if (! type)
      type = "server";

    sprintf(jvmlib, "jre/lib/" CPU "/%s/libjvm.so", type);

    if (! is_file(java_home, jvmlib)) {
      fprintf(stderr, "Can't load -%s JVM at %s/%s.\n",
              type, java_home, jvmlib);
      exit(1);
    }

    strcat(newpath, java_home);
    strcat(newpath, "/jre/lib/" CPU "/");
    strcat(newpath, type);
    strcat(newpath, ":");

    strcat(newpath, java_home);
    strcat(newpath, "/jre/lib/" CPU "/native_threads:");
    
    strcat(newpath, java_home);
    strcat(newpath, "/jre/lib/" CPU);
  }
  else {
    fprintf(stderr, "can't use JAVA_HOME=%s\n", java_home);
    exit(1);
  }
  
  strcpy(envpath, "LD_LIBRARY_PATH=");
  strcat(envpath, newpath);

  if (strlen(envpath) > sizeof(envpath)) {
    fprintf(stderr, "path too long\n");
    exit(1);
  }
  
  putenv(strdup(envpath));
}

/**
 * Sets the dynamic loading library path LD_LIBRARY_PATH so the JVM classes
 * will be automatically picked up.
 *
 * @param java_home the JAVA_HOME directory
 * @param resin_home the RESIN_HOME directory
 */
void
set_jvmpath(char *java_home, char *resin_home, char *type, char *path)
{
  /* IBM JDK */
  if (is_file(java_home, "jre/bin/libjava.so")) {
    if (! type)
      type = "classic";

    sprintf(path, "%s/jre/bin/%s/libjvm.so", java_home, type);
  }
  /* Sun JDK */
  else if (is_file(java_home, "jre/lib/" CPU "/libjava.so")) {
    if (! type)
      type = "server";

    sprintf(path, "%s/jre/lib/" CPU "/%s/libjvm.so", java_home, type);
  }
  else {
    fprintf(stderr, "can't use JAVA_HOME=%s\n", java_home);
    exit(1);
  }
}

void
setEnvVars(char *java_home)
{
  char newpath[8192];
  char *p;
  char *buffer = newpath;

    /* Now add our locale files to the beginning of XFILESEARCHPATH */

  strcpy(buffer, "XFILESEARCHPATH=");
  strcat(buffer, java_home);
  strcat(buffer, "/jre/lib/locale/%L/%T/%N%S:");

  p = getenv("XFILESEARCHPATH");
  if (p != NULL) {
    strcat(buffer, p);
  }
    
  putenv(strdup(buffer));
}

int cse_test() { return 666; }

static int
create_vm(JavaVM **p_vm, void **env, JavaVMInitArgs *vm_args, char *jvmpath)
{
  void *handle;
  
  int (*p_create_vm) (JavaVM **, void **, JavaVMInitArgs *);
  
  handle = dlopen(jvmpath, RTLD_NOW + RTLD_GLOBAL);
  /* handle = dlopen("libjvm.so", RTLD_LAZY); */
  if (! handle) {
    fprintf(stderr, "%s\n", dlerror());
    exit(1);
  }

  p_create_vm = dlsym(handle, "JNI_CreateJavaVM");
  if (! p_create_vm) {
    fprintf(stderr, "can't find JNI_CreateJavaVM\n");
    exit(1);
  }

  return p_create_vm(p_vm, env, vm_args);
}

/**
 * Parses the command-line arguments
 */
static char **
parse_command_line(int argc, char **argv)
{
  char **new_argv = (char **) malloc((argc + 10) * sizeof(char *));
  int i;
  int j = 0;

  for (i = 1; i < argc; i++) {
    if (! strcmp(argv[i], "-java_home") ||
        ! strcmp(argv[i], "-java-home")) {
      g_options.java_home = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-J-classic")) {
      g_options.jvm_type = "classic";
    }
    else if (! strcmp(argv[i], "-J-server")) {
      g_options.jvm_type = "server";
    }
    else if (! strcmp(argv[i], "-J-client")) {
      g_options.jvm_type = "client";
    }
    else if (! strcmp(argv[i], "-resin_home") ||
             ! strcmp(argv[i], "-resin-home")) {
      g_options.resin_home = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-resin-child")) {
      g_options.is_child = 1;
    }
    else if (! strcmp(argv[i], "-server_root") ||
             ! strcmp(argv[i], "-server-root")) {
      g_options.server_root = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-conf") ||
             ! strcmp(argv[i], "-conf")) {
      new_argv[j++] = argv[i];
      new_argv[j++] = argv[i + 1];
      g_options.conf = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-classpath") ||
             ! strcmp(argv[i], "-cp")) {
      g_options.classpath = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-loadpath") ||
             ! strcmp(argv[i], "-lp")) {
      g_options.loadpath = strdup(argv[i + 1]);
      i++;
    }
    else if (! strncmp(argv[i], "-D", 2) ||
             ! strncmp(argv[i], "-X", 2)) {
      add_jvm_option("%s", argv[i]);
    }
    else if (! strncmp(argv[i], "-J", 2)) {

      if (! strcmp(argv[i] + 2, "-verbosegc"))
        add_jvm_option("-verbose:gc", "");
      else
        add_jvm_option("%s", argv[i] + 2);
    }
    else if (! strcmp(argv[i], "-verbosegc")) {
      add_jvm_option("-verbose:gc", "");
    }
    else if (! strcmp(argv[i], "-stdout")) {
      g_options.stdout_path = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-stderr")) {
      g_options.stderr_path = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-pid")) {
      g_options.pid = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-server")) {
      new_argv[j++] = argv[i];
      new_argv[j++] = argv[i + 1];
      g_options.server = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-verbose")) {
      g_options.verbose = 1;
    }
    else if (! strcmp(argv[i], "start")) {
      g_options.start = 1;
    }
    else if (! strcmp(argv[i], "stop")) {
      g_options.stop = 1;
    }
    else if (! strcmp(argv[i], "restart")) {
      g_options.restart = 1;
      g_options.start = 1;
    }
    else {
      new_argv[j++] = argv[i];
    }
  }

  new_argv[j] = 0;

  return new_argv;
}

/**
 * unwinds a linked path, returning the original directory.
 */
static char *
unwind_link(char *path, char *test_tail)
{
  struct stat st;
  int tail = 0;

  path = strdup(path);

  while ((! lstat(path, &st)) && S_ISLNK(st.st_mode)) {
    char buf[8192];

    readlink(path, buf, sizeof(buf));

    path = strdup(buf);
  }

  for (tail = strlen(path); tail >= 0; tail--) {
    if (path[tail] == '/') {
      path[tail] = 0;

      if (is_file(path, test_tail)) {
        return path;
      }
    }
  }

  return 0;
}

/**
 * Finds a link in a classpath-like path
 */
static char *
find_link_in_path(char *path, char *test_tail, char *exe)
{
  char *head;
  char *tail = 0;

  for (head = path; head && *head; head = tail) {
    char buf[8192];
    struct stat st;
      
    tail = strchr(head, ':');

    if (tail) {
      *tail = 0;
      tail++;
    }

    sprintf(buf, "%s/%s", head, exe);
      
    if (! stat(buf, &st)) {
      char *dir = unwind_link(buf, test_tail);

      if (dir)
        return dir;
    }
  }

  return 0;
}

/**
 * Finds the value of java_home
 *
 * 1) command-line
 * 2) getenv("JAVA_HOME")
 * 3) find by searching back the "java" command
 * 4) trying /usr/java
 */
static void
find_java_home()
{
  char *java_home;
  char *path;
  
  if (g_options.java_home)
    return;

  if ((java_home = getenv("JAVA_HOME"))) {
    g_options.java_home = strdup(java_home);
    return;
  }

  path = getenv("PATH");
  if (path) {
    path = strdup(path);
    
    g_options.java_home = find_link_in_path(path, "jre/lib/rt.jar", "java");
  }

  if (! g_options.java_home)
    g_options.java_home = "/usr/java";
}

/**
 * Finds the value of resin_home
 *
 * 1) command-line
 * 2) getenv("RESIN_HOME")
 * 3) trying /usr/local/resin
 */
static void
find_resin_home(char *cmd)
{
  char *resin_home;
  
  if (g_options.resin_home) {
    return;
  }

  if ((resin_home = getenv("RESIN_HOME"))) {
    g_options.resin_home = strdup(resin_home);
    return;
  }

  g_options.resin_home = unwind_link(cmd, "lib/jsdk23.jar");
  if (g_options.resin_home)
    return;

  if (! g_options.resin_home) {
    char buf[8192];
    getcwd(buf, sizeof(buf));

    g_options.resin_home = strdup(buf);
  }
}

/**
 * Convert a possibly relative path to an absolute path.
 */
static char *
normalize_path(char *path)
{
  char buf[8192];

  if (path[0] == '/')
    return path;

  buf[0] = 0;

  getcwd(buf, sizeof(buf));
  strcat(buf, "/");
  strcat(buf, path);

  return strdup(buf);
}

static void
init_java(JNIEnv *env)
{
  jclass classID;
  jmethodID methodID;
  
  /* jni_vfs_init(env); */

  classID = (*env)->FindClass(env, "com/caucho/server/http/JniServer");
  if (classID == 0) {
    fprintf(stderr, "Class not found: com.caucho.server.http.JniServer\n");
    exit(1);
  }
  g_java.jni_server = classID;

  /* Get the application's main method */
  methodID = (*env)->GetStaticMethodID(env, g_java.jni_server, "main",
                                       "(I[Ljava/lang/String;)V");
  if (! methodID) {
    fprintf(stderr, "Can't find static main com.caucho.server.http.JniServer\n");
    exit(1);
  }
  g_java.main = methodID;
}

/**
 * Reads the configuration file
 *
 * @param name the configuration file name, if specified on the command-line.
 */
config_t *
read_config(char *name)
{
  FILE *file;
  config_t *config = cse_malloc(sizeof(config_t));
  
  memset(config, 0, sizeof(config_t));

  config->p = cse_create_pool(config);
  config->resin_home = g_options.server_root;

  if (! name)
    name = "conf/resin.conf";

  file = fopen(name, "r");
  if (! file) {
    fprintf(stderr, "can't find config file %s\n", name);
    exit(1);
  }

  config->registry = cse_parse(file, config, name);

  fclose(file);

  return config;
}

static int
ssl_create_node(server_socket_t *ss, registry_t *node)
{
  ssl_config_t *config;
  char *flags;
  char *verify_mode;

  config = cse_malloc(sizeof(ssl_config_t));
  memset(config, 0, sizeof(ssl_config_t));

  ss->ssl_config = config;

  config->crypto_device = cse_find_value(node->first, "crypto-device");
  config->alg_flags = ~0;
  
  flags = cse_find_value(node->first, "ssl2");
  if (flags && ! strcmp(flags, "false"))
    config->alg_flags &= ~ALG_SSL2;
  flags = cse_find_value(node->first, "ssl3");
  if (flags && ! strcmp(flags, "false"))
    config->alg_flags &= ~ALG_SSL3;
  flags = cse_find_value(node->first, "tls1");
  if (flags && ! strcmp(flags, "false"))
    config->alg_flags &= ~ALG_TLS1;

  config->certificate_file = cse_find_value(node->first, "certificate-file");
  if (! config->certificate_file)
    config->certificate_file = cse_find_value(node->first, "certificate-pem");
  
  config->key_file = cse_find_value(node->first, "certificate-key-file");
  if (! config->key_file)
    config->key_file = cse_find_value(node->first, "key-pem");

  if (! config->key_file)
    config->key_file = config->certificate_file;
  if (! config->certificate_file)
    config->certificate_file = config->key_file;

  config->password = cse_find_value(node->first, "certificate-key-password");
  if (! config->password)
    config->password = cse_find_value(node->first, "key-store-password");
  
  config->certificate_chain_file = cse_find_value(node->first,
                                                 "certificate-chain-file");
  config->ca_certificate_path = cse_find_value(node->first,
                                              "ca-certificate-path");
  config->ca_certificate_file = cse_find_value(node->first,
                                              "ca-certificate-file");
  config->ca_revocation_path = cse_find_value(node->first,
                                             "ca-revocation-path");
  config->ca_revocation_file = cse_find_value(node->first,
                                             "ca-revocation-file");

  verify_mode = cse_find_value(node->first, "verify-client");
  if (! verify_mode || ! *verify_mode)
    config->verify_client = Q_VERIFY_NONE;
  else if (! strcmp(verify_mode, "none"))
    config->verify_client = Q_VERIFY_NONE;
  else if (! strcmp(verify_mode, "optional_no_ca"))
    config->verify_client = Q_VERIFY_OPTIONAL_NO_CA;
  else if (! strcmp(verify_mode, "optional"))
    config->verify_client = Q_VERIFY_OPTIONAL;
  else if (! strcmp(verify_mode, "require"))
    config->verify_client = Q_VERIFY_REQUIRE;
  else {
    fprintf(stderr, "unknown verify-client value %s\n", verify_mode);
    return 0;
  }

  return ssl_create(ss, config);
}

/**
 * Binds the appropriate ports
 *
 * @param node the port's configuration node.
 *
 * @return the filedescriptor for the bound port
 */
static server_socket_t *
bind_port(registry_t *node)
{
  char *value;
  char *host;
  int port;
  int ss;
  server_socket_t *socket;

  if (! node) {
    fprintf(stderr, "no <http> server entries defined.\n");
    exit(1);
  }

  host = cse_find_value(node->first, "host");
  if (! host)
    host = "*";
  
  value = cse_find_value(node->first, "port");
  if (value)
    port = atoi(value);
  else
    port = 8080;
  
  ss = cse_bind(host, port);
  if (ss < 0) {
    fprintf(stderr,
            "Can't bind to port %s:%d.  Check for conflicting servers\n",
            host, port);
    exit(1);
  }

  value = cse_find_value(node->first, "request-timeout");
  
  socket = (server_socket_t *) cse_malloc(sizeof(server_socket_t));
  memset(socket, 0, sizeof(server_socket_t));

  socket->fd = ss;
  socket->port = port;
  pthread_mutex_init(&socket->ssl_lock, 0);
  
  if (value)
    socket->request_timeout = atoi(value);
  else
    socket->request_timeout = 30;

  if (socket->request_timeout <= 0)
    socket->request_timeout = 30;
    
  if (socket->request_timeout < 10)
    socket->request_timeout = 10;

  if (cse_next_link(node->first, "ssl")) {
    if (! ssl_create_node(socket, node))
      exit(1);
    fprintf(stdout, "%ss listening to %s:%d\n", node->key, host, port);
  }
  else {
    socket->accept = &std_accept;
    fprintf(stdout, "%s listening to %s:%d\n", node->key, host, port);
  }

  return socket;
}

/**
 * Count the number of ports to bind.
 */
static int
count_ports(registry_t *node, char *server)
{
  int count = 0;

  if (node)
    node = cse_next_link(node->first, "caucho.com");
  if (node)
    node = cse_next_link(node->first, "http-server");

  if (! node)
    return 0;

  for (node = node->first; node; node = node->next) {
    if (! strcmp(node->key, "http") ||
        ! strcmp(node->key, "srun") ||
        ! strcmp(node->key, "srun-backup") ||
        ! strcmp(node->key, "srun-backup") ||
        ! strcmp(node->key, "server")) {
      if ((! server && ! node->value) ||
          (server && node->value && ! strcmp(server, node->value)))
        count++;
    }
  }

  return count;
}

/**
 * Binds the appropriate ports
 *
 * @param node the port's configuration node.
 *
 * @return the filedescriptor for the bound port
 */
static void
configure_ports(registry_t *node, char *server)
{
  int count;

  if (node)
    node = cse_next_link(node->first, "caucho.com");
  if (node)
    node = cse_next_link(node->first, "http-server");

  if (node)
    node = node->first;

  count = 0;
  g_port_count = 0;
  for (; node; node = node->next) {
    if (! strcmp(node->key, "http") ||
        ! strcmp(node->key, "srun") ||
        ! strcmp(node->key, "srun-backup") ||
        ! strcmp(node->key, "srun-backup") ||
        ! strcmp(node->key, "server")) {
      if ((! server && ! node->value) ||
          (server && node->value && ! strcmp(server, node->value))) {
        server_socket_t *ss;

        ss = bind_port(node);

        if (ss == 0) {
          fprintf(stderr, "failure opening port.\n");
          exit(1);
        }
  
        g_ports[g_port_count] = ss;

        g_port_count++;

        ss->server_index = count;
      }
      
      count++;
    }
  }
}

/**
 * If necessary, sets the user id.
 *
 * @param node top-level configuration node.
 */
static void
set_user_id(registry_t *node)
{
  char *user_name = 0;
  char *group_name = 0;
  registry_t *subnode = 0;
  struct passwd *passwd;
  struct group *group;
  int uid;
  int gid;
  
  if (node)
    node = cse_next_link(node->first, "caucho.com");

  if (node)
    subnode = cse_next_link(node->first, "http-server");

  if (node)
    user_name = cse_find_value(node->first, "user-name");
  
  if (! user_name && subnode)
    user_name = cse_find_value(subnode->first, "user-name");
  
  if (node)
    group_name = cse_find_value(node->first, "group-name");
  
  if (! group_name && subnode)
    group_name = cse_find_value(subnode->first, "group-name");

  if (! user_name)
    return;

  passwd = getpwnam(user_name);
  
  if (passwd) {
    uid = passwd->pw_uid;
    gid = passwd->pw_gid;
  }
  else {
    fprintf(stderr, "Can't change to unknown user %s\n", user_name);
    exit(1);
  }

  if (group_name) {
    group = getgrnam(group_name);
    if (group)
      gid = group->gr_gid;
  }

  if (uid >= 0) {
    if (gid >= 0) {
      setgid(gid);
      initgroups(user_name, gid);
    }

    if (setuid(uid) < 0) {
      fprintf(stderr, "Can't run as user %s(uid=%d)\n", user_name, uid);
      exit(1);
    }
  }
  
  fprintf(stderr, "Running as %s(uid=%d)\n", user_name, uid);
}

/**
 * Creates a new JniServer object
 *
 * @param env the JVM environment
 * @param argc the command-line argc
 * @param argv the comment-line argv
 *
 * @return the JniServer handle
 */
/*
static jobject
create_server(JNIEnv *env, char **argv)
{
  jobject server_obj = 0;
  jarray java_args;
  jclass string_class;
  int i;
  int argc;

  for (argc = 0; argv[argc]; argc++) {
  }

  string_class = (*env)->FindClass(env, "java/lang/String");
  
  java_args = (*env)->NewObjectArray(env, argc, string_class, 0);
  for (i = 0; i < argc; i++) {
    jstring arg = (*env)->NewStringUTF(env, argv[i]);
    
    (*env)->SetObjectArrayElement(env, java_args, i, arg);
  }
  
  server_obj = (*env)->NewObject(env, g_java.jni_server,
                                 g_java.new_jni_server, java_args);
  if ((*env)->ExceptionOccurred(env)) {
    (*env)->ExceptionDescribe(env);
    exit(1);
  }
  server_obj = (*env)->NewGlobalRef(env, server_obj);

  return server_obj;
}
*/

static int
resin_get_server_socket(resin_t *resin)
{
  if (resin->count < g_port_count)
    return (int) g_ports[resin->count++];
  else
    return 0;
}

/**
 * Forks the server for the background server.
 */
static void
fork_server(int argc, char **argv)
{
  char *pid = "resin.pid";
  char pid_file[1024];
  int pipes[2];
  FILE *file;
  int child_pid;
  struct stat st;
  int status;
  char **new_argv;
  int i, j;
  int has_stdout;
  int has_stderr;

  if (g_options.pid)
    pid = g_options.pid;

  if (pid[0] == '/')
    sprintf(pid_file, "%s", pid);
  else
    sprintf(pid_file, "%s/%s", g_options.server_root, pid);

  if (! stat(pid_file, &st)) {
    file = fopen(pid_file, "r");
    fscanf(file, "%d", &child_pid);
    fclose(file);

    if (kill(child_pid, 0) == 0) {
      fprintf(stderr, "server %s has already started (pid %d).\n",
              pid, child_pid);
      exit(1);
    }
  }
  
  setpgid(0, 0);

  /* become a daemon */
  if (fork()) {
    sleep(5);
    
    exit(0);
  }

  if (fork()) {
    exit(0);
  }
  
  file = fopen(pid_file, "w+");
  fprintf(file, "%d\n", getpid());
  fclose(file);

  new_argv = (char **) cse_malloc((argc + 10) * sizeof(char *));
  j = 0;

  for (i = 0; i < argc; i++) {
    /* cleanup if necessary */
    new_argv[j++] = argv[i];
  }
  
  new_argv[j] = 0;

  while (1) {
    /*
     * Pipe for the deadwait.  When the parent closes, the child
     * will automatically close (see the tail of main()).
     */
    pipe(pipes);

    child_pid = fork();

    if (! child_pid) {
      char *exe = new_argv[0];
    
      dup2(pipes[0], 0);
      if (pipes[0] != 0)
	close(pipes[0]);
      if (pipes[1] != 0)
	close(pipes[1]);

      execv(exe, new_argv);

      fprintf(stderr, "can't start %s\n", new_argv[0]);
      exit(1);
    }

    close(pipes[0]);

    wait(&status);

    close(pipes[1]);

    kill(child_pid, SIGQUIT);

    sleep(10);
  }
}

static void
stop_server()
{
  char *pid = "resin.pid";
  char pid_file[1024];
  struct stat st;
  FILE *file;
  int child_pid;

  if (g_options.pid)
    pid = g_options.pid;
  
  if (pid[0] == '/')
    sprintf(pid_file, "%s", pid);
  else
    sprintf(pid_file, "%s/%s", g_options.server_root, pid);

  if (stat(pid_file, &st)) {
    fprintf(stderr, "no server %s has been started.\n", pid);

    if (g_options.start || g_options.restart)
      return;
    
    exit(1);
  }

  file = fopen(pid_file, "r");
  fscanf(file, "%d", &child_pid);
  fclose(file);

  unlink(pid_file);

  fprintf(stderr, "shutting down client %d\n", child_pid);
  
  if (child_pid > 0)
    kill(child_pid, 15);
}

static void
open_logs()
{
  char log_path[1024];
  int fd;

  sprintf(log_path, "%s/log", g_options.server_root);
  mkdir(log_path, 0775);

  if (is_path_absolute(g_options.stdout_path))
    sprintf(log_path, "%s", g_options.stdout_path);
  else
    sprintf(log_path, "%s/%s", g_options.server_root, g_options.stdout_path);
  fd = open(log_path, O_RDWR|O_CREAT|O_APPEND, 0664);
  
  dup2(fd, 1);
  close(fd);

  if (is_path_absolute(g_options.stderr_path))
    sprintf(log_path, "%s", g_options.stderr_path);
  else
    sprintf(log_path, "%s/%s", g_options.server_root, g_options.stderr_path);
  fd = open(log_path, O_RDWR|O_CREAT|O_APPEND, 0664);
  dup2(fd, 2);
  close(fd);
}

int
main(int argc, char **argv)
{
  JavaVM *vm = 0;
  JNIEnv *env = 0;
  JavaVMInitArgs vm_args;
  JavaVMAttachArgs vm_attach_args;
  char envpath[8192];
  char jvmpath[8192];
  int res;
  config_t *config;
  char **new_argv;
  int i, j;
  resin_t *resin;
  jarray java_args;
  jclass string_class;

  memset(&g_options, 0, sizeof(g_options));
  memset(&vm_args, 0, sizeof(vm_args));
  memset(&vm_attach_args, 0, sizeof(vm_attach_args));
  
  new_argv = parse_command_line(argc, argv);
  
  find_java_home();
  find_resin_home(argv[0]);
  g_options.resin_home = normalize_path(g_options.resin_home);

  if (! g_options.server_root)
    g_options.server_root = g_options.resin_home;
  else
    g_options.server_root = normalize_path(g_options.server_root);

  if (! g_options.is_child && (g_options.stop || g_options.restart)) {
    stop_server();

    if (g_options.restart || g_options.start) {
      sleep(5);
    }
    else
      exit(0);
  }

  /*
   * First pass through needs to set the LD_LIBRARY_PATH and reexecute
   * the server.
   */
  if (! g_options.is_child) {
    char **new_argv;

    new_argv = (char **) malloc((argc + 10) * sizeof (char *));
    memcpy(new_argv, argv, argc * (sizeof (char *)));
    new_argv[argc] = "-resin-child";
    new_argv[argc + 1] = 0;

    set_library_path(g_options.java_home, g_options.resin_home,
                     g_options.jvm_type);

    if (g_options.start || g_options.restart) {
      fork_server(argc + 1, new_argv);
      exit(1);
    }
    else {
      execv(new_argv[0], new_argv);

      fprintf(stderr, "Can't start server %s.\n", argv[0]);
      exit(1);
    }
  }

  set_jvmpath(g_options.java_home, g_options.resin_home,
              g_options.jvm_type, jvmpath);

  chdir(g_options.server_root);

  setEnvVars(g_options.java_home);
  set_classpath();

  config = read_config(g_options.conf);

  g_config = config;

  for (j = 0; new_argv[j]; j++) {
  }

  /* extra length was already allocated */
  if (g_options.stdout_path) {
    new_argv[j++] = "-stdout";
    new_argv[j++] = g_options.stdout_path;
  }
  
  if (g_options.stderr_path) {
    new_argv[j++] = "-stderr";
    new_argv[j++] = g_options.stderr_path;
  }
  new_argv[j] = 0;

  fprintf(stdout, "%s\n", FULL_VERSION);

  if (g_options.verbose) {
    fprintf(stdout, "JAVA_HOME:   %s\n", g_options.java_home);
    fprintf(stdout, "RESIN_HOME:  %s\n", g_options.resin_home);
    fprintf(stdout, "SERVER_ROOT: %s\n", g_options.server_root);
    fprintf(stdout, "CLASSPATH:   %s\n", g_options.classpath);
    fprintf(stdout, "LDPATH:      %s\n", getenv("LD_LIBRARY_PATH"));

    for (i = 0; new_argv[i]; i++)
      fprintf(stdout, "arg %d:      %s\n", i, new_argv[i]);
  }

  if (config->error) {
    printf("%s\n", config->error);
    exit(1);
  }
  
  g_port_count = count_ports(config->registry, g_options.server);
  
  if (g_port_count <= 0) {
    fprintf(stderr, "No servers defined.\n");
    exit(1);
  }
  
  configure_ports(config->registry, g_options.server);

  /* XXX: now, change ownership */
  set_user_id(config->registry);

  if (g_options.start || g_options.restart) {
    if (! g_options.stdout_path)
      g_options.stdout_path = "log/stdout.log";
    if (! g_options.stderr_path)
      g_options.stderr_path = "log/stderr.log";

    open_logs();
  }

  resin = (resin_t *) cse_malloc(sizeof(resin_t));
  memset(resin, 0, sizeof(resin_t));
  resin->get_server_socket = resin_get_server_socket;

  strcpy(envpath, "-Djava.library.path=");
  strcat(envpath, getenv("LD_LIBRARY_PATH"));

  add_jvm_option("-Djava.library.path=%s", getenv("LD_LIBRARY_PATH"));
  add_jvm_option("-Dresin.home=%s", g_options.server_root);

  memset(&vm_args, 0, sizeof(vm_args));
  vm_args.version = JNI_VERSION_1_2;
  vm_args.options = g_options.options;
  vm_args.nOptions = g_options.n_options;
  vm_args.ignoreUnrecognized = 0;

  res = create_vm(&vm, (void **)&env, &vm_args, jvmpath);
  if (res < 0) {
    fprintf(stderr, "Failed to create JVM.\n");
    exit(1);
  }
  
  g_java.vm = vm;

  init_java(env);

  for (argc = 0; new_argv[argc]; argc++) {
  }

  string_class = (*env)->FindClass(env, "java/lang/String");

  java_args = (*env)->NewObjectArray(env, argc, string_class, 0);
  for (i = 0; i < argc; i++) {
    jstring arg = (*env)->NewStringUTF(env, new_argv[i]);
    
    (*env)->SetObjectArrayElement(env, java_args, i, arg);
  }
  
  (*env)->CallStaticVoidMethod(env, g_java.jni_server, g_java.main,
                               resin, java_args);
  
  if ((*env)->ExceptionOccurred(env))
    (*env)->ExceptionDescribe(env);
  
  (*vm)->DetachCurrentThread(vm);
  (*vm)->DestroyJavaVM(vm);

  return 0;
}
