/***********************************************************
* Artsoft Retro-Game Library                               *
*----------------------------------------------------------*
* (c) 1994-2006 Artsoft Entertainment                      *
*               Holger Schemel                             *
*               Detmolder Strasse 189                      *
*               33604 Bielefeld                            *
*               Germany                                    *
*               e-mail: info@artsoft.org                   *
*----------------------------------------------------------*
* text.c                                                   *
***********************************************************/

#include <stdio.h>
#include <stdarg.h>

#include "text.h"
#include "misc.h"


/* ========================================================================= */
/* font functions                                                            */
/* ========================================================================= */

#if defined(TARGET_X11_NATIVE_PERFORMANCE_WORKAROUND)
static GC	font_clip_gc = None;

static void InitFontClipmasks()
{
  XGCValues clip_gc_values;
  unsigned long clip_gc_valuemask;
  GC copy_clipmask_gc;
  int i, j;

  /* This stuff is needed because X11 (XSetClipOrigin(), to be precise) is
     often very slow when preparing a masked XCopyArea() for big Pixmaps.
     To prevent this, create small (tile-sized) mask Pixmaps which will then
     be set much faster with XSetClipOrigin() and speed things up a lot. */

  clip_gc_values.graphics_exposures = False;
  clip_gc_valuemask = GCGraphicsExposures;
  font_clip_gc = XCreateGC(display, window->drawable,
			   clip_gc_valuemask, &clip_gc_values);

  /* create graphic context structures needed for clipping */
  clip_gc_values.graphics_exposures = False;
  clip_gc_valuemask = GCGraphicsExposures;
  copy_clipmask_gc = XCreateGC(display,
			       gfx.font_bitmap_info[0].bitmap->clip_mask,
			       clip_gc_valuemask, &clip_gc_values);

  /* create only those clipping Pixmaps we really need */
  for (i = 0; i < gfx.num_fonts; i++)
  {
    if (gfx.font_bitmap_info[i].bitmap == NULL)
      continue;

    gfx.font_bitmap_info[i].clip_mask =
      checked_calloc(gfx.font_bitmap_info[i].num_chars * sizeof(Pixmap));

    for (j = 0; j < gfx.font_bitmap_info[i].num_chars; j++)
    {
      Bitmap *src_bitmap = gfx.font_bitmap_info[i].bitmap;
      Pixmap src_pixmap = src_bitmap->clip_mask;
      int xpos = j % gfx.font_bitmap_info[i].num_chars_per_line;
      int ypos = j / gfx.font_bitmap_info[i].num_chars_per_line;
      int width  = gfx.font_bitmap_info[i].width;
      int height = gfx.font_bitmap_info[i].height;
      int src_x = gfx.font_bitmap_info[i].src_x + xpos * width;
      int src_y = gfx.font_bitmap_info[i].src_y + ypos * height;

      gfx.font_bitmap_info[i].clip_mask[j] =
	XCreatePixmap(display, window->drawable, width, height, 1);

      XCopyArea(display, src_pixmap, gfx.font_bitmap_info[i].clip_mask[j],
		copy_clipmask_gc, src_x, src_y, width, height, 0, 0);
    }
  }

  XFreeGC(display, copy_clipmask_gc);
}

static void FreeFontClipmasks()
{
  int i, j;

  if (gfx.num_fonts == 0 || gfx.font_bitmap_info[0].bitmap == NULL)
    return;

  for (i = 0; i < gfx.num_fonts; i++)
  {
    if (gfx.font_bitmap_info[i].clip_mask)
    {
      for (j = 0; j < gfx.font_bitmap_info[i].num_chars; j++)
	XFreePixmap(display, gfx.font_bitmap_info[i].clip_mask[j]);
      free(gfx.font_bitmap_info[i].clip_mask);
    }

    gfx.font_bitmap_info[i].clip_mask = NULL;
    gfx.font_bitmap_info[i].num_chars = 0;
  }

  if (font_clip_gc)
    XFreeGC(display, font_clip_gc);
  font_clip_gc = None;
}
#endif /* TARGET_X11_NATIVE_PERFORMANCE_WORKAROUND */

void InitFontInfo(struct FontBitmapInfo *font_bitmap_info, int num_fonts,
		  int (*select_font_function)(int))
{
  gfx.num_fonts = num_fonts;
  gfx.font_bitmap_info = font_bitmap_info;
  gfx.select_font_function = select_font_function;

#if defined(TARGET_X11_NATIVE_PERFORMANCE_WORKAROUND)
  InitFontClipmasks();
#endif
}

void FreeFontInfo(struct FontBitmapInfo *font_bitmap_info)
{
  if (font_bitmap_info == NULL)
    return;

#if defined(TARGET_X11_NATIVE_PERFORMANCE_WORKAROUND)
  FreeFontClipmasks();
#endif

  free(font_bitmap_info);
}

struct FontBitmapInfo *getFontBitmapInfo(int font_nr)
{
  int font_bitmap_id = gfx.select_font_function(font_nr);

  return &gfx.font_bitmap_info[font_bitmap_id];
}

int getFontWidth(int font_nr)
{
  int font_bitmap_id = gfx.select_font_function(font_nr);

  return gfx.font_bitmap_info[font_bitmap_id].width;
}

int getFontHeight(int font_nr)
{
  int font_bitmap_id = gfx.select_font_function(font_nr);

  return gfx.font_bitmap_info[font_bitmap_id].height;
}

int getTextWidth(char *text, int font_nr)
{
  return (text != NULL ? strlen(text) * getFontWidth(font_nr) : 0);
}

static char getFontCharPosition(int font_nr, char c)
{
  int font_bitmap_id = gfx.select_font_function(font_nr);
  struct FontBitmapInfo *font = &gfx.font_bitmap_info[font_bitmap_id];
  boolean default_font = (font->num_chars == DEFAULT_NUM_CHARS_PER_FONT);
  int font_pos = c - 32;

  /* map some special characters to their ascii values in default font */
  if (default_font)
    font_pos = MAP_FONT_ASCII(c) - 32;

  /* this allows dynamic special characters together with special font */
  if (font_pos < 0 || font_pos >= font->num_chars)
    font_pos = 0;

  return font_pos;
}

void getFontCharSource(int font_nr, char c, Bitmap **bitmap, int *x, int *y)
{
  int font_bitmap_id = gfx.select_font_function(font_nr);
  struct FontBitmapInfo *font = &gfx.font_bitmap_info[font_bitmap_id];
  int font_pos = getFontCharPosition(font_nr, c);

  *bitmap = font->bitmap;
  *x = font->src_x + (font_pos % font->num_chars_per_line) * font->width;
  *y = font->src_y + (font_pos / font->num_chars_per_line) * font->height;
}

void DrawInitText(char *text, int ypos, int font_nr)
{
  if (window &&
      gfx.num_fonts > 0 &&
      gfx.font_bitmap_info[font_nr].bitmap != NULL)
  {
    ClearRectangle(window, 0, ypos, video.width, getFontHeight(font_nr));
    DrawTextExt(window, (video.width - getTextWidth(text, font_nr)) / 2, ypos,
		text, font_nr, BLIT_OPAQUE);
    FlushDisplay();
  }
}

void DrawTextF(int x, int y, int font_nr, char *format, ...)
{
  char buffer[MAX_OUTPUT_LINESIZE + 1];
  va_list ap;

  va_start(ap, format);
  vsprintf(buffer, format, ap);
  va_end(ap);

  if (strlen(buffer) > MAX_OUTPUT_LINESIZE)
    Error(ERR_EXIT, "string too long in DrawTextF() -- aborting");

  DrawText(gfx.sx + x, gfx.sy + y, buffer, font_nr);
}

void DrawTextFCentered(int y, int font_nr, char *format, ...)
{
  char buffer[MAX_OUTPUT_LINESIZE + 1];
  va_list ap;

  va_start(ap, format);
  vsprintf(buffer, format, ap);
  va_end(ap);

  if (strlen(buffer) > MAX_OUTPUT_LINESIZE)
    Error(ERR_EXIT, "string too long in DrawTextFCentered() -- aborting");

  DrawText(gfx.sx + (gfx.sxsize - getTextWidth(buffer, font_nr)) / 2,
	   gfx.sy + y, buffer, font_nr);
}

void DrawTextS(int x, int y, int font_nr, char *text)
{
  DrawText(gfx.sx + x, gfx.sy + y, text, font_nr);
}

void DrawTextSCentered(int y, int font_nr, char *text)
{
  DrawText(gfx.sx + (gfx.sxsize - getTextWidth(text, font_nr)) / 2,
	   gfx.sy + y, text, font_nr);
}

void DrawText(int x, int y, char *text, int font_nr)
{
  int mask_mode = BLIT_OPAQUE;

  if (DrawingOnBackground(x, y))
    mask_mode = BLIT_ON_BACKGROUND;

  DrawTextExt(drawto, x, y, text, font_nr, mask_mode);

  if (x < gfx.dx)
    redraw_mask |= REDRAW_FIELD;
  else if (y < gfx.vy || gfx.vy == 0)
    redraw_mask |= REDRAW_DOOR_1;
}

void DrawTextExt(DrawBuffer *dst_bitmap, int dst_x, int dst_y, char *text,
		 int font_nr, int mask_mode)
{
  int font_bitmap_id = gfx.select_font_function(font_nr);
  struct FontBitmapInfo *font = &gfx.font_bitmap_info[font_bitmap_id];
  int font_width = getFontWidth(font_nr);
  int font_height = getFontHeight(font_nr);
#if 0
  int border_1 = gfx.sx + gfx.sxsize;
  int border_2 = gfx.dx + gfx.dxsize;
  int dst_x_start = dst_x;
#endif
  Bitmap *src_bitmap;
  int src_x, src_y;
  char *text_ptr = text;

  if (font->bitmap == NULL)
    return;

  /* skip text to be printed outside the window (left/right will be clipped) */
  if (dst_y < 0 || dst_y + font_height > video.height)
    return;

  /* add offset for drawing font characters */
  dst_x += font->draw_xoffset;
  dst_y += font->draw_yoffset;

  while (*text_ptr)
  {
    char c = *text_ptr++;

    if (c == '\n')
      c = ' ';		/* print space instaed of newline */

    getFontCharSource(font_nr, c, &src_bitmap, &src_x, &src_y);

    /* clip text at the left side of the window */
    if (dst_x < 0)
    {
      dst_x += font_width;

      continue;
    }

    /* clip text at the right side of the window */
#if 1
    if (dst_x + font_width > video.width)
      break;
#else
    /* (this does not work well when trying to print text to whole screen) */
    if ((dst_x_start < border_1 && dst_x + font_width > border_1) ||
	(dst_x_start < border_2 && dst_x + font_width > border_2))
      break;
#endif

    if (mask_mode == BLIT_INVERSE)	/* special mode for text gadgets */
    {
      /* first step: draw solid colored rectangle (use "cursor" character) */
      if (strlen(text) == 1)	/* only one char inverted => draw cursor */
      {
	Bitmap *cursor_bitmap;
	int cursor_x, cursor_y;

	getFontCharSource(font_nr, FONT_ASCII_CURSOR, &cursor_bitmap,
			  &cursor_x, &cursor_y);

	BlitBitmap(cursor_bitmap, dst_bitmap, cursor_x, cursor_y,
		   font_width, font_height, dst_x, dst_y);
      }

#if defined(TARGET_SDL)
      /* second step: draw masked inverted character */
      SDLCopyInverseMasked(src_bitmap, dst_bitmap, src_x, src_y,
			   font_width, font_height, dst_x, dst_y);
#else
      /* second step: draw masked black rectangle (use "space" character) */
      SetClipOrigin(src_bitmap, src_bitmap->stored_clip_gc,
		    dst_x - src_x, dst_y - src_y);
      BlitBitmapMasked(src_bitmap, dst_bitmap, 0, 0,
		       font_width, font_height, dst_x, dst_y);
#endif
    }
    else if (mask_mode == BLIT_MASKED || mask_mode == BLIT_ON_BACKGROUND)
    {
      if (mask_mode == BLIT_ON_BACKGROUND)
      {
	/* clear font character background */
	ClearRectangleOnBackground(dst_bitmap, dst_x, dst_y,
				   font_width, font_height);
      }

#if defined(TARGET_X11_NATIVE_PERFORMANCE_WORKAROUND)
      /* use special font tile clipmasks */
      {
	int font_pos = getFontCharPosition(font_nr, c);

	SetClipMask(src_bitmap, font_clip_gc, font->clip_mask[font_pos]);
	SetClipOrigin(src_bitmap, font_clip_gc, dst_x, dst_y);
      }
#else
      SetClipOrigin(src_bitmap, src_bitmap->stored_clip_gc,
		    dst_x - src_x, dst_y - src_y);
#endif

      BlitBitmapMasked(src_bitmap, dst_bitmap, src_x, src_y,
		       font_width, font_height, dst_x, dst_y);
    }
    else	/* normal, non-masked font blitting */
    {
      BlitBitmap(src_bitmap, dst_bitmap, src_x, src_y,
		 font_width, font_height, dst_x, dst_y);
    }

    dst_x += font_width;
  }
}

void DrawTextToTextArea(int x, int y, char *text, int font_nr, int line_length,
			int area_xsize, int area_ysize, int mask_mode)
{
  int area_line = 0;
  int font_height = getFontHeight(font_nr);

  if (text == NULL)
    return;

  while (*text && area_line < area_ysize)
  {
    char buffer[MAX_OUTPUT_LINESIZE + 1];
    int i;

    for (i = 0; i < line_length && *text; i++)
      if ((buffer[i] = *text++) == '\n')
	break;
    buffer[MIN(i, area_xsize)] = '\0';

    DrawTextExt(drawto, x, y + area_line * font_height, buffer, font_nr,
		mask_mode);

    area_line++;
  }

  redraw_mask |= REDRAW_FIELD;
}

boolean RenderLineToBuffer(char **src_buffer_ptr, char *dst_buffer,
			   int *dst_buffer_len, boolean last_line_was_empty,
			   int line_length)
{
  char *text_ptr = *src_buffer_ptr;
  char *buffer = dst_buffer;
  int buffer_len = *dst_buffer_len;
  boolean buffer_filled = FALSE;

  while (*text_ptr)
  {
    char *word_ptr;
    int word_len;

    /* skip leading whitespaces */
    while (*text_ptr == ' ' || *text_ptr == '\t')
      text_ptr++;

    word_ptr = text_ptr;
    word_len = 0;

    /* look for end of next word */
    while (*word_ptr != ' ' && *word_ptr != '\t' && *word_ptr != '\0')
    {
      word_ptr++;
      word_len++;
    }

    if (word_len == 0)
    {
      continue;
    }
    else if (*text_ptr == '\n')		/* special case: force empty line */
    {
      if (buffer_len == 0)
	text_ptr++;

      /* prevent printing of multiple empty lines */
      if (buffer_len > 0 || !last_line_was_empty)
	buffer_filled = TRUE;
    }
    else if (word_len < line_length - buffer_len)
    {
      /* word fits into text buffer -- add word */

      if (buffer_len > 0)
	buffer[buffer_len++] = ' ';

      strncpy(&buffer[buffer_len], text_ptr, word_len);
      buffer_len += word_len;
      buffer[buffer_len] = '\0';
      text_ptr += word_len;
    }
    else if (buffer_len > 0)
    {
      /* not enough space left for word in text buffer -- print buffer */

      buffer_filled = TRUE;
    }
    else
    {
      /* word does not fit at all into empty text buffer -- cut word */

      strncpy(buffer, text_ptr, line_length);
      buffer[line_length] = '\0';
      text_ptr += line_length;
      buffer_filled = TRUE;
    }

    if (buffer_filled)
      break;
  }

  *src_buffer_ptr = text_ptr;
  *dst_buffer_len = buffer_len;

  return buffer_filled;
}

void DrawTextWrapped(int x, int y, char *text, int font_nr, int line_length,
		     int max_lines)
{
  char *text_ptr = text;
  char buffer[line_length + 1];
  int buffer_len;
  int current_line = 0;
  int font_height = getFontHeight(font_nr);

  while (*text_ptr && current_line < max_lines)
  {
    buffer[0] = '\0';
    buffer_len = 0;

    RenderLineToBuffer(&text_ptr, buffer, &buffer_len, TRUE, line_length);

    DrawText(x, y + current_line * font_height, buffer, font_nr);
    current_line++;
  }
}

int DrawTextFromFile(int x, int y, char *filename, int font_nr,
		     int line_length, int max_lines)
{
  int font_height = getFontHeight(font_nr);
  char line[MAX_LINE_LEN];
  char buffer[line_length + 1];
  int buffer_len;
  int current_line = 0;
  FILE *file;

  if (current_line >= max_lines)
    return 0;

  if (filename == NULL)
    return 0;

  if (!(file = fopen(filename, MODE_READ)))
    return 0;

  buffer[0] = '\0';
  buffer_len = 0;

  while (!feof(file) && current_line < max_lines)
  {
    char *line_ptr;
    boolean last_line_was_empty = TRUE;

    /* read next line of input file */
    if (!fgets(line, MAX_LINE_LEN, file))
      break;

    /* skip comments (lines directly beginning with '#') */
    if (line[0] == '#')
      continue;

    /* cut trailing newline from input line */
    for (line_ptr = line; *line_ptr; line_ptr++)
    {
      if (*line_ptr == '\n' || *line_ptr == '\r')
      {
	*line_ptr = '\0';
	break;
      }
    }

    if (strlen(line) == 0)		/* special case: force empty line */
      strcpy(line, "\n");

    line_ptr = line;

    while (*line_ptr && current_line < max_lines)
    {
      boolean buffer_filled = RenderLineToBuffer(&line_ptr,
						 buffer, &buffer_len,
						 last_line_was_empty,
						 line_length);
      if (buffer_filled)
      {
	DrawText(x, y + current_line * font_height, buffer, font_nr);
	current_line++;

	last_line_was_empty = (buffer_len == 0);

	buffer[0] = '\0';
	buffer_len = 0;
      }
    }
  }

  fclose(file);

  if (buffer_len > 0 && current_line < max_lines)
  {
    DrawText(x, y + current_line * font_height, buffer, font_nr);
    current_line++;
  }

  return current_line;
}
