#--
# $Id: eet.rb 53 2005-06-02 20:01:45Z tilman $
#
# Copyright (c) 2005 Tilman Sauerbeck (tilman at code-monkey de)
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

require "eet_ext"

class Object
	def to_eet_chunks(tag, type = nil) # :nodoc:
		[Eet::Chunk.new(tag, to_eet)]
	end

	protected

	# :call-seq:
	#  object.to_eet_name -> string
	#
	# Returns the tag that's stored with the data for _object_.
	# If your class doesn't override this method, the class name will be
	# used.
	def to_eet_name
		self.class.name
	end

	# :call-seq:
	#  object.to_eet_properties -> hash
	#
	# Returns a hash that contains the properties that are stored for
	# _object_.
	# If your class doesn't override this method, all instance variables
	# of _object_ will be stored.
	def to_eet_properties
		instance_variables.inject({}) do |h, var|
			h[var[1..-1]] = [instance_variable_get(var)]
			h
		end
	end
end

class Integer # :nodoc:
	def to_eet_chunks(tag, type = nil)
		fmt = case type
		when :char: "c"
		when :short: "v"
		when :long_long: "q"
		else "V"
		end

		data = [self].pack(fmt)
		[Eet::Chunk.new(tag, data)]
	end
end

class Float # :nodoc:
	def to_eet_chunks(tag, type = nil)
		fmt = case type
		when :double: "%32.32f"
		else "%16.16f"
		end

		data = fmt % self
		[Eet::Chunk.new(tag, data + "\0")]
	end
end

class String # :nodoc:
	def to_eet_chunks(tag, type = nil)
		[Eet::Chunk.new(tag, self + "\0")]
	end
end

class TrueClass # :nodoc:
	def to_eet_chunks(tag, type = nil)
		[Eet::Chunk.new(tag, "\1")]
	end
end

class FalseClass # :nodoc:
	def to_eet_chunks(tag, type = nil)
		[Eet::Chunk.new(tag, "\0")]
	end
end

class Array # :nodoc:
	def to_eet_chunks(tag, type = nil)
		case type
		when :sub
			[Eet::Chunk.new(tag, self.to_eet)]
		else
			# lists always hold subtypes
			map { |item| Eet::Chunk.new(tag, item.to_eet) }
		end
	end
end

class Hash # :nodoc:
	def to_eet_chunks(tag, type = nil)
		# lists always hold subtypes
		map { |(key, value)| Eet::Chunk.new(tag, value.to_eet) }
	end
end

module Eet
	VERSION = "0.1.3"

	class ChunkError < EetError; end

	class Stream # :nodoc:
		def initialize(chunk = nil)
			super(chunk.nil? ? 0 : 1, chunk)
		end

		def Stream.deserialize(data)
			if data.to_str.empty?
				raise(ArgumentError, "buffer is empty")
			end

			s = Stream.new
			offset = 0

			while offset < data.length
				c, bytes = Chunk.deserialize(data[offset..-1])

				s << c
				offset += bytes
			end

			s
		end
	end

	class Chunk # :nodoc:
		def Chunk.deserialize(data)
			if data.to_str.empty?
				raise(ArgumentError, "buffer is empty")
			end

			if data.length < 8 || data[0, 4] != "CHnK"
				raise(ChunkError, "invalid data")
			end

			size = data[4, 4].unpack("V").first
			if size >= (1 << 31) || size > data.length - 8
				raise(ChunkError, "invalid chunk size")
			end

			unless data[8, size].include?(0)
				raise(ChunkError, "invalid chunk data")
			end

			c = Chunk.new(*data[8, size].split("\0", 2))

			[c, 8 + size]
		end
	end
end
