# $Id: test_misc.rb 1 2005-03-26 01:45:38Z tilman $

require "eet"
require "test/unit"
require "ftools"

class Foo
	def initialize
		@foo = "foo"
		@bar = true
		@baz = 512
	end
end

class MiscTest < Test::Unit::TestCase
	def setup
		@keys = ["key_foo", "key_bar", "key_baz"].sort
		@dest = __FILE__.sub(/\.rb$/, ".eet")
		@foo_data = Foo.new.to_eet

		assert_not_nil(@foo_data)

		File.rm_f(@dest)

		Eet::File.open(@dest, "w") do |ef|
			@keys.each do |k|
				assert(ef.write(k, @foo_data) > 0)
			end
		end
	end

	def test_stream_chunks
		stream = nil

		assert_nothing_raised do
			stream = Eet::Stream.deserialize(@foo_data)
		end

		assert_equal(1, stream.length)
		assert_equal("Foo", stream.first.tag)

		assert_nothing_raised do
			stream = Eet::Stream.deserialize(stream.first.data)
		end

		assert_equal(3, stream.length)
		["foo", "bar", "baz"].each do |tag|
			found = stream.find { |c| c.tag == tag }
			assert_not_nil(found)
		end
	end

	def test_create_new
		assert_raise(RuntimeError) do
			Eet::File.open("/foo/bar/baz/xyzzy", "w") { |ef| }
		end
	end

	def test_catch_invalid_open_mode
		assert_raise(ArgumentError) do
			Eet::File.open(@dest, @keys[0]) { |ef| }
		end
	end

	def test_zero_write
		assert_raise(IOError) do
			Eet::File.open(@dest, "w") do |ef|
				ef.write("empty", "")
			end
		end
	end

	def test_open_existing_ro
		# if mode isn't specified, the file must be opened read-only.
		# assert that we cannot write in read-only mode.
		Eet::File.open(@dest) do |ef|
			assert_raise(IOError) do
				ef.write("xyzzy", "data")
			end
		end
	end

	def test_open_existing_w
		# if the file is opened in write-only or read-write mode,
		# we must be able to write to it.
		["w", "r+"].each do |mode|
			Eet::File.open(@dest, mode) do |ef|
				assert_equal(4, ef.write("xyzzy", "data"))
			end
		end
	end

	def test_accessing_closed_file
		ef = Eet::File.open(@dest, "r")
		ef.close

		assert_raise(IOError) do
			ef.read(@keys[0])
		end
	end

	def test_delete_entry_ro
		assert_raise(IOError) do
			Eet::File.open(@dest, "r") do |ef|
				ef.delete(@keys[0])
			end
		end
	end

	def test_delete_nonexisting_entry
		assert_raise(IOError) do
			Eet::File.open(@dest, "w") do |ef|
				ef.delete("nonexistingkey")
			end
		end
	end

	def test_read
		Eet::File.open(@dest, "r") do |ef|
			@keys.each do |k|
				assert_equal(@foo_data, ef.read(k))
			end
		end

		assert_raise(IOError) do
			Eet::File.open(@dest, "w") do |ef|
				ef.read(@keys[0])
			end
		end
	end

	def test_list
		["r", "r+"].each do |mode|
			Eet::File.open(@dest, mode) do |ef|
				assert_equal(@keys, ef.list.sort)
				assert_equal(@keys, ef.list("*").sort, @keys)
				assert_equal([@keys[0]], ef.list(@keys[0]))
			end
		end

		assert_raise(IOError) do
			Eet::File.open(@dest, "w") do |ef|
				ef.list
			end
		end
	end

	def test_delete
		assert_nothing_raised do
			Eet::File.open(@dest, "r+") do |ef|
				ef.delete(@keys[0])
			end
		end

		Eet::File.open(@dest) do |ef|
			tmp = @keys.shift
			assert_equal(@keys, ef.list.sort)
			@keys.unshift(tmp)
		end
	end

	def test_invalid_key
		Eet::File.open(@dest, "r+") do |ef|
			assert_raise(ArgumentError) do
				ef.read("key_\0foo")
			end

			assert_raise(ArgumentError) do
				ef.write("key_\0foo", "data")
			end

			assert_raise(ArgumentError) do
				ef.delete("key_\0foo")
			end
		end
	end
end
