/***********************************************************************
 * FXRuby -- the Ruby language bindings for the FOX GUI toolkit.
 * Copyright (c) 2001-2003 by J. Lyle Johnson. All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For further information please contact the author by e-mail
 * at "lyle@users.sourceforge.net".
 ***********************************************************************/

/// Double-precision 4-element vector
class FXVec4d {
public:
  FXdouble x;
  FXdouble y;
  FXdouble z;
  FXdouble w;
public:

  /// Default constructor
  FXVec4d();

  /// Copy constructor
  FXVec4d(const FXVec4d& v);

  /// Initialize with 3-vector
  FXVec4d(const FXVec3d& v);

  /// Initialize with components
  FXVec4d(FXdouble xx,FXdouble yy,FXdouble zz,FXdouble ww=1.0);

  /// Initialize with color
  FXVec4d(FXColor color);

  %extend {
    /// Indexing
    FXdouble __getitem__(FXint i) const {
      if(i<0||i>3) rb_raise(rb_eIndexError,"index %d out of bounds",i);
      return (*self)[i];
      }
    FXdouble __setitem__(FXint i,FXdouble d){
      if(i<0||i>3) rb_raise(rb_eIndexError,"index %d out of bounds",i);
      (*self)[i]=d;
      return d;
      }

    /// Unary
    FXVec4d operator-() const { return -(*self); }

    /// Adding
    FXVec4d operator+(const FXVec4d& other) const { return (*self)+other; }

    /// Subtracting
    FXVec4d operator-(const FXVec4d& other) const { return (*self)-other; }

    /// Scaling
    FXVec4d operator*(FXdouble n) const { return (*self)*n; }
    FXVec4d operator/(FXdouble n) const {
      if(n==0.0) rb_raise(rb_eZeroDivError,"divided by 0");
      return (*self)/n;
      }

    /// Dot product
    FXdouble operator*(const FXVec4d& other) const { return (*self)*other; }
    FXdouble dot(const FXVec4d& other) const { return (*self)*other; }

    /// Equality tests
    bool operator==(const FXVec4d& other) const { return (*self)==other; }

    /// Length and square of length
    FXdouble len2() const { return len2(*self); }
    FXdouble len() const { return len(*self); }

    /// Normalize vector
    FXVec4d normalize() const { return normalize(*self); }

    /// Lowest or highest components
    FXVec4d lo(const FXVec4d& other) const { return lo(*self,other); }
    FXVec4d hi(const FXVec4d& other) const { return hi(*self,other); }
    
    /// Convert to array
    VALUE to_a() const { return rb_ary_new3(4,rb_float_new(self->x),rb_float_new(self->y),rb_float_new(self->z),rb_float_new(self->w)); }

    /// Convert to string
    FXString to_s() const { return FXStringFormat("[%g, %g, %g, %g]",self->x,self->y,self->z,self->w); }
    }
  };

