/***********************************************************************
 * FXRuby -- the Ruby language bindings for the FOX GUI toolkit.
 * Copyright (c) 2001-2003 by J. Lyle Johnson. All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For further information please contact the author by e-mail
 * at "lyle@users.sourceforge.net".
 ***********************************************************************/

%define DECLARE_FXAPP_VIRTUALS(klass)
%extend klass {
  /// Create application's windows
  virtual void create();

  /// Destroy application's windows
  virtual void destroy();

  /// Detach application's windows
  virtual void detach();

  /**
   * Initialize application.
   * Parses and removes common command line arguments, reads the registry.
   * Finally, if connect is TRUE, it opens the display.
   */
  virtual void init(VALUE arr,FXbool connect=TRUE);

  /**
   * Exit application.
   * Closes the display and writes the registry.
   */
  virtual void exit(FXint code=0);
  }
%enddef

%define DECLARE_FXBITMAP_VIRTUALS(klass)
%extend klass {
  /// Render pixels
  virtual void render();

  /// Release the client-side pixels buffer, freeing it if it was owned.
  virtual void release();

  /// Save pixel data only
  virtual FXbool savePixels(FXStream& store) const;
  
  /// Load pixel data
  virtual FXbool loadPixels(FXStream& store);

  /// Rescale pixels to the specified width and height
  virtual void scale(FXint w,FXint h);

  /// Mirror bitmap horizontally and/or vertically
  virtual void mirror(FXbool horizontal,FXbool vertical);

  /// Rotate bitmap by degrees ccw
  virtual void rotate(FXint degrees);

  /// Crop bitmap to given rectangle
  virtual void crop(FXint x,FXint y,FXint w,FXint h);

  /// Fill bitmap with uniform value
  virtual void fill(FXbool color);
  }
%enddef

%define DECLARE_FXCURSOR_VIRTUALS(klass)
%extend klass {
  /// Save pixel data only
  virtual FXbool savePixels(FXStream& store) const;
  
  /// Load pixel data
  virtual FXbool loadPixels(FXStream& store);
  }
%enddef

%define DECLARE_FXDC_VIRTUALS(klass)
%extend klass {
  /// Read back pixel
  virtual FXColor readPixel(FXint x,FXint y);

  /// Draw points
  virtual void drawPoint(FXint x,FXint y);
  virtual void drawPoints(const FXPoint* points,FXuint npoints);
  virtual void drawPointsRel(const FXPoint* points,FXuint npoints);

  /// Draw lines
  virtual void drawLine(FXint x1,FXint y1,FXint x2,FXint y2);
  virtual void drawLines(const FXPoint* points,FXuint npoints);
  virtual void drawLinesRel(const FXPoint* points,FXuint npoints);
  virtual void drawLineSegments(const FXSegment* segments,FXuint nsegments);
  
  /// Draw rectangles
  virtual void drawRectangle(FXint x,FXint y,FXint w,FXint h);
  virtual void drawRectangles(const FXRectangle* rectangles,FXuint nrectangles);

  /**
  * Draw arcs.
  * The argument ang1 specifies the start of the arc relative to the 
  * three-o'clock position from the center, in units of degrees*64.
  * The argument ang2 specifies the path and extent of the arc relative 
  * to the start of the arc, in units of degrees*64.
  * The arguments x,y,w,h specify the bounding rectangle.
  */
  virtual void drawArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2);
  virtual void drawArcs(const FXArc* arcs,FXuint narcs);

  /// Filled rectangles
  virtual void fillRectangle(FXint x,FXint y,FXint w,FXint h);
  virtual void fillRectangles(const FXRectangle* rectangles,FXuint nrectangles);

  /// Draw chords
  virtual void fillChord(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2);
  virtual void fillChords(const FXArc* chord,FXuint nchords);

  /// Draw arcs
  virtual void fillArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2);
  virtual void fillArcs(const FXArc* arcs,FXuint narcs);

  /// Filled polygon
  virtual void fillPolygon(const FXPoint* points,FXuint npoints);
  virtual void fillConcavePolygon(const FXPoint* points,FXuint npoints);
  virtual void fillComplexPolygon(const FXPoint* points,FXuint npoints);

  /// Filled polygon with relative points
  virtual void fillPolygonRel(const FXPoint* points,FXuint npoints);
  virtual void fillConcavePolygonRel(const FXPoint* points,FXuint npoints);
  virtual void fillComplexPolygonRel(const FXPoint* points,FXuint npoints);

  /// Draw hashed box
  virtual void drawHashBox(FXint x,FXint y,FXint w,FXint h,FXint b=1);

  /// Draw focus rectangle
  virtual void drawFocusRectangle(FXint x,FXint y,FXint w,FXint h);
  
  /// Draw area from source
  virtual void drawArea(const FXDrawable* source,FXint sx,FXint sy,FXint sw,FXint sh,FXint dx,FXint dy);

  /// Draw image 
  virtual void drawImage(const FXImage* image,FXint dx,FXint dy);

  /// Draw bitmap
  virtual void drawBitmap(const FXBitmap* bitmap,FXint dx,FXint dy);
  
  /// Draw icon 
  virtual void drawIcon(const FXIcon* icon,FXint dx,FXint dy);
  virtual void drawIconShaded(const FXIcon* icon,FXint dx,FXint dy);
  virtual void drawIconSunken(const FXIcon* icon,FXint dx,FXint dy);

  /// Draw string
  virtual void drawText(FXint x,FXint y,const FXchar* string,FXuint length);
  virtual void drawImageText(FXint x,FXint y,const FXchar* string,FXuint length);

  /// Set foreground drawing color
  virtual void setForeground(FXColor clr);
  
  /// Set background drawing color
  virtual void setBackground(FXColor clr);
  
  /**
  * Set dash pattern and dash offset.
  * A dash pattern of [1 2 3 4] is a repeating pattern of 1 foreground pixel, 
  * 2 background pixels, 3 foreground pixels, and 4 background pixels.
  * The offset is where in the pattern the system will start counting.
  * The maximum length of the dash pattern is 32.
  */
  virtual void setDashes(FXuint dashoffset,const FXchar *dashpattern,FXuint dashlength);

  /// Set line width:- 0 means thinnest/fastest possible
  virtual void setLineWidth(FXuint linewidth=0);
  
  /// Set line cap style
  virtual void setLineCap(FXCapStyle capstyle=CAP_BUTT);
  
  /// Set line join style
  virtual void setLineJoin(FXJoinStyle joinstyle=JOIN_MITER);
  
  /// Set line style
  virtual void setLineStyle(FXLineStyle linestyle=LINE_SOLID);
  
  /// Set fill style
  virtual void setFillStyle(FXFillStyle fillstyle=FILL_SOLID);
  
  /// Set fill rule
  virtual void setFillRule(FXFillRule fillrule=RULE_EVEN_ODD);
  
  /// Set rasterop function
  virtual void setFunction(FXFunction func=BLT_SRC);
  
  /// Set the tile image
  virtual void setTile(FXImage* image,FXint dx=0,FXint dy=0);
  
  /// Set the stipple pattern 
  virtual void setStipple(FXBitmap *bitmap,FXint dx=0,FXint dy=0);
  
  /// Set the stipple pattern 
  virtual void setStipple(FXStipplePattern pat,FXint dx=0,FXint dy=0);
  
  /// Set clip region
  virtual void setClipRegion(const FXRegion& region);
  
  /// Set clip rectangle
  virtual void setClipRectangle(FXint x,FXint y,FXint w,FXint h);

  /// Change clip rectangle
  virtual void setClipRectangle(const FXRectangle& rectangle);
  
  /// Clear clipping
  virtual void clearClipRectangle();
  
  /// Set clip mask
  virtual void setClipMask(FXBitmap* bitmap,FXint dx=0,FXint dy=0);
  
  /// Clear clip mask
  virtual void clearClipMask();
  
  /// Set font to draw text with
  virtual void setFont(FXFont *fnt);
  
  /// Clip against child windows
  virtual void clipChildren(FXbool yes);
  }
%enddef

%define DECLARE_FXDIALOGBOX_VIRTUALS(klass)
%extend klass {
  /// Run modal invocation of the dialog
  virtual FXuint execute(FXuint placement=PLACEMENT_CURSOR);
  }
%enddef

%define DECLARE_FXDRAWABLE_VIRTUALS(klass)
%extend klass {
  /// Resize drawable to the specified width and height 
  virtual void resize(FXint w,FXint h);
  }
%enddef

%define DECLARE_FXFILEDICT_VIRTUALS(klass)
%extend klass {
  virtual FXFileAssoc* findFileBinding(const FXchar* pathname);
  virtual FXFileAssoc* findDirBinding(const FXchar* pathname);
  virtual FXFileAssoc* findExecBinding(const FXchar* pathname);
  }
%enddef

%define DECLARE_FXFOLDINGITEM_VIRTUALS(klass)
%extend klass {
  virtual void setText(const FXString& txt);
  virtual void setOpenIcon(FXIcon* icn);
  virtual void setClosedIcon(FXIcon* icn);
  virtual void setFocus(FXbool focus);
  virtual void setSelected(FXbool selected);
  virtual void setOpened(FXbool opened);
  virtual void setExpanded(FXbool expanded);
  virtual void setEnabled(FXbool enabled);
  virtual void setDraggable(FXbool draggable);
  virtual void setIconOwned(FXuint owned);
  virtual FXint getWidth(const FXFoldingList* list) const;
  virtual FXint getHeight(const FXFoldingList* list) const;
  virtual void create();
  virtual void detach();
  virtual void destroy();
  }
%enddef

%define DECLARE_FXGLCANVAS_VIRTUALS(klass)
%extend klass {
  /// Make OpenGL context current prior to performing OpenGL commands
  virtual FXbool makeCurrent();
  
  /// Make OpenGL context non current
  virtual FXbool makeNonCurrent();

  /// Return TRUE if context is current
  virtual FXbool isCurrent() const;

  /// Swap front and back buffer
  virtual void swapBuffers();
  }
%enddef

%define DECLARE_FXGLOBJECT_VIRTUALS(klass)
%extend klass {
  /// Clone this object
  virtual FXGLObject* copy();

  /// Called by the viewer to get bounds for this object
  virtual FXRangef bounds();

  /// Draw this object in a viewer
  virtual void draw(FXGLViewer* viewer);

  /// Draw this object for hit-testing purposes
  virtual void hit(FXGLViewer* viewer);

  /// Return true if this object can be dragged around
  virtual FXbool canDrag() const;

  /// Return true if this object can be deleted from the scene
  virtual FXbool canDelete() const;

  /// Drag this object from one position to another
  virtual FXbool drag(FXGLViewer* viewer,FXint fx,FXint fy,FXint tx,FXint ty);
  }
%enddef

%define DECLARE_FXGLSHAPE_VIRTUALS(klass)
%extend klass {
  virtual void drawshape(FXGLViewer*);
  }
%enddef

%define DECLARE_FXGLVIEWER_VIRTUALS(klass)
%extend klass {
  /// Return a NULL-terminated list of all objects in the given rectangle, or NULL
  virtual FXGLObject** select(FXint x,FXint y,FXint w,FXint h);

  /// Perform a pick operation, returning the object at the given x,y position, or NULL
  virtual FXGLObject* pick(FXint x,FXint y);
  }
%enddef

%define DECLARE_FXHEADERITEM_VIRTUALS(klass)
%extend klass {
  virtual void setText(const FXString& txt);
  virtual void setIcon(FXIcon *ic);
  virtual FXint getWidth(const FXHeader* header) const;
  virtual FXint getHeight(const FXHeader* header) const;
  virtual void create();
  virtual void detach();
  virtual void destroy();
  }
%enddef

%define DECLARE_FXICONITEM_VIRTUALS(klass)
%extend klass {
  virtual void draw(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual FXint hitItem(const FXIconList* list,FXint rx,FXint ry,FXint rw=1,FXint rh=1) const;
  virtual void drawBigIcon(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual void drawMiniIcon(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual void drawDetails(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual void setText(const FXString& text);
  virtual void setBigIcon(FXIcon* icn);
  virtual void setMiniIcon(FXIcon* icn);
  virtual void setFocus(FXbool focus);
  virtual void setSelected(FXbool selected);
  virtual void setEnabled(FXbool enabled);
  virtual void setDraggable(FXbool draggable);
  virtual void setIconOwned(FXuint owned);
  virtual FXint getWidth(const FXIconList* list) const;
  virtual FXint getHeight(const FXIconList* list) const;
  virtual void create();
  virtual void detach();
  virtual void destroy();
  }
%enddef

%define DECLARE_FXICONLIST_VIRTUALS(klass)
%extend klass {
  /// Select item at index
  virtual FXbool selectItem(FXint index,FXbool notify=FALSE);
  
  /// Deselect item at index
  virtual FXbool deselectItem(FXint index,FXbool notify=FALSE);
  
  /// Toggle item at index
  virtual FXbool toggleItem(FXint index,FXbool notify=FALSE);

  /// Select items in rectangle
  virtual FXbool selectInRectangle(FXint x,FXint y,FXint w,FXint h,FXbool notify=FALSE);

  /// Extend selection from anchor index to index
  virtual FXbool extendSelection(FXint index,FXbool notify=FALSE);

  /// Deselect all items
  virtual FXbool killSelection(FXbool notify=FALSE);

  /// Change current item index
  virtual void setCurrentItem(FXint index,FXbool notify=FALSE);
  }
%enddef

%define DECLARE_FXID_VIRTUALS(klass)
%extend klass {
  virtual void create();
  virtual void detach();
  virtual void destroy();
  }
%enddef

%define DECLARE_FXIMAGE_VIRTUALS(klass)
%extend klass {
  /**
  * Retrieves pixels from the server-side image.  For example, to make
  * screen snapshots, or to retrieve an image after it has been drawin
  * into by various means.
  */
  virtual void restore();

  /**
  * Render the server-side representation of the image from client-side
  * pixels.  Normally, IMAGE_DITHER is used which causes the server-side
  * representation to be rendered using a 16x16 ordered dither if necessary;
  * however if IMAGE_NEAREST is used a faster (but uglier-looking), nearest
  * neighbor algorithm is used.  
  */
  virtual void render();

   /**
   * Release the client-side pixels buffer, free it if it was owned.
   * If it is not owned, the image just forgets about the buffer.
   */
  virtual void release();
  
  /**
  * Rescale pixels image to the specified width and height; this calls 
  * resize() to adjust the client and server side representations.
  */
  virtual void scale(FXint w,FXint h,FXint quality=0);
  
  /// Mirror image horizontally and/or vertically
  virtual void mirror(FXbool horizontal,FXbool vertical);

  /**
  * Rotate image by degrees ccw; this calls resize() to adjust the client 
  * and server side representations if necessary.
  */
  virtual void rotate(FXint degrees);

  /**
  * Crop image to given rectangle; this calls resize() to adjust the client 
  * and server side representations.
  */
  virtual void crop(FXint x,FXint y,FXint w,FXint h);

  /// Fill image with uniform color
  virtual void fill(FXColor color);

  /// Fade image to uniform color
  virtual void fade(FXColor color,FXint factor=255);

  /**
  * Shear image horizontally; the number of pixels is equal to the
  * shear parameter times 256.  The area outside the image is filled
  * with transparent black, unless another color is specified.
  */
  virtual void xshear(FXint shear,FXColor clr=0);

  /**
  * Shear image vertically; the number of pixels is equal to the
  * shear parameter times 256.  The area outside the image is filled
  * with transparent black, unless another color is specified.
  */
  virtual void yshear(FXint shear,FXColor clr=0);

  /// Fill horizontal gradient
  virtual void hgradient(FXColor left,FXColor right);

  /// Fill vertical gradient
  virtual void vgradient(FXColor top,FXColor bottom);

  /// Fill with gradient
  virtual void gradient(FXColor topleft,FXColor topright,FXColor bottomleft,FXColor bottomright);

  /// Blend image over uniform color
  virtual void blend(FXColor color);

  virtual FXbool savePixels(FXStream& store) const;
  virtual FXbool loadPixels(FXStream& store);
  }
%enddef

%define DECLARE_FXLISTITEM_VIRTUALS(klass)
%extend klass {
  virtual void setText(const FXString& txt);
  virtual void setIcon(FXIcon* icn);
  virtual void setFocus(FXbool focus);
  virtual void setSelected(FXbool selected);
  virtual void setEnabled(FXbool enabled);
  virtual void setDraggable(FXbool draggable);
  virtual void setIconOwned(FXuint owned);
  virtual FXint getWidth(const FXList* list) const;
  virtual FXint getHeight(const FXList* list) const;
  virtual void create();
  virtual void detach();
  virtual void destroy();
  }
%enddef

%define DECLARE_FXLIST_VIRTUALS(klass)
%extend klass {
  /// Select item
  virtual FXbool selectItem(FXint index,FXbool notify=FALSE);
  
  /// Deselect item
  virtual FXbool deselectItem(FXint index,FXbool notify=FALSE);
  
  /// Toggle item selection state
  virtual FXbool toggleItem(FXint index,FXbool notify=FALSE);

  /// Extend selection from anchor item to index
  virtual FXbool extendSelection(FXint index,FXbool notify=FALSE);
  
  /// Deselect all items
  virtual FXbool killSelection(FXbool notify=FALSE);

  /// Change current item
  virtual void setCurrentItem(FXint index,FXbool notify=FALSE);
  }
%enddef

%define DECLARE_FXMDICHILD_VIRTUALS(klass)
%extend klass {
  virtual FXbool minimize(FXbool notify=FALSE);
  virtual FXbool maximize(FXbool notify=FALSE);
  virtual FXbool restore(FXbool notify=FALSE);
  virtual FXbool close(FXbool notify=FALSE);
  }
%enddef

%define DECLARE_FXMDICLIENT_VIRTUALS(klass)
%extend klass {
  /// Set active MDI Child
  virtual FXbool setActiveChild(FXMDIChild* child=NULL,FXbool notify=TRUE);

  // Cascade windows
  virtual void cascade(FXbool notify=FALSE);

  // Layout horizontally
  virtual void horizontal(FXbool notify=FALSE);

  // Layout vertically
  virtual void vertical(FXbool notify=FALSE);
  }
%enddef

%define DECLARE_FXOBJECT_VIRTUALS(klass)
%extend klass {
  /// Save object to stream
  virtual void save(FXStream& store) const;

  /// Load object from stream
  virtual void load(FXStream& store);
  }
%enddef

%define DECLARE_FXPOPUP_VIRTUALS(klass)
%extend klass {
  virtual void popup(FXWindow* grabto,FXint x,FXint y,FXint w=0,FXint h=0);
  virtual void popdown();
  }
%enddef

%define DECLARE_FXREALSPINNER_VIRTUALS(klass)
%extend klass {
  virtual void setValue(FXdouble value);
  }
%enddef

%define DECLARE_FXSCROLLAREA_VIRTUALS(klass)
%extend klass {
  virtual FXint getViewportHeight();
  virtual FXint getViewportWidth();
  virtual FXint getContentHeight();
  virtual FXint getContentWidth();

  /**
   * Get the current position.
   * Note: This is not a virtual member function in the C++ library,
   * but we need to redeclare it here so that SWIG's overloading
   * mechanism will correctly handle both this version and the the
   * four-argument version of position() declared in class FXWindow.
   */
  VALUE position() const {
    FXint x, y;
    self->getPosition(x, y);
    VALUE pos = rb_ary_new();
    rb_ary_push(pos, INT2NUM(x));
    rb_ary_push(pos, INT2NUM(y));
    return pos;
  }
}
%enddef

%define DECLARE_FXSPINNER_VIRTUALS(klass)
%extend klass {
  virtual void setValue(FXint value);
  }
%enddef

%define DECLARE_FXSHUTTER_VIRTUALS(klass)
%extend klass {
  virtual void setCurrent(FXint panel);
  }
%enddef

%define DECLARE_FXSTREAM_VIRTUALS(klass)
%rename("setPosition") klass::position(long offset,FXWhence whence);
%extend klass {
  virtual FXbool close();
  virtual FXbool flush();
  virtual FXbool position(long offset,FXWhence whence=FXFromStart);
  }
%enddef

%define DECLARE_FXTABBAR_VIRTUALS(klass)
%extend klass {
  virtual void setCurrent(FXint panel,FXbool notify=FALSE);
  }
%enddef

%define DECLARE_FXTABLEITEM_VIRTUALS(klass)
%extend klass {
  virtual void draw(const FXTable* table,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual void drawBorders(const FXTable* table,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual void drawContent(const FXTable* table,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual void drawPattern(const FXTable* table,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual void drawBackground(const FXTable* table,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual void setText(const FXString& txt);
  virtual void setIcon(FXIcon* icn);
  virtual void setFocus(FXbool focus);
  virtual void setSelected(FXbool selected);
  virtual void setEnabled(FXbool enabled);
  virtual void setDraggable(FXbool draggable);
  virtual void setIconOwned(FXuint =0x00000040); // value of ICONOWNED
  virtual FXint getWidth(const FXTable* table) const;
  virtual FXint getHeight(const FXTable* table) const;
  virtual void create();
  virtual void detach();
  virtual void destroy();
  }
%enddef

%define DECLARE_FXTABLE_VIRTUALS(klass)
%extend klass {
  virtual void drawCell(FXDC& dc,FXint sr,FXint er,FXint sc,FXint ec);
  virtual void drawRange(FXDC& dc,FXint rlo,FXint rhi,FXint clo,FXint chi);
  virtual void drawHGrid(FXDC& dc,FXint rlo,FXint rhi,FXint clo,FXint chi);
  virtual void drawVGrid(FXDC& dc,FXint rlo,FXint rhi,FXint clo,FXint chi);
  virtual void drawContents(FXDC& dc,FXint x,FXint y,FXint w,FXint h);
  virtual FXTableItem* createItem(const FXString& text,FXIcon* icon,void* ptr);
  
  /// Set the table size to nr rows and nc columns; all existing items will be removed
  virtual void setTableSize(FXint nr,FXint nc,FXbool notify=FALSE);

  /// Insert new row
  virtual void insertRows(FXint TABLE_ROW_ALL,FXint nr=1,FXbool notify=FALSE);

  /// Insert new column
  virtual void insertColumns(FXint TABLE_COLUMN_ALL,FXint nc=1,FXbool notify=FALSE);

  /// Remove rows of cells
  virtual void removeRows(FXint row,FXint nr=1,FXbool notify=FALSE);

  /// Remove column of cells
  virtual void removeColumns(FXint col,FXint nc=1,FXbool notify=FALSE);

  /// Clear single cell
  virtual void removeItem(FXint row,FXint col,FXbool notify=FALSE);

  /// Clear all cells in the given range
  virtual void removeRange(FXint startrow,FXint endrow,FXint startcol,FXint endcol,FXbool notify=FALSE);

  /// Remove all items from table
  virtual void clearItems(FXbool notify=FALSE);

  /// Change column width
  virtual void setColumnWidth(FXint col,FXint cwidth);

  /// Change row height
  virtual void setRowHeight(FXint row,FXint rheight);

  /// Change X coordinate of column c
  virtual void setColumnX(FXint colEdge,FXint x);

  /// Change Y coordinate of row r
  virtual void setRowY(FXint rowEdge,FXint y);

  /// Change current item
  virtual void setCurrentItem(FXint r,FXint c,FXbool notify=FALSE);

  /// Select a row
  virtual FXbool selectRow(FXint row,FXbool notify=FALSE);

  /// Select a column
  virtual FXbool selectColumn(FXint col,FXbool notify=FALSE);

  /// Select range
  virtual FXbool selectRange(FXint startrow,FXint endrow,FXint startcol,FXint endcol,FXbool notify=FALSE);

  /// Extend selection
  virtual FXbool extendSelection(FXint r,FXint c,FXbool notify=FALSE);

  /// Kill selection
  virtual FXbool killSelection(FXbool notify=FALSE);
  }
%enddef

%define DECLARE_FXTEXT_VIRTUALS(klass)
%extend klass {
  virtual void setCursorPos(FXint pos,FXbool notify=FALSE);
  virtual FXbool extendSelection(FXint pos,FXTextSelectionMode mode=SELECT_CHARS,FXbool notify=FALSE);
  virtual FXbool killSelection(FXbool notify=FALSE);
  }
%enddef

%define DECLARE_FXTOOLBAR_VIRTUALS(klass)
%extend klass {
  virtual void dock(FXuint side=LAYOUT_SIDE_TOP,FXWindow* TOOLBAR_DOCK_AFTER=(FXWindow*)-1);
  virtual void undock();
  }
%enddef

%define DECLARE_FXTOPWINDOW_VIRTUALS(klass)
%extend klass {
  virtual void show(FXuint placement);
  virtual FXbool maximize(FXbool notify=FALSE);
  virtual FXbool minimize(FXbool notify=FALSE);
  virtual FXbool restore(FXbool notify=FALSE);
  virtual FXbool close(FXbool notify=FALSE);
  }
%enddef

%define DECLARE_FXTREEITEM_VIRTUALS(klass)
%extend klass {
  virtual void setText(const FXString& txt);
  virtual void setOpenIcon(FXIcon* icn);
  virtual void setClosedIcon(FXIcon* icn);
  virtual void setFocus(FXbool focus);
  virtual void setSelected(FXbool selected);
  virtual void setOpened(FXbool opened);
  virtual void setExpanded(FXbool expanded);
  virtual void setEnabled(FXbool enabled);
  virtual void setDraggable(FXbool draggable);
  virtual FXint getWidth(const FXTreeList* list) const;
  virtual FXint getHeight(const FXTreeList* list) const;
  virtual void create();
  virtual void detach();
  virtual void destroy();
  }
%enddef

%define DECLARE_FXTREELIST_VIRTUALS(klass)
%extend klass {
  /// Select item
  virtual FXbool selectItem(FXTreeItem* item,FXbool notify=FALSE);

  /// Deselect item
  virtual FXbool deselectItem(FXTreeItem* item,FXbool notify=FALSE);

  /// Toggle item selection
  virtual FXbool toggleItem(FXTreeItem* item,FXbool notify=FALSE);

  /// Extend selection from anchor item to item
  virtual FXbool extendSelection(FXTreeItem* item,FXbool notify=FALSE);
  
  /// Deselect all items
  virtual FXbool killSelection(FXbool notify=FALSE);

  /// Open item
  virtual FXbool openItem(FXTreeItem* item,FXbool notify=FALSE);

  /// Close item
  virtual FXbool closeItem(FXTreeItem* item,FXbool notify=FALSE);

  /// Collapse tree
  virtual FXbool collapseTree(FXTreeItem* tree,FXbool notify=FALSE);

  /// Expand tree 
  virtual FXbool expandTree(FXTreeItem* tree,FXbool notify=FALSE);

  /// Change current item
  virtual void setCurrentItem(FXTreeItem* item,FXbool notify=FALSE);
  }
%enddef

%define DECLARE_FXWINDOW_VIRTUALS(klass)
%rename("raiseWindow") klass::raise();
%extend klass {
  virtual FXint getDefaultWidth();
  virtual FXint getDefaultHeight();
  virtual FXint getWidthForHeight(FXint givenheight);
  virtual FXint getHeightForWidth(FXint givenwidth);
  virtual FXbool canFocus() const;
  virtual void setFocus();
  virtual void killFocus();
  virtual void setDefault(FXbool enable=TRUE);
  virtual void enable();
  virtual void disable();
  virtual void raise();
  virtual void lower();
  virtual void move(FXint x,FXint y);
  virtual void position(FXint x,FXint y,FXint w,FXint h);
  virtual void layout();
  virtual void recalc();
  virtual void reparent(FXComposite* newparent);
  virtual void show();
  virtual void hide();
  virtual FXbool isComposite() const;
  virtual FXbool contains(FXint parentx,FXint parenty) const;
  virtual FXbool doesSaveUnder() const;
  virtual void setBackColor(FXColor clr);
  }
%enddef

