/***********************************************************************
 * FXRuby -- the Ruby language bindings for the FOX GUI toolkit.
 * Copyright (c) 2001-2003 by J. Lyle Johnson. All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For further information please contact the author by e-mail
 * at "lyle@users.sourceforge.net".
 ***********************************************************************/

/// Double-precision quaternion
class FXQuatd : public FXVec4d {
public:

  /// Constructors
  FXQuatd();

  /// Copy constructor
  FXQuatd(const FXQuatd& q);

  /// Construct from components
  FXQuatd(FXdouble xx,FXdouble yy,FXdouble zz,FXdouble ww);

  /// Construct from axis and angle
  FXQuatd(const FXVec3d& axis,FXdouble phi=0.0);

  /// Construct from euler angles yaw (z), pitch (y), and roll (x)
  FXQuatd(FXdouble roll,FXdouble pitch,FXdouble yaw);

  /// Adjust quaternion length
  FXQuatd& adjust();

  /// Set quaternion from axis and angle
  void setAxisAngle(const FXVec3d& axis,FXdouble phi=0.0);

  %extend {
    /// Obtain axis and angle from quaternion
    VALUE getAxisAngle() const {
      FXVec3d axis;
      FXdouble phi;
      self->getAxisAngle(axis,phi);
      return rb_ary_new3(2,rb_ary_new3(3,rb_float_new(axis.x),rb_float_new(axis.y),rb_float_new(axis.z)),rb_float_new(phi));
      }
    }

  /// Set quaternion from yaw (z), pitch (y), and roll (x)
  void setRollPitchYaw(FXdouble roll,FXdouble pitch,FXdouble yaw);

  %extend {
    /// Obtain yaw, pitch, and roll from quaternion
    VALUE getRollPitchYaw() const {
      FXdouble roll,pitch,yaw;
      self->getRollPitchYaw(roll,pitch,yaw);
      return rb_ary_new3(3,rb_float_new(roll),rb_float_new(pitch),rb_float_new(yaw));
      }

    /// Exponentiate quaternion
    FXQuatd exp() const { return exp(*self); }
  
    /// Take logarithm of quaternion
    FXQuatd log() const { return log(*self); }
  
    /// Invert quaternion
    FXQuatd invert() const { return invert(*self); }
  
    /// Invert unit quaternion
    FXQuatd unitinvert() const { return unitinvert(*self); }
    
    /// Conjugate quaternion
    FXQuatd conj() const { return conj(*self); }
  
    /// Multiply quaternions
    FXQuatd operator*(const FXQuatd& other) const { return (*self)*other; }
  
    // Rotation of a vector by a quaternion
    FXVec3d operator*(const FXVec3d& vec) const { return (*self)*vec; }

    /// Construct quaternion from arc a->b on unit sphere
    static FXQuatd arc(const FXVec3d& a,const FXVec3d& b) { return arc(a,b); }
  
    /// Spherical lerp
    static FXQuatd lerp(const FXQuatd& u,const FXQuatd& v,FXdouble f) { return lerp(u,v,f); }
    }
  };

