/***********************************************************************
 * FXRuby -- the Ruby language bindings for the FOX GUI toolkit.
 * Copyright (c) 2001-2003 by J. Lyle Johnson. All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For further information please contact the author by e-mail
 * at "lyle@users.sourceforge.net".
 ***********************************************************************/

%rename("adjust!") FXQuatf::adjust();

/// Single-precision quaternion
class FXQuatf : public FXVec4f {
public:

  /// Constructors
  FXQuatf();

  /// Construct from axis and angle
  FXQuatf(const FXVec3f& axis,FXfloat phi=0.0);

  /// Construct from euler angles yaw (z), pitch (y), and roll (x)
  FXQuatf(FXfloat roll,FXfloat pitch,FXfloat yaw);

  /// Construct from components
  FXQuatf(FXfloat x,FXfloat y,FXfloat z,FXfloat w);

  /// Adjust quaternion length
  FXQuatf& adjust();

  /// Set quaternion from axis and angle
  void setAxisAngle(const FXVec3f& axis,FXfloat phi=0.0f);

  %extend {
    /// Obtain axis and angle from quaternion
    VALUE getAxisAngle() const {
      FXVec3f axis;
      FXfloat phi;
      self->getAxisAngle(axis,phi);
      return rb_ary_new3(2,rb_ary_new3(3,rb_float_new(axis.x),rb_float_new(axis.y),rb_float_new(axis.z)),rb_float_new(phi));
      }
    }

  /// Set quaternion from yaw (z), pitch (y), and roll (x)
  void setRollPitchYaw(FXfloat roll,FXfloat pitch,FXfloat yaw);

  %extend {
    /// Obtain yaw, pitch, and roll from quaternion
    VALUE getRollPitchYaw() const {
      FXfloat roll,pitch,yaw;
      self->getRollPitchYaw(roll,pitch,yaw);
      return rb_ary_new3(3,rb_float_new(roll),rb_float_new(pitch),rb_float_new(yaw));
      }

    /// Exponentiate quaternion
    FXQuatf exp() const { return exp(*self); }

    /// Take logarithm of quaternion
    FXQuatf log() const { return log(*self); }

    /// Invert quaternion
    FXQuatf invert() const { return invert(*self); }

    /// Invert unit quaternion
    FXQuatf unitinvert() const { return unitinvert(*self); }

    /// Conjugate quaternion
    FXQuatf conj() const { return conj(*self); }

    /// Multiply quaternions
    FXQuatf operator*(const FXQuatf& other) const { return (*self)*other; }

    // Rotation of a vector by a quaternion
    FXVec3f operator*(const FXVec3f& vec) const { return (*self)*vec; }

    /// Construct quaternion from arc a->b on unit sphere
    static FXQuatf arc(const FXVec3f& a,const FXVec3f& b){ return arc(a,b); }

    /// Spherical lerp
    static FXQuatf lerp(const FXQuatf& u,const FXQuatf& v,FXfloat f){ return lerp(u,v,f); }
    }
  };
