/***********************************************************************
 * FXRuby -- the Ruby language bindings for the FOX GUI toolkit.
 * Copyright (c) 2001-2003 by J. Lyle Johnson. All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For further information please contact the author by e-mail
 * at "lyle@users.sourceforge.net".
 ***********************************************************************/

/// Tree List Box styles
enum {
  TREELISTBOX_NORMAL         = 0          /// Normal style
  };

  
class FXButton;
class FXMenuButton;
class FXTreeList;
class FXPopup;


/// Tree List Box
class FXTreeListBox : public FXPacker {
protected:
  FXButton      *field;
  FXMenuButton  *button;
  FXTreeList    *tree;
  FXPopup       *pane;
protected:
  FXTreeListBox(){}
public:
  long onFocusUp(FXObject*,FXSelector,void* PTR_EVENT);
  long onFocusDown(FXObject*,FXSelector,void* PTR_EVENT);
  long onFocusSelf(FXObject*,FXSelector,void* PTR_EVENT);
  long onChanged(FXObject*,FXSelector,void* PTR_TREEITEM);
  long onCommand(FXObject*,FXSelector,void* PTR_TREEITEM);
  long onFieldButton(FXObject*,FXSelector,void* PTR_IGNORE);
  long onTreeChanged(FXObject*,FXSelector,void* PTR_TREEITEM);
  long onTreeClicked(FXObject*,FXSelector,void* PTR_TREEITEM);
  long onUpdFmTree(FXObject*,FXSelector,void* PTR_IGNORE);
public:
  enum{
    ID_TREE=FXPacker::ID_LAST,
    ID_FIELD,
    ID_LAST
    };
public:
  
  /// Construct tree list box
  %extend {
    FXTreeListBox(FXComposite* p,FXObject* tgt=NULL,FXSelector sel=0,FXuint opts=FRAME_SUNKEN|FRAME_THICK|TREELISTBOX_NORMAL,FXint x=0,FXint y=0,FXint w=0,FXint h=0,FXint pl=DEFAULT_PAD,FXint pr=DEFAULT_PAD,FXint pt=DEFAULT_PAD,FXint pb=DEFAULT_PAD){
      return new FXRbTreeListBox(p,tgt,sel,opts,x,y,w,h,pl,pr,pt,pb);
      }
    }

  /// Return number of items
  FXint getNumItems() const;
  
  /// Return number of visible items
  FXint getNumVisible() const;
  
  /// Set number of visible items to determine default height
  void setNumVisible(FXint nvis);
  
  /// Return first top-level item
  FXTreeItem* getFirstItem() const;
  
  /// Return last top-level item
  FXTreeItem* getLastItem() const;

  %extend {
    /// Add item as first child of parent p
    FXTreeItem* addItemFirst(FXTreeItem* p,FXTreeItem* item){
      if(item->isMemberOf(FXMETACLASS(FXRbTreeItem))){
        dynamic_cast<FXRbTreeItem*>(item)->owner=self;
	}
      return self->addItemFirst(p,item);
    }
  
    /// Add item as last child after parent p
    FXTreeItem* addItemLast(FXTreeItem* p,FXTreeItem* item){
      if(item->isMemberOf(FXMETACLASS(FXRbTreeItem))){
        dynamic_cast<FXRbTreeItem*>(item)->owner=self;
	}
      return self->addItemLast(p,item);
    }
  
    /// Add item after other item
    FXTreeItem* addItemAfter(FXTreeItem* other,FXTreeItem* item){
      if(item->isMemberOf(FXMETACLASS(FXRbTreeItem))){
        dynamic_cast<FXRbTreeItem*>(item)->owner=self;
	}
      return self->addItemAfter(other,item);
    }
  
    /// Add item before other item
    FXTreeItem* addItemBefore(FXTreeItem* other,FXTreeItem* item){
      if(item->isMemberOf(FXMETACLASS(FXRbTreeItem))){
        dynamic_cast<FXRbTreeItem*>(item)->owner=self;
	}
      return self->addItemBefore(other,item);
    }
  
    /// Add item as first child of parent p
    FXTreeItem* addItemFirst(FXTreeItem* p,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ITEMDATA=NULL){
      FXTreeItem* item=self->addItemFirst(p,text,oi,ci,ITEMDATA);
      if(item->isMemberOf(FXMETACLASS(FXRbTreeItem))){
        dynamic_cast<FXRbTreeItem*>(item)->owner=self;
	}
      return item;
      }
    
    /// Add item as last child of parent p
    FXTreeItem* addItemLast(FXTreeItem* p,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ITEMDATA=NULL){
      FXTreeItem* item=self->addItemLast(p,text,oi,ci,ITEMDATA);
      if(item->isMemberOf(FXMETACLASS(FXRbTreeItem))){
        dynamic_cast<FXRbTreeItem*>(item)->owner=self;
	}
      return item;
      }
    
    /// Add item after other item
    FXTreeItem* addItemAfter(FXTreeItem* other,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ITEMDATA=NULL){
      FXTreeItem* item=self->addItemAfter(other,text,oi,ci,ITEMDATA);
      if(item->isMemberOf(FXMETACLASS(FXRbTreeItem))){
        dynamic_cast<FXRbTreeItem*>(item)->owner=self;
	}
      return item;
      }
    
    /// Add item before other item
    FXTreeItem* addItemBefore(FXTreeItem* other,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ITEMDATA=NULL){
      FXTreeItem* item=self->addItemBefore(other,text,oi,ci,ITEMDATA);
      if(item->isMemberOf(FXMETACLASS(FXRbTreeItem))){
        dynamic_cast<FXRbTreeItem*>(item)->owner=self;
	}
      return item;
      }
    
    /// Remove item
    void removeItem(FXTreeItem* item){
      // Save pointer(s) to the soon-to-be-destroyed items
      FXObjectListOf<FXTreeItem> items;
      FXRbTreeList::enumerateItem(item,items);
      
      // Do the deed
      self->removeItem(item);
      
      // Now zero-out pointers held by still-alive Ruby objects
      for(FXint i=0;i<items.no();i++){
        FXRbUnregisterRubyObj(items[i]);
        }
      }
    
    /// Remove all items in range [fm...to]
    void removeItems(FXTreeItem* fm,FXTreeItem* to){
      // Save pointer(s) to the soon-to-be-destroyed items
      FXObjectListOf<FXTreeItem> items;
      FXRbTreeList::enumerateItems(fm,to,items);

      // Do the deed
      self->removeItems(fm,to);

      // Now zero-out pointers held by still-alive Ruby objects
      for(FXint i=0;i<items.no();i++){
        FXRbUnregisterRubyObj(items[i]);
        }
      }
    
    /// Remove all items from list
    void clearItems(){
      // Save pointer(s) to the soon-to-be-destroyed items
      FXObjectListOf<FXTreeItem> items;
      FXRbTreeList::enumerateItems(self->getFirstItem(),self->getLastItem(),items);

      // Do the deed
      self->clearItems();

      // Now zero-out pointers held by still-alive Ruby objects
      for(FXint i=0;i<items.no();i++){
        FXRbUnregisterRubyObj(items[i]);
        }
      }
  }
  
  /**
  * Search items for item by name, starting from start item; the
  * flags argument controls the search direction, and case sensitivity.
  */
  FXTreeItem* findItem(const FXString& text,FXTreeItem* start=NULL,FXuint flags=SEARCH_FORWARD|SEARCH_WRAP) const;

  /// Return TRUE if item is the current item
  FXbool isItemCurrent(const FXTreeItem* item) const;
  
  /// Return TRUE if item is leaf-item, i.e. has no children
  FXbool isItemLeaf(const FXTreeItem* item) const;
  
  /// Sort the toplevel items with the sort function
  void sortRootItems();
 
  /// Sort all items recursively
  void sortItems();

  /// Sort child items of item
  void sortChildItems(FXTreeItem* item);
  
  /// Change current item
  void setCurrentItem(FXTreeItem* item,FXbool notify=FALSE);
  
  /// Return current item
  FXTreeItem* getCurrentItem() const;
  
  /// Change item label
  void setItemText(FXTreeItem* item,const FXString& text);
  
  /// Return item label
  FXString getItemText(const FXTreeItem* item) const;
  
  /// Change item's open icon
  void setItemOpenIcon(FXTreeItem* item,FXIcon* icon);
  
  /// Return item's open icon
  FXIcon* getItemOpenIcon(const FXTreeItem* item) const;
  
  /// Change item's closed icon
  void setItemClosedIcon(FXTreeItem* item,FXIcon* icon);
  
  /// Return item's closed icon
  FXIcon* getItemClosedIcon(const FXTreeItem* item) const;
  
  %extend {
    /// Change item's user data
    void setItemData(FXTreeItem* item,VALUE ptr){
      self->setItemData(item,(void*) ptr);
      }
  
    /// Return item's user data
    VALUE getItemData(const FXTreeItem* item) const {
      return self->getItemData(item) ? (VALUE) self->getItemData(item) : Qnil;
      }
  }
  
  /// Is the pane shown
  FXbool isPaneShown() const;
  
  /// Change font
  void setFont(FXFont* fnt);
  
  /// Return font
  FXFont* getFont() const;
  
  /// Return list style
  FXuint getListStyle() const;
  
  /// Change list style
  void setListStyle(FXuint style);
  
  /// Change help text
  void setHelpText(const FXString& txt);
  
  /// Return help text
  FXString getHelpText() const;
  
  /// Change tip text
  void setTipText(const FXString& txt);
  
  /// Return tip text
  FXString getTipText() const;

  /// Destructor
  virtual ~FXTreeListBox();
  };


DECLARE_FXOBJECT_VIRTUALS(FXTreeListBox)
DECLARE_FXID_VIRTUALS(FXTreeListBox)
DECLARE_FXDRAWABLE_VIRTUALS(FXTreeListBox)
DECLARE_FXWINDOW_VIRTUALS(FXTreeListBox)

%{
static swig_type_info *FXTreeListBox_dynamic_cast(void **ptr) {
    FXTreeListBox **ppTreeListBox = reinterpret_cast<FXTreeListBox **>(ptr);
    FXDirBox *pDirBox=dynamic_cast<FXDirBox*>(*ppTreeListBox);
    if(pDirBox){
      *ptr=reinterpret_cast<void*>(pDirBox);
      return SWIG_TypeQuery("FXDirBox *");
      }
    return 0;
}
%}

DYNAMIC_CAST(SWIGTYPE_p_FXTreeListBox, FXTreeListBox_dynamic_cast);

