/***********************************************************************
 * FXRuby -- the Ruby language bindings for the FOX GUI toolkit.
 * Copyright (c) 2001-2003 by J. Lyle Johnson. All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For further information please contact the author by e-mail
 * at "lyle@users.sourceforge.net".
 ***********************************************************************/

/// Construction options for FXVisual class
enum FXVisualOptions {
  VISUAL_DEFAULT      = 0,            /// Default visual
  VISUAL_MONOCHROME   = 1,            /// Must be monochrome visual
  VISUAL_BEST         = 2,            /// Best (deepest) visual
  VISUAL_INDEXCOLOR   = 4,            /// Palette visual
  VISUAL_GRAYSCALE    = 8,            /// Gray scale visual
  VISUAL_TRUECOLOR    = 16,           /// Must be true color visual
  VISUAL_OWNCOLORMAP  = 32,           /// Allocate private colormap
  VISUAL_DOUBLEBUFFER = 64,           /// Double-buffered [FXGLVisual]
  VISUAL_STEREO       = 128,          /// Stereo [FXGLVisual]
  VISUAL_NOACCEL      = 256,          /// No hardware acceleration [for broken h/w]
  VISUAL_SWAP_COPY    = 512           /// Buffer swap by copying [FXGLVisual]
  };


/// Visual type
enum FXVisualType {
  VISUALTYPE_UNKNOWN,                 /// Undetermined visual type
  VISUALTYPE_MONO,                    /// Visual for drawing into 1-bpp surfaces
  VISUALTYPE_TRUE,                    /// True color
  VISUALTYPE_INDEX,                   /// Index [palette] color
  VISUALTYPE_GRAY                     /// Gray scale
  };
  
%rename("visualType") FXVisual::getType() const;

/// Visual describes pixel format of a drawable
class FXVisual : public FXId {
protected:
  FXuint        flags;                  // Visual flags
  FXuint        hint;                   // Depth Hint
  FXuint        depth;                  // Visual depth, significant bits/pixel
  FXuint        numred;                 // Number of reds
  FXuint        numgreen;               // Number of greens
  FXuint        numblue;                // Number of blues
  FXuint        numcolors;              // Total number of colors
  FXuint        maxcolors;              // Maximum number of colors
  FXVisualType  type;                   // Visual type
  void         *info;                   // Opaque data
  FXID          colormap;               // Color map, if any
  FXbool        freemap;                // We allocated the map
#ifndef WIN32
protected:
  void         *visual;                 // Application visual [Visual]
  void*         gc;                     // Drawing GC
  void*         scrollgc;               // Scrolling GC
  FXPixel       rpix[16][256];          // Mapping from red -> pixel
  FXPixel       gpix[16][256];          // Mapping from green -> pixel
  FXPixel       bpix[16][256];          // Mapping from blue -> pixel
  FXPixel       lut[256];               // Color lookup table
protected:
  void setuptruecolor();
  void setupdirectcolor();
  void setuppseudocolor();
  void setupstaticcolor();
  void setupgrayscale();
  void setupstaticgray();
  void setuppixmapmono();
  void setupcolormap();
#else
protected:
  int           pixelformat;            // PIXELFORMAT number
#endif
protected:
  FXVisual();
public:

  /// Construct default visual
  %extend {
    FXVisual(FXApp* a,FXuint flgs,FXuint d=32){
      return new FXRbVisual(a,flgs,d);
      }
    }

  /// Get visual type
  FXVisualType getType() const;

  /// Get flags (see FXVisualOptions)
  FXuint getFlags() const;

  /// Get depth, i.e. number of significant bits in color representation
  FXuint getDepth() const;

  /// Get number of colors
  FXuint getNumColors() const;

  /// Get number of reds
  FXuint getNumRed() const;

  /// Get number of greens
  FXuint getNumGreen() const;

  /// Get number of blues
  FXuint getNumBlue() const;

  /// Get device pixel value for color
  FXPixel getPixel(FXColor clr);

  /// Get color value for device pixel value
  FXColor getColor(FXPixel pix);

  /// Set maximum number of colors to allocate
  void setMaxColors(FXuint maxcols);

  /// Get maximum number of colors
  FXuint getMaxColors() const;

  /// Destructor
  virtual ~FXVisual();
  };


DECLARE_FXOBJECT_VIRTUALS(FXVisual)
DECLARE_FXID_VIRTUALS(FXVisual)

%{
static swig_type_info *FXVisual_dynamic_cast(void **ptr) {
    FXVisual **ppVisual = reinterpret_cast<FXVisual **>(ptr);
    FXGLVisual *pGLVisual=dynamic_cast<FXGLVisual*>(*ppVisual);
    if(pGLVisual){
      *ptr=reinterpret_cast<void*>(pGLVisual);
      return SWIG_TypeQuery("FXGLVisual *");
      }
    return 0;
}
%}

DYNAMIC_CAST(SWIGTYPE_p_FXVisual, FXVisual_dynamic_cast);

