=begin
= GSL::Vector class

== Class methods

--- GSL::Vector.alloc(ary) 
--- GSL::Vector.new(ary) 
--- GSL::Vector.new(range) 
--- GSL::Vector.new(size) 
--- GSL::Vector.new(elm0, elm1, ....) 
--- GSL::Vector[elm0, elm1, ....]
    Constructors.

    Ex:
       irb(main):002:0> v1 = Vector.alloc(5)
       => GSL::Vector: [ 0.000e+00 0.000e+00 0.000e+00 0.000e+00 0.000e+00 ]
       irb(main):003:0> v2 = Vector.alloc(1, 3, 5, 2)
       => GSL::Vector: [ 1.000e+00 3.000e+00 5.000e+00 2.000e+00 ]
       irb(main):004:0> v3 = Vector[1, 3, 5, 2]
       => GSL::Vector: [ 1.000e+00 3.000e+00 5.000e+00 2.000e+00 ]
       irb(main):005:0> v4 = Vector.alloc([1, 3, 5, 2])
       => GSL::Vector: [ 1.000e+00 3.000e+00 5.000e+00 2.000e+00 ]
       irb(main):006:0> v5 = Vector[1..6]
       => GSL::Vector: [ 1.000e+00 2.000e+00 3.000e+00 4.000e+00 5.000e+00 6.000e+00 ]

--- GSL::Vector.calloc(size)
    This method creates a vector object, and initializes all the elements to zero.

--- GSL::Vector.linspace(min, max, n = 10)
    Returns an (({GSL::Vector})) with ((|n|)) linearly spaced elements 
    between ((|min|)) and ((|max|)). If ((|min|)) is greater than ((|max|)), 
    the elements are stored in decreasing order. 

    Ex: 
        irb(main):002:0> x = Vector.linspace(0, 10, 5)
        [ 0.000e+00 2.500e+00 5.000e+00 7.500e+00 1.000e+01 ]
        irb(main):003:0> y = Vector.linspace(10, 0, 5)
        [ 1.000e+01 7.500e+00 5.000e+00 2.500e+00 0.000e+00 ]

--- GSL::Vector.logspace(min, max, n)
    Similar to (({GSL::Vector#linspace})) except that the values are 
    logarithmically spaced from 10^((|min|)) to 10^((|max|)).

    Ex:
        irb(main):007:0* x = Vector.logspace(1, 3, 5)
        [ 1.000e+01 3.162e+01 1.000e+02 3.162e+02 1.000e+03 ]
        irb(main):008:0> x = Vector.logspace(3, 1, 5)
        [ 1.000e+03 3.162e+02 1.000e+02 3.162e+01 1.000e+01 ]

--- GSL::Vector.logspace2(min, max, n)
    Similar to (({GSL::Vector#linspace})) except that the values are 
    logarithmically spaced from ((|min|)) to ((|max|)).

    Ex:
        irb(main):010:0* x = Vector.logspace2(10, 1000, 5)
        [ 1.000e+01 3.162e+01 1.000e+02 3.162e+02 1.000e+03 ]
        irb(main):011:0> x = Vector.logspace2(1000, 10, 5)
        [ 1.000e+03 3.162e+02 1.000e+02 3.162e+01 1.000e+01 ]

--- GSL::Vector.indgen(n, start=0, step=1)
    This creates a vector of length ((|n|)) with elements from ((|start|))
    with interval ((|step|)) (mimics NArray#indgen).
    
    Ex:
        irb(main):019:0> v = Vector::Int.indgen(5)
        => GSL::Vector::Int: 
        [ 0 1 2 3 4 ]
        irb(main):020:0> v = Vector::Int.indgen(5, 3)
        => GSL::Vector::Int: 
        [ 3 4 5 6 7 ]
        irb(main):021:0> v = Vector::Int.indgen(5, 3, 2)
        => GSL::Vector::Int: 
        [ 3 5 7 9 11 ]

=== NArray Extension
If an (({NArray})) object is given, a newly allocated vector is created.

Ex:
        na = NArray[1.0, 2, 3, 4, 5]
        p na                <----- NArray.float(5): 
                                   [ 1.0, 2.0, 3.0, 4.0, 5.0]
        v = Vector.new(na)  
        p v                 <----- [ 1 2 3 4 5 ]
                                   #<GSL::Vector:0x367ff4>

== NOTE:
In Ruby/GSL, vector lendth is limited within the range of Fixnum.
For 32-bit CPU, the maximum of vector length is 2^30 ~ 1e9.

== Methods

--- GSL::Vector#get(indices)
--- GSL::Vector#[indices]
    Return elements(s) of the vector ((|self|)). 

--- GSL::Vector#set(i, val)
--- GSL::Vector#[]=
    Set the ((|i|))-th element of the vector ((|self|)) to ((|val|)).

    Ex:
          irb(main):001:0> require("gsl")
          => true
          irb(main):002:0> v = Vector[0..5]
          => GSL::Vector: [ 0.000e+00 1.000e+00 2.000e+00 3.000e+00 4.000e+00 5.000e+00 ]
          irb(main):003:0> v[2]
          => 2.0
          irb(main):004:0> v[1, 3, 4]
          => GSL::Vector: [ 1.000e+00 3.000e+00 4.000e+00 ]
          irb(main):005:0> v[1..3]
          => GSL::Vector::View: [ 1.000e+00 2.000e+00 3.000e+00 ]
          irb(main):006:0> v[3] = 9
          => 9
          irb(main):007:0> v[-1] = 123
          => 123
          irb(main):008:0> v
          => GSL::Vector: [ 0.000e+00 1.000e+00 2.000e+00 9.000e+00 4.000e+00 1.230e+02 ]

--- GSL::Vector#set_all(x)
    This method sets all the elements of the vector to the value ((|x|)).

--- GSL::Vector#set_zero
    This method sets all the elements of the vector to zero.

--- GSL::Vector#set_basis!(i)
    This method makes a basis vector by setting all the elements of the vector
    to zero except for the ((|i|))-th element, which is set to one. 
    For a vector (({v})) of size 10, the method
      v.set_basis!(4)
    sets the vector ((|v|)) to a basis vector (({[0, 0, 0, 0, 1, 0, 0, 0, 0, 0]})). 

--- GSL::Vector#set_basis(i)
    This method returns a new basis vector by setting all the elements of the 
    vector to zero except for the i-th element which is set to one. 
    For a vector (({v})) of size 10, the method
      vb = v.set_basis(4)
    creates a new vector ((|vb|)) with elements (({[0, 0, 0, 0, 1, 0, 0, 0, 0, 0]})). 
    The vector ((|v|)) is not changed.

--- GSL::Vector#indgen!(start=0, step=1)
--- GSL::Vector#indgen(start=0, step=1)
    Mimics NArray#indgen!.

--- GSL::Vector#each
    An iterator for each of the vector elements, used as
      v.each do |x|
        p x
      end

--- GSL::Vector#each_index
    Another iterator.

--- GSL::Vector#collect { |item| .. }

=== IO
--- GSL::Vector#print
--- GSL::Vector#fprintf(io, format = "%e")
--- GSL::Vector#fprintf(filename, format = "%e")
--- GSL::Vector#fscanf(io)
--- GSL::Vector#fscanf(filename)
--- GSL::Vector#fwrite(io)
--- GSL::Vector#fwrite(filename)
--- GSL::Vector#fread(io)
--- GSL::Vector#fread(filename)
    Methods for writing or reading the vector. The first argument is IO or String object.

=== Copying vectors
--- GSL::Vector#clone
--- GSL::Vector#duplicate
    Create a new vector of the same elements.

=== Vector views
The (({GSL::Vector::View})) class is defined to be used as "references" to
vectors. Since the (({Vector::View})) class is a subclass of (({Vector})),
an instance of the (({View})) class created by slicing a (({Vector})) object 
can be used same as the original vector. A
(({View})) object shares the data with the original vector, i.e. any changes
in the elements of the (({View})) object affect to the original vector.

--- GSL::Vector#subvector
--- GSL::Vector#subvector(n)
--- GSL::Vector#subvector(offset, n)
--- GSL::Vector#subvector(offset, stride, n)
    Create a (({Vector::View})) object slicing ((|n|)) elements
    of the vector ((|self|)) from the offset ((|offset|)). If called with one
    argument ((|n|)), ((|offset|)) is set to 0. With no arguments, a view is
    created with the same length of the original vector.

    * Ex:
       #!/usr/bin/env ruby
       require("gsl")

       v = Vector[1, 2, 3, 4, 5, 6]
       view = v.subvector(1, 4)
       p view.class         <----- GSL::Vector::View
       view.print           <----- [ 2 3 4 5 ]

       view[2] = 99
       view.print           <----- [ 2 3 99 5 ]
       v.print              <----- [ 1 2 3 99 5 6 ]

--- GSL::Vector#subvector_with_stride(offset, n, stride)
    Return a (({Vector::View})) object of a subvector of another vector ((|self|)) 
    with an additional stride argument. The subvector is formed in the same way 
    as for (({Vector#subvector})) but the new vector view has ((|n|)) elements 
    with a step-size of ((|stride|)) from one element to the next in the original vector. 

--- GSL::Vectir#matrix_view(n1, n2)
    This creates a (({Matrix::View})) object from the vector ((|self|)).
    It enables to use the vector as a ((<Matrix|URL:matrix.html>)) object.

    * Ex:

        irb(main):019:0> v = Vector::Int.alloc(1..9)
        => GSL::Vector::Int: 
        [ 1 2 3 4 5 6 7 8 9 ]
        irb(main):020:0> m = v.matrix_view(3, 3)
        => GSL::Matrix::Int::View: 
        [ 1 2 3 
          4 5 6 
          7 8 9 ]
        irb(main):021:0> m[1][2] = 99
        => 99
        irb(main):022:0> v
        => GSL::Vector::Int: 
        [ 1 2 3 4 5 99 7 8 9 ]

=== Vector operations

--- GSL::Vector#swap_elements(i, j)
    This method exchanges the i-th and j-th elements of the vector ((|in-place|)). 

--- GSL::Vector#reverse
    Reverses the order of the elements of the vector.

      irb(main):025:0> v = Vector::Int[1..5]
      => GSL::Vector::Int: 
      [ 1 2 3 4 5 ]
      irb(main):026:0> v.reverse
      => GSL::Vector::Int: 
      [ 5 4 3 2 1 ]

--- GSL::Vector#trans
--- GSL::Vector#transpose
--- GSL::Vector#col
--- GSL::Vector#row
    Transpose the vector from a row vector into a column vector and vice versa.

      irb(main):029:0> v = Vector::Int[1..5]
      => GSL::Vector::Int: 
      [ 1 2 3 4 5 ]
      irb(main):030:0> v.col
      => GSL::Vector::Int::Col: 
      [ 1 
        2 
        3 
        4 
        5 ]

--- GSL::Vector#add(b)
    Adds the elements of vector ((|b|)) to the elements 
    of the vector ((|self|)). A new vector is created, and the vector
    ((|self|)) is not changed.

--- GSL::Vector#sub(b)
    Subtracts the element of vector ((|b|)) from the elements of ((|self|)).
    A new vector is created, and the vector ((|self|)) is not changed.

--- GSL::Vector#mul(b)
    Multiplies the elements of vector ((|self|)) by the elements of vector ((|b|)).
--- GSL::Vector#div(b)
    Divides the elements of vector ((|self|)) by the elements of vector ((|b|)).

--- GSL::Vector#scale(x)
    This method multiplies the elements of vector ((|self|)) by 
    the constant factor ((|x|)).

--- GSL::Vector#add_constant(x)
    Adds the constant value ((|x|)) to the elements of the vector ((|self|)).

--- GSL::Vector#+(b)
    For ((|b|)),
      * a Number: ---> (({self.add_constanb(b)}))
      * a Vector: ---> (({self.add(b)}))
--- GSL::Vector#-(b)
    For ((|b|)),
      * a Number: ---> (({self.add_constanb(-b)}))
      * a Vector: ---> (({self.sub(b)}))
--- GSL::Vector#/(b)
    For ((|b|)),
      * a Number: ---> (({self.scale(1/b)}))
      * a Vector: ---> (({self.div(b)}))

--- GSL::Vector#*(b)
    Vector multiplication.

    (1) Scale
          irb(main):027:0> v = Vector[1, 2]
          [ 1 2 ]
          irb(main):028:0> v*2
          [ 2 4 ]                           
    (2) Element-by-element multiplication
          irb(main):018:0> a = Vector[1, 2]; b = Vector[3, 4]
          [ 3 4 ]
          irb(main):020:0> a*b
          [ 3 8 ]                             
    (3) Inner product
          irb(main):023:0> a = Vector[1, 2]; b = Vector[3, 4]
          [ 3 
            4 ]
          irb(main):024:0> a*b.col
          => 11.0                        
    (4) Vector::Col*Vector -> Matrix
          irb(main):025:0> a = Vector::Col[1, 2]; b = Vector[3, 4]
          [ 3 4 ]
          irb(main):026:0> a*b
          [ 3 4 
            6 8 ]
    (5) Matrix*Vector::Col -> Vector::Col
          irb(main):029:0> a = Vector[1, 2]; m = Matrix[[2, 3], [4, 5]]
          [ 2 3 
            4 5 ]
          irb(main):030:0> m*a          <--- Error
          TypeError: Operation with GSL::Vector is not defined (GSL::Vector::Col expected)
                  from (irb):30:in `*'
                  from (irb):30
          irb(main):031:0> m*a.col
          [ 8 
            14 ]


--- GSL::Vector#swap_elements(i, j)
    This exchanges the ((|i|))-th and ((|j|))-th elements of the vector ((|self|)) in-place.
--- GSL::Vector#clone
--- GSL::Vector#duplicate
    These create a copy of the vector ((|self|)).
    
--- GSL::Vector.connect(v1, v2, v3, ...)
--- GSL::Vector#connect(v1, v2, v3, ...)
    Creates a new vector by connecting all the elements of the given vectors.

      irb(main):031:0> v1 = Vector::Int[1, 3]
      => GSL::Vector::Int: 
      [ 1 3 ]
      irb(main):032:0> v2 = Vector::Int[4, 3, 5]
      => GSL::Vector::Int: 
      [ 4 3 5 ]
      irb(main):033:0> v1.connect(v2)
      => GSL::Vector::Int: 
      [ 1 3 4 3 5 ]

--- GSL::Vector#abs
    Creates a new vector, with elements ((|fabs(x_i)|)).

      irb(main):034:0> v = Vector::Int[-3, 2, -5, 4]
      => GSL::Vector::Int: 
      [ -3 2 -5 4 ]
      irb(main):035:0> v.abs
      => GSL::Vector::Int: 
      [ 3 2 5 4 ]

--- GSL::Vector#square
    Creates a new vector, with elements ((|x_i*x_i|)).

      irb(main):036:0> v = Vector::Int[1..4]
      => GSL::Vector::Int: 
      [ 1 2 3 4 ]
      irb(main):037:0> v.square
      => GSL::Vector::Int: 
      [ 1 4 9 16 ]

--- GSL::Vector#sqrt
    Creates a new vector, with elements (({sqrt}))(((|x_i|))).

--- GSL::Vector#normalize(nrm = 1.0)
    Creates a new vector of norm ((|nrm|)), by scaling the vector ((|self|)).
--- GSL::Vector#normalize!(nrm = 1.0)
    This normalizes the vector ((|self|)) in-place.

    Ex:
      tcsh> irb
      irb(main):001:0> require("gsl")
      => true
      irb(main):002:0> a = Vector[-1, -2, -3, -4]
      => GSL::Vector: 
      [ -1.000e+00 -2.000e+00 -3.000e+00 -4.000e+00 ]
      irb(main):003:0> b = a.abs
      => GSL::Vector: 
      [ 1.000e+00 2.000e+00 3.000e+00 4.000e+00 ]
      irb(main):004:0> b.sqrt
      => GSL::Vector: 
      [ 1.000e+00 1.414e+00 1.732e+00 2.000e+00 ]
      irb(main):005:0> b.square
      => GSL::Vector: 
      [ 1.000e+00 4.000e+00 9.000e+00 1.600e+01 ]
      irb(main):006:0> c = b.normalize(2)
      => GSL::Vector: 
      [ 2.582e-01 5.164e-01 7.746e-01 1.033e+00 ]
      irb(main):007:0> c.square.sum
      => 2.0

=== Finding maximum and minimum elements of vectors

--- GSL::Vector#max
    This method returns the maximum value in the vector.

--- GSL::Vector#min
    This method returns the minimum value in the vector.

--- GSL::Vector#minmax
    This method returns an array of two elements, the minimum and the maximum values
    in the vector ((|self|)).

--- GSL::Vector#max_index
    This method returns the index of the maximum value in the vector. When there are 
    several equal maximum elements then the lowest index is returned. 

--- GSL::Vector#min_index
    This method returns the index of the minimum value in the vector. When there are 
    several equal minimum elements then the lowest index is returned. 

--- GSL::Vector#minmax_index
    This method returns an array of two elements which has the indices of the minimum 
    and the maximum values in the vector ((|self|)).

=== Vector Properties
--- GSL::Vector#size
--- GSL::Vector#len
    Return the vector length.

--- GSL::Vector#sum
    Returns the sum of the vector elements.

--- GSL::Vector#prod
    Returns the product of the vector elements.

--- GSL::Vector#isnull
    Returns 1 if all the elements of the vector ((|self|)) 
    are zero, and 0 otherwise.
--- GSL::Vector#isnull?
    Returns (({true})) if all the elements of the vector ((|self|)) 
    are zero, and (({false})) otherwise.

--- GSL::Vector#equal?(other, eps = 1e-10)
--- GSL::Vector#==(other, eps = 1e-10)
    Returns (({true})) if the vectors have same size and elements
    equal to absolute accurary ((|eps|)) for all the indices, 
    and (({false})) otherwise.

=== Sorting

--- GSL::Vector#sort
--- GSL::Vector#sort!
    These methods sort the vector ((|self|)) in ascending numerical order.

--- GSL::Vector#sort_index
    This method indirectly sorts the elements of the vector ((|self|)) into 
    ascending order, and returns the resulting permutation. 
    The elements of permutation give the index of the vector element which 
    would have been stored in that position if the vector had been sorted in place. 
    The first element of permutation gives the index of the least element in the
    vector, and the last element of permutation gives the index of the greatest 
    vector element. The vector ((|self|)) is not changed.

--- GSL::Vector#sort_smallest(n)
--- GSL::Vector#sort_largest(n)
--- GSL::Vector#sort_smallest_index(n)
--- GSL::Vector#sort_largest_index(n)

    Ex:

        irb(main):005:0> v = Vector::Int[8, 2, 3, 7, 9, 1, 4]
        => GSL::Vector::Int: 
        [ 8 2 3 7 9 1 4 ]
        irb(main):006:0> v.sort
        => GSL::Vector::Int: 
        [ 1 2 3 4 7 8 9 ]
        irb(main):007:0> v.sort_index
        => GSL::Permutation: 
        [ 5 1 2 6 3 0 4 ]
        irb(main):008:0> v.sort_largest(3)
        => GSL::Vector::Int: 
        [ 9 8 7 ]
        irb(main):009:0> v.sort_smallest(3)
        => GSL::Vector::Int: 
        [ 1 2 3 ]

=== BLAS Methods
--- GSL::Vector#nrm2
--- GSL::Vector#dnrm2
    Compute the Euclidean norm ||x||_2 = sqrt {sum x_i^2} of the vector.

--- GSL::Vector#asum
--- GSL::Vector#dasum
    Compute the absolute sum \sum |x_i| of the elements of the vector.

=== Data type conversions
--- GSL::Vector#to_a 
    This method converts the vector into a Ruby array. A Ruby array also can be
    converted into a GSL::Vector object with the (({to_gv})) method. For example,

      v = GSL::Vector.alloc([1, 2, 3, 4, 5])
      a = v.to_a   -> GSL::Vector to an array
      p a          -> [1.0, 2.0, 3.0, 4.0, 5.0]
      a[2] = 12.0
      v2 = a.to_gv  -> a new GSL::Vector object
      v2.print     -> 1.0000e+00 2.0000e+00 1.2000e+01 4.0000e+00 5.0000e+00

--- GSL::Vector#to_m(nrow, ncol)
    Creates a (({GSL::Matrix})) object of ((|nrow|)) rows and ((|ncol|)) columns.

      irb(main):011:0> v = Vector::Int[1..5]
      => GSL::Vector::Int: 
      [ 1 2 3 4 5 ]
      irb(main):012:0> v.to_m(2, 3)
      => GSL::Matrix::Int: 
      [ 1 2 3 
        4 5 0 ]
      irb(main):013:0> v.to_m(2, 2)
      => GSL::Matrix::Int: 
      [ 1 2 
        3 4 ]
      irb(main):014:0> v.to_m(3, 2)
      => GSL::Matrix::Int: 
      [ 1 2 
        3 4 
        5 0 ]

--- GSL::Vector#to_m_diagonal
    Converts the vector into a diagonal matrix. 
    See also ((<GSL::Matrix.diagonal(v)|URL:matrix.html>)).

       irb(main):012:0> v = Vector[1..4].to_i
       => GSL::Vector::Int: 
       [ 1 2 3 4 ]
       irb(main):013:0> v.to_m_diagonal
       => GSL::Matrix::Int: 
       [ 1 0 0 0 
         0 2 0 0 
         0 0 3 0 
         0 0 0 4 ]

--- GSL::Vector#to_m_circulant
    Creates a circulant matrix.

      irb(main):002:0> v = Vector::Int[1..5]
      => GSL::Vector::Int: 
      [ 1 2 3 4 5 ]
      irb(main):003:0> v.to_m_circulant
      => GSL::Matrix::Int: 
      [ 5 1 2 3 4   
        4 5 1 2 3   
        3 4 5 1 2   
        2 3 4 5 1   
        1 2 3 4 5 ]

--- GSL::Vector#to_tensor(rank, dimension)

--- GSL::Vector#to_na
    The Vector object ((|self|)) is converted into an (({NArray})) object. 
    The data are copied to newly allocated memory.

--- NArray#to_gv
--- NArray#to_gslv
    Create (({GSL::Vector})) object from the (({NArray})) object ((|self|)).

--- NArray#to_gv_view
--- NArray#to_gslv_view
    A (({GSL::Vector::View})) object is created from the NArray object ((|self|)). 
    This method does not allocate memory for the data: the data of ((|self|)) 
    are not copied, but shared with the (({View})) object created, thus 
    any modifications to the (({View})) object affect on the original NArray 
    object. In other words, the (({View})) object can be used as a ((|reference|)) 
    to the NArray object. 

    Ex:
       tcsh> irb
       irb(main):001:0> require("gsl")
       => true
       irb(main):002:0> na = NArray[1.0, 2, 3, 4, 5]    
       => NArray.float(5): 
       [ 1.0, 2.0, 3.0, 4.0, 5.0 ]
       irb(main):003:0> vv = na.to_gv_view   # Create a view sharing the memory
       => GSL::Vector::View: 
       [ 1.000e+00 2.000e+00 3.000e+00 4.000e+00 5.000e+00 ]
       irb(main):004:0> vv[3] = 9
       => 9
       irb(main):005:0> na
       => NArray.float(5): 
       [ 1.0, 2.0, 3.0, 9.0, 5.0 ]           # The data are changed
       irb(main):006:0> v = na.to_gv         # A vector with newly allocated memory
       => GSL::Vector: 
       [ 1.000e+00 2.000e+00 3.000e+00 9.000e+00 5.000e+00 ]
       irb(main):007:0> v[1] = 123
       => 123
       irb(main):008:0> v
       => GSL::Vector: 
       [ 1.000e+00 1.230e+02 3.000e+00 9.000e+00 5.000e+00 ]
       irb(main):009:0> na                   
       => NArray.float(5): 
       [ 1.0, 2.0, 3.0, 9.0, 5.0 ]           # v and na are independent 

=== Graphics
--- GSL::Vector.graph(y)
--- GSL::Vector.graph(y, options)
--- GSL::Vector.graph(x, y)
--- GSL::Vector.graph(x, y, options)
--- GSL::Vector#graph(options)
--- GSL::Vector#graph(x, options)
    These methods use the GNU plotutils (({graph})) application to plot
    vector ((|self|)). The option (({graph})) as "-T X -C" is given by a String.

((<prev|URL:sf.html>))
((<next|URL:matrix.html>))

((<Reference index|URL:ref.html>))
((<top|URL:index.html>))
    
=end
