
module Logging

  #(mostly) static Logger
  class Logger
    # Set level of visibility.
    #  DEBUG => 5
    #  INFO  => 4
    #  WARN  => 3
    #  ERROR => 2
    #  FAIL  => 1
    #  NO    => 0
    ActualLevel = ROTConfig::LOGLEVEL
    # Display which method is called? (SLOW!!)
    MethodInfo = ROTConfig::LOGMETHODINFO

    #statics
    @@stackregexp = /.*:in `(\w+)'/ 
    @@appender = nil

    # Call this method to log into the specified appender.
    # If no appender was given, no logging is done.
    def Logger.log_message(name, level, message)
      if (!@@appender.nil?)
        if (MethodInfo)
          @@stackregexp.match(caller().at(2))
          methodinfo = $1
          @@appender.log("[#{name} (#{level})] #{methodinfo}: #{message}\n")
        else
          @@appender.log("[#{name} (#{level})] #{message}\n")
        end
      end
    end

    # Use this method, to specify your appender.
    def Logger.set_appender(appender)
      @@appender= appender
    end
  end

  module Log
    #debug message
    def debug(name, message)
      log_message(name, "DEBUG", message) if (Logger::ActualLevel>=5)
    end
    #info message
    def info(name, message)
      log_message(name, "INFO", message) if (Logger::ActualLevel>=4)
    end
    #warn message
    def warn(name, message)
      log_message(name, "WARN", message) if (Logger::ActualLevel>=3)
    end
    #error message
    def error(name, message)
      log_message(name, "ERROR", message) if (Logger::ActualLevel>=2)
    end
    #fail message
    def fail(name, message)
      log_message(name, "FAIL", message) if (Logger::ActualLevel>=1)
    end
    #log message to logging instance
    def log_message(name, level, message)
      Logger.log_message(name.to_s, level, message)
    end
  end

  module NamedLog
    #name of the category to log
    attr :category_logname
    #aka ctor ...
    def init_log(name)
      if (name.is_a?(Class))
        @category_logname = name.class.name
      else
        @category_logname = name
      end
    end
    #debug message
    def debug(message)
      log_message("DEBUG", message) if (Logger::ActualLevel>=5)
    end
    #info message
    def info(message)
      log_message("INFO", message) if (Logger::ActualLevel>=4)
    end
    #warn message
    def warn(message)
      log_message("WARN", message) if (Logger::ActualLevel>=3)
    end
    #error message
    def error(message)
      log_message("ERROR", message) if (Logger::ActualLevel>=2)
    end
    #fail message
    def fail(message)
        log_message("FAIL", message) if (Logger::ActualLevel>=1)
    end
    #log message to logging instance
    def log_message(level, message)
      Logger.log_message(category_logname, level, message)
    end
  end

  # Abstract appender. Inherit, override and rule!
  class Appender
    def log(message)
      #raise "override this method!"
    end
  end
      
  # Console Appender - simply dump to console.
  class ConsoleAppender < Appender
    def log(message)
      print("#{message}")
    end
  end

  # File Appender - write to file.
  class FileAppender < Appender
    attr :file
    attr :count
    FLUSH = 10
    def initialize(file)
      begin
        @file = File.open(file, "a+")
      rescue => ex
        puts ">>>!! FileLogger - Error: #{ex} - no logging during session !!"
        puts ">>>!! Change file to log in Logger.rb line 146 !!"
        @file = nil
      end
      @count = 0
    end
    def log(message)
      if (@file)
        @file.write("#{message}")
        @count = (@count+1)%FLUSH 
        if (@count==0)
          @file.flush
        end
      end
    end
  end
  # Which appender to log to.
  Logger.set_appender(ConsoleAppender.new())
  #Logger.set_appender(FileAppender.new("/tmp/teamlog"))
end


# Logger.rb   April 2002
#
# Copyright (c) 2002 by Matthias Veit <matthias_veit@yahoo.de>
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option) 
# any later version.
#  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of 
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
# GNU General Public License for more details.
#  
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
# 02111-1307, USA.
