=begin
= Description
   spreadsheet - A package for generating MS Excel compatable files on
   any platform.

   Current version: 0.2.8
   Based on version .26 of John McNamara's Perl module Spreadsheet::WriteExcel.
= Synopsis
   require "spreadsheet/excel"
   include Spreadsheet

   workbook = Excel.new("test.xls")

   # There are three ways to create a format
   format = workbook.add_format(:color=>"blue",:bold=>1,:underline=>1)

   format2 = Format.new(
      :color     => "green",
      :bold      => true,
      :underline => true
   )
   workbook.add_format(format2)

   format3 = Format.new{ |f|
      f.color = "red"
      f.bold  = true
      f.underline = true
   }
   workbook.add_format(format3)

   worksheet1 = workbook.add_worksheet
   worksheet2 = workbook.add_worksheet("Numbers")
   worksheet3 = workbook.add_worksheet("Text")
   
   worksheet1.write(0,0,"Hello",format)
   worksheet1.write(1,1,["Matz","Larry","Guido"])

   worksheet2.write(1,3,8876,format2)
   worksheet2.write_column(4,4,[1,2,3])

   worksheet3.write(2,2,"World",format3)
   worksheet3.write(3,3,[[1,2,3],[4,5,6],[7,8,9]])

   worksheet1.format_row(1,25,format1)
   worksheet2.format_column(0..2,30,format2)

   workbook.close
= Constants
--- VERSION
    The current version number (a string).
= Classes
== Excel
--- Excel.new(filename)
    Returns a workbook object.  You may only have one workbook per file.  A
    ".xls" extension for your filename is recommended but not enforced.
== Workbook
--- Workbook#add_worksheet(sheet_name)
    Adds a worksheet to the workbook object.  You may optionally pass a sheet
    name.  Otherwise, it will default to 'Sheet1', 'Sheet2', etc.  Returns
    a (({Worksheet})) instance.
--- Workbook#add_format(attributes/format_object)
    Adds a Format to the workbook.  When included as part of the 'write' method,
    the cells specified are formatted appropriately.  Returns a (({Format}))
    instance.

    See the synopsis above for different ways to add formats.
--- Workbook#close
    Closes the workbook (and the file).  Be sure to do this.
== Worksheet
--- Worksheet#write(row,column,value,format=nil)
    Writes data to the cell you provide.  If ((*value*)) is an Array, the
    (({write_row})) method is called internally.
    
    If ((*value*)) is a multi-dimensional array, then each element of the array
    is written within its own row at the appropriate column.  In other words,
    it's written to the spreadsheet in the same manner you would visualize it.
    
    If a format is provided, the cells are each formatted appropriately.
--- Worksheet#write_row(row,column,Array,format=nil)
    Writes a row of data starting at ((*row*)) and ((*column*)) in a left to
    right manner, with one array element per cell.

    Any formatting will be applied to each cell.
--- Worksheet#write_column(row,column,Array,format=nil)
    Writes a column of data starting at ((*row*)) and ((*column*)) in a top to
    bottom manner, with one array element per cell.

    Any formatting will be applied to each cell.
--- Worksheet#format_row(int/range,height=nil,format=nil)
    Applies formatting for an entire row or range of rows.  In addition, you
    can specify the row height.  If you wish to apply row-level formatting
    without modifying the height, simply pass (({nil})) as the second argument.

    Note that this will override any previously defined cell formatting.
--- Worksheet#format_column(int/range,width=nil,format=nil)
    Applies formatting for an entire row or range of columns.  In addition, you
    can specify the column width.  If you wish to apply column-level formatting
    without modifying the width, simply pass (({nil})) as the second argument.

    Note that this will override any previously defined cell formatting.
== Format
--- Format.new({attributes})
    Creates a new Format object.  See the Format documentation for details.
= Notes
    This is a port of John McNamara's Spreadsheet::WriteExcel Perl module,
    version .26. There is no support for formulas yet.
= Design Changes
    The only somewhat major change was to make OLEWriter a subclass of File,
    rather than store a filehandle as an attribute within the class.  This
    seems to have worked out fine.

    The set_row() and set_column() methods were renamed as format_row()
    and format_column(), respectively.  All other methods are either
    identical in name or very similar.
   
    Other changes consisted mostly of minor code optimizations.  Occasionally
    I was more terse than John was (for better or for worse).
= Questions
    Questions about MS Excel should be directed to Microsoft.
    Questions about the MS Excel format should be directed to Microsoft.
    Questions about why I use the hex values that I use should be directed to
    John McNamara (jmcnamara at cpan dot org) or to Microsoft.
= Future Plans
    Add support for files > 7MB - need help
    Add formulas - need help
    RDoc documentation
= Thanks
    Many thanks go to John McNamara for his tireless dedication to a very useful
    and popular module.  I also thank him for taking the time to answer some of
    the questions I had for him while porting his code.
= License
    Ruby's
= Copyright
     2002-2004, Daniel J. Berger

    All Rights Reserved. This module is free software. It may be used,
    redistributed and/or modified under the same terms as Ruby itself.
= Warranty
    This package is provided "as is" and without any express or
    implied warranties, including, without limitation, the implied
    warranties of merchantability and fitness for a particular purpose.
= Author
    Daniel J. Berger
    djberg96 at yahoo dot com
    rubyhacker1/imperator on IRC
=end
