/*file.c

--GNU LGPL
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#ifndef WIN32
#include <sys/file.h>
#endif


#include "../command.h"
#include "../filesys.h"
#include "../match.h"
#include "../matchc.h"

#define THISFILEP  pFCO->Descriptor[FileNumber].fp
#define THISSOCKET pFCO->Descriptor[FileNumber].sp
/*
File functions and commands

This file contains the code for the commands and functions that deal with files.
*/
#ifndef WIN32
int stricmp(char *,char*);
#endif

#define MAXFILES 512
typedef struct _FileCommandObject {
  union {
    FILE *fp;  /* the file pointer to the opened file */
    SOCKET sp;
    }Descriptor[MAXFILES];
  long RecordSize[MAXFILES]; /* the length of a record */
  char mode[MAXFILES]; /* the mode the file was opened 'i', 'o', 'a', 'r' or 'b' */
                       /* 's' for client sockets */
                       /* '\0' for not opened file or socket */
  int SocketState[MAXFILES]; /*0 normal, -1 kind of EOF */
  } FileCommandObject, *pFileCommandObject;

static void close_all_files(pExecuteObject pEo){
  pFileCommandObject pFCO;
  long FileNumber;

  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);
  for( FileNumber = 0 ; FileNumber < MAXFILES ; FileNumber++ ){
    if( pFCO->mode[FileNumber] ){
      if( pFCO->mode[FileNumber] == 's' )
        HOOK_TCPCLOSE(THISSOCKET);
      else
        HOOK_FCLOSE(THISFILEP);
      }
    THISFILEP = NULL;
    }
  }

static int init(pExecuteObject pEo){
#define INITIALIZE init(pEo)
  pFileCommandObject pFCO;
  int i;

  /* initialize only once */
  if( PARAMPTR(CMD_OPEN) )return 0;
  PARAMPTR(CMD_OPEN) = ALLOC(sizeof(FileCommandObject));
  if( PARAMPTR(CMD_OPEN) == NULL )return COMMAND_ERROR_MEMORY_LOW;

  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);
  for( i=0 ; i < MAXFILES ; i++ )
    pFCO->mode[i] = (char)0;
  FINALPTR(CMD_OPEN) = close_all_files;
  return 0;
  }

/* Create the directory for a file in case it does not exists. */
static void prepare_directory(pExecuteObject pEo,char *pszFileName){
  int i;
  char *s;

  s = pszFileName + (i=strlen(pszFileName)) -1;
  while( i ){
    if( *s == '/' || *s == '\\' ){
      i = *s;
      *s = (char)0;
      HOOK_MAKEDIRECTORY(pszFileName);
      *s = (char)i;
      return;
      }
    i--;
    s--;
    }
  }

/* check that a file name is secure */
static int FileIsSecure(pExecuteObject pEo,VARIABLE vFileName){
  unsigned long i;

  for( i=0 ; i < STRLEN(vFileName) ; i++ ){
    if( STRINGVALUE(vFileName)[i] == 0 )return 0;
    }
  return 1;
  }

#define SECUREFILE(x) if( ! FileIsSecure(pEo,x) )ERROR(COMMAND_ERROR_FILE_CANNOT_BE_OPENED);

/**OPEN
=section file
=title OPEN file_name FOR mode AS [ # ] i [ LEN=record_length ]
Open or create and open a file. The syntax of the line is 

=verbatim
OPEN file_name FOR mode AS [ # ] i [ LEN=record_length ]
=noverbatim

The parameters:
=itemize
=item T<file_name> if the name of the file to be opened. If the mode allows the file to be written
the file is created if it did not existed before. If neede directory is created for the file.

=item T<mode> is the mode the file is opened. It can be:
 =itemize
 =item T<input> open the file for reading
 =item T<output> open the file for writing. If the file existed it's content is deleted first.
 =item T<append> open a possibly existing file and write after the current content.
 =item T<random> open the file for reading and writing (textual mode)
 =item T<binary> open the file for reading and writing (binary mode)
 =item T<socket> open a socket
 =noitemize

=item T<#i> is the file number. After the file has been opened this number has to be used in later
file handling functions and commands, like R<CLOSE> to refer to the file. The T<#> character is optional
and is allowed for compatibility with other BASIC languages. The number can be between 1 and 512.
This number is quite big for most of the applications and provides compatibility with VisualBasic.

=item T<record_length> is optional and specify the length of a racord in the file. The default
record length is 1 byte.
=noitemize
*/
COMMAND(OPEN)
#if NOTIMP_OPEN
NOTIMPLEMENTED;
#else


  long FileNumber,RecLen;
  char *FileName;
  char *ModeString;
  pFileCommandObject pFCO;
  VARIABLE Op1,FN,vLEN;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  Op1 = CONVERT2STRING(_EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  NEXTPARAMETER;
  ModeString = pEo->StringTable+pEo->CommandArray[_ActualNode-1].Parameter.CommandArgument.Argument.szStringValue;
  NEXTPARAMETER;
  FN = _EVALUATEEXPRESSION(PARAMETERNODE);
  ASSERTOKE;
  NEXTPARAMETER;
  vLEN = CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;

  if( vLEN == NULL )ERROR(COMMAND_ERROR_BAD_RECORD_LENGTH);
  RecLen = LONGVALUE(vLEN); /* the default record length is 1 */
  if( RecLen < 1 )ERROR(COMMAND_ERROR_BAD_RECORD_LENGTH);

  if( FN == NULL )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  if( TYPE(FN) == VTYPE_LONG && LONGVALUE(FN) == 0 ){/* we have to automatically allocate the file number */

    for( FileNumber = 1 ; FileNumber < MAXFILES ; FileNumber++ )
      if( ! pFCO->mode[FileNumber] )break;
    if( FileNumber >= MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
    LONGVALUE(FN) = FileNumber;
    }
  FileNumber = LONGVALUE(CONVERT2LONG(FN));

  if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  FileNumber --;
  if( pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_NUMBER_IS_USED);

  FileName = ALLOC(STRLEN(Op1)+1);
  if( FileName == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  memcpy(FileName,STRINGVALUE(Op1),STRLEN(Op1));
  FileName[STRLEN(Op1)] = (char)0;

  if( !stricmp(ModeString,"socket") ){
    if( HOOK_TCPCONNECT(&(THISSOCKET),FileName) ){
      FREE(FileName);
      pFCO->mode[FileNumber] = (char)0;
      ERROR(COMMAND_ERROR_FILE_CANNOT_BE_OPENED);
      }
    pFCO->mode[FileNumber] = 's';
    FREE(FileName);
    pFCO->RecordSize[FileNumber] = RecLen;
    pFCO->SocketState[FileNumber] = 0;
    RETURN;
    }else
  if( !stricmp(ModeString,"input") ){
    pFCO->mode[FileNumber] = 'i';
    THISFILEP = HOOK_FOPEN(FileName,"r");
    }else
  if( !stricmp(ModeString,"output") ){
    pFCO->mode[FileNumber] = 'o';
    prepare_directory(pEo,FileName);
    THISFILEP = HOOK_FOPEN(FileName,"w");
    }else
  if( !stricmp(ModeString,"append") ){
    pFCO->mode[FileNumber] = 'a';
    prepare_directory(pEo,FileName);
    THISFILEP = HOOK_FOPEN(FileName,"a");
    }else
  if( !stricmp(ModeString,"random") ){
    pFCO->mode[FileNumber] = 'r';
    prepare_directory(pEo,FileName);
    THISFILEP = HOOK_FOPEN(FileName,"r+");
    if( THISFILEP == NULL )
      THISFILEP = HOOK_FOPEN(FileName,"w+");
    }else
  if( !stricmp(ModeString,"binary") ){
    pFCO->mode[FileNumber] = 'b';
    prepare_directory(pEo,FileName);
    THISFILEP = HOOK_FOPEN(FileName,"rb+");
    if( THISFILEP == NULL )
      THISFILEP = HOOK_FOPEN(FileName,"wb+");
    }
  FREE(FileName);
  if( THISFILEP == NULL ){
    pFCO->mode[FileNumber] = (char)0;
    ERROR(COMMAND_ERROR_FILE_CANNOT_BE_OPENED);
    }
  pFCO->RecordSize[FileNumber] = RecLen;
#endif
END

static char *ReadFileLine(pExecuteObject pEo,
                          FILE *fp,
                          unsigned long *plCharactersRead,
                          int (*pfExtIn)(void *)){
  char *s,*r;
  unsigned long lBufferSize;
  int ch;
#define BUFFER_INCREASE 256

  s = ALLOC(BUFFER_INCREASE);
  if( s == NULL )return NULL;
  lBufferSize = BUFFER_INCREASE;
  *plCharactersRead = 0L;
  while( 1 ){
    if( (ch= (pfExtIn == NULL ? HOOK_FGETC(fp) : pfExtIn(pEo->pEmbedder) )) == EOF )break;
    if( lBufferSize <= *plCharactersRead ){
      r = ALLOC(lBufferSize+BUFFER_INCREASE);
      if( r == NULL ){
        FREE(s);
        return NULL;
        }
      memcpy(r,s,lBufferSize);
      lBufferSize += BUFFER_INCREASE;
      FREE(s);
      s = r;
      }
    s[(*plCharactersRead)++] = ch;
    if( ch == '\n' )break;
    }
  return s;
  }

static char *ReadSocketLine(pExecuteObject pEo,
                            SOCKET sp,
                            unsigned long *plCharactersRead){
  char *s,*r;
  unsigned long lBufferSize;
  char ch;
#define BUFFER_INCREASE 256

  s = ALLOC(BUFFER_INCREASE);
  if( s == NULL )return NULL;
  lBufferSize = BUFFER_INCREASE;
  *plCharactersRead = 0L;
  while( 1 ){
    if( HOOK_TCPRECV(sp,&ch,1,0) == 0 ){
      break;
      }
    if( lBufferSize <= *plCharactersRead ){
      r = ALLOC(lBufferSize+BUFFER_INCREASE);
      if( r == NULL ){
        FREE(s);
        return NULL;
        }
      memcpy(r,s,lBufferSize);
      lBufferSize += BUFFER_INCREASE;
      FREE(s);
      s = r;
      }
    s[(*plCharactersRead)++] = ch;
    if( ch == '\n' )break;
    }
  return s;
  }



/**LINEINPUT
=section file
=display LINE INPUT
=title LINE INPUT
Read a line from a file or from the standard input.

The syntax of the command is
=verbatim
LINE INPUT [# i , ] variable
=noverbatim

The parameter T<i> is the file number used in the open statement. If this is not specified the
standard input is read.

The T<variable> will hold a single line from the file read containing the possible new line character
terminating the line.

See also R<CHOMP>

*/
COMMAND(LINPUTF)
#if NOTIMP_LINPUTF
NOTIMPLEMENTED;
#else

  long FileNumber;
  pFileCommandObject pFCO;
  VARIABLE Result;
  LEFTVALUE LetThisVariable;
  unsigned long lCharactersRead;
  char *s;
  long refcount;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;
  NEXTPARAMETER;
  /* we get the pointer to the variable that points to the value */
  LetThisVariable = EVALUATELEFTVALUE(PARAMETERNODE);
  ASSERTOKE;
  DEREFERENCE(LetThisVariable)

  if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_IS_NOT_OPENED);

  if( pFCO->mode[FileNumber] == 's' ){
    s = ReadSocketLine(pEo,THISSOCKET,&lCharactersRead);
    if( lCharactersRead == 0 )pFCO->SocketState[FileNumber] = -1;
    }
  else
    s = ReadFileLine(pEo,THISFILEP,&lCharactersRead,NULL);
  if( s == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  Result = NEWSTRING(lCharactersRead);
  memcpy(STRINGVALUE(Result),s,lCharactersRead);
  FREE(s);

  /* if this variable had value assigned to it then release that value */
  if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);

  /* and finally assign the code to the variable */
  *LetThisVariable = Result;

#endif
END

COMMAND(LINPUT)
#if NOTIMP_LINPUT
NOTIMPLEMENTED;
#else

  VARIABLE Result;
  LEFTVALUE LetThisVariable;
  unsigned long lCharactersRead;
  char *s;
  long refcount;

  INITIALIZE;

  /* we get the pointer to the variable that points to the value */
  LetThisVariable = EVALUATELEFTVALUE(PARAMETERNODE);
  ASSERTOKE;
  DEREFERENCE(LetThisVariable)


  s = ReadFileLine(pEo,stdin,&lCharactersRead,pEo->fpStdinFunction);
  if( s == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  Result = NEWSTRING(lCharactersRead);
  memcpy(STRINGVALUE(Result),s,lCharactersRead);
  FREE(s);

  /* if this variable had value assigned to it then release that value */
  if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);

  /* and finally assign the code to the variable */
  *LetThisVariable = Result;
#endif
END

/**EOF
=section file

This function accepts one parameter, an opened file number. The return value is true if and
only if the reading has reached the end of the file.
*/
COMMAND(EOFFUN)
#if NOTIMP_EOFFUN
NOTIMPLEMENTED;
#else

  long FileNumber;
  pFileCommandObject pFCO;
  NODE nItem;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  USE_CALLER_MORTALS;

  nItem = PARAMETERLIST;
  FileNumber = LONGVALUE(CONVERT2LONG(_EVALUATEEXPRESSION(CAR(nItem))));
  ASSERTOKE;

  if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_IS_NOT_OPENED);

  if( pFCO->mode[FileNumber] == 's' ){
    RESULT = NEWMORTALLONG;
    if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
    LONGVALUE(RESULT) = (long)pFCO->SocketState[FileNumber];
    }else{
    RESULT = NEWMORTALLONG;
    if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
    if( HOOK_FEOF(THISFILEP) )
      LONGVALUE(RESULT) = -1L;
    else
      LONGVALUE(RESULT) =  0L;
    }
#endif
END

/**INPUTFUN
=display INPUT()
=section file
=title INPUT(n,fn)
Input specified number of characters/bytes from a file.

The first argument of the function is T<n> the number
of character to be read, the second argument T<fn> is the file number. The actual number of bytes
read can be retrieved using the T<LEN> function of the result string.

Note that some Basic languages allow the form
=verbatim
a = INPUT(20,#1)
=noverbatim
however this extra T<#> is not allowed in ScriptBasic.
*/
COMMAND(INPUTFUN)
#if NOTIMP_INPUTFUN
NOTIMPLEMENTED;
#else


  long FileNumber,BytesToRead,CharsRead;
  pFileCommandObject pFCO;
  NODE nItem;
  FILE *fp;
  VARIABLE fpN,vBTR;
  char *s;
  int ch;
  int (*pfExtIn)(void *); /* function to return a single character from the embedder standard input*/

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  USE_CALLER_MORTALS;

  nItem = PARAMETERLIST;
  vBTR = CONVERT2LONG(EVALUATEEXPRESSION(CAR(nItem)));
  ASSERTOKE;
  if( vBTR == NULL ){
    /* we won't read anything because the number of characters is undefined, and therefore the
       result is undef, but we still evaluate the second argument if any */
    nItem = CDR(nItem);
    if( nItem ){
      _EVALUATEEXPRESSION(CAR(nItem));
      ASSERTOKE;
      }
    RESULT = NULL;
    RETURN;
    }
  BytesToRead = LONGVALUE(vBTR);
  nItem = CDR(nItem);
  if( nItem ){
    fpN = _EVALUATEEXPRESSION(CAR(nItem));
    ASSERTOKE;

    if( fpN != NULL ){
      FileNumber = LONGVALUE(CONVERT2LONG(fpN));

      if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
      FileNumber --;
      if( ! pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_IS_NOT_OPENED);
      BytesToRead *= pFCO->RecordSize[FileNumber];
      fp = THISFILEP;
      }else fp = stdin;

    }else{
    if( pEo->fpStdinFunction != NULL ){
      pfExtIn = pEo->fpStdinFunction;
      RESULT = NEWMORTALSTRING(BytesToRead);
      if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
      s = STRINGVALUE(RESULT);
      CharsRead = 0;
      while( BytesToRead && ( ch = pfExtIn(pEo->pEmbedder) ) != EOF ){
        *s++ = ch;
        BytesToRead --;
        CharsRead ++;
        }
      STRLEN(RESULT) = CharsRead;
      RETURN;
      }
    fp = stdin;
    }
  RESULT = NEWMORTALSTRING(BytesToRead);
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  if( pFCO->mode[FileNumber] == 's' ){
    STRLEN(RESULT) = HOOK_TCPRECV(THISSOCKET,STRINGVALUE(RESULT),BytesToRead,0);
    if( STRLEN(RESULT) == 0 )
      pFCO->SocketState[FileNumber] = -1;
    }else{
      s = STRINGVALUE(RESULT);
      CharsRead = 0;
      while( BytesToRead && ( ch = HOOK_FGETC(fp) ) != EOF ){
        *s++ = ch;
        BytesToRead --;
        CharsRead ++;
        }
      STRLEN(RESULT) = CharsRead;
    }

#endif
END

/**CLOSE
=section file
Close a previously successfully opened file. If the file number is
not associated with a successfully opened file then error is raised.
*/
COMMAND(CLOSE)
#if NOTIMP_CLOSE
NOTIMPLEMENTED;
#else

  long FileNumber;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);
  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;

  if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_IS_NOT_OPENED);
  if( pFCO->mode[FileNumber] == 's' )
    HOOK_TCPCLOSE(THISSOCKET);
  else
    HOOK_FCLOSE(THISFILEP);
  pFCO->mode[FileNumber] = (char)0;
  THISFILEP = NULL;
#endif
END

/**RESET
=section file
Close all files.
*/
COMMAND(RESET)
#if NOTIMP_RESET
NOTIMPLEMENTED;
#else

  long FileNumber;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);
  for( FileNumber = 0 ; FileNumber < MAXFILES ; FileNumber++ ){
    if( pFCO->mode[FileNumber] )
      if( pFCO->mode[FileNumber] == 's' )
        HOOK_TCPCLOSE(THISSOCKET);
      else
        HOOK_FCLOSE(THISFILEP);
    pFCO->mode[FileNumber] = (char)0;
    THISFILEP = NULL;
    }
#endif
END

/**SEEK
=section file
=title SEEK fn,position
Go to a specified position in a file.
*/
COMMAND(SEEK)
#if NOTIMP_SEEK
NOTIMPLEMENTED;
#else


  long FileNumber,Position;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);
  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;
  if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_IS_NOT_OPENED);
  if( pFCO->mode[FileNumber] == 's' )ERROR(COMMAND_ERROR_SOCKET_FILE);
  NEXTPARAMETER;
  Position = pFCO->RecordSize[FileNumber] * LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;

  fflush(THISFILEP);
  fseek(THISFILEP,Position,SEEK_SET);

#endif
END

/**TRUNCATE
=section file
=title TRUNCATE fn,new_length
Truncate a file to the specified size.
*/
COMMAND(TRUNCATEF)
#if NOTIMP_TRUNCATEF
NOTIMPLEMENTED;
#else


  long FileNumber,Position;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);
  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;
  if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_IS_NOT_OPENED);
  if( pFCO->mode[FileNumber] == 's' )ERROR(COMMAND_ERROR_SOCKET_FILE);
  NEXTPARAMETER;
  Position = pFCO->RecordSize[FileNumber] * LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;

  fflush(THISFILEP);
  HOOK_TRUNCATE(THISFILEP,Position);

#endif
END
/**REWIND
=section file
=title REWIND [ # ]fn
Positions the file cursor to the start of the file.
This is the same as T<SEEK fn,0> or T<SEEK#fn,0>
*/
COMMAND(REWIND)
#if NOTIMP_REWIND
NOTIMPLEMENTED;
#else


  long FileNumber;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);
  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;
  if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_IS_NOT_OPENED);
  if( pFCO->mode[FileNumber] == 's' )ERROR(COMMAND_ERROR_SOCKET_FILE);

  fflush(THISFILEP);
  fseek(THISFILEP,0L,SEEK_SET);

#endif
END

/**LOC
=display LOC()
=title LOC()
=section file
Return current file pointer position of the opened file.
*/
COMMAND(LOC)
#if NOTIMP_LOC
NOTIMPLEMENTED;
#else


  VARIABLE Op1;
  long FileNumber;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  Op1 = EVALUATEEXPRESSION(CAR(PARAMETERLIST));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  FileNumber = LONGVALUE(CONVERT2LONG(Op1));
  RESULT = NULL;
  if( FileNumber < 1 || FileNumber > MAXFILES )RETURN;
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )RETURN;
  if( pFCO->mode[FileNumber] == 's' )RESULT;

  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = (long)(ftell(THISFILEP) / pFCO->RecordSize[FileNumber]);

#endif
END

/**LOF
=section file
=display LOF()
=title LOF()
Length Of File. Return length of the opened file referenced with the file number.
*/
COMMAND(LOF)
#if NOTIMP_LOF
NOTIMPLEMENTED;
#else


  VARIABLE Op1;
  long FileNumber,SavePosition;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  Op1 = EVALUATEEXPRESSION(CAR(PARAMETERLIST));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  FileNumber = LONGVALUE(CONVERT2LONG(Op1));
  RESULT = NULL;
  if( FileNumber < 1 || FileNumber > MAXFILES )RETURN;
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )RETURN;
  if( pFCO->mode[FileNumber] == 's' )RETURN;

  RESULT = NEWMORTALLONG;
  SavePosition = ftell(THISFILEP);
  fseek(THISFILEP,0,SEEK_END);
  LONGVALUE(RESULT) = (long)(ftell(THISFILEP) / pFCO->RecordSize[FileNumber]);
  fseek(THISFILEP,SavePosition,SEEK_SET);

#endif
END

/**FREEFILE
=section file
=title FREEFILE()
This function returns a free file number which is currently not associated with any opened
file. If there is no such file number it returns T<undef>.
*/
COMMAND(FREEFILE)
#if NOTIMP_FREEFILE
NOTIMPLEMENTED;
#else


  VARIABLE Op1;
  long FileNumber,Range;
  pFileCommandObject pFCO;
  NODE nItem;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  if( nItem = PARAMETERLIST ){
    Op1 = EVALUATEEXPRESSION(CAR(PARAMETERLIST));
    ASSERTOKE;
    if( Op1 == NULL )
      Range = -1;
    else
      Range = LONGVALUE(CONVERT2LONG(Op1));
    }else Range = -1;

  if( Range == -1 ){
    for( FileNumber = 1 ; FileNumber < MAXFILES ; FileNumber++ )
      if( ! pFCO->mode[FileNumber] ){
        Range = -2;
        break;
        }
    }else
  if( Range == 0 ){
    for( FileNumber = 1 ; FileNumber < 255 ; FileNumber++ )
      if( ! pFCO->mode[FileNumber]  ){
        Range = -2;
        break;
        }
    }else{
    for( FileNumber = 255 ; FileNumber < MAXFILES ; FileNumber++ )
      if( ! pFCO->mode[FileNumber]  ){
        Range = -2;
        break;
        }
    }
  if( Range != -2 ){
    RESULT = NULL;
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = FileNumber+1;

#endif
END

/**PRINT
=section file
=title PRINT [ # fn , ] print_list
The command prints the T<print_list> to an opened file given by the file number T<fn>. If T<fn>
is not specified the command prints to the standard output.

If there is no T<print_list> specified the command prints a new line.

*/
COMMAND(FPRINT)
#if NOTIMP_FPRINT
NOTIMPLEMENTED;
#else

  char buffer[80]; /* should be enough to print a long or a double */
  NODE nItem;
  VARIABLE ItemResult;
  long FileNumber;
  pFileCommandObject pFCO;
  char *s;
  unsigned long slen;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  nItem = PARAMETERNODE;
  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(nItem)));
  ASSERTOKE;
  if( FileNumber < 1 || FileNumber > MAXFILES )RETURN;
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )RETURN;


  NEXTPARAMETER;
  nItem = PARAMETERNODE;

  while( nItem ){
    ItemResult = EVALUATEEXPRESSION(CAR(nItem));
    ASSERTOKE;

    if( ItemResult == NULL ){
      s = "undef";
      slen = 5;
      }
    else
    switch( TYPE(ItemResult) ){
      case VTYPE_LONG:
        sprintf(buffer,"%ld",LONGVALUE(ItemResult));
        s = buffer;
        slen = strlen(buffer);
        break;
      case VTYPE_DOUBLE:
        sprintf(buffer,"%lf",DOUBLEVALUE(ItemResult));
        s = buffer;
        slen = strlen(buffer);
        break;
      case VTYPE_STRING:
        s = STRINGVALUE(ItemResult);
        slen = STRLEN(ItemResult);
        break;
      case VTYPE_ARRAY:
        sprintf(buffer,"ARRAY@#%08X",LONGVALUE(ItemResult));
        s = buffer;
        slen = strlen(buffer);
        break;
      }

    if( pFCO->mode[FileNumber] == 's' )
      HOOK_TCPSEND(THISSOCKET,s,slen,0);
    else
      while( slen -- )HOOK_PUTC(((int)*s++),THISFILEP);

    nItem = CDR(nItem);
    }

  if( pFCO->mode[FileNumber] != 's' &&
      fflush(THISFILEP) == EOF )ERROR(COMMAND_ERROR_PRINT_FAIL);

#endif
END

COMMAND(FPRINTNL)
#if NOTIMP_FPRINTNL
NOTIMPLEMENTED;
#else


  NODE nItem;
  long FileNumber;
  pFileCommandObject pFCO;


  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  nItem = PARAMETERNODE;
  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(nItem)));
  ASSERTOKE;
  if( FileNumber < 1 || FileNumber > MAXFILES )RETURN;
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )RETURN;


  if( pFCO->mode[FileNumber] == 's' )
    HOOK_TCPSEND(THISSOCKET,"\n",1,0);
  else{
    HOOK_PUTC('\n',THISFILEP);
    if( fflush(THISFILEP) == EOF )ERROR(COMMAND_ERROR_PRINT_FAIL);
    }

#endif
END
#define NOCOMMAND(XXX) \
COMMAND(XXX)\
NOTIMPLEMENTED;\
END

#define FILE_FUN(XXX,yyy) \
COMMAND(XXX)\
\
  char *FileName;\
  VARIABLE Op;\
  long lRes;\
\
  USE_CALLER_MORTALS;\
\
  Op = _EVALUATEEXPRESSION(CAR(PARAMETERLIST));\
  ASSERTOKE;\
  if( Op == NULL ){\
    RESULT = NULL;\
    RETURN;\
    }\
\
  Op = CONVERT2STRING(Op);\
\
  FileName = ALLOC(STRLEN(Op)+1);\
  if( FileName == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);\
  memcpy(FileName,STRINGVALUE(Op),STRLEN(Op));\
  FileName[STRLEN(Op)] = (char)0;\
  lRes = yyy(FileName);\
  if( lRes == -1 ){\
    RESULT = NULL;\
    RETURN;\
    }\
\
  RESULT = NEWMORTALLONG;\
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);\
  LONGVALUE(RESULT) = lRes;\
  FREE(FileName);\
\
END

#define FILE_BFUN(XXX,yyy) \
COMMAND(XXX)\
\
  char *FileName;\
  VARIABLE Op;\
  long lRes;\
\
  USE_CALLER_MORTALS;\
\
  Op = _EVALUATEEXPRESSION(CAR(PARAMETERLIST));\
  ASSERTOKE;\
  if( Op == NULL ){\
    RESULT = NULL;\
    RETURN;\
    }\
\
  Op = CONVERT2STRING(Op);\
\
  FileName = ALLOC(STRLEN(Op)+1);\
  if( FileName == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);\
  memcpy(FileName,STRINGVALUE(Op),STRLEN(Op));\
  FileName[STRLEN(Op)] = (char)0;\
  lRes = yyy(FileName);\
  RESULT = NEWMORTALLONG;\
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);\
  LONGVALUE(RESULT) = lRes;\
  FREE(FileName);\
\
END

/**FILELEN
=section file
=display FILELEN()
=title FILELEN(file_name)
Get the length of a named file. If the length of the file can not be determined 
(for example the file does not exists, or the
process running the code does not have permission to read the file) then the return value is T<undef>.
*/

#if NOTIMP_FILELEN
NOCOMMAND(FILELEN)
#else
FILE_FUN(FILELEN,HOOK_SIZE)
#endif

/**FILEACCESSTIME
=section file
=display FILEACCESSTIME()
=title FILEACCESSTIME(file_name)
Get the time the file was accessed last time.
*/
#if NOTIMP_FTACCESS
NOCOMMAND(FTACCESS)
#else
FILE_FUN(FTACCESS,HOOK_TIME_ACCESSED)
#endif

/**FILEMODIFYTIME
=section file
=display FILEMODIFYTIME()
=title FILEMODIFYTIME(file_name)
Get the time the file was modified last time.
*/
#if NOTIMP_FTMODIFY
NOCOMMAND(FTMODIFY)
#else
FILE_FUN(FTMODIFY,HOOK_TIME_MODIFIED)
#endif

/**FILECREATETIME
=section file
=display FILECREATETIME()
=title FILECREATETIME(file_name)
Get the time the file was modified last time.
*/
#if NOTIMP_FTCREATED
NOCOMMAND(FTCREATED)
#else
FILE_FUN(FTCREATED,HOOK_TIME_CREATED)
#endif

/**ISDIR
=section file
=display ISDIR()
=title ISDIR(file_name)
Returns true if the named file is a directory.
*/
#if NOTIMP_ISDIR
NOCOMMAND(ISDIR)
#else
FILE_BFUN(ISDIR,HOOK_ISDIR)
#endif

/**ISREG
=section file
=display ISREG()
=title ISREG(file_name)
Returns if the named file is a regular file.
*/
#if NOTIMP_ISREG
NOCOMMAND(ISREG)
#else
FILE_BFUN(ISREG,HOOK_ISREG)
#endif

/**FILEEXISTS
=section file
=display FILEEXISTS()
=title FILEEXISTS(file_name)
Returns true if the named file exists.
*/
#if NOTIMP_FILEXISTS
NOCOMMAND(FILEXISTS)
#else
FILE_BFUN(FILEXISTS,HOOK_EXISTS)
#endif

/**FLOCK
=section file
=display LOCK
=title LOCK # fn, mode
Lock a file or release a lock on a file. The T<mode> parameter can be T<read>, T<write> or T<release>.
There can be more than one simultaneous read lock on a file but there can be only one write lock.

*/
COMMAND(FLOCK)
#if NOTIMP_FLOCK
NOTIMPLEMENTED;
#else


  long FileNumber;
  char *ModeString;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;\
  if( FileNumber < 1 || FileNumber > MAXFILES )RETURN;
  FileNumber --;
  if( ! pFCO->mode[FileNumber] || pFCO->mode[FileNumber] == 's' )RETURN;

  NEXTPARAMETER;
  ModeString = pEo->StringTable+pEo->CommandArray[_ActualNode-1].Parameter.CommandArgument.Argument.szStringValue;

  if( !stricmp(ModeString,"read") ){
    HOOK_FLOCK(THISFILEP,LOCK_SH);
    }else
  if( !stricmp(ModeString,"write") ){
    HOOK_FLOCK(THISFILEP,LOCK_EX);
    }else
  if( !stricmp(ModeString,"release") ){
    HOOK_FLOCK(THISFILEP,LOCK_UN);
    }else ERROR(COMMAND_ERROR_INVALID_LOCK);

#endif
END

/**LOCKR
=display LOCK REGION
=title LOCK REGION # fn FROM start TO end FOR mode
Lock a region of a file. The region starts with the position T<start> and ends with the position
T<end> including both end positions.

The mode can be T<read>, T<write> and T<release>.
*/
COMMAND(RLOCK)
#if NOTIMP_RLOCK
NOTIMPLEMENTED;
#else


  long FileNumber,lFrom,lTo,lSwap;
  char *ModeString;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);

  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;
  if( FileNumber < 1 || FileNumber > MAXFILES )RETURN;
  FileNumber --;
  if( ! pFCO->mode[FileNumber] || pFCO->mode[FileNumber] == 's' )RETURN;

  NEXTPARAMETER;
  lFrom = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;
  if( lFrom < 0 )RETURN;
  NEXTPARAMETER;
  lTo   = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;
  if( lTo < 0 )RETURN;
  if( lFrom > lTo ){
    lSwap = lTo;
    lTo = lFrom;
    lFrom = lSwap;
    }

  NEXTPARAMETER;
  ModeString = pEo->StringTable+pEo->CommandArray[_ActualNode-1].Parameter.CommandArgument.Argument.szStringValue;

  if( !stricmp(ModeString,"read") ){
    HOOK_LOCK(THISFILEP,LOCK_SH,lFrom,(lTo-lFrom+1)*pFCO->RecordSize[FileNumber]);
    }else
  if( !stricmp(ModeString,"write") ){
    HOOK_LOCK(THISFILEP,LOCK_EX,lFrom,(lTo-lFrom+1)*pFCO->RecordSize[FileNumber]);
    }else
  if( !stricmp(ModeString,"release") ){
    HOOK_LOCK(THISFILEP,LOCK_UN,lFrom,(lTo-lFrom+1)*pFCO->RecordSize[FileNumber]);
    }else ERROR(COMMAND_ERROR_INVALID_LOCK);

#endif
END
/**MKDIR
=section file
=title MKDIR firectory_name

Create a directory. Note that all directories on the path are created 
recursively if neccessary.
*/
COMMAND(MKDIR)
#if NOTIMP_MKDIR
NOTIMPLEMENTED;
#else

  VARIABLE Op;
  char *s;

  Op = CONVERT2STRING(EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  s = ALLOC(STRLEN(Op)+1);
  if( s == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  memcpy(s,STRINGVALUE(Op),STRLEN(Op));
  s[STRLEN(Op)] = (char)0;
  if( HOOK_MAKEDIRECTORY(s) == -1 ){
    FREE(s);
    ERROR(COMMAND_ERROR_MKDIR_FAIL);
    }
  FREE(s);
#endif
END

/**DELETE
=title DELETE file/directory_name
=section file

Delete a file or an B<empty> directory. See R<DELTREE> for a more powerful and dangerous delete.
*/
COMMAND(DELETE)
#if NOTIMP_DELETE
NOTIMPLEMENTED;
#else

  VARIABLE Op;
  char *s;
  int iResult;

  Op = CONVERT2STRING(EVALUATEEXPRESSION(PARAMETERNODE));
  s = ALLOC(STRLEN(Op)+1);
  ASSERTOKE;
  if( s == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  memcpy(s,STRINGVALUE(Op),STRLEN(Op));
  s[STRLEN(Op)] = (char)0;
  if( ! HOOK_EXISTS(s) ){
    FREE(s);
    RETURN;
    }
  if( HOOK_ISDIR(s) ){
    iResult = HOOK_RMDIR(s);
    FREE(s);
    if( iResult == -1 )ERROR(COMMAND_ERROR_DELETE_FAIL);
    RETURN;
    }
  iResult = HOOK_REMOVE(s);
  FREE(s);
  if( iResult == -1 )ERROR(COMMAND_ERROR_DELETE_FAIL);
  RETURN;

#endif
END

/**DELTREE
=title DELTREE file/directory_name
=section file

Delete a file or a directory. If a directory is to be deleted all files and subdirectories
are also deleted.
*/
COMMAND(DELETEF)
#if NOTIMP_DELETEF
NOTIMPLEMENTED;
#else

  VARIABLE Op;
  char *s;
  int iResult;

  Op = CONVERT2STRING(EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  s = ALLOC(STRLEN(Op)+1);
  if( s == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  memcpy(s,STRINGVALUE(Op),STRLEN(Op));
  s[STRLEN(Op)] = (char)0;
  if( ! HOOK_EXISTS(s) ){
    FREE(s);
    RETURN;
    }
  if( ! HOOK_ISDIR(s) ){
    iResult = HOOK_REMOVE(s);
    FREE(s);
    if( iResult == -1 )ERROR(COMMAND_ERROR_DELETE_FAIL);
    RETURN;
    }

  /* Delete the files recursively inside the directory. */
  iResult = HOOK_DELTREE(s);
  FREE(s);
  if( iResult == -1 )ERROR(COMMAND_ERROR_DELETE_FAIL);
  RETURN;

#endif
END

#define MAXDIRS 512

/* A DirList contains the file names.

   The field cbFileName points to an array.
   Each element of the array contains the length of a file name.

   SortValue is an array that stores file size, creation date or
   other parameter according to the sort condition.

   ppszFileName points to an array of char * each pointing to a
   file name. This array is altered during sort.

   cFileNames is the number of file names in the list.

   FileIndex is the index to the cbFileName array for the
   current file name during file iterations.

*/
typedef struct _DirList {
  unsigned long *cbFileName; /* file name length */
  unsigned long *SortValue;  /* sort value or string offset */
  char **ppszFileName;       /* file name */
  unsigned long cFileNames;  /* number of file names */
  unsigned long FileIndex;   /* current file index */
  } DirList, *pDirList;

#define FILES_INCREMENT 10
static int store_file_name(pExecuteObject pEo,
                           pDirList p,
                           char *buffer,
                           unsigned long ThisSortValue

  ){
  unsigned long ulNewSize;
  unsigned long *plNewCbFileName,*NewSortValue;
  char **ppszNewppszFileName;
  unsigned long i;

  /* first of all check that there is enough space to store the file name on the current index location. */
  if( p->FileIndex >= (ulNewSize = p->cFileNames) ){
    while( p->FileIndex >= ulNewSize )/* this while is redundant unless there is a programming error somwhere else */
      ulNewSize += FILES_INCREMENT;
    plNewCbFileName = ALLOC( ulNewSize * sizeof(long) );
    if( plNewCbFileName == NULL )return 1;
    NewSortValue = ALLOC( ulNewSize * sizeof(long));
    if( NewSortValue == NULL )return 1;
    ppszNewppszFileName = ALLOC( ulNewSize * sizeof(char *));
    if( ppszNewppszFileName == NULL )return 1;
    for( i=0 ; i < p->cFileNames ; i++ ){
       plNewCbFileName[i] = p->cbFileName[i];
       NewSortValue[i] = p->SortValue[i];
       ppszNewppszFileName[i] = p->ppszFileName[i];
       }
    if( p->cbFileName )FREE(p->cbFileName);
    if( p->SortValue )FREE(p->SortValue);
    if( p->ppszFileName )FREE(p->ppszFileName);
    p->cbFileName = plNewCbFileName;
    p->SortValue = NewSortValue;
    p->ppszFileName = ppszNewppszFileName;
    p->cFileNames = ulNewSize;
    }

  if( (p->ppszFileName[p->FileIndex] = ALLOC( (p->cbFileName[p->FileIndex] = strlen(buffer)) )) == NULL )
    return 1;
  memcpy(p->ppszFileName[p->FileIndex],buffer,p->cbFileName[p->FileIndex]);
  p->SortValue[p->FileIndex] = ThisSortValue;
  p->FileIndex++;
  return 0;
  }

typedef struct _DirCommandObject {
  pDirList dp[MAXDIRS];
  } DirCommandObject, *pDirCommandObject;

static void close_directory_list(pExecuteObject pEo,
                                 unsigned long i
  ){
  pDirCommandObject pDCO;
  unsigned long j;

  pDCO = (pDirCommandObject)PARAMPTR(CMD_OPENDIR);
  if( pDCO == NULL )return;

  for( j=0 ; j<pDCO->dp[i]->cFileNames ; j++ )
     FREE(pDCO->dp[i]->ppszFileName[j]);
  /* the following two pointers may be null if the directory list contained no file name*/
  if( pDCO->dp[i]->cbFileName )FREE( pDCO->dp[i]->cbFileName);
  if( pDCO->dp[i]->ppszFileName )FREE( pDCO->dp[i]->ppszFileName);
  /* this should never be null when we get here */
  if( pDCO->dp[i] )FREE( pDCO->dp[i]);
  pDCO->dp[i] = NULL;
  return;
  }

static void close_all_dirs(pExecuteObject pEo){
  pDirCommandObject pDCO;
  long i;

  pDCO = (pDirCommandObject)PARAMPTR(CMD_OPENDIR);
  if( pDCO == NULL )return;
  for( i = 0 ; i < MAXDIRS ; i++ )
    if( pDCO->dp[i] )
      close_directory_list(pEo,i);
  }

static int initdir(pExecuteObject pEo){
#define INITDIR initdir(pEo)
  pDirCommandObject pDCO;
  int i;

  /* initialize only once */
  if( PARAMPTR(CMD_OPENDIR) )return 0;

  PARAMPTR(CMD_OPENDIR) = ALLOC(sizeof(DirCommandObject));
  if( PARAMPTR(CMD_OPENDIR) == NULL )return COMMAND_ERROR_MEMORY_LOW;

  pDCO = (pDirCommandObject)PARAMPTR(CMD_OPENDIR);
  for( i=0 ; i < MAXDIRS ; i++ )
    pDCO->dp[i] = NULL;

  FINALPTR(CMD_OPENDIR) = close_all_dirs;
  return 0;
  }


#define COLLECT_DIRS   0x0001 /* collect directories as well */
#define COLLECT_DOTS   0x0002 /* collect the dot directories */
#define COLLECT_RECU   0x0004 /* collect recursively */
#define SORTBY_SIZE    0x0008 /* sort by file size */
#define SORTBY_CRETI   0x0010 /* sort by creation time */
#define SORTBY_ACCTI   0x0020 /* sort by access time */
#define SORTBY_MODTI   0x0040 /* sort by modification time */
#define SORTBY_NAME    0x0080 /* sort by name (no directory part is compared) */
#define SORTBY_FNAME   0x0100 /* sort by full name */
#define COLLECT_FULLP  0x0200 /* collect and return "full" path names including the original directory name in
                                 each file name retuned (can directly be used in file handling statements */
#define SORTBY_DESCEN  0x0400 /* sort descending order */

#define MAX_FNLEN 1024

/* This function recursively collects
*/
static int collect_dirs_r(pExecuteObject pEo,
                          char * buffer,
                          unsigned long fAction,
                          pDirList pThisDirList,
                          char *pattern,
                          unsigned long StartCharIndex
  ){
  tDIR DL;
  DIR *pDL;
  struct dirent *pD;
  int dirlen;

  unsigned long sL,pL;
  unsigned long cArraySize;
  pPatternParam pLastResult;
  int iError;
  unsigned long ulSortValue;

  pLastResult = (pPatternParam)PARAMPTR(CMD_LIKEOP);

  dirlen=strlen(buffer);
  if( buffer[dirlen-1] != '/' ){
    dirlen++;
    if( dirlen >= MAX_FNLEN )return -1;
    strcpy(buffer+dirlen-1,"/");
    }
  pDL = HOOK_OPENDIR(buffer,&DL);
  if( pDL == NULL )return -1;
  while( pD = HOOK_READDIR(pDL) ){
    /* skip . and .. directories */
    if( pD->d_name[0] == '.' && 
       ( pD->d_name[1] == (char)0 ||
         ( pD->d_name[1] == '.' && pD->d_name[2] == (char)0 ) ) ){
      if( fAction&COLLECT_DOTS ){
        if( dirlen+strlen(pD->d_name) >= MAX_FNLEN )return -1;
        strcpy(buffer+dirlen,pD->d_name);
        if( fAction & SORTBY_SIZE )ulSortValue = HOOK_SIZE(buffer); else
        if( fAction & SORTBY_CRETI )ulSortValue = HOOK_TIME_CREATED(buffer); else
        if( fAction & SORTBY_MODTI )ulSortValue = HOOK_TIME_MODIFIED(buffer); else
        if( fAction & SORTBY_ACCTI )ulSortValue = HOOK_TIME_ACCESSED(buffer); else
        if( fAction & SORTBY_NAME  )ulSortValue = dirlen - StartCharIndex; else
        ulSortValue = 0; /* SORTBY_FNAME */
        if( store_file_name(pEo,pThisDirList,buffer+StartCharIndex,ulSortValue) )return -1;
        }
      continue;
      }
    if( dirlen+(sL=strlen(pD->d_name)) >= MAX_FNLEN )return -1;
    strcpy(buffer+dirlen,pD->d_name);
    if( *pattern ){
      pL = strlen(pattern);
      cArraySize = match_count(pattern,pL);
      if( cArraySize > pLastResult->cArraySize ){
        if( pLastResult->pcbParameterArray )FREE(pLastResult->pcbParameterArray);
        if( pLastResult->ParameterArray)FREE(pLastResult->ParameterArray);
        pLastResult->cArraySize = 0;
        pLastResult->pcbParameterArray = ALLOC(cArraySize*sizeof(unsigned long));
        if( pLastResult->pcbParameterArray == NULL )return -1;
        pLastResult->ParameterArray    = ALLOC(cArraySize*sizeof(char *));
        if( pLastResult->ParameterArray == NULL ){
          FREE(pLastResult->pcbParameterArray);
          pLastResult->pcbParameterArray = NULL;
          return -1;
          }
        pLastResult->cArraySize = cArraySize;
        }
      if( pLastResult->cbBufferSize < sL ){
        pLastResult->cbBufferSize = 0;
        if( pLastResult->pszBuffer )FREE(pLastResult->pszBuffer);
        pLastResult->pszBuffer = ALLOC(sL*sizeof(char));
        if( pLastResult->pszBuffer == NULL )return -1;
        pLastResult->cbBufferSize = sL;
        }

      iError = match_match(pattern,
                           pL,
                           buffer+dirlen,
                           sL,
                           pLastResult->ParameterArray,
                           pLastResult->pcbParameterArray,
                           pLastResult->pszBuffer,
                           pLastResult->cArraySize,
                           pLastResult->cbBufferSize,
#ifdef WIN32
                           0,
#else
                           !(OPTION("compare")&1),
#endif
                           pLastResult->pThisMatchSets,
                           &(pLastResult->iMatches));
      }
    if( (!*pattern) || pLastResult->iMatches ){
      if( fAction & SORTBY_SIZE )ulSortValue = HOOK_SIZE(buffer); else
      if( fAction & SORTBY_CRETI )ulSortValue = HOOK_TIME_CREATED(buffer); else
      if( fAction & SORTBY_MODTI )ulSortValue = HOOK_TIME_MODIFIED(buffer); else
      if( fAction & SORTBY_ACCTI )ulSortValue = HOOK_TIME_ACCESSED(buffer); else
      if( fAction & SORTBY_NAME  )ulSortValue = dirlen - StartCharIndex; else
      ulSortValue = 0;
      store_file_name(pEo,pThisDirList,buffer+StartCharIndex,ulSortValue);
      }
    pLastResult->iMatches = 0; /* no joker() after file pattern match */
    if( HOOK_ISDIR(buffer) && (fAction & COLLECT_RECU) )
      collect_dirs_r(pEo,buffer,fAction,pThisDirList,pattern,StartCharIndex);
    }
  HOOK_CLOSEDIR(pDL);
  dirlen--;
  buffer[dirlen] = (char)0;
  return 0;
  }

static int collect_dirs(pExecuteObject pEo,
                        unsigned long fAction,
                        pDirList pThisDirList,
                        char *Directory,
                        unsigned long cDirectory,
                        char *pattern,
                        unsigned long c_pattern
  ){
  char buffer[MAX_FNLEN];
  char puffer[MAX_FNLEN];
  unsigned long StartCharIndex;

  if( initialize_like(pEo) )return -1;
  memcpy(buffer,Directory,cDirectory);
  buffer[cDirectory] = (char)0;
  if( buffer[cDirectory-1] != '/' ){
    cDirectory++;
    if( cDirectory >= MAX_FNLEN )return -1;
    strcpy(buffer+cDirectory-1,"/");
    }
  if( pattern )
    memcpy(puffer,pattern,c_pattern);
  puffer[c_pattern] = (char)0;

  StartCharIndex = strlen(buffer);
  if( fAction & COLLECT_FULLP )StartCharIndex = 0;
  if( collect_dirs_r(pEo,buffer,fAction,pThisDirList,puffer,StartCharIndex) == -1 )return -1;
  pThisDirList->cFileNames = pThisDirList->FileIndex;
  pThisDirList->FileIndex = 0;
  return 0;
  }

static int sort_dirs(pExecuteObject pEo,
                        unsigned long fAction,
                        pDirList p
  ){
  unsigned long i,j;
  unsigned long lSwap;
  char *pszSwap;
  unsigned long CompareLength,Leni,Lenj;
  int CompareSult;

#define SWAP do{\
                lSwap = p->cbFileName[i];\
                p->cbFileName[i] = p->cbFileName[j];\
                p->cbFileName[j] = lSwap;\
                lSwap = p->SortValue[i];\
                p->SortValue[i] = p->SortValue[j];\
                p->SortValue[j] = lSwap;\
                pszSwap = p->ppszFileName[i];\
                p->ppszFileName[i] = p->ppszFileName[j];\
                p->ppszFileName[j] = pszSwap;\
                }while(0)

  /* if there is nothing to sort by */
  if( !(fAction & ( SORTBY_SIZE | SORTBY_CRETI | SORTBY_ACCTI | SORTBY_MODTI | SORTBY_NAME | SORTBY_FNAME)) )
    return 0;

  if( fAction & (SORTBY_NAME | SORTBY_FNAME) ){
    /* string type of comparision */
    for( i=1 ; i < p->cFileNames ; i++ )
     for( j=0 ; j < i ; j++ ){
       CompareLength = Leni = p->cbFileName[i] - p->SortValue[i];
       if( CompareLength > (Lenj=p->cbFileName[j] - p->SortValue[j]) )
         CompareLength = Lenj;
       CompareSult = memcmp(p->ppszFileName[i]+p->SortValue[i],p->ppszFileName[j]+p->SortValue[j],CompareLength);
       CompareSult = CompareSult > 0 || (CompareSult == 0 && Leni > Lenj);
       if( fAction & SORTBY_DESCEN )CompareSult = !CompareSult;
       if( CompareSult )
         SWAP;
       }
    }else{
    /* numeric comparision based on collected value */
    for( i=1 ; i < p->cFileNames ; i++ )
     for( j=0 ; j < i ; j++ )
       if( (fAction & SORTBY_DESCEN) ? p->SortValue[i] < p->SortValue[j] : p->SortValue[i] > p->SortValue[j] )
         SWAP;
    }
  return 0;
  }

/**OPENDIR
=display OPEN DIRECTORY
=title OPEN DIRECTORY dir_name PATTERN pattern OPTION option AS dn

Open a directory to retrieve the list of files.

=itemize
=item T<dir_name> is the name of the directory.
=item T<parttern> is a wilde card pattern to filter the file list.
=item T<option> is an integer value that can be composed AND-ing some of the following values
 =itemize
 =item T<SbCollectDirectories>	Collect the directory names as well as file names into the file list.
 =item T<SbCollectDots>	Collect the virtual . and .. directory names into the list.
 =item T<SbCollectRecursively>	Collect the files from the directory and from all the directories below.
 =item T<SbCollectFullPath>	The list will contain the full path to the file names. This means that the file names returned by the function NextFile will contain the directory path specified in the open directory statement and therefore can be used as argument to file handling commands and functions.
 =item T<SbCollectFiles>	Collect the files. This is the default behavior.
 =item T<SbSortBySize>	The files will be sorted by file size.
 =item T<SbSortByCreateTime>	The files will be sorted by creation time.
 =item T<SbSortByAccessTime>	The files will be sorted by access time.
 =item T<SbSortByModifyTime>	The files will be sorted by modify time.
 =item T<SbSortByName>	The files will be sorted by name. The name used for sorting is the bare file name without any path. 
 =item T<SbSortByPath>	The files will be sorted by name including the path. The path is the relative to the directory, which is currently opened. This sorting option is different from the value sbSortByName only when the value sbCollectRecursively is also used.
 =item T<SbSortAscending>	Sort the file names in ascending order. This is the default behavior.
 =item T<SbSortDescending>	Sort the file names in descending order.
 =item T<SbSortByNone>	Do not sort. Specify this value if you do not need sorting. In this case directory opening can be much faster especially for large directories.
 =noitemize
=item T<dn> is the directory number used in later references to the opened directory.
=noitemize

See also R<CLOSEDIR> and R<NEXTFILE>.

*/
COMMAND(OPENDIR)
#if NOTIMP_OPENDIR
NOTIMPLEMENTED;
#else

  long DirNumber;
  char *DirName;
  unsigned long i;
  pDirCommandObject pDCO;
  VARIABLE vDirName,vPattern,vDirNumber,vOption;
  unsigned long fAction;

  INITDIR;
  pDCO = (pDirCommandObject)PARAMPTR(CMD_OPENDIR);

  /* get the directory name */
  vDirName = CONVERT2STRING(_EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  NEXTPARAMETER;
  vPattern = CONVERT2STRING(_EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  NEXTPARAMETER;
  vOption = CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  NEXTPARAMETER;
  vDirNumber = _EVALUATEEXPRESSION(PARAMETERNODE);
  ASSERTOKE;
  if( vDirNumber == NULL )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  if( TYPE(vDirNumber) == VTYPE_LONG && LONGVALUE(vDirNumber) == 0 ){/* we have to automatically allocate the file number */
    for( i = 1 ; i < MAXFILES ; i++ )
      if( pDCO->dp[i] == NULL )break;
    if( i >= MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
    LONGVALUE(vDirNumber) = i;
    }
  DirNumber = LONGVALUE(CONVERT2LONG(vDirNumber));
  if( DirNumber <1 || DirNumber >= MAXDIRS )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  if( pDCO->dp[DirNumber] )ERROR(COMMAND_ERROR_FILE_NUMBER_IS_USED);

  /* copy the dir name to DirName zchar terminated. */
  if( vDirName == NULL )ERROR(COMMAND_ERROR_INV_DNAME);
  SECUREFILE(vDirName)
  CONVERT2ZCHAR(vDirName,DirName);

  pDCO->dp[DirNumber] = ALLOC( sizeof(DirList) );
  if( pDCO->dp[DirNumber] == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  pDCO->dp[DirNumber]->cFileNames = 0;
  pDCO->dp[DirNumber]->FileIndex  = 0;
  pDCO->dp[DirNumber]->cbFileName = NULL;
  pDCO->dp[DirNumber]->SortValue = NULL;
  pDCO->dp[DirNumber]->ppszFileName = NULL;

  if( vOption == NULL )ERROR(COMMAND_ERROR_INV_DO_OPTION);
  fAction = ~ (LONGVALUE(vOption));

#define SORTBY_SIZE    0x0008 /* sort by file size */
#define SORTBY_CRETI   0x0010 /* sort by creation time */
#define SORTBY_ACCTI   0x0020 /* sort by access time */
#define SORTBY_MODTI   0x0040 /* sort by modification time */
#define SORTBY_NAME    0x0080 /* sort by name (no directory part is compared) */
#define SORTBY_FNAME   0x0100 /* sort by full name */


  /* If the user specifies more than one sorting criuteria, we have to correct it,
     because the underlying layers assume that only one bit is set. Former version
     crashed when open directory option was sloppyly set to 0. */
  if( fAction & SORTBY_SIZE  )
    fAction &= ~(    0      |SORTBY_CRETI|SORTBY_ACCTI|SORTBY_MODTI|SORTBY_NAME|SORTBY_FNAME); else
  if( fAction & SORTBY_CRETI )
    fAction &= ~(SORTBY_SIZE|       0    |SORTBY_ACCTI|SORTBY_MODTI|SORTBY_NAME|SORTBY_FNAME); else
  if( fAction & SORTBY_ACCTI )
    fAction &= ~(SORTBY_SIZE|SORTBY_CRETI|      0     |SORTBY_MODTI|SORTBY_NAME|SORTBY_FNAME); else
  if( fAction & SORTBY_MODTI )
    fAction &= ~(SORTBY_SIZE|SORTBY_CRETI|SORTBY_ACCTI|      0     |SORTBY_NAME|SORTBY_FNAME); else
  if( fAction & SORTBY_NAME  )
    fAction &= ~(SORTBY_SIZE|SORTBY_CRETI|SORTBY_ACCTI|SORTBY_MODTI|      0    |SORTBY_FNAME); else
  if( fAction & SORTBY_FNAME )
    fAction &= ~(SORTBY_SIZE|SORTBY_CRETI|SORTBY_ACCTI|SORTBY_MODTI|SORTBY_NAME|      0     );

  if( collect_dirs(pEo,
                   fAction,
                   pDCO->dp[DirNumber],
                   STRINGVALUE(vDirName),
                   STRLEN(vDirName),
                   vPattern ? STRINGVALUE(vPattern) : NULL ,
                   vPattern ? STRLEN(vPattern) : 0 ) == -1 ){
    close_directory_list(pEo,DirNumber);
    ERROR(COMMAND_ERROR_DIR_NO_OPEN); 
    }
  sort_dirs(pEo,
            fAction,
            pDCO->dp[DirNumber]);
#endif
END

/**NEXTFILE
=section file
=title NEXTFILE(dn)
=display NEXTFILE()

Retrieve the next file name from an opened directory list. If there is no more file names it
returns T<undef>.

See also R<OPENDIR> and R<CLOSEDIR>.

*/
COMMAND(NEXTFILE)
#if NOTIMP_NEXTFILE
NOTIMPLEMENTED;
#else

  VARIABLE Op1;
  pDirCommandObject pDCO;
  unsigned long DirNumber;

  INITDIR;
  pDCO = (pDirCommandObject)PARAMPTR(CMD_OPENDIR);

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  Op1 = CONVERT2LONG(EVALUATEEXPRESSION(CAR(PARAMETERLIST)));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }
  DirNumber = LONGVALUE(Op1);
  if( DirNumber <1 || DirNumber >= MAXDIRS )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);

  if( pDCO->dp[DirNumber]->FileIndex >= pDCO->dp[DirNumber]->cFileNames ){
    RESULT = NULL;
    RETURN;
    }

  RESULT = NEWMORTALSTRING(pDCO->dp[DirNumber]->cbFileName[pDCO->dp[DirNumber]->FileIndex]);
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  memcpy(STRINGVALUE(RESULT),pDCO->dp[DirNumber]->ppszFileName[pDCO->dp[DirNumber]->FileIndex],
         (STRLEN(RESULT)=pDCO->dp[DirNumber]->cbFileName[pDCO->dp[DirNumber]->FileIndex]));
  pDCO->dp[DirNumber]->FileIndex++;
#endif
END

/**EOD
=display EOD()
=title EOD(dn)
=section file

Checks if there is still some file names in the directory opened for reading using the directory
number T<dn>.

See also R<NEXTFILE>.

*/
COMMAND(EODFUN)
#if NOTIMP_EODFUN
NOTIMPLEMENTED;
#else

  VARIABLE Op1;
  pDirCommandObject pDCO;
  unsigned long DirNumber;

  INITDIR;
  pDCO = (pDirCommandObject)PARAMPTR(CMD_OPENDIR);

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  Op1 = CONVERT2LONG(EVALUATEEXPRESSION(CAR(PARAMETERLIST)));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }
  DirNumber = LONGVALUE(Op1);
  if( DirNumber <1 || DirNumber >= MAXDIRS )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  if( pDCO->dp[DirNumber]->FileIndex >= pDCO->dp[DirNumber]->cFileNames ){
    RESULT = NEWMORTALLONG;
    if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
    LONGVALUE(RESULT) = -1;
    RETURN;
    }
  RESULT = NEWMORTALLONG;
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  LONGVALUE(RESULT) = 0;
#endif
END

/**RESETDIR
=section file
=title RESET DIRECTORY [#] dn
=display RESET DIRECTORY

Reset the directory file name list and start from the first file name when the next call to R<NEXTFILE>
is performed.

See also R<OPENDIR>, R<CLOSEDIR>, R<NEXTFILE>, R<EOD>.

*/
COMMAND(RESETDIR)
#if NOTIMP_RESETDIR
NOTIMPLEMENTED;
#else

  VARIABLE Op1;
  pDirCommandObject pDCO;
  unsigned long DirNumber;

  INITDIR;
  pDCO = (pDirCommandObject)PARAMPTR(CMD_OPENDIR);

  Op1 = CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }
  DirNumber = LONGVALUE(Op1);
  if( DirNumber <1 || DirNumber >= MAXDIRS )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  pDCO->dp[DirNumber]->FileIndex = 0;
#endif
END

/**CLOSEDIR
=section file
=title CLOSE DIRECTORY [#] dn
=display CLOSE DIRECTORY

Close an opened directory and release all memory that was used by the file list.

See also R<OPENDIR>.
*/
COMMAND(CLOSEDIR)
#if NOTIMP_CLOSEDIR
NOTIMPLEMENTED;
#else

  VARIABLE Op1;
  pDirCommandObject pDCO;
  unsigned long DirNumber;

  INITDIR;
  pDCO = (pDirCommandObject)PARAMPTR(CMD_OPENDIR);

  Op1 = CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }
  DirNumber = LONGVALUE(Op1);
  if( DirNumber <1 || DirNumber >= MAXDIRS )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  close_directory_list(pEo,DirNumber);
#endif
END

/**SLEEP
=section misc
=title SLEEP(n)
=display SLEEP()

Suspend the execution of the interpreter (process or thread) for T<n> seconds.
*/
COMMAND(SLEEP)
#if NOTIMP_SLEEP
NOTIMPLEMENTED;
#else

  VARIABLE Op1;

  Op1 = CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  if( Op1 != NULL )
    sys_sleep(LONGVALUE(Op1));

#endif
END

/**HOSTNAME
=title HOSTNAME()
=display HOSTNAME()
=section misc
Return the local host name.
*/
COMMAND(HOSTNAME)
#if NOTIMP_HOSTNAME
NOTIMPLEMENTED;
#else

  char *pszBuffer;
  long cbBuffer;
  int err;

  cbBuffer = 256;
  pszBuffer = ALLOC(cbBuffer);
  err = HOOK_GETHOSTNAME(pszBuffer,cbBuffer);
  if( err == 0 ){
    cbBuffer = strlen(pszBuffer);
    RESULT = NEWMORTALSTRING(cbBuffer);
    if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
    memcpy(STRINGVALUE(RESULT),pszBuffer,cbBuffer);
    }else RESULT = NULL;
#endif
END

/**CURDIR
=title CURDIR()
=displax CURDIR()
=section misc
Return the current directory.
*/
COMMAND(CURDIR)
#if NOTIMP_CURDIR
NOTIMPLEMENTED;
#else

 char *Buffer;
 long cBuffer;

  USE_CALLER_MORTALS;

 cBuffer = 256;
 Buffer = ALLOC(cBuffer);
 while( HOOK_CURDIR(Buffer,cBuffer) == -1 ){
   FREE(Buffer);
   cBuffer += 256;
   if( cBuffer > 1024 )ERROR(COMMAND_ERROR_CURDIR);
   Buffer = ALLOC(cBuffer);
   }
 cBuffer = strlen(Buffer);
 RESULT = NEWMORTALSTRING(cBuffer);
 if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);

 memcpy(STRINGVALUE(RESULT),Buffer,cBuffer);
#endif
END

/**CHDIR
=section misc
=title CHDIR directory

Change the current working directory.
*/
COMMAND(CHDIR)
#if NOTIMP_CHDIR
NOTIMPLEMENTED;
#else

  VARIABLE Op;
  char *Buffer;
  int i;

  Op = CONVERT2STRING(_EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  if( Op == NULL )ERROR(COMMAND_ERROR_UNDEF_DIR);

  SECUREFILE(Op)
  CONVERT2ZCHAR(Op,Buffer);

  i = HOOK_CHDIR(Buffer);
  FREE(Buffer);
  if( i )ERROR(COMMAND_ERROR_CHDIR);
  
#endif
END

/**SETFILE
=section file
=title SET FILE filename parameter=value
=display SET FILE

Set some of the parameters of a file. The parameter can be:

=itemize
=item T<owner> set the owner of the file. This operation requires T<root> permission on UNIX or
T<Administrator> privileges on Windows NT. The value should be the string representation of the
UNIX user or the Windows NT domain user.
=item T<createtime> 
=item T<modifytime> 
=item T<accesstime>
=item Set the time of the file. The value should be the file time in
seconds since January 1,1970. 00:00GMT.
=noitemize

*/
COMMAND(SETFILE)
#if NOTIMP_SETFILE
NOTIMPLEMENTED;
#else

  VARIABLE vAttribute,vFile;
  long iErrorC;
  char *pszAttribute,*pszFile,*pszAttributeSymbol;

  vFile = CONVERT2STRING(_EVALUATEEXPRESSION(PARAMETERNODE));
  ASSERTOKE;
  NEXTPARAMETER;
  pszAttributeSymbol = pEo->StringTable+pEo->CommandArray[_ActualNode-1].Parameter.CommandArgument.Argument.szStringValue;
  NEXTPARAMETER;
  vAttribute = EVALUATEEXPRESSION(PARAMETERNODE);
  ASSERTOKE;

  if( vAttribute == NULL )ERROR(COMMAND_ERROR_SETFILE_INVALID_ATTRIBUTE);
  if( vFile == NULL )ERROR(COMMAND_ERROR_INVALID_FILE_NAME);

  SECUREFILE(vFile)
  CONVERT2ZCHAR(  vFile  , pszFile  );
  if( !stricmp(pszAttributeSymbol,"owner") ){
    vAttribute = CONVERT2STRING(vAttribute);
    CONVERT2ZCHAR(  vAttribute , pszAttribute );
    iErrorC = HOOK_CHOWN(pszFile,pszAttribute);
    FREE(pszAttribute);
    }
  else if( !stricmp(pszAttributeSymbol,"createtime") ){
    CONVERT2LONG(vAttribute);
    iErrorC = HOOK_SETCREATETIME(pszFile,LONGVALUE(vAttribute));
    }
  else if( !stricmp(pszAttributeSymbol,"modifytime") ){
    CONVERT2LONG(vAttribute);
    iErrorC = HOOK_SETMODIFYTIME(pszFile,LONGVALUE(vAttribute));
    }
  else if( !stricmp(pszAttributeSymbol,"accesstime") ){
    CONVERT2LONG(vAttribute);
    iErrorC = HOOK_SETACCESSTIME(pszFile,LONGVALUE(vAttribute));
    }
  else{
    FREE(pszFile);
    ERROR(COMMAND_ERROR_SETFILE_INVALID_ATTRIBUTE);
    }
  FREE(pszFile);
  if( iErrorC )ERROR(iErrorC);
#endif
END

/**KILL
=section process
=display KILL()
=title KILL(pid)

This function kills a process given by the pid and returns true if the
process was successfully killed. Otherwise it returns false.
*/
COMMAND(KILL)
#if NOTIMP_KILL
NOTIMPLEMENTED;
#else

  long pid;
  NODE nItem;

  USE_CALLER_MORTALS;

  nItem = PARAMETERLIST;
  pid = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(CAR(nItem))));
  ASSERTOKE;

  RESULT = NEWMORTALLONG;
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  if( HOOK_KILLPROC(pid) )
    LONGVALUE(RESULT) = 0L;
  else
    LONGVALUE(RESULT) = -1L;

#endif
END

/**FOWNER
=section file
=title FILEOWNER(FileName)

This function returns the name of the ownerof a file.
*/
#define MAXOWNERLEN 512
COMMAND(FOWNER)
#if NOTIMP_FOWNER
NOTIMPLEMENTED;
#else

  char *pszOwnerBuffer,*pszFileName;
  long cbOwnerBuffer;
  VARIABLE vFileName;

  USE_CALLER_MORTALS;

  vFileName = CONVERT2STRING(_EVALUATEEXPRESSION(CAR(PARAMETERLIST)));
  ASSERTOKE;
  SECUREFILE(vFileName)
  CONVERT2ZCHAR(vFileName,pszFileName);
  pszOwnerBuffer = ALLOC( MAXOWNERLEN );
  if( pszOwnerBuffer == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  cbOwnerBuffer = MAXOWNERLEN;
  if( HOOK_GETOWNER(pszFileName,pszOwnerBuffer,cbOwnerBuffer) ){
    RESULT = NULL;
    RETURN;
    }
  FREE(pszFileName);
  RESULT = NEWMORTALSTRING(cbOwnerBuffer=strlen(pszOwnerBuffer));
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  memcpy(STRINGVALUE(RESULT),pszOwnerBuffer,cbOwnerBuffer);

#endif
END

/**CRYPT
=section file misc
=display CRYPT()
=title CRYPT(string,salt)

This function returns the encoded DES digest of the string using the salt
as it is used to encrypt passwords under UNIX.

Note that only the first 8 characters of the string are taken into account.
*/
COMMAND(FCRYPT)
#if NOTIMP_FCRYPT
NOTIMPLEMENTED;
#else

  char *pszString,*pszSalt;
  char szResult[13];
  VARIABLE vString,vSalt;
  NODE nItem;

  USE_CALLER_MORTALS;

  nItem = PARAMETERLIST;

  vString = CONVERT2STRING(_EVALUATEEXPRESSION(CAR(nItem)));
  ASSERTOKE;
  nItem = CDR(nItem);
  vSalt = CONVERT2STRING(_EVALUATEEXPRESSION(CAR(nItem)));
  ASSERTOKE;

  if( vString == NULL || vSalt == NULL ){
    RESULT = NULL;
    RETURN;
    }

  CONVERT2ZCHAR(vString,pszString);
  CONVERT2ZCHAR(vSalt,pszSalt);

  HOOK_FCRYPT(pszString,pszSalt,szResult);

  FREE(pszString);
  FREE(pszSalt);

  RESULT = NEWMORTALSTRING(12);
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  memcpy(STRINGVALUE(RESULT),szResult,12);

#endif
END

/**FORK
=section process
=display FORK()
=title FORK()
=subtitle NOT IMPLEMENTED
Not implemented yet. Cygwin has an implementation, but we need a better one for NT.
Until then we do not support it on UNIX as an intrinsic, because we support all functions
and commands portable on UNIX and NT.
*/
COMMAND(FORK)
#if NOTIMP_FORK
NOTIMPLEMENTED;
#else
NOTIMPLEMENTED;
#endif
END

/**SYSTEM
=section process
=title SYSTEM(executable_program)
=display SYSTEM()
Run a program.
*/
COMMAND(CREATEPROCESS)
#if NOTIMP_CREATEPROCESS
NOTIMPLEMENTED;
#else

  char *pszCommandLine;
  VARIABLE vCommandLine;
  long lPid;

  USE_CALLER_MORTALS;

  vCommandLine = CONVERT2STRING(_EVALUATEEXPRESSION(CAR(PARAMETERLIST)));
  ASSERTOKE;
  if( vCommandLine == NULL ){
    RESULT = NULL;
    RETURN;
    }

  SECUREFILE(vCommandLine)
  CONVERT2ZCHAR(vCommandLine,pszCommandLine);

  lPid = HOOK_CREATEPROCESS(pszCommandLine);

  FREE(pszCommandLine);

  RESULT = NEWMORTALLONG;
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  LONGVALUE(RESULT) = lPid;

#endif
END

/**BINMODE
=section file
=title BINMODE [ # fn ] | input | output
Set an opened file handling to binary mode.

The argument is either a file number with which the file was opened or one of keywords
T<input> and T<output>. In the latter case the standard input or output is set.

See also R<TEXTMODE>
*/
COMMAND(BINMF)
#if NOTIMP_BINMF
NOTIMPLEMENTED;
#else

  long FileNumber;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);
  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;

  if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_IS_NOT_OPENED);
  if( pFCO->mode[FileNumber] != 's' )/* sockets are binary and binary only */
    HOOK_BINMODE(THISFILEP);
#endif
END

/**TEXTMODE
=section file
=title TEXTMODE [ # fn] | input | output
Set an opened file handling to text mode. 

The argument is either a file number with which the file was opened or one of keywords
T<input> and T<output>. In the latter case the standard input or output is set.

See also R<BINMODE>
*/
COMMAND(TXTMF)
#if NOTIMP_TEXTMF
NOTIMPLEMENTED;
#else

  long FileNumber;
  pFileCommandObject pFCO;

  INITIALIZE;
  pFCO = (pFileCommandObject)PARAMPTR(CMD_OPEN);
  FileNumber = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(PARAMETERNODE)));
  ASSERTOKE;

  if( FileNumber < 1 || FileNumber > MAXFILES )ERROR(COMMAND_ERROR_BAD_FILE_NUMBER);
  FileNumber --;
  if( ! pFCO->mode[FileNumber] )ERROR(COMMAND_ERROR_FILE_IS_NOT_OPENED);
  if( pFCO->mode[FileNumber] != 's' )/* sockets are binary and binary only */
    HOOK_TEXTMODE(THISFILEP);
#endif
END

COMMAND(BINMO)
#if NOTIMP_BINMO
NOTIMPLEMENTED;
#else
  HOOK_BINMODE(stdout);
#endif
END

COMMAND(BINMI)
#if NOTIMP_BINMI
NOTIMPLEMENTED;
#else
  HOOK_BINMODE(stdin);
#endif
END

COMMAND(TXTMO)
#if NOTIMP_TEXTMO
NOTIMPLEMENTED;
#else
  HOOK_TEXTMODE(stdout);
#endif
END

COMMAND(TXTMI)
#if NOTIMP_TEXTMI
NOTIMPLEMENTED;
#else
  HOOK_TEXTMODE(stdin);
#endif
END
