/*function.c

--GNU LGPL
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdlib.h>
#include <stdio.h>

#include "../command.h"

/*POD
=H Function and sub commands

This code implements the start and the end of a function.

When the program execution reaches the code that was generated from the code

=verbatim
     fun(x,y,z)     
=noverbatim

if saves some of the state variables, like the function result pointer, local variable pointer,
program counter, error resume and erroro goto values. The function evaluation does NOT allocate
local variables, and does not evaluate the function arguments. Instead it calls the executor
function recursively to perform execution starting at the node, which was generated from

=verbatim
        FUNCTION fname(a,b,c)
=noverbatim

The code of the command should evaluate the arguments (it gets the expression list node in a
class variable), and should allocate the local variables and put the values into the local
variables.

Note that the implementation of the command T<FUNCTION> is very sophisticated, and uses many
features of the execution system, which is usually not needed by other commands. Note that for this
reasons it accesses variables directly without using the hiding macros supplied via T<command.c>

The code implementing T<FUNCTION> should access the global variables, the local variables
of the caller and the local variables of the just starting function, at the same time.

Also note that the parameter evaluation is special. It first checks the expression. If this is a normal,
compound expression it does evaluate it the normal way and puts the result to the respective local variable.
However if the expression is a variable then it generates a referring variable. Whenever value is assigned to
a variable that holds a referring value the instruction T<LET> searches for the referred variable and the
assignment is done using that variable.

This means that all variables are passed by reference. If you want to pass a variable by value you should
apply some tricks, like:

=verbatim
a=1
call f(a+0)
print a
print
call f(a)
print a
print
function f(x)
 x=x+1
end function
=noverbatim

CUT*/

/**FUNCTION
=section misc
=title FUNCTION fun()

Declare a function with possible arguments, local variables and local error handling.
Function value is returned assigned to the function name.

=verbatim
FUNCTION fun(a,b,c)
...
fun = returnvalue
...
END FUNCTION
=noverbatim
*/
COMMAND(FUNCTION)
#if NOTIMP_FUNCTION
NOTIMPLEMENTED;
#else

  IDENTICAL_COMMAND(FUNCTIONARG)

#endif
END

/*POD
=section FUNCTIONARG
=H FUNCTIONARG

This command implements the function head.

CUT*/
void COMMAND_FUNCTIONARG(pExecuteObject pEo
  ){
#if NOTIMP_FUNCTIONARG
NOTIMPLEMENTED;
#else
  MortalList _ThisCommandMortals=NULL;
  pMortalList _pThisCommandMortals = &_ThisCommandMortals;
  unsigned long _ActualNode=pEo->ProgramCounter;
  int iErrorCode;
  NODE nItem,nExpression;
  unsigned long i;
  unsigned long NumberOfArguments;
  long Opcode;
  pFixSizeMemoryObject ItemResult;

/* note that this code should allocate and put values into the function local variables, but
   should evaluate expressions and code using the caller local variables. Therefore we store
   the caller local variables pointer in CallerLocalVariables and whenever we need a call
   using the caller local variables we swap the two pointers for the shortest time possible.
*/
  pFixSizeMemoryObject CallerLocalVariables,SwapLVP;
#define SWAP_LOCAL_VARIABLES SwapLVP = CallerLocalVariables; \
                             CallerLocalVariables = pEo->LocalVariables;\
                             pEo->LocalVariables = SwapLVP;

  nItem = pEo->CommandArray[pEo->ProgramCounter-1].Parameter.NodeList.actualm ;
  Opcode = pEo->CommandArray[nItem-1].OpCode;
  pEo->cLocalVariables = pEo->CommandArray[nItem-1].Parameter.CommandArgument.Argument.lLongValue;
  nItem = pEo->CommandArray[nItem-1].Parameter.CommandArgument.next;
  NumberOfArguments = pEo->CommandArray[nItem-1].Parameter.CommandArgument.Argument.lLongValue;
  nItem = pEo->CommandArray[nItem-1].Parameter.CommandArgument.next;
  nItem = pEo->CommandArray[nItem-1].Parameter.CommandArgument.Argument.lLongValue;

  if( ! pEo->fWeAreCallingFuction ){
    SETPROGRAMCOUNTER(CDR(nItem));
    return;
    }

  CallerLocalVariables = pEo->LocalVariables;
  if( pEo->cLocalVariables ){
    pEo->LocalVariables = memory_NewArray(pEo->pMo,1,pEo->cLocalVariables);
    if( pEo->LocalVariables == NULL ){
      pEo->fStop = fStopSTOP;
      return;
      }
    }else pEo->LocalVariables = NULL; /* it should have been null anyway */

  nItem = pEo->FunctionArgumentsNode;
  i = 0;
  while( nItem ){
    i++ ;
    nExpression = CAR(nItem);
    switch( OPCODE(nExpression) ){
      case eNTYPE_ARR:
        ItemResult = memory_NewRef(pEo->pMo);
        SWAP_LOCAL_VARIABLES;
        ItemResult->Value.aValue = execute_LeftValueArray(pEo,nExpression,_pThisCommandMortals,&iErrorCode);
        SWAP_LOCAL_VARIABLES;
        break;
      case eNTYPE_SAR:
        ItemResult = memory_NewRef(pEo->pMo);
        SWAP_LOCAL_VARIABLES;
        ItemResult->Value.aValue = execute_LeftValueSarray(pEo,nExpression,_pThisCommandMortals,&iErrorCode);
        SWAP_LOCAL_VARIABLES;
        break;
      case eNTYPE_LVR:
        ItemResult = memory_NewRef(pEo->pMo);
        SWAP_LOCAL_VARIABLES;
        ItemResult->Value.aValue = &(pEo->LocalVariables->Value.aValue[pEo->CommandArray[nExpression-1].Parameter.Variable.Serial-1]);
        SWAP_LOCAL_VARIABLES;
        break;
      case eNTYPE_GVR:
        ItemResult = memory_NewRef(pEo->pMo);
        ItemResult->Value.aValue = &(pEo->GlobalVariables->Value.aValue[pEo->CommandArray[nExpression-1].Parameter.Variable.Serial-1]);
        break;
      default:
        SWAP_LOCAL_VARIABLES;
        ItemResult = EVALUATEEXPRESSION(nExpression);
        SWAP_LOCAL_VARIABLES;
        ASSERTOKE;
        if( ItemResult)
          memory_Immortalize(ItemResult,_pThisCommandMortals);
        break;
      }
    if( i <= NumberOfArguments ){
      pEo->LocalVariables->Value.aValue[i-1] = ItemResult;
      }
    nItem = CDR(nItem);
    }

  memory_ReleaseMortals(pEo->pMo,&_ThisCommandMortals);
  /* and finally we start to execute the function when executing the next command */
  pEo->lFunctionLevel++;
  /* some macros need this label */
#endif
_FunctionFinishLabel: ;
  }

/*

This structure is used to maintain the return gosub stack. This structure
becomes empty when there is no any GOSUB/RETURN pairs pending. If some
of the GOSUB-s did not return when the code finishes the memory is released
when the segment is released. We do leak here memory at this level, because
upper levels release this memory before exiting the interpreter.

*/
typedef struct _GosubStack {
  struct _GosubStack *next;
  long lFunctionLevel;
  NODE nReturnNode;
  } GosubStack , *pGosubStack;

#define GosubStackRoot ((pGosubStack)PARAMPTR(CMD_GOSUB))

/**GOSUB
=section misc 
=H Gosub commands
=title GOSUB label

Call a local subroutine. This is the good old way implementation of the BASIC T<GOSUB>
command. See also R<RETURN>.
*/
COMMAND(GOSUB)
#if NOTIMP_GOSUB
NOTIMPLEMENTED;
#else

  pGosubStack pGSS;

  pGSS = ALLOC(sizeof(GosubStack));
  if( pGSS == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  pGSS->lFunctionLevel = pEo->lFunctionLevel;
  pGSS->nReturnNode = pEo->CommandArray[pEo->ProgramCounter-1].Parameter.NodeList.rest;
  pGSS->next = GosubStackRoot;
  GosubStackRoot = pGSS;
  SETPROGRAMCOUNTER(PARAMETERNODE);

#endif
END

/**RETURN
=section misc

Return from a subroutine started with R<GOSUB>.
*/
COMMAND(RETURNC)
#if NOTIMP_RETURNC
NOTIMPLEMENTED;
#else

  pGosubStack pGSS;

  pGSS = GosubStackRoot;
  if( pGSS == NULL || pGSS->lFunctionLevel < pEo->lFunctionLevel )ERROR(COMMAND_ERROR_RETURN_WITHOUT_GOSUB);
  GosubStackRoot = GosubStackRoot->next;
  SETPROGRAMCOUNTER(pGSS->nReturnNode);
  FREE(pGSS);
#endif
END

/*EXITFUNC
=section misc
=title EXIT FUNCTION
=display EXIT FUNCTION

Exit from a function.
*/
COMMAND(EXITFUNC)
#if NOTIMP_EXITFUNC
NOTIMPLEMENTED;
#else

  pGosubStack pGSS;

  /* step back the function level because we are leaving the function */
  pEo->lFunctionLevel--;

  /* clean up the gosub stack */
  pGSS = GosubStackRoot;
  while( pGSS && pGSS->lFunctionLevel > pEo->lFunctionLevel ){
    GosubStackRoot = GosubStackRoot->next;
    FREE(pGSS);
    pGSS = GosubStackRoot;
    }
  pEo->fStop = fStopRETURN;
  
#endif
END

COMMAND(ENDFUNC)
#if NOTIMP_ENDFUNC
NOTIMPLEMENTED;
#else

  IDENTICAL_COMMAND(EXITFUNC)
  
#endif
END

COMMAND(FLET)
#if NOTIMP_FLET
NOTIMPLEMENTED;
#else


  VARIABLE ItemResult;

  /* here we get a mortal value as result. This should be mortal, because it is immortalized and
     the code later assumes that this is an immortal memory piece that is assigned only to this
     "variable"
  */
  ItemResult = EVALUATEEXPRESSION(PARAMETERNODE);
  ASSERTOKE;

  if( pEo->pFunctionResult )
    memory_ReleaseVariable(pEo->pMo,pEo->pFunctionResult);
  if( ItemResult )IMMORTALIZE(ItemResult);
  pEo->pFunctionResult = ItemResult;

#endif
END

/**ADDRESSF
=section misc
=title ADDRESS( myFunc() )
=display ADDRESS()

Return the entry point of a function or subroutine. The returned value is to be used solely
in a corresponding R<ICALL>. Faking aroud with the value may crash the interpreter.

*/
COMMAND(ADDRESSF)
#if NOTIMP_ADDRESSF
NOTIMPLEMENTED;
#else

  NODE z;

  USE_CALLER_MORTALS;
  z = PARAMETERLIST;

  if( OPCODE(CAR(z)) != eNTYPE_FUN )ERROR(COMMAND_ERROR_INVALID_ARGUMENT_FOR_FUNCTION_ADDRESS);

  RESULT = NEWMORTALLONG;
  if( RESULT == NULL )ERROR(COMMAND_ERROR_MEMORY_LOW);
  LONGVALUE(RESULT) = CAR(pEo->CommandArray[z-1].Parameter.UserFunction.NodeId);

#endif
END

/**ICALL
=section misc
=title ICALL n,v1,v2, ... ,vn
=display ICALL

ICALL is implicit call. The first argument of an T<ICALL> command or T<ICALL> function
should be the integer value returned by the function R<ADDRESS> as the address of a user
defined function.

=details

Whenever you call a function or subroutine you have to know the name of 
the subroutine or function. In some situation programmers want to call a function 
without knowing the name of the function. For ex-ample you want to write a 
sorting subroutine that sorts general elements and the caller should provide a subroutine that makes 
the comparison. This way the sorting algorithm can be implemented only once and need 
not be rewritten each time a new type of data is to be sorted. The sorting subroutine gets 
the comparing function as an argument and calls the function indirectly. ScriptBasic can not pass 
functions as arguments to other functions, but it can pass integer numbers. The function 
R<ADDRESS> can be used to convert a function into integer. The result of the 
built-in function R<ADDRESS> is an integer number, which is associated inside the 
basic code with the function. You can pass this value to 
the T<ICALL> command or function as first argument. The icall command is the command 
for indirect subroutine call. The call

=verbatim
icall Address(MySubroutine()),arg1,arg2,arg3
=noverbatim

is equivalent to

=verbatim
call MySubroutine( arg1,arg2,arg3)
=noverbatim

If you call a function that has return value use can use the icall function instead of the icall state-ment:

=verbatim
A = icall(Address(MyFunction()),arg1,arg2,arg3)
=noverbatim

is equivalent to

=verbatim
A = MyFunction(arg1,arg2,arg3)
=noverbatim

The real usage of the function Address and icall can be seen in the following example:

=verbatim
sub MySort(sfun,q)
local ThereWasNoChange,SwapVar
repeat
 ThereWasNoChange = 1
 for i=lbound(q) to ubound(q)-1

  if  icall(sfun,q[i],q[i+1]) > 0 then
   ThereWasNoChange = 0
   SwapVar = q[i]
   q[i] = q[i+1]
   q[i+1] = SwapVar
  endif

 next i
until ThereWasNoChange

end sub

function IntegerCompare(a,b)
  if a < b then
   cmp = -1
  elseif a = b then
   cmp = 0
  else
   cmp = 1
  endif
end function

h[0] = 2
h[1] = 7
h[2] = 1

MySort address(IntegerCompare()) , h

for i=lbound(h) to ubound(h)
 print h[i],"\n"
next i
=noverbatim

Note that the argument of the function Address is a function call. ScriptBasic allows variables 
and func-tions to share the same name. Address is a built-in function just as any other 
built in function, and therefore the expression

Address(MySub) B<THIS IS WRONG!>

Is syntactically correct. The only problem is that it tries to calculate 
the address of the variable MySub, which it can not and results a run-time error.
Instead you have to write

=verbatim
Address( MySub() )
=noverbatim

using the parentheses. In this situation the function or subroutine 
T<MySub()> will not be invoked. The parentheses tell the compiler that 
this is a function and not a variable.
*/
COMMAND(ICALLFUN)
#if NOTIMP_ICALLFUN
NOTIMPLEMENTED;
#else

  NODE nItem;
  VARIABLE ItemResult;
  pFixSizeMemoryObject ThisFunctionResultPointer;
  unsigned long SaveProgramCounter,SaveStepCounter;
  unsigned long SavefErrorGoto,SaveErrorGoto,SaveErrorResume;
  pFixSizeMemoryObject SaveLocalVariablesPointer;
  pFixSizeMemoryObject SaveFunctionResultPointer;

  USE_CALLER_MORTALS;

  if( pEo->FunctionLevelLimit && pEo->lFunctionLevel > pEo->FunctionLevelLimit )
    ERROR(EXE_ERROR_TOO_DEEP_CALL);

  SaveLocalVariablesPointer = pEo->LocalVariables;
  SaveProgramCounter = pEo->ProgramCounter;

  nItem = PARAMETERLIST;
  ItemResult = CONVERT2LONG(EVALUATEEXPRESSION(CAR(nItem)));
  ASSERTOKE;
  pEo->ProgramCounter = LONGVALUE(ItemResult);

  if( pEo->ProgramCounter == 0 )ERROR(EXE_ERROR_USERFUN_UNDEFINED);

  pEo->FunctionArgumentsNode = CDR(nItem);
  SaveFunctionResultPointer = pEo->pFunctionResult;
  pEo->pFunctionResult = NULL;
  SaveStepCounter = pEo->lStepCounter;
  pEo->lStepCounter = 0;
  pEo->fWeAreCallingFuction = 1;
  SaveErrorGoto = pEo->ErrorGoto;
  pEo->ErrorGoto = 0;
  SaveErrorResume = pEo->ErrorResume;
  pEo->ErrorResume = 0;
  SavefErrorGoto = pEo->fErrorGoto;
  pEo->fErrorGoto = ONERROR_NOTHING;
  if( pEo->pHookers->HOOK_ExecCall && 
      (iErrorCode = pEo->pHookers->HOOK_ExecCall(pEo)) )
    ERROR(iErrorCode);
  /* function entering code needs access to the caller local variables, therefore
     WE SHOULD NOT NULL pEo->LocalVariables */
  execute_Execute_r(pEo,&iErrorCode);
  if( pEo->pHookers->HOOK_ExecReturn &&
      (iErrorCode = pEo->pHookers->HOOK_ExecReturn(pEo)) )
    ERROR(iErrorCode);

  pEo->lStepCounter = SaveStepCounter;
  if( pEo->LocalVariables )/* this is null if the function did not have arguments and no local variables */
    memory_ReleaseVariable(pEo->pMo,pEo->LocalVariables);
  pEo->ProgramCounter = SaveProgramCounter;
  pEo->LocalVariables = SaveLocalVariablesPointer;
  ThisFunctionResultPointer = pEo->pFunctionResult;
  pEo->pFunctionResult = SaveFunctionResultPointer;
  /* Functions return their value as immortal values assigned to the very global
     variable pEo->pFunctionResult. Here this variable is restored to point to the
     saved value and the value returned should be mortalized.                   */
  if( ThisFunctionResultPointer && ! IsMortal(ThisFunctionResultPointer) )
    memory_Mortalize(ThisFunctionResultPointer,_pThisCommandMortals);

  pEo->ErrorGoto = SaveErrorGoto;
  pEo->fErrorGoto = SavefErrorGoto;
  pEo->ErrorResume = SaveErrorResume;
  if( iErrorCode )ERROR(iErrorCode);

  RESULT = ThisFunctionResultPointer;
#endif
END

COMMAND(ICALL)
#if NOTIMP_ICALL
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  unsigned long SaveProgramCounter,SaveStepCounter;
  unsigned long SavefErrorGoto,SaveErrorGoto,SaveErrorResume;
  pFixSizeMemoryObject SaveLocalVariablesPointer;
  pFixSizeMemoryObject SaveFunctionResultPointer;

  if( pEo->FunctionLevelLimit && pEo->lFunctionLevel > pEo->FunctionLevelLimit )
    ERROR(EXE_ERROR_TOO_DEEP_CALL);

  SaveLocalVariablesPointer = pEo->LocalVariables;
  SaveProgramCounter = pEo->ProgramCounter;

  ItemResult = CONVERT2LONG(EVALUATEEXPRESSION(CAR(PARAMETERNODE)));
  ASSERTOKE;
  pEo->ProgramCounter = LONGVALUE(ItemResult);

  if( pEo->ProgramCounter == 0 )ERROR(EXE_ERROR_USERFUN_UNDEFINED);

  pEo->FunctionArgumentsNode = CDR(PARAMETERNODE);
  SaveFunctionResultPointer = pEo->pFunctionResult;
  pEo->pFunctionResult = NULL;
  SaveStepCounter = pEo->lStepCounter;
  pEo->lStepCounter = 0;
  pEo->fWeAreCallingFuction = 1;
  SaveErrorGoto = pEo->ErrorGoto;
  pEo->ErrorGoto = 0;
  SaveErrorResume = pEo->ErrorResume;
  pEo->ErrorResume = 0;
  SavefErrorGoto = pEo->fErrorGoto;
  pEo->fErrorGoto = ONERROR_NOTHING;
  if( pEo->pHookers->HOOK_ExecCall && 
      (iErrorCode = pEo->pHookers->HOOK_ExecCall(pEo)) )
    ERROR(iErrorCode);
  /* function entering code needs access to the caller local variables, therefore
     WE SHOULD NOT NULL pEo->LocalVariables */
  execute_Execute_r(pEo,&iErrorCode);
  if( pEo->pHookers->HOOK_ExecReturn &&
      (iErrorCode = pEo->pHookers->HOOK_ExecReturn(pEo)) )
    ERROR(iErrorCode);

  pEo->lStepCounter = SaveStepCounter;
  if( pEo->LocalVariables )/* this is null if the function did not have arguments and no local variables */
    memory_ReleaseVariable(pEo->pMo,pEo->LocalVariables);
  pEo->ProgramCounter = SaveProgramCounter;
  pEo->LocalVariables = SaveLocalVariablesPointer;
  memory_ReleaseVariable(pEo->pMo,pEo->pFunctionResult);
  pEo->pFunctionResult = SaveFunctionResultPointer;

  pEo->ErrorGoto = SaveErrorGoto;
  pEo->fErrorGoto = SavefErrorGoto;
  pEo->ErrorResume = SaveErrorResume;
  if( iErrorCode )ERROR(iErrorCode);

#endif
END

/**CALL
=section misc
=title CALL subroutine

Call a subroutine.
*/
COMMAND(CALL)
#if NOTIMP_CALL
NOTIMPLEMENTED;
#else


  _EVALUATEEXPRESSION(PARAMETERNODE);
  ASSERTOKE;

#endif
END

/**SUB
=section misc
=title SUB fun()

Declare a subroutine with possible arguments, local variables and local error handling.

=verbatim
SUB fun(a,b,c)
...
...
END SUB
=noverbatim

Note that functions and subroutines are not quite different is ScriptBasic. ScriptBasic allows you
to return a value from a subroutine and to call a function using the command T<CALL>. It is just
a convention to have separately T<SUB> and T<FUNCTION> declarations.
*/
COMMAND(SUB)
#if NOTIMP_SUB
NOTIMPLEMENTED;
#else

  IDENTICAL_COMMAND(FUNCTIONARG)

#endif
END

/*POD
=section SUBARG
=H SUBARG

Same as R<FUNCTIONARG>

CUT*/
COMMAND(SUBARG)
#if NOTIMP_SUBARG
NOTIMPLEMENTED;
#else

  IDENTICAL_COMMAND(FUNCTIONARG)

#endif
END


/**EXITSUB
=section misc
=title EXIT SUB
=display EXIT SUB

Same as R<EXITFUNC>

CUT*/
COMMAND(EXITSUB)
#if NOTIMP_EXITSUB
NOTIMPLEMENTED;
#else

  IDENTICAL_COMMAND(EXITFUNC)

#endif
END


COMMAND(ENDSUB)
#if NOTIMP_ENDSUB
NOTIMPLEMENTED;
#else

  IDENTICAL_COMMAND(EXITFUNC)

#endif
END
