/*print.c

--GNU LGPL
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdlib.h>
#include <stdio.h>

#include "../command.h"

/*POD
=H Commands that assign values to variables

This file contains commands that assign value to variables.

CUT*/

/**LET
=section misc
=title v = expression
=display LET

Assign a value to a variable. You should NOT write the keyword T<LET> before the variable.
T<LET> is not a valid keyword in ScriptBasic.

The variable can be a global or local variable, array element or associative array element.

*/
COMMAND(LET)
#if NOTIMP_LET
NOTIMPLEMENTED;
#else
  VARIABLE ExpressionResult;
  LEFTVALUE LetThisVariable;
  long refcount;

  /* we get the pointer to the variable that points to the value */
  LetThisVariable = EVALUATELEFTVALUE(PARAMETERNODE);
  ASSERTOKE;

  /* if this points to a reference value then we search the "real" variable
     to modify */
  DEREFERENCE(LetThisVariable);

  /* get the next parameter of the command, which is the expression */
  NEXTPARAMETER;

  /* Evaluate the expression, and if the expression is simple create a copy of the result
     to be sure that no two variables point to the same value */
  ExpressionResult = execute_Evaluate(pEo,PARAMETERNODE,_pThisCommandMortals,&iErrorCode,0);
  ASSERTOKE;

  /* after the references have been looked after we duplicate the result to have an own copy of the value. */
  ExpressionResult = memory_DupMortalize(pEo->pMo,ExpressionResult,_pThisCommandMortals,&iErrorCode);

  /* if the result of the expression is not undef then immortalize */
  if( ExpressionResult ){
    /* we immortalize the new variable if it is a variable and not NULL meaning undef */
    IMMORTALIZE(ExpressionResult);
    }

  /* if this variable had value assigned to it then release that value */
  if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);

  /* and finally assign the code to the variable */
  *LetThisVariable = ExpressionResult;

#endif
END

/*UNDEF
=section misc
=title UNDEF variable
=display UNDEF

Undefines a variable or release whole array. When this command is used as a function, it simply
returns the value T<undef>.

=details

Note that when this command is called in a function then the
local variable is undefed and the caller variable passed by reference
is not changed. Therefore

=verbatim

sub xx(a)
 undef a
end sub

q = 1
xx q
print q
=noverbatim

will print 1 and not T<undef>.

On the other hand

=verbatim

sub xx(a)
 a = undef
end sub

q = 1
xx q
print q
=noverbatim

does print T<undef>.
*/
COMMAND(CUNDEF)
#if NOTIMP_CUNDEF
NOTIMPLEMENTED;
#else

  NODE nItem;
  LEFTVALUE LetThisVariable;

  nItem = PARAMETERNODE;
  while( nItem ){

    LetThisVariable = EVALUATELEFTVALUE_A(CAR(nItem));
    ASSERTOKE;
    /* do NOT dereference by definition */
    if( *LetThisVariable == NULL ){
      nItem = CDR(nItem);
      continue;
      }

    memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
    *LetThisVariable = NULL;
    nItem = CDR(nItem);
    }

#endif
END

/*BYVAL
=section misc

Convert an argument variable to local. Saying

=verbatim
SUB fun(a,b,c)
BYVAL a,b

 ....

END SUB
=noverbatim

will make the variables T<a> and T<b> truely local, and assigning a value to them will not
alter the variable that was passed as parameter when calling T<fun>.

This keyword can also be used as an operator, like

=verbatim
CALL fun(BYVAL a, BYVAL B)
=noverbatim

that will pass the variables T<a> and T<b> to the function by value instead of reference.

=details

T<BYVAL> as an operator actually does nothing but returns the value of its argument.
When T<BYVAL> is used as a command it assignes the value of the variable to the
local variable that may have been a reference to a global variable before that.
*/
COMMAND(CBYVAL)
#if NOTIMP_CBYVAL
NOTIMPLEMENTED;
#else

  NODE nItem;
  LEFTVALUE LetThisVariable;
  VARIABLE NewValue;
  unsigned long refcount;

  nItem = PARAMETERNODE;
  while( nItem ){

    LetThisVariable = EVALUATELEFTVALUE_A(CAR(nItem));
    ASSERTOKE;

    if( *LetThisVariable == NULL || TYPE(*LetThisVariable) != VTYPE_REF ){
      nItem = CDR(nItem);
      continue;
      }
    NewValue = *LetThisVariable;
    refcount = pEo->maxderef;
    while( NewValue && TYPE(NewValue) == VTYPE_REF ){
      NewValue = *(NewValue->Value.aValue);
      if( ! refcount-- )ERROR(COMMAND_ERROR_CIRCULAR);
      }
    if( NewValue ){
      NewValue = memory_DupVar(pEo->pMo,NewValue,_pThisCommandMortals,&iErrorCode);
      memory_Immortalize(NewValue,_pThisCommandMortals);
      }
    if( *LetThisVariable )
      memory_ReleaseVariable(pEo->pMo,*LetThisVariable); /* release the ref value */
    *LetThisVariable = NewValue;
    nItem = CDR(nItem);
    }

#endif
END