/*mathfunc.c

--GNU LGPL
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdlib.h>
#include <stdio.h>

#include <math.h>

#include "../command.h"

/*POD
=H Mathematical Functions

This file defines all the mathematical functions that are implemented in scriba.
CUT*/


/**POW10
=section math
=display POW10()

Calculates the x-th exponent of 10.

T<POW10(undef)> is T<undef>.
*/
COMMAND(POW)
#if NOTIMP_POW
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2DOUBLE(Op1);
  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = pow(10.0,DOUBLEVALUE(Op1));
  RETURN;

#endif
END


/**EXP
=section math
=display EXP()

Calculates the x-th exponent of e.

T<EXP(undef)> is T<undef>.
*/
COMMAND(EXP)
#if NOTIMP_EXP
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2DOUBLE(Op1);
  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = exp(DOUBLEVALUE(Op1));
  RETURN;

#endif
END

/**LOG
=display LOG()
=section math

Calculates the natural log of the argument. If the argument is zero or less than zero the
result is T<undef>

T<LOG(undef)> is T<undef>.
*/
COMMAND(LOG)
#if NOTIMP_LOG
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2DOUBLE(Op1);
  if( DOUBLEVALUE(Op1) <= 0.0 ){
    RESULT = NULL;
    RETURN;
    }
  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = log(DOUBLEVALUE(Op1));
  RETURN;

#endif
END

/**LOG10
=display LOG10()
=section math

Calculates the log of the argument. If the argument is zero or less than zero the
result is T<undef>

T<LOG10(undef)> is T<undef>.
*/
COMMAND(LOG10)
#if NOTIMP_LOG10
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2DOUBLE(Op1);
  if( DOUBLEVALUE(Op1) <= 0.0 ){
    RESULT = NULL;
    RETURN;
    }
  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = log10(DOUBLEVALUE(Op1));
  RETURN;

#endif
END

/**SIN
=section math
=display SIN()

Calculates the sine of the argument.

T<SIN(undef)> is T<undef>.
*/
COMMAND(SIN)
#if NOTIMP_SIN
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2DOUBLE(Op1);
  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = sin(DOUBLEVALUE(Op1));
  RETURN;

#endif
END

/**ASIN
=section math
=display ASIN()

Calculates the arcus sine of the argument, which is the inverse of the function R<SIN>.
If the argument is not between (-1.0,+1.0) the return value is T<undef>.

T<ASIN(undef)> is T<undef>.
*/
COMMAND(ASIN)
#if NOTIMP_ASIN
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2DOUBLE(Op1);
  if( DOUBLEVALUE(Op1) < -1 || DOUBLEVALUE(Op1) > +1 ){
    RESULT = NULL;
    RETURN;
    }

  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = asin(DOUBLEVALUE(Op1));
  RETURN;

#endif
END

/**ACOS
=display ACOS()
=section math

Calculates the arcus cosine of the argument, which is the inverse of the function R<COS>.
If the argument is not between (-1.0,+1.0) the return value is T<undef>.

T<ACOS(undef)> is T<undef>.
*/
COMMAND(ACOS)
#if NOTIMP_ACOS
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2DOUBLE(Op1);
  if( DOUBLEVALUE(Op1) < -1 || DOUBLEVALUE(Op1) > +1 ){
    RESULT = NULL;
    RETURN;
    }

  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = acos(DOUBLEVALUE(Op1));
  RETURN;

#endif
END

/**COS
=section math
=display COS()
Calculates the cosine of the argument.

T<COS(undef)> is T<undef>.
*/
COMMAND(COS)
#if NOTIMP_COS
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2DOUBLE(Op1);
  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = cos(DOUBLEVALUE(Op1));
  RETURN;

#endif
END

/**RND
=section math
=display RND()

Returns a random number as generated by the C function T<rand()>.

*/
COMMAND(RND)
#if NOTIMP_RND
NOTIMPLEMENTED;
#else


  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = rand();
  RETURN;

#endif
END

/*SGN
=display SGN()
=section math

Calculates the signum of the argument. This is -1 for negative numbers, +1 for positive numbers
and 0 for zero. The return value is always long but T<SGN(undef)> is T<undef>. If the argument is
a string it is converted to long or double.

*/
COMMAND(SGN)
#if NOTIMP_SGN
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  if( TYPE(Op1) == VTYPE_STRING )
    if( ISSTRINGINTEGER(Op1) )
      Op1 = CONVERT2LONG(Op1);
    else
      Op1 = CONVERT2DOUBLE(Op1);

  if( TYPE(Op1) == VTYPE_LONG ){
    if( LONGVALUE(Op1) > 0 )
      LONGVALUE(RESULT) = 1;
    if( LONGVALUE(Op1) < 0 )
      LONGVALUE(RESULT) = -1;
    if( LONGVALUE(Op1) == 0 )
      LONGVALUE(RESULT) = 0;
    RETURN;
    }

  if( DOUBLEVALUE(Op1) > 0 )
    LONGVALUE(RESULT) = 1;
  if( DOUBLEVALUE(Op1) < 0 )
    LONGVALUE(RESULT) = -1;
  if( DOUBLEVALUE(Op1) == 0 )
    LONGVALUE(RESULT) = 0;

#endif
END

/**SQR
=section math
=display SQR()

Calculates the square root of the argument. If the argument is a string it is converted
to double. If the argument is negative the result is T<undef>.

T<SQR(undef)> is T<undef>.
*/
COMMAND(SQR)
#if NOTIMP_SQR
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2DOUBLE(Op1);

  if( DOUBLEVALUE(Op1) < 0 ){
    RESULT = NULL;
    RETURN;
    }

  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = sqrt(DOUBLEVALUE(Op1));

#endif
END

/**ABS
=display ABS()
=section math
Returns the absolute value of the argument. If the argument is a string then
it first converts it to double or long. The return value is double or long
depending on the argument.

T<ABS(undef)> is T<undef>.

*/
COMMAND(ABS)
#if NOTIMP_ABS
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  if( TYPE(Op1) == VTYPE_STRING )
    if( ISSTRINGINTEGER(Op1) )
      Op1 = CONVERT2LONG(Op1);
    else
      Op1 = CONVERT2DOUBLE(Op1);

  if( TYPE(Op1) == VTYPE_LONG ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = LONGVALUE(Op1) > 0 ? LONGVALUE(Op1) : -LONGVALUE(Op1);
    RETURN;
    }

  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = DOUBLEVALUE(Op1) > 0 ? DOUBLEVALUE(Op1) : -DOUBLEVALUE(Op1);

#endif
END

/**VAL
=display VAL()
=section math

Converts a string to numeric value. If the string is integer it returns a long.
If the string contains a number presentation which is a float number the returned
value is double. In case the argument is long or double no conversion is done.

T<VAL(undef)> is T<undef>


*/
COMMAND(VAL)
#if NOTIMP_VAL
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  if( TYPE(Op1) == VTYPE_STRING ){
    if( ISSTRINGINTEGER(Op1) )
      RESULT = CONVERT2LONG(Op1);
    else
      RESULT = CONVERT2DOUBLE(Op1);
    }else{
    RESULT = Op1;
    }

#endif
END

/**PI
=section math

This built-in contant is implemented as an argument less function. Returns the approximate
value of the constant PI which is the ratio of the circumference of a circle to its diameter.

*/
COMMAND(PI)
#if NOTIMP_PI
NOTIMPLEMENTED;
#else


  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = 3.1415926;

#endif
END

COMMAND(UNDEF)
#if NOTIMP_UNDEF
NOTIMPLEMENTED;
#else


  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  RESULT = NULL;

#endif
END

/**TRUE
=section math

This built-in contant is implemented as an argument less function. Returns the value T<true>.
*/
COMMAND(TRUE)
#if NOTIMP_TRUE
NOTIMPLEMENTED;
#else


  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = -1;

#endif
END

/**FALSE
=section math

This built-in contant is implemented as an argument less function. Returns the value T<false>.
*/
COMMAND(FALSE)
#if NOTIMP_FALSE
NOTIMPLEMENTED;
#else


  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = 0;

#endif
END

/**FIX
=display FIX()
=section math

This function returns the integral part of the argument. If the argument is T<undef> then the
return value is T<undef>. Otherwise the return value is always long.


The difference between T<int> and T<fix> is that T<int> truncates down while T<fix> truncates
torward zero. The two functions are identical for positive numbers. In case of negative arguments
T<int> will give a smaller number if the argument is not integer. For example:

=verbatim
  int(-3.3) = -4
  fix(-3.3) = -3
=noverbatim

See R<INT>.
*/
COMMAND(FIX)
#if NOTIMP_FIX
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2LONG(Op1);
  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = LONGVALUE(Op1);

#endif
END

/**INT
=display INT()
=section math

This function returns the integral part of the argument. If the argument is T<undef> then the
return value is T<undef>. Otherwise the return value is allways long.

The difference between T<int> and T<fix> is that T<int> truncates down while T<fix> truncates
torward zero. The two functions are identical for positive numbers. In case of negative arguments
T<int> will give a smaller number if the argument is not integer. For example:

=verbatim
  int(-3.3) = -4
  fix(-3.3) = -3
=noverbatim

See R<FIX>.

*/
COMMAND(INT)
#if NOTIMP_INT
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  if( TYPE(Op1) == VTYPE_STRING ){
    if( ISSTRINGINTEGER(Op1) ){
      RESULT = CONVERT2LONG(Op1);
      RETURN;
      }else{
      Op1 = CONVERT2DOUBLE(Op1);
      }
    }
  else
  if( TYPE(Op1) == VTYPE_LONG ){
    RESULT = Op1;
    RETURN;
    }


  if( DOUBLEVALUE(Op1) < 0.0 ){
    if( DOUBLEVALUE(Op1) != (double)(long)DOUBLEVALUE(Op1) )
      DOUBLEVALUE(Op1) -= 1 ;
    }
  Op1 = CONVERT2LONG(Op1);
  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = LONGVALUE(Op1);

#endif
END

/**FRAC
=display FRAC()
=section math

The fractional part of the argument. This function always returns a double except that
T<frac(undef)> is T<undef>.

Negative arguments return negative value (or zero if the argument is a negative integer),
positive arguments result positive values (or zero if the argument is integer).
*/
COMMAND(FRAC)
#if NOTIMP_FRAC
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  if( TYPE(Op1) == VTYPE_STRING ){
    if( ISSTRINGINTEGER(Op1) ){
      RESULT = NEWMORTALDOUBLE;
      DOUBLEVALUE(RESULT) = 0.0;
      RETURN;
      }else{
      Op1 = CONVERT2DOUBLE(Op1);
      }
    }
  else
  if( TYPE(Op1) == VTYPE_LONG ){
    RESULT = NEWMORTALDOUBLE;
    DOUBLEVALUE(RESULT) = 0.0;
    RETURN;
    }

  RESULT = NEWMORTALDOUBLE;
  DOUBLEVALUE(RESULT) = DOUBLEVALUE(Op1) - (double)(long)DOUBLEVALUE(Op1);

#endif
END


/**ROUND
=display ROUND()
=section math

This function rounds its argument. The first argument is the number to round, and
the optional second argument specifies the number of fractional digits to round to.

If the second argument is missing the function rounds to integer value.

The return value is long if the number of decimal places to keep is zero, otherwise
the return value is double.

Negative value for the number of decimal places result rounding to integer value.

T<ROUND(undef)> is T<undef>. Also T<ROUND(val,undef)> is equivalent to T<ROUND(value)>.
See R<INT>.

*/
COMMAND(ROUND)
#if NOTIMP_ROUND
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;
  long iNumberOfDigits;
  double Multiplier;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }
  nItem = CDR(nItem);

  iNumberOfDigits = 0;
  if( nItem ){
    iNumberOfDigits = LONGVALUE(CONVERT2LONG(EVALUATEEXPRESSION(CAR(nItem))));
    ASSERTOKE;
    }
  if( iNumberOfDigits < 0 )iNumberOfDigits = 0;
  if( iNumberOfDigits > 100 )iNumberOfDigits = 100;

  if( TYPE(Op1) == VTYPE_STRING ){
    if( ISSTRINGINTEGER(Op1) ){
      RESULT = CONVERT2LONG(Op1);
      RETURN;
      }else{
      Op1 = CONVERT2DOUBLE(Op1);
      }
    }
  else
  if( TYPE(Op1) == VTYPE_LONG ){
    RESULT = Op1;
    RETURN;
    }

  if( iNumberOfDigits == 0 ){
    DOUBLEVALUE(Op1) += 0.5;
    if( DOUBLEVALUE(Op1) < 0.0 ){
      if( DOUBLEVALUE(Op1) != (double)(long)DOUBLEVALUE(Op1) )
        DOUBLEVALUE(Op1) -= 1 ;
      }
    Op1 = CONVERT2LONG(Op1);
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = LONGVALUE(Op1);
    RETURN;
    }
  Multiplier = 1.0;
  while( iNumberOfDigits -- )Multiplier *= 10.0;
  DOUBLEVALUE(Op1) *= Multiplier;
  DOUBLEVALUE(Op1) += 0.5;
  if( DOUBLEVALUE(Op1) < 0.0 ){
    if( DOUBLEVALUE(Op1) != (double)(long)DOUBLEVALUE(Op1) )
      DOUBLEVALUE(Op1) -= 1 ;
    }
  DOUBLEVALUE(Op1) = (long)DOUBLEVALUE(Op1);
  DOUBLEVALUE(Op1) /= Multiplier;

  RESULT = Op1;

#endif
END

/**ODD
=section math
=display ODD()

Return T<true> if the argument is an odd number. T<odd(undef)> is T<undef>
*/
COMMAND(ODD)
#if NOTIMP_ODD
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  Op1 = CONVERT2LONG(Op1);

  if( LONGVALUE(Op1) & 1 )
    LONGVALUE(RESULT) = -1;
  else
    LONGVALUE(RESULT) = 0;

#endif
END

/**EVEN
=display EVEN()
=section math
Return T<true> if the argument is an even number. T<even(undef)> is T<undef>.
*/
COMMAND(EVEN)
#if NOTIMP_EVEN
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  Op1 = CONVERT2LONG(Op1);

  if( LONGVALUE(Op1) & 1 )
    LONGVALUE(RESULT) = 0;
  else
    LONGVALUE(RESULT) = -1;

#endif
END

/**LBOUND
=section array
=display LBOUND()

Return the lowest index of the argument array.

*/
COMMAND(LBOUND)
#if NOTIMP_LBOUND
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ){
    RESULT = NULL;
    RETURN;
    }
  switch( TYPE(ItemResult) ){
    case VTYPE_LONG:
    case VTYPE_DOUBLE:
    case VTYPE_STRING:
      RESULT = NULL;
      RETURN;
    case VTYPE_ARRAY:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = ItemResult->ArrayLowLimit;
      RETURN;
    }

#endif
END

COMMAND(UBOUND)
#if NOTIMP_UBOUND
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ){
    RESULT = NULL;
    RETURN;
    }
  switch( TYPE(ItemResult) ){
    case VTYPE_LONG:
    case VTYPE_DOUBLE:
    case VTYPE_STRING:
      RESULT = NULL;
      RETURN;
    case VTYPE_ARRAY:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = ItemResult->ArrayHighLimit;
      RETURN;
    }

#endif
END

COMMAND(ISARRAY)
#if NOTIMP_ISARRAY
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = 0;
    RETURN;
    }
  switch( TYPE(ItemResult) ){
    case VTYPE_LONG:
    case VTYPE_DOUBLE:
    case VTYPE_STRING:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = 0;
      RETURN;
    case VTYPE_ARRAY:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = -1;
      RETURN;
    }

#endif
END

COMMAND(ISSTRING)
#if NOTIMP_ISSTRING
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = 0;
    RETURN;
    }
  switch( TYPE(ItemResult) ){
    case VTYPE_LONG:
    case VTYPE_DOUBLE:
    case VTYPE_ARRAY:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = 0;
      RETURN;
    case VTYPE_STRING:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = -1;
      RETURN;
    }

#endif
END

COMMAND(ISLONG)
#if NOTIMP_ISLONG
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = 0;
    RETURN;
    }
  switch( TYPE(ItemResult) ){
    case VTYPE_DOUBLE:
    case VTYPE_ARRAY:
    case VTYPE_STRING:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = 0;
      RETURN;
    case VTYPE_LONG:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = -1;
      RETURN;
    }

#endif
END

COMMAND(ISDOUBLE)
#if NOTIMP_ISDOUBLE
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = 0;
    RETURN;
    }
  switch( TYPE(ItemResult) ){
    case VTYPE_ARRAY:
    case VTYPE_STRING:
    case VTYPE_LONG:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = 0;
      RETURN;
    case VTYPE_DOUBLE:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = -1;
      RETURN;
    }

#endif
END

COMMAND(ISNUMERIC)
#if NOTIMP_ISNUMERIC
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = 0;
    RETURN;
    }
  switch( TYPE(ItemResult) ){
    case VTYPE_ARRAY:
    case VTYPE_STRING:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = 0;
      RETURN;
    case VTYPE_LONG:
    case VTYPE_DOUBLE:
      RESULT = NEWMORTALLONG;
      LONGVALUE(RESULT) = -1;
      RETURN;
    }
#endif
END

COMMAND(ISDEF)
#if NOTIMP_ISDEF
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = 0;
    RETURN;
    }
  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = -1;
#endif
END

COMMAND(ISUNDEF)
#if NOTIMP_ISUNDEF
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = -1;
    RETURN;
    }
  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = 0;
#endif
END

COMMAND(ISEMPTY)
#if NOTIMP_ISEMPTY
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  if( ItemResult == NULL ||
     ( TYPE(ItemResult) == VTYPE_STRING && STRLEN(ItemResult) == 0 ) ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = -1;
    RETURN;
    }
  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = 0;
#endif
END

COMMAND(TYPE)
#if NOTIMP_TYPE
NOTIMPLEMENTED;
#else

  VARIABLE ItemResult;
  NODE nItem;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameter */
  nItem = PARAMETERLIST;
  ItemResult = _EVALUATEEXPRESSION_A(CAR(nItem));
  ASSERTOKE;

  RESULT = NEWMORTALLONG;

  if( ItemResult == NULL ){
    LONGVALUE(RESULT) = 0;
    RETURN;
    }
  switch( TYPE(ItemResult) ){
    case VTYPE_LONG:
      LONGVALUE(RESULT) = 3;
      break;
    case VTYPE_DOUBLE:
      LONGVALUE(RESULT) = 2;
      break;
    case VTYPE_STRING:
      LONGVALUE(RESULT) = 1;
      break;
    case VTYPE_ARRAY:
      LONGVALUE(RESULT) = 4;
      break;
    }
#endif
END



/*
1.4.8. ATN, ATAN
1.4.9. TAN
1.4.10. COTAN
1.4.11. ACTAN
1.4.23. SECANT
1.4.24. COSECANT
1.4.25. ASECANT
1.4.26. ACOSECANT
1.4.27. HSIN
1.4.28. HCOS
1.4.29. HTAN
1.4.30. HSECANT
1.4.31. HCOSECANT
1.4.32. HCOTAN
*/
