/*mathop.c

--GNU LGPL
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
/* comparision operators compare strings as well */
#include <string.h>

#include "../command.h"

/* stringcompare two string values. The values SHOULD be string.
*/
static int STRCMP(VARIABLE Op1, VARIABLE Op2, int iCase){
  unsigned long n;
  char *a,*b;
  char ca,cb;

  if( Op1 == NULL && Op2 == NULL )return 0;
  if( Op1 == NULL )return 1;
  if( Op2 == NULL )return -1;
  iCase &= 1;/* only the lowest bit is about case sensitivity */
  n = STRLEN(Op1);
  if( n > STRLEN(Op2) ) n= STRLEN(Op2);
  a = STRINGVALUE(Op1);
  b = STRINGVALUE(Op2);
  while( n-- ){
    ca = *a;
    cb = *b;
    if( iCase ){
      if( isupper(ca) )ca = tolower(ca);
      if( isupper(cb) )cb = tolower(cb);
      }
    if( ca != cb )return ( (ca)-(cb) );
    a++;
    b++;
    }
  if( STRLEN(Op1) == STRLEN(Op2) )return 0;
  if( STRLEN(Op1) > STRLEN(Op2) )return 1;
  return -1;
  }

static long longpow(long a,long b){
  long result;

  result = 1;
  while( b ){
    if( b&1 )result *= a;
    b /= 2;
    a *= a;
    }
  return result;
}

/*POD
=H Mathematical operators

This file defines all the mathematical operators that are implemented in scriba.

CUT*/

/*POD
=section MULT
=H Multiplication

This operator multiplies two numbers. If one of the arguments is double then the result
is double, otherwise the result is long.

If one of the operators is undefined the result is undefined.

CUT*/
COMMAND(MULT)
#if NOTIMP_MULT
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1,Op2;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  nItem = CDR(nItem);
  Op2 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op2 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  /* if any of the arguments is double then the result is double */
  if( (TYPE(Op1) == VTYPE_DOUBLE || TYPE(Op2) == VTYPE_DOUBLE) ||

     /* if the first argument is string and is NOT integer */
     ( TYPE(Op1) == VTYPE_STRING && !ISSTRINGINTEGER(Op1)) ||

     /* if the second argument is string and is NOT integer */
     ( TYPE(Op2) == VTYPE_STRING && !ISSTRINGINTEGER(Op2))
     ){
    RESULT = NEWMORTALDOUBLE;
    Op1 = CONVERT2DOUBLE(Op1);
    Op2 = CONVERT2DOUBLE(Op2);
    DOUBLEVALUE(RESULT) = DOUBLEVALUE(Op1) * DOUBLEVALUE(Op2);
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  Op1 = CONVERT2LONG(Op1);
  Op2 = CONVERT2LONG(Op2);
  LONGVALUE(RESULT) = LONGVALUE(Op1) * LONGVALUE(Op2);

#endif
END

COMMAND(EQ)
#if NOTIMP_EQ
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1,Op2;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  nItem = CDR(nItem);
  Op2 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  /* undef is equal to undef */

  if( Op1 == NULL && Op2 == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = -1;
    RETURN;
    }

  /* undef is not equal to anything else */
  if( Op1 == NULL || Op2 == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = 0;
    RETURN;
    }

  /* if any of the arguments is string then we compare strings */
  if( TYPE(Op1) == VTYPE_STRING || TYPE(Op2) == VTYPE_STRING ){
    Op1 = CONVERT2STRING(Op1);
    Op2 = CONVERT2STRING(Op2);
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = STRCMP(Op1,Op2,OPTION("compare")) == 0 ? -1L : 0;

    RETURN;
    }

  /* if any of the arguments is double then we compare double */
  if( TYPE(Op1) == VTYPE_DOUBLE || TYPE(Op2) == VTYPE_DOUBLE ){
    RESULT = NEWMORTALLONG;
    Op1 = CONVERT2DOUBLE(Op1);
    Op2 = CONVERT2DOUBLE(Op2);
    LONGVALUE(RESULT) = DOUBLEVALUE(Op1) == DOUBLEVALUE(Op2) ? -1L : 0L;
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  Op1 = CONVERT2LONG(Op1);
  Op2 = CONVERT2LONG(Op2);
  LONGVALUE(RESULT) = LONGVALUE(Op1) == LONGVALUE(Op2) ? -1L : 0L;

#endif
END

COMMAND(NE)
#if NOTIMP_NE
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1,Op2;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  nItem = CDR(nItem);
  Op2 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  /* undef is equal to undef */
  if( Op1 == NULL && Op2 == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = 0;
    RETURN;
    }

  /* undef is not equal to anything else */
  if( Op1 == NULL || Op2 == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = -1L;
    RETURN;
    }

  /* if any of the arguments is string then we compare strings */
  if( TYPE(Op1) == VTYPE_STRING || TYPE(Op2) == VTYPE_STRING ){
    Op1 = CONVERT2STRING(Op1);
    Op2 = CONVERT2STRING(Op2);
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = STRCMP(Op1,Op2,OPTION("compare")) != 0 ?  -1L : 0;
    RETURN;
    }

  /* if any of the arguments is double then we compare double */
  if( TYPE(Op1) == VTYPE_DOUBLE || TYPE(Op2) == VTYPE_DOUBLE ){
    RESULT = NEWMORTALLONG;
    Op1 = CONVERT2DOUBLE(Op1);
    Op2 = CONVERT2DOUBLE(Op2);
    LONGVALUE(RESULT) = DOUBLEVALUE(Op1) != DOUBLEVALUE(Op2) ? -1L : 0L;
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  Op1 = CONVERT2LONG(Op1);
  Op2 = CONVERT2LONG(Op2);
  LONGVALUE(RESULT) = LONGVALUE(Op1) != LONGVALUE(Op2) ? -1L : 0L;

#endif
END

#define LOGOP(NAME,OP) \
COMMAND(NAME)\
  NODE nItem;\
  VARIABLE Op1,Op2;\
\
  /* this is an operator and not a command, therefore we do not have our own mortal list */\
  USE_CALLER_MORTALS;\
\
  /* evaluate the parameters */\
  nItem = PARAMETERLIST;\
  Op1 = EVALUATEEXPRESSION(CAR(nItem));\
  ASSERTOKE;\
  nItem = CDR(nItem);\
  Op2 = EVALUATEEXPRESSION(CAR(nItem));\
  ASSERTOKE;\
\
  /* undef is not comparable except for equality */\
  if( Op1 == NULL || Op2 == NULL ){\
    RESULT = NEWMORTALLONG;\
    LONGVALUE(RESULT) = 0;\
    RETURN;\
    }\
\
  /* if any of the arguments is string then we compare strings */\
  if( TYPE(Op1) == VTYPE_STRING || TYPE(Op2) == VTYPE_STRING ){\
    Op1 = CONVERT2STRING(Op1);\
    Op2 = CONVERT2STRING(Op2);\
    RESULT = NEWMORTALLONG;\
    LONGVALUE(RESULT) = STRCMP(Op1,Op2,OPTION("compare")) OP 0 ?  -1L : 0;\
    RETURN;\
    }\
\
  /* if any of the arguments is double then we compare double */\
  if( TYPE(Op1) == VTYPE_DOUBLE || TYPE(Op2) == VTYPE_DOUBLE ){\
    RESULT = NEWMORTALLONG;\
    Op1 = CONVERT2DOUBLE(Op1);\
    Op2 = CONVERT2DOUBLE(Op2);\
    LONGVALUE(RESULT) = DOUBLEVALUE(Op1) OP DOUBLEVALUE(Op2) ? -1L : 0L;\
    RETURN;\
    }\
\
  RESULT = NEWMORTALLONG;\
  Op1 = CONVERT2LONG(Op1);\
  Op2 = CONVERT2LONG(Op2);\
  LONGVALUE(RESULT) = LONGVALUE(Op1) OP LONGVALUE(Op2) ? -1L : 0L;\
\
END

/*POD
=section compare
=H Comparing operators

The comparing operators compare long, double and string values. Whenever any of the
arguments is string the comparisionis done stringwise. If none of the arguments are strings
but one of then is double then the comparision is done between doubles. Otherwise we compare
long values.

The comparing operators are

=itemize
=item = equality operator
=item <> non equality operator
=item < less than
=item > greather than
=item <= less than or equal
=item >= greather than or equal
=noitemize

When comparing T<undef> values the following statements should be taken into account:

=itemize
=item T<undef> is equal to T<undef>
=item T<undef> is not equal anything else than T<undef>
=item T<undef> is comparable with anything only for equality or non equality. Any other comparision
      having an operand T<undef> results an undefined value.
=noitemize


CUT*/

#define NOCOMMAND(XXX) \
COMMAND(XXX)\
NOTIMPLEMENTED;\
END

#if NOTIMP_LT
NOCOMMAND(LT)
#else
LOGOP(LT,<)
#endif

#if NOTIMP_LE
NOCOMMAND(LE)
#else
LOGOP(LE,<=)
#endif


#if NOTIMP_GT
NOCOMMAND(GT)
#else
LOGOP(GT,>)
#endif

#if NOTIMP_GE
NOCOMMAND(GE)
#else
LOGOP(GE,>=)
#endif

#define LONGOP(NAME,OP) \
COMMAND(NAME)\
  NODE nItem;\
  VARIABLE Op1,Op2;\
\
  /* this is an operator and not a command, therefore we do not have our own mortal list */\
  USE_CALLER_MORTALS;\
\
  /* evaluate the parameters */\
  nItem = PARAMETERLIST;\
  Op1 = EVALUATEEXPRESSION(CAR(nItem));\
  ASSERTOKE;\
  if( Op1 == NULL ){\
    RESULT = NULL;\
    RETURN;\
    }\
  nItem = CDR(nItem);\
  Op2 = EVALUATEEXPRESSION(CAR(nItem));\
  ASSERTOKE;\
  if( Op2 == NULL ){\
    RESULT = NULL;\
    RETURN;\
    }\
\
  RESULT = NEWMORTALLONG;\
  Op1 = CONVERT2LONG(Op1);\
  Op2 = CONVERT2LONG(Op2);\
  LONGVALUE(RESULT) = LONGVALUE(Op1) OP LONGVALUE(Op2);\
\
END

/*POD
=section longoperators
=H Long operators

These operators are defined only for long arguments, and they result long value.
If any of their argument is T<undef> the result is T<undef>. The operators are

=itemize
=item T<and> bitwise and
=item T<or> bitwise or
=item T<xor> bitwise xor
=noitemize

Note that the logical operators can be used to evaluate logical expressions as
well as bitwise expressions, because logical TRUE value is -1L which means all
bits set to 1. In commands that take a logical value any nonzero value is true.

CUT*/

#if NOTIMP_AND
NOCOMMAND(AND)
#else
LONGOP(AND,&)
#endif

#if NOTIMP_OR
NOCOMMAND(OR)
#else
LONGOP(OR,|)
#endif

#if NOTIMP_XOR
NOCOMMAND(XOR)
#else
LONGOP(XOR,^)
#endif

/*POD
=section mod
=H Modulo operators

This operator calculates the modulo of two numbers.
CUT*/
COMMAND(MOD)
#if NOTIMP_MOD
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1,Op2;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }
  nItem = CDR(nItem);
  Op2 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op2 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  Op1 = CONVERT2LONG(Op1);
  Op2 = CONVERT2LONG(Op2);
  if( LONGVALUE(Op2) == 0 ){
    RESULT = NULL;
    RETURN;
    }
  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = LONGVALUE(Op1) % LONGVALUE(Op2);

#endif
END

/*POD
=section plusminus
=H unary and binary plus and minus

These functions implement the unary and binary plus and minus operands.

If any of the arguments is T<undef> then the result is T<undef>.

If any of the arguments is double or a string evaluating to a float value
then the result is double.

The result is long or double, never string.

CUT*/
COMMAND(PLUS)
#if NOTIMP_PLUS
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1,Op2;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }
  nItem = CDR(nItem);
  if( nItem ){
    Op2 = EVALUATEEXPRESSION(CAR(nItem));
    ASSERTOKE;
    if( Op2 == NULL ){
      RESULT = NULL;
      RETURN;
      }

    /* if any of the arguments is double then the result is double */
    if( (TYPE(Op1) == VTYPE_DOUBLE || TYPE(Op2) == VTYPE_DOUBLE) ||

       /* if the first argument is string and is NOT integer */
       ( TYPE(Op1) == VTYPE_STRING && !ISSTRINGINTEGER(Op1)) ||

       /* if the second argument is string and is NOT integer */
       ( TYPE(Op2) == VTYPE_STRING && !ISSTRINGINTEGER(Op2))
       ){
      RESULT = NEWMORTALDOUBLE;
      Op1 = CONVERT2DOUBLE(Op1);
      Op2 = CONVERT2DOUBLE(Op2);
      DOUBLEVALUE(RESULT) = DOUBLEVALUE(Op1) + DOUBLEVALUE(Op2);
      RETURN;
      }

    RESULT = NEWMORTALLONG;
    Op1 = CONVERT2LONG(Op1);
    Op2 = CONVERT2LONG(Op2);
    LONGVALUE(RESULT) = LONGVALUE(Op1) + LONGVALUE(Op2);
    RETURN;
    }

  /* this is unary */
  if( TYPE(Op1) == VTYPE_DOUBLE ||
     /* if the first argument is string and is NOT integer */
     ( TYPE(Op1) == VTYPE_STRING && !ISSTRINGINTEGER(Op1)) ){
    RESULT = NEWMORTALDOUBLE;
    Op1 = CONVERT2DOUBLE(Op1);
    DOUBLEVALUE(RESULT) = + DOUBLEVALUE(Op1);
    RETURN;
    }
  RESULT = NEWMORTALLONG;
  Op1 = CONVERT2LONG(Op1);
  LONGVALUE(RESULT) = + LONGVALUE(Op1);

#endif
END

COMMAND(MINUS)
#if NOTIMP_MINUS
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1,Op2;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }
  nItem = CDR(nItem);
  if( nItem ){
    Op2 = EVALUATEEXPRESSION(CAR(nItem));
    ASSERTOKE;
    if( Op2 == NULL ){
      RESULT = NULL;
      RETURN;
      }

    /* if any of the arguments is double then the result is double */
    if( (TYPE(Op1) == VTYPE_DOUBLE || TYPE(Op2) == VTYPE_DOUBLE) ||

       /* if the first argument is string and is NOT integer */
       ( TYPE(Op1) == VTYPE_STRING && !ISSTRINGINTEGER(Op1)) ||

       /* if the second argument is string and is NOT integer */
       ( TYPE(Op2) == VTYPE_STRING && !ISSTRINGINTEGER(Op2))
       ){
      RESULT = NEWMORTALDOUBLE;
      Op1 = CONVERT2DOUBLE(Op1);
      Op2 = CONVERT2DOUBLE(Op2);
      DOUBLEVALUE(RESULT) = DOUBLEVALUE(Op1) - DOUBLEVALUE(Op2);
      RETURN;
      }

    RESULT = NEWMORTALLONG;
    Op1 = CONVERT2LONG(Op1);
    Op2 = CONVERT2LONG(Op2);
    LONGVALUE(RESULT) = LONGVALUE(Op1) - LONGVALUE(Op2);
    RETURN;
    }

  /* this is unary */
  if( TYPE(Op1) == VTYPE_DOUBLE ||
     /* if the first argument is string and is NOT integer */
     ( TYPE(Op1) == VTYPE_STRING && !ISSTRINGINTEGER(Op1)) ){
    RESULT = NEWMORTALDOUBLE;
    Op1 = CONVERT2DOUBLE(Op1);
    DOUBLEVALUE(RESULT) = - DOUBLEVALUE(Op1);
    RETURN;
    }
  RESULT = NEWMORTALLONG;
  Op1 = CONVERT2LONG(Op1);
  LONGVALUE(RESULT) = - LONGVALUE(Op1);

#endif
END

/*POD
=section NOT
=H unary NOT operator

This operator takes one argument converts it to long and inverts all bits.
If the argument is T<undef> the result is -1L which is the absolute TRUE value,
havinbg all bits set.

CUT*/
COMMAND(NOT)
#if NOTIMP_NOT
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;

  if( Op1 == NULL ){
    RESULT = NEWMORTALLONG;
    LONGVALUE(RESULT) = -1L; /* NOT undef is true */
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  Op1 = CONVERT2LONG(Op1);
  LONGVALUE(RESULT) = ~ LONGVALUE(Op1);

#endif
END

/*POD
=section POWER
=H powering operator

This is a binary operator that calculates I<x> powered to I<y>.

If any of the arguments is T<undef> then the result is also T<undef>.

If the exponent is negative the result is double.

If any of the operators is double or is a string evaluating to a non-integer
value then the result is double.

Otherwise the operator makes integer operations calculating the power value.
CUT*/
COMMAND(POWER)
#if NOTIMP_POWER
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1,Op2;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  nItem = CDR(nItem);
  Op2 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op2 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  if( TYPE(Op2) == VTYPE_LONG && LONGVALUE(Op2) < 0 )
    Op2 = CONVERT2DOUBLE(Op2);

  /* if any of the arguments is double then the result is double */
  if( (TYPE(Op1) == VTYPE_DOUBLE || TYPE(Op2) == VTYPE_DOUBLE) ||

     /* if the first argument is string and is NOT integer */
     ( TYPE(Op1) == VTYPE_STRING && !ISSTRINGINTEGER(Op1)) ||

     /* if the second argument is string and is NOT integer */
     ( TYPE(Op2) == VTYPE_STRING && !ISSTRINGINTEGER(Op2))
     ){
    Op1 = CONVERT2DOUBLE(Op1);
    if( DOUBLEVALUE(Op1) < 0.0 ){
      RESULT = NULL;
      RETURN;
      }
    RESULT = NEWMORTALDOUBLE;
    Op2 = CONVERT2DOUBLE(Op2);
    DOUBLEVALUE(RESULT) = pow(DOUBLEVALUE(Op1),DOUBLEVALUE(Op2));
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  Op1 = CONVERT2LONG(Op1);
  Op2 = CONVERT2LONG(Op2);
  LONGVALUE(RESULT) = longpow(LONGVALUE(Op1),LONGVALUE(Op2));

#endif
END

/*POD
=section IDIV
=H Integer division

This operator converts the arguments to long and divides the first argument with the second.
The result is a truncated long. The truncation is done towards zero like it is done by the function
FIX and B<unlike> by the function INT.

CUT*/
COMMAND(IDIV)
#if NOTIMP_IDIV
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1,Op2;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  nItem = CDR(nItem);
  Op2 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op2 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  /* if any of the arguments is double then the result is double */
  if( (TYPE(Op1) == VTYPE_DOUBLE || TYPE(Op2) == VTYPE_DOUBLE) ||

     /* if the first argument is string and is NOT integer */
     ( TYPE(Op1) == VTYPE_STRING && !ISSTRINGINTEGER(Op1)) ||

     /* if the second argument is string and is NOT integer */
     ( TYPE(Op2) == VTYPE_STRING && !ISSTRINGINTEGER(Op2))
     ){
    Op1 = CONVERT2DOUBLE(Op1);
    Op2 = CONVERT2DOUBLE(Op2);
    if( DOUBLEVALUE(Op2) == 0.0 ){
      RESULT = NULL;
      RETURN;
      }
    RESULT = NEWMORTALDOUBLE;
    DOUBLEVALUE(RESULT) = DOUBLEVALUE(Op1) / DOUBLEVALUE(Op2);
    RETURN;
    }

  Op1 = CONVERT2LONG(Op1);
  Op2 = CONVERT2LONG(Op2);
  if( LONGVALUE(Op2) == 0 ){
    RESULT = NULL;
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = LONGVALUE(Op1) / LONGVALUE(Op2);

#endif
END

/*POD
=section DIV
=H Division

This operator divides two numbers. If some of the arguments are strings then they are
converted to double or long. The result is double unless both operands are long and
the operation can be performed to result an integer value without truncation.
CUT*/
COMMAND(DIV)
#if NOTIMP_DIV
NOTIMPLEMENTED;
#else


  NODE nItem;
  VARIABLE Op1,Op2;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  nItem = PARAMETERLIST;
  Op1 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op1 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  nItem = CDR(nItem);
  Op2 = EVALUATEEXPRESSION(CAR(nItem));
  ASSERTOKE;
  if( Op2 == NULL ){
    RESULT = NULL;
    RETURN;
    }

  /* if any of the arguments is double then the result is double */
  if( (TYPE(Op1) == VTYPE_DOUBLE || TYPE(Op2) == VTYPE_DOUBLE) ||

     /* if the first argument is string and is NOT integer */
     ( TYPE(Op1) == VTYPE_STRING && !ISSTRINGINTEGER(Op1)) ||

     /* if the second argument is string and is NOT integer */
     ( TYPE(Op2) == VTYPE_STRING && !ISSTRINGINTEGER(Op2))
     ){
    Op1 = CONVERT2DOUBLE(Op1);
    Op2 = CONVERT2DOUBLE(Op2);
    if( DOUBLEVALUE(Op2) == 0.0 ){
      RESULT = NULL;
      RETURN;
      }
    RESULT = NEWMORTALDOUBLE;
    DOUBLEVALUE(RESULT) = DOUBLEVALUE(Op1) / DOUBLEVALUE(Op2);
    RETURN;
    }

  Op1 = CONVERT2LONG(Op1);
  Op2 = CONVERT2LONG(Op2);
  if( LONGVALUE(Op2) == 0 ){
    RESULT = NULL;
    RETURN;
    }
  if( LONGVALUE(Op1) % LONGVALUE(Op2) ){
    RESULT = NEWMORTALDOUBLE;
    DOUBLEVALUE(RESULT) = ((double)LONGVALUE(Op1)) / ((double)LONGVALUE(Op2));
    RETURN;
    }

  RESULT = NEWMORTALLONG;
  LONGVALUE(RESULT) = LONGVALUE(Op1) / LONGVALUE(Op2);

#endif
END

/*POD
=section BYVAL
=H unary ByVal operator

This operator does nothing. Does it? It can be used to alter pass by reference
variables and help the caller to pass a variable by value. Istead of writing

=verbatim
call sub(a)
=noverbatim

it can do

=verbatim
call sub(ByVal a)
=noverbatim

and this way the subroutine can NOT alter the value of the variable T<a>.

CUT*/
COMMAND(BYVAL)
#if NOTIMP_BYVAL
NOTIMPLEMENTED;
#else


  VARIABLE Op1;

  /* this is an operator and not a command, therefore we do not have our own mortal list */
  USE_CALLER_MORTALS;

  /* evaluate the parameters */
  Op1 = EVALUATEEXPRESSION(CAR(PARAMETERLIST));
  ASSERTOKE;

  RESULT = Op1;

#endif
END
