/* while.c
--GNU LGPL
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdlib.h>
#include <stdio.h>

#include "../command.h"
/*POD
=H Implement the different looping constructs

This file implements the various looping constructs. These include the followings:

=verbatim
while expression
wend

repeat
until expression

do while expression
loop

do until expression
loop

do
loop while expression

do
loop until expression

for i=expression to expression [step expression]
next
=noverbatim

The different formats result the same, they are presented here to allow
the user to use the one that fits his or her taste.

The implementation of the 'for' loop is extremely sophisticated and tricky. Try to
understand when you are really familiar with the code structure.

CUT*/

/**WHILE
=section loop
=title WHILE condition

Implements the while loop as it is usually done in most basic implementations.

=verbatim
while expression
  ...
  commands to repeat
  ...
wend
=noverbatim

This command implements a looping construct checking the condition at start and 
does loop when the condition is T<TRUE>.
*/
COMMAND(WHILE)
#if NOTIMP_WHILE
NOTIMPLEMENTED;
#else



  NODE nItem;
  NODE nGoForward;
  VARIABLE ItemResult;

  nItem = PARAMETERNODE;
  ItemResult = EVALUATEEXPRESSION(nItem);
  ASSERTOKE;
  NEXTPARAMETER;
  /* we step forward to the node AFTER the while statement */
  nGoForward = CDR(PARAMETERNODE);
  if( ItemResult == NULL ){
    SETPROGRAMCOUNTER(nGoForward);
    RETURN;
    }

   switch( TYPE(ItemResult) ){
      case VTYPE_LONG:
        if( ! LONGVALUE(ItemResult) ){
          SETPROGRAMCOUNTER(nGoForward);
          RETURN;
          }
        break;
      case VTYPE_DOUBLE:
        if( ! DOUBLEVALUE(ItemResult) ){
          SETPROGRAMCOUNTER(nGoForward);
          RETURN;
          }
        break;
      case VTYPE_STRING:
        if( * STRINGVALUE(ItemResult) == (char)0 ){
          SETPROGRAMCOUNTER(nGoForward);
          RETURN;
          }
        break;
      case VTYPE_ARRAY:
        break;
      }

#endif
END

/**DOUNTIL
=section loop
=title DO UNTIL condition

Implements front checking looping construct repeating so long as long the condition is T<FALSE>
=verbatim
do until expression
 ...
 commands to repeat
 ...
loop
=noverbatim
*/
COMMAND(DOUNTIL)
#if NOTIMP_DOUNTIL
NOTIMPLEMENTED;
#else


  NODE nItem;
  NODE nGoForward;
  VARIABLE ItemResult;

  nItem = PARAMETERNODE;
  ItemResult = EVALUATEEXPRESSION(nItem);
  ASSERTOKE;
  NEXTPARAMETER;

  /* we step forward to the node AFTER the loop statement */
  nGoForward = CDR(PARAMETERNODE);
  if( ItemResult == NULL ){
    RETURN;
    }

   switch( TYPE(ItemResult) ){
      case VTYPE_LONG:
        if( LONGVALUE(ItemResult) ){
          SETPROGRAMCOUNTER(nGoForward);
          RETURN;
          }
        break;
      case VTYPE_DOUBLE:
        if( DOUBLEVALUE(ItemResult) ){
          SETPROGRAMCOUNTER(nGoForward);
          RETURN;
          }
        break;
      case VTYPE_STRING:
        if( * STRINGVALUE(ItemResult) != (char)0 ){
          SETPROGRAMCOUNTER(nGoForward);
          RETURN;
          }
        break;
      case VTYPE_ARRAY:
        break;
      }

#endif
END

COMMAND(WEND)
#if NOTIMP_WEND
NOTIMPLEMENTED;
#else


  SETPROGRAMCOUNTER(PARAMETERNODE);

#endif
END

/**DOWHILE
=section loop
=title DO WHILE condition

Same as R<WHILE> with a different syntax to be compatible with different BASIC implementations.

=verbatim
do while
 ...
loop
=noverbatim
*/
COMMAND(DOWHILE)
#if NOTIMP_DOWHILE
NOTIMPLEMENTED;
#else
 IDENTICAL_COMMAND(WHILE)
#endif
END

COMMAND(LOOP)
#if NOTIMP_LOOP
NOTIMPLEMENTED;
#else
 IDENTICAL_COMMAND(WEND)
#endif
END

/**REPEAT
=section loop

Implements a tail checking loop that repeats the commands so long as long the condition is false.

=verbatim
repeat
  ...
  commands to repeat
  ...
until expression
=noverbatim
*/
COMMAND(REPEAT)
#if NOTIMP_REPEAT
NOTIMPLEMENTED;
#else

 /* we do not do anything just start a loop */
#endif
END

/**DO
=section loop

Implements a tail checking loop that repeats commands so long as long the condition is
either T<TRUE> (T<do/while>) or T<FALSE> (T<do/until>)

=verbatim
do
 ...
 commands to repeat
 ...
loop while expression
=noverbatim

or

=verbatim
do
 ...
 commands to repeat
 ...
loop until expression
=noverbatim
*/
COMMAND(DO)
#if NOTIMP_DO
NOTIMPLEMENTED;
#else

 /* we do not do anything just start a loop */
#endif
END

COMMAND(LOOPWHILE)
#if NOTIMP_LOOPWHILE
NOTIMPLEMENTED;
#else
  IDENTICAL_COMMAND(DOUNTIL)
#endif
END

COMMAND(UNTIL)
#if NOTIMP_UNTIL
NOTIMPLEMENTED;
#else
  IDENTICAL_COMMAND(WHILE)
#endif
END

COMMAND(LOOPUNTIL)
#if NOTIMP_LOOPUNTIL
NOTIMPLEMENTED;
#else
  IDENTICAL_COMMAND(WHILE)
#endif
END

/**FOR
=section loop
=title FOR var=exp_start TO exp_stop [ STEP exp_step ]

Implements a FOR loop. The variable T<var> gets the value of the start expression T<exp_start>,
and after each execution of the loop body it is incremented ir decremented by the value 
T<exp_step> until it reaches the stop value T<exp_stop>.

=verbatim
FOR var=expression TO expression [ STEP expression ]
 ...
 commands to repeat
 ...
next
=noverbatim

=details
B<THIS IS DETAILS FOR C PROGRAMMERS> We discuss here the specialities how FOR is implemented
in the syntax analyzer!

This command implements a for loop. The for loop is implemented with a slight trick. The
syntax definition system of scriba does not allo optional parts of a command, like the STEP
part of the for command. In such a case two different command should be defined, like

=verbatim
FOR: 'for' lval '=' expression 'to' expression nl
FORS: 'for' lval '=' expression 'to' expression 'step' expression nl
=noverbatim

in the T<syntax.def> file. B<This does not work!>

The reason is that an T<expression> is a terminal symbol which can not be parsed without side effects.
Therefore the syntax of a line should be designed in a way which can assure that the line contains the
very command by the time the parser starts to evaluate an expression.

Assume that we use the syntax definition above and scriba tries to analyze a line

=verbatim
for i=1 to 100 step 2
=noverbatim

The syntax analyzer analized the left value before the '=' sign, two expressions when it realizes that
the command does not fit the current syntax definition line, because a predefined symbol (step) is coming
instead of the new line. At this stage the syntax analyzer drops the line and treats it syntactically
incorrect, because the left value analysis and the expression analysis generated a lot of code which, 
in the current of scriba can not be undone.

I<What do we do then to implement a FOR loop?>

A FOR loop head is defined as three consecutive commands:

=itemize
=item a T<FOR> command
=item a T<FORTO> command
=item and a T<STEP> command
=noitemize
(And yes, we have a NEXT command as well, but that is implemented normal.)

The syntax definition lines in T<syntax.def> are:
=verbatim
FOR:     'for' lval '=' expression come_back(FORTO)
FORTO:   'to' expression go_back(FORTO) go_forward(FOR) come_back(FOR)
FORSTEP: 'step' expression nl
=noverbatim

This means that a

=verbatim
for i=1 to 100
print
=noverbatim

code fragment is analyzed as

=verbatim
for i=1
=noverbatim

a for command,

=verbatim
        to 100
=noverbatim

a to command. Then a single new line character comes, which is analyzed as an EMPTY command (see the syntax.def file!). And finally a
print command.

This also generates two or three commands in case you do or do not have a 'step' part. This helps implementing the initialization
code, the increment code into two different command and therefore it is easy to distinguish between starting the loop and
jumping from the end to the start. (Note that the other looping construct do not have initialization code inside them,
and therefore they do not care if the execution is the first one entering the loop or we just jumped back to the loop head from
the end.)

When we jump back from the command T<NEXT> we always jump back to the command T<FORTO>.

The sorry side of this solution is that one can write the following or similar code:

=verbatim
print "hello"
for i=10
print "here I am"
=noverbatim

or

=verbatim
print "hello"
to 63
print "here I am"
next
=noverbatim

without getting syntax error. The error occures when the execution starts.

Let us hope two things:

=itemize
=item People doing nasty things with scriba using this "feature" won't exist.
=item Unintentional syntax erorrs uncovered by this "feature" are very rare.
=noitemize

For more details see the code.
*/
COMMAND(FOR)
#if NOTIMP_FOR
NOTIMPLEMENTED;
#else


  VARIABLE StartExpressionResult,EndExpressionResult,StepExpressionResult;
  LEFTVALUE LetThisVariable;
  NODE nToStatement;
  NODE nStepStatement;
  NODE nLoopStart,nLoopEnd;
  int iGoingUp; /* are we stepping up or down */
  long refcount;

  /* there should be a to statement following the for statement */
  nToStatement = CDR(pEo->ProgramCounter);

  if( nToStatement == 0 ){
    /* If this is the last statement then we can stop. */
    SETPROGRAMCOUNTER(0);
    RETURN;
    }

  nStepStatement = CDR(nToStatement);
  nToStatement = CAR(nToStatement);

  if( nStepStatement ){
    if( pEo->CommandArray[CAR(nStepStatement)-1].OpCode == CMD_FORSTEP ){
      nLoopStart = CDR(nStepStatement);
      nStepStatement = CAR(nStepStatement);
      }else{
      nLoopStart = nStepStatement;
      nStepStatement = 0; /* there is no step statement */
      }
    }else{
    nLoopStart = 0;
    }

  if( OPCODE(nToStatement) != CMD_FORTO ){
    pEo->fStop = fStopSTOP; /* at least at runtime we recognize the mistake */
    RETURN;
    }

  LetThisVariable = EVALUATELEFTVALUE(PARAMETERNODE);
  ASSERTOKE;
  DEREFERENCE(LetThisVariable)

  NEXTPARAMETER;
  StartExpressionResult = EVALUATEEXPRESSION(PARAMETERNODE);
  ASSERTOKE;
  if( StartExpressionResult )IMMORTALIZE(StartExpressionResult);
  if( TYPE(StartExpressionResult) == VTYPE_STRING ){
        if( ISSTRINGINTEGER(StartExpressionResult) )
          StartExpressionResult = CONVERT2LONG(StartExpressionResult);
        else
          StartExpressionResult = CONVERT2DOUBLE(StartExpressionResult);
    }

  if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);

  *LetThisVariable = StartExpressionResult;

  EndExpressionResult = 
    EVALUATEEXPRESSION(pEo->CommandArray[nToStatement-1].Parameter.CommandArgument.Argument.pNode);
  ASSERTOKE;
  if( TYPE(EndExpressionResult) == VTYPE_STRING ){
        if( ISSTRINGINTEGER(EndExpressionResult) )
          EndExpressionResult = CONVERT2LONG(EndExpressionResult);
        else
          EndExpressionResult = CONVERT2DOUBLE(EndExpressionResult);
    }
  nLoopEnd = pEo->CommandArray[nToStatement-1].Parameter.CommandArgument.next;/* come_back(FORTO) */
  nLoopEnd = pEo->CommandArray[nLoopEnd    -1].Parameter.CommandArgument.next;
  nLoopEnd = pEo->CommandArray[nLoopEnd-1].Parameter.CommandArgument.Argument.pNode;
  if( nLoopEnd )nLoopEnd = CDR(nLoopEnd);

  if( nStepStatement && OPCODE(nStepStatement) == CMD_FORSTEP ){
    StepExpressionResult = EVALUATEEXPRESSION(pEo->CommandArray[nStepStatement-1].Parameter.CommandArgument.Argument.pNode);
    ASSERTOKE;
    if( StepExpressionResult ){
      if( TYPE(StepExpressionResult) == VTYPE_LONG ){
        iGoingUp = 0 < LONGVALUE(StepExpressionResult);
        }else
      if( TYPE(StepExpressionResult) == VTYPE_DOUBLE ){
        iGoingUp = 0 < DOUBLEVALUE(StepExpressionResult);
        }else
      if( TYPE(StepExpressionResult) == VTYPE_STRING ){
        if( ISSTRINGINTEGER(StepExpressionResult) ){
          iGoingUp = 0 < LONGVALUE(CONVERT2LONG(StepExpressionResult));
          }else{
          iGoingUp = 0 < DOUBLEVALUE(CONVERT2DOUBLE(StepExpressionResult));
          }
        }
      }else
      iGoingUp = 1; /* if step value is undef then we go up */
    }else{
    iGoingUp = 1; /* by default, if there is no step statement we are going upward */
    }

  if( StartExpressionResult && TYPE(StartExpressionResult) == VTYPE_LONG && 
      EndExpressionResult   && TYPE(EndExpressionResult) == VTYPE_LONG ){
    if( iGoingUp ? LONGVALUE(StartExpressionResult) <= LONGVALUE(EndExpressionResult)
                 : LONGVALUE(StartExpressionResult) >= LONGVALUE(EndExpressionResult) ){
       /* We have to start the loop */
       SETPROGRAMCOUNTER(nLoopStart);
       }else{
       /* We do not even step into the loop */
       SETPROGRAMCOUNTER(nLoopEnd);
       }
    RETURN;
    }

  if( StartExpressionResult && TYPE(StartExpressionResult) == VTYPE_LONG && 
      EndExpressionResult   && TYPE(EndExpressionResult) == VTYPE_DOUBLE ){
    if( iGoingUp ? ((double)LONGVALUE(StartExpressionResult)) <= DOUBLEVALUE(EndExpressionResult)
                 : ((double)LONGVALUE(StartExpressionResult)) >= DOUBLEVALUE(EndExpressionResult) ){
       /* We have to start the loop */
       SETPROGRAMCOUNTER(nLoopStart);
       }else{
       /* We do not even step into the loop */
       SETPROGRAMCOUNTER(nLoopEnd);
       }
    RETURN;
    }

  if( StartExpressionResult && TYPE(StartExpressionResult) == VTYPE_DOUBLE && 
      EndExpressionResult   && TYPE(EndExpressionResult) == VTYPE_DOUBLE ){
    if( iGoingUp ? DOUBLEVALUE(StartExpressionResult) <= DOUBLEVALUE(EndExpressionResult)
                 : DOUBLEVALUE(StartExpressionResult) >= DOUBLEVALUE(EndExpressionResult) ){
       /* We have to start the loop */
       SETPROGRAMCOUNTER(nLoopStart);
       }else{
       /* We do not even step into the loop */
       SETPROGRAMCOUNTER(nLoopEnd);
       }
    RETURN;
    }

  if( StartExpressionResult && TYPE(StartExpressionResult) == VTYPE_DOUBLE && 
      EndExpressionResult   && TYPE(EndExpressionResult) == VTYPE_LONG ){
    if( iGoingUp ? DOUBLEVALUE(StartExpressionResult) <= (double)LONGVALUE(EndExpressionResult)
                 : DOUBLEVALUE(StartExpressionResult) >= (double)LONGVALUE(EndExpressionResult) ){
       /* We have to start the loop */
       SETPROGRAMCOUNTER(nLoopStart);
       }else{
       /* We do not even step into the loop */
       SETPROGRAMCOUNTER(nLoopEnd);
       }
    RETURN;
    }
  SETPROGRAMCOUNTER(nLoopEnd);

#endif
END

COMMAND(FORTO)
#if NOTIMP_FORTO
NOTIMPLEMENTED;
#else


  LEFTVALUE LetThisVariable;
  VARIABLE EndExpressionResult;
  VARIABLE StepExpressionResult;
  VARIABLE NewLoopValue;
  NODE nStepStatement;
  NODE nForStatement;
  NODE nLoopStart,nLoopEnd;
  int iGoingUp;
  long refcount;

  EndExpressionResult = EVALUATEEXPRESSION(PARAMETERNODE);
  ASSERTOKE;
  NEXTPARAMETER;
  nForStatement = CAR(PARAMETERNODE);
  NEXTPARAMETER;
  nLoopEnd = CDR(PARAMETERNODE);

  LetThisVariable = EVALUATELEFTVALUE( pEo->CommandArray[nForStatement-1].Parameter.CommandArgument.Argument.pNode );
  ASSERTOKE;
  DEREFERENCE(LetThisVariable)

  nStepStatement = CDR(pEo->ProgramCounter);

  if( nStepStatement ){
    if( pEo->CommandArray[CAR(nStepStatement)-1].OpCode == CMD_FORSTEP ){
      nLoopStart = CDR(nStepStatement);
      nStepStatement = CAR(nStepStatement);
      }else{
      nLoopStart = nStepStatement;
      nStepStatement = 0; /* there is no step statement */
      }
    }else{
    nLoopStart = 0;
    }

  if( nStepStatement && OPCODE(nStepStatement) == CMD_FORSTEP ){
    StepExpressionResult = EVALUATEEXPRESSION(pEo->CommandArray[nStepStatement-1].Parameter.CommandArgument.Argument.pNode);
    ASSERTOKE;
    if( StepExpressionResult ){
      if( TYPE(StepExpressionResult) == VTYPE_LONG ){
        iGoingUp = 0 < LONGVALUE(StepExpressionResult);
        if( TYPE( (*LetThisVariable) ) == VTYPE_LONG ){
          NewLoopValue = NEWMORTALLONG;
          LONGVALUE(NewLoopValue) = LONGVALUE( (*LetThisVariable) ) + LONGVALUE(StepExpressionResult);
          if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
          if( NewLoopValue ){
            IMMORTALIZE(NewLoopValue);
            }
          *LetThisVariable = NewLoopValue;
          }else{
          NewLoopValue = NEWMORTALDOUBLE;
          DOUBLEVALUE(NewLoopValue) = DOUBLEVALUE( (*LetThisVariable) ) + (double)LONGVALUE(StepExpressionResult);
          if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
          if( NewLoopValue ){
            IMMORTALIZE(NewLoopValue);
            }
          *LetThisVariable = NewLoopValue;
          }
        }else
      if( TYPE(StepExpressionResult) == VTYPE_DOUBLE ){
        iGoingUp = 0 < DOUBLEVALUE(StepExpressionResult);
        if( TYPE( *LetThisVariable ) == VTYPE_LONG ){
          NewLoopValue = NEWMORTALDOUBLE;
          DOUBLEVALUE(NewLoopValue) = ((double)LONGVALUE(*LetThisVariable) ) + DOUBLEVALUE(StepExpressionResult);
          if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
          if( NewLoopValue ){
            IMMORTALIZE(NewLoopValue);
            }
          *LetThisVariable = NewLoopValue;
          }else{
          NewLoopValue = NEWMORTALDOUBLE;
          DOUBLEVALUE(NewLoopValue) = DOUBLEVALUE( (*LetThisVariable) ) + DOUBLEVALUE(StepExpressionResult);
          if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
          if( NewLoopValue ){
            IMMORTALIZE(NewLoopValue);
            }
          *LetThisVariable = NewLoopValue;
          }
        }else
      if( TYPE(StepExpressionResult) == VTYPE_STRING ){
        if( ISSTRINGINTEGER(StepExpressionResult) ){
          iGoingUp = 0 < LONGVALUE(StepExpressionResult=CONVERT2LONG(StepExpressionResult));
          if( TYPE( *LetThisVariable ) == VTYPE_LONG ){
            NewLoopValue = NEWMORTALLONG;
            LONGVALUE(NewLoopValue) = LONGVALUE( *LetThisVariable ) + LONGVALUE(StepExpressionResult);
            if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
            if( NewLoopValue ){
              IMMORTALIZE(NewLoopValue);
              }
            *LetThisVariable = NewLoopValue;
            }else{
            NewLoopValue = NEWMORTALDOUBLE;
            DOUBLEVALUE(NewLoopValue) = DOUBLEVALUE( (*LetThisVariable) ) + (double)LONGVALUE(StepExpressionResult);
            if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
            IMMORTALIZE(NewLoopValue);
            *LetThisVariable = NewLoopValue;
            }
          }else{
          iGoingUp = 0 < DOUBLEVALUE(StepExpressionResult=CONVERT2DOUBLE(StepExpressionResult));
          if( TYPE( *LetThisVariable ) == VTYPE_LONG ){
            NewLoopValue = NEWMORTALLONG;
            LONGVALUE(NewLoopValue) = LONGVALUE( *LetThisVariable) + (long)DOUBLEVALUE(StepExpressionResult);
            if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
            IMMORTALIZE(NewLoopValue);
            *LetThisVariable = NewLoopValue;
            }else{
            NewLoopValue = NEWMORTALDOUBLE;
            DOUBLEVALUE(NewLoopValue) = DOUBLEVALUE( *LetThisVariable) + DOUBLEVALUE(StepExpressionResult);
            if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
            IMMORTALIZE(NewLoopValue);
            *LetThisVariable = NewLoopValue;
            }
          }
        }
      }else{
      iGoingUp = 1; /* if step value is undef then we go up */
      if( TYPE( *LetThisVariable ) == VTYPE_LONG ){
        NewLoopValue = NEWMORTALLONG;
        LONGVALUE(NewLoopValue) = LONGVALUE( *LetThisVariable) + 1L;
        if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
        IMMORTALIZE(NewLoopValue);
        *LetThisVariable = NewLoopValue;
      }else{
        NewLoopValue = NEWMORTALDOUBLE;
        DOUBLEVALUE(NewLoopValue) = DOUBLEVALUE( *LetThisVariable) + (double)1.0;
        if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
        IMMORTALIZE(NewLoopValue);
        *LetThisVariable = NewLoopValue;
        }
      }
    }else{
    iGoingUp = 1; /* by default, if there is no step statement we are going upward */
    if( (!*LetThisVariable) || TYPE( *LetThisVariable ) == VTYPE_LONG ){
      NewLoopValue = NEWMORTALLONG;
      if( *LetThisVariable )
        LONGVALUE(NewLoopValue) = LONGVALUE( *LetThisVariable) + 1L;
      else
        LONGVALUE(NewLoopValue) = 1L;
      if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
      IMMORTALIZE(NewLoopValue);
      *LetThisVariable = NewLoopValue;
    }else{
      NewLoopValue = NEWMORTALDOUBLE;
      DOUBLEVALUE(NewLoopValue) = DOUBLEVALUE( *LetThisVariable) + (double)1.0;
      if( *LetThisVariable )memory_ReleaseVariable(pEo->pMo,*LetThisVariable);
      IMMORTALIZE(NewLoopValue);
      *LetThisVariable = NewLoopValue;
      }
    }

  if( *LetThisVariable && TYPE(*LetThisVariable) == VTYPE_LONG &&
      EndExpressionResult && TYPE(EndExpressionResult) == VTYPE_LONG ){
    if( iGoingUp ? LONGVALUE(*LetThisVariable) <= LONGVALUE(EndExpressionResult)
                 : LONGVALUE(*LetThisVariable) >= LONGVALUE(EndExpressionResult) ){
       /* We have to start the loop */
       SETPROGRAMCOUNTER(nLoopStart);
       }else{
       /* We do not even step into the loop */
       SETPROGRAMCOUNTER(nLoopEnd);
       }
    RETURN;
    }

  if( *LetThisVariable && TYPE(*LetThisVariable) == VTYPE_LONG &&
      EndExpressionResult &&  TYPE(EndExpressionResult) == VTYPE_DOUBLE ){
    if( iGoingUp ? ((double)LONGVALUE(*LetThisVariable)) <= DOUBLEVALUE(EndExpressionResult)
                 : ((double)LONGVALUE(*LetThisVariable)) >= DOUBLEVALUE(EndExpressionResult) ){
       /* We have to start the loop */
       SETPROGRAMCOUNTER(nLoopStart);
       }else{
       /* We do not even step into the loop */
       SETPROGRAMCOUNTER(nLoopEnd);
       }
    RETURN;
    }

  if( *LetThisVariable && TYPE(*LetThisVariable) == VTYPE_DOUBLE &&
      EndExpressionResult &&  TYPE(EndExpressionResult) == VTYPE_DOUBLE ){
    if( iGoingUp ? DOUBLEVALUE(*LetThisVariable) <= DOUBLEVALUE(EndExpressionResult)
                 : DOUBLEVALUE(*LetThisVariable) >= DOUBLEVALUE(EndExpressionResult) ){
       /* We have to start the loop */
       SETPROGRAMCOUNTER(nLoopStart);
       }else{
       /* We do not even step into the loop */
       SETPROGRAMCOUNTER(nLoopEnd);
       }
    RETURN;
    }

  if( *LetThisVariable && TYPE(*LetThisVariable) == VTYPE_DOUBLE &&
      EndExpressionResult &&  TYPE(EndExpressionResult) == VTYPE_LONG ){
    if( iGoingUp ? DOUBLEVALUE(*LetThisVariable) <= (double)LONGVALUE(EndExpressionResult)
                 : DOUBLEVALUE(*LetThisVariable) >= (double)LONGVALUE(EndExpressionResult) ){
       /* We have to start the loop */
       SETPROGRAMCOUNTER(nLoopStart);
       }else{
       /* We do not even step into the loop */
       SETPROGRAMCOUNTER(nLoopEnd);
       }
    RETURN;
    }
  SETPROGRAMCOUNTER(nLoopEnd);

#endif
END

COMMAND(FORSTEP)
#if NOTIMP_FORSTEP
NOTIMPLEMENTED;
#else


 /* we should never ever get here */

#endif
END

COMMAND(NEXT)
#if NOTIMP_NEXT
NOTIMPLEMENTED;
#else
  IDENTICAL_COMMAND(WEND)
#endif
END

COMMAND(NEXTI)
#if NOTIMP_NEXTI
NOTIMPLEMENTED;
#else
  NEXTPARAMETER; /* we have to step over the symbol */
  SETPROGRAMCOUNTER(PARAMETERNODE);
#endif
END
