/*
FILE:   configurer.c
HEADER: configurer.h

The functions in this file handle the configuration file.

TO_HEADER:

#include "sym.h"

typedef struct _ConfigurationData {
  void *(*memory_allocating_function)(size_t, void *);
  void (*memory_releasing_function)(void *, void *);
  void *pMemorySegment;

  SymbolTable ST_ConfigData;

  } ConfigurationData, *pConfigurationData;

*/


#if _WIN32
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include "configurer.h"
#include "errcodes.h"
#include "filesys.h"

#define SYMLOOKUP(x) sym_LookupSymbol((x),p->ST_ConfigData,1,p->memory_allocating_function,p->memory_releasing_function,p->pMemorySegment)
#define ALLOC(x) p->memory_allocating_function((x),p->pMemorySegment)
#define FREE(x) p->memory_releasing_function((x),p->pMemorySegment)

/*POD
=H Read configuration information

The configuration information is different on each installation. The module files, system
include files, the extension for dynamic load libraries may be different. All this information
is stored in a text file that the program reads at startup. The format of the text file is
very simple. It is

=verbatim
key value
=noverbatim

for each line.

The configuration file name is given in the system registry on Windows NT under the key

=verbatim
HKEY_LOCAL_MACHINE\Software\ScriptBasic
=noverbatim

the string value named T<config> should specify the configuration file.

On other operating systems the environment variable T<SCRIBACONF> gives the name
of the configuration file.

The default configuration file name is T</etc/scriba/basic.conf> under UNIX and
T<WINNT\SCRIBA.INI> under Windows operating systems.

CUT*/
static int GetConfigFileName(char *pszConfigFile, unsigned int *pcbConfigFile){
#if _WIN32
#define DEFAULTCONF "scriba.ini"
#define REGKEY "Software\\ScriptBasic"
#define SVNAME "config"
#define STRING_BUFFER_LENGTH 256
  HKEY  hKey ;
  long   Ret;
  unsigned int i;
  char *s;
  CHAR   ValueName[STRING_BUFFER_LENGTH];
  DWORD  cbValueName = STRING_BUFFER_LENGTH;
  DWORD  dwType;

  char   sData[STRING_BUFFER_LENGTH];
  char   xData[STRING_BUFFER_LENGTH];
  DWORD  cbData;

  Ret = RegOpenKeyEx(HKEY_LOCAL_MACHINE,REGKEY,0,KEY_EXECUTE,&hKey);
  if( Ret == ERROR_SUCCESS ){
    for( i=0 ; 1 ; i++ ){
      cbValueName = STRING_BUFFER_LENGTH;
      cbData      = STRING_BUFFER_LENGTH;
      Ret = RegEnumValue (hKey, 
                          i,           // Value index, taken from listbox.
                          ValueName,   // Name of value.
                          &cbValueName,// Size of value name.
                          NULL,        // Reserved, dword = NULL.
                          &dwType,     // Type of data.
                          sData,       // Data buffer.
                          &cbData);    // Size of data buffer.
      if( Ret != ERROR_SUCCESS )break;

      if( dwType == REG_EXPAND_SZ ){
        ExpandEnvironmentStrings(sData,xData,cbData);
        strcpy(sData,xData);
        dwType = REG_SZ;
        }
      if( dwType == REG_MULTI_SZ )dwType = REG_SZ;

      if( dwType != REG_SZ )continue;

      if( !strcmp(ValueName,SVNAME) ){
        if( strlen(sData) > *pcbConfigFile ){
          *pcbConfigFile = strlen(sData);
          return 1;
          }
        strcpy(pszConfigFile,sData);
        return 0;
        }
      }
    }
  s = getenv("windir");
  if( s == NULL )s = getenv("systemroot");
  if( s == NULL )s = "c:\\WINDOWS";
  if( strlen(s)+strlen(DEFAULTCONF) + 1 > *pcbConfigFile ){
    *pcbConfigFile = strlen(s)+strlen(DEFAULTCONF);
    return 1;
    }
  strcpy(pszConfigFile,s);
  strcat(pszConfigFile,"\\");
  strcat(pszConfigFile,DEFAULTCONF);
  return 0;
#else
  char *s;
#define SCRIBACONF "SCRIBACONF"
#define DEFAULTCONF "/etc/scriba/basic.conf"

  s = getenv(SCRIBACONF);

  if( s == NULL ){
    if( strlen(DEFAULTCONF) > *pcbConfigFile ){
      *pcbConfigFile = strlen(DEFAULTCONF);
      return 1;
      }
    strcpy(pszConfigFile,DEFAULTCONF);
    return 0;
    }
  if( strlen(s) > *pcbConfigFile ){
    *pcbConfigFile = strlen(s);
    return 1;
    }
  strcpy(pszConfigFile,s);
  return 0;
#endif
  }

/*

This function copies a configuration value into a memory space. When there are more than one
values they are copied one after the other. The strings are zchar terminated. The last string
is terminated by a double zchar.

s gives the value to store
pStore points to the pointer that points to the list of strings with the current values
cb points to an int that holds the length of the string list pointed by *pStore
p is class pointer

Upon return the *pStore points to a newly allocated space and the new string
is copied into the new location after the old values. *pStore should be NULL when there
are no strings stored yet. *cb is also altered to hold the new length

*/
static int CopyConfigValue(char *s, char **pStore, int *cb, pConfigurationData p){
  int k;
  char *r;

  while( isspace(*s) )s++;
  k = strlen(s) + 1;
  r = *pStore;
  *pStore = ALLOC( (*cb) + k +1);
  if( ! *pStore )return CONFIG_ERROR_MEMORY_LOW;
  if( r ){
    memcpy(*pStore,r,*cb);
    FREE(r);
    r = (*pStore) + *cb;
    }else r = *pStore;
  strcpy(r,s);
  r[k] = (char)0; /* the double terminating zero */
  *cb += k;
  return 0;
  }

/*POD
=section ReadConfigData
=H Read the configuration data from the config file

Call this function to read the configuration data. This is usually done from the
T<main> function. Currently only the command T<external> uses configuration information
and the reader routines.

/*FUNCTION*/
int configurer_ReadConfigData(pConfigurationData p
  ){
/*noverbatim
CUT*/
#define LINLEN 1024
  int cbConfigFileName=LINLEN;
  char szBuffer[LINLEN];
  FILE *fpConfigFile;
  int ch,i;
  void **pValue;
  char *s,*pszKey,*pszValue;
  int cbValue;

  p->ST_ConfigData = sym_NewSymbolTable(p->memory_allocating_function,p->pMemorySegment);
  if( p->ST_ConfigData == NULL )return 0;
  if( GetConfigFileName(szBuffer,(unsigned int *)&cbConfigFileName) ||
    (fpConfigFile = file_fopen(szBuffer,"r")) == NULL
    ){
    /* if the config file can not be used then the default
       values are filled into the config data. This is hard coded ScriptBasic. */
    pValue = SYMLOOKUP("dll");
    if( *pValue == NULL )return CONFIG_ERROR_MEMORY_LOW;
    *pValue = ALLOC(5);
    strcpy(*pValue,".dll");

    pValue = SYMLOOKUP("module");
    if( *pValue == NULL )return CONFIG_ERROR_MEMORY_LOW;
    *pValue = ALLOC(1);
    strcpy(*pValue,"");

    pValue = SYMLOOKUP("include");
    if( *pValue == NULL )return CONFIG_ERROR_MEMORY_LOW;
    *pValue = ALLOC(1);
    strcpy(*pValue,"");

    return 0;
    }


  ch = !EOF;
  while( ch != EOF ){
    i=0;
    ch = ' ';
    /* eat the space at the start of the file */
    while( isspace(ch) )ch=getc(fpConfigFile);
    while( i < LINLEN && ch != EOF && ch != '\n' ){
      szBuffer[i++] = (char)ch;
      ch = getc(fpConfigFile);
      }
    szBuffer[i] = (char)0;

    /* remove the trailing spaces */
    if( i )i--;
    while( isspace(szBuffer[i]) && i )szBuffer[i--] = (char)0;

    /* skip empty and comment lines */
    if( !*szBuffer || *szBuffer == '#' )continue;

    pszKey = szBuffer;
    pszValue = szBuffer;

    while( *pszValue && ! isspace(*pszValue) )pszValue++;
    if( *pszValue )*pszValue++ = (char)0;
    while( isspace(*pszValue) )pszValue ++;

    pValue = SYMLOOKUP(pszKey);
    cbValue = 0;
    if( *pValue ){
      s = (char *)*pValue;
      while( *s ){
        cbValue += strlen(s)+1;/* the characters and the zchar */
        s = ((char *)*pValue) + cbValue;
        }
      }

    if( CopyConfigValue(pszValue,(char **)pValue,&cbValue,p) )return CONFIG_ERROR_MEMORY_LOW;

    }
  file_fclose(fpConfigFile);
  return 0;
  }

/*POD
=section ConfigData
=H Get config data out of the config symbol table



/*FUNCTION*/
char *configurer_ConfigData(pConfigurationData p,
                            char *s
  ){
/*noverbatim
CUT*/
  void **pValue;
#undef SYMLOOKUP
#define SYMLOOKUP(x) sym_LookupSymbol((x),p->ST_ConfigData,0,p->memory_allocating_function,p->memory_releasing_function,p->pMemorySegment)
  pValue = SYMLOOKUP(s);
  if( pValue == NULL )return NULL;
  return *pValue;
  }
