/* 
FILE:   dynlolib.c
HEADER: dynlolib.h

--GNU LGPL
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

TO_HEADER:

*/

#include <stdlib.h>

#ifdef _WIN32
#include <windows.h>
#include <winbase.h>
#else
#include <dlfcn.h>
#endif

/*POD
=H Handling Dynamic Load Libraries

The Dynamic Load Libraries are handled different on all operating systems. This file
implements a common functional base handling the DLLs for ScriptBasic. All other modules
of ScriptBasic that want to use DLLs should call only the functions implemented in this
file.

=toc

CUT*/

/*POD
=section LoadLibrary
=H Load a library

This function loads a library and returns a pointer that can be used in other functions
referencing the loaded library.
/*FUNCTION*/
void *dynlolib_LoadLibrary(
  char *pszLibraryFile
  ){
/*noverbatim
The argument T<pszLibraryFile> is the ZCHAR file name.

The file name is either absolute or relative. When a relative file name is specified the
directories searched may be different on different operating systems.
CUT*/
  char *s,*r;
  void *pLib;

/* This code was modified in v1.0b16 to make a copy of the library string before altering \ and /
   This was needed, because the module loader depends on the actual file name to recognize
   modules already loaded. The user provided the exactly same file name for each function it declared
   but using forward slashes under WNT and the module loader loaded the module many times because 
   the stored file name was stored with the converted slash.
*/

  r = s = malloc(strlen(pszLibraryFile)+1);
  if( s == NULL )return NULL;
  strcpy(s,pszLibraryFile);

#ifdef _WIN32
  for(  ;  *s ; s++ )
    if( *s == '/' )*s = '\\';
  pLib = (void *)LoadLibrary(r);
#else
  for( ;  *s ; s++ )
    if( *s == '\\' )*s = '/'; /* just to ease the life of lazy DOS programmers. */
  pLib = (void *)dlopen(r,RTLD_LAZY);
#endif
  free(r);
  return pLib;
  }

/*POD
=section FreeLibrary
=H Release a library

This function releases the library that was loaded before using R<LoadLibrary>
/*FUNCTION*/
void dynlolib_FreeLibrary(
  void *pLibrary
  ){
/*noverbatim
The argument T<pLibrary> is the pointer, which was returned by the function R<LoadLibrary>
CUT*/

#ifdef _WIN32
  FreeLibrary((HMODULE)pLibrary);
#else
  dlclose(pLibrary);
#endif
  return;
  }

/*POD
=section GetFunctionByName
=H Get the entry point of a function by its name

This function can be used to get the entry point of a function of a loaded module
specifying the name of the function.
/*FUNCTION*/
void *dynlolib_GetFunctionByName(
  void *pLibrary,
  char *pszFunctionName
  ){
/*noverbatim
The argument T<pLibrary> is the pointer, which was returned by the function R<LoadLibrary>

The argument T<pszFunctionName> is the ZCAR function name.
CUT*/

#ifdef _WIN32
  return GetProcAddress((HMODULE)pLibrary,pszFunctionName);
#else
  return dlsym(pLibrary,pszFunctionName);
#endif
  }
