/* 
FILE:   memory.c
HEADER: memory.h

--GNU LGPL
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

TO_HEADER:

typedef unsigned char BYTE, *PBYTE;

typedef struct _FixSizeMemoryObject {

  union {
    PBYTE pValue; // the value of the object
    long lValue;  // the long value of the object
    double dValue;// the double value of the object
    struct _FixSizeMemoryObject **aValue;
    } Value;
  unsigned long Size; // the actual size used (may be different from the size indicated by the type)
  BYTE sType; //index to the array SizeOfType. If this is LARGE_BLOCK_TYPE then
              //Size is the memory allocated for the value
  BYTE vType; // the variable type, VTYPE_/LONG/DOUBLE/STRING/REF
  BYTE State; // state of the variable (mortal, free or assigned)
  // these two pointers help to maintain either the free list or the mortal lists
  // a variable can never be in a free and in a mortal list at the same time
  struct _FixSizeMemoryObject *next, // link to the next object
                              *prev; // link to the previous object
  long ArrayLowLimit, ArrayHighLimit; // If this is an array then these fields tell you where to start
                                      // and where to end
  } FixSizeMemoryObject, *pFixSizeMemoryObject, 
    *MortalList, **pMortalList;

// these macros can be used to access the actual values of the objects
#define DOUBLEVALUE(x) ((x)->Value.dValue)
#define LONGVALUE(x)   ((x)->Value.lValue)
#define STRINGVALUE(x) ((char *)((x)->Value.pValue))
#define ARRAYVALUE(x,i)  (((x)->Value.aValue)[(i)-(x)->ArrayLowLimit])
#define ARRAYLOW(x)    ((x)->ArrayLowLimit)
#define ARRAYHIGH(x)   ((x)->ArrayHighLimit)
#define STRLEN(x)      ((x)->Size)

#define MAX_NUMBER_OF_FIX_TYPES 254
#define LARGE_BLOCK_TYPE 255

typedef struct _MemoryObject {
  unsigned long SizeOfType[MAX_NUMBER_OF_FIX_TYPES]; // SizeOfType[i] is the size of the type i
  BYTE iNumberOfFixTypes;    // the number of fix length type memory objects
                             // we can have at most MAX_NUMBER_OF_FIX_TYPES fix size types
                             // LARGE_BLOCK_TYPE means large block which is not stored in free list
  BYTE FirstStringIndex,LastStringIndex;
  pFixSizeMemoryObject MemoryObjectsFreeList[MAX_NUMBER_OF_FIX_TYPES];
  void *(*memory_allocating_function)(size_t);
  void (*memory_releasing_function)(void *);
  void *pMemorySegment; //this pointer is passed to the memory allocating functions
                        //this pointer is initialized in ex_init

  } MemoryObject, *pMemoryObject;

#define TYPESIZE(pMo,X) (pMo->SizeOfType(X))
#define IsMortal(x) ((x)->State == STATE_MORTAL)

// these are fixed and stored inside the FixSizeMemoryObject in the place of the pointer
// these constants should be used for the field sType
#define FIX_TYPE_LONG   0
#define FIX_TYPE_DOUBLE 1

#define STATE_UNKNOWN  0
#define STATE_FREE     1
#define STATE_MORTAL   2
#define STATE_IMMORTAL 3

// these constants are for vType (which is variable type)
// you can define other variables in case you need other types
#define VTYPE_LONG   0 // long value stored inside the struct
#define VTYPE_DOUBLE 1 // double value stored inside the struct
#define VTYPE_STRING 2 // string pointed by Value.pValue
#define VTYPE_ARRAY  3 // array pointing to other values
#define VTYPE_REF    4 // reference value pointing to another variable
*/
#include <stdlib.h>
#include <string.h>

#ifdef _DEBUG
#include <stdio.h>
#endif

#include "errcodes.h"
#include "memory.h"
#include "myalloc.h"

/*POD
=H Memory handling routines to allocate and release memory for variable storage

CUT*/

/*POD
=section InitStructure
=H Initialize a memory object

/*FUNCTION*/
void memory_InitStructure(pMemoryObject pMo
  ){
/*noverbatim
CUT*/
  int i;

  for( i=0 ; i < MAX_NUMBER_OF_FIX_TYPES ; i++ ){
    pMo->SizeOfType[i] = 0;
    pMo->MemoryObjectsFreeList[i] = NULL;
    }
  pMo->iNumberOfFixTypes = 2; /* long and double are implicit types that 
                                 are stored inside the FixSizeMemoryObject */
  pMo->pMemorySegment = alloc_InitSegment(pMo->memory_allocating_function,pMo->memory_releasing_function);
  }

/*POD
=section RegisterType
=H Register a fix length variable type

This function should be used to register a variable type. The return value
is the serial number of the type that should later be used to reference the type.

/*FUNCTION*/
int memory_RegisterType(pMemoryObject pMo,
                        unsigned long SizeOfThisType
  ){
/*noverbatim
The argument of the function is the size of the type in terms of bytes. Usually this
is calculated using the C structure T<sizeof>.

If the type can not be registered -1 is returned.
CUT*/
  if( pMo->iNumberOfFixTypes >= MAX_NUMBER_OF_FIX_TYPES )return -1;
  pMo->SizeOfType[pMo->iNumberOfFixTypes++] = SizeOfThisType;
  return pMo->iNumberOfFixTypes-1;
  }


/*POD
=section RegisterTypes
=H Register the usual fix size object types

This function should be used to initialize the usual T<FixSizeMemoryObject> types. This
sets some usual string sizes, but the caller may not call this function and set different
size objects.

/*FUNCTION*/
void memory_RegisterTypes(pMemoryObject pMo
  ){
/*noverbatim
This function registers the different string sizes. In the current implementation a string has
at least 32 characters. If this is longer that that (including the terminating zchar) then
a 64 byte fix size object is allocated. If this is small enough then a 128 byte fix size memory
object is allocated and so on up to 1024 bytes. If a string is longer that that then a LARGE_OBJECT_TYPE
is allocated.

CUT*/

  pMo->FirstStringIndex = memory_RegisterType(pMo,32);
                          memory_RegisterType(pMo,64);
                          memory_RegisterType(pMo,128);
                          memory_RegisterType(pMo,256);
                          memory_RegisterType(pMo,512);
  pMo->LastStringIndex = memory_RegisterType(pMo,1024);
  }

/*POD
=section NewVariable
=H Allocate a new variable

This function should be used whenever a new variable is to be allocated.
The function returns a pointer to a T<FixSizeMemoryObject> structure that
holds the variable information and pointer to the memory that stores the
actual value for the memory.

If there is not engough memory or the calling is illegal the returned value is T<NULL>
/*FUNCTION*/
pFixSizeMemoryObject memory_NewVariable(pMemoryObject pMo,
                                        int type,
                                        unsigned long LargeBlockSize
  ){
/*noverbatim

The second argument gives the type of the memory object to be allocated. If this
value is T<LARGE_BLOCK_TYPE> then the third argument is used to determine the size of the
memory to be allocated. If the type if NOT T<LARGE_BLOCK_TYPE> then this argument is
ignored and the proper size is allocated.

If the type has memory that was earlier allocated and released it is stored in a free list
and is reused.

CUT*/
  pFixSizeMemoryObject p;

  if( type == LARGE_BLOCK_TYPE ){
    p = alloc_Alloc(sizeof(FixSizeMemoryObject),pMo->pMemorySegment);
    if( p == NULL )return NULL;
    p->sType = type;
    p->State = STATE_UNKNOWN;
    p->next = NULL;
    p->prev = NULL;
    p->ArrayHighLimit = 0;
    p->ArrayLowLimit  = 1;
    p->Size = LargeBlockSize;
    p->Value.pValue = alloc_Alloc(LargeBlockSize,pMo->pMemorySegment);
    if( p->Value.pValue == NULL ){
      alloc_Free(p,pMo->pMemorySegment);
      return NULL;
      }
    return p;
    }

  if( type >= MAX_NUMBER_OF_FIX_TYPES )return NULL;

  if( pMo->MemoryObjectsFreeList[type] == NULL ){
    p = alloc_Alloc(sizeof(FixSizeMemoryObject),pMo->pMemorySegment);
    if( p == NULL )return NULL;
    p->sType = type;
    p->State = STATE_UNKNOWN;
    p->next = NULL;
    p->prev = NULL;
    /* if size is null then the variable is stored inside the FixSizeMemoryObject */
    if( pMo->SizeOfType[type] != 0 ){
      p->Value.pValue = alloc_Alloc(pMo->SizeOfType[type],pMo->pMemorySegment);
      if( p->Value.pValue == NULL ){
        alloc_Free(p,pMo->pMemorySegment);
        return NULL;
        }
      }else{
      p->Value.pValue = NULL;
      }
    return p;
    }

  p = pMo->MemoryObjectsFreeList[type];
  pMo->MemoryObjectsFreeList[type] = pMo->MemoryObjectsFreeList[type]->next;
  if( pMo->MemoryObjectsFreeList[type] )
    pMo->MemoryObjectsFreeList[type]->prev = NULL;

  p->next = NULL;
  p->prev = NULL;
  p->ArrayHighLimit = 0;
  p->ArrayLowLimit  = 1;
  p->State = STATE_UNKNOWN;
  return p;
  }

/*POD
=section ReleaseVariable
=H Release a variable value which is not used anymore

This function should be used to release a memory object.

/*FUNCTION*/
void memory_ReleaseVariable(pMemoryObject pMo,
                            pFixSizeMemoryObject p
  ){
/*noverbatim
CUT*/
   long i;
#ifdef _DEBUG
pFixSizeMemoryObject q;
#endif
  /* ease the life of caller if it does not check that the value is undef. */
  if( p == NULL )return;

#ifdef _DEBUG
  if( p->sType != LARGE_BLOCK_TYPE ){
    q = pMo->MemoryObjectsFreeList[p->sType];
    while( q ){
      if( q == p ){
        printf("A released memory location is rereleased.\n");
        exit(666);
        }
      q = q->next;
      }
    }
#endif


  /* if the variable is an array then all the elements should be released */
  if( p->vType == VTYPE_ARRAY ){
    for( i = p->ArrayLowLimit ; i<= p->ArrayHighLimit ; i++ )
      if( p->Value.aValue[i - p->ArrayLowLimit] )
        memory_ReleaseVariable(pMo,p->Value.aValue[i - p->ArrayLowLimit]);
    }

  /* if it is a large block, then release it */
  if( p->sType == LARGE_BLOCK_TYPE ){
    alloc_Free(p->Value.pValue,pMo->pMemorySegment);
    alloc_Free(p,pMo->pMemorySegment);
    return;
    }

  /* link the item into the free list */
  p->next = pMo->MemoryObjectsFreeList[p->sType];
  p->prev = NULL;
  if( p->next )
    p->next->prev = p;
  p->State = STATE_FREE;
  pMo->MemoryObjectsFreeList[p->sType] = p;
  return;
  }

/*POD
=section NewString
=H Allocate a new string variable

This function should be used to allocate string variable.
/*FUNCTION*/
pFixSizeMemoryObject memory_NewString(pMemoryObject pMo,
                                      unsigned long StringSize
  ){
/*noverbatim

The second argument specifies the length of th required string including the terminating
zchar.

The function checks the desired length and if this is small then is allocates a fix size
object. If this is too large then it allocates a T<LARGE_BLOCK_TYPE>

CUT*/
  BYTE i;
  pFixSizeMemoryObject p;

  for( i=pMo->FirstStringIndex ; i<= pMo->LastStringIndex ; i++ ){
    if( StringSize <= pMo->SizeOfType[i] ){
      p = memory_NewVariable(pMo,i,0);
      if( p == NULL )return NULL;
      p->vType = VTYPE_STRING;
      p->Size = StringSize;
      return p;
      }
    }
  p = memory_NewVariable(pMo,(int)LARGE_BLOCK_TYPE,StringSize);
  if( p == NULL )return NULL;
  p->vType = VTYPE_STRING;
  return p;
  }

/*POD
=section NewRef
=H Allocate a new reference variable

/*FUNCTION*/
pFixSizeMemoryObject memory_NewRef(pMemoryObject pMo
  ){
/*noverbatim
CUT*/
  pFixSizeMemoryObject p;
  p = memory_NewVariable(pMo,(BYTE)FIX_TYPE_LONG,0);
  if( p == NULL )return NULL;
  p->vType = VTYPE_REF;
  p->State = STATE_IMMORTAL;
  p->prev = p->next = NULL;
  return p;
  }


/*POD
=section NewLong
=H Allocate a new long variable

/*FUNCTION*/
pFixSizeMemoryObject memory_NewLong(pMemoryObject pMo
  ){
/*noverbatim
CUT*/
  pFixSizeMemoryObject p;
  p = memory_NewVariable(pMo,(BYTE)FIX_TYPE_LONG,0);
  if( p == NULL )return NULL;
  p->vType = VTYPE_LONG;
  p->State = STATE_IMMORTAL;
  p->prev = p->next = NULL;
  return p;
  }

/*POD
=section NewDouble
=H Allocate a new double variable

/*FUNCTION*/
pFixSizeMemoryObject memory_NewDouble(pMemoryObject pMo
  ){
/*noverbatim
CUT*/
  pFixSizeMemoryObject p;
  p = memory_NewVariable(pMo,FIX_TYPE_DOUBLE,0);
  if( p == NULL )return NULL;
  p->vType = VTYPE_DOUBLE;
  p->prev = p->next = NULL;
  p->State = STATE_IMMORTAL;
  return p;
  }

/*POD
=section NewArray
=H Allocate a new array

This function should be used whenever a new array is to be allocated.

/*FUNCTION*/
pFixSizeMemoryObject memory_NewArray(pMemoryObject pMo,
                                     long LowIndex,
                                     long HighIndex
  ){
/*noverbatim
The index variables define the indices that are to be used when accessing an
array element. The index values are inclusive.
CUT*/
  pFixSizeMemoryObject p;
  long i;

  p = memory_NewVariable(pMo,LARGE_BLOCK_TYPE, (HighIndex-LowIndex+1)*sizeof(void *));
  if( p == NULL )return NULL;
  p->vType = VTYPE_ARRAY;
  p->ArrayHighLimit = HighIndex;
  p->ArrayLowLimit = LowIndex;
  /* initialize the array */
  for( i=LowIndex ; i<=HighIndex ; i++ )
    p->Value.aValue[i - LowIndex] = NULL;
  p->prev = p->next = NULL;
  p->State = STATE_IMMORTAL;
  return p;
  }

/*POD
=section ReDimArray
=H Redimension an array

This function should be used when an array needs redimensioning.
If the redimensioning is succesful the function returns the pointer
to the argument T<p>. If memory allocation is needed and the memory
allocation fails the function returns T<NULL>. In this case the 
original array is not changed.

If the redimensioned array is smaller that the original no memory allocation
takes place, only the array elements (pointers) are moved.

/*FUNCTION*/
pFixSizeMemoryObject memory_ReDimArray(pMemoryObject pMo,
                                       pFixSizeMemoryObject p,
                                       long LowIndex,
                                       long HighIndex
  ){
/*noverbatim
CUT*/
  unsigned long NewSize;
  long i;
  pFixSizeMemoryObject *pValue;

  NewSize = (HighIndex-LowIndex+1)*sizeof(void*);
  if( NewSize > p->Size ){
    pValue = alloc_Alloc(NewSize,pMo->pMemorySegment);
    if( pValue == NULL )return NULL;

    for( i = LowIndex ; i<= HighIndex ; i++ )
      if( i < p->ArrayLowLimit || i > p->ArrayHighLimit )
        pValue[i-LowIndex] = NULL;
      else
        pValue[i-LowIndex] = p->Value.aValue[i-p->ArrayLowLimit];

    alloc_Free(p->Value.pValue,pMo->pMemorySegment);
    p->Value.aValue = pValue;
    p->ArrayHighLimit = HighIndex;
    p->ArrayLowLimit = LowIndex;
    return p;
    }

  pValue = p->Value.aValue;
  if( LowIndex < p->ArrayLowLimit )
    for( i = HighIndex ; i>= LowIndex ; i-- )
      if( i < p->ArrayLowLimit || i > p->ArrayHighLimit )
        pValue[i-LowIndex] = NULL;
      else
        pValue[i-LowIndex] = pValue[i-p->ArrayLowLimit];
  else
    for( i = LowIndex ; i<= HighIndex ; i++ )
      if( i < p->ArrayLowLimit || i > p->ArrayHighLimit )
        pValue[i-LowIndex] = NULL;
      else
        pValue[i-LowIndex] = pValue[i-p->ArrayLowLimit];

  p->ArrayHighLimit = HighIndex;
  p->ArrayLowLimit = LowIndex;
  return p;
  }

/*POD
=section CheckArrayIndex
=H Check a particular index of an array and redimension on demand

This function should be called before accessing a certain element of an array.
The function checks that the index is within the index limitsof the array
and in case the index is outside the index limits of the array it redimensionate the
array.

The function returns the pointer passed as parameter T<p> or NULL in case there is a
memory allocation error.
/*FUNCTION*/
pFixSizeMemoryObject memory_CheckArrayIndex(pMemoryObject pMo,
                                            pFixSizeMemoryObject p,
                                            long Index
  ){
/*noverbatim
CUT*/

  if( p->ArrayHighLimit < Index )
    return memory_ReDimArray(pMo,p,p->ArrayLowLimit,Index);
  if( p->ArrayLowLimit > Index )
    return memory_ReDimArray(pMo,p,Index,p->ArrayHighLimit);
  return p;
  }


/*POD
=section Mortalize
=H Make an immortal variable mortal

This function should be used when a variable is to be put in a mortal list.

/*FUNCTION*/
void memory_Mortalize(pFixSizeMemoryObject p,
                      pMortalList pMortal
  ){
/*noverbatim
Note that care should be taken to be sure that the variable is NOT on a mortal
list. If the variable is already on a mortal list calling this function will
break the original list and therefore may loose the variables that follow this one.
CUT*/

  p->next = *pMortal;
  if( p->next )
    p->next->prev = p;
  p->prev = NULL;
  p->State = STATE_MORTAL;
  *pMortal = p;
  return;
  }

/*POD
=section Immortalize
=H Remove a variable from the mortal list

Use this function to immortalize a variable. This can be used when the result of an expression
evaluation gets into a mortal variable and instead of copiing the value from the mortal variable to
an immortal variable the caller can immortalize the variable. However it should know which mortal list
the variable is on.

/*FUNCTION*/
void memory_Immortalize(pFixSizeMemoryObject p,
                        pMortalList pMortal
  ){
/*noverbatim
CUT*/
  if( ! p )return;/* the code tries to immortalize undef at several location, therefore it is better
                     to handle this situation here. */

  if( p->prev )
    p->prev->next = p->next;
  else
    *pMortal = p->next;

  if( p->next )
    p->next->prev = p->prev;

  p->prev = NULL;
  p->next = NULL;
  p->State = STATE_IMMORTAL;
  return;
  }

/*POD
=section NewMortal
=H Allocate a new mortal variable

When an expression is evaluated mortal variables are needed to store
the intermediate results. These variables are called mortal variables.
Such a variable is is allocated using this function and specifying a
variable of type T<MortalList> to assign the mortal to the list of
mortal variables.

When the expression is evaluated all mortal variables are to be released
and they are calling the function T<memory_ReleaseMortals> (see R<ReleaseMortals>).
/*FUNCTION*/
pFixSizeMemoryObject memory_NewMortal(pMemoryObject pMo,
                                      BYTE type,
                                      unsigned long LargeBlockSize,
                                      pMortalList pMortal
  ){
/*noverbatim
If the parameter T<pMortal> is T<NULL> the generated variable is not mortal.
CUT*/
  pFixSizeMemoryObject p;

  p = memory_NewVariable(pMo,type,LargeBlockSize);
  if( p == NULL )return NULL;
  if( pMortal )
    memory_Mortalize(p,pMortal);
  return p;
  }


/*POD
=section DupVar
=H Duplicate a value creating a new mortal

This function creates a new mortal and copies the argument T<pVar> into this
new mortal.

/*FUNCTION*/
pFixSizeMemoryObject memory_DupVar(pMemoryObject pMo,
                                   pFixSizeMemoryObject pVar,
                                   pMortalList pMyMortal,
                                   int *piErrorCode
  ){
/*noverbatim
This function is vital, when used in operations that convert the
values to T<long> or T<double>. Expression evaluation may return an immortal
value, when the expression is a simple variable access. Conversion of the
result would modify the value of the variable itself. Therefore functions and
operators call this function to duplicate the result to be sure that the value
they convert is mortal and to be sure they do not change the value of a variable
when they are not supposed to.

Note that you can duplicate T<long>, T<double> and T<string> values, but you can not
duplicate arrays! The string value is duplicated and the characters are copied to
the new location. This is perfect. However if you do the same with an array the array
pointers will point to the same variables, which are not going to be duplicated. This
result multiple reference to a single value. This situation is currently not supported
by this system as we do not have either garbage collection or
CUT*/
  pFixSizeMemoryObject mypVar ;

  *piErrorCode = MEM_ERROR_SUCCESS;
  if( pVar == NULL )return NULL;

  mypVar = memory_NewMortal(pMo,pVar->sType,pVar->Size,pMyMortal);
  if( mypVar == NULL ){
    *piErrorCode = MEM_ERROR_MEMORY_LOW;
    return NULL;
    }

  mypVar->vType = pVar->vType;
  mypVar->Size  = pVar->Size;
  if( pVar->sType == LARGE_BLOCK_TYPE ){
    if( pVar->Size )
      memcpy(mypVar->Value.pValue, pVar->Value.pValue, mypVar->Size);
    }else{
    if( pVar->vType == VTYPE_STRING ){
      if( pVar->Size )
        memcpy(mypVar->Value.pValue,pVar->Value.pValue,mypVar->Size);
      }else{
      mypVar->Value = pVar->Value;
      }
    }
  return mypVar;
  }

/*POD
=section DupMortalize
=H Duplicate a value creating a new mortal from new immortal

This function creates a new mortal and copies the argument T<pVar> into this
new mortal only if the value is immortal. If the value is mortal the it returns
the original value.

/*FUNCTION*/
pFixSizeMemoryObject memory_DupMortalize(pMemoryObject pMo,
                                         pFixSizeMemoryObject pVar,
                                         pMortalList pMyMortal,
                                         int *piErrorCode
  ){
/*noverbatim
CUT*/

  if( *piErrorCode )return NULL;/* if it was called after an erroneous evaluate then do not reset the error code. */
  if( pVar && IsMortal(pVar) )return pVar;
  return memory_DupVar(pMo,pVar,pMyMortal,piErrorCode);
  }

/*POD
=section ReleaseMortals
=H Release the list of mortal variables

This function should be used to release the mortal variables.

When an expression is evaluated mortal variables are needed to store
the intermediate results. These variables are called mortal variables.
Such a variable is is allocated using this function and specifying a
variable of type T<MortalList> to assign the mortal to the list of
mortal variables.

/*FUNCTION*/
void memory_ReleaseMortals(pMemoryObject pMo,
                           pMortalList pMortal
  ){
/*noverbatim
CUT*/
  MortalList p;

  if( pMortal == NULL )return; /* this is to help lasy callers */
  while( p=*pMortal ){
    *pMortal = p->next;
    p->prev = NULL; /* some bug during development caused the free list and the */
    p->next = NULL; /* mortal list joined together criss crossing               */
    memory_ReleaseVariable(pMo,p);
    }
  *pMortal = NULL;
  }

/*POD
=section NewMortalString
=H Allocate a new mortal string variable

/*FUNCTION*/
pFixSizeMemoryObject memory_NewMortalString(pMemoryObject pMo,
                                            unsigned long StringSize,
                                            pMortalList pMortal
  ){
/*noverbatim
If the parameter T<pMortal> is T<NULL> the generated variable is not mortal.
CUT*/
  pFixSizeMemoryObject p;

  p = memory_NewString(pMo,StringSize);
  if( p == NULL )return NULL;
  if( pMortal )
    memory_Mortalize(p,pMortal);
  return p;
  }

/*POD
=section NewMortalLong
=H Allocate a new mortal long variable

/*FUNCTION*/
pFixSizeMemoryObject memory_NewMortalLong(pMemoryObject pMo,
                                            pMortalList pMortal
  ){
/*noverbatim
If the parameter T<pMortal> is T<NULL> the generated variable is not mortal.
CUT*/
  pFixSizeMemoryObject p;

  p = memory_NewLong(pMo);
  if( p == NULL )return NULL;
  if( pMortal )
    memory_Mortalize(p,pMortal);
  return p;
  }

/*POD
=section NewMortalRef
=H Allocate a new mortal reference variable

/*FUNCTION*/
pFixSizeMemoryObject memory_NewMortalRef(pMemoryObject pMo,
                                            pMortalList pMortal
  ){
/*noverbatim
If the parameter T<pMortal> is T<NULL> the generated variable is not mortal.
CUT*/
  pFixSizeMemoryObject p;
  p = memory_NewRef(pMo);
  if( p == NULL )return NULL;
  if( pMortal )
    memory_Mortalize(p,pMortal);
  return p;
  }


/*POD
=section NewMortalDouble
=H Allocate a new mortal double variable

/*FUNCTION*/
pFixSizeMemoryObject memory_NewMortalDouble(pMemoryObject pMo,
                                            pMortalList pMortal
  ){
/*noverbatim
If the parameter T<pMortal> is T<NULL> the generated variable is not mortal.
CUT*/
  pFixSizeMemoryObject p;

  p = memory_NewDouble(pMo);
  if( p == NULL )return NULL;
  if( pMortal )
    memory_Mortalize(p,pMortal);
  return p;
  }


/*POD
=section NewMortalArray
=H Allocate a new mortal array variable

/*FUNCTION*/
pFixSizeMemoryObject memory_NewMortalArray(pMemoryObject pMo,
                                           pMortalList pMortal,
                                           long IndexLow,
                                           long IndexHigh
  ){
/*noverbatim
If the parameter T<pMortal> is T<NULL> the generated variable is not mortal.
CUT*/
  pFixSizeMemoryObject p;

  p = memory_NewArray(pMo,IndexLow,IndexHigh);
  if( p == NULL )return NULL;
  if( pMortal )
    memory_Mortalize(p,pMortal);
  return p;
  }
