/*
FILE:   scriba.c
HEADER: scriba.h

--GNU LGPL
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

TO_HEADER:
#include "report.h"
#include "lexer.h"
#include "sym.h"
#include "expression.h"
#include "syntax.h"
#include "reader.h"
#include "myalloc.h"
#include "builder.h"
#include "memory.h"
#include "execute.h"
#include "buildnum.h"
#include "conftree.h"
#include "filesys.h"
#include "errcodes.h"
#include "testalloc.h"
#include "command.h"
#include "epreproc.h"
#include "uniqfnam.h"

typedef struct _SbProgram {
  void *pMEM;
  void * (*maf)(size_t);
  void   (*mrf)(void *);
  unsigned long fErrorFlags;
  char *pszFileName;
  char *pszCacheFileName;
  char *FirstUNIXline;

  void *fpStdouFunction;
  void *fpStdinFunction;
  void *fpEnvirFunction;
  void *pEmbedder;
  void *fpReportFunction;
  void *pReportPointer;

  tConfigTree   *pCONF;
  ReadObject    *pREAD;
  LexObject     *pLEX;
  eXobject      *pEX;
  BuildObject   *pBUILD;
  ExecuteObject *pEXE;
  } SbProgram, *pSbProgram;

// type to pass and receive arguments and result values from ScriptBasic functions
typedef struct _SbData {
  unsigned char type;
  unsigned long size;
  union {
    double d;
    long   l;
    unsigned char *s;
    } v;
  } SbData, *pSbData;
#define SBT_UNDEF  0
#define SBT_DOUBLE 1
#define SBT_LONG   2
#define SBT_STRING 3
#define SBT_ZCHAR  4

// Access SbData content. Y is present to emulate class argument passing.
#define scriba_GetType(Y,X)   ( (X).type )
#define scriba_GetLength(Y,X) ( (X).size )
#define scriba_GetString(Y,X) ( (X).v.s  )
#define scriba_GetLong(Y,X)   ( (X).v.l  )
#define scriba_GetDouble(Y,X) ( (X).v.d  )

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "scriba.h"

/*POD
=H Interface functions to ScriptBasic

This file implements the interface functions that a program embeddign ScriptBasic
may use. This file was introduced inScriptBasic v1.0b20. Earier embedding
applicatios had to implement the functions implemented in this file.

CUT*/

/*POD
=section model
=H The OO model for executing a ScriptBasic program
=abstract
This section introduces the overall object oriented structure of the interface.
=end

To embed the ScriptBasic interpreter into an application the easiest method is to use
the interface implemented in this file. This file provides you an object oriented model
to ScriptBasic. The implementation language is C, which is not object oriented itself, but
the underlyig model is. Later version or external programmers may develop C++ or Java
interface to these functions with a few hours of effort.

To execute a ScriptBasic program the embedding code should create an T<SbProgram> object.
The leading T<Sb> obviously stands for ScriptBasic. To do this you can call the function
T<scriba_new> R<new>.

All functions implemented in this file have the prefix T<scriba_> and all functions get
a pointer to  T<SbProgram> object as first argument. All operations are performed on that
object.


CUT*/

/*POD
=section new
=H Create an object to handle a ScriptBasic program

To create a new T<SbProgram> object you have to call this function. The two arguments
should point to T<malloc> and T<free> or similar functions. All later memory allocation
and releasing will be performed using these functions.

Note that this is the only function that does not require a pointer to an
T<SbProgram> object.

/*FUNCTION*/
pSbProgram scriba_new(void * (*maf)(size_t),
                      void   (*mrf)(void *)
  ){
/*noverbatim
CUT*/
  pSbProgram pProgram;
  void *p;

  p = alloc_InitSegment(maf,mrf);
  if( p == NULL )return NULL;

  pProgram = (pSbProgram)alloc_Alloc(sizeof(SbProgram),p);
  if( pProgram == NULL ){
    alloc_FinishSegment(p);
    return NULL;
    }

  pProgram->maf              = maf;
  pProgram->mrf              = mrf;
  pProgram->pMEM             = p;
  pProgram->fErrorFlags      = 0;
  pProgram->pszFileName      = NULL;
  pProgram->pszCacheFileName = NULL;
  pProgram->FirstUNIXline    = NULL;
  pProgram->fpStdouFunction  = NULL;
  pProgram->fpStdinFunction  = NULL;
  pProgram->fpEnvirFunction  = NULL;
  pProgram->pEmbedder        = NULL;
  pProgram->fpReportFunction = report_report;
  pProgram->pReportPointer   = (void *)stderr;

  pProgram->pCONF  = NULL;
  pProgram->pREAD  = NULL;
  pProgram->pLEX   = NULL;
  pProgram->pEX    = NULL;
  pProgram->pBUILD = NULL;
  pProgram->pEXE   = NULL;

  return pProgram;
  }

/*POD
=section destroy
=H Destroy a SbProgram object

After a ScriptBasic program was successfully execued and there is no need to
run it anymore call this function to release all memory associated with the
code.

/*FUNCTION*/
void scriba_destroy(pSbProgram pProgram
  ){
/*noverbatim
CUT*/


  scriba_PurgeReaderMemory(pProgram);
  scriba_PurgeLexerMemory(pProgram);
  scriba_PurgeSyntaxerMemory(pProgram);
  scriba_PurgeBuilderMemory(pProgram);
  scriba_PurgeExecuteMemory(pProgram);

  /* Note that finishing this segment will release the configuration 
     information in case it was loaded for this object and not 
     inherited from another one. */
  alloc_FinishSegment(pProgram->pMEM);
  }

/*POD
=section NewSbData
=H Allocate new SbData

/*FUNCTION*/
pSbData scriba_NewSbData(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  return alloc_Alloc(sizeof(SbData),pProgram->pMEM);
  }

/*POD
=section NewSbLong
=H Allocate new SbData to store a long

/*FUNCTION*/
pSbData scriba_NewSbLong(pSbProgram pProgram,
                         long lInitValue
  ){
/*Noverbatim
CUT*/
  pSbData p;

  p = scriba_NewSbData(pProgram);
  if( p == NULL )return NULL;
  p->type = SBT_LONG;
  p->v.l = lInitValue;
  return p;
  }

/*POD
=section NewSbDouble
=H Allocate new SbData to store a double

/*FUNCTION*/
pSbData scriba_NewSbDouble(pSbProgram pProgram,
                           double dInitValue
  ){
/*Noverbatim
CUT*/
  pSbData p;

  p = scriba_NewSbData(pProgram);
  if( p == NULL )return NULL;
  p->type = SBT_DOUBLE;
  p->v.d = dInitValue;
  return p;
  }

/*POD
=section NewSbUndef
=H Allocate new SbData to store an undef value

/*FUNCTION*/
pSbData scriba_NewSbUndef(pSbProgram pProgram
  ){
/*Noverbatim
CUT*/
  pSbData p;

  p = scriba_NewSbData(pProgram);
  if( p == NULL )return NULL;
  p->type = SBT_UNDEF;
  return p;
  }


/*POD
=section NewSbString
=H Allocate new SbData to store a string

/*FUNCTION*/
pSbData scriba_NewSbString(pSbProgram pProgram,
                           char *pszInitValue
  ){
/*Noverbatim
CUT*/
  pSbData p;

  if( pszInitValue == NULL )return scriba_NewSbUndef(pProgram);

  p = scriba_NewSbData(pProgram);
  if( p == NULL )return NULL;
  p->type = SBT_STRING;
  p->size = strlen(pszInitValue);
  if( p->size ){
    p->v.s = alloc_Alloc(p->size,pProgram->pMEM);
    if( p->v.s == NULL ){
      alloc_Free(p,pProgram->pMEM);
      return NULL;
      }
    memcpy(p->v.s,pszInitValue,p->size);
    }else{
    p->v.s = NULL;
    }
  return p;
  }

/*POD
=section NewSbBytes
=H Allocate new SbData to store a byte array

/*FUNCTION*/
pSbData scriba_NewSbBytes(pSbProgram pProgram,
                          unsigned long len,
                          unsigned char *pszInitValue
  ){
/*Noverbatim
CUT*/
  pSbData p;

  if( pszInitValue == NULL )return scriba_NewSbUndef(pProgram);

  p = scriba_NewSbData(pProgram);
  if( p == NULL )return NULL;
  p->type = SBT_STRING;
  p->size = len;
  if( p->size ){
    p->v.s = alloc_Alloc(p->size,pProgram->pMEM);
    if( p->v.s == NULL ){
      alloc_Free(p,pProgram->pMEM);
      return NULL;
      }
    memcpy(p->v.s,pszInitValue,p->size);
    }else{
    p->v.s = NULL;
    }
  return p;
  }

/*POD
=section DestroySbData
=H Destroy data allocated by any of the functions NewSbXXX

/*FUNCTION*/
void scriba_DestroySbData(pSbProgram pProgram,
                          pSbData p
  ){
/*noverbatim
CUT*/
  if( p->type == SBT_STRING )
    alloc_Free(p->v.s,pProgram->pMEM);
  alloc_Free(p,pProgram->pMEM);
  }

/*POD
=section PurgeReaderMemory
=H Purge memory needed by the reader

You can call this function to release all memory that was allocated by the
reader module. The memory data is needed so long as long the lexical analyzer
has finished.
/*FUNCTION*/
void scriba_PurgeReaderMemory(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  if( pProgram->pREAD ){
    alloc_FinishSegment(pProgram->pREAD->pMemorySegment);
    alloc_Free(pProgram->pREAD,pProgram->pMEM);
    }
  pProgram->pREAD = NULL;
  }

/*POD
=section PurgeLexerMemory
=H Purge memory needed by the lexical analyzer

/*FUNCTION*/
void scriba_PurgeLexerMemory(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  if( pProgram->pLEX )
    alloc_FinishSegment(pProgram->pLEX->pMemorySegment);
  alloc_Free(pProgram->pLEX,pProgram->pMEM);
  pProgram->pLEX = NULL;
  }

/*POD
=section PurgeSyntaxerMemory
=H Purge memory needed by the syntax analyzer

/*FUNCTION*/
void scriba_PurgeSyntaxerMemory(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  if( pProgram->pEX )
    ex_free( pProgram->pEX );
  alloc_Free(pProgram->pEX,pProgram->pMEM);
  pProgram->pEX = NULL;
  }

/*POD
=section PurgeBuilderMemory
=H Purge memory needed by the builder

/*FUNCTION*/
void scriba_PurgeBuilderMemory(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  if( pProgram->pBUILD )
    alloc_FinishSegment(pProgram->pBUILD->pMemorySegment);
  alloc_Free(pProgram->pBUILD,pProgram->pMEM);
  pProgram->pBUILD = NULL;
  }

/*POD
=section PurgeExecuteMemory
=H Purge memory needed by the execute subsystem

This function purges the memory that was needed to execute the program,
but before that it executes the finalization part of the execution.

/*FUNCTION*/
void scriba_PurgeExecuteMemory(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  int iErrorCode;

  if( pProgram->pEXE ){
    execute_FinishExecute(pProgram->pEXE,&iErrorCode);
    if( pProgram->pEXE->pMo &&
        pProgram->pEXE->pMo->pMemorySegment )alloc_FinishSegment(pProgram->pEXE->pMo->pMemorySegment);
    alloc_FinishSegment(pProgram->pEXE->pMemorySegment);
    }
  alloc_Free(pProgram->pEXE,pProgram->pMEM);
  pProgram->pEXE = NULL;
  }

/*POD
=section SetFileName
=H Specify the input file for the program

/*FUNCTION*/
int scriba_SetFileName(pSbProgram pProgram,
                        char *pszFileName
  ){
/*noverbatim
CUT*/
  if( pProgram->pszFileName )alloc_Free(pProgram->pszFileName,pProgram->pMEM);
  pProgram->pszFileName = NULL;
  if( pszFileName ){
    pProgram->pszFileName = alloc_Alloc(strlen(pszFileName)+1,pProgram->pMEM);
    if( pProgram->pszFileName == NULL )SCRIBA_ERROR_MEMORY_LOW;
    strcpy(pProgram->pszFileName,pszFileName);
    }
  return SCRIBA_ERROR_SUCCESS;
  }

/*POD
=section GettingConfiguration
=H How to load the configuration information
=abstract
R<LoadConfiguration> and R<InheritConfiguration> can be used to
specify configuration information for a ScriptBasic program. Here
we describe the differences and how to use the two functions for
single-process single-basic and for single-process multiple-basic
applications.
=end

To execute a ScriptBasic program you usually need configuration information.
The configuration information for the interpreter is stored in a file.
The function R<LoadConfiguration> reads the file and loads it into memory
into the T<SbProgram> object. When the object is destroyed the configuration
information is automatically purged from memory.

Some implementations like the Windows NT ISAPI variation of ScriptBasic starts
several interpreter thread within the same process. In this case the configuration
information is read only once and all the running interpreters share the same
configuration information.

To do this the embedding program has to create a pseudo T<SbProgram> object that
does not run any ScriptBasic program, but is used only to load the configuration
information calling the function R<LoadConfiguration>. Other T<SbProgram> objects
that do intepret ScriptBasic program should inherit this configuration calling the
function R<InheritConfiguration>. When a T<SbProgram> object that inherited the
configuration is destroyed the configuration is NOT destroyed. It remains in memory
and can later be used by other intrepreter instances.

See the configuration handling functions R<LoadConfiguration> and R<InheritConfiguration>.
CUT*/

/*POD
=section LoadConfiguration
=H Load the configuration information from the config file

This function should be used to load the configuration information
from a file.

The return value iz zero on success and the error code when error happens.
/*FUNCTION*/
int scriba_LoadConfiguration(pSbProgram pProgram,
                             char *pszForcedConfigurationFileName
  ){
/*noverbatim
CUT*/
  int iError;
  FILE *fp;
  pProgram->pCONF = alloc_Alloc(sizeof(tConfigTree),pProgram->pMEM);
  if( pProgram->pCONF == NULL )return SCRIBA_ERROR_MEMORY_LOW;

  iError = cft_start(pProgram->pCONF,alloc_Alloc,alloc_Free,pProgram->pMEM,
#ifdef WIN32
            "Software\\ScriptBasic\\config",
            "SCRIBA.INI",
#else
            "SCRIBACONF",
            "/etc/scriba/basic.conf",
#endif
            pszForcedConfigurationFileName);
/*DEBUG*/
  fp = fopen("e:\\MyProjects\\sb\\isapi.log","a");
  if( fp ){
    fprintf(fp,"LoadConfiguration returned %d.\n",iError);
    fclose(fp);
    fp = NULL;
    }

  return iError;
  }

/*POD
=section InheritConfiguration
=H Inherit the configuration information from another program object

Use this function to get the configuration from another program object.

The return value is zero on success and error code if error has happened.
/*FUNCTION*/
int scriba_InheritConfiguration(pSbProgram pProgram,
                                pSbProgram pFrom
  ){
/*noverbatim
CUT*/
  if( pFrom == NULL )return 1;
  pProgram->pCONF = pFrom->pCONF;
  if( pProgram->pCONF == NULL )return 1;
  return 0;
  }

/*POD
=section SetCgiFlag
=H Specify that this is CGI execution

You can call this function to tell the reporting subsystem that
this code runs in a CGI environment and therefore it should format
error messages according to the CGI standard sending to the 
standard output.

/*FUNCTION*/
void scriba_SetCgiFlag(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  pProgram->fErrorFlags |= REPORT_F_CGI;
  }

/*POD
=section SetReportFunction
=H Set the report function

This function should be used to set the report function for a program. The report function
is used to send info, warning, error, fatal and internal error messages to the user.

In case you want to implement a specific report function see the sample implementation in the
file T<report.c>. The documentation of the function T<report_report> describes not only the details
of the sample implementation but also the implementation requests for other reporting functions.

/*FUNCTION*/
void scriba_SetReportFunction(pSbProgram pProgram,
                              void *fpReportFunction
  ){
/*noverbatim
CUT*/
  pProgram->fpReportFunction = fpReportFunction;
  if( pProgram->pEXE )pProgram->pEXE->report = fpReportFunction;
  }

/*POD
=section SetReportPointer
=H Set the report pointer

/*FUNCTION*/
void scriba_SetReportPointer(pSbProgram pProgram,
                             void *pReportPointer
  ){
/*noverbatim
CUT*/
  pProgram->pReportPointer = pReportPointer;
  if( pProgram->pEXE )pProgram->pEXE->reportptr = pReportPointer;
  }

/*POD
=section SetStdin
=H Specify special standard input function

You can call this function to define a special standard input function. This
pointer should point to a function that accepts a T<void *> pointer
as argument. Whenever the ScriptBasic program tries to read from the
standard input it calls this function pasing the embedder pointer as
argument.

If the T<stdin> function is not defined or the parameter is T<NULL>
the interpreter will read the normal T<stdin> stream.

/*FUNCTION*/
void scriba_SetStdin(pSbProgram pProgram,
                     void *fpStdinFunction
  ){
/*noverbatim
CUT*/
  pProgram->fpStdinFunction = fpStdinFunction;
  if( pProgram->pEXE )pProgram->pEXE->fpStdinFunction = fpStdinFunction;
  }

/*POD
=section SetStdout
=H Specify special standard output function

You can call this function to define a special standard output function. This
pointer should point to a function that accepts a T<(char, void *)> arguments.
Whenever the ScriptBasic program tries to send a character to the standard output
it calls this function. The first parameter is the character to write, the second
is the embedder pointer.

If the standard output function is not defined or the parameter is T<NULL>
the interpreter will write the normal T<stdout> stream.

/*FUNCTION*/
void scriba_SetStdout(pSbProgram pProgram,
                      void *fpStdoutFunction
  ){
/*noverbatim
CUT*/
  pProgram->fpStdouFunction = fpStdoutFunction;
  if( pProgram->pEXE )pProgram->pEXE->fpStdouFunction = fpStdoutFunction;
  }

/*POD
=section SetEmbedPointer
=H Set the embedder pointer

This function should be used to set the embed pointer.

The embed pointer is a pointer that is not used by ScriptBasic itself. This
pointer is remembered by ScriptBasic and is passed to call-back functions,
like the standard input, output and environment functions that the embedding
application may provide and is also available to external modules implemented
in C or other compiled language in DLL or SO files.

The embedder pointer should usually point to the T<struct> of the thread local data.
For example the Windows NT IIS variation of ScriptBasic sets this variable to point to
the extension control block.

If this pointer is not set ScriptBasic will pass T<NULL> pointer to the extensions and
to the call-back function.
/*FUNCTION*/
void scriba_SetEmbedPointer(pSbProgram pProgram,
                            void *pEmbedder
  ){
/*noverbatim
CUT*/
  pProgram->pEmbedder = pEmbedder;
  if( pProgram->pEXE )pProgram->pEXE->pEmbedder = pEmbedder;
  }

/*POD
=section SetEnvironment
=H Specify special environment query function

You can call this function to define a special environment query function. This
pointer should point to a function that accepts a T<(void *, char *, long )> arguments.

Whenever the ScriptBasic program tries to get the value of an enviroment variable
it calls this function. The first argument is the embedder pointer.

The second argument is the name of the environment variable to retrieve or T<NULL>.

The third argument is either zero or is the serial number of the environment variable.

ScriptBasic never calls this function with both specifying the environment variable name
and the serial number.

The return value of the function should either be T<NULL> or should point to a string that
holds the zero character terminated value of the environment variable. This string is not
changed by ScriptBasic.

If the special environment function is not defined or is T<NULL> ScriptBasic uses the
usual environment of the process.

/*FUNCTION*/
void scriba_SetEnvironment(pSbProgram pProgram,
                           void *fpEnvirFunction
  ){
/*noverbatim
CUT*/
  pProgram->fpEnvirFunction = fpEnvirFunction;
  if( pProgram->pEXE )pProgram->pEXE->fpEnvirFunction = fpEnvirFunction;
  }


/*POD
=section LoadBinaryProgram
=H Load a binary program from a file

Use this function to load ScriptBasic program from a file that is already compiled into
internal form.

The return value is the number of errors (hopefully zero) during program load.

/*FUNCTION*/
int scriba_LoadBinaryProgram(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  pProgram->pBUILD = alloc_Alloc( sizeof(BuildObject) , pProgram->pMEM);
  if( pProgram->pBUILD == NULL )return 1;

  pProgram->pBUILD->memory_allocating_function = pProgram->maf;
  pProgram->pBUILD->memory_releasing_function  = pProgram->mrf;
  pProgram->pBUILD->iErrorCounter = 0;
  pProgram->pBUILD->reportptr = pProgram->pReportPointer;
  pProgram->pBUILD->report   = pProgram->fpReportFunction;
  pProgram->pBUILD->fErrorFlags = pProgram->fErrorFlags;
  build_LoadCode(pProgram->pBUILD,pProgram->pszFileName);
  return pProgram->pBUILD->iErrorCounter;
  }

/*POD
=section InheritBinaryProgram
=H Inherit binary code from another program object

Use this function in application that keeps the program code in memory.

/*FUNCTION*/
int scriba_InheritBinaryProgram(pSbProgram pProgram,
                                pSbProgram pFrom
  ){
/*noverbatim

The function inherits the binary code from the program object T<pFrom>.
In server type applications the compiled binary code of a BASIC program may
be kept in memory. To do this a pseudo program object should be created that
loads the binary code and is not destroyed.

The program object used to execute the code should inherit the binary code from
this pseudo object calling this function. This is similar to the configuration
inheritance.

CUT*/
  pProgram->pBUILD = alloc_Alloc( sizeof(BuildObject) , pProgram->pMEM);
  if( pProgram->pBUILD == NULL )return SCRIBA_ERROR_MEMORY_LOW;

  memcpy(pProgram->pBUILD,pFrom->pBUILD,sizeof(BuildObject));
  pProgram->pBUILD->memory_allocating_function = pProgram->maf;
  pProgram->pBUILD->memory_releasing_function  = pProgram->mrf;
  pProgram->pBUILD->iErrorCounter = 0;
  pProgram->pBUILD->reportptr = pProgram->pReportPointer;
  pProgram->pBUILD->report   = pProgram->fpReportFunction;
  pProgram->pBUILD->fErrorFlags = pProgram->fErrorFlags;
  return SCRIBA_ERROR_SUCCESS;
  }


/*POD
=section ReadSource
=H Read the source code of a ScriptBasic program from a file

Loads the source code of a ScriptBasic program from a text file.

The return code is the number of errors happened during read.

/*FUNCTION*/
int scriba_ReadSource(pSbProgram pProgram
  ){
/*noverbatim
CUT*/

  pProgram->pREAD = alloc_Alloc( sizeof(ReadObject) , pProgram->pMEM );
  if( pProgram->pREAD == NULL )return 1;

  reader_InitStructure(pProgram->pREAD);
  pProgram->pREAD->memory_allocating_function = alloc_Alloc;
  pProgram->pREAD->memory_releasing_function = alloc_Free;
  pProgram->pREAD->pMemorySegment = alloc_InitSegment(
                                     pProgram->maf,
                                     pProgram->mrf);
  pProgram->pREAD->report = pProgram->fpReportFunction;
  pProgram->pREAD->reportptr = pProgram->pReportPointer;
  pProgram->pREAD->iErrorCounter = 0;
  pProgram->pREAD->fErrorFlags = pProgram->fErrorFlags;
  pProgram->pREAD->pConfig = pProgram->pCONF;
  if( ! reader_ReadLines(pProgram->pREAD,pProgram->pszFileName) ){
    if( pProgram->pREAD->FirstUNIXline ){
      pProgram->FirstUNIXline = alloc_Alloc(strlen(pProgram->pREAD->FirstUNIXline)+1,pProgram->pMEM);
      if( pProgram->FirstUNIXline == NULL )return SCRIBA_ERROR_MEMORY_LOW;
      strcpy(pProgram->FirstUNIXline,pProgram->pREAD->FirstUNIXline);
      }
    }
  return pProgram->pREAD->iErrorCounter;
  }

/*POD
=section DoLexicalAnalysis
=H Perform lexical analysis

/*FUNCTION*/
int scriba_DoLexicalAnalysis(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  pProgram->pLEX = alloc_Alloc( sizeof(LexObject) , pProgram->pMEM );
  if( pProgram->pLEX == NULL )return 1;

  reader_StartIteration(pProgram->pREAD);

  pProgram->pLEX->memory_allocating_function = alloc_Alloc;
  pProgram->pLEX->memory_releasing_function = alloc_Free;
  pProgram->pLEX->pMemorySegment = alloc_InitSegment(
                                       pProgram->maf,
                                       pProgram->mrf);
  lex_InitStructure(  pProgram->pLEX);

  pProgram->pLEX->pfGetCharacter = reader_NextCharacter;
  pProgram->pLEX->pfFileName = reader_FileName;
  pProgram->pLEX->pfLineNumber = reader_LineNumber;

  pProgram->pLEX->pNASymbols = NASYMBOLS;
  pProgram->pLEX->pASymbols  = ASYMBOLS;
  pProgram->pLEX->pCSymbols  = CSYMBOLS;
  pProgram->pLEX->report = pProgram->fpReportFunction;
  pProgram->pLEX->reportptr = pProgram->pReportPointer;
  pProgram->pLEX->fErrorFlags = pProgram->fErrorFlags;
  pProgram->pLEX->iErrorCounter = 0;
  pProgram->pLEX->pLexResult = (void *)stderr;


  pProgram->pLEX->pvInput = (void *)  pProgram->pREAD;
  lex_ReadInput(pProgram->pLEX);

  if( pProgram->pLEX->iErrorCounter )return pProgram->pLEX->iErrorCounter;
  lex_RemoveComments(pProgram->pLEX);
  lex_HandleContinuationLines(pProgram->pLEX);
  return pProgram->pLEX->iErrorCounter;
  }

/*POD
=section DoSyntaxAnalysis
=H Perform syntactical analysis

/*FUNCTION*/
int scriba_DoSyntaxAnalysis(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  peNODE_l CommandList;

  pProgram->pEX = alloc_Alloc( sizeof(eXobject) , pProgram->pMEM);
  if( pProgram->pEX == NULL )return 1;

  pProgram->pEX->memory_allocating_function = pProgram->maf;
  pProgram->pEX->memory_releasing_function = pProgram->mrf;
  pProgram->pEX->cbBuffer = 1024; /* init will allocate the space of this number of characters */
  pProgram->pEX->cbCurrentNameSpace = 1024; /* init will allocate the space of this number of characters */
  pProgram->pEX->pLex = pProgram->pLEX;

  pProgram->pEX->Unaries  = UNARIES;
  pProgram->pEX->Binaries = BINARIES;
  pProgram->pEX->BuiltInFunctions = INTERNALFUNCTIONS;
  pProgram->pEX->MAXPREC  = MAX_BINARY_OPERATOR_PRECEDENCE;
  pProgram->pEX->PredeclaredLongConstants = PREDLCONSTS;
  pProgram->pEX->reportptr = pProgram->pReportPointer;
  pProgram->pEX->report   = pProgram->fpReportFunction;
  pProgram->pEX->fErrorFlags = pProgram->fErrorFlags;
  pProgram->pEX->iErrorCounter = 0;

  pProgram->pEX->Command = COMMANDS;

  ex_init(pProgram->pEX);

  ex_Command_l(pProgram->pEX,&CommandList);

  if( pProgram->pEX->iErrorCounter)return pProgram->pEX->iErrorCounter;

  pProgram->pEX->pCommandList = CommandList;
  return 0;
  }

/*POD
=section BuildCode
=H Build code from the syntax analysis result

/*FUNCTION*/
int scriba_BuildCode(pSbProgram pProgram
  ){
/*noverbatim
CUT*/

  pProgram->pBUILD = alloc_Alloc( sizeof(BuildObject) , pProgram->pMEM );
  if( pProgram->pBUILD == NULL )return 1;

  pProgram->pBUILD->memory_allocating_function = pProgram->maf;
  pProgram->pBUILD->memory_releasing_function  = pProgram->mrf;
  pProgram->pBUILD->pEx =   pProgram->pEX;
  pProgram->pBUILD->iErrorCounter = 0;
  pProgram->pBUILD->fErrorFlags = pProgram->pEX->fErrorFlags;
  pProgram->pBUILD->FirstUNIXline = pProgram->FirstUNIXline;

  build_Build(pProgram->pBUILD);

  if( pProgram->pBUILD->iErrorCounter )return pProgram->pBUILD->iErrorCounter;
  return 0;
  }

/*POD
=section IsFileBinaryFormat
=H Decide if a file is binary format ScriptBasic code or not

This function decides if a file is a correct binary format ScriptBasic
code file and returns true if it is binary. If the file is a ScriptBasic
source file or an older version binary of ScriptBasic or any other file
it returns zero.

This function just calls the function T<build_IsFileBinaryFormat>

/*FUNCTION*/
int scriba_IsFileBinaryFormat(pSbProgram pProgram
  ){
/*noverbatim
CUT*/

  return  build_IsFileBinaryFormat(pProgram->pszFileName);
  }

/*POD
=section GetCacheFileName
=H Get the name of the cache file

Calculate the name of the cache file for the given source file name and
calculate .

/*FUNCTION*/
int scriba_GetCacheFileName(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
#define FULL_PATH_BUFFER_LENGTH 256
  char *pszCache;
  char *s,*q;
  char CachedFileName[FULL_PATH_BUFFER_LENGTH];

  pszCache = cft_GetString(pProgram->pCONF,"cache");
  if( pszCache == NULL)return SCRIBA_ERROR_FAIL;
  if( strlen(pszCache) >= FULL_PATH_BUFFER_LENGTH )return SCRIBA_ERROR_BUFFER_SHORT;
  strcpy(CachedFileName,pszCache);
  s = CachedFileName + strlen(CachedFileName); /* point to the end of the cache directory */

#ifdef WIN32
/* under Win32 we convert the argv[0] to the full path file name */
  if( GetFullPathName(pProgram->pszFileName,
                      FULL_PATH_BUFFER_LENGTH-strlen(CachedFileName),s,&q)==0 )
    return SCRIBA_ERROR_FAIL;
#else
/* under UNIX we can not convert, but it usually contains the full path */
  if( strlen(pProgram->pszFileName) > FULL_PATH_BUFFER_LENGTH - strlen(CachedFileName) )
    return SCRIBA_ERROR_BUFFER_SHORT;
  strcpy(s,pProgram->pszFileName);
#endif
  /* convert the full path to MD5 digest unique file name */
  uniqfnam(s,s);
  pProgram->pszCacheFileName = alloc_Alloc(strlen(CachedFileName)+1,pProgram->pMEM);
  if( pProgram->pszCacheFileName == NULL )return SCRIBA_ERROR_MEMORY_LOW;
  strcpy(pProgram->pszCacheFileName,CachedFileName);
  return SCRIBA_ERROR_SUCCESS;
  }

/*POD
=section UseCacheFile
=H Test if cache file is usable

/*FUNCTION*/
int scriba_UseCacheFile(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  unsigned long FileTime,CacheTime;

  if( scriba_GetCacheFileName(pProgram) )return SCRIBA_ERROR_MEMORY_LOW;

  FileTime  = file_time_modified(pProgram->pszFileName);
  CacheTime = file_time_modified(pProgram->pszCacheFileName);
  if( FileTime && CacheTime && CacheTime > FileTime &&
      build_IsFileBinaryFormat(pProgram->pszCacheFileName) ){
    alloc_Free(pProgram->pszFileName,pProgram->pMEM);
    pProgram->pszFileName = alloc_Alloc(strlen(pProgram->pszCacheFileName)+1,
                                                               pProgram->pMEM);
    if( pProgram->pszFileName == NULL )return SCRIBA_ERROR_MEMORY_LOW;
    strcpy(pProgram->pszFileName,pProgram->pszCacheFileName);
    return SCRIBA_ERROR_SUCCESS;
    }
  return SCRIBA_ERROR_FAIL;
  }

/*POD
=section SaveCacheFile
=H Save the cache file

/*FUNCTION*/
void scriba_SaveCacheFile(pSbProgram pProgram
  ){
/*noverbatim
CUT*/

  if( pProgram->pszCacheFileName )
    scriba_SaveCode(pProgram,pProgram->pszCacheFileName);
  }

/*POD
=section RunExternalPreprocessor
=H Execute external preprocessors

This function should be called to execute external preprocessors.

This function does almost nothing else but calls the function
R<epreproc/epreproc>.

/*FUNCTION*/
int scriba_RunExternalPreprocessor(pSbProgram pProgram,
                                   char **ppszArgPreprocessor
  ){
/*noverbatim
CUT*/
  int iError;
  char *pszPreprocessedFileName=NULL;

  iError = epreproc(pProgram->pCONF,
                    pProgram->pszFileName,
                    &pszPreprocessedFileName,
                    ppszArgPreprocessor,
                    pProgram->maf,
                    pProgram->mrf);

  /* If there was error then return it. */
  if( iError )return iError;

  /* If there was no error, but there is no need to preprocess. */
  if( pszPreprocessedFileName == NULL )return SCRIBA_ERROR_SUCCESS;

  if( pProgram->pszFileName ){
    alloc_Free(pProgram->pszFileName,pProgram->pMEM);
    pProgram->pszFileName = NULL;
    }

  /* Allocated space for the preprocessed file name and store it in the
     memory segment pProgram->pMEM. */
  pProgram->pszFileName = alloc_Alloc(strlen(pszPreprocessedFileName)+1,pProgram->pMEM);
  if( pProgram->pszFileName == NULL )return SCRIBA_ERROR_MEMORY_LOW;
  strcpy(pProgram->pszFileName,pszPreprocessedFileName);
  pProgram->mrf(pszPreprocessedFileName);
  return SCRIBA_ERROR_SUCCESS;
  }

/*POD
=section SaveCode
=H Save the compiled code of the program into file

/*FUNCTION*/
void scriba_SaveCode(pSbProgram pProgram,
                     char *pszCodeFileName
  ){
/*noverbatim
CUT*/
  build_SaveCode(pProgram->pBUILD,pszCodeFileName);
  }

/*POD
=section SaveCCode
=H Save the compiled C code of the program into file

/*FUNCTION*/
void scriba_SaveCCode(pSbProgram pProgram,
                      char *pszCodeFileName
  ){
/*noverbatim
CUT*/
  build_SaveCCode(pProgram->pBUILD,pszCodeFileName);
  }

/*POD
=section LoadSourceProgram
=H Load a program from source code and compile it

/*FUNCTION*/
int scriba_LoadSourceProgram(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  int iError;

  if( iError = scriba_ReadSource(pProgram) )return iError;
  if( iError = scriba_DoLexicalAnalysis(pProgram) )return iError;
  if( iError = scriba_DoSyntaxAnalysis(pProgram) )return iError;
  if( iError = scriba_BuildCode(pProgram) )return iError;

  /* we can not purge these memory areas sooner because some
     error messages even during build refer to names read
     by the reader and still stored intheir memory heap. */
  scriba_PurgeReaderMemory(pProgram);
  scriba_PurgeLexerMemory(pProgram);
  scriba_PurgeSyntaxerMemory(pProgram);
  return SCRIBA_ERROR_SUCCESS;
  }

static int scriba_PreRun(pSbProgram pProgram){
  int iError;

  if( pProgram->pEXE == NULL ){
    pProgram->pEXE = alloc_Alloc( sizeof(ExecuteObject) , pProgram->pMEM );
    if( pProgram->pEXE == NULL )return SCRIBA_ERROR_MEMORY_LOW;

    pProgram->pEXE->memory_allocating_function = pProgram->maf;
    pProgram->pEXE->memory_releasing_function = pProgram->mrf;
    pProgram->pEXE->reportptr = pProgram->pReportPointer;
    pProgram->pEXE->report   = pProgram->fpReportFunction;
    pProgram->pEXE->fErrorFlags = pProgram->fErrorFlags;

    pProgram->pEXE->pConfig = pProgram->pCONF;
    build_MagicCode(&(pProgram->pEXE->Ver));
    if( iError=execute_InitStructure(  pProgram->pEXE,pProgram->pBUILD) )
      return iError;
    pProgram->pEXE->fpStdouFunction = pProgram->fpStdouFunction;
    pProgram->pEXE->fpStdinFunction = pProgram->fpStdinFunction;
    pProgram->pEXE->fpEnvirFunction = pProgram->fpEnvirFunction;
    pProgram->pEXE->pEmbedder       = pProgram->pEmbedder;
    }else{
    if( iError=execute_ReInitStructure(  pProgram->pEXE,pProgram->pBUILD) )
      return iError;
    }

  pProgram->pEXE->CmdLineArgument = NULL;

  return SCRIBA_ERROR_SUCCESS;
  }

/*POD
=section Run
=H Run the program

Call this function to execute a program. Note that you can call this function
many times. Repetitive execution of the same program will execute the
ScriptBasic code again and again with the global variables keeping their values.

If you want to reset the global variables you have to call R<ResetVariables>.

There is no way to keep the value of the local variables.

The argument T<pszCommandLineArgument> is the command part that is passed to the
BASIC program.
/*FUNCTION*/
int scriba_Run(pSbProgram pProgram,
               char *pszCommandLineArgument
  ){
/*noverbatim
CUT*/
  int iError;


  scriba_PreRun(pProgram);

  pProgram->pEXE->CmdLineArgument = pszCommandLineArgument;  
  execute_InitExecute(pProgram->pEXE,&iError);
  execute_Execute_r(pProgram->pEXE,&iError);
  return iError;
  }

/*POD
=section NoRun
=H Don't run the program

In case the embedding program want to set global variables and
execute subroutines without starting the main program it has to call this
function first. It does all the initializations that are done by
R<Run> except that it does not actually execute the program.

After calling this function the main program may access global variables
and call BASIC functions.

/*FUNCTION*/
int scriba_NoRun(pSbProgram pProgram
  ){
/*noverbatim
CUT*/
  int iError;

  scriba_PreRun(pProgram);
  execute_InitExecute(pProgram->pEXE,&iError);
  return iError;
  }

/*POD
=section ResetVariables
=H Reset the global variables between executions

/*FUNCTION*/
void scriba_ResetVariables(pSbProgram pProgram
  ){
/*noverbatim
CUT*/

  memory_ReleaseVariable(pProgram->pEXE->pMo,pProgram->pEXE->GlobalVariables);
  pProgram->pEXE->GlobalVariables = NULL;
  }

/*POD
=section Call
=H Execute a function or sub

This function can be used to call a function or subroutine. This function does not
get any arguments and does not provide any return value.

/*FUNCTION*/
int scriba_Call(pSbProgram pProgram,
                unsigned long lEntryNode
  ){
/*noverbatim
CUT*/
  int iError;

  execute_ExecuteFunction(pProgram->pEXE,lEntryNode,0,NULL,NULL,&iError);

  return iError;
  }

/*POD
=section CallArg
=H Execute a function or sub with arguments

This function can be used to call a function or subroutine with arguments passed by value.
Neither the return value of the SUB nor the modified argument variables are not accessible 
via this function. T<CallArg> is a simple interface to call a ScriptBasic SUB with argument.

/*FUNCTION*/
int scriba_CallArg(pSbProgram pProgram,
                   unsigned long lEntryNode,
                   char *pszFormat, ...
  ){
/*noverbatim
Arguments

=itemize
=item T<pProgram> is the class variable.
=item T<lEntryNode> is the start node of the SUB
=item T<pszFormat> is a format string that defines the rest of the areguments
=noitemize

The format string is case insensitive. The characters T<u>, T<i>, T<r>, T<b> and T<s> have meaning.
All other characters are ignored. The format characters define the type of the arguments
from left to right.

=itemize
=item T<u> means to pass an T<undef> to the SUB. This format character is exceptional that it does not
consume any function argument.
=item T<i> means that the next argument has to be T<long> and it is passed to the BASIC SUB as an integer.
=item T<r> means that the next argument has to be T<double> and it is passed to the BASIC SUB as a real.
=item T<s> means that the next argument has to be T<char *> and it is passed to the BASIC SUB as a string.
=item T<b> means that the next two arguments has to be T<long cbBuffer> and T<unsigned char *Buffer>.
The T<cbBuffer> defines the length of the T<Buffer>.
=noitemize

Note that this SUB calling function is a simple interface and has no access to the modified values of the argument
after the call or the return value.

If you need any of the functionalities that are not implemented in this function call a more sophisticated
function.

Example:
=verbatim

  iErrorCode = scriba_CallArg(&MyProgram,lEntry,"i i s d",13,22,"My string.",54.12);

=noverbatim

CUT*/
  int iError;
  VARIABLE vArgs;
  va_list marker;
  unsigned long cArgs,i,slen;
  char *s;
  char *arg;

  cArgs = 0;
  if( pszFormat ){
    s = pszFormat;
    while( *s ){
      switch( *s++ ){
        case 'U': /* undef argument */
        case 'u': /* It eats no actual C level caller argument */

        case 'B': /* byte argument   */
        case 'b': /* it eats two arguments: a length and the pointer to the byte stream */

        case 'S': /* string argument */
        case 's':
        case 'I': /* Integer argument */
        case 'i':
        case 'R': /* Real number argument */
        case 'r':
          cArgs ++;
          break;
        default:; /* ignore all non-format characters */
        }
      }
    }

  if( cArgs )
    vArgs = memory_NewArray(pProgram->pEXE->pMo,0,cArgs-1);
  else
    vArgs = NULL;

  if( vArgs ){
    i = 0;
    va_start(marker,pszFormat);
    s = pszFormat;
    while( *s ){
      switch( *s++ ){
        case 'U':
        case 'u':
          vArgs->Value.aValue[i] = NULL;
          i++;
          break;
        case 'B': /* byte stream argument */
        case 'b':
          slen = va_arg(marker, long);
          arg = va_arg(marker, char *);
          if( arg == NULL )arg = "";
          vArgs->Value.aValue[i] = memory_NewString(pProgram->pEXE->pMo,slen);
          memcpy(STRINGVALUE(vArgs->Value.aValue[i]),arg,slen);
          i++;
          break;
        case 'S': /* string argument */
        case 's':
          arg = va_arg(marker, char *);
          if( arg == NULL )arg = "";
          slen = strlen(arg);
          vArgs->Value.aValue[i] = memory_NewString(pProgram->pEXE->pMo,slen);
          memcpy(STRINGVALUE(vArgs->Value.aValue[i]),arg,slen);
          i++;
          break;
        case 'I': /* Integer argument */
        case 'i':
          vArgs->Value.aValue[i] = memory_NewLong(pProgram->pEXE->pMo);
          LONGVALUE(vArgs->Value.aValue[i]) = va_arg(marker, long);
          i++;
          break;
        case 'R': /* Real number argument */
        case 'r':
          vArgs->Value.aValue[i] = memory_NewDouble(pProgram->pEXE->pMo);
          DOUBLEVALUE(vArgs->Value.aValue[i]) = va_arg(marker, double);
          i++;
          break;
        }
      }
    }

  execute_ExecuteFunction(pProgram->pEXE,lEntryNode,cArgs,vArgs ? vArgs->Value.aValue : NULL ,NULL,&iError);
  memory_ReleaseVariable(pProgram->pEXE->pMo,vArgs);
  return iError;
  }

/*POD
=section DestroySbArgs
=H Release memory allocated by NewSbArgs

This function can be used to release the memory used by arguments created by the
function R<NewSbArgs>.

/*FUNCTION*/
void scriba_DestroySbArgs(pSbProgram pProgram,
                          pSbData Args,
                          unsigned long cArgs
  ){
/*noverbatim

Arguments:
=itemize
=item T<pProgram> class variable
=item T<Args> pointer returned by R<NewSbArgs>
=item T<cArgs> the number of arguments pointed by T<Args>
=noitemize

CUT*/
  unsigned long i;

  for( i=0 ; i<cArgs ; i++ )
    if( Args[i].type == SBT_STRING )
      alloc_Free(Args[i].v.s,pProgram->pMEM);
  alloc_Free(Args,pProgram->pMEM);
  }

/*POD
=section NewSbArgs
=H Create argument array

Whenever you want to handle the variable values that are returned by the scriba subroutine
you have to call R<CallArgEx>. This function needs the arguments passed in an array of T<SbDtata> type.

This function is a usefuly tool to convert C variables to an array of T<SbData>

/*FUNCTION*/
pSbData scriba_NewSbArgs(pSbProgram pProgram,
                         char *pszFormat, ...
  ){
/*noverbatim
The arguments passed are 

=itemize
=item T<pProgram> is the class variable
=item T<pszFormat> is the format string
=noitemize

The format string is case insensitive. The characters T<u>, T<i>, T<r>, T<b> and T<s> have meaning.
All other characters are ignored. The format characters define the type of the arguments
from left to right.

=itemize
=item T<u> means to pass an T<undef> to the SUB. This format character is exceptional that it does not
consume any function argument.
=item T<i> means that the next argument has to be T<long> and it is passed to the BASIC SUB as an integer.
=item T<r> means that the next argument has to be T<double> and it is passed to the BASIC SUB as a real.
=item T<s> means that the next argument has to be T<char *> and it is passed to the BASIC SUB as a string.
=item T<b> means that the next two arguments has to be T<long cbBuffer> and T<unsigned char *Buffer>.
The T<cbBuffer> defines the leng of the T<Buffer>.
=noitemize

Example:

=verbatim

pSbData MyArgs;


  MyArgs = scriba_NewSbArgs(pProgram,"i i r s b",13,14,3.14,"string",2,"two character string");
  if( MyArgs == NULL )error("memory alloc");

  scriba_CallArgEx(pProgram,lEntry,NULL,5,MyArgs);

=noverbatim

This example passes five arguments to the ScriptBasic subroutine. Note that the last one is only
two character string, the rest of the characters are ignored.

CUT*/
  va_list marker;
  unsigned long cArgs,i;
  char *s;
  char *arg;
  pSbData p;

  if( pszFormat == NULL )return NULL;

  cArgs = 0;
  s = pszFormat;
  while( *s ){
    switch( *s++ ){
      case 'U': /* undef argument */
      case 'u': /* It eats no actual C level caller argument */

      case 'B': /* byte argument   */
      case 'b': /* it eats two arguments: a length and the pointer to the byte stream */

      case 'S': /* string argument */
      case 's':
      case 'I': /* Integer argument */
      case 'i':
      case 'R': /* Real number argument */
      case 'r':
        cArgs ++;
        break;
      default:; /* ignore all non-format characters */
      }
    }
  p = alloc_Alloc(sizeof(SbData)*cArgs,pProgram->pMEM);
  if( p == NULL )return NULL;  

  i = 0;
  va_start(marker,pszFormat);
  s = pszFormat;
  while( *s ){
    switch( *s++ ){
      case 'U':
      case 'u':
        p[i].type = SBT_UNDEF;
        i++;
        break;
      case 'B': /* byte stream argument */
      case 'b':
        p[i].type = SBT_STRING;
        p[i].size = va_arg(marker, long);
        arg = va_arg(marker, char *);
        if( arg == NULL && p[i].size != 0 ){
          p[i++].type = SBT_UNDEF;
          break;
          }
        p[i].size =  strlen(arg);
        if( p[i].size ){
          p[i].v.s = alloc_Alloc(p[i].size,pProgram->pMEM);
          if( p[i].v.s == NULL ){
            while( i ){
              if( p[i].type == SBT_STRING && p[i].v.s )alloc_Free(p[i].v.s,pProgram->pMEM);
              i--;
              }
            alloc_Free(p,pProgram->pMEM);
            return NULL;
            }
          memcpy(p[i].v.s,arg,p[i].size);
          }else{
          p[i].v.s = NULL;
          }
        i++;
        break;
      case 'S': /* string argument */
      case 's':
        p[i].type = SBT_STRING;
        arg = va_arg(marker, char *);
        if( arg == NULL )arg = "";
        p[i].size = strlen(arg);
        if( p[i].size ){
          p[i].v.s = alloc_Alloc(p[i].size,pProgram->pMEM);
          if( p[i].v.s == NULL ){
            while( i ){
              if( p[i].type == SBT_STRING && p[i].v.s )alloc_Free(p[i].v.s,pProgram->pMEM);
              i--;
              }
            alloc_Free(p,pProgram->pMEM);
            return NULL;
            }
          memcpy(p[i].v.s,arg,p[i].size);
          }else{
          p[i].v.s = NULL;
          }
        i++;
        break;
      case 'I': /* Integer argument */
      case 'i':
        p[i].type = SBT_LONG;
        p[i].v.l = va_arg(marker, long);
        i++;
        break;
      case 'R': /* Real number argument */
      case 'r':
        p[i].type = SBT_DOUBLE;
        p[i].v.d = va_arg(marker, double);
        i++;
        break;
      }
    }

  return p;
  }

/*POD
=section CallArgEx
=H Execute a function or sub with arguments and return value

This is the most sophisticated function of the ones that call a ScriptBasic subroutine.
This function is capable handling parameters to scriba subroutines, and returning the
modified argument variables and the return value.

/*FUNCTION*/
int scriba_CallArgEx(pSbProgram pProgram,
                     unsigned long lEntryNode,
                     pSbData ReturnValue,
                     unsigned long cArgs,
                     pSbData Args
  ){
/*noverbatim
CUT*/
  int iError;
  VARIABLE vArgs;
  VARIABLE vReturn;
  unsigned long i;

  if( cArgs )
    vArgs = memory_NewArray(pProgram->pEXE->pMo,0,cArgs-1);
  else
    vArgs = NULL;

  if( vArgs ){
    for( i = 0 ; i < cArgs ; i ++ ){
      switch( Args[i].type ){
        case SBT_UNDEF:
          vArgs->Value.aValue[i] = NULL;
          break;
        case SBT_STRING:
          vArgs->Value.aValue[i] = memory_NewString(pProgram->pEXE->pMo,Args[i].size);
          memcpy(STRINGVALUE(vArgs->Value.aValue[i]),Args[i].v.s,Args[i].size);
          alloc_Free(Args[i].v.s,pProgram->pMEM);
          break;
        case SBT_LONG: /* Integer argument */
          vArgs->Value.aValue[i] = memory_NewLong(pProgram->pEXE->pMo);
          LONGVALUE(vArgs->Value.aValue[i]) = Args[i].v.l;
          break;
        case SBT_DOUBLE: /* Real number argument */
          vArgs->Value.aValue[i] = memory_NewDouble(pProgram->pEXE->pMo);
          DOUBLEVALUE(vArgs->Value.aValue[i]) = Args[i].v.d;
          break;
        }
      }
    }

  execute_ExecuteFunction(pProgram->pEXE,lEntryNode,cArgs,vArgs ? vArgs->Value.aValue : NULL ,&vReturn,&iError);

  if( iError || vReturn == NULL ){
    ReturnValue->type = SBT_UNDEF;
    }else{
    switch( vReturn->vType ){
      case VTYPE_LONG:
        ReturnValue->v.l = LONGVALUE(vReturn);
        break;
      case VTYPE_DOUBLE:
        ReturnValue->v.d = DOUBLEVALUE(vReturn);
        break;
      case VTYPE_STRING:
        /* we allocate a one byte longer buffer and append a terminating zero */
        ReturnValue->size=STRLEN(vReturn);/* size is w/o the terminating zero */
        ReturnValue->v.s = alloc_Alloc(ReturnValue->size+1,pProgram->pMEM);
        if( ReturnValue->v.s ){
          memcpy(ReturnValue->v.s,STRINGVALUE(vReturn),ReturnValue->size);
          ReturnValue->v.s[ReturnValue->size] = (char)0;
          }
        break;
      default:
        ReturnValue->type = SBT_UNDEF;
        break;
      }
    }

  if( vArgs && ! iError ){
    for( i = 0 ; i < cArgs ; i ++ ){
      if( vArgs->Value.aValue[i] == NULL ){
        Args[i].type = SBT_UNDEF;
        continue;
        }
      switch( vArgs->Value.aValue[i]->vType ){
        case VTYPE_LONG:
          Args[i].v.l = LONGVALUE(vArgs->Value.aValue[i]);
          break;
        case VTYPE_DOUBLE:
          Args[i].v.d = DOUBLEVALUE(vArgs->Value.aValue[i]);
          break;
        case VTYPE_STRING:
          /* we allocate a one byte longer buffer and append a terminating zero */
          Args[i].size=STRLEN(vArgs->Value.aValue[i]);/* size is w/o the terminating zero */
          Args[i].v.s = alloc_Alloc(Args[i].size+1,pProgram->pMEM);
          if( Args[i].v.s ){
            memcpy(Args[i].v.s,STRINGVALUE(vArgs->Value.aValue[i]),Args[i].size);
            Args[i].v.s[Args[i].size] = (char)0;
            }
          break;
        default:
          Args[i].type = SBT_UNDEF;
          break;
        }
      }
    }

  memory_ReleaseVariable(pProgram->pEXE->pMo,vArgs);
  return iError;
  }

/*POD
=section LookupFunctionByName
=H LookupFunctionByName

This function should be used to get the entry point of a function
knowing the name of the function. The entry point should not be treated as a
numerical value rather as a handle and to pass it to functions like
R<CallArgEx>.

/*FUNCTION*/
long scriba_LookupFunctionByName(pSbProgram pProgram,
                                 char *pszFunctionName
  ){
/*noverbatim
CUT*/
  return build_LookupFunctionByName(pProgram->pBUILD,pszFunctionName);
  }

/*POD
=section LookupVariableByName
=H LookupVariableByName

This function can be used to get the serial number of a global variable
knowing the name of the variable.

/*FUNCTION*/
long scriba_LookupVariableByName(pSbProgram pProgram,
                                 char *pszVariableName
  ){
/*noverbatim
CUT*/
  if( pProgram->pBUILD == NULL )return 0;
  return build_LookupVariableByName(pProgram->pBUILD,pszVariableName);
  }

/*POD
=section GetVariableType
=H GetVariableType

Get the type of the value that a variable is currently holding. This
value can be

=itemize
=item T<SBT_UNDEF>
=item T<SBT_DOUBLE>
=item T<SBT_LONG>
=item T<SBT_STRING>
=noitemize

/*FUNCTION*/
long scriba_GetVariableType(pSbProgram pProgram,
                            long lSerial
  ){
/*noverbatim
The argument T<lSerial> should be the serial number of 
the variable as returned by R<LookupVariableByName>.

CUT*/
  if( lSerial <= 0 || lSerial > pProgram->pEXE->cGlobalVariables )return SBT_UNDEF;

  if( pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] == NULL )return SBT_UNDEF;

  switch( pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]->vType ){
    case VTYPE_LONG: return SBT_LONG;
    case VTYPE_DOUBLE: return SBT_DOUBLE;
    case VTYPE_STRING: return SBT_STRING;
    default: return SBT_UNDEF;
    }  
  }

/*POD
=section GetVariable
=H  GetVariable

This function retrieves the value of a variable.
A new T<SbData> object is created and the pointer to it
is returned in T<pVariable>. This memory space is automatically
reclaimed when the program object is destroyed or the function
T<DestroySbData> can be called.

/*FUNCTION*/
int scriba_GetVariable(pSbProgram pProgram,
                       long lSerial,
                       pSbData *pVariable
  ){
/*noverbatim
The argument T<lSerial> should be the serial number of the global variable
as returned by R<LookupVariableByName>.

The funtion returns T<SCRIBA_ERROR_SUCCESS> on success,

T<SCRIBA_ERROR_MEMORY_LOW> if the data cannot be created or

T<SCRIBA_ERROR_FAIL> if the parameter T<lSerial> is invalid. 
CUT*/

  if( lSerial <= 0 || lSerial > pProgram->pEXE->cGlobalVariables )return SCRIBA_ERROR_FAIL;

  if( pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] == NULL ){
    *pVariable = scriba_NewSbUndef(pProgram);
    if( *pVariable )return SCRIBA_ERROR_SUCCESS;
    return SCRIBA_ERROR_FAIL;
    }

  switch( pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]->vType ){
    case VTYPE_LONG:
      *pVariable = scriba_NewSbLong(pProgram,
                                    pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]->Value.lValue);
      if( *pVariable )return SCRIBA_ERROR_SUCCESS;
      return SCRIBA_ERROR_MEMORY_LOW;
    case VTYPE_DOUBLE:
      *pVariable = scriba_NewSbDouble(pProgram,
                                      pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]->Value.dValue);
      if( *pVariable )return SCRIBA_ERROR_SUCCESS;
      return SCRIBA_ERROR_MEMORY_LOW;
    case VTYPE_STRING:
      *pVariable = scriba_NewSbBytes(pProgram,
                                             pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]->Size,
                                             pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]->Value.pValue);
      if( *pVariable )return SCRIBA_ERROR_SUCCESS;
      return SCRIBA_ERROR_MEMORY_LOW;
    default:
      *pVariable = scriba_NewSbUndef(pProgram);
      if( *pVariable )return SCRIBA_ERROR_SUCCESS;
      return SCRIBA_ERROR_FAIL;
    }  
  }


/*POD
=section SetVariable
=H  SetVariable

This function sets the value of a global variable.

/*FUNCTION*/
int scriba_SetVariable(pSbProgram pProgram,
                       long lSerial,
                       int type,
                       long lSetValue,
                       double dSetValue,
                       char *pszSetValue,
                       unsigned long size
  ){
/*noverbatim
The argument T<lSerial> should be the serial number of the global variable
as returned by R<LookupVariableByName>.

The argument T<type> should be one of the followings:

=itemize
=item T<SBT_UNDEF>
=item T<SBT_DOUBLE>
=item T<SBT_LONG>
=item T<SBT_STRING>
=item T<SBT_ZCHAR>
=noitemize

The function uses one of the arguments T<lSetValue>, T<dSetValue> or T<pszSetValue> and
the other two are ignored based on the value of the argument T<type>.

If the value of the argument T<type> is T<SBT_UNDEF> all initialization arguments are ignored and the
global variable will get the value T<undef>.

If the value of the argument T<type> is T<SBT_DOUBLE> the argument T<dSetValue> will be used and the global
variable will be double holding the value.

If the value of the argument T<type> is T<SBT_LONG> the argument T<lSetValue> will be used and the global
variable will be long holding the value.

If the value of the argument T<type> is T<SBT_STRING> the argument T<pszSetValue> 
will be used and the global variable will be long holding the value. The length of the string
should in this case be specified by the variable T<size>.

If the value of the argument T<type> is T<SBT_ZCHAR> the argument T<pszSetValue> 
will be used and the global variable will be long holding the value. The length of the string
is automatically calculated and the value passed in the variable T<size> is ignored. In this case the
string T<pszSetValue> should be zero character terminated.

The funtion returns T<SCRIBA_ERROR_SUCCESS> on success,

T<SCRIBA_ERROR_MEMORY_LOW> if the data cannot be created or

T<SCRIBA_ERROR_FAIL> if the parameter T<lSerial> is invalid. 
CUT*/

  if( lSerial <= 0 || lSerial > pProgram->pEXE->cGlobalVariables )return SCRIBA_ERROR_FAIL;

  if( pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] ){
    memory_ReleaseVariable(pProgram->pEXE->pMo,pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]);
    pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] = NULL;
    }
  if( type == SBT_UNDEF )return SCRIBA_ERROR_SUCCESS;
  switch( type ){
    case SBT_DOUBLE:
         pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] =
           memory_NewDouble(pProgram->pEXE->pMo);
         if( pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] == NULL )return SCRIBA_ERROR_MEMORY_LOW;
         pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]->Value.dValue = dSetValue;
         return SCRIBA_ERROR_SUCCESS;
    case SBT_LONG:
         pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] =
           memory_NewLong(pProgram->pEXE->pMo);
         if( pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] == NULL )return SCRIBA_ERROR_MEMORY_LOW;
         pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]->Value.lValue = lSetValue;
         return SCRIBA_ERROR_SUCCESS;
    case SBT_ZCHAR:
         size = strlen(pszSetValue);
         type = SBT_STRING;
         /* nobreak flow over */
    case SBT_STRING:
         pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] =
           memory_NewString(pProgram->pEXE->pMo,size);
         if( pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1] == NULL )return SCRIBA_ERROR_MEMORY_LOW;
         memcpy(pProgram->pEXE->GlobalVariables->Value.aValue[lSerial-1]->Value.pValue,pszSetValue,size);
         return SCRIBA_ERROR_SUCCESS;

    default: return SCRIBA_ERROR_FAIL;
    }
  }