//
// Copyright 1996-2005 by Manolo Gouy.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// A copy of the GNU General Public License is available
// from http://www.gnu.org/licenses/gpl.txt or by mail from the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.
//
// Please report all bugs and problems to "mgouy@biomserv.univ-lyon1.fr".
//


#include "seaview.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "FL/Fl_Double_Window.H"
#include "FL/Fl_Pack.H"
#include "FL/Fl_Hold_Browser.H"

// myFL_CTRL is the modifier for menu shortcuts (Meta on Apple and Ctrl otherwise)
#if defined(__APPLE__) 
#if	(FL_MAJOR_VERSION*10000 + FL_MINOR_VERSION*100 + FL_PATCH_VERSION) >= 10106
#define myFL_CTRL FL_META
#else //on old fltk versions, Ctrl and Meta keyw are exchanged
#undef FL_CTRL
#define FL_CTRL 0x00400000
#define myFL_CTRL 0x00040000
#endif
#else
#define myFL_CTRL FL_CTRL
#endif


/* allongement maximal prevu pour les seqs suite a edition */
#define MAX_SEQ_ALLONG 5000
#define MINI_MAX_LENGTH  10000 /* pour depart avec fichier vide */

/* 
Purpose
resource	argument	type		default			exemple

Default file format for saving operations
save		-save		{mase, phylip,	mase			phylip
			    clustal, msf, fasta, nexus}

Control of printout options
printoutblock	N/A		integer		10			3
printoutcpl	N/A		integer		80			90
	[this is the paper width, not the # of residues desired on each line]
printoutlpp	N/A		integer		66			90
Name of help file
helpfile	N/A		string		"seaview.help"

Standard coloring of protein sequences (<= 10 colors + white for gaps)
(colors are red, green, yellow, blue, cyan, magenta, salmon, purple, aquamarine,
and dark-gray)
stdcolorgroups	N/A		string		EDQNHRKBZ,ILMV,APSGT,FY,WC
						      BZDE,ACWY,FGHIK,LMNPQRSTV
Alternate coloring of protein sequences
altcolorgroups	N/A		string		\0	AC,DEFGHIK,LMNPQRS,WY,TV

Faster but less smooth writing
fast		-fast		no-value	False

Residues colored on background
inverted	-inverted	no-value	False

N/A		filename
*/	

Fl_Group *create_dna_scroller(SEA_VIEW *view, int x, int y, int w, int h, 
	int dbl_buff);
SEA_VIEW *create_the_form(int double_buffer, int inverted,
		known_format default_format, int numb_dnacolors,
		int *dnacolors, 
		int numb_stdprotcolors, int *stdprotcolors,
		int numb_altprotcolors, int *altprotcolors,
		color_choice curr_color_choice, char *progname, char *movekeys,
		char *win_size, int allow_lower, char *help_file,
		int back_color, int region_back_color, int defaultfontsize);
int prep_custom_colors(int *colors, char *customcolors, 
	int max_colors);
color_choice prep_aa_color_code(char *list_std, char *list_alt, 
	int maxprotcolors, int *numb_stdprotcolors, int *numb_altprotcolors);
void deplacer_grp_seqs(SEA_VIEW *view, int target);
void del_gap_only_sites(SEA_VIEW *view);
void reference_toggle(SEA_VIEW *view, int on);
void handle_mouse(SEA_VIEW *view, int mx, int my, 
	int *p_selecting_seqs, int *p_sel_seq_move, int *p_modifying_segment);
void handle_keyboard(SEA_VIEW *view, unsigned int key, int istext);
void handle_push(SEA_VIEW *view, int mx, int my, int key, 
	int *p_modifying_segment, int *p_selecting_seqs, int *p_sel_seq_move);
int delete_gaps_before(SEA_VIEW *view, int numseq, int numsite, int total);
void set_save_format(SEA_VIEW *view, int val);
void use_initial_file(SEA_VIEW *view, char *masename);
int ask_with_custom_yesno(const char *text, const char *yes, const char *no);

/* external functions */
void draw_comment_lines(Fl_Widget *ob, SEA_VIEW *view);
void get_menu_taille_valeur(Fl_Widget *menu, int *taille, int *valeur);
void init_menu_taille_valeur(Fl_Widget *menu, int taille, int valeur, 
	int offset);
void update_menu_taille_valeur(Fl_Widget *menu, int taille, int valeur, 
	int offset);
char* fl_file_chooser_plus(const char* message, const char* pat, 
	const char* fname, int use_only_button);
char* fl_file_chooser_save_as(const char* message, const char* pat, 
	const char* fname, void *view);
void *load_resources(char *progname);
int save_resources(void);
char *get_res_value(char *name, char *def_value, void *v_db);
int int_res_value(char *name, int def_value, void *v_db);
int bool_res_value(char *name, int def_value, void *v_db);
void minuscules(char *);
known_format what_format(const char *filename);
void custom_callback(Fl_Widget *obj, void *data);
void adjust_menu_edit_modes(SEA_VIEW *view);
extern void mainwin_close_callback(Fl_Widget *form, void *data);
#ifdef __APPLE__
extern char* MAC_file_chooser(const char* message, const char* pat, const char* fname);
int set_res_value(const char *name, const char *value);
extern "C" { 
	void MGinit_apple_events(SEA_VIEW *view);
	void add_apropos(char *progname, void *data);
#if !TARGET_RT_MAC_MACHO
	char *OSXpathtoOS9path(const char *OSXpath);
#else
	char *MG_GetBundleResourcesDir(void);
#endif
	}
#elif defined(WIN32)
extern char *MG_win32_file_chooser(const char* message,
        const char* pattern,
        const char* fname,
        const char* defext,
        bool save, int *pfilterindex);
#endif
extern int pdf_printout(SEA_VIEW *view, const char *filename, 
	int fontsize, int block_size, paperformat pageformat, int vary_only, int ref0, 
	int blackonly);
extern void pdf_options_dialog(SEA_VIEW *view);
extern char* fl_file_chooser(const char* message, const char* pat, const char* fname, int);


/* variables globales */
/* pour controle parametres option printout */
paperformat printout_pageformat;
int printout_block, printout_fontsize, printout_black = FALSE;
int printout_vary = FALSE;
char *f_format_names[] = {"Mase", "Phylip", "Clustal", "MSF", "Fasta", "NEXUS"};
char *f_format_exts[] = {"mase", "phy", "aln", "msf", "fst", "nxs"};
int nbr_formats = sizeof(f_format_names)/sizeof(char *);
int emul_3_buttons;
//char def_stdcolorgroups[] = "EDQNHRKBZ,ILMV,APSGT,FY,WC";
char def_stdcolorgroups[] = "KR,AFILMVW,NQST,HY,C,DE,P,G";
//int def_protcolors[] =
//	{ FL_WHITE, FL_RED, FL_GREEN, FL_YELLOW, FL_BLUE, FL_CYAN,
//	FL_MAGENTA, 9/*salmon*/, 13/*purple*/, 14/*aquamarine*/, 8/*darkgray*/};
int *def_protcolors;
int def_protcolors_rgb[] = 
	{255,255,255, /* white */
	230,51,25,   25,128,230,   25,204,25,   25,179,179,   230,128,128,
	204,77,204,   204,204,0,   230,153,77,
	128,128,128,    100,100,100 };
int max_protcolors = sizeof(def_protcolors_rgb) / sizeof(int) / 3;

#ifdef WIN32
#define LINE_HEIGHT_FACTOR  0.85
#else
#define LINE_HEIGHT_FACTOR  1.
#endif


int main(int argc, char *argv[])
{
char *masename, *progname, *p;
known_format defaultformat;
char **possible_formats; 
static int dnacolors[] =
	{ FL_WHITE, FL_RED, FL_GREEN, FL_YELLOW, FL_BLUE };
int numb_dnacolors = sizeof(dnacolors) / sizeof(int);
int numb_stdprotcolors, numb_altprotcolors, i, back_color, region_back_color;
color_choice curr_color_choice;

static int quick_and_dirty, inverted, allow_lower; 
static char stdcolorgroups[50], altcolorgroups[50];
static char customdnacolors[200];
static char customprotcolors[300];
static char save_format[10];
static char movekeys[5], win_size[12];
static char *help_file;
void *resources;
int defaultfontsize = 
#ifdef WIN32
			12;
#else
			14;
#endif

progname = (char *)"seaview";
resources = load_resources(progname);

possible_formats = (char **)malloc(nbr_formats * sizeof(char *));
for(i= 0; i < nbr_formats; i++) {
	possible_formats[i] = (char  *)malloc( strlen(f_format_names[i]) + 1 );
	strcpy(possible_formats[i], f_format_names[i]);
	minuscules(possible_formats[i]);
	}

/* access to resources */
quick_and_dirty = bool_res_value("fast", 0, resources);
inverted = bool_res_value("inverted", TRUE, resources);
strcpy(save_format, get_res_value("save", possible_formats[NEXUS_FORMAT], resources) );
strcpy(stdcolorgroups, 
	get_res_value("stdcolorgroups", def_stdcolorgroups, resources) );
strcpy(altcolorgroups, get_res_value("altcolorgroups", "", resources) );
printout_block = int_res_value("printoutblock", 10, resources);
printout_fontsize = int_res_value("printoutfontsize", 10, resources);
if(strcmp( get_res_value("printoutpageformat", "A4", resources) , "A4") == 0)
	printout_pageformat = A4;
else printout_pageformat = LETTER;
allow_lower = bool_res_value("lowercase", 0, resources);
#if defined(__APPLE__)
#if TARGET_RT_MAC_MACHO
p = MG_GetBundleResourcesDir();
help_file = (char *)malloc(strlen(p) + 20);
strcpy(help_file, p );
strcat(help_file, "/seaview.help");
#else
help_file = get_res_value("helpfile", 	NULL, 	resources );
#endif
#elif defined(DEFAULT_HELPFILE)   /* to ease FreeBSD port */
help_file = get_res_value("helpfile", 	__XSTRING(DEFAULT_HELPFILE), 	resources );
#else
help_file = get_res_value("helpfile", 	"seaview.help", 	resources );
#endif
strcpy(customdnacolors, get_res_value("dnacolors", "", resources) );
strcpy(customprotcolors, get_res_value("protcolors", "", resources) );
strcpy(movekeys, get_res_value("movekeys", MOVEKEYS_DEFAULT, resources) );
strcpy(win_size, get_res_value("window", 
#if defined(WIN32) || defined(__APPLE__)
	"700x500", 
#else
	"1000x700", 
#endif
	resources) );
#if defined(WIN32) || defined(__APPLE__)
	emul_3_buttons = TRUE;
#else
	emul_3_buttons = bool_res_value("emulate_3_b", FALSE, resources);
#endif
#ifdef WIN32
back_color = FL_DARK1; region_back_color = 43;
#else
back_color = FL_GRAY; region_back_color = FL_DARK2;
#endif
back_color = int_res_value("background", back_color, resources);
region_back_color = int_res_value("sites_background", region_back_color, 
	resources);

/* process custom color settings */
prep_custom_colors(dnacolors, customdnacolors, numb_dnacolors);
int *protcolors = (int *)malloc(sizeof(def_protcolors_rgb) / 3);
def_protcolors = (int *)malloc(sizeof(def_protcolors_rgb) / 3);
for(i=0; i<max_protcolors; i++) def_protcolors[i] = 
	fl_rgb_color(def_protcolors_rgb[3*i], def_protcolors_rgb[3*i+1], def_protcolors_rgb[3*i+2]);

memcpy(protcolors, def_protcolors, sizeof(def_protcolors_rgb) / 3 );
int cur_protcolors = prep_custom_colors(protcolors, customprotcolors, 
	max_protcolors);
/* process resource-read stdcolorgroups and altcolorgroups */
curr_color_choice = prep_aa_color_code(stdcolorgroups, altcolorgroups, 
		cur_protcolors, &numb_stdprotcolors, &numb_altprotcolors);

/* argument processing */
masename = NULL;
i = 1;
while(i < argc) {
	if(argv[i][0] != '-') masename = argv[i];
	if(strcmp(argv[i], "-fast") == 0) quick_and_dirty = TRUE;
	if(strcmp(argv[i], "-inverted") == 0) inverted = TRUE;
	if(strcmp(argv[i], "-save") == 0) {
		if(++i < argc) strcpy(save_format, argv[i]);
		}
	if(strcmp(argv[i], "-fontsize") == 0) {
		if(++i < argc) sscanf(argv[i], "%d", &defaultfontsize);
		}
	i++;
	}
for( i=0; i < nbr_formats; i++) 
	if(strcmp(save_format, possible_formats[i]) == 0) break;
if( i >= nbr_formats ) 
	defaultformat = (known_format)MASE_FORMAT;
else
	defaultformat = (known_format)i;

fl_message_font(FL_HELVETICA_BOLD, FL_NORMAL_SIZE );

SEA_VIEW *view = create_the_form(!quick_and_dirty, inverted,
	defaultformat, numb_dnacolors, dnacolors, 
	numb_stdprotcolors, protcolors,
	numb_altprotcolors, protcolors, curr_color_choice, 
	progname, movekeys, win_size, allow_lower, help_file,
	back_color, region_back_color, defaultfontsize);
#ifdef __APPLE__ 
MGinit_apple_events(view);
add_apropos(progname, help_file);
#else
if(masename != NULL) use_initial_file(view, masename);
#endif
view->dnawin->show();
view->DNA_obj->take_focus();
fl_reset_cursor(view->dnawin);
Fl::run(); 
/*
while( Fl::wait() ) { // reparation du bug focus perdu
	if(Fl::focus() == NULL) {
		view->DNA_obj->take_focus();
		}
	} */
return 0;
}


void use_initial_file(SEA_VIEW *view, char *masename)
{
known_format defaultformat;

defaultformat = what_format(masename);
if(defaultformat >= 0) {
		set_save_format(view, defaultformat);
		load_alignment_file(view, masename, NULL, defaultformat);
		}
else
		fl_alert("File %s\nis not of a format readable by seaview", masename);
}


#ifdef _AIX
/* sur IBM RISC __filbuf est en fait _filbuf utilise a l'interieur de xforms */
int __filbuf(FILE *fich)
{
return _filbuf(fich);
}
#endif


class DNA_obj : public Fl_Widget {
    FL_EXPORT void draw(void);
    FL_EXPORT int handle(int);
public:
    FL_EXPORT DNA_obj(int x,int y,int w,int h, void *view) :
		Fl_Widget(x,y,w,h,NULL) {
	this->user_data(view);
	}
};


char *majuscules(char *p)
{
char *q = p;
if(p == NULL) return NULL;
while(*q != 0) { *q = toupper(*q); q++; }
return p;
}


void out_of_memory(void)
{
fl_alert("Error: Not enough memory!");
exit(1);
}


void minuscules(char *p)
{
if(p == NULL) return;
while(*p) {
	*p = tolower(*p);
	p++;
	}
return;
}


char *extract_filename(char *fname)
{
static char *p, *q;
q = (char *)fname;
if(q == NULL) return "";
do	{ 
#ifdef __VMS
	p = strchr(q,']'); 
	if(p == NULL) p = strchr(q,':'); 
#elif defined(WIN32)
	p = strchr(q,'\\'); 
#elif defined(__APPLE__) && !TARGET_RT_MAC_MACHO
	p = strchr(q,':'); 
#else
	p = strchr(q,'/'); 
#endif
	if(p != NULL) q = p+1; 
	}
while 	(p != NULL);
return q;
}


void allonge_seqs(char **seq, int totseqs, int maxlen, int *eachlength,
	int tot_comment_lines, char **comment_line)
{
int num, l;
char *newseq;

for(num = 0; num < totseqs; num++) {
	l = eachlength[num];
	newseq = (char *)malloc(maxlen+1);
	if(newseq==NULL) out_of_memory();
	memcpy(newseq,seq[num],l+1);
	free(seq[num]); 
	seq[num] = newseq;
	}
for(num = 0; num < tot_comment_lines; num++) {
	l = strlen(comment_line[num]);
	newseq = (char *)malloc(maxlen+1);
	if(newseq == NULL) out_of_memory();
	memcpy(newseq, comment_line[num], l+1);
	free(comment_line[num]); 
	comment_line[num] = newseq;
	}
}


/* memoire pour contenir coloriage standard */
static int color_for_aa_gaps;
static char std_aminoacids[30];
static int std_aa_color[30];
/* memoire pour contenir coloriage alternatif */
static char alt_aminoacids[30] = "";
static int alt_aa_color[30];
/* pointeurs vers coloriage courant */
static char *current_aminoacids;
static int *current_aa_color;


int decode_color_scheme(char *vlist,  char *aminoacids, int *aa_color,
	int maxprotcolors)
{
int nbr_colors = 1, current = 0, i;
char *p, *list;
list = strdup(vlist);
if(list == NULL) return 1;
aminoacids[0] = 0;
p=strtok(list,",");
while( p!= NULL && nbr_colors < maxprotcolors) {
	strcat(aminoacids,p);
	for(i=0; i < (int) strlen(p); i++)
		aa_color[current++] = nbr_colors;
	nbr_colors++;
	p=strtok(NULL,",");
	}
free(list);
return nbr_colors;
}


int prep_custom_colors(int *colors, char *customcolors, 
	int max_colors)
{
char *nom;
int rank = 0, r, g, b;
Fl_Color col;

if(*customcolors == 0) return max_colors;
nom = strtok(customcolors, ",");
while(nom != NULL && rank < max_colors - 1) {
	r = g = b = -1;
	sscanf(nom, "%d%d%d", &r, &g, &b);
	if( r>=0 && g>=0 && b>=0 ) 
		col = fl_rgb_color(r, g, b);
	else col = FL_BLACK;
	colors[++rank] = (int)col;
	nom = strtok(NULL, ",");
	}
return rank + 1;
}


color_choice prep_aa_color_code(char *list_std, char *list_alt, 
	int maxprotcolors, int *numb_stdprotcolors, int *numb_altprotcolors)
{
/* couleur pour gaps = 1ere couleur connue (comptee a partir de 0) */
color_for_aa_gaps = 0;
current_aminoacids = std_aminoacids;
current_aa_color = std_aa_color;
/* decodage du coloriage standard des proteines */
*numb_stdprotcolors = 
	decode_color_scheme(list_std, std_aminoacids, std_aa_color,
		maxprotcolors);
*numb_altprotcolors = 0;
if(*list_alt == 0) return NO_ALT_COLORS;
/* decodage du coloriage alternatif des proteines */
*numb_altprotcolors = 
	decode_color_scheme(list_alt, alt_aminoacids, alt_aa_color,
		maxprotcolors);
return USING_STANDARD_COLORS;
}


void set_aa_color_mode(color_choice choice)
{
if(choice == USING_ALT_COLORS) {
	current_aminoacids = alt_aminoacids;
	current_aa_color = alt_aa_color;
	}
else	{
	current_aminoacids = std_aminoacids;
	current_aa_color = std_aa_color;
	}
}


int get_color_for_aa( int key )
/* returns the color # used to display character key in protein */
{
char *pos;
pos = strchr(current_aminoacids, toupper(key) ); 
return ( pos == NULL ? 
	color_for_aa_gaps : current_aa_color[pos - current_aminoacids] );
}


int get_color_for_base( int key )
/* returns the color # used to display character key in DNA */
{
static char bases[]="ACGTU";
static int base_color[] = {1,2,3,4,4};
char *pos;
pos=strchr(bases, toupper(key) ); 
return ( pos == NULL ? 0 : base_color[pos - bases] );
}


char ***prepcolseqs(char **seq, int totseqs, int maxlen, int *eachlength,
	int (*calc_color_function)( int ), int numb_gc, int allow_lower)
{
int num, l, col, i, res;
char ***colseq;
if(totseqs == 0 || numb_gc == 1) return NULL;
colseq=(char ***)malloc(totseqs*sizeof(char **));
if(colseq == NULL) out_of_memory();
for(num=0; num<totseqs; num++) {
	colseq[num] = (char **)malloc(numb_gc * sizeof(char *));
	if(colseq[num] == NULL) out_of_memory();
	l = eachlength[num];
	for(col=0; col < numb_gc; col++) {
		colseq[num][col]=(char *)malloc(maxlen+1);
		if(colseq[num][col] == NULL) out_of_memory();
		memset(colseq[num][col], ' ', l);
		colseq[num][col][l]=0;
		}
	for(i=0; i<l; i++) {
		res = seq[num][i]; 
		col = calc_color_function( res );
		colseq[num][col][i] = ( allow_lower ? res : toupper(res) );
		}
	}
return colseq;
}


char ***prepcolseqs_by_difference(char **seq, int totseqs, int ref_seq0,
	int maxlen, int *eachlength,
	int (*calc_color_function)( int ), int numb_gc, int allow_lower)
{
int num, l, col, i, res;
char ***colseq;
if(totseqs == 0) return NULL;
colseq = (char ***)malloc(totseqs*sizeof(char **));
if(colseq == NULL) return NULL;
for(num=0; num<totseqs; num++) { /* allocation memoire */
	colseq[num] = (char **)malloc(numb_gc * sizeof(char *));
	if(colseq[num] == NULL) return NULL;
	l = eachlength[num];
	for(col=0; col < numb_gc; col++) {
		colseq[num][col]=(char *)malloc(maxlen+1);
		if(colseq[num][col] == NULL) return NULL;
		memset(colseq[num][col], ' ', l);
		colseq[num][col][l]=0;
		}
	}
for(i=0; i<eachlength[ref_seq0]; i++) { /* coloration seq de reference */
	res = seq[ref_seq0][i];
	col = calc_color_function( res );
	colseq[ref_seq0][col][i] = ( allow_lower ? res : toupper(res) );
	}
for(num=0; num<totseqs; num++) { /* coloration des autres sequences */
	if(num == ref_seq0) continue;
	l = eachlength[num];
	for(i=0; i<l; i++) {
		res = seq[num][i];
		if(toupper(res) != toupper(seq[ref_seq0][i])) {
			col = calc_color_function( res );
			colseq[num][col][i] =(allow_lower ? res : toupper(res));
			}
		else	{
			colseq[num][0][i] = '.';
			}
		}
	}
return colseq;
}


void draw_cursor(Fl_Widget *ob, int on_off, int site, int seq, 
	int cursor_in_comment)
{
SEA_VIEW *view = (SEA_VIEW *)ob->user_data() ;
int x, y, cursor_x, cursor_y, c, max_curs_coord;
char *debut, *fin;
int background, foreground;
static char lettre[]="A";
static char cursor_coord[100];

if( (!cursor_in_comment) && (seq == 0 || site > view->each_length[seq-1] + 1) ) 
	return;
cursor_x = site - view->first_site;
if(cursor_in_comment) {
	if(view->tot_comment_lines == 0 || !view->show_comment_lines) return;
	if(cursor_x < 0 || cursor_x >= view->tot_sites ) return;
	if(seq <= 0 || seq > view->tot_comment_lines ) return;
	y = view->y_seq + (seq + view->pos_first_comment_line - 2) * 	
		view->line_height;
	x = view->x_seq + cursor_x * view->char_width;
	lettre[0] = view->comment_line[seq-1][site-1];
	}
else	{
	if(view->tot_seqs == 0) return;
	cursor_y = seq - view->first_seq;
	if(cursor_x < 0 || cursor_x >= view->tot_sites ) return;
	if(cursor_y < 0 || cursor_y >= view->tot_lines ) return;
	y = view->y_seq + cursor_y * view->line_height;
	x = view->x_seq + cursor_x * view->char_width;
	lettre[0] = view->sequence[seq-1][site-1];
	}
if(view->numb_gc == 1) { /* cas noir et blanc */
	if(on_off) {
		foreground = FL_WHITE;
		background = FL_BLACK;
		}
	else	{
		foreground = FL_BLACK;
		background = FL_WHITE;
		}
	}
else if( (!cursor_in_comment) && view->inverted_colors) { 
	/* cas inverted colors */
	if(site <= view->each_length[seq - 1]) {
		for(c=0; c<view->numb_gc; c++)
			if(view->col_seq[seq-1][c][site-1] != ' ') break;
		if(c == 0) background = ob->color();
		else background = view->curr_colors[c];
		}
	else
		background = ob->color();
	if(on_off) {
		foreground = background;
		background = FL_BLACK;
		}
	else	{
		if(view->active_region == NULL || 
				view->region_line[site - 1] == 'X')
			foreground = FL_BLACK;
		else
			foreground = ob->selection_color();
		}
	}
else if(cursor_in_comment) { /* dans les lignes comment sur ecran couleur */
	if(on_off) {
		foreground = FL_WHITE; background = FL_BLACK;
		}
	else	{
		foreground = FL_BLACK; background = ob->color();
		}
	}
else	{ /* cas colored letters */
	if( site <= view->each_length[seq - 1]) {
		for(c=0; c<view->numb_gc; c++)
			if(view->col_seq[seq-1][c][site-1] != ' ') break;
		lettre[0] = view->col_seq[seq-1][c][site-1];
		foreground = view->curr_colors[c];
		}
	else	{
		lettre[0] = ' ';
		foreground = FL_BLACK;
		}
	if(on_off) {
		background = FL_BLACK;
		}
	else	{
		if( (!cursor_in_comment) && view->active_region != NULL && 
				view->region_line[site - 1] == 'X')
			background = ob->selection_color();
		else
			background = ob->color();
		}
	}
fl_font(ob->labelfont(), ob->labelsize());
fl_color(background);
fl_rectf( x, y - view->line_height + fl_descent(), 
	view->char_width, view->line_height);
fl_color(foreground); 
fl_draw(lettre, 1, x, y);
if(on_off) {
/* ecriture des coordonnees du curseur */
	max_curs_coord = view->tot_sites - 12;
	cursor_coord[max_curs_coord] = 0;
	x = view->x_seq + 6 * view->char_width;
	y = view->y_seq - view->line_height;
	fl_font(FL_COURIER, ob->labelsize());
	fl_color(ob->color());
	fl_rectf( x, y - view->line_height, 
		(max_curs_coord ) * view->char_width, 
		view->line_height + fl_descent());
	if(!cursor_in_comment) {
		debut = view->sequence[seq-1] - 1; fin = debut + site; c = 0;
		if(*fin == 0) fin--;
		while(++debut <= fin) if( *debut != '-' ) c++;	
		sprintf(cursor_coord, "Seq:%d Pos:%d|%d [%s]", seq, 
			site, c, view->seqname[seq - 1]);
		}
	else
		sprintf(cursor_coord, "Pos:%d", site);
	fl_color(view->namecolor); 
	fl_draw(cursor_coord, x, y);
	}
}


void draw_seq_names(Fl_Widget *ob, SEA_VIEW *view)
{
int x, y, num, debut, fin;
int couleur;
static char trunc_name[20];

x = view->x_name; y = view->y_name;
fl_font(FL_COURIER, ob->labelsize() );
if(view->draw_names == -1) { /* ecrire tous les noms */
	fl_color(ob->color());
	fl_rectf( ob->x(), ob->y(),  /* fond pour les noms */
		view->x_seq - ob->x(),
		ob->h());
	debut = view->first_seq - 1; 
	fin = view->first_seq - 2 + view->tot_lines;
	if(fin >= view->tot_seqs) fin = view->tot_seqs - 1;
/* write region name */
	if(view->active_region != NULL) {
		fl_color(view->region_color); 
		fl_font(ob->labelfont(), ob->labelsize());
		fl_draw(view->active_region->name, 
		   FL_min(view->wid_names, strlen(view->active_region->name)), 
			view->x_name,
			view->y_name + FL_min(view->tot_lines, view->tot_seqs) *
			view->line_height );
		fl_font(FL_COURIER, ob->labelsize() );
		}
	}
else if(view->draw_names == -2) { /* ecrire tous les noms mais rien qu'eux */
	debut = view->first_seq - 1; 
	fin = view->first_seq - 2 + view->tot_lines;
	if(fin >= view->tot_seqs) fin = view->tot_seqs - 1;
	fl_color(ob->color());
	fl_rectf( ob->x(),   /* fond pour les noms */
		y - view->line_height + fl_descent(), 
		view->x_seq - ob->x(),
		(fin - debut + 1) * view->line_height);
	}
else	{ /* ecrire un seul nom */
	debut = fin = view->draw_names - 1;
	y += view->line_height * (view->draw_names - view->first_seq);
	if( !view->sel_seqs[view->draw_names - 1] ) {
		fl_color(ob->color());
		fl_rectf( x, y - view->line_height + fl_descent(), 
			(view->wid_names+1) * view->char_width, 
			view->line_height);
		}
	}
if(view->tot_seqs > 0) { /* nbre de seqs selectionnees */
	fl_color(ob->color());
	fl_rectf( x, view->y_name - 2 * view->line_height, 
		(view->wid_names+1) * view->char_width, view->line_height);
	sprintf(trunc_name, "sel=%d", view->tot_sel_seqs);
	fl_color(view->namecolor); 
	fl_draw(trunc_name, x, view->y_name - view->line_height );
	}
/* le(s) nom(s) a ecrire */
for(num = debut; num <= fin; num++) {
	if(view->sel_seqs[num]) {
		fl_color(FL_BLACK);
		fl_rectf( x, y - view->line_height + fl_descent(), 
			(view->wid_names+1) * view->char_width, 
			view->line_height);
		couleur = FL_WHITE;
		}
	else
		couleur = view->namecolor;
	fl_color(couleur); 
	fl_draw(view->seqname[num], 
		FL_min(view->wid_names, strlen(view->seqname[num])), x, y );
	y += view->line_height;
	}
}


void draw_header(Fl_Widget *ob, SEA_VIEW *view)
{
int l_line, c;
static char site_line[300];

if(view->mod_seq) return;
if(view->tot_seqs == 0) return;
if(view->active_region != NULL) draw_region_line(ob, view);
/* write site numbers */
l_line = view->tot_sites;
if(l_line < 5) return;
memset(site_line,' ',l_line);
sprintf(site_line,"%d",view->first_site); c=strlen(site_line); site_line[c]=' ';
sprintf(site_line + l_line - 5,"%5d",view->first_site + l_line - 1);
fl_color(ob->color());
fl_rectf( view->x_seq, 
	view->y_seq - 2 * view->line_height + fl_descent(),
	view->tot_sites * view->char_width, view->line_height);
fl_color(view->namecolor); fl_font(FL_COURIER, ob->labelsize() );
fl_draw(site_line, view->x_seq, view->y_seq - view->line_height);
}  


void draw_dna_seqs(Fl_Widget *ob, SEA_VIEW *view)
{
int nline, c, offset, x, y, l_line, y_back, need_back;
char *pos;
int debut, fin, use_region;
list_segments *segment, *first_segment;

if(view->mod_seq == 0) { /* draw all lines */
	debut = view->first_seq - 1;
	fin = FL_min(view->first_seq + view->tot_lines - 1, view->tot_seqs) - 1;
	}
else if(view->mod_seq == -1) { /* draw only selected lines */
	debut = view->first_seq - 1;
	fin = FL_min(view->first_seq + view->tot_lines - 1, view->tot_seqs) - 1;
	while(fin >= debut && !view->sel_seqs[fin]) fin--;
	if(debut > fin) return;
	while(!view->sel_seqs[debut]) debut++;
	}
else	{ /* draw just line # view->mod_seq */
	debut = fin = view->mod_seq - 1;
	if(debut < view->first_seq - 1) fin = debut - 1;
	}
// prepare for drawing region background
use_region = (view->active_region != NULL && view->numb_gc > 1 &&
		(first_segment = view->active_region->list) != NULL);
if(use_region) {
	do	{
		if(first_segment->debut > 
			view->first_site + view->tot_sites - 1) {
				use_region = FALSE;
				break;
			}
		if(first_segment->fin >= view->first_site) break;
		first_segment = first_segment->next;
		}
	while(first_segment != NULL);
	if(first_segment == NULL) use_region = FALSE;
	}
need_back = TRUE;
if(use_region) {
	if(first_segment->debut <= view->first_site &&
		first_segment->fin >= view->first_site + view->tot_sites - 1)
		need_back = FALSE;
	}

/* write sequences */
fl_font( ob->labelfont(), ob->labelsize() );
x = view->x_seq;
y = view->y_seq + (debut - (view->first_seq - 1)) * view->line_height;
y_back = y - view->line_height + fl_descent();
offset = view->first_site - 1;
for(nline = debut; nline <= fin; nline++) {
	if( view->mod_seq != -1 || view->sel_seqs[nline] ) {
		if(need_back) { /* write seq background */
			fl_color(ob->color());
			fl_rectf( x, y_back, 
				view->tot_sites * view->char_width, 
				view->line_height);
			}
		if(use_region) { /* write regions background */
			int deb_block, fin_block;
			fl_color(ob->selection_color());
			segment = first_segment;
			do	{
				deb_block = FL_max(
					segment->debut, view->first_site);
				fin_block = FL_min(segment->fin, 
					view->first_site + view->tot_sites - 1);
				fl_rectf(x+(deb_block - view->first_site)*
						view->char_width, 
					y_back, 
					(fin_block - deb_block + 1) *
						view->char_width, 
					view->line_height);
				segment = segment->next;
				}
			while(segment != NULL && segment->debut <= 
				view->first_site + view->tot_sites - 1);
			}
		if(view->each_length[nline] > offset) {
			l_line = FL_min( view->tot_sites, 
					view->each_length[nline] - offset );
			for(c=0; c < view->numb_gc; c++) { /* write DNA seqs */
				if(view->numb_gc > 1) 
					pos = view->col_seq[nline][c];
				else
					pos = view->sequence[nline];
				fl_color(view->curr_colors[c]);
				fl_draw(pos + offset, l_line, x, y );
				}
			}
		}
	y += view->line_height;
	y_back += view->line_height;
	}
draw_header(ob, view);
}  


void draw_dna_seqs_inverted(Fl_Widget *ob, SEA_VIEW *view)
{
int nline, c, offset, x, y, l_line, xx, yy, firstline, lastline, use_region,
	debut, fin, der_site;
char *lastpos, *pos;
list_segments *segment, *first_segment;
int save_col0;
static char up_line[1000]; char *p, *q; int i;

if(view->tot_seqs == 0) {
	return;
	}
fl_font(ob->labelfont(), ob->labelsize() );
save_col0 = view->curr_colors[0];
view->curr_colors[0] = ob->color();
/* write sequences */
x = view->x_seq;
offset= view->first_site - 1;
if(view->mod_seq <= 0) {
	firstline = view->first_seq-1;
	lastline = FL_min(view->first_seq + view->tot_lines -1, view->tot_seqs);
	}
else	{
	firstline = view->mod_seq - 1;
	lastline = firstline + 1;
	if(firstline < view->first_seq - 1) lastline = firstline - 1;
	}
use_region = (view->active_region != NULL &&
		(first_segment = view->active_region->list) != NULL);
if(use_region) {
	do	{
		if(first_segment->fin >= view->first_site) break;
		first_segment = first_segment->next;
		}
	while(first_segment != NULL);
	}
y = view->y_seq + (firstline - view->first_seq + 1) * view->line_height; 
yy = y - view->line_height + fl_descent();
for(nline=firstline; nline < lastline; /* ecriture des fonds de couleur */
		nline++, y += view->line_height, yy += view->line_height ) {
	l_line = ( offset + view->tot_sites < view->each_length[nline] ? 
		view->tot_sites : view->each_length[nline] - offset);
	l_line = FL_max(l_line, 0); /* nbre de caract a ecrire */
	if(l_line > 0) {
		for(c = 0; c < view->numb_gc; c++) { 
			xx = x; 
			pos = view->col_seq[nline][c] + offset;
			lastpos = pos + l_line;
			fl_color(view->curr_colors[c]);
			while(pos < lastpos) {
				if( *(pos++) != ' ' ) {
					fl_rectf( xx, yy, 
						view->char_width, 
						view->line_height);
					}
				xx += view->char_width;
				}
			}
		}
	if(l_line < view->tot_sites) { /* au dela de la fin de la seq */
		fl_color(ob->color());
	 	fl_rectf( x + l_line * view->char_width, yy,
			view->char_width * (view->tot_sites - l_line), 
			view->line_height);
		}
	if(l_line == 0) continue;
/*ecriture des seqs: regions en noir et autres en col2 */
	fl_color( use_region ? ob->selection_color() : FL_BLACK );
	p = view->sequence[nline] + offset; 
	if(!view->allow_lower) {
		q = p + l_line; i = 0;
		while( p < q) up_line[i++] = toupper(*(p++));
		p = up_line;
		}
	fl_draw(p, l_line, x, y);
	if(use_region) { /* re-ecrire les regions en noir */
		der_site = view->first_site + l_line - 1;
		segment = first_segment;
		fl_color( FL_BLACK );
		while( segment  != NULL ) {
			if(segment->debut > der_site) break;
			debut = FL_max(segment->debut, view->first_site);
			fin = FL_min(segment->fin, der_site);
			if(view->allow_lower) 
				p = view->sequence[nline] + debut - 1;
			else p = up_line + debut - offset - 1;
			fl_draw(p,
				fin - debut + 1, 
				x + (debut - offset - 1)*view->char_width, y);
			segment = segment->next;
			}
		}
	}
view->curr_colors[0] = save_col0;
draw_header(ob, view);
}


void set_tot_lines(SEA_VIEW *view, int new_val)
{
int l;
double x;
view->tot_lines = new_val;
l = view->tot_seqs - view->tot_lines + 1;
if(l<1) l=1;
if(view->first_seq > l) {
	view->first_seq = l;
	}
if(view->tot_seqs > 0) {
	x = view->tot_lines / ( (double) view->tot_seqs ); 
	if(x>1) x=1;
	}
else	x = 1;
view->vertsli->slider_size(x);
view->vertsli->bounds(1, l);
view->vertsli->Fl_Slider::value(view->first_seq);
}


int compute_size_params(SEA_VIEW *view, int force_recompute)
{
static int old_w = 0;
static int old_h = 0;
Fl_Widget *ob = view->DNA_obj;
double x;
int l, tot_lines, possible_lines;

if( !force_recompute && ob->w() == old_w && ob->h() == old_h) return FALSE;
old_w = ob->w(); old_h = ob->h();
view->x_name = view->char_width/2 + ob->x();
view->y_name =  2 * view->line_height + ob->y();
view->x_seq = (view->wid_names +2) * view->char_width + ob->x();
view->y_seq = view->y_name;
view->tot_sites = ( ob->w() - view->x_seq + ob->x() ) / view->char_width;
possible_lines = ( ob->h() - 4 ) / view->line_height  - 1;
tot_lines = possible_lines;
if(view->active_region != NULL) tot_lines--;
if(view->show_comment_lines) {
	tot_lines -= view->tot_comment_lines;
	view->pos_first_comment_line = 
		FL_min(tot_lines, view->tot_seqs) + 1;
	if(view->active_region != NULL) ++(view->pos_first_comment_line);
	}
/* init sliders bounds and size */
if(view->tot_seqs > 0) {
	x = ( (double) view->tot_sites ) / ( view->seq_length + 3 ); 
	if(x>1) x=1;
	}
else	x = 1;
view->horsli->slider_size(x);
l = view->seq_length - view->tot_sites+3;
if(l<1) l=1;
view->horsli->bounds(1,l);
if(view->first_site > l) {
	view->first_site = l;
	}
view->horsli->Fl_Slider::value(view->first_site);
set_tot_lines(view, tot_lines);
if(view->tot_seqs + 
	(view->show_comment_lines ? view->tot_comment_lines : 0) + 
	(view->active_region != NULL ? 1 : 0) < possible_lines) {
/* forcer effacement de tout l'alignement pour ne pas garder 
de mauvaises lignes regions ou footers en bas */
	ob->parent()->redraw();
	}
return TRUE;
}


void init_dna_scroller(SEA_VIEW *view, int totseqs, const char *masename, 
	int protein, char *header)
{
int i, l;
list_regions *mylregion;
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)view->bouton_props;
props_menu_parts *props_parts = 
		(props_menu_parts *)menu_props->user_data();
Fl_Menu_Item *item;

view->header = header;
view->tot_seqs = totseqs;
view->seq_length = 0;
view->protein = protein;
view->allow_seq_edit = FALSE;
item = (Fl_Menu_Item *)menu_props->menu() + props_parts->edit;
item->activate(); item->clear();
item = (Fl_Menu_Item *)menu_props->menu() + props_parts->keys;
item->clear(); (item + 1)->clear(); (item + 2)->clear(); 
item = (Fl_Menu_Item *)menu_props->menu() + props_parts->keys - 1;
if(protein) item->deactivate();
else item->activate();
item = (Fl_Menu_Item *)menu_props->menu() + props_parts->reference;
item->clear();
item->deactivate();
view->hjkl = FALSE;
view->modif_but_not_saved = FALSE;
if(masename != NULL) {
	view->masename = (char *)malloc(strlen(masename)+1);
	if(view->masename == NULL) out_of_memory();
	strcpy(view->masename, masename);
	}
else	view->masename = NULL;
if(totseqs > 0) {
	view->each_length = (int *)malloc(totseqs * sizeof(int));
	if(view->each_length == NULL) out_of_memory();
	}
for(i = 0; i < totseqs; i++) {
	l = strlen(view->sequence[i]);
	view->each_length[i] = l;
	if(l > view->seq_length) view->seq_length = l;
	}
for(i = 0; i < view->tot_comment_lines; i++) {
	l = strlen(view->comment_line[i]);
	if(l > view->seq_length) view->seq_length = l;
	}
view->max_seq_length = FL_max(view->max_seq_length, 
	view->seq_length + MAX_SEQ_ALLONG);
view->wid_names = 0;
for(i=0; i<totseqs; i++) {
	l=strlen(view->seqname[i]);
	while(l > 0 && view->seqname[i][l-1] == ' ') l--; 
	view->seqname[i][l] = 0;
	if(l > view->wid_names) view->wid_names = l;
	}
if(totseqs > 0 && view->wid_names < 10) view->wid_names = 10;
if(view->wid_names > 20) view->wid_names = 20;
allonge_seqs(view->sequence, totseqs, view->max_seq_length, 
	view->each_length, view->tot_comment_lines, view->comment_line);
if(view->numb_dnacolors > 1 && view->bouton_props != NULL) { 
	/* DNA/prot modes useful only with colors */
	item = (Fl_Menu_Item *)menu_props->menu() + props_parts->colors;
	if(view->alt_colors != NO_ALT_COLORS) (item+2)->activate();
	if(protein) {
		if(view->alt_colors == USING_ALT_COLORS) {
			(item+2)->setonly();
			set_aa_color_mode(USING_ALT_COLORS);
			view->numb_gc = view->numb_altprotcolors;
			view->curr_colors = view->altprotcolors;
			}
		else	{
			(item+1)->setonly();
			set_aa_color_mode(USING_STANDARD_COLORS);
			view->numb_gc = view->numb_stdprotcolors;
			view->curr_colors = view->stdprotcolors;
			}
		}
	else	{
		item->setonly();
		view->numb_gc = view->numb_dnacolors;
		view->curr_colors = view->dnacolors;
		}
	}
if(view->numb_gc > 1) {
	view->col_seq = prepcolseqs(view->sequence, totseqs, 
		view->max_seq_length, view->each_length, 
		( view->protein ? get_color_for_aa : get_color_for_base ), 
		view->numb_gc, view->allow_lower);
	view->alt_col_seq = NULL;
	}
view->first_seq = view->first_site = 1;
fl_font(view->DNA_obj->labelfont(), view->DNA_obj->labelsize());
view->line_height = (int)( LINE_HEIGHT_FACTOR * fl_height() );
view->char_width = (int)fl_width('W');
view->draw_names = -1;
view->mod_cursor = FALSE;
view->mod_seq = 0;
view->multipl->argument(0);
if(view->tot_seqs == 0)
	view->cursor_seq = view->cursor_site = 0;
else
	view->cursor_seq = view->cursor_site = 1;
view->old_cursor_seq = view->cursor_seq;
view->old_cursor_site = view->cursor_site;
view->cursor_in_comment = view->old_cursor_in_comment = FALSE;
view->tot_sel_seqs = 0;
if(view->menu_edit != NULL) {
	Fl_Menu_ *obj = (Fl_Menu_ *)(view->menu_edit);
	Fl_Menu_Item *items = (Fl_Menu_Item *)obj->menu();
	items[RENAME_SEQ].deactivate();
	items[EDIT_COMMENTS].deactivate();
	items[DUPLICATE_SEQ].deactivate();
	items[DELETE_SEQ].deactivate();
	items[COMPLEMENT_SEQ].deactivate();
	items[REVERSE_SEQ].deactivate();
	items[EXCHANGE_UT].deactivate();
	items[ALIGN_SEQS].deactivate();
	items[DOT_PLOT].deactivate();
	items[CONSENSUS_SEQ].deactivate();
	items[DELETE_GAP_ONLY_SITES].activate();
	}
if(view->tot_seqs > 0) {
	view->sel_seqs = (int *)calloc(view->tot_seqs, sizeof(int));
	if(view->sel_seqs == NULL) out_of_memory();
	}
/* initialiser les regions et leur menu avec pas de region active */
view->mod_region_line = FALSE;
if(view->tot_seqs > 0) {
	view->region_line = (char *)malloc(view->max_seq_length + 1);
	if(view->region_line == NULL) out_of_memory();
	}
view->region_length = 0;
view->active_region = NULL;
Fl_Menu_ *menu_regions = (Fl_Menu_ *)view->menu_regions;
if(menu_regions != NULL) {
	menu_regions->clear();
	if(view->tot_seqs > 0) {
		menu_regions->add("Create set", 0, NULL, NULL, 0);
		menu_regions->add("Save set", 0, NULL, NULL, FL_MENU_INACTIVE);
		menu_regions->add("Rename set", 0, NULL, NULL,FL_MENU_INACTIVE);
		menu_regions->add("Hide set", 0, NULL, NULL, FL_MENU_INACTIVE);
		menu_regions->add("Delete set", 0, NULL, NULL, 
			FL_MENU_INACTIVE | FL_MENU_DIVIDER);
		}
	mylregion = view->regions;
	i = 0;
	while(mylregion != NULL) {
		i++;
		menu_regions->add(mylregion->element->name, 0, NULL, NULL, FL_MENU_RADIO);
		mylregion = mylregion->next;
		}
	init_menu_taille_valeur(menu_regions, i, 0, 5);
	}
Fl_Menu_ *menu_species = (Fl_Menu_ *)view->menu_species;
if(menu_species != NULL) {
	menu_species->clear();
	if(view->tot_seqs > 0) {
		menu_species->add("Create group", 0, NULL, NULL, FL_MENU_INACTIVE);
		menu_species->add("Delete group", 0, NULL, NULL, FL_MENU_INACTIVE | FL_MENU_DIVIDER);
		}
	for(i=0; i < view->numb_species_sets; i++) {
		menu_species->add(view->name_species_sets[i], 0, NULL, NULL, FL_MENU_RADIO);
		}
	init_menu_taille_valeur(menu_species, view->numb_species_sets, 0, 2);
	}
view->show_comment_lines = (view->tot_comment_lines > 0);
view->active_comment_line = 0;
update_menu_footers(view);
compute_size_params(view, TRUE);
view->DNA_obj->take_focus();
}


void mod_multipl(SEA_VIEW *view, int newval)
{
Fl_Widget *obj = view->multipl;
static char label[16];
int old = obj->argument();

if(newval == 0) sprintf(label, "%s-+_", view->movekeys);
else	sprintf(label, "mult=%d", newval);
obj->argument(newval);
obj->label(label);
if(newval != old) obj->redraw();
}


void rename_sequence(SEA_VIEW *view)
{
char *new_name;
const char *rep;
int num;
if(view->tot_sel_seqs != 1) return; /* only by security, should not happen */
num = 0;
while( ! view->sel_seqs[num] ) num++;
rep = fl_input("Rename:",view->seqname[num]);
if(rep == NULL) return;
new_name = (char *)malloc(strlen(rep)+1);
if(new_name == NULL) return;
free(view->seqname[num]);
strcpy(new_name, rep);
view->seqname[num] = new_name;
if(num+1 >= view->first_seq && num+1 <= view->first_seq+view->tot_lines-1) {
	view->draw_names = num + 1;
	view->DNA_obj->damage(1);
	}
view->modif_but_not_saved = TRUE;
}


void del_gap_only_sites(SEA_VIEW *view)
{
int position, numseq, allgaps, inrun, debut, count;

view->seq_length = 0;
for(numseq = 0; numseq < view->tot_seqs; numseq++) 
	if(view->each_length[numseq] > view->seq_length)
		view->seq_length = view->each_length[numseq];
inrun = FALSE;
position = -1;
while( ++position <= view->seq_length) {
	for(numseq = 0; numseq < view->tot_seqs; numseq++) 
		if(position < view->each_length[numseq] &&
			view->sequence[numseq][position] != '-') break;
	allgaps = (numseq >= view->tot_seqs);
	if(position >= view->seq_length) allgaps = FALSE;
	if(inrun == allgaps) continue;
	if(allgaps && !inrun) {
		inrun = TRUE;
		debut = position;
		}
	else	{
		inrun = FALSE;
		count = position - debut;
		for(numseq = 1; numseq <= view->tot_seqs; numseq++)
			delete_gaps_before(view, numseq, debut + count + 1, 
				count);
		view->seq_length -= count;
		if(view->cursor_site > position) view->cursor_site -= count;
		if(view->regions != NULL) delete_region_part(view, 
			debut + count + 1, count);
		if(view->tot_comment_lines > 0)	delete_in_all_comments(count, 
			debut + count + 1, view);
		position -= count;
		}
	}
view->seq_length = 0;
for(numseq = 0; numseq < view->tot_seqs; numseq++) 
	if(view->each_length[numseq] > view->seq_length)
		view->seq_length = view->each_length[numseq];	
}


int delete_selected_seqs(SEA_VIEW *view)
{
void **temp_data;
int new_tot_seqs, i, j, c, *temp_int, numset;
new_tot_seqs = view->tot_seqs - view->tot_sel_seqs;
if(new_tot_seqs > 0) {
	temp_data = (void **)malloc(new_tot_seqs * sizeof(void *));
	if(temp_data == NULL) return TRUE;
	temp_int = (int *)malloc(new_tot_seqs * sizeof(int));
	if(temp_int == NULL) return TRUE;
	}

for(i = 0, j = 0; i<  view->tot_seqs; i++) /* process sequences */
	if( ! view->sel_seqs[i] ) temp_data[j++] = (void *) view->sequence[i];
	else free(view->sequence[i]);
memcpy(view->sequence , temp_data, new_tot_seqs * sizeof(char *) );

if(view->comments != NULL) {
	for(i = 0, j = 0; i<  view->tot_seqs; i++) /* process comments */
		if( ! view->sel_seqs[i] ) temp_data[j++] = 
						(void *) view->comments[i];
		else free(view->comments[i]);
	memcpy(view->comments , temp_data, new_tot_seqs * sizeof(char *) );
	}

for(i = 0, j = 0; i<  view->tot_seqs; i++) /* process seq names */
	if( ! view->sel_seqs[i] ) temp_data[j++] = (void *) view->seqname[i];
	else free(view->seqname[i]);
memcpy(view->seqname , temp_data, new_tot_seqs * sizeof(char *) );

for(i = 0, j = 0; i<  view->tot_seqs; i++) /* process seq lengths */
	if( ! view->sel_seqs[i] ) temp_int[j++]= view->each_length[i];
memcpy(view->each_length , temp_int, new_tot_seqs * sizeof(int) );

if(view->numb_gc > 1) { /* process color-coded sequences */
	for(i = 0, j = 0; i<  view->tot_seqs; i++)
		if( ! view->sel_seqs[i] ) temp_data[j++] = 
						(void *) view->col_seq[i];
		else 	{
			for(c=0; c<view->numb_gc; c++)
				free(view->col_seq[i][c]);
			free(view->col_seq[i]);
			}
	memcpy(view->col_seq , temp_data, new_tot_seqs * sizeof(char **) );
   	}
for(numset = 0; numset < view->numb_species_sets; numset++) {
/* process species sets */
	for(i = 0, j = 0; i < view->tot_seqs; i++) { 
		if( ! view->sel_seqs[i] ) 
			temp_int[j++]= view->list_species_sets[numset][i];
		}
	memcpy(view->list_species_sets[numset], temp_int, 
		new_tot_seqs * sizeof(int) );
	}

if(! view->cursor_in_comment) view->cursor_seq = 1;
memset(view->sel_seqs, 0, new_tot_seqs * sizeof(int));
view->tot_seqs = new_tot_seqs;
select_deselect_seq(view, 0);
if(view->menu_species != NULL) {
	numset = 0;
	while(numset < view->numb_species_sets) { /* check for empty sets */
		for(i = 0, j = 0; i < view->tot_seqs; i++) 
			if( view->list_species_sets[numset][i] ) j++;
		if( j == 0 ) { /* set became empty => remove it */
			int taille, valeur;
			free(view->list_species_sets[numset]);
			free(view->name_species_sets[numset]);
			memcpy(view->list_species_sets + numset,
			view->list_species_sets + numset + 1,
			(view->numb_species_sets - numset - 1)*sizeof(int *) );
			memcpy(view->name_species_sets + numset,
			view->name_species_sets + numset + 1,
			(view->numb_species_sets - numset - 1)*sizeof(char *) );
			((Fl_Menu_Button*)view->menu_species)->remove(numset + 3 - 1);
			(view->numb_species_sets)--;
			get_menu_taille_valeur(view->menu_species, &taille,
				&valeur);
			init_menu_taille_valeur(view->menu_species, taille - 1,
				0, 2);
			}
		else	numset++;
		}
	}
view->modif_but_not_saved = TRUE;
return FALSE;
}


void vh_sliders_callback(Fl_Widget *ob, void *data)
{
int old, new_val, page;
int which = ((user_data_plus *)data)->value;
SEA_VIEW *view = (SEA_VIEW *)((user_data_plus *)data)->p;
Fl_Scrollbar *sli = (Fl_Scrollbar *)ob;

new_val = sli->value();
if (which == 0) { /* vertical */
      	old = view->first_seq;
	if( Fl::event_is_click() ) {
		page = view->tot_lines - 2;
		if(page < 1) page = 1;
		if(new_val > old + sli->linesize() ) new_val = old + page;
		else if(new_val < old - sli->linesize() ) new_val = old - page;
		new_val = (int) sli->clamp(new_val);
		((Fl_Valuator *)sli)->value(new_val);
		}
      	view->first_seq = new_val;
	if(old != new_val) view->DNA_obj->redraw();
	}
else	{ /* horizontal */
      	old = view->first_site;
	if( Fl::event_is_click() ) {
		page = view->tot_sites - 20;
		if(page < 10) page = 10;
		if(new_val > old + sli->linesize() ) new_val = old + page;
		else if(new_val < old - sli->linesize() ) new_val = old - page;
		new_val = (int)sli->clamp(new_val);
		((Fl_Valuator *)sli)->value(new_val);
		}
      	view->first_site = new_val;
	view->draw_names = 0;
	if(old != new_val) view->DNA_obj->damage(1);
	}
view->DNA_obj->take_focus();
}


void lrdu_button_callback(Fl_Widget *ob, void *data)
{
Fl_Slider *sli;
int *pval, newval;
int mini, maxi;

int which = ((user_data_plus *)data)->value;
SEA_VIEW *view = (SEA_VIEW *)((user_data_plus *)data)->p;
if(which <= 3) { /* mouvement horizontal */
	sli = view->horsli;
	pval = &view->first_site;
	}
else	{ /* mouvement vertical */
	sli = view->vertsli;
	pval = &view->first_seq;
	}
mini = (int)(sli->minimum());
maxi = (int)(sli->maximum());

if(which == 1 || which == 5) newval = mini;
else if(which == 3 || which == 7) newval = maxi;

if(newval < mini) newval = mini;
if(newval > maxi) newval = maxi;
if(newval != *pval) {
	*pval = newval;
	sli->value(*pval);
	if(which <= 3)  view->draw_names = 0;
	view->DNA_obj->damage(view->draw_names ? FL_DAMAGE_ALL : 1);
	}
view->DNA_obj->take_focus();
}


void font_callback(Fl_Widget *ob, void *data)
{
int taille;
SEA_VIEW *view = ((props_menu_parts *)data)->view;
sscanf( ((Fl_Menu_Button *)ob)->text(), "%d", &taille);
if(view->DNA_obj->labelsize() == taille ) return;
my_watch_cursor(view->dnawin);
view->DNA_obj->labelsize(taille);
fl_font( view->DNA_obj->labelfont(), view->DNA_obj->labelsize() );
view->line_height = (int)( LINE_HEIGHT_FACTOR * fl_height() );
view->char_width = (int)fl_width('W');
compute_size_params(view, TRUE);
view->DNA_obj->parent()->redraw();
fl_reset_cursor(view->dnawin);
view->DNA_obj->take_focus();
}


void save_as_callback(Fl_Widget *ob, void *data)
{
SEA_VIEW *view = ((props_menu_parts *)data)->view;
int debut = ((props_menu_parts *)data)->saveformat;
int reponse =  ((Fl_Menu_Button *)ob)->value();
int previous_format = view->format_for_save + debut;
if(previous_format == reponse) return;
set_save_format(view, reponse - debut);
view->DNA_obj->take_focus();
}


void set_save_format(SEA_VIEW *view, int val)
{
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)view->bouton_props;
props_menu_parts *props_parts = 
		(props_menu_parts *)menu_props->user_data();
Fl_Menu_Item *saveas_popup = 
	(Fl_Menu_Item *)menu_props->menu() + props_parts->saveformat;

if(val >= nbr_formats) return;
if( view->format_for_save == (known_format)val) return;
saveas_popup[val].setonly();
view->format_for_save = (known_format)val;
Fl_Menu_Item *items = (Fl_Menu_Item *) ((Fl_Menu_ *)view->menu_file)->menu();
items[SAVE].deactivate();
if(view->masename != NULL) {
	free(view->masename);
	view->masename = NULL;
	}
}


void colors_callback(Fl_Widget *ob, void *data)
{
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)ob;
props_menu_parts *parts = (props_menu_parts *)data;
SEA_VIEW *view = parts->view;
int reponse =  menu_props->value();
Fl_Menu_Item *item;

/* processing inverted colors */
if(reponse == parts->inverted) {
	int style;
	my_watch_cursor(view->dnawin);
	view->inverted_colors = !view->inverted_colors;
	if(view->inverted_colors) {
		style = FL_COURIER;
		}
	else	{
		style = FL_COURIER_BOLD;
		}
	view->DNA_obj->labelfont(style);
	view->DNA_obj->redraw();
	fl_reset_cursor(view->dnawin);
	item = (Fl_Menu_Item *)menu_props->menu() + parts->reference;
	if( (!view->inverted_colors) && view->tot_sel_seqs == 1) item->activate();
	else item->deactivate();
	return;
	}

/* changing DNA / protein / Alternate protein mode */
enum {DNARNA, Protein, Alt };
int debut = parts->colors;
reponse -= debut;
if(reponse == DNARNA && !view->protein) return;
if(reponse == Protein && view->protein && 
	( view->alt_colors == USING_STANDARD_COLORS ||
	view->alt_colors == NO_ALT_COLORS) ) return;
if(reponse == Alt && view->alt_colors == USING_ALT_COLORS) return;
view->protein = (reponse >= Protein);
my_watch_cursor(view->dnawin);
if(view->numb_gc > 1) { /* free col_seq */
	int num, col;
	for(num = 0; num < view->tot_seqs; num++) {
		for(col = 0; col < view->numb_gc; col++) 
			free(view->col_seq[num][col]);
		free(view->col_seq[num]);
		}
	}
item = (Fl_Menu_Item *)menu_props->menu() + parts->keys - 1;
if(reponse == DNARNA) {
	view->numb_gc = view->numb_dnacolors;
	view->curr_colors = view->dnacolors;
	item->activate();
	adjust_menu_edit_modes(view);
	}
else if(reponse == Protein) {
	set_aa_color_mode(USING_STANDARD_COLORS);
	view->numb_gc = view->numb_stdprotcolors;
	view->curr_colors = view->stdprotcolors;
	item->deactivate();
	adjust_menu_edit_modes(view);
	}
else if(reponse == Alt) {
	set_aa_color_mode(USING_ALT_COLORS);
	view->numb_gc = view->numb_altprotcolors;
	view->curr_colors = view->altprotcolors;
	item->deactivate();
	}
view->col_seq = prepcolseqs(view->sequence, view->tot_seqs, 
	view->max_seq_length, view->each_length, 
	(view->protein ? get_color_for_aa : get_color_for_base),
	view->numb_gc, view->allow_lower);
if(view->alt_colors != NO_ALT_COLORS) {
	if(reponse >= Protein)  view->alt_colors = 
		(color_choice)reponse;
	}
view->DNA_obj->redraw();
fl_reset_cursor(view->dnawin);
}


void allow_edit_callback(Fl_Widget *ob, void *data)
{
props_menu_parts *parts = (props_menu_parts *)data;
SEA_VIEW *view = parts->view;
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)ob;
Fl_Menu_Item *item = (Fl_Menu_Item *)menu_props->menu() + parts->edit;

view->allow_seq_edit = !view->allow_seq_edit;
if(view->allow_seq_edit) 
	item->set();
else
	item->clear();
}


void consensus_allowgaps_callback(Fl_Widget *ob, void *data)
{
int *poption = (int *)data;
Fl_Menu_Item *item = (Fl_Menu_Item *)( ((Fl_Menu_ *)ob)->mvalue() );
*poption = !(*poption);
}


void slow_fast_callback(Fl_Widget *ob, void *data)
{
props_menu_parts *parts = (props_menu_parts *)data;
SEA_VIEW *view = parts->view;
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)ob;
Fl_Menu_Item *item = (Fl_Menu_Item *)menu_props->menu() + parts->slow_fast;
view->double_buffer = !view->double_buffer;
if(!view->double_buffer) 
	item->set();
else
	item->clear();
#ifdef __APPLE__
set_res_value("fast", view->double_buffer ? "false" : "true");
#endif
Fl_Window *win = (Fl_Window *)view->DNA_obj->parent();
Fl_Group *win_parent = (Fl_Group *)win->parent();
int x,y,w,h;
x = win->x(); y = win->y(); w = win->w(); h = win->h();
int size = view->DNA_obj->labelsize();
int font = view->DNA_obj->labelfont();
Fl_Color col = view->DNA_obj->color();
Fl_Color sel_col = view->DNA_obj->selection_color();
win->hide();
win_parent->remove(*win);
delete win;
Fl_Group::current(win_parent);
if(view->double_buffer)	
	win = (Fl_Window *)new Fl_Double_Window(x, y, w, h );
else 	win = new Fl_Window(x,y,w,h);
win->box(FL_DOWN_BOX);
win->resizable(win);
view->DNA_obj = (Fl_Widget*)new DNA_obj(3, 3, w - 6, h - 6, view);
view->DNA_obj->labelfont(font);
view->DNA_obj->labelsize(size);
view->DNA_obj->color(col, sel_col);
win->color(col);
win->end();
win->show();
view->DNA_obj->take_focus();
}


void allow_lower_callback(Fl_Widget *ob, void *data)
{
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)ob;
props_menu_parts *parts = (props_menu_parts *)data;
SEA_VIEW *view = parts->view;
int num, col, pos; char c;
int (*calc_color_function)(int);
Fl_Menu_Item *item = (Fl_Menu_Item *)menu_props->menu() + parts->allow_lower;

view->allow_lower = !view->allow_lower;
if(view->allow_lower) 
	item->set();
else
	item->clear();
my_watch_cursor(view->dnawin);
calc_color_function = view->protein ? get_color_for_aa : get_color_for_base;
if(view->allow_lower) {
	for(num = 0; num < view->tot_seqs; num++) {
		for(pos = 0; pos < view->each_length[num]; pos++) {
			c = view->sequence[num][pos];
			if(islower(c)) {
				col = calc_color_function(c);
			   	view->col_seq[num][col][pos] = c;
				}
			}
		}
	}
else	{
	for(num = 0; num < view->tot_seqs; num++) {
		for(pos = 0; pos < view->each_length[num]; pos++) {
			c = view->sequence[num][pos];
			if(islower(c)) {
				col = calc_color_function(c);
				view->col_seq[num][col][pos] = toupper(c);
				}
			}
		}
	}
view->DNA_obj->parent()->redraw();
fl_reset_cursor(view->dnawin);
}


void toggle_reference_callback(Fl_Widget *ob, void *data)
{
int on, etat;
props_menu_parts *parts = (props_menu_parts *)data;
SEA_VIEW *view = parts->view;
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)ob;
Fl_Menu_Item *item = (Fl_Menu_Item *)menu_props->menu() + parts->edit;
on = (menu_props->menu() + parts->reference)->value();
reference_toggle(view, on);
if(on) 	{
	item->deactivate();
	}
else	{
	item->activate();
	if(view->allow_seq_edit) item->set();
	else item->clear();
	}
}


void dnakeys_callback(Fl_Widget *ob, void *data)
{
props_menu_parts *parts = (props_menu_parts *)data;
SEA_VIEW *view = parts->view;
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)ob;
int reponse =  menu_props->value() - parts->keys;
if( ! view->hjkl ) { /* activer un choix */
	view->hjkl = reponse + 1;
	}
else if(view->hjkl == reponse + 1) { /* desactiver choix courant */
	((Fl_Menu_Item *)menu_props->mvalue())->clear();
	view->hjkl = 0;
	}
else	{ /* changer de choix */
	view->hjkl = reponse + 1;
	}
}


void clustal_opt_callback(Fl_Widget *ob, void *data)
{
props_menu_parts *parts = (props_menu_parts *)data;
SEA_VIEW *view = parts->view;
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)ob;
int reponse =  menu_props->value();
if(reponse == parts->clustalopt) { // toggle use of option
	; // toggle automatique 
	}
else	{ // edit options
	const char *rep;
	int l; 
	rep = fl_input("Clustal options "
#if defined(WIN32) || (defined(__APPLE__) && ! TARGET_RT_MAC_MACHO)
		"(ex: /gapopen=5 /quicktree)", 
#else
		"(ex: -gapopen=5 -quicktree)", 
#endif
		(view->clustal_options == NULL ? "": view->clustal_options) );
	if(rep == NULL) return;
	if(view->clustal_options != NULL) free(view->clustal_options);
	l = strlen(rep);
	while(l > 0 && rep[l-1] == ' ') l--;
	if(l == 0) {
		view->clustal_options = NULL; 
		return;
		}
	view->clustal_options = (char *)malloc(l+1);
	memcpy(view->clustal_options, rep, l);
	view->clustal_options[l] = 0;
	menu_props->replace(parts->clustalopt, view->clustal_options);
	menu_props->mode(parts->clustalopt, FL_MENU_TOGGLE | FL_MENU_VALUE | FL_MENU_DIVIDER);
	}
}


void consensus_opt_callback(Fl_Widget *ob, void *data)
{
props_menu_parts *parts = (props_menu_parts *)data;
SEA_VIEW *view = parts->view;
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)ob;
int reponse =  menu_props->value();
if(reponse != parts->consensusopt + 2) return; // securite
const char *rep;
char tmp[100];
int value;
sprintf(tmp, "%d %%", view->consensus_threshold);
rep = fl_input("Set consensus threshold value in %%", tmp );
if(rep == NULL) return;
value = -1;
sscanf(rep, "%d", &value);
if(value <= 0 || value > 100) return;
view->consensus_threshold = value;
sprintf(tmp, "%d %%", view->consensus_threshold);
menu_props->replace(parts->consensusopt, tmp);
}


void create_props_menu(Fl_Menu_Button *obj, SEA_VIEW *view, 
	int curr_color_choice, int inverted, int black_and_white, 
	int def_fontsize)
{
int defaut, mode;
props_menu_parts *parts =(props_menu_parts *)calloc(1,sizeof(props_menu_parts));
obj->user_data(parts);

parts->view = view;
// Ne pas utiliser d'initialisation directe (obj->menu) car edition impossible

/* partie Fontsize */
obj->add("Fontsize", 0, 0, 0, FL_SUBMENU);
#define NBER_FONTSIZES 10
parts->fontsize = 
	obj->add("Fontsize/8", 0, font_callback, parts, FL_MENU_RADIO);
obj->add("Fontsize/10", 0, font_callback, parts, FL_MENU_RADIO);
obj->add("Fontsize/12", 0, font_callback, parts, FL_MENU_RADIO );
obj->add("Fontsize/14", 0, font_callback, parts, FL_MENU_RADIO );
obj->add("Fontsize/16", 0, font_callback, parts, FL_MENU_RADIO );
obj->add("Fontsize/18", 0, font_callback, parts, FL_MENU_RADIO );
obj->add("Fontsize/24", 0, font_callback, parts, FL_MENU_RADIO );
obj->add("Fontsize/28", 0, font_callback, parts, FL_MENU_RADIO );
obj->add("Fontsize/32", 0, font_callback, parts, FL_MENU_RADIO );
obj->add("Fontsize/36", 0, font_callback, parts, FL_MENU_RADIO );

for(defaut = parts->fontsize; defaut < parts->fontsize + NBER_FONTSIZES; defaut++) {
	int taille;
	sscanf( obj->text(defaut), "%d", &taille);
	if(taille >= def_fontsize) break;
	}
if(defaut >= parts->fontsize + NBER_FONTSIZES) defaut = parts->fontsize + 3;
((Fl_Menu_Item *)obj->menu() + defaut)->set();

/* partie Save Format */
obj->add("Save Format", 0, 0, 0, FL_SUBMENU);
parts->saveformat = obj->add("Save Format/Save as Mase", 0, save_as_callback, 
	parts, FL_MENU_RADIO);
obj->add("Save Format/Save as Phylip",0, save_as_callback, parts,FL_MENU_RADIO);
obj->add("Save Format/Save as Clustal",0,save_as_callback, parts,FL_MENU_RADIO);
obj->add("Save Format/Save as MSF", 0, save_as_callback, parts, FL_MENU_RADIO );
obj->add("Save Format/Save as Fasta", 0, save_as_callback, parts,FL_MENU_RADIO);
obj->add("Save Format/Save as NEXUS", 0, save_as_callback, parts,FL_MENU_RADIO);
defaut = parts->saveformat + view->format_for_save;
((Fl_Menu_Item *)obj->menu() + defaut)->set();

/* partie Colors */
int submenu = obj->add("Colors", 0, 0, 0, FL_SUBMENU);
parts->colors = obj->add("Colors/DNA-RNA colors", 0, colors_callback, parts, 
	FL_MENU_RADIO);
obj->add("Colors/Protein colors",0, colors_callback, parts, FL_MENU_RADIO);
int alt = obj->add("Colors/Alt. colors",0,colors_callback, parts, 
	FL_MENU_RADIO | FL_MENU_DIVIDER );
parts->inverted = obj->add("Colors/Inverted colors", 0, colors_callback, parts, FL_MENU_TOGGLE );
if(black_and_white) {
	((Fl_Menu_Item *)obj->menu() + submenu)->deactivate();
	}
else	{
	defaut = parts->colors; if(view->protein) defaut++;
	((Fl_Menu_Item *)obj->menu() + defaut)->set();
	if(curr_color_choice == NO_ALT_COLORS) 
		((Fl_Menu_Item *)obj->menu() + alt)->deactivate();
	if(inverted) ((Fl_Menu_Item *)obj->menu() + parts->inverted)->set();
	}

/* partie Edit */
parts->edit = obj->add("Allow seq. edition", 0, allow_edit_callback, 
	parts, FL_MENU_TOGGLE);

/* partie by reference */
parts->reference = obj->add("by Reference", 0, toggle_reference_callback, 
	parts, FL_MENU_TOGGLE);
if(black_and_white) 
	((Fl_Menu_Item *)obj->menu() + parts->reference)->deactivate();
else	{
// bug: FL_ALT ne marche pas, c'est Meta qu'il faut taper a la place
	obj->shortcut(parts->reference, myFL_CTRL | 'r');
	}

/* partie keys */
obj->add("DNA keys", 0, 0, 0, FL_SUBMENU);
parts->keys = obj->add("DNA keys/hjkl<space> => GATCN", 0, dnakeys_callback, 
	parts, FL_MENU_RADIO);
obj->add("DNA keys/hjkl<space> => TCGAN", 0, dnakeys_callback, 
	parts, FL_MENU_RADIO);
obj->add("DNA keys/hjkl<space> => ACGTN", 0, dnakeys_callback, 
	parts, FL_MENU_RADIO);

/* partie clustal options */
obj->add("Clustalw options", 0, 0, 0, FL_SUBMENU);
parts->clustalopt = obj->add("Clustalw options/<none>", 0, clustal_opt_callback, 
	parts, FL_MENU_DIVIDER | FL_MENU_INACTIVE);
obj->add("Clustalw options/Edit options", 0, clustal_opt_callback, 
	parts, 0);

/* partie consensus options */
obj->add("Consensus options", 0, 0, 0, FL_SUBMENU);
char tmp[50];
sprintf(tmp, "Consensus options/%d %%", view->consensus_threshold);
parts->consensusopt = obj->add(tmp, 0, NULL, parts, FL_MENU_DIVIDER);
obj->add("Consensus options/allow gaps", 0, consensus_allowgaps_callback, &(view->consensus_allowgaps), FL_MENU_TOGGLE);
obj->add("Consensus options/Edit threshold", 0, consensus_opt_callback, parts, 0);

/* partie Fast/Slow */
parts->slow_fast = obj->add("Fast-Rough", 0, slow_fast_callback, parts,
	view->double_buffer ? FL_MENU_TOGGLE : FL_MENU_TOGGLE | FL_MENU_VALUE );

/* partie allow lower case */
parts->allow_lower = obj->add("Allow lowercase", 0, allow_lower_callback, parts,
	view->allow_lower ? FL_MENU_TOGGLE | FL_MENU_VALUE : FL_MENU_TOGGLE );
if(black_and_white) 
	((Fl_Menu_Item *)obj->menu() + parts->allow_lower)->deactivate();

/* partie customize */
parts->custom = obj->add("Customize", 0, custom_callback, view, 0);
}


void edit_menu_callback(Fl_Widget *ob, long which)
{
SEA_VIEW *view;
int reponse = ((Fl_Menu_Button*)ob)->value();
view = (SEA_VIEW *) ob->user_data();
if(reponse == RENAME_SEQ) { /* rename the selected sequence */
	rename_sequence(view);
	}
else if(reponse == EDIT_COMMENTS) {
	edit_comments_dialog(view);
	}
else if(reponse == DELETE_SEQ) { /* delete selected sequences from alignment */
	char temp[100];
	sprintf(temp,"Confirm request of deletion of %d sequence(s)",
		view->tot_sel_seqs);
	if( ask_with_custom_yesno(temp, "Delete", "Cancel") ) { 
		if( delete_selected_seqs(view) )
			fl_alert("Not enough memory for this operation");
		else	{
			compute_size_params(view, TRUE);
			update_menu_footers(view);
			view->DNA_obj->redraw();
			view->vertsli->redraw();
			}
		}
	}
else if(reponse == CREATE_SEQ) { /* create a new sequence */
	char *newname;
	newname = (char *) fl_input("Name of the new sequence?", "");
	if(newname == NULL || strlen(newname) == 0) return;
	add_seq_to_align(view, newname, "-", 1);
	}
else if(reponse == LOAD_SEQ) { /* load a new sequence */
	load_seq_dialog(view);
	}
else if(reponse == DUPLICATE_SEQ || reponse == COMPLEMENT_SEQ ||
		 reponse == REVERSE_SEQ) { 
	int num, lenseq, old_first_seq;
	char *newseq, *p, *q, newname[100];
	if(view->tot_sel_seqs != 1 ) return; /* par securite */
	if(reponse != DUPLICATE_SEQ && view->protein) return; /* par securite */
	for(num = 0; num < view->tot_seqs; num++)
		if(view->sel_seqs[num]) break;
	lenseq = view->each_length[num];
	newseq = (char *)malloc(lenseq + 1);
	if(newseq == NULL) {
		fl_alert("Not enough memory\nto create a new sequence");
		Fl_Menu_Item *items = (Fl_Menu_Item *) ((Fl_Menu_ *)ob)->menu();
		items[reponse-1].deactivate();
		return;
		}
	if(reponse == DUPLICATE_SEQ) {
		strcpy(newseq, view->sequence[num]);
		}
	else	{
		p = view->sequence[num]; q = newseq + lenseq - 1;
		while( *p != 0) {
			if(reponse == COMPLEMENT_SEQ) *q = complement_base(*p);
			else *q = *p;
			p++; q--;
			}
 		newseq[lenseq] = 0;
		}
	if(reponse == COMPLEMENT_SEQ) strcpy(newname,"C_");
	else if(reponse == REVERSE_SEQ) strcpy(newname,"R_");
	else	strcpy(newname,"D_");
	strcat(newname,view->seqname[num]);
	old_first_seq = view->first_seq;
	add_seq_to_align(view, newname , newseq, lenseq);
	free(newseq);	 
/* placer la nouvelle seq apres celle de depart */
	view->sel_seqs[num] = FALSE;
	view->sel_seqs[view->tot_seqs - 1] = TRUE;
	deplacer_grp_seqs( view, FL_min(num + 2, view->tot_seqs) );
/* montrer les 2 seqs concernees */
	if(old_first_seq > num + 1) view->first_seq = 
		FL_min(num + 1, view->tot_seqs - view->tot_lines + 1);
	else if(old_first_seq + view->tot_lines - 1 < num + 2) 
		view->first_seq = FL_min(num + 2, 
			view->tot_seqs - view->tot_lines + 1);
	else
		view->first_seq = old_first_seq;
	view->vertsli->Fl_Slider::value(view->first_seq);
	}
else if(reponse == EXCHANGE_UT) { /* exchange Us and Ts */
	int num, col; char *p, *q;
	if(view->tot_sel_seqs == 0 || view->protein) return; /* par securite */
	my_watch_cursor(view->dnawin);
	col = get_color_for_base('T');
	for(num = 0; num < view->tot_seqs; num++) {
		if( ! view->sel_seqs[num] ) continue;
		p = view->sequence[num] - 1;
		q = view->col_seq[num][col] - 1;
		while( *(++p) != 0 ) {
			++q;
			if( *p == 'U' ) {
			   	*p = *q = 'T'; 
			   	}
			else if( *p == 'T' ) {
			   	*p = *q = 'U'; 
			   	}
			else if( *p == 't' ) {
			   	*p = 'u';
				*q = (view->allow_lower ? 'u' : 'U'); 
				}
			else if( *p == 'u' ) {
			   	*p = 't';
				*q = (view->allow_lower ? 't' : 'T'); 
			   	}
			}
		}
	view->modif_but_not_saved = TRUE;
	view->draw_names = 0;
	view->DNA_obj->damage(1);
	fl_reset_cursor(view->dnawin);
	}
else if(reponse == ALIGN_SEQS) { /* align selected sites */
	align_selected_parts(view);
	view->DNA_obj->redraw();
	view->horsli->redraw();
	fl_reset_cursor(view->dnawin);
	}
else if(reponse == DOT_PLOT) { /* dot plot */
	int num1, num2;
	extern void show_dot_plot(char *seq1, char *seq2, char *seqname1, 
		char *seqname2, int l1, int l2, int maxseqlength, 
		void *seaview_data);

	if(view->tot_sel_seqs != 2) return;
	for(num1 = 0; num1 < view->tot_seqs; num1++)
		if(view->sel_seqs[num1]) break;
	for(num2 = num1 + 1; num2 < view->tot_seqs; num2++)
		if(view->sel_seqs[num2]) break;
	show_dot_plot(view->sequence[num1], view->sequence[num2],
		view->seqname[num1], view->seqname[num2],
		view->each_length[num1], view->each_length[num2],
		view->max_seq_length, (void *)view);
	}
else if (reponse == CONSENSUS_SEQ) {
	char *newseq, newname[100];
	int old_total, *tmp, new_pos, i, old_first_seq;

	if(view->tot_sel_seqs <= 1 ) return; /* par securite */
	newseq = cre_consensus(view, newname);
	if(newseq == NULL) {
		fl_alert("Not enough memory\nto create a new sequence");
		Fl_Menu_Item *items = (Fl_Menu_Item *) ((Fl_Menu_ *)ob)->menu();
		items[reponse-1].deactivate();
		return;
		}
	old_first_seq = view->first_seq;
	old_total = view->tot_seqs;
	add_seq_to_align(view, newname , newseq, strlen(newseq));
	free(newseq);	
	for(i=0; i < view->tot_seqs; i++)
		if(view->sel_seqs[i]) new_pos = i;
	new_pos += 2;
	if(view->tot_seqs == old_total || new_pos == view->tot_seqs) {
		return; 
		}
/* placer la nouvelle seq apres la derniere des selectionnees */
	tmp = (int *)calloc(view->tot_seqs, sizeof(int));
	if(tmp == NULL) {
		return; 
		}
	memcpy(tmp, view->sel_seqs, view->tot_seqs * sizeof(int) );
	memset(view->sel_seqs, 0, view->tot_seqs * sizeof(int) );
	view->sel_seqs[view->tot_seqs - 1] = TRUE;
	old_total = view->tot_sel_seqs;
	view->tot_sel_seqs = 1;
	deplacer_grp_seqs( view, new_pos );
	memcpy(view->sel_seqs, tmp, view->tot_seqs * sizeof(int) );
	view->tot_sel_seqs = old_total;
	free(tmp);
/* montrer la seq concernee */
	if(old_first_seq > new_pos ) view->first_seq = 
		FL_min(new_pos - 2,  1);
	else if(old_first_seq + view->tot_lines - 1 < new_pos) 
		view->first_seq = FL_min(new_pos - 2, 
			view->tot_seqs - view->tot_lines + 1);
	else
		view->first_seq = old_first_seq;
	view->vertsli->Fl_Slider::value(view->first_seq);
	}
else if(reponse == DELETE_GAP_ONLY_SITES) {
	if( !ask_with_custom_yesno("Confirm remove all gap_containing sites?", "Remove", "Cancel") )  
		return;
	my_watch_cursor(view->dnawin);
	del_gap_only_sites(view);
	compute_size_params(view, TRUE);
	view->DNA_obj->redraw();
	view->horsli->redraw();
	fl_reset_cursor(view->dnawin);
	}
}


void set_and_show_new_cursor_site(SEA_VIEW *view, int new_pos, int center,
	int force_redraw)
{
int old_pos;
old_pos = view->cursor_site;
if(new_pos != old_pos)
	view->cursor_site = new_pos;
if(new_pos >= view->first_site && new_pos < view->first_site +
	view->tot_sites - 1) {
	if( !force_redraw && 
		( (view->cursor_in_comment && view->mod_comment_line == 0) ||
		(!view->cursor_in_comment && view->mod_seq == 0) ) )
		view->mod_cursor = TRUE; 
	}
else	{
	if(center) 
		view->first_site = 
			view->cursor_site - view->tot_sites/2;
	else	{
		if(new_pos >= old_pos) 
			view->first_site = view->cursor_site + 10 - 
				view->tot_sites;
		else
			view->first_site = view->cursor_site - 10;
		}
	if(view->first_site + view->tot_sites - 1 >
		view->seq_length + 1 )
		view->first_site = view->seq_length - view->tot_sites + 2;
	if(view->first_site <=0 ) 
		view->first_site = 1;
	view->horsli->Fl_Slider::value(view->first_site);
	view->mod_seq = 0;
	view->mod_comment_line = 0;
	}
view->draw_names = 0;
view->DNA_obj->damage(1);
}


void set_and_show_new_cursor_seq(SEA_VIEW *view, int new_pos)
{
if(view->cursor_in_comment) {
	if(view->comment_length[new_pos - 1] + 1 < view->cursor_site) {
		fl_ringbell(0); 
		return;
		}
	if(new_pos == view->cursor_seq) return;
	view->cursor_seq = new_pos;
	view->mod_cursor = TRUE;
	view->draw_names = 0;
	view->DNA_obj->damage(1);
	return;
	}
if(view->each_length[new_pos - 1] + 1 < view->cursor_site) {
	fl_ringbell(0); return;
	}
if(new_pos != view->cursor_seq || new_pos < view->first_seq ||
	view->cursor_site != view->old_cursor_site ||
	new_pos >= view->first_seq + view->tot_lines) {
	view->cursor_seq = new_pos;
	if(new_pos >= view->first_seq && new_pos < view->first_seq +
		view->tot_lines) {
		view->mod_cursor = TRUE;
		view->draw_names = 0;
		view->DNA_obj->damage(1);
		}
	else	{
		view->first_seq = view->cursor_seq - view->tot_lines/2;
		if(view->first_seq + view->tot_lines >= 
			view->tot_seqs )
			view->first_seq = view->tot_seqs - view->tot_lines + 1;
		if(view->first_seq <=0 ) 
			view->first_seq = 1;
		view->vertsli->Fl_Slider::value(view->first_seq);
		view->draw_names = -1;
		view->DNA_obj->redraw();
		}
	}
}


void goto_callback(Fl_Widget *ob, void *data)
{
Fl_Input *input;
SEA_VIEW *view;
int l, maxi, num, new_pos = -1, numerique = TRUE;
char *p, *q, target[50];
static char *upname;
static int l_upname = 0;

user_data_plus *data_plus = (user_data_plus *)data;
int which = data_plus->value;
if(which == 0) { /* appele par bouton goto */
	input = (Fl_Input *)data_plus->p;
	data_plus = (user_data_plus *)input->user_data();
	}
else	input = (Fl_Input*)ob;
view = (SEA_VIEW *)data_plus->p;
if(view->tot_seqs == 0) return;
p = (char *)input->value();
q = p - 1; while(*(++q) != 0) {
	if(!isdigit(*q)) numerique = FALSE;
	}
if(numerique) { /* aller a une position de l'alignement */
	sscanf(p,"%d",&new_pos);
	if(view->cursor_in_comment)
		maxi = view->comment_length[view->cursor_seq - 1];
	else
		maxi = view->each_length[view->cursor_seq - 1];
	if( new_pos <= 0 || new_pos > maxi ) {
		input->value("");
		fl_ringbell(0);
		}
	else	{
		set_and_show_new_cursor_site(view, new_pos, TRUE, FALSE);
		}
	}
else	{ /* recherche d'une seq par son nom */
	l = strlen(p);
	if(l > sizeof(target) - 1) l = sizeof(target) - 1;
	strncpy(target, p, l); target[l] = 0; majuscules(target);
	if(view->wid_names > l_upname) {
		if(l_upname > 0) free(upname);
		upname = (char *)malloc(view->wid_names + 1);
		l_upname = view->wid_names;
		}
	for(num = 0; num < view->tot_seqs; num++) {
		strncpy(upname, view->seqname[num], l_upname);
		upname[l_upname] = 0; majuscules(upname);
		if(strstr(upname, target) != NULL) break;
		}
	if(num >= view->tot_seqs) {
		fl_ringbell(0);
		return;
		}
	if(view->cursor_site < view->first_site || 
		view->cursor_site >= view->first_site + view->tot_sites)
		view->cursor_site = view->first_site;
	set_and_show_new_cursor_seq(view, num + 1);
	}
view->DNA_obj->take_focus();
}


char *search_with_gaps(char *target, char *debut)
{
char *cherche, *trouve = debut - 1;
do	{
	debut = trouve + 1;
	trouve = NULL;
	cherche = target;
	do	{
		while( *debut == '-' ) debut++;
		if(trouve == NULL) trouve = debut;
		if ( toupper(*debut) != *cherche ) break;
		cherche++; debut++;
		}
	while( *cherche != 0 );
	}
while( *trouve != 0  && *cherche != 0);
return ( *cherche == 0 ? trouve : (char*)NULL );
}


void search_callback(Fl_Widget *ob, void *data)
{
SEA_VIEW *view;
char target[50], *pos, *debut;
int new_pos;
int l = -1;

user_data_plus *data_plus = (user_data_plus *)data;
int which = data_plus->value;
if(which == 1) { /* appele par champ input */
	pos = (char *)((Fl_Input*)ob)->value() - 1;
	}
else	{ /* appele par bouton search */
	Fl_Input *champ = (Fl_Input *)data_plus->p;
	pos = (char *)champ->value() - 1;
	data_plus = (user_data_plus *) champ->user_data();
	}
view = (SEA_VIEW *)data_plus->p;
if(view->tot_seqs == 0) return;
while( *++pos && l+1 < sizeof(target) ) 
	target[++l] = toupper(*pos);
target[++l] = 0;
if( l == 0 ) return;
if(view->cursor_in_comment)
	debut = view->comment_line[view->cursor_seq - 1] + view->cursor_site;
else
	debut = view->sequence[view->cursor_seq - 1] + view->cursor_site;
pos = search_with_gaps(target, debut);
if(pos == NULL) fl_ringbell(0);
else	{
	if(view->cursor_in_comment)
		new_pos = pos - view->comment_line[view->cursor_seq - 1] + 1;
	else
		new_pos = pos - view->sequence[view->cursor_seq - 1] + 1;
	set_and_show_new_cursor_site(view, new_pos, TRUE, FALSE);
	}
view->DNA_obj->take_focus();
}


void free_alignment(SEA_VIEW *view)
{
int num, c;
if(view->header!=NULL) { free(view->header); view->header = NULL; }
for(num = 0; num < view->tot_seqs; num++) {
	free(view->sequence[num]);
	free(view->seqname[num]);
	if(view->comments != NULL && view->comments[num] != NULL) {
		free(view->comments[num]);
		view->comments[num] = NULL;
		}
	if(view->numb_gc > 1) {
		for(c = 0; c < view->numb_gc; c++) free(view->col_seq[num][c]);
		free(view->col_seq[num]);
		}
	}
if( view->tot_seqs > 0 ) {
	free(view->sequence);
	free(view->seqname);
	if(view->comments != NULL) free(view->comments);
	}
if( view->numb_gc > 1 && view->tot_seqs > 0 ) free(view->col_seq);
if(view->masename != NULL) {
	free(view->masename);
	view->masename = NULL;
	}
if( view->tot_seqs >= 1 ) {
	free(view->each_length);
	free(view->sel_seqs);
	free(view->region_line);
	}
while (view->regions != NULL)
	delete_region(view, 1);
for(num = 0; num < view->numb_species_sets; num++) {
	free(view->list_species_sets[num]);
	free(view->name_species_sets[num]);
	}
view->numb_species_sets = 0;
view->tot_seqs = 0;
view->tot_sel_seqs = 0;
view->cursor_seq = 0;
free_region(view->active_region);
view->active_region = NULL;
if(view->menu_file != NULL) {
	Fl_Menu_Item *items = 
		(Fl_Menu_Item *) ((Fl_Menu_ *)view->menu_file)->menu();
	items[SAVE].deactivate();
	items[SAVE_AS].deactivate();
	items[SAVE_REGIONS].deactivate();
	}
if(view->tot_comment_lines > 0) {
	for(num = 0; num < view->tot_comment_lines; num++) {
		free(view->comment_name[num]);
		free(view->comment_line[num]);
		}
	free(view->comment_name);
	free(view->comment_line);
	free(view->comment_length);
	view->tot_comment_lines = 0;
	view->show_comment_lines = FALSE;
	}
}


void load_alignment_file(SEA_VIEW *view, char *filename, char *message, 
	known_format file_format)
{
char *err_message, *p;
int protein;

if(view->tot_seqs > 0 && view->modif_but_not_saved) {
	if( ! ask_with_custom_yesno("Alignment was modified but not saved\n"
		"Do you want to  read a new alignment?", "Read", "Cancel") ) return; 
	}
if(filename == NULL) {
#ifdef WIN32
	char wpattern[100];
	sprintf(wpattern, "%s format (*.%s)%c*.%s%cany%c*.*%c", 
		f_format_names[file_format], f_format_exts[file_format], 0, 
		f_format_exts[file_format], 0, 0, 0);
	filename = MG_win32_file_chooser(message, wpattern, "", f_format_exts[file_format], FALSE, NULL);
#else
	char pattern[20];
	sprintf(pattern, "*.%s", file_format == -1 ? "*" : f_format_exts[file_format]); 
#ifdef __APPLE__
	filename = (char *)MAC_file_chooser(message,pattern,"");
#else
	filename = (char *)fl_file_chooser(message,pattern,"",0);
#endif
#endif
	if(filename==NULL) return;
	}
if(file_format == -1) {
	file_format = what_format(filename);
	if(file_format < 0) {
			fl_alert("File %s\nis not of a format readable by seaview", filename);
			return;
			}
	}
my_watch_cursor(view->dnawin);
if(view->alt_col_seq != NULL) {
	reference_toggle(view, FALSE);
	}
free_alignment(view);
if(file_format == MASE_FORMAT) {
	view->tot_seqs = read_mase_seqs_header(filename, &view->sequence,
		&view->seqname, &view->comments, &view->header,
		&err_message);
/* interpreter les regions du header du fichier mase */
	view->regions = parse_regions_from_header(view->header);
/* interpreter les species sets du fichier mase */
	view->numb_species_sets = parse_species_sets_from_header(view->header,
	view->list_species_sets, view->name_species_sets, view->tot_seqs);
/* interpreter les comment lines du header */
view->tot_comment_lines = parse_comment_lines_from_header(view->header,
	&(view->comment_line), &(view->comment_name), 
	&(view->comment_length) , &(view->max_seq_length));
	}
else if(file_format == FASTA_FORMAT)
	view->tot_seqs = read_fasta_align(filename, &view->sequence,
		&view->seqname, &view->comments, &view->header, &err_message);
else if(file_format == PHYLIP_FORMAT)
	view->tot_seqs = read_phylip_align(filename, &view->sequence,
		&view->seqname, &view->comments, &view->header, &err_message);
else if(file_format == CLUSTAL_FORMAT)
	view->tot_seqs = read_clustal_align(filename, &view->sequence,
		&view->seqname, &view->comments, &view->header, &err_message);
else if(file_format == MSF_FORMAT)
	view->tot_seqs = read_msf_align(filename, &view->sequence,
		&view->seqname, &view->comments, &view->header, &err_message);
else if(file_format == NEXUS_FORMAT) {
	int **list_sp = NULL; char **name_sp = NULL; int i;
	view->tot_seqs = read_nexus_align(filename, &view->sequence,
		&view->seqname, &view->comments, &view->header,
		&err_message, &view->regions, &view->numb_species_sets,
		&list_sp, &name_sp, &view->tot_comment_lines, 
		&view->comment_name, &view->comment_line, 
		&view->comment_length, &protein);
	for(i= 0; i < view->numb_species_sets; i++) {
		view->list_species_sets[i] = list_sp[i];
		view->name_species_sets[i] = name_sp[i];
		}
	if(list_sp != NULL) free(list_sp); 
	if(name_sp != NULL) free(name_sp);
	}
if(view->tot_seqs == 0) {
	fl_reset_cursor(view->dnawin);
	fl_alert("Error while reading file %s\n%s", filename,
		err_message);
	view->DNA_obj->parent()->redraw();
	return;
	}
if(file_format != NEXUS_FORMAT) protein = is_a_protein_seq(view->sequence[0]);
p = extract_filename(filename);
view->dnawin->label(p, p);
if(view->menu_file != NULL) {
	Fl_Menu_Item *items = 
		(Fl_Menu_Item *) ((Fl_Menu_ *)view->menu_file)->menu();
	if(file_format == view->format_for_save) {
		items[SAVE].activate();
		items[SAVE_AS].activate();
		}
	else	{
		items[SAVE_AS].activate();
		if(view->masename != NULL) {
			free(view->masename);
			view->masename = filename = NULL;
			}
		}
	}
init_dna_scroller(view, view->tot_seqs, filename, protein, view->header);
view->DNA_obj->parent()->redraw();
fl_reset_cursor(view->dnawin);
view->DNA_obj->take_focus();
return;
}


int ask_with_custom_yesno(const char *text, const char *yes, const char *no)
{
const char *oldyes = fl_yes; const char *oldno = fl_no;
fl_yes = yes; fl_no = no;
int rep = fl_ask(text);
fl_yes = oldyes; fl_no = oldno;
return rep;
}


void mainwin_close_callback(Fl_Widget *form, void *data)
{
SEA_VIEW *view = (SEA_VIEW *)data;
if(view->modif_but_not_saved) {
	if(! ask_with_custom_yesno("Alignment was modified but not saved\n"
		"Do you want to quit anyway?", "Quit", "Cancel") ) return;
	}
#ifdef __APPLE__
char value[50];
sprintf(value, "%dx%d", view->dnawin->w(),  view->dnawin->h() );
set_res_value("window", value);
save_resources();
#endif
exit(0);
}


void file_menu_callback(Fl_Widget *ob, long which)
{
SEA_VIEW *view;
char pattern[20];
static char fsel_message[] = "seaview: choose file and format";
char *filename;
int reponse = ((Fl_Menu_Button*)ob)->value();

view = (SEA_VIEW *) ob->user_data();
sprintf(pattern, "*.%s", f_format_exts[view->format_for_save]);
if(reponse == QUIT) { /* quit */
	mainwin_close_callback(view->dnawin, view);
	}
else if(reponse == OPEN_ANY) { 
#ifdef __APPLE__
	Fl::e_state = 0; // contre bug fltk/MacOS qui laisse e_state plein apres cmd-O
#endif
	load_alignment_file(view, NULL, 
		"Choose an alignment file",  (known_format)-1);
	}
else if(reponse == OPEN_MASE) { 
	load_alignment_file(view, NULL, 
		"Choose a .mase file",  MASE_FORMAT);
	}
else if(reponse == OPEN_PHYLIP )	{  
	load_alignment_file(view, NULL,
		"Choose a Phylip file" ,  PHYLIP_FORMAT );
	}
else if(reponse == OPEN_CLUSTAL )	{  
	load_alignment_file(view, NULL,
		"Choose a Clustal file" ,  CLUSTAL_FORMAT );
	}
else if(reponse == OPEN_MSF )	{  
	load_alignment_file(view, NULL,
		"Choose an MSF file",  MSF_FORMAT );
	}
else if(reponse == OPEN_FASTA )	{  
	load_alignment_file(view, NULL,
		"Choose a Fasta file",  FASTA_FORMAT );
	}
else if(reponse == OPEN_NEXUS )	{  
	load_alignment_file(view, NULL,
		"Choose a NEXUS file",  NEXUS_FORMAT );
	}
else if(reponse == SAVE || reponse == SAVE_AS) 	{ 
	char *err;
	if(reponse == SAVE_AS) { /* Save as */
		FILE *in;
		filename = fl_file_chooser_save_as(fsel_message,
			pattern, 
			view->masename, view);
		if(filename==NULL) return;
#ifndef __APPLE__
		in = fopen(filename,"r");
		if(in != NULL) {
			char tmp[200];
			fclose(in);
			sprintf(tmp, "This file already exists\n%s\n"
				"Do you want to overwrite it?", filename);
			if( ! ask_with_custom_yesno(tmp, "Overwrite", "Cancel") ) 
				return;
			}
#endif
		}
	else {
		filename = view->masename;
		}
	my_watch_cursor(view->dnawin);
	save_active_region(view, TRUE);
	err = save_alignment_or_region(filename, view->sequence, view->comments,
		view->header, view->seqname, view->tot_seqs, view->each_length,
		view->regions, NULL, view->format_for_save,
		view->numb_species_sets, view->list_species_sets,
		view->name_species_sets, NULL, 0, view->protein,
		view->tot_comment_lines, view->comment_name, 
		view->comment_line);
	fl_reset_cursor(view->dnawin);
	if(err != NULL) fl_alert(err);
	else 	{
		if(reponse == SAVE_AS) {
			if(view->masename != NULL) free(view->masename);
			view->masename=(char *)malloc(strlen(filename)+1);
			if(view->masename == NULL) out_of_memory();
			strcpy(view->masename,filename);
			view->dnawin->label(extract_filename(filename));
			Fl_Menu_Item *items = 
				(Fl_Menu_Item *) ((Fl_Menu_ *)ob)->menu();
			items[SAVE].activate();
			}
		view->modif_but_not_saved = FALSE;
		}
	}
else if(reponse == SAVE_REGIONS) {  /* save current regions choice */
	char *err;
	static char regions_only_filename[200];
	static int first = TRUE;
	if(first) {
	 	strcpy(regions_only_filename, "regions");
		if( strchr(pattern,'.') != NULL)
			strcat(regions_only_filename, 
				pattern+1);
		first = FALSE;
		}
	if(view->active_region == NULL) return;
	filename = fl_file_chooser_save_as(fsel_message,
			pattern, regions_only_filename, view);
	if(filename==NULL) return;
	my_watch_cursor(view->dnawin);
	err = save_alignment_or_region(filename, view->sequence, view->comments,
		view->header, view->seqname, view->tot_seqs, view->each_length,
		NULL, view->active_region, view->format_for_save,
		0, NULL, NULL, view->sel_seqs, view->tot_sel_seqs, 
		view->protein, 0, NULL, NULL);
	fl_reset_cursor(view->dnawin);
	if(err != NULL) fl_alert(err);
	else strcpy(regions_only_filename, extract_filename(filename));
	}
else if(reponse == PRINTOUT && view->tot_seqs > 0) {
	int anerr;
	FILE *in;
	char suggested[200], *p;
	
	strcpy(suggested, extract_filename(view->masename) );
	p = strrchr(suggested, '.');
	if(p != NULL) *p = 0;
	strcat(suggested, ".pdf");
	if( view->alt_col_seq != NULL ) {
		for(anerr = 0; anerr < view->tot_seqs; anerr++)
			if(view->sel_seqs[anerr]) break;
		}
	else	anerr = -1;
#ifdef WIN32
	filename = MG_win32_file_chooser("Enter a pdf file name", "pdf format\0*.pdf\0", suggested, "pdf",
		TRUE, NULL);
#elif defined(__APPLE__)
	filename = (char *) fl_file_chooser_plus(
		"Enter a pdf file name", "*.pdf", suggested, 
		(anerr >= 0 ? TRUE : FALSE) );
#else
	filename = fl_file_chooser("Enter a pdf file name", "*.pdf", suggested, 0);
#endif
	if(filename==NULL) return;
#ifndef __APPLE__
	in = fopen(filename,"r");
	if(in != NULL) {
		char tmp[200];
		fclose(in);
		sprintf(tmp, "This file already exists\n%s\n"
				"Do you want to overwrite it?", filename);
		if( ! ask_with_custom_yesno(tmp, "Overwrite", "Cancel") ) 
			return;
		}
#endif
	my_watch_cursor(view->dnawin);
	anerr = pdf_printout(view, filename, printout_fontsize,
		printout_block, printout_pageformat, printout_vary, anerr, printout_black);
	fl_reset_cursor(view->dnawin);
	if( anerr ) fl_alert("Error while writing to file %s",
		filename);
	}
#if !defined( __APPLE__)
else if(reponse == PDFOPTIONS ) {
	pdf_options_dialog(view);
	}
#endif
}



#if defined(__APPLE__) && ! TARGET_RT_MAC_MACHO

char *read_next_help_line(char *line, int lline, void *v_in)
{
static char *pos;
char *in, *l = line;

in = (char *)v_in;
if(line == NULL) {
	pos = in;
	return NULL;
	}
in = pos - 1;
do	*(line++) = *(++in); while(*in != 0 && *in != '\n' && *in != '\r');
if(*in == 0) return NULL;
pos = ++in;
*line = 0;
return l;
}

#else

#define  read_next_help_line(a,b,c) fgets(a,b,(FILE *)c)

#endif


char *get_next_help_line(void *in, char *line, int lline)
{
#ifdef WIN32
static const char match[] = "WIN32";
#elif defined(__APPLE__)
static const char match[] = "__APPLE__";
#else
static const char match[] = "unix";
#endif
static int accept = TRUE;
static int found = FALSE;
if( read_next_help_line(line, lline, in) == NULL) { accept = TRUE; return NULL; }
while(*line != '#') {
	if(accept) return line;
	if( read_next_help_line(line, lline, in) == NULL) { accept = TRUE; return NULL; }
	}
if(strncmp(line, "#ifdef", 6) == 0 || strncmp(line, "#if ", 4) == 0) {
	accept =  ( strstr(line, match) != NULL ) ;
	found = accept;
	}
else if(strncmp(line, "#elif", 5) == 0) {
	if(found) accept = FALSE;
	else if( strstr(line, match) != NULL ) {
		accept = TRUE; found = TRUE;
		}
	}
else if(strncmp(line, "#else", 5) == 0) {
	if(found) accept = FALSE;
	else {
		accept = TRUE; found = TRUE;
		}
	}
else if(strncmp(line, "#endif", 6) == 0) {
	accept = TRUE;
	}
else
	accept = TRUE;  // should not occur
return get_next_help_line(in, line, lline);
}


typedef struct {
	Fl_Widget *br_item, *box;
	void *in;
	} browser_struct;

void browser_titres_callback(Fl_Widget *ob, void *data)
{
int reponse;
static char line[200];
const char *p;
browser_struct *br_data;
int w;
int charwidth, newwidth, maxl;
Fl_Browser *browser_item;

reponse = ((Fl_Browser*)ob)->value();
if(reponse == 0) return;
br_data = (browser_struct *)data;
p = ((Fl_Browser*)ob)->text(reponse);
#if defined(__APPLE__) && !TARGET_RT_MAC_MACHO
get_next_help_line(br_data->in, NULL, 0);
#else
rewind((FILE *)br_data->in);
#endif
br_data->box->hide();
browser_item = (Fl_Browser *)br_data->br_item;
browser_item->clear();
while( get_next_help_line( br_data->in, line, sizeof(line)) != NULL) {
	if(strncmp(line, ">>>", 3) != 0) continue;
	line[ strlen(line) - 1 ] = 0;
	if(strcmp(line+3, p) != 0) continue;
	maxl = 0;
	while( get_next_help_line(br_data->in, line, sizeof(line)) != NULL && 
			strncmp(line, ">>>", 3) != 0 ) {
		line[ strlen(line) - 1 ] = 0;
		if(*line == 0) strcpy(line," ");
		browser_item->add(line);
		maxl = FL_max(maxl, (int)strlen(line));
		}
	/* agrandir fenetre si pas assez large */
	w = browser_item->w();
	fl_font( browser_item->textfont(), browser_item->textsize() );
	charwidth = (int)fl_width('W');
	maxl++;
	if( w < maxl * charwidth) {
		newwidth = ob->window()->w() + (maxl * charwidth - w);
		ob->window()->size(newwidth, ob->window()->h() );
		}
	br_data->box->label(p);
	break;
	}
br_data->box->show();
}


void browser_ok_callback(Fl_Widget *ob, long which)
{
ob->window()->hide();
}


void free_colseqs_by_difference(char ***alt_col_seq, int total, int numb_gc)
{
int num, c;

if(alt_col_seq == NULL) return;
for(num = 0; num < total; num++) {
	for(c = 0; c < numb_gc; c++) free(alt_col_seq[num][c]);
	free(alt_col_seq[num]);
	}
if( total > 0 ) free(alt_col_seq);
}


void reference_toggle(SEA_VIEW *view, int on)
{
char ***tmp;
static int old_pos;

if(view->numb_gc == 1) return;
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)view->bouton_props;
props_menu_parts *props_parts = (props_menu_parts *)menu_props->user_data();
if(on) { /* tenter de passer en mode par reference */
	if(view ->inverted_colors || view->tot_sel_seqs != 1 || 
			view->numb_gc == 1) {
		((Fl_Menu_Item *)menu_props->menu() + props_parts->reference)->clear();
		return;
		}
	my_watch_cursor(view->dnawin);
	for(old_pos = 0; old_pos < view->tot_seqs; old_pos++)
		if(view->sel_seqs[old_pos]) break;
	deplacer_grp_seqs(view, 1);
	view->first_seq = 1;
	view->vertsli->Fl_Slider::value(1);
	view->alt_col_seq = prepcolseqs_by_difference(view->sequence, 
		view->tot_seqs, 0, 
		view->max_seq_length,
		view->each_length, 
		( view->protein ? get_color_for_aa : get_color_for_base ), 
		view->numb_gc, view->allow_lower);
	fl_reset_cursor(view->dnawin);
	if(view->alt_col_seq == NULL) {
		view->DNA_obj->redraw();
		((Fl_Menu_Item *)menu_props->menu() + props_parts->reference)->clear();
		return;
		}
	tmp = view->alt_col_seq;
	view->alt_col_seq = view->col_seq;
	view->col_seq = tmp;		
	view->DNA_obj->redraw();
	((Fl_Menu_Item *)menu_props->menu() + props_parts->colors - 1)->deactivate();
	((Fl_Menu_Item *)menu_props->menu() + props_parts->reference)->set();
	view->menu_edit->labelcolor(FL_INACTIVE_COLOR);
	view->menu_edit->deactivate();
	view->menu_species->labelcolor(FL_INACTIVE_COLOR);
	view->menu_species->deactivate();
	}
else	{ /* retour mode normal */
	my_watch_cursor(view->dnawin);
	tmp = view->alt_col_seq;
	view->alt_col_seq = view->col_seq;
	view->col_seq = tmp;		
	free_colseqs_by_difference(view->alt_col_seq, view->tot_seqs, 
		view->numb_gc);
	view->alt_col_seq = NULL;
	deplacer_grp_seqs(view, old_pos + 1);
	view->DNA_obj->redraw();
	((Fl_Menu_Item *)menu_props->menu() + props_parts->colors - 1)->activate();
	((Fl_Menu_Item *)menu_props->menu() + props_parts->reference)->clear();
	view->menu_edit->labelcolor(FL_BLACK);
	view->menu_edit->activate();
	view->menu_species->labelcolor(FL_BLACK);
	view->menu_species->activate();
	fl_reset_cursor(view->dnawin);
	}
}


void help_callback(Fl_Widget *ob, void *data)
{
static browser_struct browser_data;
static int first = TRUE;
static Fl_Window *help_form;

if(first) {
	char line[85];
	Fl_Browser *browser_titres, *browser_item;
	Fl_Button *ok_button;
	Fl_Widget *br_item_label;
	int titres_width, fin_titres;
#if defined(__APPLE__) 
#if ! TARGET_RT_MAC_MACHO
	get_next_help_line(data, NULL, 0);
#else
	data = (void *)fopen( (char *)data, "r" );
#endif
#else
	data = (void *)open_path( (char *)data );
#endif
	if(data == NULL) {
		fl_alert(
#ifdef __APPLE__
	"No help information in program resources"
#else
	"Help file %s\nnot found in PATH directories nor in current directory", (char *)data
#endif
			);
		return;
		}
	first = FALSE;
	help_form = new Fl_Window( 705, 350);
	help_form->label("Help");
	help_form->box(FL_FLAT_BOX);
	fl_font(FL_HELVETICA, 12);
	titres_width = (int)fl_width("XXXXXX"); /* mieux ensuite */
	browser_titres = new Fl_Hold_Browser(5, 20, titres_width, 
		help_form->h() - 50, "Choose help item");
	browser_titres->textsize(12);
	browser_titres->has_scrollbar(Fl_Browser_::VERTICAL);
	browser_titres->callback(browser_titres_callback, &browser_data);
	browser_titres->align(FL_ALIGN_TOP);
	browser_titres->color(FL_LIGHT1, browser_titres->selection_color());
	while( get_next_help_line(data, line, sizeof(line)) != NULL) {
		if(strncmp(line, ">>>", 3) != 0) continue;
		line[ strlen(line) - 1 ] = 0;
		browser_titres->add(line+3);
		titres_width = FL_max(titres_width, 
			(int)fl_width(line + 3) + (int)fl_width("XX") );
		}
	((Fl_Widget*)browser_titres)->size(titres_width, browser_titres->h() );
	ok_button = new Fl_Button(5,help_form->h() - 25, titres_width,20,"OK");
	ok_button->callback(browser_ok_callback, 0);
	fin_titres = titres_width + 10;
	browser_item = new Fl_Browser(fin_titres, 20, 
		700 - fin_titres, help_form->h() - 25, "");
	browser_item->has_scrollbar(Fl_Browser_::VERTICAL);
	browser_item->align(FL_ALIGN_TOP);
	browser_item->color(FL_LIGHT1, browser_item->selection_color());
	browser_item->textsize(12);
	browser_item->textfont(FL_COURIER);
	br_item_label = new Fl_Box(FL_NO_BOX, fin_titres, 0, 
		700 - fin_titres, 20, "Help Info");
	br_item_label->align(FL_ALIGN_CENTER);
	browser_data.box = br_item_label;
	browser_data.br_item = browser_item;
	browser_data.in = data;
	help_form->add_resizable(
		*new Fl_Box(browser_item->x(), browser_item->y(),
		browser_item->w(), browser_titres->h() ) );
	help_form->end();
	}
help_form->show();
}


int insert_gaps_at(SEA_VIEW *view, int seq, int site, int total)
{
char *pos, **psequence;
int l, c, gapcolor, *plength;
if(view->cursor_in_comment) {
	psequence = view->comment_line;
	plength = view->comment_length;
	}
else	{
	psequence = view->sequence;
	plength = view->each_length;
	gapcolor = ( view->protein ? 
		get_color_for_aa('-') : get_color_for_base('-') );
	}
l = plength[seq-1];
if(site > l + 1) return total;
if( l + total > view->max_seq_length) total = view->max_seq_length - l;
pos = psequence[seq-1] + site - 1;
memmove(pos+total, pos, l - site + 2);
memset(pos, '-', total);
if( (!view->cursor_in_comment) && view->numb_gc > 1) {
	for (c=0; c<view->numb_gc; c++) {
	   pos= &view->col_seq[seq-1][c][site-1];
	   memmove(pos+total, pos, l - site + 2);
	   memset(pos,' ',total);
	   }
	memset(view->col_seq[seq-1][gapcolor] + site - 1, '-', total);
	}
plength[seq-1] += total;
view->modif_but_not_saved = TRUE;
return total;
}


int delete_gaps_before(SEA_VIEW *view, int numseq, int numsite, int total)
{
char *site, *finseq, **psequence;
int c, count = -1, l, retval, *plength;

psequence = view->sequence;
plength = view->each_length;
site = psequence[numseq-1] + numsite - 1;
finseq = psequence[numseq-1] + plength[numseq-1] - 1;
do	{ site--; count++; }
while ( count < total && site >= psequence[numseq-1] && 
	( view->allow_seq_edit || *site == '-' || site > finseq) );
if(count == 0) return 0;
/* ne rien faire si on efface au dela de la fin de la seq */
if(numsite - count > plength[numseq-1]) return count;
l = plength[numseq-1];
retval = count;
if(numsite > l) { /* effacer depuis au dela fin jusqu'a interieur seq */
	count -= (numsite - l - 1);
	numsite = l + 1;
	}
site = psequence[numseq-1] + numsite - 1;
memmove(site-count, site, l - numsite + 2);
if( view->numb_gc > 1) {
	for (c=0; c < view->numb_gc; c++) {
		site= view->col_seq[numseq-1][c] + numsite - 1;
		memmove(site-count,site, l - numsite + 2);
		}
	}
plength[numseq-1] -= count;
view->modif_but_not_saved = TRUE;
return retval;
}


void adjust_menu_edit_modes(SEA_VIEW *view)
{
if(view->menu_edit != NULL) {
	Fl_Menu_Item *items = (Fl_Menu_Item *) ((Fl_Menu_ *)view->menu_edit)->menu();
	if(view->tot_sel_seqs != 0) {
		items[DELETE_SEQ].activate();
		items[CONSENSUS_SEQ].activate();
		if(view->protein) items[EXCHANGE_UT].deactivate();
		else items[EXCHANGE_UT].activate();
		}
	else	{
		items[DELETE_SEQ].deactivate();
		items[EXCHANGE_UT].deactivate();
		}
Fl_Menu_Button *menu_props = (Fl_Menu_Button *)view->bouton_props;
props_menu_parts *props_parts = (props_menu_parts *)menu_props->user_data();
	Fl_Menu_Item *byref = (Fl_Menu_Item *)menu_props->menu() + props_parts->reference;
	if(view->tot_sel_seqs == 1) {
		if(!view->inverted_colors) byref->activate();
		items[RENAME_SEQ].activate();
		items[DUPLICATE_SEQ].activate();
		items[EDIT_COMMENTS].activate();
		if(view->protein) {
			items[COMPLEMENT_SEQ].deactivate();
			items[REVERSE_SEQ].deactivate();
			}
		else {
			items[COMPLEMENT_SEQ].activate();
			items[REVERSE_SEQ].activate();
			}
		}
	else	{
		byref->deactivate();
		items[RENAME_SEQ].deactivate();
		items[DUPLICATE_SEQ].deactivate();
		items[EDIT_COMMENTS].deactivate();
		items[COMPLEMENT_SEQ].deactivate();
		items[REVERSE_SEQ].deactivate();
		}
	if(view->tot_sel_seqs == 2) 
		items[DOT_PLOT].activate();
	else
		items[DOT_PLOT].deactivate();
	if(view->tot_sel_seqs >= 2) {
		items[ALIGN_SEQS].activate();
		items[CONSENSUS_SEQ].activate();
		}
	else	{
		items[ALIGN_SEQS].deactivate();
		items[CONSENSUS_SEQ].deactivate();
		}
	}
}


void select_deselect_seq(SEA_VIEW *view, int new_seq)
{
/* new_seq = # seq a select/deselect; si 0: tout deselectionner; 
   si -1: tout selectionner 
   si -2: ne pas changer la selection mais ajuster l'interface selon son etat
*/
if(new_seq > 0) { /* traiter une sequence */
	view->sel_seqs[new_seq-1] = !view->sel_seqs[new_seq-1];
	if(view->sel_seqs[new_seq-1])
		++view->tot_sel_seqs;
	else
		--view->tot_sel_seqs;
	}
else if(new_seq == 0)	{ /* tout deselectionner */
	view->tot_sel_seqs = 0;
	memset(view->sel_seqs, 0, view->tot_seqs * sizeof(int));
	}
else if(new_seq == -1)	{ /* tout selectionner */
	int i;
	view->tot_sel_seqs = view->tot_seqs;
	for(i=0; i < view->tot_seqs; i++) view->sel_seqs[i] = TRUE;
	}
adjust_menu_edit_modes(view);
if(view->menu_species != NULL) {
	int taille, valeur;
	Fl_Menu_Item *items =
		(Fl_Menu_Item *)((Fl_Menu_*)view->menu_species)->menu();
	if(view->tot_sel_seqs == 0)
		items[0].deactivate();
	else
		items[0].activate();
	items[1].deactivate();
	get_menu_taille_valeur(view->menu_species, &taille, &valeur);
	if(valeur != 0) 
		update_menu_taille_valeur(view->menu_species, taille, 0, 2);
	}
}


void deplacer_grp_seqs(SEA_VIEW *view, int target)
{
/* deplacer toutes les seqs selectionnees pour positionner la premiere
d'entre elles en position target */
int *new_rank, *old_rank, old, new_val, numset;
char **aux;
-- target;
new_rank = (int *)malloc(view->tot_seqs * sizeof(int));
old_rank = (int *)malloc(view->tot_seqs * sizeof(int));
aux = (char **)malloc( view->tot_seqs * sizeof(char *) );
if(new_rank == NULL || old_rank == NULL || aux == NULL) out_of_memory();
/* compute old_rank[new_val] = old */
new_val = -1;
/* place first all non selected seqs */
for(old = 0; old < view->tot_seqs; old++) {
	if(!view->sel_seqs[old]) old_rank[++new_val] = old;
	}
/* allocate room for selected seqs */
if(target + view->tot_sel_seqs > view->tot_seqs)
	target = view->tot_seqs - view->tot_sel_seqs;
old = view->tot_seqs - view->tot_sel_seqs - target;
if(old != 0)
	memmove(old_rank + target + view->tot_sel_seqs, old_rank + target,
		old * sizeof(int));
/* insert selected seqs */
for(old = 0; old < view->tot_seqs; old++)
	if(view->sel_seqs[old]) old_rank[target++] = old;
/* compute new_rank[old] = new_val */
for(new_val = 0; new_val < view->tot_seqs; new_val++)
	new_rank[old_rank[new_val]] = new_val;
/* displace all sequence order-dependent ingredients */
/* deplacer la position du curseur */
if(!view->cursor_in_comment) {
	view->cursor_seq = new_rank[view->cursor_seq - 1] + 1;
	view->old_cursor_seq = view->cursor_seq;
	}
/* deplacer les seqs */
for(old = 0; old < view->tot_seqs; old++)
	aux[new_rank[old]] = view->sequence[old];
memcpy(view->sequence, aux, view->tot_seqs * sizeof(char *) );
/* deplacer les noms */
for(old = 0; old < view->tot_seqs; old++)
	aux[new_rank[old]] = view->seqname[old];
memcpy(view->seqname, aux, view->tot_seqs * sizeof(char *) );
if(view->comments != NULL) {
	/* deplacer les commentaires */
	for(old = 0; old < view->tot_seqs; old++)
		aux[new_rank[old]] = view->comments[old];
	memcpy(view->comments, aux, view->tot_seqs * sizeof(char *) );
	}
/* deplacer les seqs en couleurs */
if(view->numb_gc > 1) {
	for(old = 0; old < view->tot_seqs; old++)
		aux[new_rank[old]] = (char *) view->col_seq[old];
	memcpy(view->col_seq, aux, view->tot_seqs * sizeof(char *) );
	}
/* deplacer les sequences selectionnees */
for(old = 0; old < view->tot_seqs; old++)
	old_rank[new_rank[old]] = view->sel_seqs[old];
memcpy(view->sel_seqs, old_rank, view->tot_seqs * sizeof(int) );
/* deplacer les longueurs de sequences */
for(old = 0; old < view->tot_seqs; old++)
	old_rank[new_rank[old]] = view->each_length[old];
memcpy(view->each_length, old_rank, view->tot_seqs * sizeof(int) );
/* process species sets */
for(numset = 0; numset < view->numb_species_sets; numset++) {
	for(old = 0; old < view->tot_seqs; old++)
		old_rank[new_rank[old]] = view->list_species_sets[numset][old];
	memcpy(view->list_species_sets[numset], old_rank, 
		view->tot_seqs * sizeof(int) );
	}
free(aux); free(old_rank); free(new_rank);
view->modif_but_not_saved = TRUE;
}


void update_current_seq_length(int newlength, SEA_VIEW *view)
{
double x; int l;
if(newlength > view->seq_length) {
	view->seq_length = 
		( newlength+20 < view->max_seq_length ? 
		newlength+20 : view->max_seq_length );
	l = view->seq_length - view->tot_sites+3;
	if(l<1) l=1;
	view->horsli->bounds(1,l);
	x = ( (double) view->tot_sites ) / 
		( view->seq_length + 3 ) ;
	if(x>1) x=1;
	view->horsli->slider_size(x);
	}
}


int insert_char_in_seq( int key, int total, SEA_VIEW *view)
/* to insert the typed key in the sequence at cursor location if it is visible
returns # of inserted chars if ok, 0 if error (= cursor not visible or 
max seq size is reached) 
*/
{
char *pos;
int l, c, *plength;
if(view->cursor_in_comment) {
	if( view->cursor_seq < 1 ||
	   view->cursor_seq >= view->tot_comment_lines ||
	   view->cursor_site < view->first_site ||
	   view->cursor_site >= view->first_site + view->tot_sites ) return 0;
	l = view->comment_length[view->cursor_seq - 1];
	}
else	{
	if( view->cursor_seq < view->first_seq ||
	   view->cursor_seq >=view->first_seq+view->tot_lines ||
	   view->cursor_site < view->first_site ||
	   view->cursor_site >= view->first_site + view->tot_sites ) return 0;
	l = view->each_length[view->cursor_seq-1];
	}
if(view->cursor_site > l + 1) return 0;
if( l + total > view->max_seq_length) total = view->max_seq_length - l;
if(total <= 0) return 0;
if(view->cursor_in_comment) 
	pos = view->comment_line[view->cursor_seq - 1] + view->cursor_site - 1;
else
	pos = view->sequence[view->cursor_seq - 1] + view->cursor_site - 1;
memmove(pos+total, pos, l - view->cursor_site + 2);
memset(pos, view->cursor_in_comment || view->allow_lower ? key : toupper(key) ,
	 total);
if( (!view->cursor_in_comment) && view->numb_gc > 1) {
	for (c=0; c<view->numb_gc; c++) {
	   pos= &view->col_seq[view->cursor_seq-1][c][view->cursor_site-1];
	   memmove(pos+total, pos, l - view->cursor_site + 2);
	   memset(pos, ' ' ,total);
	   }
	c = (view->protein ? 
		get_color_for_aa(key) : get_color_for_base(key) );
	memset(view->col_seq[view->cursor_seq-1][c] + view->cursor_site - 1, 
		view->allow_lower ? key : toupper(key), total);
	}
if(view->cursor_in_comment) 
	plength = &(view->comment_length[view->cursor_seq-1]);
else
	plength = &(view->each_length[view->cursor_seq-1]);
(*plength) += total;
update_current_seq_length(*plength, view);
view->modif_but_not_saved = TRUE;
return total;
}


/*  The routine that does drawing */
void DNA_obj::draw(void)
{
SEA_VIEW *view = (SEA_VIEW *)this->user_data();

/* returns TRUE if window size was changed by user */
if( compute_size_params( view, FALSE) ) {
	view->horsli->redraw();
	view->vertsli->redraw();
	}
if( ( this->damage() & FL_DAMAGE_ALL ) != 0 ) {
// appel autre que uniquement par damage partiel
	view->draw_names = -1;
	view->mod_cursor = view->mod_region_line = view->mod_comment_line = 
		FALSE; 
	view->mod_seq = 0;
	}
if(view->draw_names) { /* soit tous (<= -1) soit un seul ( >= 1) */
	draw_seq_names(view->DNA_obj, view);
	if(view->draw_names > 0 || view->draw_names == -2){
		/* si > 0 ou -2, ne pas ecrire les seqs*/
		view->draw_names = -1;
		return;
		}
	}
if(view->mod_cursor) {
	/* effacer old_cursor en ecrivant dessus */
	draw_cursor(view->DNA_obj, FALSE, view->old_cursor_site, 
		view->old_cursor_seq, view->old_cursor_in_comment);
	view->mod_cursor = FALSE;
	}
else if(view->mod_region_line) {
	draw_region_line(view->DNA_obj, view);
	view->mod_region_line = FALSE;
	}
else if(view->mod_comment_line) {
	draw_comment_lines(view->DNA_obj, view);
	view->mod_comment_line = FALSE;
	}
else	{
#ifdef __APPLE__
	if(view->tot_seqs == 0) {
#define DRAG_MESS "alignment file drag zone"
		fl_color(FL_WHITE);
		fl_font(FL_TIMES_BOLD_ITALIC, 28 );
		fl_draw(DRAG_MESS, this->x()+this->w()/2-fl_width(DRAG_MESS)/2, this->y()+this->h()/2);
		}
#endif
	if(view->inverted_colors)
		    draw_dna_seqs_inverted(view->DNA_obj, view);
	else
		    draw_dna_seqs(view->DNA_obj, view);
	draw_comment_lines(view->DNA_obj, view);
	}
view->mod_seq = 0;
view->draw_names = -1;
draw_cursor(view->DNA_obj, TRUE , view->cursor_site, view->cursor_seq, 
	view->cursor_in_comment);
view->old_cursor_seq = view->cursor_seq;
view->old_cursor_site = view->cursor_site;
view->old_cursor_in_comment = view->cursor_in_comment;
}


int my_event_button(void)
{
int key;

key = Fl::event_button();  /* key: 1=gauche, 2=centre, 3=droit souris */
if(emul_3_buttons) {
	if(Fl::event_state(FL_CTRL)) key = 2;// bouton du milieu par Ctrl-Click
	else if(Fl::event_state(FL_SHIFT)) key = 3;// bouton  droit par Shift-Click
	}
return key;
}



int DNA_obj::handle(int event) 
{
SEA_VIEW *view;
/* numero courant de la derniere seq selectionnee pendant selection de seqs
par glissement de la souris 
*/
static int selecting_seqs = 0, sel_seq_move = 0;
static int modifying_segment = 0;
int mx = Fl::event_x();
int my = Fl::event_y();

view = (SEA_VIEW *) ( this->user_data() );
if(view == NULL) return 1;
    switch (event)
    {
#if defined(__APPLE__) || defined(WIN32)
// D&D doesn't seem to work under X11, and OK with Win98 but incomplete with WinXP
    case FL_DND_ENTER:
   	 	return 1;
    case FL_DND_DRAG:
   	 	return 1; 
    case FL_DND_RELEASE:
 		{ Fl_Widget *o = this; while(o->parent()->parent() != NULL) o = o->parent();   
   	 	Fl::focus(o); } //fltk bug: focus must be to direct child of Fl_Window for D&D 
   	 	return 1;
    case FL_PASTE:
   	 	use_initial_file(view, 
#if defined(__APPLE__) && !TARGET_RT_MAC_MACHO
   		 OSXpathtoOS9path( Fl::event_text() )
#else
   		 (char *)Fl::event_text() 
#endif
    );
   		Fl::focus(this);
   		return 1;
#endif
    case FL_FOCUS: 
    case FL_UNFOCUS: 
		return 1;
    case FL_PUSH: 
	int key;
	key = my_event_button();
	if( Fl::event_clicks() ) { /* double click */
		int new_seq;
		Fl::event_clicks(0);
		new_seq = (my + view->line_height/2 - 
			view->y_seq)/view->line_height + view->first_seq;
		if( new_seq < view->first_seq || new_seq > view->tot_seqs ||
			new_seq >= view->first_seq + view->tot_lines ) break;
		if(mx < view->x_name || mx >= view->x_seq - view->char_width ||
			key != 1) break;
		/* double click sur nom de seq: selection de toutes les seqs */
		if(view->alt_col_seq != NULL) break;
		if(view->multipl->argument() > 0) mod_multipl(view,0);
		select_deselect_seq(view, -1);
		selecting_seqs = 0;
		view->draw_names = -2;
// ceci signifie redraw partiel commande' par draw_names, mod_cursor, ...
		this->damage(1);
		}
	else	{ /* simple click */
		Fl::focus(this);
		handle_push(view, mx, my, key, &modifying_segment, 
			&selecting_seqs, &sel_seq_move);
		}
	break;
     case FL_DRAG: /* mouvement avec souris enfoncee */
	handle_mouse(view, mx, my, &selecting_seqs, &sel_seq_move, 
		&modifying_segment);
	break;
     case FL_RELEASE:
//     case FL_LEAVE:
	if(selecting_seqs) {
		if(sel_seq_move) {
			select_deselect_seq(view, sel_seq_move);
			view->draw_names = sel_seq_move;
			this->damage(1);
			}
		else			
			select_deselect_seq(view, -2);
		selecting_seqs = 0;
		}
	else if(modifying_segment) {
		end_change_segment(view);
		view->draw_names = 0;
		this->damage(1);
		modifying_segment = 0;
		}
	break;
     case FL_KEYBOARD:
	if( Fl::event_state(myFL_CTRL) ) { /* vrai si touche ctrl */
		return 0; /* ne pas traiter ici car shortcut d'autres widgets */
		}
  	if(Fl::event_length() > 0) { /* du vrai texte */
		handle_keyboard(view, Fl::event_text()[0], TRUE );
		}
	else	{ /* une touche genre fleche ou fonction */
		handle_keyboard(view, Fl::event_key(), FALSE );
		}
	break;
#ifdef __APPLE__
	 case FL_HIDE:
	 Fl::e_state = 0; //bug qui conserve cmd modifier apres cmd-H
	 return 0;
#endif
    default :
	return 0;
    }
    return 1;
}


void handle_mouse(SEA_VIEW *view, int mx, int my, 
	int *p_selecting_seqs, int *p_sel_seq_move, int *p_modifying_segment)
{ /* mouvement avec souris enfoncee */
int debut, fin, step, num, new_seq, new_site;

if(*p_selecting_seqs != 0) {
	new_seq = (my + view->line_height/2 - 
		view->y_seq)/view->line_height + view->first_seq;
	if(new_seq == *p_selecting_seqs) return;
	if( new_seq < view->first_seq || new_seq > view->tot_seqs ||
		new_seq >= view->first_seq + view->tot_lines ) return;
	if(!view->sel_seqs[new_seq - 1]) 
		{ debut= new_seq; fin = *p_selecting_seqs; }
	else
		{ debut= *p_selecting_seqs; fin = new_seq; }
	if(debut < fin) step = 1;
	else	step = -1;
	*p_selecting_seqs = new_seq;
	for(num = debut; num != fin; num += step) {
		new_seq = debut + fin - step - num - 1;
		if(view->sel_seqs[new_seq]) {
			view->sel_seqs[new_seq] = FALSE;
			--(view->tot_sel_seqs);
			}
		else	{
			view->sel_seqs[new_seq] = TRUE;
			++(view->tot_sel_seqs);
			}
		if(*p_sel_seq_move == new_seq + 1) *p_sel_seq_move = 0;
		}
	if(*p_sel_seq_move) {
		if( view->sel_seqs[*p_sel_seq_move - 1] ) {
			view->sel_seqs[*p_sel_seq_move - 1] = FALSE;
			--(view->tot_sel_seqs);
			}
		else	{
			view->sel_seqs[*p_sel_seq_move - 1] = TRUE;
			++(view->tot_sel_seqs);
			}
		*p_sel_seq_move = 0;
		}
	view->draw_names = -2;
	view->DNA_obj->damage(1);
	}
else if(*p_modifying_segment != 0) {
	new_site = (mx - view->x_seq )/view->char_width + 
		view->first_site;
	if(new_site == *p_modifying_segment) return;
	if( new_site < view->first_site || 
		new_site > view->first_site + view->tot_sites ||
		new_site > view->region_length ) return;
	if( continue_change_segment(view, new_site) ) {
		*p_modifying_segment = new_site;
		view->draw_names = 0;
		view->mod_region_line = TRUE;
		view->DNA_obj->damage(1);
		}
	}
}


void handle_push(SEA_VIEW *view, int mx, int my, int key, 
	int *p_modifying_segment, int *p_selecting_seqs, int *p_sel_seq_move)
/* key: 1=bouton gauche, 2=centre, 3=droit de la souris */
{
int new_site, new_seq, new_line;

if(view->multipl->argument() > 0) mod_multipl(view,0);
new_seq = (my + view->line_height/2 - view->y_seq)/view->line_height + 
	view->first_seq;
new_line = new_seq - view->first_seq + 1;
new_site = (mx - view->x_seq )/view->char_width + 
	view->first_site;
if(view->active_region != NULL && 
  new_seq == view->first_seq + FL_min(view->tot_lines,view->tot_seqs) &&
	new_site >= view->first_site && 
	new_site < view->first_site + view->tot_sites && 
	new_site <= view->region_length ) {
/* work with segments: extend, or create, or delete */
	if(key == 2) { /* middle button:extend left neighbor segment */
		new_seq = extend_segment_at_left(view, new_site);
		if(new_seq) fl_ringbell(0);
		else	{
			view->draw_names = 0;
			}
		}
	else if(key == 3) { /* right button=>delete segment */
		new_seq = suppr_segment(view->active_region, new_site,
			view->region_line);
		if(new_seq) fl_ringbell(0);
		else	{
			view->draw_names = 0;
			}
		}
	else	{ /* left button=>extend or create segment */
		new_seq = begin_change_segment(view, new_site);
		if(new_seq) {
			view->mod_region_line = TRUE;
			*p_modifying_segment = new_site;
			}
		view->draw_names = 0;
		}
	if(view->draw_names == 0) {
		view->DNA_obj->damage(1);
		}
	return;
	}
if( view->show_comment_lines && new_line >= view->pos_first_comment_line &&
	new_line < view->pos_first_comment_line + view->tot_comment_lines ) {
/* dans les comment lines */
	int num, old;
	if(key != 1) return;
	num = new_line - view->pos_first_comment_line + 1;
	if(mx >= view->x_name && mx < view->x_seq - view->char_width) {
	/* click sur nom de comment line: selection/deselection */
		old = view->active_comment_line;
		if(old == num)
			view->active_comment_line = 0;
		else
			view->active_comment_line = num;
		if(old == num || old == 0)
			view->mod_comment_line = num;
		else	{
			view->mod_comment_line = 0;
			view->mod_seq = 1; // astuce
			}
		view->draw_names = 0;
		update_menu_footers(view);
		view->DNA_obj->damage(1);
		}
	else if( new_site >= view->first_site && 
		new_site < view->first_site + view->tot_sites &&
		new_site <= view->comment_length[num - 1] + 1 ) {
		/* click sur comment: positionnement du curseur */
		view->cursor_site = new_site;
		view->cursor_seq = num;
		view->draw_names = 0;
		view->mod_cursor = TRUE;
		view->cursor_in_comment = TRUE;
		view->DNA_obj->damage(1);
		}
	return;
	}
	
if( new_seq < view->first_seq || new_seq > view->tot_seqs ||
	new_seq >= view->first_seq + view->tot_lines ) return;
if(mx >= view->x_name && mx < view->x_seq - view->char_width) {
/* click sur nom de seq: selection/deselection */
	if(view->alt_col_seq != NULL) return;
	if(key == 1) {
		*p_selecting_seqs = new_seq;
		*p_sel_seq_move = new_seq;
		return;
		}
	else if(key == 3) {
		select_deselect_seq(view, 0);
		view->draw_names = -2;
		view->DNA_obj->damage(1);
		return;
		}
	else	{ /* milieu: depl des seqs selectionnees */
		if(view->tot_sel_seqs == 0 || 
			view->sel_seqs[new_seq - 1]) 
			{ fl_ringbell(0); return; }
		deplacer_grp_seqs(view, new_seq);
		view->DNA_obj->redraw();
		}
	return;
	}
if(key != 1)  return;
/* click sur seq: positionnement du curseur */
if( new_site >= view->first_site && 
	new_site < view->first_site + view->tot_sites &&
	new_site <= view->each_length[new_seq-1] + 1 ) {
	view->cursor_site = new_site;
	view->cursor_seq = new_seq;
	view->cursor_in_comment = FALSE;
	view->draw_names = 0;
	view->mod_cursor = TRUE;
	view->DNA_obj->damage(1);
	}
return;
}


void handle_keyboard(SEA_VIEW *view, unsigned int key, int istext)
{
int new_pos, multipl, num;

multipl = view->multipl->argument();
if(multipl == 0) multipl = 1;
if(key == FL_Right) { /* right arrow */		
	new_pos = view->cursor_site + multipl;
    if(view->cursor_in_comment) {
	if(new_pos > view->comment_length[view->cursor_seq-1]+1)
	   new_pos = view->comment_length[view->cursor_seq-1]+1;
	}
    else {
	if(new_pos > view->each_length[view->cursor_seq-1] + 1) 
	   new_pos = view->each_length[view->cursor_seq-1] + 1;
	}
    set_and_show_new_cursor_site(view, new_pos,FALSE,FALSE);
    }
else if(key == FL_Left) { /* left arrow */		
	new_pos = FL_max(1, view->cursor_site - multipl);
	set_and_show_new_cursor_site(view, new_pos,FALSE,FALSE);
	}
else if(key == FL_Up) { /* up arrow */
	new_pos = FL_max(1, view->cursor_seq - multipl);
	set_and_show_new_cursor_seq(view, new_pos);
	}
else if(key == FL_Down){ /* down arrow */
	new_pos = view->cursor_seq + multipl;
	if(view->cursor_in_comment) {
		if(new_pos > view->tot_comment_lines) 
			new_pos = view->tot_comment_lines;
		}
	else	{
		if(new_pos > view->tot_seqs) 
			new_pos = view->tot_seqs;
		}
	set_and_show_new_cursor_seq(view, new_pos);
	}
else if(view->cursor_in_comment && 
	view->active_comment_line == view->cursor_seq) {
	unsigned char c_key = (unsigned)key;
	if(view->alt_col_seq != NULL) return;
	if( key == 0x7f || key == 0x8 || key == FL_Delete || key == FL_BackSpace ) /* del or BS */
		delete_char_in_comment(view, 1, 
			view->active_comment_line, 
			view->cursor_site, FALSE);
	else if( istext && ( ( c_key >= 32 && c_key <= 126 ) || 
		( c_key >= 160 && c_key <= 255 )  ) ) 
		insert_char_in_comment(c_key, 1, view);
	else
		return;
	}
else if(strchr(view->movekeys, key) != NULL) { 
					/* ][>< touches depl droite/gauche */
	int oldpos;
	int upper_step=50, bracket_step=5 ;
	int max_w;
	max_w = (int)(view->horsli->maximum());
	oldpos = (int)( view->horsli->value() );
	new_pos = oldpos;
	upper_step *= multipl;
	bracket_step *= multipl;
	if(key == view->movekeys[2] /* > */ ) {
		new_pos=oldpos+upper_step;
		if(new_pos>max_w) new_pos=max_w;
		}
	else if(key == view->movekeys[3] /* < */ ) {
		new_pos=oldpos-upper_step;
		if(new_pos<1) new_pos=1;
		}
	else if(key == view->movekeys[0] /* ] */ ) {
		new_pos=oldpos+bracket_step;
		if(new_pos>max_w) new_pos=max_w;
		}
	else if(key == view->movekeys[1] /* [ */ ) {
		new_pos=oldpos-bracket_step;
		if(new_pos<1) new_pos=1;
		}
	if(new_pos!=oldpos) {
		view->horsli->Fl_Slider::value(new_pos);
		view->draw_names = 0;
		view->first_site = new_pos;
		view->DNA_obj->damage(1);
		}
	}
else if(key == 0x7f || key == 0x8 || key == FL_Delete || key == FL_BackSpace ) { /* delete or backspace */
	int count, count_each, debut, fin, test;
	    if(view->multipl->argument() > 0)
		mod_multipl(view,0);
	if(view->cursor_in_comment)
		test = view->cursor_seq < 1 ||
	   		view->cursor_seq > view->tot_comment_lines;
	else
		test = view->cursor_seq < view->first_seq ||
	   		view->cursor_seq >= view->first_seq+view->tot_lines;
	if( test ||
	   view->cursor_site < view->first_site ||
	   view->cursor_site >=view->first_site+view->tot_sites)
		{ fl_ringbell(0); return; }
	if(view->cursor_in_comment) {
		if( delete_char_in_comment(view, multipl, 
			view->cursor_seq, view->cursor_site,
			TRUE) != multipl) fl_ringbell(0);
		return;
		}
	if(view->alt_col_seq != NULL) return;
	if(view->tot_sel_seqs > 1 &&
		view->sel_seqs[view->cursor_seq - 1])
		{ debut = 1; fin = view->tot_seqs; test = TRUE;}
	else
		{ debut = fin = view->cursor_seq; test = FALSE;}
	for(num = debut; num<=fin; num++) 
		{
		if(test && !view->sel_seqs[num-1]) continue;
		count_each = delete_gaps_before(view,
			num, 
			view->cursor_site, multipl);
		if(count_each < multipl) fl_ringbell(0);
		if(num == view->cursor_seq) count = count_each;
		}
	/* si ttes seqs selectionnees, traiter aussi regions et comments */
	if(count_each == multipl && 
		(!view->cursor_in_comment) &&
		view->tot_sel_seqs == view->tot_seqs ) {
		if(view->regions != NULL)
			delete_region_part(view, 
				view->cursor_site,multipl);
		if(view->tot_comment_lines > 0)
			delete_in_all_comments(multipl,
				view->cursor_site, view);
		}
	new_pos = view->cursor_site - count;
	if(new_pos <= 0) new_pos = 1;
	if(view->cursor_in_comment) {
		view->mod_comment_line = view->cursor_seq;
		}
	else	{
		if(view->tot_sel_seqs > 1 && 
			view->tot_sel_seqs != view->tot_seqs &&
			view->sel_seqs[view->cursor_seq - 1])
			view->mod_seq = -1;
		else if(view->tot_sel_seqs <= 1 || 
			!view->sel_seqs[view->cursor_seq - 1])
			view->mod_seq = view->cursor_seq;
		}
	set_and_show_new_cursor_site(view, new_pos, 
		FALSE,TRUE);
	}
else if(key == '_' ) { /* del gap in all but current seq(s) */
	int count_each;
	    if(view->multipl->argument() > 0)
		mod_multipl(view,0);
	if(view->cursor_in_comment) return;
	if( view->cursor_seq < view->first_seq ||
	   	view->cursor_seq >=view->first_seq+view->tot_lines ||
	  	view->cursor_site < view->first_site ||
	   	view->cursor_site >= 
	   	view->first_site + view->tot_sites ||
	   	view->tot_sel_seqs == view->tot_seqs )
		 { fl_ringbell(0); return; }
	if(view->alt_col_seq != NULL) return;
	for( num = 1; num <= view->tot_seqs; num++) {
		if(num == view->cursor_seq || 
			(view->sel_seqs[view->cursor_seq-1] &&
			view->sel_seqs[num-1] ) ) continue;
		count_each = delete_gaps_before(view, 
		    	num, view->cursor_site, multipl);
		if(count_each < multipl) {
			fl_ringbell(0);
			return;
			}
		}
	if(count_each == multipl && view->regions != NULL)
		delete_region_part(view, view->cursor_site, multipl);
	if(count_each == multipl && view->tot_comment_lines > 0)
		delete_in_all_comments(multipl, view->cursor_site, view);
	new_pos = view->cursor_site - multipl;
	if(new_pos <= 0) new_pos = 1;
	set_and_show_new_cursor_site(view, new_pos, FALSE, TRUE);
	}
else if( key == '-' || (key == ' ' && !view->hjkl)
			  /* gap key = - or space */
	|| key == '+' ) { /* insert gap in other seqs key */
	int newlength = 0, count = 0, count_each, debut, fin, test;
	    if(view->multipl->argument() > 0)
		mod_multipl(view,0);
	if(view->cursor_in_comment && key == '+') return;
	if(view->cursor_in_comment)
		test = FALSE;
	else
		test = view->cursor_seq < view->first_seq ||
	   		view->cursor_seq >=view->first_seq+view->tot_lines;
	if( test || view->cursor_site < view->first_site ||
	   	view->cursor_site >= 
		view->first_site + view->tot_sites )
		 { fl_ringbell(0); return; }
	if(view->alt_col_seq != NULL) return;
	if(key != '+') { /* gap key */
	    if(view->tot_sel_seqs > 1 && 
		(!view->cursor_in_comment) &&
		view->sel_seqs[view->cursor_seq - 1])
		{ debut = 1; fin = view->tot_seqs; test = TRUE;}
	    else
		{ debut = fin = view->cursor_seq; test = FALSE;}
	    for(num = debut; num<=fin; num++) 
		{
		if(test && !view->sel_seqs[num-1]) continue;
		count_each = insert_gaps_at(view, num, 
			view->cursor_site, multipl);
		if(count_each < multipl) fl_ringbell(0);
		if(num == view->cursor_seq) count = count_each;
		if(view->cursor_in_comment) {
		   if(newlength < view->comment_length[num-1])
			newlength = view->comment_length[num-1];
		     }
		else {
		   if(newlength < view->each_length[num-1])
			   newlength = view->each_length[num-1];
		     }
		}
/* si ttes seqs selectionnees, traiter aussi regions et comments */
	    if(count_each == multipl && 
		(!view->cursor_in_comment) &&
		view->tot_sel_seqs == view->tot_seqs) {
		if(view->regions != NULL)
			insert_region_part(view, view->cursor_site, multipl);
		if(view->tot_comment_lines > 0)
			insert_gap_all_comments(multipl,view->cursor_site, 
				view);
		}
	    }
	else	{ /* + ==> gap in other sequences */
		if(view->tot_sel_seqs == view->tot_seqs) {
			fl_ringbell(0); return;
			}
		for( num = 1; num <= view->tot_seqs; num++) {
			if(num == view->cursor_seq || 
			     (view->sel_seqs[view->cursor_seq-1] &&
			      view->sel_seqs[num-1] ) ) continue;
			count_each = insert_gaps_at(view, 
			    num, view->cursor_site, multipl);
			if(count_each < multipl) {
				fl_ringbell(0); return;
				}
			if(newlength < view->each_length[num-1])
			   	newlength = view->each_length[num-1];
			}
		count = multipl;
		if(count_each == multipl && 
			view->regions != NULL)
			insert_region_part(view, view->cursor_site, multipl);
		if(count_each == multipl && 
			view->tot_comment_lines > 0) {
			insert_gap_all_comments(multipl,view->cursor_site, 
				view);
			   }
		}
	new_pos = view->cursor_site + count;
	if(view->cursor_in_comment) {
	 	if(new_pos> view->comment_length[view->cursor_seq-1]+1)
	    		new_pos= view->comment_length[view->cursor_seq-1]+1;
	 	}
	else 	{
	 	if(new_pos > view->each_length[view->cursor_seq-1] + 1)
	   		 new_pos = view->each_length[view->cursor_seq-1] + 1;
	 	}
	if(view->cursor_in_comment)
		view->mod_comment_line = view->cursor_seq;
	else if(key != '+' ) {
		if(view->tot_sel_seqs > 1 && 
			view->tot_sel_seqs != view->tot_seqs &&
			view->sel_seqs[view->cursor_seq - 1])
			view->mod_seq = -1;
		else if(view->tot_sel_seqs <= 1 ||
			!view->sel_seqs[view->cursor_seq - 1] )
			view->mod_seq = view->cursor_seq;
		}
	update_current_seq_length(newlength, view);
	set_and_show_new_cursor_site(view, new_pos, FALSE,TRUE);
	}
else if( key >= '0' && key <= '9' ) { /* multiplicateur */
	multipl = view->multipl->argument() * 10;
	multipl += (key - '0');
	mod_multipl(view, multipl);
	return;
	}
else if( view->allow_seq_edit && (view->alt_col_seq == NULL) &&
	(!view->cursor_in_comment) &&
	( (key>='a' && key<='z') || (key>='A' && key<='Z') || key == ' ' ) ) {
	if(view->hjkl) {
		static char typedkey[]= "hjklHJKL ";
		static char dnaequivs[3][10]={
			"gatcGATCN", "tcgaTCAGN", "acgtACGTN"};
		char *p;
		p = strchr(typedkey, key);
		if(p != NULL) 
			key = *( dnaequivs[view->hjkl - 1] + (p - typedkey) );
		}
	if(key == ' ') num = 0;
	else	num = insert_char_in_seq(key, multipl, view);
	if( num == 0 ) fl_ringbell(0);
	else 	{
		view->mod_seq = view->cursor_seq;
		set_and_show_new_cursor_site(view, 
			view->cursor_site + num, FALSE, TRUE);
		}
	}
else	return;
if(view->multipl->argument() > 0) mod_multipl(view, 0);
}


Fl_Group *create_dna_scroller(SEA_VIEW *view, int x, int y, int w, int h, 
	int double_buffer)
{
  Fl_Group *dna_group;
  Fl_Widget *obj;
  int wmultipl, x_pos;
  user_data_plus *data;
dna_group = new Fl_Group(x,y,w,h);
int scroll_w = 15;
#ifdef __APPLE__
#define DELTA_G  13 // laisser place pour poigne de dimensionnement de fenetre
#else
#define DELTA_G  0
#endif
/* screen move haut */
view->up_screen_move = obj = 
(Fl_Widget*)new Fl_Repeat_Button(x+3, y,scroll_w,scroll_w,"@8>>");
obj->labeltype(FL_SYMBOL_LABEL);
data = new user_data_plus;
data->p = view;
data->value = 5;
obj->callback(lrdu_button_callback, data);
/* ascenc. vertical */
view->vertsli =  new Fl_Scrollbar(x+3, y + scroll_w + 2, scroll_w, 
		h - 3*scroll_w - 12, "");
view->vertsli->box(FL_DOWN_BOX);
data = new user_data_plus;
data->p = view;
data->value = 0;
((Fl_Scrollbar *)view->vertsli)->linesize(1);
view->vertsli->callback(vh_sliders_callback, data);
  view->vertsli->bounds(1,1);
  view->vertsli->slider_size(1);
  view->vertsli->Fl_Slider::value(1);
  view->vertsli->step(1);
view->vertsli->when(FL_WHEN_CHANGED);
/* screen move bas */
view->down_screen_move = obj = 
(Fl_Widget*)new Fl_Repeat_Button(x+3,
	y + h - 2 * scroll_w - 8, scroll_w, scroll_w, "@2>>");
obj->labeltype(FL_SYMBOL_LABEL);
data = new user_data_plus;
data->p = view;
data->value = 7;
obj->callback(lrdu_button_callback, data);

int y_scroll = y + h - scroll_w - 3;
/* valeur du multiplicateur */
fl_font(FL_HELVETICA, FL_NORMAL_SIZE);
  wmultipl = (int)fl_width("mult=9999");
  x_pos = 5;
  view->multipl = obj = (Fl_Widget*)new Fl_Box(FL_FLAT_BOX,
	x+x_pos, y_scroll - 2, wmultipl, FL_NORMAL_SIZE + 4, "");
  x_pos += wmultipl + 5;
  obj->labelfont(FL_HELVETICA);
  obj->labelsize(FL_NORMAL_SIZE);
  obj->align(FL_ALIGN_CENTER);


/* screen move gauche */
  view->left_screen_move = obj = 
(Fl_Widget*)new Fl_Repeat_Button(x+x_pos,y_scroll,scroll_w,scroll_w,"@<<");
obj->labeltype(FL_SYMBOL_LABEL);
  x_pos += scroll_w + 2;
data = new user_data_plus;
data->p = view;
data->value = 1;
obj->callback(lrdu_button_callback, data);
/* ascens. horizontal */ 
view->horsli =  
new Fl_Scrollbar(x + x_pos, y_scroll, w - x_pos - scroll_w - 2 - DELTA_G, scroll_w,"");
view->horsli->type(FL_HORIZONTAL);
view->horsli->box(FL_DOWN_BOX);
((Fl_Scrollbar *)view->horsli)->linesize(1);
data = new user_data_plus;
data->p = view;
data->value = 1;
view->horsli->callback(vh_sliders_callback, data);
view->horsli->bounds(1,1);
view->horsli->slider_size(1);
view->horsli->Fl_Slider::value(1);
view->horsli->step(1);
view->horsli->when(FL_WHEN_CHANGED);
/* screen move a droite */
  view->right_screen_move = obj = 
(Fl_Widget*)new Fl_Repeat_Button(x+w - scroll_w - DELTA_G, y_scroll, scroll_w, scroll_w,"@>>");
obj->labeltype(FL_SYMBOL_LABEL);
data = new user_data_plus;
data->p = view;
data->value = 3;
obj->callback(lrdu_button_callback, data);

/* noms + sequences */
Fl_Window *viewer;
if(double_buffer)
	  viewer = (Fl_Window *)new Fl_Double_Window(
					x+25-3, y, w - 25 - 1 + 6, h - 30 + 6);
else
	  viewer = new Fl_Window(x+25, y+3, w - 25 - 1, h - 30);
viewer->box(FL_DOWN_BOX);
viewer->resizable(viewer);
view->double_buffer = double_buffer;
view->DNA_obj = (Fl_Widget*)new DNA_obj(3, 3, w - 25 - 1, h - 30, view);
view->DNA_obj->labelfont(FL_COURIER_BOLD);
viewer->end();

dna_group->end();
dna_group->add_resizable(
	*new Fl_Box(x + x_pos, y + scroll_w + 2, w - x_pos - scroll_w - 2 - DELTA_G, 
		h - 3*scroll_w - 12) );
view->menu_regions = view->menu_file = view->menu_edit = 
		view->bouton_props = view->menu_species = NULL;
return dna_group;
}


SEA_VIEW *create_the_form(int double_buffer, int inverted,
		known_format default_format, int numb_dnacolors,
		int *dnacolors, 
		int numb_stdprotcolors, int *stdprotcolors,
		int numb_altprotcolors, int *altprotcolors,
		color_choice curr_color_choice, char *progname, char *movekeys,
		char *win_size, int allow_lower, char *help_file,
		int back_color, int region_back_color, int defaultfontsize)
{
Fl_Group *dna_group;
Fl_Window *my_form;
Fl_Widget *obj, *bouton_search, *champ_search, *bouton_goto, *champ_goto,
	*bouton_props, *menu_regions, *menu_edit, *menu_species,
	*menu_footers, *bouton_help;
Fl_Menu_Button *menu_file;
int black_and_white;
SEA_VIEW *view;
user_data_plus *data;
int labelSize = 12;
int borderWidth = 2;
int w_w = 700, w_h = 500;

sscanf(win_size, "%dx%d", &w_w, &w_h);
my_form = new Fl_Window(w_w, w_h);
my_form->color(FL_LIGHT1);
my_form->label(progname);
my_form->box(FL_FLAT_BOX);
int extra_menu_width = 2*(labelSize - 3);//largeur symbole menu
#define calc_width(nom) \
	(fl_font(FL_HELVETICA, labelSize), (int)fl_width(nom) + 4 * borderWidth)
#define add_menu(nom) \
	new Fl_Menu_Button(0,0,calc_width(nom)+extra_menu_width,25,nom)
#define add_button(nom) \
	new Fl_Button(0,0,calc_width(nom),25,nom)

Fl_Pack *menus = new Fl_Pack(5,5,10,25); /* groupe fixe des menus */
menus->type(FL_HORIZONTAL);
menus->spacing(2);

/* menu File */
menu_file = add_menu("File");
((Fl_Menu_Button*)menu_file)->add(
	"Open|Open Mase|Open Phylip|Open Clustal|Open MSF|Open Fasta|Open NEXUS|"
	"Save|Save as...|Save current sites|Prepare pdf|"
#if ! defined( __APPLE__) 
	"pdf options...|"
#endif
	"Quit");
menu_file->labelsize(labelSize);
menu_file->mode(OPEN_ANY, FL_MENU_DIVIDER);
menu_file->shortcut(OPEN_ANY, myFL_CTRL | 'o');
menu_file->mode(OPEN_NEXUS, FL_MENU_DIVIDER);
menu_file->mode(SAVE, FL_MENU_INACTIVE);
menu_file->shortcut(SAVE, myFL_CTRL | 's');
menu_file->mode(SAVE_AS, FL_MENU_INACTIVE);
menu_file->mode(SAVE_REGIONS, FL_MENU_DIVIDER | FL_MENU_INACTIVE);
#if defined( __APPLE__) 
menu_file->mode(PRINTOUT, FL_MENU_DIVIDER);
#else
menu_file->mode(PDFOPTIONS, FL_MENU_DIVIDER);
#endif
menu_file->shortcut(QUIT, myFL_CTRL | 'q');
menu_file->callback(file_menu_callback, 0);
menu_file->align((Fl_Align) (FL_ALIGN_INSIDE | FL_ALIGN_LEFT) );
menu_file->box(FL_UP_BOX);

/* menu Props */
bouton_props = obj = add_menu("Props");
obj->labelsize(labelSize);
obj->align((Fl_Align) (FL_ALIGN_INSIDE | FL_ALIGN_LEFT) );

/* menu regions */
menu_regions = obj = add_menu("Sites");
data = new user_data_plus;
obj->labelsize(labelSize);
obj->callback(regions_menu_callback, data);
obj->align((Fl_Align) (FL_ALIGN_INSIDE | FL_ALIGN_LEFT) );
obj->box(FL_UP_BOX);

/* menu species */
menu_species = obj = add_menu("Species");
data = new user_data_plus;
obj->labelsize(labelSize);
obj->callback(species_menu_callback, data);
obj->align((Fl_Align) (FL_ALIGN_INSIDE | FL_ALIGN_LEFT) );
obj->box(FL_UP_BOX);

/* menu footers */
menu_footers = obj = add_menu("Footers");
((Fl_Menu_Button*)obj)->add("Show footers|Create footer|Delete footer");
((Fl_Menu_ *)obj)->mode(SHOW_HIDE_FOOTERS, FL_MENU_INACTIVE);
obj->labelsize(labelSize);
obj->callback(footers_menu_callback, 0);
obj->align((Fl_Align) (FL_ALIGN_INSIDE | FL_ALIGN_LEFT) );
obj->box(FL_UP_BOX);

/* bouton search + champ Search */
bouton_search = obj = add_button("Search:");
obj->labelsize(labelSize);
data = new user_data_plus;
data->value = 0;
obj->callback(search_callback, data);
champ_search = obj = (Fl_Widget*)new Fl_Input(0,0, 80, 25, "");
((Fl_Input*)obj)->type(FL_NORMAL_INPUT);
((Fl_Input*)obj)->when(FL_WHEN_ENTER_KEY);
data = new user_data_plus;
data->value = 1;
champ_search->callback(search_callback, data);
data = (user_data_plus *)bouton_search->user_data();
data->p = champ_search;

/*  bouton + champ Goto */
bouton_goto = obj = add_button("Goto:");
obj->labelsize(labelSize);
data = new user_data_plus;
data->value = 0;
obj->callback(goto_callback, data);
champ_goto = obj = (Fl_Widget*)new Fl_Input(0, 0, 80, 25, "");
((Fl_Input*)obj)->type(FL_NORMAL_INPUT);
data = new user_data_plus;
data->value = 1;
champ_goto->callback(goto_callback, data);
data = (user_data_plus *)bouton_goto->user_data();
data->p = champ_goto;
((Fl_Input*)obj)->when(FL_WHEN_ENTER_KEY);

/* menu Edit */
menu_edit = obj = add_menu("Edit");
((Fl_Menu_Button*)obj)->add( 
   "Rename sequence|Edit comments|Delete sequence(s)|Create sequence|"
   "Load sequence|Duplicate sequence|"
   "Complement sequence|Reverse sequence|Exchange Us and Ts|Align sites|"
   "Dot plot|Consensus sequence|Del. gap-only sites");
obj->labelsize(labelSize);
obj->callback(edit_menu_callback, 0);
obj->align((Fl_Align) (FL_ALIGN_INSIDE | FL_ALIGN_LEFT) );
obj->box(FL_UP_BOX);

/* bouton help */
bouton_help = add_button("Help");
bouton_help->labelsize(labelSize);
bouton_help->callback(help_callback, help_file);

#undef add_menu
#undef add_button
#undef calc_width

menus->end();
menus->resizable(NULL);

view = (SEA_VIEW *)calloc(1, sizeof(SEA_VIEW));
if(view == NULL) out_of_memory();

/* tout le groupe scroller */
dna_group = create_dna_scroller(view,
	0, 35, my_form->w() - 5, my_form->h() - 35, double_buffer);
my_form->resizable( dna_group );
my_form->end();
my_form->callback(mainwin_close_callback, view);
my_form->position((Fl::w() - my_form->w())/2, (Fl::h() - my_form->h())/2);

/* initialisation independantes des sequences */
view->movekeys = movekeys;
view->menu_regions = menu_regions;
view->menu_species = menu_species;
view->menu_file = menu_file;
view->menu_edit = menu_edit;
view->menu_footers = menu_footers;
view->bouton_props = bouton_props;
menu_file->user_data( view );
data = (user_data_plus *)menu_regions->user_data();
data->p = view;
data = (user_data_plus *)menu_species->user_data();
data->p = view;
menu_edit->user_data( view );
menu_footers->user_data( view );
data = (user_data_plus *)champ_search->user_data();
data->p = view;
data = (user_data_plus *)champ_goto->user_data();
data->p = view;
view->format_for_save = default_format; /* default format for saving */
view->tot_seqs = 0;
view->first_seq = 1; view->tot_sites = 1;
view->numb_gc = 1;
view->line_height = 1;
view->char_width = 1;
view->draw_names = -1;
view->mod_seq = 0;
view->mod_comment_line = 0;
view->consensus_threshold = 60;
view->consensus_allowgaps = FALSE;
view->dnawin = my_form;
view->max_seq_length = MINI_MAX_LENGTH;
mod_multipl(view, 0);

#ifdef WIN32
black_and_white = FALSE;
#elif defined(__APPLE__)
black_and_white = FALSE;
#else
black_and_white = (fl_xpixel(FL_BLACK) == fl_xpixel(FL_RED));
#endif
if ( ! black_and_white) { 
	/* couleurs du fond pour seqs */
  	view->DNA_obj->color(back_color, region_back_color); 
	view->region_color = FL_WHITE;
	view->inverted_colors = inverted;
	if(inverted) {
		view->DNA_obj->labelfont( FL_COURIER);
		}
	view->allow_lower = allow_lower;
	}
else	{ /* the Black and White case */
	numb_dnacolors = numb_stdprotcolors = numb_altprotcolors = 1;
	dnacolors[0] = FL_BLACK;
	/* couleur du fond pour seqs */
  	view->DNA_obj->color(FL_WHITE, FL_WHITE); 
	view->region_color = FL_BLACK;
	view->numb_gc = 1;
	view->curr_colors = dnacolors;
	view->inverted_colors = FALSE;
	view->allow_lower = TRUE;
	}
view->DNA_obj->parent()->color( view->DNA_obj->color() );
/* taille par defaut des lettres des sequences */
view->DNA_obj->labelsize(defaultfontsize);
/* creation du menu Props et de ses sous-menus */
create_props_menu((Fl_Menu_Button *)bouton_props, view,
	curr_color_choice, inverted, black_and_white, 
	view->DNA_obj->labelsize());
view->dnacolors = dnacolors;
view->numb_dnacolors = numb_dnacolors;
view->stdprotcolors = stdprotcolors;
view->numb_stdprotcolors = numb_stdprotcolors;
view->altprotcolors = altprotcolors;
view->numb_altprotcolors = numb_altprotcolors;
view->namecolor = FL_BLACK;
view->alt_colors = curr_color_choice;
view->clustal_options = NULL;
view->show_comment_lines = FALSE;
view->tot_comment_lines = 0;
return view;
}
