#include <errno.h>
#include "uint16.h"
#include "alloc.h"
#include "bytestr.h"
#include "error.h"
#include "fmtscan.h"
#include "buffer.h"
#include "strerr2.h"
#include "gen_alloc.h"
#include "stralloc.h"
#include "bufalloc.h"
#include "tai.h"
#include "iopause.h"
#include "sgetopt.h"
#include "djbunix.h"
#include "skamisc.h"
#include "skadns.h"

char const *PROG = "skadnsfilter" ;
#define USAGE "skadnsfilter [ -c concurrency ] [ -l lines ]"

typedef struct line line, *line_ref ;
typedef enum { STATE_READ, STATE_SENT, STATE_OK } state_t ;

struct line
{
  unsigned int offset ;
  stralloc data ;
  char ip[4] ;
  uint16 id ;
  state_t state ;
} ;

static inline void die (void)
{
  strerr_diefu1sys(111, "run") ;
}

int main (int argc, char const *const *argv)
{
  unsigned int maxactive = SKADNS_MAXCONCURRENCY ;
  unsigned int maxlines = SKADNS_MAXCONCURRENCY << 2 ;
  unsigned int pendingp = 0, pendinglen = 0, active = 0, zp = 0, zlen = 0 ;
  unsigned int linenb[SKADNS_MAXCONCURRENCY] ;
  unsigned int *pending ;
  line_ref z ;
  skadnsinfo_ref a ;
  stralloc in = GEN_ALLOC_ZERO ;
  iopause_fd x[3] ;
  unsigned char reading = 1 ;

  {
    register int opt ;
    while ((opt = subgetopt(argc, argv, "c:l:")) != opteof)
      switch(opt)
      {
        case 'c' :
        {
          unsigned int c ;
          if (!uint0_scan(optarg, &c)) strerr_dieusage(100, USAGE) ;
          if (c < 1) c = 1 ;
          if (c <= SKADNS_MAXCONCURRENCY) maxactive = c ;
          break ;
        }
        case 'l' :
        {
          unsigned int l ;
          if (!uint0_scan(optarg, &l)) strerr_dieusage(100, USAGE) ;
          if (l < 1) l = 1 ;
          if (l > 1000000) l = 1000000 ;
          maxlines = l ;
          break ;
        }
        default:
          strerr_dieusage(100, USAGE) ;
      }
  }

  z = (line_ref)alloc(maxlines * sizeof(line)) ;
  if (!z) die() ;

  pending = (unsigned int *)alloc((maxlines - maxactive) * sizeof(unsigned int)) ;
  if (!pending) die() ;

  a = skadns_startf() ;
  if (!a) strerr_diefu2sys(111, "run ", SKADNSD_PROG) ;  

  byte_zero(z, maxlines * sizeof(line)) ;
  x[0].fd = skadns_fd(a) ;
  x[0].events = IOPAUSE_EXCEPT | IOPAUSE_READ ;
  x[1].fd = 1 ;
  x[2].fd = 0 ;
  ndelay_on(0) ; ndelay_on(1) ;

  while (reading || bufalloc_len(bufalloc_1) || zlen)
  {
    int r ;
    x[1].events = IOPAUSE_EXCEPT | (bufalloc_len(bufalloc_1) ? IOPAUSE_WRITE : 0) ;
    x[2].events = reading ? (IOPAUSE_EXCEPT | ((zlen < maxlines) ? IOPAUSE_READ : 0)) : 0 ;

    r = iopause(x, reading ? 3 : 2, 0, 0) ;

   /* exceptions */
    if (r < 0) strerr_dief1sys(111, "trouble in iopause") ;
    if (x[1].revents & IOPAUSE_EXCEPT) strerr_diefu1x(111, "write to stdout") ;
    if (x[0].revents & IOPAUSE_EXCEPT) strerr_dief1x(111, "server hung up") ;

   /* can write */
    if (x[1].revents & IOPAUSE_WRITE)
    {
      if ((bufalloc_flush(bufalloc_1) == -1) && !error_isagain(errno))
        strerr_diefu1sys(111, "flush stdout") ;
    }

   /* can read */
    if (buffer_len(buffer_0) || (reading && (x[2].revents & IOPAUSE_READ)))
    {
      r = 1 ;
      for (;;)
      {
        int i ;
        if (zlen >= maxlines) break ;
        r = skagetln(buffer_0, &in, '\n') ;
        if (r <= 0) break ;
        i = (zp + zlen++) % maxlines ;
        z[i].data = in ;
        in.s = 0 ; in.a = 0 ; in.len = 0 ;
        z[i].offset = ip4_scan(z[i].data.s, z[i].ip) ;
        if (z[i].offset)
        {
          if (active >= maxactive)
          {
            z[i].state = STATE_READ ;
            pending[(pendingp + pendinglen++) % (maxlines - maxactive)] = i ;
          }
          else
          {
            z[i].id = skadns_name4_send(a, z[i].ip) ;
            if (!z[i].id)
            {
              if (!stralloc_inserts(&z[i].data, z[i].offset, ":skadns_send-failed")) die() ;
              z[i].state = STATE_OK ;
            }
            else
            {
              z[i].state = STATE_SENT ;
              linenb[z[i].id-1] = i ;
              active++ ;
            }
          }
        }
        else z[i].state = STATE_OK ;
      }
      if ((r == 0) || ((r == -1) && (errno == EPIPE)))
        reading = 0 ;
      else if ((r == -1) && !error_isagain(errno))
        strerr_diefu1sys(111, "read from stdin") ;
    }

   /* stdin close exception */
    if (reading && (x[2].revents & IOPAUSE_EXCEPT)) reading = 0 ;

   /* some answers have arrived */
    if (x[0].revents & IOPAUSE_READ)
    {
      int j = skadns_readanswers(a) ;
      uint16 const *idlist = skadns_recvlist(a) ;
      if (j == -1) strerr_dief1x(111, "server sent garbage") ;
      while (j--)
      {
        int r ;
        unsigned int tmpbase = satmp.len ;
        unsigned int i = linenb[idlist[j] - 1] ;
        r = skadns_name4_recv(a, z[i].id, &satmp) ;
        if (!r) strerr_dief1x(101, "error in skadns") ;
        if (r == -1)
        {
          if (!stralloc_inserts(&z[i].data, z[i].offset, ":error-in-skadns")) die() ;
        }
        else if (r == -2)
        {
          if (!stralloc_inserts(&z[i].data, z[i].offset, ":dns-error")) die() ;
        }
        else if (r == -3)
        {
          if (!stralloc_inserts(&z[i].data, z[i].offset, ":timed-out")) die() ;
        }
        else if (satmp.len)
        {
          if (!stralloc_insertb(&z[i].data, z[i].data.len - 1, "=", 1)
          || (!stralloc_insert(&z[i].data, z[i].offset + 1, &satmp)))
            die() ;
        }
        satmp.len = tmpbase ;
        z[i].state = STATE_OK ;
        active-- ;
      }

     /* fill active table when possible */
      while (pendinglen && (active < maxactive))
      {
        unsigned int i = pending[pendingp] ;
        pendingp = (pendingp + 1) % (maxlines - maxactive) ;
        pendinglen-- ;
        z[i].id = skadns_name4_send(a, z[i].ip) ;
        if (!z[i].id)
        {
          if (!stralloc_inserts(&z[i].data, z[i].offset, ":skadns_send-failed")) die() ;
          z[i].state = STATE_OK ;
        }
        else
        {
          z[i].state = STATE_SENT ;
          linenb[z[i].id-1] = i ;
          active++ ;
        }
      }
    }

   /* put available results on stdout */
    while (zlen && (z[zp].state == STATE_OK))
    {
      if (bufalloc_put(bufalloc_1, z[zp].data.s, z[zp].data.len) == -1) die() ;
      stralloc_free(&z[zp].data) ;
      zp = (zp + 1) % maxlines ;
      zlen-- ;
    }
  }
  skadns_end(a) ;
  return 0 ;
}
