/*
 * relay.c
 *
 * Description:  Check relay tools
 * Developed by: Alexander Djourik <sasha@iszf.irk.ru>
 *		 Anton Gorbunov <anton@iszf.irk.ru>
 *		 Pavel Zhilin <pzh@iszf.irk.ru>
 *
 * Copyright (c) 2003,2004,2005 Alexander Djourik. All rights reserved.
 *
 */

#include <pwd.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <resolv.h>
#include <ctype.h>
#include <syslog.h>
#include <errno.h>
#include <fcntl.h>
#include <sysexits.h>
#include <openssl/md5.h>
#include "smc-milter.h"
#include "relay.h"
#include "utils.h"
#include "smtp.h"

#define MAXPACKET	8192
#define PWD_SIZE	256

#ifndef MAXLINE
    #define MAXLINE	4096
#endif

typedef union {
	HEADER hdr;
	u_char buf[MAXPACKET];
} querybuf;

int
find_user (char *usersfile, char *name) {
	FILE *fh;
	char buffer[MAXLINE];
	int found = 0;

	/* check for required data presented */
	if (!name || *name == '\0') return 0;

	if (!(fh = fopen(usersfile, "r"))) return 0;

	while (fgets(buffer, MAXLINE, fh)) {
	    if (*buffer == '#') continue;
	    if (strncasecmp(buffer, name, strlen(name)) == 0) {
		found = 1;
		break;
	    }
	}

	fclose(fh);

	return found;
}


int
check_relay (char *conn, char *access_file) {
	FILE *fh = fopen(access_file, "r");
	unsigned long addr, value, mask;
	char buffer[MAXLINE], *p;
	int d1, d2, d3, d4;
	int found = 0;

	if (!fh) {
	    syslog(LOG_ERR, "Failed to open %s: %s",
		access_file, strerror(errno));
	    return -1;
	}

	addr = ntohl(inet_addr(conn));
	while (fgets (buffer, MAXLINE, fh)) {
	    if (*buffer == '#') continue;

	    p = buffer + strlen(buffer) - 1;
	    if (*p == '\n') *p = '\0';
	    if (*(p - 1) == '\r') *(--p) = '\0';

	    if (strcasecmp (p - 2, "OK") &&
		strcasecmp (p - 5, "RELAY")) continue;

	    d1 = d2 = d3 = d4 = 0;
	    if (sscanf(buffer, "%3d.%3d.%3d.%3d", &d1, &d2, &d3, &d4) > 0) {
		sprintf(buffer, "%d.%d.%d.%d", d1, d2, d3, d4);
		value = ntohl(inet_addr(buffer));
		mask  = (d1|d2|d3|d4) ? 0xff000000 : 0;
		mask |= (d2|d3|d4) ? 0x00ff0000 : 0;
		mask |= (d3|d4) ? 0x0000ff00 : 0;
		mask |= (d4) ? 0x000000ff : 0;
		if ((addr & mask) == value) {
		    found = 1; break;
		}
	    }
	}

	fclose (fh);
	return found;
}

int
r_check_dnsbl (char *dnsbl_domain, char *client_addr)
{
	querybuf answer;
	char query[MAXDNAME];
	int ret, d1, d2, d3, d4;
	struct __res_state res_local;
	int result = 1;

	/* make a query */
	if (sscanf(client_addr, "%3d.%3d.%3d.%3d", &d1, &d2, &d3, &d4) == 4)
	    sprintf(query, "%d.%d.%d.%d.%s", d4, d3, d2, d1, dnsbl_domain);
	else return -1;

	if (resolver_init(&res_local) < 0)
	    return -1;

	/* look up given domain name */
	do { ret = res_nquery(&res_local, query, C_IN, T_ANY,
	    (u_char *) &answer, sizeof(answer));
	} while (ret < 0 && errno == EINTR);

	if (ret < 0) {
	    if (res_local.res_h_errno == HOST_NOT_FOUND) result = 0;
	    else {
		syslog(LOG_ERR, "Could not connect to %s: %s",
		    dnsbl_domain, strerror(errno));
		result =  -1;
	    }
	}
	resolver_close(&res_local);

	return result;
}

int
check_dnsbl (char *dnsbl_domain, char *client_addr, int cache_time) 
{
	char md_sign[MD5_STRING_LENGTH + 1];
	rec  value;
	time_t t;
    	int result;
	
	t = time(NULL);

	md5sign(dnsbl_domain, client_addr, md_sign);

	if (get_record(HASH_DNSBL_DB, md_sign, &value) == 0) {
	    if ((t - value.time1 < cache_time))
		return value.data;
	    else
		del_record (HASH_DNSBL_DB, md_sign);
	}

	result = r_check_dnsbl (dnsbl_domain, client_addr);

	if (result >= 0) {
	    value.time1 = t;
	    value.data = result;
	    add_record (HASH_DNSBL_DB, md_sign, &value, cache_time);
	}

	return result;
}
int
r_check_connect (char *client_addr, char *domain_name) {
	querybuf answer;
	char buffer[8192];
	struct hostent host_buf, *host = NULL;
	uint32_t host_addr;
	char host_name[MAXDNAME];
	char client_name[MAXDNAME];
	register u_char *cp, *eom;
	int count, type, dlen, pref, ret;
	char nsaddr_list[MAXNS][MAXDNAME];
	int nscount = 0, found = 0;
	int i, d1, d2, d3, d4, result = -1;
	struct __res_state res_local;

	if (resolver_init(&res_local) < 0)
	    return -1;

	/* look up MX records for the given domain name */
	do { ret = res_nquery(&res_local, domain_name, C_IN, T_MX,
	    (u_char *) &answer, sizeof(answer));
	} while (ret < 0 && errno == EINTR);

	if (ret < 0) {
	    result = NOREC_MX;
	    if (res_local.res_h_errno == NETDB_INTERNAL) {
		syslog(LOG_ERR, "DNS error: %s", strerror(errno));
		result =  -1;
	    }
	    resolver_close(&res_local);
	    return result;
	}
	
	eom = answer.buf + ret;
	cp = answer.buf + sizeof(HEADER);
	cp += dn_skipname(cp, eom) + QFIXEDSZ;
	count = ntohs(answer.hdr.ancount) +
		ntohs(answer.hdr.nscount);
	host_addr = inet_addr(client_addr);

	while (--count >= 0 && cp < eom) {
	    cp += dn_skipname(cp, eom);

	    GETSHORT(type, cp);
	    cp += (INT16SZ + INT32SZ);
	    GETSHORT(dlen, cp);

	    if (type == T_MX) {
		GETSHORT(pref, cp);
		if ((ret = dn_expand(answer.buf, eom,
		    cp, host_name, MAXDNAME-1)) < 0) {
		    resolver_close(&res_local);
		    return -1;
		}

#if defined __FreeBSD__
		host = gethostbyname_r(host_name, &host_buf, buffer,
					sizeof(buffer), &ret);
		if(!ret) {
#else
		if (!(gethostbyname_r(host_name, &host_buf, buffer,
					sizeof(buffer), &host, &ret))) {
#endif /* __FreeBSD__ */
		    while (*host->h_addr_list) {
			if (host_addr == *(uint32_t *)(*host->h_addr_list)) {
			    resolver_close(&res_local);
			    return HOST_FOUND;
			}
			host->h_addr_list++;
		    }
		}

		cp += ret;
	    } else cp += dlen;
	}

	/* Check relay by NS algorithm:

	Lookup the NS hosts for given "client site" (domain's NS list) and
	for given "connect address" (connect NS list); Search for each of hosts
	from connect NS list in the domain's NS list; Relay is OK if found.

	Copyright (c) 2003 Alexander Djourik. All rights reserved.       
	Please see the file COPYING in this directory for full copyright
	information. */

	/* look up NS records for the given domain name */
	do { ret = res_nquery(&res_local, domain_name, C_IN, T_ANY,
	    (u_char *) &answer, sizeof(answer));
	} while (ret < 0 && errno == EINTR);
	if (ret < 0) {
	    result = NOREC_DOMAIN;
	    if (res_local.res_h_errno == NETDB_INTERNAL) {
	    	syslog(LOG_ERR, "DNS error: %s", strerror(errno));
		result =  -1;
	    }
	    resolver_close(&res_local);
	    return result;
	}
	
	eom = answer.buf + ret;
	cp = answer.buf + sizeof(HEADER);
	cp += dn_skipname(cp, eom) + QFIXEDSZ;
	count = ntohs(answer.hdr.ancount) +
		ntohs(answer.hdr.nscount);

	while (--count >= 0 && cp < eom) {
	    cp += dn_skipname(cp, eom);

	    GETSHORT(type, cp);
	    cp += (INT16SZ + INT32SZ);
	    GETSHORT(dlen, cp);

	    if (type == T_NS && nscount < MAXNS) {
		if ((ret = dn_expand(answer.buf, eom,
		    cp, host_name, MAXDNAME-1)) < 0) {
		    resolver_close(&res_local);
		    return -1;
		}

		strcpy(nsaddr_list[nscount++], host_name);
		cp += ret;
	    } else cp += dlen;
	}

	/* make client host record */
	if (sscanf(client_addr, "%3d.%3d.%3d.%3d", &d1, &d2, &d3, &d4) == 4)
	    sprintf(client_name, "%d.%d.%d.%d.in-addr.arpa", d4, d3, d2, d1);
	else {
	    resolver_close(&res_local);
	    return -1;
	}

	/* reverse look up NS records for the client address */
	do { ret = res_nquery(&res_local, client_name, C_IN, T_PTR,
	    (u_char *) &answer, sizeof(answer));
	} while (ret < 0 && errno == EINTR);
	if (ret < 0) {
	    result = NOREC_CLIENT;
	    if (res_local.res_h_errno == NETDB_INTERNAL) {
	    	syslog(LOG_ERR, "DNS error: %s", strerror(errno));
		result =  -1;
	    }
	    resolver_close(&res_local);
	    return result;
	}
	
	eom = answer.buf + ret;
	cp = answer.buf + sizeof(HEADER);
	cp += dn_skipname(cp, eom) + QFIXEDSZ;
	count = ntohs(answer.hdr.ancount) +
		ntohs(answer.hdr.nscount);

	while (--count >= 0 && cp < eom) {
	    cp += dn_skipname(cp, eom);

	    GETSHORT(type, cp);
	    cp += (INT16SZ + INT32SZ);
	    GETSHORT(dlen, cp);

	    if (!found && type == T_PTR) {
		if ((ret = dn_expand(answer.buf, eom,
		    cp, client_name, MAXDNAME-1)) < 0) {
		    resolver_close(&res_local);
		    return -1;
		}

		found = 1;
		cp += ret;
	    } else if (type == T_NS) {
		if ((ret = dn_expand(answer.buf, eom,
		    cp, host_name, MAXDNAME-1)) < 0) {
		    resolver_close(&res_local);
		    return -1;
		}

		/* search for client's NS in the domain's NS list */
		for (i = 0; i < nscount; i++) {
		    if (strcasecmp(nsaddr_list[i], host_name) == 0) {
			resolver_close(&res_local);
			return HOST_FOUND;
		    }
		}
		cp += ret;
	    } else cp += dlen;
	}

	/* look up NS records for the found client host name */
	do { ret = res_nquery(&res_local, client_name, C_IN, T_ANY,
	    (u_char *) &answer, sizeof(answer));
	} while (ret < 0 && errno == EINTR);
	if (ret < 0) {
	    result =  HOST_UNKNOWN;
	    if (res_local.res_h_errno == NETDB_INTERNAL) {
	    	syslog(LOG_ERR, "DNS error: %s", strerror(errno));
		result =  -1;
	    }
	    resolver_close(&res_local);
	    return result;
	}
	
	eom = answer.buf + ret;
	cp = answer.buf + sizeof(HEADER);
	cp += dn_skipname(cp, eom) + QFIXEDSZ;
	count = ntohs(answer.hdr.ancount) +
		ntohs(answer.hdr.nscount);

	while (--count >= 0 && cp < eom) {
	    cp += dn_skipname(cp, eom);

	    GETSHORT(type, cp);
	    cp += (INT16SZ + INT32SZ);
	    GETSHORT(dlen, cp);

	    if (type == T_NS) {
		if ((ret = dn_expand(answer.buf, eom,
		    cp, host_name, MAXDNAME-1)) < 0) {
		    resolver_close(&res_local);
		    return -1;
		}

		/* search for client's NS in the domain's NS list */
		for (i = 0; i < nscount; i++) {
		    if (strcasecmp(nsaddr_list[i], host_name) == 0) {
			resolver_close(&res_local);
			return HOST_FOUND;
		    }
		}
		cp += ret;
	    } else cp += dlen;
	}

	resolver_close(&res_local);

	return HOST_UNKNOWN;
}
int
check_connect (char *client_addr, char *domain_name, int cache_time) 
{
	char md_sign[MD5_STRING_LENGTH + 1];
	rec  value;
	time_t t;
    	int result;
	
	t = time(NULL);

	md5sign(domain_name, client_addr, md_sign);

	if (get_record(HASH_NSRELAY_DB, md_sign, &value) == 0) {
	    if ((t - value.time1 < cache_time))
		return value.data;
	    else
		del_record (HASH_DNSBL_DB, md_sign);
	}

	result = r_check_connect(client_addr, domain_name);

	if (result >= 0) {
	    value.time1 = t;
	    value.data = result;
	    add_record (HASH_NSRELAY_DB, md_sign, &value, cache_time);
	}

	return result;
}

int
check_local (char *conn) {
	unsigned long addr;

	addr = ntohl(inet_addr(conn));
	if ((addr & 0xffffff00) == 0xc0000200 /* 192.0.2.0/24    test network   */
	||  (addr & 0xffffff00) == 0xc0586300 /* 192.88.99.0/24  RFC 3068       */
	||  (addr & 0xffff0000) == 0xa9fe0000 /* 169.254.0.0/16  link local     */
	||  (addr & 0xffff0000) == 0xc0a80000 /* 192.168.0.0/16  private use    */
	||  (addr & 0xfffe0000) == 0xc6120000 /* 198.18.0.0/15   RFC 2544       */
	||  (addr & 0xfff00000) == 0xac100000 /* 172.16.0.0/12   private use    */
	||  (addr & 0xff000000) == 0x00000000 /* 0.0.0.0/8       "this" network */
	||  (addr & 0xff000000) == 0x7f000000 /* 127.0.0.0/8     loopback       */
	||  (addr & 0xff000000) == 0x0a000000 /* 10.0.0.0/8      private use    */
	||  (addr & 0xf0000000) == 0xe0000000 /* 224.0.0.0/4     RFC 3171       */
	) return 1;

	return 0;
}

int
r_verify (char *mail_addr, char *mail_host, char *msg_id, char *domain) 
{
	char vrfy_addr[MAXLINE];
	char mxlist[MAXMX][MAXLINE];
	int rc, i, mxcount;

	/* check for required data presented */
	if (!mail_addr || !mail_host) 
	    return -1;

	/* check for address is local */
	if (!strchr(mail_addr, '@')) return 1;

	/* get best mx for recipient address */
	mxcount = get_mxlist(mail_host, mxlist);
	if (mxcount <= 0)
	    return mxcount;

	/* make local address */
	sprintf(vrfy_addr, POSTMASTER "@%s", domain);

	/* verify address */
	for (i = 0, rc = 0; i < mxcount; i++) {
	    rc = mailer(domain, vrfy_addr, mail_addr, mxlist[i], NULL, 1);
	    if (rc >= 0) break;
	}

	switch (rc) {
	case -1:
	    syslog(LOG_MAIL|LOG_INFO, "%s: verify: <%s>, stat=Temp",
	    msg_id, mail_addr); break;
	case 0:
	    syslog(LOG_MAIL|LOG_INFO, "%s: verify: <%s>, relay=%s, stat=Fail",
	    msg_id, mail_addr, mxlist[i]); break;
	default:
	    syslog(LOG_MAIL|LOG_INFO, "%s: verify: <%s>, relay=%s, stat=Ok",
	    msg_id, mail_addr, mxlist[i]); break;
	}

	return rc;
}

int
verify (char *mail_addr, char *mail_host, char *msg_id, char *domain, int cache_time) 
{
	char md_sign[MD5_STRING_LENGTH + 1];
	rec  value;
	time_t t;
    	int result;
	
	t = time(NULL);

	md5sign(mail_addr, mail_host, md_sign);

	if (get_record(HASH_VERIFY_DB, md_sign, &value) == 0) {
	    if ((t - value.time1 < cache_time))
		return value.data;
	    else
		del_record (HASH_VERIFY_DB, md_sign);
	}

	result = r_verify(mail_addr, mail_host, msg_id, domain);

	if (result >= 0) {
	    value.time1 = t;
	    value.data = result;
	    add_record (HASH_VERIFY_DB, md_sign, &value, cache_time);
	}

	return result;
}

int
resolver_init (struct __res_state *res_local) {
	if (res_ninit(res_local) < 0) return -1;

	_res.options &= ~(RES_DEFNAMES|RES_DNSRCH);
	_res.options |= (RES_STAYOPEN|RES_USEVC);
	_res.retrans = DNS_RETRANS;
	_res.retry = DNS_RETRY;
	
	return 0;
}

void
resolver_close (struct __res_state *res_local) {
	if (_res.options & RES_INIT)
	    res_nclose(res_local);
}

/* eof */
